/*
 * Apollo vnc server
 *
 * Copyright Jim Rees, September 2004
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * This software is provided "as-is" and without warranty.
 */

#include <sys/param.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include <stdio.h>
#include <time.h>
#include <netdb.h>
#include <errno.h>

#include <apollo/base.h>
#include <apollo/gpr.h>
#include <apollo/pad.h>

typedef unsigned char  CARD8;
typedef unsigned short CARD16;
typedef unsigned long  CARD32;

#include "rfbproto.h"

#define PORT 5900
#define RS 16

extern u_char monotab[256][8];
extern u_char stipple[];

extern char *optarg;
extern int optind;

extern struct {
    unsigned int keysym;
    char *keydef;
} keydefs[];

char *clmsgnames[] = {
    "rfbSetPixelFormat",
    "rfbFixColourMapEntries",
    "rfbSetEncodings",
    "rfbFramebufferUpdateRequest",
    "rfbKeyEvent",
    "rfbPointerEvent",
    "rfbClientCutText",
};

void proto_init(FILE *f);
void serve(FILE *f);
void update(FILE *f, int inc, pinteger x0, pinteger y0, pinteger w, pinteger h);
void sendrhd(FILE *f, int x, int y, int enc);
void sendrect(FILE *f, u_char *p, int lw);
void sendcopy(FILE *f, int x, int y);
void sendsolid(FILE *f, u_char px);
int ispx(u_char *p, int lw, pinteger px);
int rectcmp(u_char *p0, int lw0, u_char *p1, int lw1);
void sendkey(FILE *f, int down, unsigned int key);
void client_cut(FILE *f, char *s);
int netread(void *p, int elemsize, int numelem, FILE *f);
char *ErrMsg(status_$t st);
void ErrPrint(char *s, status_$t st);

int vflag, need_refresh;
stream_$id_t padfd;
int clbig = 1, willCopyRect, willRRE;
gpr_$disp_char_t dispchar;
gpr_$offset_t bmsize;
gpr_$bitmap_desc_t bmdesc;

int
main(ac, av)
int ac;
char *av[];
{
    struct sockaddr_in saddr, claddr;
    int i, sfd, fd, slen;
    int port = PORT;
    FILE *f, *logf = stdout;
    time_t t;
    short sn;
    status_$t st;

    while ((i = getopt(ac, av, "p:v")) != -1) {
	switch (i) {
	case 'p':
	    port = atoi(optarg);
	    break;
	case 'v':
	    vflag = 1;
	    break;
	}
    }

    saddr.sin_family = AF_INET;
    saddr.sin_port = htons(port);
    saddr.sin_addr.s_addr = INADDR_ANY;

    if ((sfd = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
	perror("socket");
	exit(1);
    }
    i = 1;
    setsockopt(sfd, SOL_SOCKET, SO_REUSEADDR, &i, sizeof i);

    if (bind(sfd, (struct sockaddr *) &saddr, sizeof saddr) < 0) {
	perror("bind");
	exit(1);
    }

    if (listen(sfd, 0) < 0) {
	perror("listen");
	exit(1);
    }

    pad_$isa_dm_pad(1, &st);
    if (st.all == status_$ok)
	padfd = 1;
    else {
	fprintf(stderr, "stdin not a pad, trying /dev/pad0000\n");
	padfd = open("/dev/pad0000", 2);
	if (padfd < 0)
	    perror("padfd");
	pad_$isa_dm_pad(padfd, &st);
	if (st.all != status_$ok) {
	    ErrPrint("/dev/pad0000", st);
	    fprintf(stderr, "can't find a pad fd\n");
	}
    }

    gpr_$inq_disp_characteristics(gpr_$borrow_nc, padfd,
				  sizeof dispchar, &dispchar, &sn, &st);
    if (vflag)
	printf("display is %dx%d depth %d\n",
	       dispchar.x_window_size, dispchar.y_window_size,
	       dispchar.n_planes);
    bmsize.x_size = dispchar.x_window_size;
    bmsize.y_size = dispchar.y_window_size;

    while (1) {
	if ((fd = accept(sfd, (struct sockaddr *) &saddr, &slen)) < 0) {
	    perror("accept");
	    exit(1);
	}

	i = sizeof claddr;
	getpeername(fd, (struct sockaddr *) &claddr, &i);
	time(&t);
	fprintf(logf, "%s C %s", inet_ntoa(claddr.sin_addr), ctime(&t));
	fflush(logf);

	f = fdopen(fd, "w");
	setbuffer(f, (char *) malloc(16 * 1024), 16 * 1024);

	proto_init(f);
	serve(f);
	time(&t);
	fprintf(logf, "%s D %s", inet_ntoa(claddr.sin_addr), ctime(&t));
	fflush(logf);
	fclose(f);
    }
}

void
proto_init(FILE *f)
{
    int n;
    CARD32 auth;
    rfbProtocolVersionMsg pvmsg;
    rfbClientInitMsg clmsg;
    rfbServerInitMsg msg;
    char hostname[MAXHOSTNAMELEN];

    fprintf(f, rfbProtocolVersionFormat, rfbProtocolMajorVersion, rfbProtocolMinorVersion);
    fflush(f);
    n = read(fileno(f), pvmsg, sz_rfbProtocolVersionMsg);
    pvmsg[n] = '\0';
    if (vflag)
	printf("client version %s", pvmsg);

    auth = htonl(rfbNoAuth);
    fwrite(&auth, sizeof auth, 1, f);

    netread(&clmsg, sz_rfbClientInitMsg, 1, f);

    msg.framebufferWidth = htons(dispchar.x_window_size);
    msg.framebufferHeight = htons(dispchar.y_window_size);
    msg.format.bitsPerPixel = 8;
    msg.format.depth = 8;
    msg.format.bigEndian = clbig;
    msg.format.trueColour = 1;
    msg.format.redMax = htons(7);
    msg.format.greenMax = htons(7);
    msg.format.blueMax = htons(3);
    msg.format.redShift = 0;
    msg.format.greenShift = 3;
    msg.format.blueShift = 6;
    msg.format.pad1 = 0;
    msg.format.pad2 = htons(0);
    gethostname(hostname, MAXHOSTNAMELEN);
    msg.nameLength = htonl(strlen(hostname));
    fwrite(&msg, sz_rfbServerInitMsg, 1, f);
    fputs(hostname, f);

    if (vflag)
	printf("proto_init done\n");
}

void
serve(FILE *f)
{
    int i;
    rfbClientToServerMsg msg;
    CARD32 enc;
    char *cp;

    while (netread(&msg.type, sizeof msg.type, 1, f) == 1) {
	if (vflag && msg.type != rfbPointerEvent)
	    printf("client message %s\n", clmsgnames[msg.type]);
	switch(msg.type) {
	case rfbSetPixelFormat:
	    netread(&msg.type + 1, sz_rfbSetPixelFormatMsg - 1, 1, f);
	    clbig = msg.spf.format.bigEndian;
	    break;
	case rfbFixColourMapEntries:
	    netread(&msg.type + 1, sz_rfbFixColourMapEntriesMsg - 1, 1, f);
	    break;
	case rfbSetEncodings:
	    netread(&msg.type + 1, sz_rfbSetEncodingsMsg - 1, 1, f);
	    for (i = 0; i < ntohs(msg.se.nEncodings); i++) {
		netread(&enc, sizeof enc, 1, f);
		switch(ntohl(enc)) {
		case rfbEncodingCopyRect:
		    willCopyRect = 1;
		    break;
		case rfbEncodingRRE:
		    willRRE = 1;
		    break;
		}
	    }
	    break;
	case rfbFramebufferUpdateRequest:
	    netread(&msg.type + 1, sz_rfbFramebufferUpdateRequestMsg - 1, 1, f);
	    update(f, msg.fur.incremental, ntohs(msg.fur.x), ntohs(msg.fur.y), ntohs(msg.fur.w), ntohs(msg.fur.h));
	    break;
	case rfbKeyEvent:
	    netread(&msg.type + 1, sz_rfbKeyEventMsg - 1, 1, f);
	    sendkey(f, msg.ke.down, ntohl(msg.ke.key));
	    break;
	case rfbPointerEvent:
	    netread(&msg.type + 1, sz_rfbPointerEventMsg - 1, 1, f);
	    break;
	case rfbClientCutText:
	    netread(&msg.type + 1, sz_rfbClientCutTextMsg - 1, 1, f);
	    i = ntohl(msg.cct.length);
	    cp = (char *) malloc(i+1);
	    netread(cp, 1, i, f);
	    cp[i] = '\0';
	    client_cut(f, cp);
	    free(cp);
	    break;
	}
    }
}

void
update(FILE *f, int inc, pinteger x0, pinteger y0, pinteger w, pinteger h)
{
    int x, y, x1, y1, r, xstip = -1, ystip, ncpy = 0, nwht = 0, nblk = 0, nraw = 0;
    rfbFramebufferUpdateMsg umsg;
    u_char *p;
    static u_char *p0, *p1;
    short lw;
    status_$t st;

    if (vflag)
	printf("client wants %d %d %d %d %s\n", x0, y0, w, h, inc ? "inc" : "full");

    if (inc && !need_refresh)
	return;

    gpr_$init(gpr_$borrow_nc, padfd, bmsize, 7, &bmdesc, &st);
    if (st.all != status_$ok)
	ErrPrint("gpr_$init", st);
    gpr_$inq_bitmap(&bmdesc, &st);

    r = x0 % RS;
    x0 -= r;
    w = roundup(w + r, RS);
    x1 = MIN(x0 + w, bmsize.x_size);
    r = y0 % RS;
    y0 -= r;
    h = roundup(h + r, RS);
    y1 = MIN(y0 + h, bmsize.y_size);

    if (vflag)
	printf("rounding to %d %d %d %d\n", x0, y0, w, h);

    /* Put previous frame in p0, current frame in p1 */
    p = p0;
    p0 = p1;
    p1 = p;

    gpr_$inq_bitmap_pointer(bmdesc, (char **) &p, &lw, &st);
    if (st.all != status_$ok) {
	ErrPrint("gpr_$inq_bitmap_pointer", st);
	gpr_$terminate(false, &st);
	return;
    }
    lw *= 2;

    if (p1 == NULL)
	p1 = (u_char *) malloc(bmsize.y_size * lw);
    memmove(p1, p, bmsize.y_size * lw);
    gpr_$terminate(false, &st);

    umsg.type = rfbFramebufferUpdate;
    umsg.pad = 0;
    umsg.nRects = htons((y1 - y0) / RS * (x1 - x0) / RS);
    fwrite(&umsg, sizeof umsg, 1, f);

    for (y = y0; y < y1; y += RS) {
	for (x = x0; x < x1; x += RS) {
	    p = &p1[y * lw + x / 8];
	    if (willCopyRect && !rectcmp(p, lw, stipple, RS / 8)) {
		if (xstip < 0) {
		    /* no stipple sent yet; send one and remember where */
		    sendrhd(f, x, y, rfbEncodingRaw);
		    sendrect(f, p, lw);
		    xstip = x;
		    ystip = y;
		    nraw++;
		} else {
		    /* stipple already sent; copy it */
		    sendrhd(f, x, y, rfbEncodingCopyRect);
		    sendcopy(f, xstip, ystip);
		    ncpy++;
		}
	    } else if (willRRE && ispx(p, lw, 0x0000)) {
		sendrhd(f, x, y, rfbEncodingRRE);
		sendsolid(f, 0x00);
		nwht++;
	    } else if (willRRE && ispx(p, lw, 0xffff)) {
		sendrhd(f, x, y, rfbEncodingRRE);
		sendsolid(f, 0xff);
		nblk++;
	    } else {
		sendrhd(f, x, y, rfbEncodingRaw);
		sendrect(f, p, lw);
		nraw++;
	    }
	}
    }

    if (vflag)
	printf("%d copy, %d black, %d white, %d raw\n", ncpy, nblk, nwht, nraw);

    need_refresh = 0;
}

void
sendrhd(FILE *f, int x, int y, int enc)
{
    rfbFramebufferUpdateRectHeader rhd;

    rhd.r.x = htons(x);
    rhd.r.y = htons(y);
    rhd.r.w = htons(RS);
    rhd.r.h = htons(RS);
    rhd.encoding = htonl(enc);
    fwrite(&rhd, sizeof rhd, 1, f);
}

void
sendrect(FILE *f, u_char *p, int lw)
{
    int x, y;

    for (y = 0; y < RS; y++)
	for (x = 0; x < RS / 8; x++)
	    fwrite(monotab[p[y * lw + x]], 1, 8, f);
}

void
sendcopy(FILE *f, int x, int y)
{
    rfbCopyRect rcr;

    rcr.srcX = x;
    rcr.srcY = y;
    fwrite(&rcr, sz_rfbCopyRect, 1, f);
}

void
sendsolid(FILE *f, u_char px)
{
    struct {
	rfbRREHeader rre;
	CARD8 px;
    } h;

    h.rre.nSubrects = 0;
    h.px = 0xff ^ px;
    fwrite(&h, sz_rfbRREHeader + 1, 1, f);
}

/*
 * ispx and rectcomp have been optimized for RS=16.
 * They won't work for other values.
 */
int
ispx(u_char *p, int lw, pinteger px)
{
    int y;

    for (y = 0; y < RS; y++)
	if (*(pinteger *)(p + (y * lw)) != px)
	    return 0;
    return 1;
}

int
rectcmp(u_char *p0, int lw0, u_char *p1, int lw1)
{
    int y;

    if (((pinteger *) p0)[0] != ((pinteger *) p1)[0])
	return 1;
    for (y = 0; y < RS; y++)
	if (((pinteger *) p0)[y * lw0 / 2] != ((pinteger *) p1)[y * lw1 / 2])
	    return 1;
    return 0;
}

void
sendkey(FILE *f, int down, unsigned int key)
{
    int i;
    static int mods;
    char *cmd, escmd[8];
    status_$t st;

    if ((key & ~0xf) == 0xffe0) {
	/* modifiers */
	i = key & 0xf;
	if (i == 2)
	    i = 1;
	else if (i == 3)
	    i = 4;
	i <<= 16;
	if (down)
	    mods |= i;
	else
	    mods &= ~i;
	return;
    }
    if (!down)
	return;

    if (key >= ' ' && key <= '~' && (mods & ~0x10000) == 0) {
	if (key == '\'')
	    cmd = "es '@''";
	else {
	    sprintf(escmd, "es '%c'", key);
	    cmd = escmd;
	}
    } else {
	key |= mods;
	if (key == 0x4ff0d) {
	    /* ctl-ret */
	    exit(0);
	}
	for (i = 0; keydefs[i].keydef != NULL; i++)
	    if (keydefs[i].keysym == key)
		break;
	if (keydefs[i].keydef == NULL) {
	    if (vflag)
		printf("keysym 0x%05x\n", key);
	    return;
	}
	cmd = keydefs[i].keydef;
	if (vflag)
	    printf("keydef %s\n", cmd);
    }

    pad_$dm_cmd(padfd, cmd, strlen(cmd), &st);

    if (st.all != status_$ok)
	ErrPrint("pad_$dm_cmd", st);

    need_refresh = 1;
}

void
client_cut(FILE *f, char *s)
{
    FILE *pf;

    pf = fopen("`node_data/paste_buffers/default.txt", "w");
    if (pf == NULL)
	return;
    fputs(s, pf);
    fclose(pf);
}

int
netread(void *p, int elemsize, int numelem, FILE *f)
{
    int n, len = elemsize * numelem;
    u_char *cp = p;

    fflush(f);

    while (len > 0) {
	n = read(fileno(f), cp, len);
	if (n <= 0)
	    break;
	cp += n;
	len -= n;
    }
    return ((elemsize * numelem - len) / elemsize);
}

char *
ErrMsg(status_$t st)
{
	static char msg[120];
	short n;
	extern std_$call error_$get_string();

	error_$get_string(st, msg, (short) (sizeof msg - 1), n);
	msg[n] = '\0';
	return msg;
}

void
ErrPrint(char *s, status_$t st)
{
	char buf[200];

	sprintf(buf, "%s: %s\n", s, ErrMsg(st));
	write(2, buf, strlen(buf));
}
