/*
X paint program
Jim Rees, May, 1993
 */

#include <stdio.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Xatom.h>
#include <X11/Xaw/Box.h>
#include <X11/Xaw/Command.h>
#include <X11/Xaw/Paned.h>

#include "Draw.h"

#define PALETTEHEIGHT 32
#define MAXCELLS 512
#define GSCL 1024
#define GSCLD (GSCL - 1)

#define pixtocolor(px) (mappx ? mappx[(px)] : (px))
#define colortopix(cl) (pxmap ? pxmap[(cl)] : (cl))

static char appname[] = "xpaint";

static
String fallback_resources[] = {
    "xpaint.size:				640x480",
    "*foreground:				black",
    "*background:				white",
    "*showGrip:					False",
    "*skipAdjust:				True",
    NULL
};

typedef struct _AppResources {
    int		cells;
    int		bgp;
    String	size;
    String	fsize;
    Boolean	debug;
    Boolean	zpixmap;
} AppResources;

AppResources app_resources;

static
XtResource resources[] = {
    {"size", "Size", XtRString, sizeof(String),
     XtOffset(AppResources *, size), XtRImmediate, ""},
    {"fsize", "Size", XtRString, sizeof(String),
     XtOffset(AppResources *, fsize), XtRImmediate, ""},
    {"debug", "Debug", XtRBoolean, sizeof(Boolean),
     XtOffset(AppResources *, debug), XtRImmediate, False},
    {"cells", "Ncells", XtRInt, sizeof(int),
     XtOffset(AppResources *, cells), XtRImmediate, 0},
    {"bgp", "Bgp", XtRInt, sizeof(int),
     XtOffset(AppResources *, bgp), XtRImmediate, 0},
    {"zpixmap", "Zpixmap", XtRBoolean, sizeof(Boolean),
     XtOffset(AppResources *, zpixmap), XtRImmediate, True},
};

static
XrmOptionDescRec options[] = {
    {"-d", 	"debug",		XrmoptionNoArg,         "True"},
    {"-cells",	"cells",		XrmoptionSepArg,	"0"},
    {"-size",	"fsize",		XrmoptionSepArg,	NULL},
    {"-mono",	"bgp",			XrmoptionNoArg,		"1"},
    {"-gray",	"bgp",			XrmoptionNoArg,		"2"},
    {"-color",	"bgp",			XrmoptionNoArg,		"3"},
};

#define NSTIPPLES 5

static char s0_bits[] = {
   0x0f, 0x0f, 0x0f, 0x0f};
static char s1_bits[] = {
   0x0f, 0x0a, 0x0f, 0x05};
static char s2_bits[] = {
   0x05, 0x0a, 0x0a, 0x05};
/*   0x05, 0x0a, 0x05, 0x0a};*/
static char s3_bits[] = {
   0x05, 0x00, 0x0a, 0x00};
static char s4_bits[] = {
   0x00, 0x00, 0x00, 0x00};

struct stipplestruct {
    char *bits;
    Pixmap pxmap;
} stipples[] = {
    {s0_bits},
    {s1_bits},
    {s2_bits},
    {s3_bits},
    {s4_bits},
};

static char palettecursor_bits[] = {
   0xf0, 0x01, 0xfc, 0x07, 0x0e, 0x0e, 0x06, 0x0c, 0x03, 0x18, 0x03, 0x18,
   0x03, 0x18, 0x03, 0x18, 0x03, 0x18, 0x06, 0x0c, 0x0e, 0x0e, 0xfc, 0x07,
   0xf0, 0x01};
static char palettecursormask_bits[] = {
   0xf0, 0x01, 0xfc, 0x07, 0xfe, 0x0f, 0xfe, 0x0f, 0xff, 0x1f, 0xff, 0x1f,
   0xff, 0x1f, 0xff, 0x1f, 0xff, 0x1f, 0xfe, 0x0f, 0xfe, 0x0f, 0xfc, 0x07,
   0xf0, 0x01};

struct dotstruct {
	int n;
	Pixmap pxmap;
	Widget btn;
	Cursor cursor;
} dots[] = {
	{ 1},
	{ 3},
	{ 5},
	{ 7},
	{ 9},
	{13},
	{ 0, NULL},
};

int dflag, iflag;

XtAppContext app;
Display *dpy;
int screen;
XStandardColormap cmapinfo;
Colormap colormap;
XColor color, fcolor, bcolor;
Window win, palettewin;
GC gc, gc0, gc1;
char *filename;
int width, height, pbmwidth, pbmheight;
int ncells, rncells, stipn, bgp, dpyform;
struct dotstruct *dot;
Cursor palettecursor;
XImage *memimage, *pbmimage;
FILE *pbmf;
int ptype;
char plett;
short *mappx;
int *pxmap;
int exposed;

static void excb(), rscb(), percb(), incb(), pincb(), chdot();
static void Save(), Clear(), Quit(), Mem();
FILE *openpbm();
XImage *crimage();

extern char *malloc(), *realloc();
extern void abort();

main(ac, av)
int ac;
char *av[];
{
    int i, px;
    Widget top, paned, box, palette, draw;
    XGCValues gcv;
    Widget btn;
    int btnsize;
    struct dotstruct *d;
    char buf[8];
    XColor xcolor;

    top = XtVaAppInitialize(&app, appname, options, XtNumber(options), &ac, av, fallback_resources, NULL);
    XtGetApplicationResources(top, (XtPointer) &app_resources, resources, XtNumber(resources), NULL, 0);
    dpy = XtDisplay(top);
    screen = DefaultScreen(dpy);

    dflag = app_resources.debug;
    if (app_resources.size)
	XParseGeometry(app_resources.size, &px, &px, &width, &height);

    if (dflag) {
	XSynchronize(dpy, True);
	XSetErrorHandler(abort);
    }

    if (ac > 1 && av[1][0] != '-') {
	filename = av[1];
	pbmf = openpbm(filename);
    } else
	pbmf = NULL;

    if (app_resources.fsize && app_resources.fsize[0])
	XParseGeometry(app_resources.fsize, &px, &px, &width, &height);

    rncells = XDisplayCells(dpy, screen);
    if (app_resources.cells)
	ncells = app_resources.cells;
    if (!ncells || ncells > rncells)
	ncells = rncells;
    if (ncells > MAXCELLS)
	ncells = MAXCELLS;
    dpyform = (rncells > 2 && app_resources.zpixmap) ? ZPixmap : XYPixmap;

    /* need to find a better way of determining whether we're on a color display */
    if (app_resources.bgp)
	bgp = app_resources.bgp;
    else if (rncells == 2)
	bgp = 1;
    else if (rncells <= 16)
	bgp = 2;
    else
	bgp = 3;

    if (bgp > 1) {
	if (XGetStandardColormap(dpy, DefaultRootWindow(dpy),
	    &cmapinfo, (bgp == 2) ? XA_RGB_GRAY_MAP : XA_RGB_BEST_MAP))
	    colormap = cmapinfo.colormap;
	else
	    rollyourown();
    } else {
	mappx = (short *) malloc(rncells * sizeof (short));
	for (i = 0; i < rncells; i++)
	    mappx[i] = ncells - 1;
	mappx[BlackPixel(dpy, screen)] = 0;
    }

    paned = XtVaCreateManagedWidget("paned", panedWidgetClass, top,
	colormap ? XtNcolormap : NULL, colormap, NULL);
    box = XtCreateManagedWidget("buttonBox", boxWidgetClass, paned, NULL, 0);

    /* buttons */
    btn = XtCreateManagedWidget("quit", commandWidgetClass, box, NULL, 0);
    XtAddCallback(btn, XtNcallback, Quit, NULL);
#ifdef MEMBUTTON
    btn = XtCreateManagedWidget("mem", commandWidgetClass, box, NULL, 0);
    XtAddCallback(btn, XtNcallback, Mem, NULL);
#endif
    btn = XtCreateManagedWidget("save", commandWidgetClass, box, NULL, 0);
    XtAddCallback(btn, XtNcallback, Save, NULL);
    btn = XtCreateManagedWidget("clear", commandWidgetClass, box, NULL, 0);
    XtAddCallback(btn, XtNcallback, Clear, NULL);

    for (d = dots; d->n; d++) {
	sprintf(buf, "%d-%d", d - dots, d->n);
	d->btn = XtVaCreateManagedWidget(buf, commandWidgetClass, box,
					 XtNresize, False, NULL);
	XtAddCallback(d->btn, XtNcallback, chdot, d);
    }

    palette = XtVaCreateManagedWidget("palette", drawWidgetClass, paned,
	XtNwidth, width, XtNheight, PALETTEHEIGHT, NULL);
    XtAddCallback(palette, XtNexpose, percb, NULL);
    XtAddCallback(palette, XtNcallback, pincb, NULL);

    draw = XtVaCreateManagedWidget("draw", drawWidgetClass, paned,
	XtNwidth, width, XtNheight, height, NULL);
    XtAddCallback(draw, XtNexpose, excb, NULL);
    XtAddCallback(draw, XtNresize, rscb, NULL);
    XtAddCallback(draw, XtNcallback, incb, NULL);

    XtRealizeWidget(top);

    win = XtWindow(draw);
    palettewin = XtWindow(palette);

    XtVaGetValues(draw, XtNcolormap, &colormap, 0);
    XtVaGetValues(btn, XtNheight, &btnsize, 0);

    color.pixel = fcolor.pixel = BlackPixel(dpy, screen);
    bcolor.pixel = WhitePixel(dpy, screen);
    XQueryColor(dpy, colormap, &fcolor);
    XQueryColor(dpy, colormap, &bcolor);

    gcv.foreground = fcolor.pixel;
    gcv.background = bcolor.pixel;
    gc0 = XCreateGC(dpy, win, (GCForeground | GCBackground), &gcv);

    gcv.foreground = fcolor.pixel;
    gcv.background = bcolor.pixel;
    gc = XCreateGC(dpy, win, (GCForeground | GCBackground), &gcv);

    /* I hate X */
    gcv.foreground = 1;
    gcv.background = 0;

    for (d = dots; d->n; d++) {
	d->pxmap = XCreatePixmap(dpy, win, d->n, d->n, 1);
	if (gc1 == NULL)
	    gc1 = XCreateGC(dpy, d->pxmap, (GCForeground | GCBackground), &gcv);
	if (d->n == 1)
	    XFillRectangle(dpy, d->pxmap, gc1, 0, 0, 1, 1);
	else {
	    XSetForeground(dpy, gc1, 0);
	    XFillRectangle(dpy, d->pxmap, gc1, 0, 0, d->n, d->n);
	    XSetForeground(dpy, gc1, 1);
	    XFillArc(dpy, d->pxmap, gc1, 0, 0, d->n, d->n, 0, 23040);
	}
	XtVaSetValues(d->btn, XtNbitmap, d->pxmap, NULL);
	d->cursor = XCreatePixmapCursor(dpy, d->pxmap, d->pxmap,
	  &fcolor, &bcolor, d->n / 2, d->n / 2);
    }

    if (bgp == 1) {
	for (i = 0; i < NSTIPPLES; i++)
	    stipples[i].pxmap = XCreateBitmapFromData(dpy, win, stipples[i].bits, 4, 4);
	XSetFillStyle(dpy, gc, FillOpaqueStippled);
	stipn = 0;
    }

    palettecursor = XCreatePixmapCursor(dpy,
	XCreateBitmapFromData(dpy, palettewin, palettecursor_bits, 13, 13),
	XCreateBitmapFromData(dpy, palettewin, palettecursormask_bits, 13, 13),
	&fcolor, &bcolor, 13 / 2, 13 / 2);
    XDefineCursor(dpy, palettewin, palettecursor);

    chdot(draw, &dots[3], NULL);

    XtAppMainLoop(app);
}

static void
incb(w, a, ev)
Widget w;
XtPointer a;
XEvent *ev;
{
    static int down;
    int i;
    char buf[4];
    XComposeStatus compstatus;

    switch(ev->xany.type) {
    case ButtonRelease:
	switch(ev->xbutton.button) {
	case Button3:
	    XSetForeground(dpy, gc, color.pixel);
	case Button1:
#ifndef MEMBUTTON
	    Mem();
#endif
	    break;
	}
	down = 0;
	break;
    case ButtonPress:
	switch(ev->xbutton.button) {
	case Button1:
	    down = 1;
	    break;
	case Button2:
	    if (bgp > 1 && memimage) {
		color.pixel = XGetPixel(memimage, ev->xmotion.x, ev->xmotion.y);
		XSetForeground(dpy, gc, color.pixel);
	    } else
		XBell(dpy, 100);
	    break;
	case Button3:
	    XSetForeground(dpy, gc, bcolor.pixel);
	    down = 3;
	    break;
	}
    case MotionNotify:
	if (down) {
	    if (ev->xmotion.x < 0 || ev->xmotion.x >= width)
		return;
	    if (dot->n == 1)
		XFillRectangle(dpy, win, gc, ev->xmotion.x, ev->xmotion.y, 1, 1);
	    else
		XFillArc(dpy, win, gc, ev->xmotion.x - dot->n / 2, ev->xmotion.y - dot->n / 2, dot->n, dot->n, 0, 23040);
	}
	break;
    case KeyPress:
	i = XLookupString((XKeyEvent *) ev, buf, sizeof buf, NULL, &compstatus);
	buf[i] = '\0';
	switch (buf[0]) {
	case 'q':
	    Quit();
	    break;
	case 'e':
	    XClearWindow(dpy, win);
	    break;
	}
	break;
    }
}

static void
rscb(w, a)
Widget w;
XtPointer a;
{
    XWindowAttributes xwa;

    if (!exposed)
	return;
    XGetWindowAttributes(dpy, win, &xwa);
    width = xwa.width;
    height = xwa.height;
}

static void
excb(w, a)
Widget w;
XtPointer a;
{
    if (!exposed++)
	rscb();
    if (pbmf != NULL) {
	readpbm(pbmf);
	fclose(pbmf);
	pbmf = NULL;
    }
    if (memimage == NULL)
	return;
    XPutImage(dpy, win, gc0, memimage, 0, 0, 0, 0, width, height);
}

static void
pincb(w, a, ev)
Widget w;
XtPointer a;
XEvent *ev;
{
    static int down = 0;

    switch(ev->xany.type) {
    case ButtonRelease:
	switch(ev->xbutton.button) {
	case Button2:
	    if (bgp > 1) {
		recolor(ev->xmotion.x);
		XSetForeground(dpy, gc, color.pixel);
	    } else {
		stipn = ev->xmotion.x * NSTIPPLES / width;
		XSetStipple(dpy, gc, stipples[stipn].pxmap);
	    }
	    break;
	}
	down = 0;
	break;
    case ButtonPress:
	down = 1;
    case MotionNotify:
	if (down && bgp > 1) {
	    recolor(ev->xmotion.x);
	    XQueryColor(dpy, colormap, &color);
	    XRecolorCursor(dpy, palettecursor, &fcolor, &color);
	}
	break;
    }
}

recolor(x)
int x;
{
    int pix;

    pix = x * ncells / width;
    color.pixel = colortopix(pix);
}

static void
percb(w, a)
Widget w;
XtPointer a;
{
    int i, pwid;

    if (bgp > 1) {
	pwid = (width + ncells - 1) / ncells;
	for (i = 0; i < ncells; i++) {
	    XSetForeground(dpy, gc, colortopix(i));
	    XFillRectangle(dpy, palettewin, gc, i * width / ncells, 0, pwid, PALETTEHEIGHT);
	}
	XSetForeground(dpy, gc, color.pixel);
    } else {
	pwid = (width + NSTIPPLES - 1) / NSTIPPLES;
	for (i = 0; i < NSTIPPLES; i++) {
	    XSetStipple(dpy, gc, stipples[i].pxmap);
	    XFillRectangle(dpy, palettewin, gc, i * width / NSTIPPLES, 0, pwid, PALETTEHEIGHT);
	}
	XSetStipple(dpy, gc, stipples[stipn].pxmap);
    }
}

static void
chdot(w, d, a)
Widget w;
struct dotstruct *d;
char *a;
{
    dot = d;
    if (d->cursor)
	XDefineCursor(dpy, win, d->cursor);
}

FILE *
openpbm(name)
char *name;
{
    FILE *fp;
    char buf[80];
    static char pbmletts[] = {'\0', 'b', 'g', 'p', 'b', 'g', 'p'};
    static char pbmok[] = {0, 1, 0, 0, 1, 1, 1};

    fp = fopen(name, "r");
    if (fp == NULL)
	return NULL;
    if (fgets(buf, sizeof buf, fp) == NULL || buf[0] != 'P') {
	fclose(fp);
	return NULL;
    }
    ptype = buf[1] - '0';
    if (ptype < 1 || ptype > 6 || !pbmok[ptype]) {
	fclose(fp);
	return NULL;
    }

    plett = pbmletts[ptype];

    buf[0] = '#';
    while (buf[0] == '#') {
	if (fgets(buf, sizeof buf, fp) == NULL) {
	    fclose(fp);
	    return NULL;
	}
    }

    sscanf(buf, "%d %d", &pbmwidth, &pbmheight);
    width = (pbmwidth > 192) ? pbmwidth : 192;
    height = pbmheight;

    return fp;
}

readpbm(fp)
FILE *fp;
{
    char buf[80];
    char *bm;
    int bmsize;
    XImage *ip;

    if (plett == 'g')
	readpgm(fp);
    else if (plett == 'p') {
	if (cmapinfo.colormap)
	    readppm(fp);
	else
	    readppmg(fp);
    }
    if (plett != 'b')
	return;

    ip = XtNew(XImage);
    bzero(ip, sizeof (XImage));
    ip->bytes_per_line = (pbmwidth + 7) / 8;
    bmsize = ip->bytes_per_line * pbmheight;
    bm = malloc(bmsize);

    if (ptype == 1)
	cpascii(fp, bm, bmsize);
    else if (ptype == 4)
	fread(bm, 1, bmsize, fp);
    else {
	free(bm);
	return;
    }

    ip->width = pbmwidth;
    ip->height = pbmheight;
    ip->xoffset = 0;
    ip->format = XYBitmap;
    ip->depth = 1;
    ip->data = bm;
    ip->byte_order = MSBFirst;
    ip->bitmap_unit = 8;
    ip->bitmap_bit_order = MSBFirst;
    ip->bitmap_pad = 8;

    memimage = pbmimage = ip;
}

XImage *
crimage()
{
    char *bp;

/*
    return XGetImage(dpy, win, 0, 0, width, height, AllPlanes, ZPixmap);
*/
    bp = (char *) malloc(width * height * (DisplayPlanes(dpy, screen) + 7) / 8);
    return XCreateImage(dpy, DefaultVisual(dpy, screen),
	DisplayPlanes(dpy, screen), dpyform, 0, bp, width, height, 32, 0);
}

readpgm(fp)
FILE *fp;
{
    XImage *ip;
    int npix, x, y;

    fscanf(fp, "%d", &npix);
    getc(fp);

    ip = crimage();

    for (y = 0; y < pbmheight; y++) {
	for (x = 0; x < pbmwidth; x++)
	    XPutPixel(ip, x, y, graytopix(getc(fp) * GSCLD / npix));
    }
    memimage = ip;
}

/* See X Window System, "Client Conventions," no. 14 */

readppm(fp)
FILE *fp;
{
    XImage *ip;
    int npix, x, y, r, g, b;

    fscanf(fp, "%d", &npix);
    getc(fp);

    ip = crimage();

    for (y = 0; y < pbmheight; y++) {
	for (x = 0; x < pbmwidth; x++) {
	    r = getc(fp) * cmapinfo.red_max / npix;
	    g = getc(fp) * cmapinfo.green_max / npix;
	    b = getc(fp) * cmapinfo.blue_max / npix;
	    XPutPixel(ip, x, y, colortopix(cmapinfo.base_pixel
		      + (r * cmapinfo.red_mult)
		      + (g * cmapinfo.green_mult)
		      + (b * cmapinfo.blue_mult)));
	}
    }
    memimage = ip;
}

readppmg(fp)
FILE *fp;
{
    XImage *ip;
    int npix, x, y;

    fscanf(fp, "%d", &npix);

    ip = crimage();

    for (y = 0; y < height; y++) {
	for (x = 0; x < width; x++)
	    XPutPixel(ip, x, y, graytopix(getc(fp) + getc(fp) + getc(fp) * GSCLD / npix / 3));
    }
    memimage = ip;
}

/* bug -- I don't think this works right if width % 8 != 0 */

cpascii(fp, bm, bmn)
FILE *fp;
char *bm;
int bmn;
{
    int px, n = 0, bit;

    while (fscanf(fp, "%d", &bit) == 1 && bmn) {
	if (n % 8 == 0) {
	    px = 0;
	    n = 0;
	}
	if (bit)
	    px |= 1 << (7 - n);
	if (n++ % 8 == 7) {
	    *bm++ = px;
	    bmn--;
	}
    }
}

static void
Save()
{
    FILE *fp;
    XImage *ip;
    int n, x, y, px, bitno;
    char plett, buf[100];
    static XColor *cm;

    if (bgp == 1) {
	ptype = 4;
	plett = 'b';
    } else if (bgp == 2) {
	ptype = 5;
	plett = 'g';
    } else {
	ptype = 6;
	plett = 'p';
	if (!cm) {
	    cm = (XColor *) malloc(rncells * sizeof (XColor));
	    for (px = 0; px < rncells; px++) {
		cm[px].pixel = px;
		XQueryColor(dpy, colormap, &cm[px]);
	    }
	}
    }

    if (filename == NULL) {
	sprintf(buf, "xpaint.p%cm", plett);
	filename = XtNewString(buf);
    } else {
	n = strlen(filename);
	if (n > 4
	    && filename[n-4] == '.'
	    && filename[n-3] == 'p'
	    && filename[n-2] != plett
	    && filename[n-1] == 'm') {
	    filename = XtNewString(filename);
	    filename[n-2] = plett;
	}
    }

    fp = fopen(filename, "w");
    if (fp == NULL)
	return;

    fprintf(fp, "P%1d\n", ptype);
    fprintf(fp, "# produced by xpaint\n");
    fprintf(fp, "%d %d\n", width, height);
    if (ptype == 5 || ptype == 6)
	fprintf(fp, "255\n");

    Mem();
    ip = memimage;

    for (y = 0; y < height; y++) {
	for (x = bitno = 0; x < width; x++) {
	    if (ptype == 4) {
		if (bitno % 8 == 0)
		    px = bitno = 0;
		if (pixtocolor(XGetPixel(ip, x, y)) < ncells / 2)
		    px |= 1 << (7 - bitno);
		if (bitno++ % 8 == 7)
		    putc(px, fp);
	    } else if (ptype == 5)
		putc(pixtocolor(XGetPixel(ip, x, y)) * 255 / (ncells - 1), fp);
	    else if (ptype == 6) {
		putc(cm[XGetPixel(ip, x, y)].red * 255 / 65535, fp);
		putc(cm[XGetPixel(ip, x, y)].green * 255 / 65535, fp);
		putc(cm[XGetPixel(ip, x, y)].blue * 255 / 65535, fp);
	    }
	}
	if (bitno % 8 != 0)
	    putc(px, fp);
    }
    fclose(fp);
}

static void
Mem()
{
    if (memimage && memimage != pbmimage)
	XDestroyImage(memimage);
    memimage = XGetImage(dpy, win, 0, 0, width, height, AllPlanes, dpyform);
}

static void
Clear()
{
    XFillRectangle(dpy, win, gc, 0, 0, width, height);
}

static void
Quit()
{
    XtDestroyApplicationContext(app);
    exit(0);
}

graytopix(gr)
int gr;
{
    int px;

    if (cmapinfo.colormap) {
	if (bgp < 3)
	    px = gr * cmapinfo.red_max / GSCLD * cmapinfo.red_mult;
	else
	    px = gr * cmapinfo.red_max / GSCLD * cmapinfo.red_mult
	       + gr * cmapinfo.green_max / GSCLD * cmapinfo.green_mult
	       + gr * cmapinfo.blue_max / GSCLD * cmapinfo.blue_mult
	       + cmapinfo.base_pixel;
	px = colortopix(px);
    } else if (gr >= GSCL / 2)
	px = bcolor.pixel;
    else
	px = fcolor.pixel;
    return px;
}

rollyourown()
{
    int i;

    colormap = DefaultColormap(dpy, screen);
    pxmap = (int *) malloc(ncells * sizeof (int));
    mappx = (short *) malloc(rncells * sizeof (short));

    if (bgp == 2)
	rollgray();
    else
	rollcolor();

    cmapinfo.colormap = colormap;
    cmapinfo.base_pixel = 0;

    for (i = 0; i < rncells; i++)
	mappx[i] = ncells - 1;
    for (i = 0; i < ncells; i++)
	mappx[pxmap[i]] = i;
}

rollgray()
{
    int i, opx;
    XColor c;

    /* go white to black, cause white end is more interesting */
    for (i = ncells - 1; i >= 0; i--) {
	c.red = c.green = c.blue = i * 65535 / (ncells - 1);
	pxmap[i] = XAllocColor(dpy, colormap, &c) ? (opx = c.pixel) : opx;
    }

    cmapinfo.red_max = ncells - 1;
    cmapinfo.red_mult = 1;
}

rollcolor()
{
    int np, r, g, b, nvr, nvg, nvb, opx;
    XColor c;

    if (ncells < 8) {
	rollgray();
	return;
    }

    /* find number of planes to use */
    for (np = 1; np <= 8; np++)
	if (1 << np > ncells)
	    break;
    np--;

    /* find number of planes per color */
    nvg = (np + 2) / 3;
    np -= nvg;
    nvr = (np + 1) / 2;
    np -= nvr;
    nvb = np;

    /* find number of values per color */
    nvr = 1 << nvr;
    nvg = 1 << nvg;
    nvb = 1 << nvb;

    /* allocate color cells */
    for (b = nvb - 1; b >= 0; b--)
	for (r = nvr - 1; r >= 0; r--)
	    for (g = nvg - 1; g >= 0; g--) {
		c.red = r * 65535 / (nvr - 1);
		c.green = g * 65535 / (nvg - 1);
		c.blue = b * 65535 / (nvb - 1);
		pxmap[g * nvr * nvb + r * nvb + b] =
		  XAllocColor(dpy, colormap, &c) ? (opx = c.pixel) : opx;
	    }

    ncells = nvr * nvg * nvb;
    cmapinfo.red_max = nvr - 1;
    cmapinfo.green_max = nvg - 1;
    cmapinfo.blue_max = nvb - 1;
    cmapinfo.green_mult = nvr * nvb;
    cmapinfo.red_mult = nvb;
    cmapinfo.blue_mult = 1;
}
