/*
 * x-a-sketch
 *
 * -i inverted video
 * -r realtime (flush output frequently)
 * -o outfilename
 *
 *   Written by Jim Rees April 1993 adapted from x-a-sketch,
 *   Written by Jim Rees February 1991
 */

#include <stdio.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>

#include "Draw.h"

static char appname[] = "x-a-sketch";

static
String fallback_resources[] = {
    "*geometry:					800x600+20+20",
    "*foreground:				black",
    "*background:				white",
    NULL
};

typedef struct _AppResources {
    Boolean	invert;
    Boolean	realtime;
    String	outfile;
} AppResources;

AppResources app_resources;

static
XtResource resources[] = {
    {"invert", "Invert", XtRBoolean, sizeof(Boolean),
     XtOffset(AppResources *, invert), XtRImmediate, False},
    {"realtime", "Realtime", XtRBoolean, sizeof(Boolean),
     XtOffset(AppResources *, realtime), XtRImmediate, False},
    {"outfile", "Outfile", XtRString, sizeof(String),
     XtOffset(AppResources *, outfile), XtRImmediate, ""},
};

static
XrmOptionDescRec options[] = {
    {"-i", 	"invert",		XrmoptionNoArg,         "True"},
    {"-r", 	"realtime",		XrmoptionNoArg,         "True"},
    {"-o",	"outfile",		XrmoptionSepArg,	NULL},
};

static struct {
	char c;
	char *name;
	int style;
	char *dash;
} styletab[] = {
	'.',	"dotted",	LineOnOffDash,	"\1\3",
	'_',	"solid",	LineSolid,	"\1\1",
	'-',	"longdashed",	LineOnOffDash,	"\10\10",
	',',	"shortdashed",	LineOnOffDash,	"\4\4",
	';',	"dotdashed",	LineOnOffDash,	"\1\3\4\3",
	'\0',	NULL,		0,		NULL
};

struct storedlines {
	short x0, y0, x1, y1;
} *stlines;

int rflag;
FILE *outf;
XtAppContext app;
Display *dpy;
Window win;
GC gc;
Widget top, draw;
XWindowAttributes xwa;
int wsize;
int nstoredlines, stlinesize;

static void ercb(), incb();

extern char *malloc(), *realloc();

main(ac, av)
int ac;
char *av[];
{
	XGCValues gcv;
	XColor xc;

	top = XtVaAppInitialize(&app, appname, options, XtNumber(options), &ac, av, fallback_resources, NULL);
	XtGetApplicationResources(top, (XtPointer) &app_resources, resources, XtNumber(resources), NULL, 0);
	dpy = XtDisplay(top);

	rflag = app_resources.realtime;

	if (app_resources.outfile && app_resources.outfile[0]) {
		if (!strcmp(app_resources.outfile, "-")) {
			outf = stdout;
			rflag = 1;
		} else {
			outf = fopen(app_resources.outfile, "w");
			if (outf == NULL) {
				fprintf(stderr, "can't open %s for output\n", app_resources.outfile);
				XtDestroyApplicationContext(app);
				exit(1);
			}
		}
		setbuf(outf, malloc(BUFSIZ));
	} else
		outf = NULL;

	stlinesize = 1024;
	stlines = (struct storedlines *) malloc(stlinesize * sizeof (struct storedlines));

	draw = XtCreateManagedWidget("draw", drawWidgetClass, top, NULL, 0);
	XtAddCallback(draw, XtNexpose, ercb, NULL);
	XtAddCallback(draw, XtNcallback, incb, NULL);

	XtRealizeWidget(top);

	win = XtWindow(draw);

	if (app_resources.invert) {
#ifdef BROKEN
		/* I can't figure out why this doesn't work */
		XtVaGetValues(draw, XtNbackground, &gcv.foreground, 0);
		XtVaGetValues(draw, XtNforeground, &gcv.background, 0);
#else
		gcv.foreground = WhitePixel(dpy, DefaultScreen(dpy));
		gcv.background = BlackPixel(dpy, DefaultScreen(dpy));
#endif
		XtVaSetValues(draw, XtNforeground, gcv.foreground, 0);
		XtVaSetValues(draw, XtNbackground, gcv.background, 0);
	} else {
#ifdef BROKEN
		XtVaGetValues(draw, XtNforeground, &gcv.foreground, 0);
		XtVaGetValues(draw, XtNbackground, &gcv.background, 0);
#else
		gcv.foreground = BlackPixel(dpy, DefaultScreen(dpy));
		gcv.background = WhitePixel(dpy, DefaultScreen(dpy));
#endif
	}
	XClearWindow(dpy, win);
/*
printf("fore %d back %d black %d white %d\n", gcv.foreground, gcv.background,
 BlackPixel(dpy, DefaultScreen(dpy)), WhitePixel(dpy, DefaultScreen(dpy)));
*/
	gc = XCreateGC(dpy, win, (GCForeground | GCBackground), &gcv);

	XGetWindowAttributes(dpy, win, &xwa);
	wsize = (xwa.width > xwa.height) ? xwa.width : xwa.height;

	putb('e');
	putb('s');
	putpt(0, 0);
	putpt(wsize, wsize);

	XtAppMainLoop(app);
}

static void
incb(w, a, ev)
Widget w;
XtPointer a;
XEvent *ev;
{
	static int down, x0, y0;
	int i;
	char buf[4];
	XComposeStatus compstatus;

	switch(ev->xany.type) {
	case ButtonPress:
		putb('m');
		putpt(x0, xwa.height - y0);
		down = 1;
		break;
	case ButtonRelease:
		down = 0;
		break;
	case MotionNotify:
		if (down) {
			XDrawLine(dpy, win, gc, x0, y0, ev->xmotion.x, ev->xmotion.y);
			addline(x0, y0, ev->xmotion.x, ev->xmotion.y);
			putb('n');
			putpt(ev->xmotion.x, xwa.height - ev->xmotion.y);
			if (rflag && outf)
				fflush(outf);
		}
		x0 = ev->xmotion.x;
		y0 = ev->xmotion.y;
		break;
	case KeyPress:
		i = XLookupString((XKeyEvent *) ev, buf, sizeof buf, NULL, &compstatus);
		buf[i] = '\0';
		switch (buf[0]) {
		case 'q':
			XtDestroyApplicationContext(app);
			exit(0);
			break;
		case 'e':
			XClearWindow(dpy, win);
			nstoredlines = 0;
			putb('e');
			if (outf)
				fflush(outf);
			break;
		default:
			setstyle(buf[0]);
			break;
		}
		break;
	}
}

setstyle(c)
char c;
{
	int i;

	for (i = 0; styletab[i].name != NULL; i++)
		if (styletab[i].c == c)
			break;
	if (styletab[i].name == NULL)
		return;

	XSetLineAttributes(dpy, gc, 0, styletab[i].style, CapButt, JoinMiter);
	XSetDashes(dpy, gc, 0, styletab[i].dash, strlen(styletab[i].dash));
	if (outf)
		fprintf(outf, "f%s\n", styletab[i].name);
}

addline(x0, y0, x1, y1)
int x0, y0, x1, y1;
{
	/* This doesn't store line styles. */
	if (nstoredlines >= stlinesize) {
		stlinesize *= 2;
		stlines = (struct storedlines *) realloc(stlines, stlinesize * sizeof (struct storedlines));
		if (stlines == NULL) {
			fprintf(stderr, "out of memory\n");
			exit(2);
		}
	}
	stlines[nstoredlines].x0 = x0;
	stlines[nstoredlines].y0 = y0;
	stlines[nstoredlines].x1 = x1;
	stlines[nstoredlines].y1 = y1;
	nstoredlines++;
}

static void
ercb(w, a)
Widget w;
XtPointer a;
{
	int i;

	for (i = 0; i < nstoredlines; i++)
		XDrawLine(dpy, win, gc, stlines[i].x0, stlines[i].y0, stlines[i].x1, stlines[i].y1);
}

static
putb(c)
int c;
{
	if (outf)
		putc(c, outf);
}

static
putnum(x)
short x;
{
	if (outf) {
		putc(x, outf);
		putc(x >> 8, outf);
	}
}

static
putpt(x, y)
short x, y;
{
	putnum(x);
	putnum(y);
}
