#! /bin/sh

#
# Conventions:
#
# Non-exported variables start with lower case characters, and are mixed case
# Exported variables are all upper case characters
# Function names start with upper case characters, and are mixed case
# Functions are in alphabetic order of the names
# Functions have only one entry and exit point (no returns in the middle)
#

exitvalue=0
prog=`echo $0 | sed -e 's,.*/,,g'`

tmpdir=/tmp
sed_script=
nawk_script=
tmpfile=
viewprint_args=""
outfile=""

#
# Is /tmp writeable? If not then check /var/tmp.
#
CheckTmp()
{
	retvalue=0

	if [ ! \( -d $tmpdir -a -w $tmpdir \) ]; then
		tmpdir=/var/tmp
		if [ ! \( -d $tmpdir -a -w $tmpdir \) ]; then
			echo "$prog: neither /tmp nor /var/tmp " \
				"are writeable directories" >&2
			retvalue=1
		fi
	fi

	if [ $retvalue -eq 0 ]; then
		sed_script=$tmpdir/$prog.$$.sed_script
		nawk_script=$tmpdir/$prog.$$.nawk_script
		tmpfile=$tmpdir/$prog.$$.vp
	fi

	return $retvalue
}

CleanUp()
{
	rm -f $tmpdir/$prog.$$.*
}

CreateScripts()
{
#
# Write the sed script to a file
#
	cat > $sed_script <<-__END_OF_sed_script__
	/^%!/a\\
	save %foo
	#
	/0 0 null SS/c\\
	statusdict /product get (NeWS Server) eq\\
	{\\
		0 0 null SS\\
	}{\\
		60.0 45.0 { pop pop 0.0 } SS\\
	} ifelse
	#
	/^end %PROLOGUE/a\\
	restore %foo
	#
	#eof
	__END_OF_sed_script__

#
# Write the awk script to a file
#
	cat > $nawk_script <<-__END_OF_nawk_script__
	{
		# eat stuff
		if (\$1 == "%%PageBoundingBox:" && \$2 != "(atend)") {
			print \$0

			do {
				getline
			} while (\$1 != "%%Page:" && \$0 != "%%Trailer")

			print \$0
		}
		else {	# else print it
			print \$0
		}
 
	}
	__END_OF_nawk_script__

	return 0
}

PrintIt()
{
	retvalue=0

#
# Before running the script move the real viewprint to viewprint.bin
# so we can call it here
#
	viewprint.bin -f$tmpfile $viewprint_args ||
	{
		echo "$prog: error executing viewprint.bin" >&2
		retvalue=1
	}

	if [ $retvalue -eq 0 ]; then
#
# If outfile is set then output to a file. If not then we are sending
# to a printer.
#
		if [ "x$outfile" = "x" ]; then
			sed -f $sed_script < $tmpfile | \
				nawk -f $nawk_script | lpr -P$printer ||
			{
				retvalue=1
			}
		else
			sed -f $sed_script < $tmpfile | \
				nawk -f $nawk_script > $outfile ||
			{
				retvalue=1
			}
		fi
	fi

	return $retvalue
}

#
# Process the arguments
#
while [ $# -ne 0 ]; do 
	case $1 in 
	-P*)
		if [ $1 = "-P" ]; then
			shift
			printer=$1
		else
			printer=`echo $1 | sed -e 's;\-P;;'`
		fi
		;;
	-f*)
		if [ "X$1" = "X-f" ]; then
			shift
			outfile=$1
		else
			outfile=`echo $1 | sed -e 's;\-f;;'`
		fi
		;;
	*)
		viewprint_args="$viewprint_args $1"
		;;
	esac
	shift
done

trap "echo $prog: received signal ... cleaning up; CleanUp; exit 1" \
	1 2 3 15

CheckTmp && CreateScripts && PrintIt ||
{
	exitvalue=1
}

CleanUp

exit $exitvalue
