(* Ulm's Oberon Compiler
   Copyright (c) 1989 by University of Ulm, SAI, D-W-7900 Ulm, Germany
   ----------------------------------------------------------------------------
   $Id: EmitCode.m2,v 0.8 1994/03/17 10:41:29 borchert Exp $
   ----------------------------------------------------------------------------
   $Log: EmitCode.m2,v $
   Revision 0.8  1994/03/17  10:41:29  borchert
   gas doesn't accept @d1 but @(d1:l) despite to Motorolas notification
   (d1) which isn't that ambiguous; for this reason we require the
   register of addrMode to be an address register

   Revision 0.7  1993/10/03  14:57:40  borchert
   support of rlabel component of Attribute
   string are now padded to oneword-boundaries

   Revision 0.6  1993/09/27  12:43:03  borchert
   EmitAlign4 added

   Revision 0.5  1993/06/18  15:33:00  borchert
   several instances of INTEGER and CARDINAL changed to
   LONGINT and LONGCARD for better compatibility

   Revision 0.4  1993/06/16  09:47:01  borchert
   atsize 2 is now legal because of SYSTEM.INT16

   Revision 0.3  1993/04/13  15:11:07  borchert
   %:i and %t added
   .stabs parameter fixed

   Revision 0.2  1993/02/03  12:33:09  borchert
   + and - are now accepted in format strings
   bug in Advance fixed
   nilvalue now imported from Machine
   :D added
   generation of module records moved to GenBlocks

   Revision 0.1  1992/07/30  10:48:06  borchert
   Initial revision

   ----------------------------------------------------------------------------
*)

IMPLEMENTATION MODULE EmitCode; (* AFB 7/90 *)

   (* the only assembly output generating module;
      implementation for GAS (GNU assembler, Sun3)
      supported and tested gas-versions: 1.36

      no code is generated if Scan.errorflag is TRUE
   *)

   FROM ASCII IMPORT tab;
   FROM Attributes IMPORT Label;
   FROM Exception IMPORT Assert;
   FROM IdentSys IMPORT GetIdentChar;
   FROM Lex IMPORT String, GetStringNumber, GetStringChar;
   FROM Machine IMPORT oneword;
   FROM Mnemonics IMPORT Mnemonic;
   FROM SymTab IMPORT Ident, IdentClass;
   FROM SYSTEM IMPORT BYTE;
   IMPORT ASCII, Attributes, Conversions, Exception, IdentSys, Lex, Machine,
      Memory, Mnemonics, Scan, ScanArgs, StdIO, SymTab, SysExit, SystemTypes,
      SYSTEM;

   (* (* exported from definition module *)
   TYPE
      Segment = (text, data);
   *)

   CONST
      gasversion = 136; (* 1.36 *)

   VAR
      currentSegment: Segment;
      intendedSegment: Segment;
      bfline: CARDINAL; (* line of last BEGIN *)

   MODULE Output;

      FROM ASCII IMPORT nl;
      FROM Conversions IMPORT ConvertInteger, ConvertCardinal, ConvertHex,
	 ConvertOctal;
      FROM Exception IMPORT Assert, IOFault;
      FROM Scan IMPORT errorflag, outputcount;
      FROM ScanArgs IMPORT output;
      FROM StdIO IMPORT Fopen, FILE, write, Fputc, Fclose, Fseek, Ftell; 
      FROM SysExit IMPORT EnterCleanup;
      FROM SystemTypes IMPORT OFF;

      EXPORT InitOutput, WriteString, WriteLn, Write, WriteCard, WriteInt,
	 WriteLongCard, WriteLongInt, WriteHex, WriteOct3, SetFixupPos, Fixup,
	 OpenBuffer, CloseBuffer, ReleaseBuffer,
	 SuppressOutput, ContinueOutput;

      CONST
	 bufsize = 512;

      VAR
	 out: FILE;
	 open: BOOLEAN; (* output file opened *)
	 fixup: BOOLEAN;
	 fixupPos: OFF;	(* fixup position *)
	 currPos: OFF;	(* current position *)

	 (* buffering system *)
	 buf: ARRAY [0..bufsize-1] OF CHAR;
	 intobuffer: BOOLEAN; (* write into buffer instead of out *)
	 bufindex: CARDINAL;

	 suppressOutput: BOOLEAN;

      PROCEDURE InitOutput;
      BEGIN
	 Assert(~open);
	 IF (output[0] = 0C) OR errorflag THEN
	    (* no output is to be generated *)
	 ELSIF ~Fopen(out, output, write, (* buffered = *) TRUE) THEN
	    IOFault(output);
	 ELSE
	    open := TRUE; fixup := FALSE; suppressOutput := FALSE;
	 END;
      END InitOutput;

      PROCEDURE CloseOutput;
      BEGIN
	 IF open THEN
	    IF ~Fclose(out) THEN
	       IOFault(output);
	    END;
	 END;
	 open := FALSE; fixup := FALSE;
      END CloseOutput;

      PROCEDURE Write(ch: CHAR);
      BEGIN
	 IF open & ~errorflag & ~suppressOutput THEN
	    IF intobuffer THEN
	       (* buffer exceeding is not handled as implementation
		  restriction because Oberon identifiers (which may
		  be of arbritrary length) shouldn't be
		  placed into the buffer
	       *)
	       Assert(bufindex <= HIGH(buf));
	       buf[bufindex] := ch;
	       INC(bufindex);
	    ELSE
	       IF ~Fputc(ch, out) THEN
		  IOFault(output);
	       END;
	       IF ch = nl THEN
		  INC(outputcount);
	       END;
	    END;
	 END;
      END Write;

      PROCEDURE OpenBuffer;
      BEGIN
	 Assert(~intobuffer);
	 intobuffer := TRUE;
	 bufindex := 0;
      END OpenBuffer;

      PROCEDURE CloseBuffer;
      BEGIN
	 Assert(intobuffer);
	 intobuffer := FALSE;
	 IF bufindex <= HIGH(buf) THEN
	    buf[bufindex] := 0C;
	 END;
      END CloseBuffer;

      PROCEDURE ReleaseBuffer;
      BEGIN
	 Assert(~intobuffer);
	 WriteString(buf);
      END ReleaseBuffer;

      PROCEDURE SuppressOutput;
      BEGIN
	 Assert(~suppressOutput);
	 suppressOutput := TRUE;
      END SuppressOutput;

      PROCEDURE ContinueOutput;
      BEGIN
	 Assert(suppressOutput);
	 suppressOutput := FALSE;
      END ContinueOutput;

      PROCEDURE WriteLn;
      BEGIN
	 Assert(~suppressOutput);
	 Write(nl);
      END WriteLn;

      PROCEDURE WriteString(s: ARRAY OF CHAR);
	 VAR i: CARDINAL;
      BEGIN
	 i := 0;
	 WHILE (i <= HIGH(s)) & (s[i] # 0C) DO
	    Write(s[i]); INC(i);
	 END;
      END WriteString;

      PROCEDURE WriteCard(c: CARDINAL);
	 VAR str: ARRAY [0..11] OF CHAR;
      BEGIN
	 ConvertCardinal(c, 1, str);
	 WriteString(str);
      END WriteCard;

      PROCEDURE WriteLongCard(c: LONGCARD);
	 VAR str: ARRAY [0..11] OF CHAR;
      BEGIN
	 (* FIXME: TO BE REPLACED BY ConvertLongCardinal *)
	 ConvertCardinal(CARDINAL(c), 1, str);
	 WriteString(str);
      END WriteLongCard;

      PROCEDURE WriteHex(c: CARDINAL);
	 VAR
	    str: ARRAY [0..7] OF CHAR;
	    i: CARDINAL;
      BEGIN
	 ConvertHex(c, 8, str);
	 i := 0;
	 WHILE str[i] = ' ' DO
	    str[i] := '0';
	    INC(i);
	 END;
	 WriteString(str);
      END WriteHex;

      PROCEDURE WriteOct3(c: CARDINAL);
	 CONST
	    len = 3;
	 VAR
	    str: ARRAY [0..len-1] OF CHAR;
	    i: CARDINAL;
      BEGIN
	 ConvertOctal(c, len, str);
	 i := 0;
	 WHILE str[i] = ' ' DO
	    str[i] := '0';
	    INC(i);
	 END;
	 WriteString(str);
      END WriteOct3;

      PROCEDURE WriteInt(i: INTEGER);
	 VAR str: ARRAY [0..11] OF CHAR;
      BEGIN
	 ConvertInteger(i, 1, str);
	 WriteString(str);
      END WriteInt;

      PROCEDURE WriteLongInt(i: LONGINT);
	 VAR str: ARRAY [0..11] OF CHAR;
      BEGIN
	 (* FIXME: ConvertInteger is to be replaced by ConvertLongInteger *)
	 ConvertInteger(i, 1, str);
	 WriteString(str);
      END WriteLongInt;

      PROCEDURE SetFixupPos;
      BEGIN
	 Assert(open & ~fixup);
	 IF ~Ftell(out, fixupPos) THEN
	    IOFault(output);
	 END;
	 fixup := TRUE;
      END SetFixupPos;

      PROCEDURE Fixup(s: ARRAY OF CHAR);
      BEGIN
	 Assert(open & fixup);
	 IF ~Ftell(out, currPos) OR ~Fseek(out, fixupPos, 0) THEN
	    IOFault(output);
	 END;
	 WriteString(s);
	 IF ~Fseek(out, currPos, 0) THEN
	    IOFault(output);
	 END;
	 fixup := FALSE;
      END Fixup;

   BEGIN
      open := FALSE; intobuffer := FALSE;
      EnterCleanup(CloseOutput);
   END Output;

   MODULE Formats;

      IMPORT ASCII, Attributes, Exception, IdentSys, Lex, Machine, Memory,
	 Mnemonics, Output, Scan, SymTab, SYSTEM,
	 CheckSegment, bfline, gasversion;

      EXPORT PushArg, WriteMnem, Print, WriteLabel, WriteL;

      MODULE Regs;

	 FROM Attributes IMPORT Reg;
	 FROM Exception IMPORT Assert;
	 FROM Output IMPORT WriteString;

	 EXPORT WriteReg;

	 VAR
	    regname: ARRAY Reg OF ARRAY [0..7] OF CHAR;

	 PROCEDURE WriteReg(r: Reg);
	 BEGIN
	    (* accept it for %(d,r,r) or the like
	    Assert(r # illegal);
	    *)
	    Assert(r # illegal); (* GNU as does not support %za0-notation *)
	    WriteString(regname[r]);
	 END WriteReg;

      BEGIN
         regname[d0] := "d0"; regname[d1] := "d1"; regname[d2] := "d2";
         regname[d3] := "d3"; regname[d4] := "d4"; regname[d5] := "d5";
         regname[d6] := "d6"; regname[d7] := "d7";
         regname[a0] := "a0"; regname[a1] := "a1"; regname[a2] := "a2";
         regname[a3] := "a3"; regname[a4] := "a4"; regname[a5] := "a5";
         regname[a6] := "a6"; regname[a7] := "a7"; regname[illegal] := "za0";
         regname[pc] := "pc"; regname[ccr] := "cc";
         regname[fp0] := "fp0"; regname[fp1] := "fp1"; regname[fp2] := "fp2";
         regname[fp3] := "fp3"; regname[fp4] := "fp4"; regname[fp5] := "fp5";
         regname[fp6] := "fp6"; regname[fp7] := "fp7";
         regname[fpcr] := "fpcr"; regname[fpsr] := "fpsr";
      END Regs;

      MODULE Args;

	 FROM Exception IMPORT Assert;
	 FROM SYSTEM IMPORT BYTE, ADDRESS, ADR;

	 EXPORT PushArg, NextArg, AllArgs, RewindArgs, SkipRestOfArgs;

	 CONST
	    maxarg = 10;
	 TYPE
	    Arg =
	       RECORD
		  addr: ADDRESS;
		  size: CARDINAL;
	       END;
	 VAR
	    args: ARRAY [0..maxarg-1] OF Arg;
	    argcnt: CARDINAL;
	    index: CARDINAL;

	 PROCEDURE PushArg(VAR arg: ARRAY OF BYTE);
	 BEGIN
	    Assert((index = 0) & (argcnt < maxarg));
	    WITH args[argcnt] DO
	       addr := ADR(arg);
	       size := SIZE(arg);
	    END;
	    INC(argcnt);
	 END PushArg;

	 PROCEDURE NextArg(VAR a: ADDRESS; VAR s: CARDINAL);
	 BEGIN
	    Assert(index < argcnt);
	    WITH args[index] DO
	       a := addr;
	       s := size;
	    END;
	    INC(index);
	 END NextArg;

	 PROCEDURE AllArgs;
	 BEGIN
	    Assert(index = argcnt);
	    index := 0; argcnt := 0;
	 END AllArgs;

	 PROCEDURE RewindArgs;
	 BEGIN
	    Assert(index = argcnt);
	    index := 0;
	 END RewindArgs;

	 PROCEDURE SkipRestOfArgs;
	 BEGIN
	    index := 0; argcnt := 0;
	 END SkipRestOfArgs;

      BEGIN
	 argcnt := 0; index := 0;
      END Args;

      MODULE Printing;

	 FROM Args IMPORT NextArg, AllArgs, RewindArgs, SkipRestOfArgs;
	 FROM Attributes IMPORT Attribute, Label, Reg, RegSet, AtMode;
	 FROM Exception IMPORT Assert;
	 FROM IdentSys IMPORT Identifier, GetIdentChar;
	 FROM Lex IMPORT String, GetStringNumber, Symbol, Constval;
	 FROM Machine IMPORT nilvalue, oneword;
	 FROM Memory IMPORT CheckPtr;
	 FROM Mnemonics IMPORT Mnemonic, Mnem;
	 FROM Output IMPORT WriteString, WriteLn, Write, WriteCard, WriteInt,
	    WriteLongCard, WriteLongInt, WriteHex, WriteOct3,
	    SetFixupPos, Fixup, OpenBuffer, CloseBuffer, ReleaseBuffer,
	    SuppressOutput, ContinueOutput;
	 FROM Regs IMPORT WriteReg;
	 FROM Scan IMPORT pos;
	 FROM SymTab IMPORT mainmod, Type, Size, Ident, global, IdentClass;
	 FROM SYSTEM IMPORT ADDRESS;
	 IMPORT ASCII, bfline, CheckSegment, gasversion;

	 EXPORT WriteMnem, Print, WriteLabel, WriteL;

	 VAR
	    start: BOOLEAN; (* at beginning of a new line? *)
	    comment: BOOLEAN; (* inside of a comment? *)
	    lastline: CARDINAL;
	    suppressmod: BOOLEAN; (* modifier to be suppressed? *)
	    reversemod: BOOLEAN; (* operands to be exchanged? *)
	    changemod: BOOLEAN; (* modifier to be changed: b --> s *)

	 PROCEDURE WriteLabel(l: Label);
	 BEGIN
	    WITH l DO
	       Assert(ok);
	       Write('L');
	       IF head # 0C THEN
		  Write(head);
	       END;
	       Write('.');
	       IF n1 # 0 THEN
		  WriteCard(n1);
	       END;
	       IF n2 # 0 THEN
		  Write('.'); WriteCard(n2);
	       END;
	    END;
	 END WriteLabel;

	 PROCEDURE WriteL(ch: CHAR; no: CARDINAL);
	    VAR
	       l: Label;
	 BEGIN
	    WITH l DO
	       ok := TRUE;
	       head := ch;
	       n1 := 0;
	       n2 := no;
	    END;
	    WriteLabel(l);
	 END WriteL;

	 PROCEDURE WriteTab;
	 BEGIN
	    Write(ASCII.tab); start := FALSE;
	 END WriteTab;

	 PROCEDURE WriteLine;
	 BEGIN
	    lastline := pos.line;
	    WriteString("/* @ "); WriteCard(lastline);
	    WriteString(" */"); WriteLn;
	    IF bfline # 0 THEN
	       WriteTab; WriteString(".stabn"); WriteTab;
	       WriteString("0104,0,"); WriteCard(lastline); WriteString(",.");
	       WriteLn;
	       start := TRUE;
	    END;
	 END WriteLine;

	 PROCEDURE WriteMnem(m: Mnemonic);
	    TYPE
	       MnemSet = SET OF Mnemonic;
	    CONST
	       suppressMod = MnemSet{LEA, PEA, TAS,
				 SCC, SCS, SEQ, SF, SGE, SGT, SHI, SLE,
				 SLS, SLT, SMI, SNE, SPL, ST, SVC, SVS,
				 FSGE, FSGL, FSGLE, FSGT, FSLE, FSLT};
	       changeMod = MnemSet{
				 BCC, BCS, BEQ, BGE, BGT, BHI, BLE, BLS,
				 BLT, BMI, BNE, BPL, BVC, BVS,
				 BRA};
	       reverseMod = MnemSet{CMP, CMPA, CMPI, FCMP};
		  (* the ``original'' operand order of CMP-instructions
		     is equivalent to the operand order of SUB-instructions
		     (because CMP is a SUB without result but cc);
		     Motorola itself reverses the operands of CMP-instructions
		     for its assembler for reasons of readability,
		     e.g.
			   cmp.l %d3,%d5	%d3 < %d5?
			   ble   is_less

		     SUNs as and GNU-as do not follow the convention
		     of Motorola:

			   cmpl  d5,d3		d3 < d5?
			   ble   is_less
		  *)
	 BEGIN
	    Assert(start);
	    IF pos.line > lastline THEN
	       WriteLine;
	    END;
	    WriteTab; (* start = FALSE *)
	    WriteString(Mnem[m]);
	    suppressmod := m IN suppressMod;
	    changemod := m IN changeMod;
	    reversemod := m IN reverseMod;
	 END WriteMnem;

	 PROCEDURE WriteComment;
	 BEGIN
	    IF ~comment THEN
	       IF start THEN
		  WriteString("/*   ");
		  start := FALSE;
	       ELSE
		  WriteTab; WriteString("/* ");
	       END;
	       comment := TRUE;
	    END;
	 END WriteComment;

	 PROCEDURE Print(format: ARRAY OF CHAR);

	    CONST
	       (* configuration parameters *)
	       enforceIndex = (gasversion = 136);
		  (* enforceIndex is needed if the assembler does not accept
			apc@(digits)@
		     thus we need to append (0):
			apc@(digits)@(0)
		  *)

	       immedch = "#";
	       symprefix = "O_";
	    TYPE
	       CharSet = SET OF CHAR;
	    CONST
	       Printable = CharSet{ASCII.tab, ' '..'~'};
	       Legal = CharSet{'a'..'z', 'A'..'Z', ',', '.', '_', '+', '-'};
	    VAR
	       ch: CHAR;	(* ch = format[index] *)
	       index: CARDINAL;	(* in format *)
	       mnem: BOOLEAN;   (* mnemonic written *)
	       tab: BOOLEAN;	(* tab after mnem written? *)
	       atsize: Size;	(* size attribute of %a-parameter *)
	       fixup: BOOLEAN;  (* fixup size of attribute of %A *)
	       immed: BOOLEAN;	(* & (or #) for immediate value written? *)
	       deflabel: BOOLEAN; (* %:l *)
	       bss: BOOLEAN;	(* %:b *)
	       bssSize: LONGCARD; (* size of bss-definition in bytes *)
	       (* if in reverseMod: *)
	       delimIndex: CARDINAL; (* index position of delimiter *)
	       skipat: BOOLEAN; (* skip size at in transition to readOp1 *)
	       revstate: (readAt1, readAt2, suppressOp, suppressDelim,
			  readOp1, readDelim, readOp2);

	    PROCEDURE PrepareReverseMode;
	       (* prepare operand exchanging:
		  the construction following is somewhat cumbersome but
		  it achieves two goals:
		  (1) it allows the rest of the code generation not to
		      worry about different orders of cmp operands
		  (2) exchanging is done *without* interpreting the
		      %-sequences (wouldn't be easy)

		  the args are evaluated in two passes:

		  format layout:

		     [optional size at] [first operand] [delim] [second operand]

		  state transitions:

		  pass1: (* print first operand *)
			 optional readAt1 and readAt2, suppressOp,
			 suppressDelim, readOp1, readDelim
		  pass2: (* print second operand *)
			 RewindArgs and repositioning to first operand
			 readOp2

		  printing is suppressed during suppressOp and suppressDelim
	       *)
	       CONST
		  delim = ',';
		  sizeats = CharSet{'A', 'B', 'D', 'L', 'W', 'X'};
	       VAR
		  i: CARDINAL; (* index of format *)
		  nestlevel: CARDINAL; (* of () and [] *)
	    BEGIN
	       (* look for operand delimiter *)
	       i := 0; nestlevel := 0;
	       WHILE (i <= HIGH(format)) & (format[i] # 0C) &
		     ((nestlevel > 0) OR (format[i] # delim)) DO
		  CASE format[i] OF
		  | '(', '[': INC(nestlevel);
		  | ')', ']': DEC(nestlevel);
		  ELSE
		  END;
		  INC(i);
	       END;
	       Assert((i <= HIGH(format)) & (format[i] = delim));

	       (* setup initial state *)
	       delimIndex := i;
	       IF (delimIndex > 1) &
		     (format[0] = '%') & (format[1] IN sizeats) THEN
		  revstate := readAt1;
		  skipat := TRUE;
	       ELSE
		  revstate := suppressOp; SuppressOutput;
		  skipat := FALSE;
	       END;
	       index := 0;
	    END PrepareReverseMode;

	    PROCEDURE NextCh;
	    BEGIN
	       IF reversemod THEN
		  IF revstate = readDelim THEN
		     RewindArgs;
		     IF skipat THEN
			index := 2;
		     ELSE
			index := 0;
		     END;
		     revstate := readOp2;
		  ELSE
		     INC(index);
		     IF revstate = readAt1 THEN
			revstate := readAt2;
		     ELSIF revstate = readAt2 THEN
			revstate := suppressOp; SuppressOutput;
		     ELSIF (revstate = suppressOp) & (index = delimIndex) THEN
			revstate := suppressDelim;
		     ELSIF revstate = suppressDelim THEN
			revstate := readOp1; ContinueOutput;
		     ELSIF (index > HIGH(format)) OR (format[index] = 0C) THEN
			Assert(revstate = readOp1);
			revstate := readDelim;
			index := delimIndex;
		     END;
		  END;
		  Assert((revstate # readOp2) OR (index < delimIndex));
	       ELSE
		  INC(index);
	       END;
	       Assert((index <= HIGH(format)) & (format[index] # 0C));
	       ch := format[index];
	    END NextCh;

	    PROCEDURE Advance;
	       (* advance index but do not modify ch *)
	       VAR
		  keepch: CHAR;
	    BEGIN
	       IF reversemod & (revstate <= readDelim) THEN
		  keepch := ch;
		  NextCh;
		  ch := keepch;
	       ELSE
		  INC(index);
	       END;
	    END Advance;

	    PROCEDURE EndOfFormat() : BOOLEAN;
	    BEGIN
	       IF reversemod THEN
		  RETURN (revstate = readOp2) & (index = delimIndex)
	       ELSE
		  RETURN (index > HIGH(format)) OR (format[index] = 0C)
	       END;
	    END EndOfFormat;

	    PROCEDURE CTab;
	    BEGIN
	       IF mnem & ~tab THEN
		  WriteTab;
		  tab := TRUE;
	       END;
	    END CTab;

	    PROCEDURE WriteSizeAttribute(ch: CHAR);
	    BEGIN
	       Assert(mnem & ~tab);
	       IF ~suppressmod THEN
		  Write(ch);
	       END;
	       CTab;
	    END WriteSizeAttribute;

	    PROCEDURE WriteImmed;
	    BEGIN
	       Assert(~immed);
	       immed := TRUE;
	       Write('#');
	    END WriteImmed;

	    PROCEDURE TakeAttribute;
	       VAR
		  atptr: POINTER TO Attribute;
		  size: CARDINAL;
		  at: Attribute;
		  newsize: Size;

	       PROCEDURE WriteIdent(ip: Ident);
		  (* write symbol name of variables and procedures *)
	       BEGIN
		  WITH ip^ DO
		     WriteString(symprefix);
		     WriteIdentifier(mod^.name);
		     Write("_");
		     IF exported OR (class = varC) THEN
			WriteIdentifier(name);
		     ELSE
			Assert(class = procedureC);
			WriteCard(procno);
		     END;
		  END;
	       END WriteIdent;

	       PROCEDURE WriteIndex;

		  CONST
		     scales = {1, 2, 4, 8};
		     maxscale = 8;
		  VAR
		     comma: BOOLEAN;
		     lparen: BOOLEAN; (* "(" written? *)

		  PROCEDURE Comma;
		  BEGIN
		     IF comma THEN
			Write(","); comma := FALSE;
		     END;
		  END Comma;

		  PROCEDURE Lparen;
		  BEGIN
		     IF ~lparen THEN
			Write("(");
			lparen := TRUE;
		     END;
		  END Lparen;

	       BEGIN (* WriteIndex *)
		  comma := FALSE; lparen := FALSE;
		  WITH at^ DO (* indexMode or memIndexMode *)
		     IF reg # illegal THEN
			Assert(reg IN RegSet{a0..a7});
			WriteReg(reg);
		     END;
		     Write("@");
		     IF (labelip # NIL) OR rlabel.ok THEN
			Lparen;
			IF labelip # NIL THEN
			   WriteIdent(labelip);
			ELSE
			   WriteLabel(rlabel);
			END;
			IF addr > 0 THEN
			   Write("+");
			END;
			comma := TRUE;
		     END;
		     IF addr # 0 THEN
			Lparen;
			WriteInt(addr);
			comma := TRUE;
		     END;
		     IF (mode = memIndexMode) & post THEN
			IF ~lparen THEN
			   WriteString("(0)");
			ELSE
			   Write(")");
			END;
			Write("@"); lparen := FALSE; comma := FALSE;
		     END;
		     IF (mode = memIndexMode) & post &
			   ((od # 0) OR enforceIndex) THEN
			Comma; Lparen;
			WriteInt(od);
			comma := TRUE;
		     END;
		     IF xreg # illegal THEN
			Comma; Lparen;
			WriteReg(xreg);
			WriteString(":l");
			Assert((scale <= maxscale) & (scale IN scales));
			IF scale # 1 THEN
			   Write(":");
			   WriteCard(scale);
			END;
			comma := TRUE;
		     END;
		     IF mode = memIndexMode THEN
			IF ~post THEN
			   IF ~lparen THEN
			      WriteString("(0)");
			   ELSE
			      Write(")");
			   END;
			   Write("@");
			   lparen := FALSE; comma := FALSE;
			   IF (od # 0) OR enforceIndex THEN
			      Lparen;
			      WriteInt(od);
			   END;
			END;
		     END;
		     IF lparen THEN
			Write(")");
		     END;
		  END;
	       END WriteIndex;

	    BEGIN (* TakeAttribute *)
	       NextArg(atptr, size); at := atptr^;
	       Assert((size = SIZE(Attribute)) & CheckPtr(at));
	       WITH at^ DO
		  IF fixup THEN
		     IF mode = floatRegMode THEN
			newsize := 12;
		     ELSIF (mode = constAt) & (cval.sy = longrealcon) THEN
			newsize := 8;
		     ELSE
			newsize := attype^.size;
		     END;
		     IF (atsize = 0) OR
			(newsize < atsize) & (newsize >= oneword) OR
			(newsize > atsize) & (atsize < oneword) THEN
			atsize := newsize;
		     END;
		  END;
		  CASE mode OF
		  | constAt:        IF cval.sy # stringcon THEN
				       Write(immedch);
				    END;
				    WriteConstval(cval);
		  | procAt:         Write(immedch); WriteIdent(atip);
		  | regMode,
		    floatRegMode:   WriteReg(reg);
		  | addrMode:       (* some assemblers requires an
				       explicit index mode in this case
				    *)
				    Assert(reg IN RegSet{a0..a7});
				    WriteReg(reg); Write("@");
		  | indexMode,
		    memIndexMode:   WriteIndex;
		  ELSE
		     Assert(FALSE);
		  END;
	       END;
	    END TakeAttribute;

	    PROCEDURE TakeCardinal;
	       TYPE
		  LongCardPtr = POINTER TO LONGCARD;
	       VAR
		  cardptr: POINTER TO CARDINAL;
		  longcardptr: LongCardPtr;
		  size: CARDINAL;
	    BEGIN
	       NextArg(cardptr, size);
	       IF size = SIZE(CARDINAL) THEN
		  WriteCard(cardptr^);
	       ELSIF size = SIZE(LONGCARD) THEN
		  longcardptr := LongCardPtr(cardptr);
		  WriteLongCard(longcardptr^);
	       ELSE
		  Assert(FALSE);
	       END;
	    END TakeCardinal;

	    PROCEDURE TakeInteger;
	       TYPE
		  LongIntPtr = POINTER TO LONGINT;
	       VAR
		  intptr: POINTER TO INTEGER;
		  longintptr: LongIntPtr;
		  size: CARDINAL;
	    BEGIN
	       NextArg(intptr, size);
	       IF size = SIZE(INTEGER) THEN
		  WriteInt(intptr^);
	       ELSIF size = SIZE(LONGINT) THEN
		  longintptr := LongIntPtr(intptr);
		  WriteLongInt(longintptr^);
	       ELSE
		  Assert(FALSE);
	       END;
	    END TakeInteger;

	    PROCEDURE TakeLabel;
	       VAR
		  labptr: POINTER TO Label;
		  size: CARDINAL;
	    BEGIN
	       NextArg(labptr, size); Assert(size = SIZE(Label));
	       WriteLabel(labptr^);
	    END TakeLabel;

	    PROCEDURE WriteIdentifier(id: Identifier);
	       VAR
		  ch: CHAR;
	    BEGIN
	       GetIdentChar(id, ch);
	       WHILE ch # 0C DO
		  Write(ch);
		  GetIdentChar(id, ch);
	       END;
	    END WriteIdentifier;

	    PROCEDURE TakeIdentifier;
	       VAR
		  idptr: POINTER TO Identifier;
		  id: Identifier;
		  size: CARDINAL;
	    BEGIN
	       NextArg(idptr, size); Assert(size = SIZE(Identifier));
	       id := idptr^;
	       WriteIdentifier(id);
	    END TakeIdentifier;

	    PROCEDURE TakeType;
	       (* print label of tag record *)
	       VAR
		  tptr: POINTER TO Type;
		  size: CARDINAL;
		  type: Type;
	    BEGIN
	       NextArg(tptr, size); Assert(size = SIZE(Type));
	       type := tptr^;
	       WITH type^ DO
		  WriteString(symprefix);
		  IF (ident # NIL) & ident^.exported THEN
		     WriteIdentifier(ident^.mod^.name);
		     Write("_");
		     WriteIdentifier(ident^.name);
		  ELSE
		     WriteIdentifier(mainmod^.name);
		     Write("_");
		     WriteCard(tagno);
		  END;
		  WriteString("_TAG");
	       END;
	    END TakeType;

	    PROCEDURE TakeRegister;
	       VAR
		  regptr: POINTER TO Reg;
		  size: CARDINAL;
	    BEGIN
	       NextArg(regptr, size); Assert(size = SIZE(Reg));
	       WriteReg(regptr^);
	    END TakeRegister;

	    PROCEDURE RegisterInd(ch: CHAR);
	    BEGIN
	       TakeRegister; Write("@"); Write(ch);
	    END RegisterInd;

	    PROCEDURE TakeIndirect;
	       
	       (* %()  inside () one of
		       d,r
		       d,r,r
		       r,r		register indirect with index
		       [d,r],r,d		memory indirect post-indexed
		       [d,r,r],d		memory indirect pre-indexed
		       ...
		       %-sequences instead of `d' permitted
		       `r' may be followed by 2, 4, or 8 (scale)
	       *)

	       VAR
		  regcnt: CARDINAL; 	(* count `r' *)
		  discnt: CARDINAL; 	(* count displacements *)
		  disfmt: BOOLEAN;  	(* displacement with %-format *)
		  memind: BOOLEAN;  	(* memory indirect? *)
		  displacement: BOOLEAN; (* result of d or % in buffer *)
		  xreginbuf: BOOLEAN; 	(* result of r in buffer *)
		  post: BOOLEAN; 	(* "]" seen? *)
		  comma: BOOLEAN; 	(* "," necessary? *)
		  lparen: BOOLEAN; 	(* "(" written? *)
		  open: BOOLEAN; 	(* buffer opened? *)
		  secondIndexWritten: BOOLEAN;
		     (* assert that index is appended in case of
			memory indirect and enforceIndex
		     *)

	       PROCEDURE Comma;
	       BEGIN
		  IF comma THEN
		     Write(","); comma := FALSE;
		  END;
	       END Comma;

	       PROCEDURE Lparen;
	       BEGIN
		  IF ~lparen THEN
		     Write("(");
		     lparen := TRUE;
		     comma := FALSE;
		  END;
	       END Lparen;

	       PROCEDURE GetCh() : BOOLEAN;
	       BEGIN
		  Assert(~EndOfFormat());
		  ch := format[index];
		  IF ch # ')' THEN
		     Advance;
		  END;
		  RETURN ch # ')'
	       END GetCh;

	       PROCEDURE UngetCh;
	       BEGIN
		  Assert(index > 0);
		  DEC(index);
	       END UngetCh;

	       PROCEDURE TakeRegister;
		  VAR
		     regptr: POINTER TO Reg;
		     size: CARDINAL;
	       BEGIN
		  Assert((regcnt <= 1) & (discnt <= 1));
		  NextArg(regptr, size); Assert(size = SIZE(Reg));
		  IF regptr^ # illegal THEN
		     IF (regptr^ IN RegSet{d0..d7}) & (regcnt = 0) THEN
			(* special case:
			   first index register is a data register:
			      (r      -->  @(r
			      (d,r         @(d,r
			      ([d,r        @(d,r
			*)
			Write("@");
			Lparen;
			IF displacement THEN
			   ReleaseBuffer; displacement := FALSE;
			   comma := TRUE;
			END;
			Comma;
		     END;
		     WriteReg(regptr^);
		     IF (regptr^ IN RegSet{d0..d7}) OR (regcnt = 1) THEN
			WriteString(":l");
		     END;
		     comma := TRUE;
		  END;
		  IF GetCh() THEN
		     CASE ch OF
		     | '2', '4', '8': (* scale *)
			   IF regptr^ # illegal THEN
			      Write(':'); Write(ch);
			   END;
		     ELSE
			UngetCh;
		     END;
		  END;
		  INC(regcnt);
	       END TakeRegister;

	       PROCEDURE CheckClose;
	       BEGIN
		  IF open THEN
		     CloseBuffer;
		     open := FALSE;
		  END;
	       END CheckClose;

	    BEGIN
	       regcnt := 0; discnt := 0; memind := FALSE; disfmt := FALSE;
	       post := FALSE; displacement := FALSE; xreginbuf := FALSE;
	       secondIndexWritten := FALSE;
	       comma := FALSE; lparen := FALSE; open := FALSE;
	       WHILE GetCh() DO
		  CASE ch OF
		  | 'd', '%':
			   Assert((discnt = 0) OR (discnt = 1) & memind);
			   IF discnt = 0 THEN
			      (* first displacement *)
			      (* (d,r    -->  r@(d
				 ([d,r   -->  r@(d
			      *)
			      IF ch = 'd' THEN
				 OpenBuffer; TakeInteger; CloseBuffer;
			      ELSE
				 IF ~open THEN
				    OpenBuffer; open := TRUE;
				 END;
				 Assert(GetCh());
				 WorkupParameter; disfmt := TRUE;
			      END;
			      displacement := TRUE;
			   ELSE (* memory indirect *)
			      (* outer displacement *)
			      (* ([...],r,d   --> ...)@(d,r)
				 ([...],d     --> ...)@(d)
			      *)
			      (* @ has been printed at "]" *)
			      secondIndexWritten := TRUE;
			      Write("(");
			      IF ch = 'd' THEN
				 TakeInteger;
				 IF xreginbuf THEN
				    Write(",");
				    ReleaseBuffer; xreginbuf := FALSE;
				 END;
				 Write(")");
				 lparen := FALSE;
			      ELSE
				 Assert(GetCh());
				 WorkupParameter; disfmt := TRUE;
			      END;
			   END;
			   INC(discnt);
		  | 'r':   IF post THEN
			      (* ([...],r,d)  -->  ...)@(d,r)
				 ([...],r)    -->  ...)@(r)
			      *)
			      OpenBuffer; TakeRegister; CloseBuffer;
			      xreginbuf := TRUE;
			      secondIndexWritten := TRUE;
			   ELSIF regcnt = 0 THEN
			      (* first index register *)
			      (* address register:
				    (r      -->  r@
				    (d,r         r@(d
				    ([d,r        r@(d
				 data register:
				    (r      -->  @(r
				    (d,r         @(d,r
				    ([d,r        @(d,r
				 the second case is partially handled
				 by TakeRegister
			      *)
			      TakeRegister;
			      IF ~lparen THEN
				 (* if lparen is set then TakeRegister
				    has already written @(d,r
				 *)
				 Write("@");
				 lparen := FALSE; comma := FALSE;
				 IF displacement THEN
				    Lparen;
				    ReleaseBuffer; displacement := FALSE;
				    comma := TRUE;
				 END;
			      END;
			   ELSIF regcnt = 1 THEN
			      (* (d,r,r       -->  r@(d,r
				 (r,r         -->  r@(r
			      *)
			      Lparen; Comma;
			      TakeRegister;
			   END;
		  | ',':   IF disfmt THEN INC(discnt) END; disfmt := FALSE;
			   CheckClose;
		  | '[':   Assert(~memind); memind := TRUE;
		  | ']':   Assert(memind & (discnt <= 1));
			   CheckClose;
			   IF lparen THEN
			      Write(")");
			      lparen := FALSE;
			   ELSE
			      WriteString("(0)");
			   END;
		           Write("@");
			   post := TRUE;
			   comma := FALSE;
		  ELSE
		     Assert(FALSE);
		  END;
	       END;
	       Assert(discnt <= 2);
	       CheckClose;
	       IF xreginbuf THEN
		  Lparen; Comma;
		  ReleaseBuffer; xreginbuf := FALSE;
	       ELSIF displacement THEN
		  Lparen;
		  ReleaseBuffer; displacement := FALSE;
	       ELSIF enforceIndex & post & ~secondIndexWritten THEN
		  (* fix gas bug *)
		  WriteString("(0)");
	       END;
	       IF lparen THEN
		  Write(")"); lparen := FALSE;
	       END;
	    END TakeIndirect;

	    PROCEDURE TakeBitfield;
	       (* %{}  offset/width specification of bit fields inside {} one of
		       d:d
		       d:r
		       r:d
		       r:r
		       %-sequences instead of `d' permitted
		       r  any data register
		       d  any value between 0-31 (offset) or 1-32 (width)
		       %{} is preceded by the effective address, e.g. %a%{d:d}
	       *)
	       VAR
		  state: CARDINAL;
		     (* 1:  ^ offset : width
			2:  offset ^ : width
			3:  offset : ^ width
			4:  offset : width ^
		     *)

	       PROCEDURE GetCh() : BOOLEAN;
	       BEGIN
		  Assert(~EndOfFormat());
		  ch := format[index];
		  IF ch # '}' THEN
		     Advance;
		  END;
		  RETURN ch # '}'
	       END GetCh;

	    BEGIN
	       state := 1;
	       Write('{');
	       WHILE GetCh() DO
		  CASE ch OF
		  | 'd': Assert(state # 2); Write(immedch); TakeInteger;
		  | 'r': Assert(state # 2); TakeRegister;
		  | ':': Assert(state = 2); Write(':');
		  | '%': Assert((state # 2) & GetCh()); WorkupParameter;
		  ELSE
		     Assert(FALSE);
		  END;
		  INC(state);
	       END;
	       Write('}');
	       Assert(state = 4);
	    END TakeBitfield;

	    PROCEDURE TakeCharacterString;
	       TYPE
		  CharPtr = POINTER TO CHAR;
	       VAR
		  s: CharPtr;
		  p: ADDRESS;
		  len: CARDINAL;
	    BEGIN
	       NextArg(p, len);
	       LOOP
		  IF len = 0 THEN EXIT END;
		  s := CharPtr(p); INC(p, SIZE(CHAR)); DEC(len);
		  IF s^ = 0C THEN EXIT END;
		  Write(s^);
	       END;
	    END TakeCharacterString;

	    PROCEDURE TakeLexString;
	       VAR
		  sptr: POINTER TO String;
		  size: CARDINAL;
	    BEGIN
	       NextArg(sptr, size); Assert(size = SIZE(String));
	       WriteL('S', GetStringNumber(sptr^));
	    END TakeLexString;

	    PROCEDURE WriteConstval(cval: Constval);

	       PROCEDURE WriteRealVal(r: REAL);
		  TYPE
		     Int2 = ARRAY [0..1] OF CARDINAL;
		  VAR
		     int2: Int2;
	       BEGIN
		  int2 := Int2(r);
		  WriteString("0x");
		  WriteHex(int2[0]); WriteHex(int2[1]);
	       END WriteRealVal;

	    BEGIN
	       WITH cval DO
		  CASE sy OF
		  | stringcon:   WriteL('S', GetStringNumber(string));
		  | charcon:     WriteCard(ORD(charval));
		  | intcon:      WriteInt(intval);
		  | realcon,
		    longrealcon: WriteRealVal(realval);
		  | setcon:      WriteCard(CARDINAL(setval));
		  | boolcon:     WriteCard(ORD(boolval));
		  | nilSY:       WriteCard(nilvalue);
		  END;
	       END;
	    END WriteConstval;

	    PROCEDURE TakeConstval;
	       VAR
		  cp: POINTER TO Constval;
		  size: CARDINAL;
	    BEGIN
	       NextArg(cp, size); Assert(size = SIZE(Constval));
	       WriteConstval(cp^);
	    END TakeConstval;

	    PROCEDURE DefineString;
	       TYPE
		  CharPtr = POINTER TO CHAR;
	       VAR
		  s: CharPtr;
		  p: ADDRESS;
		  len: CARDINAL;
		  cnt: CARDINAL;
	    BEGIN
	       Assert(~comment);
	       cnt := 0;
	       NextArg(p, len);
	       LOOP
		  IF len = 0 THEN EXIT END;
		  s := CharPtr(p); INC(p, SIZE(CHAR)); DEC(len);
		  IF s^ = 0C THEN EXIT END;
		  IF cnt MOD 30 = 0 THEN
		     IF cnt > 0 THEN
			Write('"'); WriteLn;
		     END;
		     WriteTab; WriteString(".ascii"); WriteTab;
		     Write('"');
		  END;
		  INC(cnt);
		  IF s^ IN (Printable - CharSet{ASCII.tab, '"', '\'}) THEN
		     Write(s^);
		  ELSE
		     Write('\');
		     CASE s^ OF
		     | ASCII.tab:   Write('t');
		     | '\':         Write('\');
		     | '"':         Write('"');
		     ELSE
			WriteOct3(ORD(s^));
		     END;
		  END;
	       END;
	       IF cnt > 0 THEN
		  (* pad string to next oneword-boundary *)
		  WHILE cnt MOD oneword # 0 DO
		     Write('\'); WriteOct3(0); INC(cnt);
		  END;
		  Write('"');
	       END;
	    END DefineString;

	    PROCEDURE Import;
	    BEGIN
	       Export;
	    END Import;

	    PROCEDURE Export;
	    BEGIN
	       Assert(start);
	       WriteTab; WriteString(".globl"); WriteTab;
	    END Export;

	    PROCEDURE LabelDef;
	    BEGIN
	       Assert(start);
	       WriteTab; WriteString(".set"); WriteTab;
	       TakeLabel; Write(',');
	    END LabelDef;

	    PROCEDURE WorkupParameter;
	    BEGIN
	       CASE ch OF
	       | 'a': CTab; TakeAttribute;
	       | 'A': fixup := TRUE; atsize := 0;
		      SetFixupPos; WriteSizeAttribute('?');
	       | 'B': IF changemod THEN
			 WriteSizeAttribute('s');
		      ELSE
			 WriteSizeAttribute('b');
		      END;
	       | 'c': CTab; TakeCardinal;
	       | 'C': CTab; WriteImmed; TakeCardinal;
	       | 'D': WriteSizeAttribute('d');
	       | 'e': Import;
	       | 'g': Export;
	       | 'i': CTab; TakeInteger;
	       | 'I': CTab; WriteImmed; TakeInteger;
	       | 'l': CTab; TakeLabel;
	       | 'L': WriteSizeAttribute('l');
	       | 'n': CTab; TakeIdentifier;
	       | 'P': IF ~start THEN (* pseudo operations *)
			 Write(';');
		      END;
		      WriteTab; TakeCharacterString; WriteTab;
	       | 'r': CTab; TakeRegister;
	       | '+': CTab; NextCh; Assert(ch = 'r'); RegisterInd('+');
	       | '-': CTab; NextCh; Assert(ch = 'r'); RegisterInd('-');
	       | '(': CTab; NextCh; TakeIndirect;
	       | '{': Assert(tab); NextCh; TakeBitfield;
	       | 's': CTab; TakeCharacterString;
	       | 'S': CTab; TakeLexString;
	       | 't': CTab; TakeType;
	       | 'v': CTab; TakeConstval;
	       | 'V': CTab; WriteImmed; TakeConstval;
	       | 'W': WriteSizeAttribute('w');
	       | 'X': WriteSizeAttribute('x');
	       | '*': WriteComment;
	       | '#': CTab; WriteImmed;
	       | '_': CTab; WriteString(symprefix);
	       | '=': LabelDef;
	       ELSE
		  Assert(FALSE);
	       END;
	    END WorkupParameter;

	    PROCEDURE BSSDefinition(global: BOOLEAN);
	       VAR argsize: CARDINAL;
		   cp: POINTER TO LONGCARD;
	    BEGIN
	       bss := TRUE; tab := TRUE; start := FALSE;
	       NextArg(cp, argsize); Assert(argsize = SIZE(LONGCARD));
	       bssSize := cp^;
	       WriteTab;
	       IF global THEN
		  WriteString(".comm");
	       ELSE
		  WriteString(".lcomm");
	       END;
	       WriteTab;
	    END BSSDefinition;

	 BEGIN (* Print *)
	    IF start & (lastline < pos.line) THEN
	       WriteLine;
	    END;
	    CheckSegment;
	    mnem := ~start; tab := FALSE; fixup := FALSE; immed := FALSE;
	    deflabel := FALSE; bss := FALSE;
	    IF reversemod THEN
	       PrepareReverseMode;
	    ELSE
	       index := 0;
	    END;
	    WHILE ~EndOfFormat() DO
	       ch := format[index];
	       IF ch = '%' THEN
		  NextCh;
		  IF ch = ':' THEN
		     NextCh;
		     Assert((ch = 'r') OR (ch = 'i') OR (ch = '=') OR start);
		     CASE ch OF
		     | 'b': BSSDefinition(TRUE);
		     | 'B': BSSDefinition(FALSE);
		     | 'D': WriteTab; WriteString(".set"); WriteTab;
		     | 'i': IF gasversion > 0 THEN
			       TakeRegister; Write(':'); TakeRegister;
			    ELSE
			       Write('('); TakeRegister; WriteString("):(");
			       TakeRegister; Write(')');
			    END;
		     | 'l': deflabel := TRUE; tab := TRUE; start := FALSE;
		     | 'L': WriteTab; WriteString(".long"); WriteTab;
		     | 'r': TakeRegister; Write(':'); TakeRegister;
		     | 's': DefineString;
		     | '=': Assert(~start); Write(',');
		     ELSE
			Assert(FALSE);
		     END;
		  ELSE
		     WorkupParameter;
		  END;
	       ELSE
		  Assert((ch IN Printable) & comment OR (ch IN Legal));
		  CTab;
		  Write(ch);
		  start := FALSE;
	       END;
	       Advance;
	    END;
	    IF deflabel THEN
	       Write(':');
	    ELSIF bss THEN
	       Write(','); WriteCard(bssSize);
	    END;
	    IF fixup THEN
	       Assert(atsize # 0);
	       CASE atsize OF
	       |  1: Fixup("b");
	       |  2: Fixup("w");
	       |  4: Fixup("l");
	       |  8: Fixup("d");
	       | 12: Fixup("x");
	       ELSE
		  Assert(FALSE);
	       END;
	    END;
	    IF comment THEN
	       WriteString(" */");
	    END;
	    WriteLn;
	    IF reversemod THEN
	       SkipRestOfArgs;
	    ELSE
	       AllArgs;
	    END;
	    start := TRUE; comment := FALSE; reversemod := FALSE;
	 END Print;

      BEGIN
	 start := TRUE; comment := FALSE; lastline := 0;
	 reversemod := FALSE;
      END Printing;

   END Formats;

   PROCEDURE Emit(m: Mnemonic; s: ARRAY OF CHAR);
   BEGIN
      WriteMnem(m); Print(s);
   END Emit;

   PROCEDURE Emit1(m: Mnemonic; s: ARRAY OF CHAR; p1: ARRAY OF BYTE);
   BEGIN
      WriteMnem(m); PushArg(p1); Print(s);
   END Emit1;

   PROCEDURE Emit2(m: Mnemonic; s: ARRAY OF CHAR; p1, p2: ARRAY OF BYTE);
   BEGIN
      WriteMnem(m); PushArg(p1); PushArg(p2); Print(s);
   END Emit2;

   PROCEDURE Emit3(m: Mnemonic; s: ARRAY OF CHAR; p1, p2, p3: ARRAY OF BYTE);
   BEGIN
      WriteMnem(m); PushArg(p1); PushArg(p2); PushArg(p3); Print(s);
   END Emit3;

   PROCEDURE Emit4(m: Mnemonic; s: ARRAY OF CHAR;
		   p1, p2, p3, p4: ARRAY OF BYTE);
   BEGIN
      WriteMnem(m);
      PushArg(p1); PushArg(p2); PushArg(p3); PushArg(p4); Print(s);
   END Emit4;

   PROCEDURE Emit5(m: Mnemonic; s: ARRAY OF CHAR;
		   p1, p2, p3, p4, p5: ARRAY OF BYTE);
   BEGIN
      WriteMnem(m);
      PushArg(p1); PushArg(p2); PushArg(p3); PushArg(p4); PushArg(p5);
      Print(s);
   END Emit5;

   PROCEDURE Emit6(m: Mnemonic; s: ARRAY OF CHAR;
		   p1, p2, p3, p4, p5, p6: ARRAY OF BYTE);
   BEGIN
      WriteMnem(m);
      PushArg(p1); PushArg(p2); PushArg(p3); PushArg(p4); PushArg(p5);
      PushArg(p6);
      Print(s);
   END Emit6;

   PROCEDURE Emit7(m: Mnemonic; s: ARRAY OF CHAR;
                   p1, p2, p3, p4, p5, p6, p7: ARRAY OF BYTE);
   BEGIN
      WriteMnem(m);
      PushArg(p1); PushArg(p2); PushArg(p3); PushArg(p4); PushArg(p5);
      PushArg(p6); PushArg(p7);
      Print(s);
   END Emit7;

   PROCEDURE Emit8(m: Mnemonic; s: ARRAY OF CHAR;
                   p1, p2, p3, p4, p5, p6, p7, p8: ARRAY OF BYTE);
   BEGIN
      WriteMnem(m);
      PushArg(p1); PushArg(p2); PushArg(p3); PushArg(p4); PushArg(p5);
      PushArg(p6); PushArg(p7); PushArg(p8);
      Print(s);
   END Emit8;

   PROCEDURE Emit9(m: Mnemonic; s: ARRAY OF CHAR;
		   p1, p2, p3, p4, p5, p6, p7, p8, p9: ARRAY OF BYTE);
   BEGIN
      WriteMnem(m);
      PushArg(p1); PushArg(p2); PushArg(p3); PushArg(p4); PushArg(p5);
      PushArg(p6); PushArg(p7); PushArg(p8); PushArg(p9);
      Print(s);
   END Emit9;

   PROCEDURE StrEmit(s: ARRAY OF CHAR);
   BEGIN
      Print(s);
   END StrEmit;

   PROCEDURE StrEmit1(s: ARRAY OF CHAR; p1: ARRAY OF BYTE);
   BEGIN
      PushArg(p1); Print(s);
   END StrEmit1;

   PROCEDURE StrEmit2(s: ARRAY OF CHAR; p1, p2: ARRAY OF BYTE);
   BEGIN
      PushArg(p1); PushArg(p2); Print(s);
   END StrEmit2;

   PROCEDURE StrEmit3(s: ARRAY OF CHAR; p1, p2, p3: ARRAY OF BYTE);
   BEGIN
      PushArg(p1); PushArg(p2); PushArg(p3); Print(s);
   END StrEmit3;

   PROCEDURE StrEmit4(s: ARRAY OF CHAR; p1, p2, p3, p4: ARRAY OF BYTE);
   BEGIN
      PushArg(p1); PushArg(p2); PushArg(p3); PushArg(p4); Print(s);
   END StrEmit4;

   PROCEDURE StrEmit5(s: ARRAY OF CHAR; p1, p2, p3, p4, p5: ARRAY OF BYTE);
   BEGIN
      PushArg(p1); PushArg(p2); PushArg(p3); PushArg(p4); PushArg(p5);
      Print(s);
   END StrEmit5;

   PROCEDURE StrEmit6(s: ARRAY OF CHAR; p1, p2, p3, p4, p5, p6: ARRAY OF BYTE);
   BEGIN
      PushArg(p1); PushArg(p2); PushArg(p3); PushArg(p4); PushArg(p5);
      PushArg(p6);
      Print(s);
   END StrEmit6;

   PROCEDURE StrEmit7(s: ARRAY OF CHAR;
                   p1, p2, p3, p4, p5, p6, p7: ARRAY OF BYTE);
   BEGIN
      PushArg(p1); PushArg(p2); PushArg(p3); PushArg(p4); PushArg(p5);
      PushArg(p6); PushArg(p7);
      Print(s);
   END StrEmit7;

   PROCEDURE StrEmit8(s: ARRAY OF CHAR;
                   p1, p2, p3, p4, p5, p6, p7, p8: ARRAY OF BYTE);
   BEGIN
      PushArg(p1); PushArg(p2); PushArg(p3); PushArg(p4); PushArg(p5);
      PushArg(p6); PushArg(p7); PushArg(p8);
      Print(s);
   END StrEmit8;

   PROCEDURE StrEmit9(s: ARRAY OF CHAR;
		   p1, p2, p3, p4, p5, p6, p7, p8, p9: ARRAY OF BYTE);
   BEGIN
      PushArg(p1); PushArg(p2); PushArg(p3); PushArg(p4); PushArg(p5);
      PushArg(p6); PushArg(p7); PushArg(p8); PushArg(p9);
      Print(s);
   END StrEmit9;

   PROCEDURE StrEmit10(s: ARRAY OF CHAR;
		   p1, p2, p3, p4, p5, p6, p7, p8, p9, p10: ARRAY OF BYTE);
   BEGIN
      PushArg(p1); PushArg(p2); PushArg(p3); PushArg(p4); PushArg(p5);
      PushArg(p6); PushArg(p7); PushArg(p8); PushArg(p9); PushArg(p10);
      Print(s);
   END StrEmit10;

   (* pseudo operations *)

   PROCEDURE CheckSegment;
   BEGIN
      IF intendedSegment # currentSegment THEN
	 Write(tab);
	 CASE intendedSegment OF
	 | text:  WriteString(".text");
	 | data:  WriteString(".data");
	 END;
	 WriteLn;
	 currentSegment := intendedSegment;
      END;
   END CheckSegment;

   PROCEDURE SetSegment(s: Segment);
   BEGIN
      intendedSegment := s;
   END SetSegment;

   PROCEDURE EmitAlign; (* word (= 2 bytes) alignment *)
   BEGIN
      (* enforce 4-byte alignment:
	 ld on Sun puts objects on 4-byte boundaries
      *)
      Write(tab); WriteString(".align"); Write(tab); Write('2'); WriteLn;
   END EmitAlign;

   PROCEDURE EmitAlign4; (* 4-byte-alignment *)
   BEGIN
      Write(tab); WriteString(".align"); Write(tab); Write('2'); WriteLn;
   END EmitAlign4;

   PROCEDURE EmitLabel(l: Label); (* define label *)
   BEGIN
      CheckSegment;
      WriteLabel(l); Write(':'); WriteLn;
   END EmitLabel;

   PROCEDURE EmitString(s: String); (* define label and string *)
      TYPE
	 CharSet = SET OF CHAR;
      CONST
	 Printable = CharSet{' '..'~'} - CharSet{'"', '\'};
      VAR
	 cnt: CARDINAL; (* number of characters printed *)
	 ch: CHAR;	(* character of `s' *)
   BEGIN
      CheckSegment;
      WriteL('S', GetStringNumber(s));
      Write(':');
      cnt := 0;
      LOOP
	 GetStringChar(s, ch);
	 IF cnt MOD 30 = 0 THEN
	    IF cnt > 0 THEN Write('"'); WriteLn END;
	    Write(tab); WriteString(".ascii"); Write(tab);
	    Write('"');
	 END;
	 INC(cnt);
	 IF ch IN Printable THEN
	    Write(ch);
	 ELSE
	    Write('\');
	    CASE ch OF
	    | tab:   Write('t');
	    | '\':   Write('\');
	    | '"':   Write('"');
	    ELSE
	       WriteOct3(ORD(ch));
	    END;
	 END;
	 IF ch = 0C THEN EXIT END;
      END;
      IF cnt > 0 THEN
	 (* pad string to next oneword-boundary *)
	 WHILE cnt MOD oneword # 0 DO
	    Write('\'); WriteOct3(0); INC(cnt);
	 END;
	 Write('"');
      END;
      WriteLn;
   END EmitString;

   PROCEDURE EmitHeader(modp: Ident); (* module header and key *)
   BEGIN
      WITH modp^ DO
	 Assert(class = moduleC);
	 StrEmit3("%g%_%n_%c_%c",
	    origname, CARDINAL(key.time), CARDINAL(key.pid));
	 StrEmit3("%:l%_%n_%c_%c",
	    origname, CARDINAL(key.time), CARDINAL(key.pid));
      END;
   END EmitHeader;

   PROCEDURE EmitKey(extmodp: Ident); (* key reference to imported module *)
   BEGIN
      WITH extmodp^ DO
	 Assert(class = moduleC);
	 StrEmit3("%e%_%n_%c_%c",
	    origname, CARDINAL(key.time), CARDINAL(key.pid));
      END;
   END EmitKey;

   (* generations of stabs and line numbers *)

   PROCEDURE EmitFileName(filename: ARRAY OF CHAR);
      VAR
	 shortname: ARRAY [0..13] OF CHAR;
	 sindex, index: CARDINAL;
   BEGIN
      index := 0; sindex := 0;
      WHILE (index <= HIGH(filename)) & (filename[index] # 0C) DO
	 IF filename[index] = '/' THEN
	    sindex := 0;
	 ELSIF sindex <= HIGH(shortname) THEN
	    shortname[sindex] := filename[index];
	    INC(sindex);
	 END;
	 INC(index);
      END;
      IF sindex <= HIGH(shortname) THEN
	 shortname[sindex] := 0C;
      END;
      StrEmit4("%P%s%s%s", ".stabs", '"', shortname, '",0144,0,0,.');
   END EmitFileName;

   PROCEDURE EmitBeginBlock1(blockp: Ident; line: CARDINAL);
   BEGIN
      WITH blockp^ DO
	 IF class = moduleC THEN
	    StrEmit5("%P%s%_%n%s%_%n",
	       ".stabs", '"', name, '",044,0,48,', name);
	 ELSE (* procedureC *)
	    StrEmit7("%P%s%_%n_%c%s%_%n_%c",
	       ".stabs", '"', mod^.name, procno, '",044,0,48,',
	       mod^.name, procno);
	 END;
      END;
   END EmitBeginBlock1;

   PROCEDURE EmitBeginBlock2(blockp: Ident; line: CARDINAL);
   BEGIN
      bfline := line;
      StrEmit2("%P%s", ".stabn", "0244,0,0,.");
   END EmitBeginBlock2;

   PROCEDURE EmitEndBlock1(blockp: Ident; line: CARDINAL);
   BEGIN
      StrEmit2("%P%s", ".stabn", "0244,0,1,.");
   END EmitEndBlock1;

   PROCEDURE EmitEndBlock2(blockp: Ident; line: CARDINAL);
   BEGIN
      (*
      WITH blockp^ DO
	 IF class = moduleC THEN
	    StrEmit7("%P%_%n%P%s%P%c%P", "def", name,
	       "val", "~", "scl", -1, "endef");
	 ELSE (* procedureC *)
	    StrEmit8("%P%_%n_%c%P%s%P%i%P",
	       "def", mod^.name, procno,
	       "val", "~", "scl", -1, "endef");
	 END;
      END;
      *)
      bfline := 0;
   END EmitEndBlock2;

   PROCEDURE EmitLineNumer(line: CARDINAL);
   BEGIN
      IF bfline # 0 THEN
	 StrEmit4("%P%s%c%s", ".stabn", "0104,0,", line, ",.");
      END;
   END EmitLineNumer;

   (* generation of command records;
      current segment is undefined afterwards
   *)

   PROCEDURE EmitName(ip: Ident);
      VAR
	 cnt: CARDINAL;
	 ch: CHAR;
   BEGIN
      WITH ip^ DO
	 SetSegment(text); CheckSegment;
	 IF class = moduleC THEN
	    StrEmit1("%:l%_%n__NAME", origname);
	 ELSE
	    StrEmit2("%:l%_%n_%n_NAME", mod^.origname, name);
	 END;
	 cnt := 0;
	 LOOP
	    GetIdentChar(name, ch);
	    IF cnt MOD 30 = 0 THEN
	       IF cnt > 0 THEN Write('"'); WriteLn END;
	       Write(tab); WriteString(".ascii"); Write(tab); Write('"');
	    END;
	    INC(cnt);
	    IF ch = 0C THEN
	       WriteString("\000"); EXIT
	    END;
	    Write(ch);
	 END;
	 Write('"');
	 WriteLn;
	 EmitAlign;
      END;
   END EmitName;

   PROCEDURE InitEmitCode;
      (* to be called after ScanArgs *)
   BEGIN
      InitOutput;
   END InitEmitCode;

BEGIN
   currentSegment := text; (* default segment of as *)
   bfline := 0;
END EmitCode.
