(* Ulm's Oberon Compiler
   Copyright (c) 1989 by University of Ulm, SAI, D-W-7900 Ulm, Germany
   ----------------------------------------------------------------------------
   $Id: GenBasicOps.d,v 0.7 1994/03/17 10:40:39 borchert Exp $
   ----------------------------------------------------------------------------
   $Log: GenBasicOps.d,v $
   Revision 0.7  1994/03/17  10:40:39  borchert
   LoadAddrReg requires now the register to be an address register

   Revision 0.6  1993/10/03  14:55:07  borchert
   MoveBytesAt added

   Revision 0.5  1993/09/27  12:43:38  borchert
   Min and Max added

   Revision 0.4  1993/06/18  15:30:33  borchert
   Const??Reg: INTEGER replaced by Types.Integer

   Revision 0.3  1992/07/31  16:03:16  borchert
   ConversionCheck added

   Revision 0.2  1992/07/31  14:38:13  borchert
   RangeCheck & DynArrayCheck added

   Revision 0.1  1992/07/30  10:46:24  borchert
   Initial revision

   ----------------------------------------------------------------------------
*)

DEFINITION MODULE GenBasicOps; (* AFB 4/89 *)

   (* generate basic operations for common use of Gen-modules;
      every procedure may generate code and is authorized to
      modify `at' but not `at^.attype'
      `at' and everything in connection with `at' must be valid
      (i.e. Scan.errorflag must be FALSE)
   *)

   FROM Attributes IMPORT ArithmeticType, Attribute, Reg, Label,
      TestType;
   FROM SymTab IMPORT Size, Ident, Type;
   FROM Types IMPORT Integer;

   VAR
      ccat: Attribute;
	 (* this variable is accessed by any code-generating routine
	    and set to the attribute which reflects the current
	    condition codes;
	    it allows to decide whether a test-instruction is necessary
	    or not;
	    ccat should be set to NIL on generation of
	    cc-destroying instructions
	 *)

   PROCEDURE Address(at: Attribute);
      (* convert `at' and generate code such that
	 `at' is a valid addressing mode for the MOVE-instruction
	 and can be given as %a-argument to Emit;
	 `at.mode' must be a leaf-node, i.e. not unaryAt,...
      *)

   PROCEDURE DynArray(at: Attribute);
      (* prepare `at' for access of the array AND the dope vector
	 at^.mode must be varAt and is modified to indexMode or memIndexMode;
	 then Dereference(at)            accesses the array
	      Offset(at, dim * oneword)  accesses LEN(array, dim)
      *)

   PROCEDURE LoadDynArraySize(varp: Ident; sizeReg: Reg);
      (* load size of `varp' into `sizeReg' (data register);
	 condition codes are set
      *)

   PROCEDURE CalcDynArraySize(type: Type; VAR lenat: Attribute; sizeReg: Reg);
      (* `type' is the type of an dynamic array;
	 `lenat' addresses the next length component;
		 (not released by CalcDynArraySize)
	 the size is calculated and stored into `sizeReg' (one of d0..d7);
	 the condition codes are set according to the result
	 -- common part of LoadDynArraySize and procedures which
	    want to calculate the element size of a multi-dimensional
	    dynamic array
      *)

   PROCEDURE AlignReg(reg: Reg);
      (* align contents of `reg' which must be a data register *)

   PROCEDURE Min(reg1, reg2: Reg);
      (* return unsigned minimum of reg1 and reg2 in reg1 *)

   PROCEDURE Max(reg1, reg2: Reg);
      (* return unsigned maximum of reg1 and reg2 in reg1 *)

   PROCEDURE Load(at: Attribute);
      (* load the corresponding value of `at' into a data
	 (or floating point) register.
	 `at' must not be a record or an array
      *)

   PROCEDURE LoadA(at: Attribute);
      (* like Load; but destination is an address register *)

   PROCEDURE LoadReg(at: Attribute; r: Reg);
      (* like Load with destination register `r' *)

   PROCEDURE LoadAddr(at: Attribute);
      (* load address of `at' into a address register and
	 convert mode of `at' into addrMode
      *)

   PROCEDURE LoadAddrReg(at: Attribute; r: Reg);
      (* like LoadAddr with destination register `r';
	 `r' must be an address register
      *)

   PROCEDURE LoadAndExtend(at: Attribute);
      (* Load(at); if `at' is byte-sized it is extended to an INTEGER *)

   PROCEDURE LoadCond(at: Attribute);
      (* set condition codes of `at';
	 at is in condMode afterwards and ccat equals at
      *)

   PROCEDURE Convert(at: Attribute; dtype: Type);
      (* convert `at' to type `dtype'; at^.attype becomes dtype;
	 `at' may be loaded
      *)

   PROCEDURE DereferenceAt(at: Attribute);
      (* dereferencing of `at'; like '^'-operator *)

   PROCEDURE OffsetAt(at: Attribute; offset: Size);
      (* `at' must have an addressable addressing mode (i.e. not regMode);
	 the address described by `at' is then incremented by `offset';
	 `offset' may be negative
      *)

   PROCEDURE IndexAtReg(at: Attribute; indexreg: Reg; scalefactor: CARDINAL);
      (* generate []-operation for `at' with index in `indexreg'
	 `indexreg' is released afterwards
      *)

   PROCEDURE ReleaseAt(at: Attribute);
      (* release anything (registers and stack reservations) of `at' *)

   PROCEDURE ReturnAt(VAR at: Attribute);
      (* release anything (registers and stack reservations) of `at' and
	 dispose `at'
      *)


   (* for following constant operations
      at must be one of shortAT, intAT, or longAT
   *)

   PROCEDURE ConstMulReg(at: ArithmeticType; r: Reg; value: Integer);

   PROCEDURE ConstDivReg(at: ArithmeticType; r: Reg; value: Integer);

   PROCEDURE ConstModReg(at: ArithmeticType; r: Reg; value: Integer);

   PROCEDURE Mult(at: ArithmeticType; desat, opat: Attribute);
      (* generate code for `desat' := `desat' * `opat'
	 and release `opat'
	 `desat' is in regMode afterwards
      *)

   PROCEDURE Div(at: ArithmeticType; desat, opat: Attribute);
      (* generate code for `desat' := `desat' DIV `opat'
	 and release `opat'
	 `desat' is in regMode afterwards
      *)

   PROCEDURE Mod(at: ArithmeticType; desat, opat: Attribute);
      (* generate code for `desat' := `desat' MOD `opat'
	 and release `opat'
	 `desat' is in regMode afterwards
      *)

   PROCEDURE ArithType(type: Type) : ArithmeticType;
      (* return arithmetic type of `at' in dependance of attype *)

   PROCEDURE InvertTest(VAR t: TestType);
      (* invert test type (NOT), e.g. `le' becomes `gt' *)

   PROCEDURE ReverseTest(VAR t: TestType);
      (* reverse test type (exchange of operands), e.g. `le' becomes `ge' *)

   PROCEDURE GenTest(t: TestType; atype: ArithmeticType; dest: Label);
      (* condition codes are set; generate code for branching
	 to `dest' if `t' is true, i.e. "bcc test,dest"
      *)

   PROCEDURE SetBool(at: Attribute; destreg: Reg);
      (* set destreg according to condition codes *)


   PROCEDURE MoveBytes(from, to: Reg; nbytes: Size);
      (* addresses are in `from' and `to'; both registers
	 are released afterwards
      *)

   PROCEDURE MoveBytesAt(from, to: Attribute; nbytes: Size);
      (* both attributes are released afterwards *)

   PROCEDURE RangeCheck(at: Attribute; upperBound: Size);
      (* at^.mode = regMode;
	 check at^.reg for being inside [0..upperBound]
      *)

   PROCEDURE DynArrayCheck(indexat, lenat: Attribute);
      (* check indexat for being inside [0..lenat-1] *)

   PROCEDURE ConversionCheck(at: Attribute);
      (* at^.attype = intptr;
	 check at for being representable as SHORTINT
      *)

END GenBasicOps.
