(* Ulm's Oberon Compiler
   Copyright (c) 1989 by University of Ulm, SAI, D-W-7900 Ulm, Germany
   ----------------------------------------------------------------------------
   $Id: IdentSys.m2,v 0.1 1992/07/30 10:48:37 borchert Exp $
   ----------------------------------------------------------------------------
   $Log: IdentSys.m2,v $
   Revision 0.1  1992/07/30  10:48:37  borchert
   Initial revision

   ----------------------------------------------------------------------------
*)

IMPLEMENTATION MODULE IdentSys;		(* Martin Hasch, Jan 1989 *)

   FROM Exception IMPORT Fatal, Assert;
   FROM Memory IMPORT ALLOCATE;

(*
 *	identifier table structure:
 *
 *	   +-----+			 +-----+
 * table-->|  o------------------------->| NIL |
 *	   +---+-+-+---+---+---+---+	 +---+-+-+---+---+---+---+---+
 *	   | s | t | r | 1 | s | t |     | r | 2 | s | 3 |   |   |   |
 *	   +---+---+---+---+---+---+	 +---+---+---+---+---+---+---+
 *	     ^		     ^			   ^       ^
 *	  +--|--+	  +--|--+	  +-----+  |       |
 *  id1-->|  o  |   id2-->|  o  |   id3-->|  o-----+       |
 *     +->+-----+      +->+-----+    +--->+-----+      endoftable
 *     |  |len=4|      |  |  4  |    |	  |  2  |
 *     |  +-----+      |  +-----+    |	  +-----+
 *     |  |  o---------+  | NIL |    |	  | NIL |
 *     |  +-----+	  +-----+    |	  +-----+
 *     |			     |
 *     +-----------+		     |
 * hash-  +-----+--|--+-----+-----+--|--+-----+-----+-----+-----+
 * table  | NIL |  o  | NIL | NIL |  o  | NIL | NIL | NIL | NIL |
 *	  +-----+-----+-----+-----+-----+-----+-----+-----+-----+
 *)

   CONST
      tablepartsize = 512;		(* table space allocated at one time *)
      hashsize = 1031;

   TYPE
      HashIndex = [0..hashsize-1];

      LargeTable = POINTER TO TablePart;

      PartIndex = [0..tablepartsize-SIZE(LargeTable)-1];
      TablePart =
	 RECORD
	    nextpart: LargeTable;
	    content:  ARRAY PartIndex OF CHAR;
	 END;
      TablePosition =
	 RECORD
	    part: LargeTable;
	    pos:  PartIndex;
	 END;
      Identifier = POINTER TO IdentRec;		(* opaque EXPORTED *)
      IdentRec =
	 RECORD
	    tablepos: TablePosition;
	    size:     CARDINAL;
	    nextid:   Identifier;
	 END;

   VAR
      table: LargeTable;
      endoftable,				(* end of valid table + 1 *)
      currentrpos,
      currentwpos: TablePosition;	(* current read/write position *)
      pcharcount,
      gcharcount: CARDINAL;			(* Put-/GetIdentChar count *)
      currentid: Identifier;			(* set by GetIdentChar *)

      hashtable: ARRAY HashIndex OF Identifier;

   PROCEDURE PutIdentChar(ch: CHAR);				(* EXPORTED *)
   BEGIN
      INC(pcharcount);
      WITH currentwpos DO
	 part^.content[pos] := ch;
	 IF pos < MAX(PartIndex) THEN
	    INC(pos);
	 ELSE
	    IF part^.nextpart = NIL THEN
	       NewTablepart(part^.nextpart);
	    END;
	    part := part^.nextpart;
	    pos := 0;
	 END;
      END;
   END PutIdentChar;

   PROCEDURE IncPos(VAR tpos: TablePosition);
   BEGIN
      WITH tpos DO
	 IF pos < MAX(PartIndex) THEN
	    INC(pos);
	 ELSE
	    Assert( part # NIL );
	    part := part^.nextpart;
	    pos := 0;
	 END;
      END;
   END IncPos;

   PROCEDURE Hash(curpos: TablePosition; size: CARDINAL): HashIndex;
      CONST
	 factor = 99;
      VAR
	 result: HashIndex;
	 index: CARDINAL;
   BEGIN
      result := 0;
      FOR index := 1 TO size DO
	 WITH curpos DO
	    result := (result * factor + ORD(part^.content[pos])) MOD hashsize;
	 END;
	 IncPos(curpos);
      END;
      RETURN result
   END Hash;

   PROCEDURE NewIdentifier(VAR id: Identifier;
			   tpos: TablePosition; siz: CARDINAL;
			   nxt: Identifier);
   BEGIN
      ALLOCATE(id,SIZE(IdentRec));
      WITH id^ DO
	 tablepos := tpos;
	 size := siz;
	 nextid := nxt;
      END;
   END NewIdentifier;

   PROCEDURE IdEqual(pos1: TablePosition; siz1: CARDINAL;
		     pos2: TablePosition; siz2: CARDINAL): BOOLEAN;
      VAR
	 cnt: CARDINAL;
   BEGIN
      IF siz1 # siz2 THEN
	 RETURN FALSE
      END;
      FOR cnt := 1 TO siz1 DO
	 IF pos1.part^.content[pos1.pos] # pos2.part^.content[pos2.pos] THEN
	    RETURN FALSE
	 END;
	 IncPos(pos1);
	 IncPos(pos2);
      END;
      RETURN TRUE
   END IdEqual;

   PROCEDURE PutIdent(VAR id: Identifier);			(* EXPORTED *)
      VAR
	 hindex: HashIndex;
   BEGIN
      hindex := Hash(endoftable,pcharcount);
      id := hashtable[hindex];
      IF id = NIL THEN
	 NewIdentifier(id,endoftable,pcharcount,NIL);
	 hashtable[hindex] := id;
	 endoftable := currentwpos;
      ELSE
	 WHILE (id # NIL) &
	       ~IdEqual(endoftable,pcharcount,id^.tablepos,id^.size) DO
	    id := id^.nextid;
	 END;
	 IF id = NIL THEN
	    NewIdentifier(id,endoftable,pcharcount,hashtable[hindex]);
	    hashtable[hindex] := id;
	    endoftable := currentwpos;
	 ELSE
	    currentwpos := endoftable;
	 END;
      END;
      pcharcount := 0;
   END PutIdent;

   PROCEDURE GetIdent(id: Identifier; VAR ident: ARRAY OF CHAR);(* EXPORTED *)
      VAR
	 isize,
	 index: CARDINAL;
	 tpos: TablePosition;
   BEGIN
      Assert ( gcharcount = 0 );	(* was last ident read completely? *)
      tpos := id^.tablepos;
      isize := id^.size;
      IF isize > HIGH(ident)+1 THEN
	 isize := HIGH(ident)+1;
      END;
      index := 0;
      WHILE index < isize DO
	 ident[index] := tpos.part^.content[tpos.pos];
	 IncPos(tpos);
	 INC(index);
      END;
      IF index <= HIGH(ident) THEN
	 ident[index] := 0C;
      END;
   END GetIdent;

   PROCEDURE GetIdentChar(id: Identifier; VAR ch: CHAR);	(* EXPORTED *)
      (* last value of ch: 0C *)
   BEGIN
      IF id # currentid THEN
	 Assert ( gcharcount = 0 );	(* was last ident read completely? *)
	 currentid := id;
      END;
      IF gcharcount = 0 THEN
	 currentrpos := id^.tablepos;
      ELSE
	 IncPos(currentrpos);
      END;
      INC(gcharcount);
      IF gcharcount <= id^.size THEN
	 ch := currentrpos.part^.content[currentrpos.pos];
      ELSE
	 ch := 0C;
	 gcharcount := 0;
      END;
   END GetIdentChar;

   PROCEDURE NewTablepart(VAR part: LargeTable);
   BEGIN
      ALLOCATE(part,SIZE(TablePart));
      part^.nextpart := NIL;
   END NewTablepart;

   PROCEDURE InitHashtab;
      VAR
	 hindex: HashIndex;
   BEGIN
      FOR hindex := 0 TO hashsize-1 DO
	 hashtable[hindex] := NIL;
      END;
   END InitHashtab;

BEGIN
   InitHashtab;
   NewTablepart(table);
   endoftable.part := table;
   endoftable.pos := 0;
   currentwpos := endoftable;
   currentrpos := endoftable;
   pcharcount := 0;
   gcharcount := 0;
   currentid := NIL;
END IdentSys.
