(* Ulm's Oberon Compiler
   Copyright (c) 1989 by University of Ulm, SAI, D-W-7900 Ulm, Germany
   ----------------------------------------------------------------------------
   $Id: Messages.m2,v 0.1 1992/07/30 10:48:52 borchert Exp $
   ----------------------------------------------------------------------------
   $Log: Messages.m2,v $
   Revision 0.1  1992/07/30  10:48:52  borchert
   Initial revision

   ----------------------------------------------------------------------------
*)

IMPLEMENTATION MODULE Messages;		(* Martin Hasch, Feb 1989 *)
(*
 *	formatting of error messages
 *)

   FROM SYSTEM IMPORT BYTE, ADDRESS, ADR;
   FROM ASCII IMPORT nl;
   FROM Calendar IMPORT Time;
   FROM Conversions IMPORT ConvertCardinal, ConvertInteger;
   FROM Errno IMPORT errno;
   FROM IdentSys IMPORT Identifier, GetIdent;
   FROM Memory IMPORT CheckPtr;
   FROM SysPerror IMPORT GetErrorString;
   FROM TimeIO IMPORT SwriteTime;

(***** EXPORTED:
   VAR
      writeproc: PROCEDURE(CHAR);		(* must be initialized! *)
*****)

   CONST
      maxarg = 100;
      maxtabs = 12;
      fmtch = "%";

   TYPE
      Argument =
	 RECORD
	    adr: ADDRESS;
	    size: CARDINAL;
	 END;
      ArgIndex = [1..maxarg];

   VAR
      argstack: ARRAY ArgIndex OF Argument;
      top: CARDINAL;			(* =0 when arg stack is empty *)
      lostargs: CARDINAL;		(* >0 at arg stack overflow *)
      tabtable: ARRAY [1..maxtabs] OF CARDINAL;
      tabs: CARDINAL;

   PROCEDURE ClearTabs;
   BEGIN
      tabs := 0;
   END ClearTabs;

   PROCEDURE SetTab(tabpos: CARDINAL);
      (* (n-th call) define n-th tabulator position, left border = 1 *)
   BEGIN
      IF tabs < maxtabs THEN
	 INC(tabs);
	 tabtable[tabs] := tabpos;
      END;
   END SetTab;

   PROCEDURE PushArgument( (*read*)VAR arg: ARRAY OF BYTE);	(* EXPORTED *)
   BEGIN
      IF top < maxarg THEN
	 INC(top);
	 WITH argstack[top] DO
	    adr := ADR(arg);
	    size := HIGH(arg)+1;
	 END;
      ELSE
	 INC(lostargs);
      END;
   END PushArgument;

   PROCEDURE PrintFormat( (*read*)VAR fmt: ARRAY OF CHAR);
   (* EXPORTED *)
      (* within fmt, the following tokens are substituted *)
      (* by parameter values:				  *)
      (* %c CARDINAL *)
      (* %C CHAR *)
      (* %Q CHAR, print quotes around it *)
      (* %i INTEGER *)
      (* %I '"' Identifier '"' *)
      (* %s ARRAY OF CHAR *)
      (* %R "implementation restriction", no parameter required *)
      (* %r "Oberon revision", no parameter required *)
      (* %t Calendar.Time, print date and time *)
      (* %d Calendar.Time, print date only *)
      (* %E system error message, parameter = errno *)
      (* %S system error message, no parameter required *)
      (* %T (n-th occurence) fill in blanks up to n-th tabulator position *)
      (* %n newline *)
      (* %% % *)

      VAR
	 pos: CARDINAL;
	 tcount: CARDINAL;

      PROCEDURE Write(ch: CHAR);
      BEGIN
	 writeproc(ch);
	 IF ch = nl THEN
	    pos := 1;
	    tcount := 0;
	 ELSE
	    INC(pos);
	 END;
      END Write;

      PROCEDURE WriteString(VAR str: ARRAY OF CHAR);
	 VAR
	    index: CARDINAL;
      BEGIN
	 index := 0;
	 WHILE (index <= HIGH(str)) & (str[index] # 0C) DO
	    Write(str[index]);
	    INC(index);
	 END;
      END WriteString;

      PROCEDURE WriteConstString(str: ARRAY OF CHAR);
      BEGIN
	 WriteString(str);
      END WriteConstString;

      PROCEDURE WriteTab;
      BEGIN
	 INC(tcount);
	 IF tcount <= tabs THEN
	    WHILE pos < tabtable[tcount] DO
	       Write(" ");
	    END;
	 END;
      END WriteTab;

      PROCEDURE GoodArg(argsize: CARDINAL): BOOLEAN;
      BEGIN
	 IF lostargs > 0 THEN
	    DEC(lostargs);
	    RETURN FALSE
	 END;
	 RETURN (top > 0) & (argsize = argstack[top].size)
      END GoodArg;

      PROCEDURE PrintCard;
	 TYPE
	    CardPtr = POINTER TO CARDINAL;
	 VAR
	    cardptr: CardPtr;
	    cardstr: ARRAY [0..15] OF CHAR;
      BEGIN
	 IF GoodArg( SIZE(CARDINAL) ) THEN
	    cardptr := argstack[top].adr;
	    ConvertCardinal( cardptr^, 0, cardstr );
	    WriteString(cardstr);
	    DEC(top);
	 END;
      END PrintCard;

      PROCEDURE PrintInt;
	 TYPE
	    IntPtr = POINTER TO INTEGER;
	 VAR
	    intptr: IntPtr;
	    intstr: ARRAY [0..15] OF CHAR;
      BEGIN
	 IF GoodArg( SIZE(CARDINAL) ) THEN
	    intptr := argstack[top].adr;
	    ConvertInteger( intptr^, 0, intstr );
	    WriteString(intstr);
	    DEC(top);
	 END;
      END PrintInt;

      PROCEDURE PrintChar;
	 TYPE
	    CharPtr = POINTER TO CHAR;
	 VAR
	    charptr: CharPtr;
      BEGIN
	 IF GoodArg( SIZE(CHAR) ) THEN
	    charptr := argstack[top].adr;
	    Write( charptr^ );
	    DEC(top);
	 END;
      END PrintChar;

      PROCEDURE PrintQuotedChar;
	 TYPE
	    CharPtr = POINTER TO CHAR;
	 VAR
	    charptr: CharPtr;
	    quote: CHAR;
      BEGIN
	 IF GoodArg( SIZE(CHAR) ) THEN
	    charptr := argstack[top].adr;
	    IF charptr^ = '"' THEN
	       quote := "'";
	    ELSE
	       quote := '"';
	    END;
	    Write(quote);
	    Write( charptr^ );
	    Write(quote);
	    DEC(top);
	 END;
      END PrintQuotedChar;

      PROCEDURE PrintIdent;
	 CONST
	    maxlength = 256;
	 TYPE
	    IdentPtr = POINTER TO Identifier;
	 VAR
	    identptr: IdentPtr;
	    identstr: ARRAY [0..maxlength-1] OF CHAR;
      BEGIN
	 IF GoodArg( SIZE(Identifier) ) THEN
	    identptr := argstack[top].adr;
	    IF CheckPtr( ADDRESS(identptr^) ) THEN
	       identstr[maxlength-3] := 0C;
	       GetIdent( identptr^, identstr );
	       IF identstr[maxlength-3] # 0C THEN
		  identstr[maxlength-3] := ".";
		  identstr[maxlength-2] := ".";
		  identstr[maxlength-1] := ".";
	       END;
	    ELSE
	       identstr := "???";
	    END;
	    Write('"');
	    WriteString(identstr);
	    Write('"');
	    DEC(top);
	 END;
      END PrintIdent;

      PROCEDURE PrintString;
	 TYPE
	    StrPtr = POINTER TO ARRAY [0..0] OF CHAR;	(* any index allowed *)
	 VAR
	    strptr: StrPtr;
	    high,
	    index: CARDINAL;
      BEGIN
	 IF lostargs > 0 THEN
	    DEC(lostargs);
	 ELSIF top > 0 THEN
	    strptr := argstack[top].adr;
	    index := 0;
	    high := argstack[top].size - 1;
	    (* $T- (turn array index checks off) *)
	    WHILE (index <= high) & (strptr^[index] # 0C) DO
	       Write(strptr^[index]);
	       INC(index);
	    END;
	    (* $T= *)
	    DEC(top);
	 END;
      END PrintString;

      PROCEDURE PrintTimeAs(timefmt: ARRAY OF CHAR);
	 TYPE
	    TimePtr = POINTER TO Time;
	 VAR
	    timeptr: TimePtr;
	    timestr: ARRAY [0..23] OF CHAR;
      BEGIN
	 IF GoodArg( SIZE(Time) ) THEN
	    timeptr := argstack[top].adr;
	    SwriteTime( timestr, timefmt, timeptr^ );
	    WriteString(timestr);
	    DEC(top);
	 END;
      END PrintTimeAs;

      PROCEDURE WriteSysErrorNo(errno : CARDINAL);
	 VAR
	    errstr: ARRAY [0..31] OF CHAR;
      BEGIN
	 GetErrorString(errno, errstr);
	 WriteString(errstr);
      END WriteSysErrorNo;

      PROCEDURE PrintSysError;
	 TYPE
	    CardPtr = POINTER TO CARDINAL;
	 VAR
	    cardptr: CardPtr;
      BEGIN
	 IF GoodArg( SIZE(CARDINAL) ) THEN
	    cardptr := argstack[top].adr;
	    WriteSysErrorNo(cardptr^);
	    DEC(top);
	 END;
      END PrintSysError;

      VAR
	 index: CARDINAL;
	 readfmt: BOOLEAN;
   BEGIN
      index := 0;
      pos := 1;
      tcount := 0;
      readfmt := FALSE;
      WHILE (index <= HIGH(fmt)) & (fmt[index] # 0C) DO
	 IF readfmt THEN
	    CASE fmt[index] OF
	    |  "c":  PrintCard;
	    |  "C":  PrintChar;
	    |  "Q":  PrintQuotedChar;
	    |  "i":  PrintInt;
	    |  "I":  PrintIdent;
	    |  "s":  PrintString;
	    |  "t":  PrintTimeAs("%d.%m.%Y %H:%M:%S");
	    |  "d":  PrintTimeAs("%d.%m.%Y");
	    |  "E":  PrintSysError;
	    |  "R":  WriteConstString("implementation restriction");
	    |  "r":  WriteConstString("Oberon revision");
	    |  "S":  WriteSysErrorNo(errno);
	    |  "T":  WriteTab;
	    |  "n":  Write(nl);
	    |  fmtch:
		     Write(fmtch);
	    |  ELSE
		     Write(fmtch);
		     Write(fmt[index]);
	    END;
	    readfmt := FALSE;
	 ELSIF fmt[index] = fmtch THEN
	    readfmt := TRUE;
	 ELSE
	    Write(fmt[index]);
	 END;
	 INC(index);
      END;
   END PrintFormat;

BEGIN
   top := 0;
   lostargs := 0;
   tabs := 0;
END Messages.
