(* Ulm's Oberon Compiler
   Copyright (c) 1989 by University of Ulm, SAI, D-W-7900 Ulm, Germany
   ----------------------------------------------------------------------------
   $Id: Parser.m2,v 0.2 1993/06/09 14:49:58 borchert Exp $
   ----------------------------------------------------------------------------
   $Log: Parser.m2,v $
   Revision 0.2  1993/06/09  14:49:58  borchert
   interface of SymDef has been changed: ImportOurDefinition has
   to be called prior to the other imports

   Revision 0.1  1992/07/30  10:49:03  borchert
   Initial revision

   ----------------------------------------------------------------------------
*)

IMPLEMENTATION MODULE Parser;		(* Martin Hasch, Jan/Feb 1989 *)
					(* Oberon revision: mh Oct 1989 *)
				(* improved error recovery: mh Apr 1990 *)

   FROM IdentSys IMPORT Identifier;
   FROM Lex IMPORT
      GetSy, Symbol, sy, ident, Constval, cval, dummyid, GetSymString,
      Mark, Insert, InsertId, pos, previdpos, Skip, statementpos, PosStatement,
      CompilerOptions, AddLegalOptions, options;
   FROM Scan IMPORT ErrorType, Message1, Message2, Error, Error1, Error2, Pos;
   FROM Exception IMPORT Fatal, Assert;
   FROM ScanArgs IMPORT defunit;
   FROM Suffixes IMPORT definitionSX, moduleSX;
   FROM SymDef IMPORT
      Type, QualIdent, OpenScope, CloseScope, Import, AliasImport, DeclConst,
      NoType, UseType, ArrayOf, Record, AddField, EndRecord, DeclType,
      PointerTo, PointerToIdent, DynArrayOf, ProcedureType, AddParameterType,
      Function, GetConstVal, DeclVar, DeclProc, AddParameter, IsVar,
      ImportOurDefinition;
   FROM CodeGen IMPORT
      Attribute, InitConstAt, InitIdentAt, EvalConst, UnaryOp, BinaryOp, Index,
      Typeguard, Qualified, FieldSelection, IsVarNoProcvar, Dereference,
      InitEmptyAt, Set, ProcDesignator, Parameter, FunctionCall, InitNILAt,
      Assignment, IfThen, ElsifThen, Else, EndIf, CaseOf, CaseLabels, CaseElse,
      EndCase, WhileDo, EndWhile, Repeat, Until, Loop, ExitLoop, EndLoop,
      Return, ReturnExpr, WithDo, EndWith, ProcedureCall,
      Labels, InitLabel, AddLabel, AddRange, ProcedureBegin, ProcedureEnd,
      ModuleHeader, ModuleBegin, ModuleEnd;
   FROM Memory IMPORT ALLOCATE, DEALLOCATE;

   CONST
      revisionOpt = "R";
      rev90Opt = "I";

   TYPE
      SymSet = SET OF Symbol;
      IdList = POINTER TO IdListRec;
      IdListRec =
	 RECORD
	    ident: Identifier;
	    nextid: IdList;
	    idpos: Pos;
	 END;
      ParseMesg = (genM, defM, modM);
   
   VAR
      noaliasmesg: BOOLEAN;		(* prevent multiple messages *)

   CONST
      reloperators = SymSet{ eql, neq, lst, leq, grt, geq, inSY, isSY };
      addoperators = SymSet{ plus, minus, orSY };
      muloperators = SymSet{ times, slash, divSY, modSY, ampersand };
      expstartsyms = SymSet{ plus, minus, intcon..longrealcon, charcon,
			stringcon, nilSY, lbrace, identSY, lparen, tilde };
      (*TBD*)
      statementsyms = SymSet{ identSY, ifSY, caseSY, whileSY, repeatSY,
			loopSY, withSY, exitSY, returnSY, semicolon };
      typesyms = SymSet{ identSY, arraySY, recordSY, pointerSY, procedureSY };

   PROCEDURE Expected(sym: Symbol);
      VAR
	 symstr: ARRAY [0..15] OF CHAR;
   BEGIN
      Insert;
      IF sym = identSY THEN
	 InsertId;
      END;
      GetSymString(sym,symstr);
      Error1("%s expected", symstr);
   END Expected;

   PROCEDURE TestGetSy(sym: Symbol);
   BEGIN
      IF sy = sym THEN
	 GetSy;
      ELSE
	 Expected(sym);
      END;
   END TestGetSy;

   PROCEDURE Test((*readonly*)VAR expectedsyms: SymSet; msg: ParseMesg);
      VAR
	 symstr: ARRAY [0..15] OF CHAR;
   BEGIN
      IF ~(sy IN expectedsyms) THEN
	 Mark;
	 CASE msg OF
	 |  genM: GetSymString(sy, symstr);
		  Error1("%s not expected", symstr);
	 |  defM: Error("error in definition");
	 |  modM: Error("error in module");
	 |  ELSE  Assert(FALSE)
	 END;
	 REPEAT
	    Skip;
	    GetSy;
	 UNTIL sy IN expectedsyms;
      END;
   END Test;

   PROCEDURE Parse;
      TYPE
	 DefinitionType = (none, const, type, var, procedure);

      PROCEDURE TestOrder(VAR lastdef: DefinitionType; newdef: DefinitionType);
	 (* Oberon revision: rule for declaration order relaxed *)
      BEGIN
	 IF (lastdef = procedure) & (newdef < procedure) THEN
	    Mark;
	    Error(
"procedure declarations must not be followed by other declarations");
	 END;
	 lastdef := newdef;
      END TestOrder;

      PROCEDURE IdentList(VAR idlist: IdList);
	 VAR
	    lastid: IdList;
      BEGIN
	 IF sy = identSY THEN
	    Mark;
	    NEW(lastid);
	    lastid^.ident := ident;
	    lastid^.idpos := pos;
	    idlist := lastid;
	    GetSy;
	    WHILE sy = comma DO
	       GetSy;
	       IF sy = identSY THEN
		  Mark;
		  NEW(lastid^.nextid);
		  lastid := lastid^.nextid;
		  lastid^.ident := ident;
		  lastid^.idpos := pos;
		  GetSy;
	       ELSE
		  Expected(identSY);
	       END;
	    END;
	    lastid^.nextid := NIL;
	 ELSE
	    Expected(identSY);
	    idlist := NIL;
	 END;
      END IdentList;

      PROCEDURE NextIdent(VAR idlist: IdList);
	 VAR
	    oldid: IdList;
      BEGIN
	 oldid := idlist;
	 idlist := idlist^.nextid;
	 DISPOSE(oldid);
      END NextIdent;

      PROCEDURE ImportList;

	 PROCEDURE SingleImport;
	    VAR
	       name: Identifier;
	 BEGIN
	    IF sy = identSY THEN
	       name := ident;
	       GetSy;
	       IF (sy = colon) OR (sy = becomes) THEN
		  IF ~noaliasmesg &
			((sy = becomes) # (rev90Opt IN options)) THEN
		     Mark;
		     Error('%r 1990: ":" replaced by ":=" in alias import');
		     noaliasmesg := TRUE;
		  END;
		  GetSy;
		  IF sy = identSY THEN
		     Mark;
		     AliasImport(name,ident);
		     GetSy;
		  ELSE
		     Expected(identSY);
		  END;
	       ELSE
		  Import(name);
	       END;
	    ELSE
	       Expected(identSY);
	    END;
	 END SingleImport;

      BEGIN
	 SingleImport;
	 WHILE sy IN SymSet{comma,identSY} DO
	    TestGetSy(comma);
	    SingleImport;
	 END;
	 TestGetSy(semicolon);
      END ImportList;

      PROCEDURE ConstDeclaration;
	 VAR
	    constval: Constval;
	    constid: Identifier;
	    constpos: Pos;
      BEGIN
	 WHILE sy = identSY DO
	    constid := ident;
	    Mark; constpos := pos;
	    GetSy;
	    TestGetSy(eql);
	    ConstExpression(constval);
	    previdpos := constpos;
	    DeclConst(constid,constval);
	    TestGetSy(semicolon);
	 END;
      END ConstDeclaration;

      PROCEDURE QualifiedIdent(VAR qualid: QualIdent);
      BEGIN
	 Assert( sy = identSY );
	 WITH qualid DO
	    idname := ident;
	    qualified := FALSE;
	    GetSy;
	    IF sy = period THEN
	       GetSy;
	       IF sy = identSY THEN
		  qualified := TRUE;
		  modname := idname;
		  idname := ident;
		  GetSy;
	       ELSE
		  Expected(identSY);
	       END;
	    END;
	 END; (*WITH qualid*)
      END QualifiedIdent;

      PROCEDURE QualIdentType(VAR qualidtype: Type);
	 VAR
	    qualident: QualIdent;
      BEGIN
	 QualifiedIdent(qualident);
	 UseType(qualident, qualidtype);
      END QualIdentType;

      PROCEDURE GetQualIdent(VAR qualidat: Attribute);
      BEGIN
	 IF sy = identSY THEN
	    Mark;
	    InitIdentAt(ident, qualidat);
	    GetSy;
	    IF sy = period THEN
	       GetSy;
	       IF sy = identSY THEN
		  Mark;
		  Qualified(ident, qualidat);
		  GetSy;
	       ELSE
		  Expected(identSY);
	       END;
	    END;
	 ELSE
	    Insert;
	    InitEmptyAt(qualidat);
	    Error("qualident expected");
	 END;
      END GetQualIdent;

      PROCEDURE FormalType(VAR ftype: Type);
	 VAR
	    elemtype: Type;
      BEGIN
	 IF sy = arraySY THEN
	    GetSy;
	    TestGetSy(ofSY);
	    FormalType(elemtype);
	    DynArrayOf(elemtype, ftype);
	 ELSIF sy = identSY THEN
	    QualIdentType(ftype);
	 ELSE
	    Insert;
	    ftype := NoType();
	    Error("formal type expected");
	 END;
      END FormalType;

      PROCEDURE TypeConstruction(VAR type: Type);

	 PROCEDURE ArrayType(VAR arraytype: Type);
	    VAR
	       dim: Constval;
	       elemtype: Type;
	 BEGIN
	    ConstExpression(dim);
	    IF sy = comma THEN
	       GetSy;
	       ArrayType(elemtype);
	    ELSE
	       TestGetSy(ofSY);
	       TypeConstruction(elemtype);
	    END;
	    ArrayOf(dim,elemtype,arraytype);
	 END ArrayType;

	 PROCEDURE RecordType(VAR rectype: Type);
	    VAR
	       basetype: Type;

	    PROCEDURE FieldListSequence;

	       PROCEDURE FieldList;
		  VAR
		     idlist: IdList;
		     typeconstr: Type;
	       BEGIN
		  IF sy = identSY THEN
		     IdentList(idlist); (* NIL in case of error *)
		     TestGetSy(colon);
		     TypeConstruction(typeconstr);
		     WHILE idlist # NIL DO
			previdpos := idlist^.idpos;
			AddField(idlist^.ident,typeconstr, rectype);
			NextIdent(idlist);
		     END;
		  END;
	       END FieldList;

	    BEGIN
	       FieldList;
	       WHILE sy = semicolon DO
		  GetSy;
		  FieldList;
	       END;
	       EndRecord(rectype);
	    END FieldListSequence;

	 BEGIN
	    basetype := NoType();
	    IF sy = lparen THEN
	       GetSy;
	       IF sy = identSY THEN
		  QualIdentType(basetype);
	       ELSE
		  Expected(identSY);		(* basetype remains NoType() *)
	       END;
	       TestGetSy(rparen);
	    END;
	    Record(basetype,rectype);
	    FieldListSequence;
	    TestGetSy(endSY);
	 END RecordType;

	 PROCEDURE PointerType(VAR pointype: Type);
	    VAR
	       reftype: Type;
	       qualid: QualIdent;
	 BEGIN
	    TestGetSy(toSY);
	    IF sy = identSY THEN
	       QualifiedIdent(qualid);
	       IF qualid.qualified THEN
		  UseType(qualid,reftype);
		  PointerTo(reftype, pointype);
	       ELSE
		  PointerToIdent(qualid.idname, pointype);
	       END;
	    ELSE
	       TypeConstruction(reftype);
	       PointerTo(reftype, pointype);
	    END;
	 END PointerType;

	 PROCEDURE ProcType(VAR proctype: Type);
	    VAR
	       rtype: Type;				(* result type *)

	    PROCEDURE VarFormalType;
	       VAR
		  isvar: BOOLEAN;
		  ftype: Type;				(* type in FT-list *)
	    BEGIN
	       isvar := sy = varSY;
	       IF isvar THEN
		  GetSy;
	       END;
	       FormalType(ftype);
	       AddParameterType(isvar,ftype, proctype);
	    END VarFormalType;

	 BEGIN
	    ProcedureType(proctype);
	    IF sy = lparen THEN
	       GetSy;
	       IF sy = rparen THEN
		  GetSy;
	       ELSE
		  VarFormalType;
		  WHILE sy = comma DO
		     GetSy;
		     VarFormalType;
		  END;
		  TestGetSy(rparen);
	       END;
	       IF sy = colon THEN
		  GetSy;
		  IF sy = identSY THEN
		     QualIdentType(rtype);
		     Function(rtype, proctype);
		  ELSE
		     Expected(identSY);
		  END;
	       END;
	    END;
	 END ProcType;

      BEGIN
	 CASE sy OF
	 |  identSY:
	       QualIdentType(type);
	 |  arraySY:
	       GetSy;
	       ArrayType(type);
	 |  recordSY:
	       GetSy;
	       RecordType(type);
	 |  pointerSY:
	       GetSy;
	       PointerType(type);
	 |  procedureSY:
	       GetSy;
	       IF revisionOpt IN options THEN
		  FormalParameters(type);
	       ELSE
		  ProcType(type);
	       END;
	 |  ELSE
	       type := NoType();
	       Error("error in type definition");
	 END;
      END TypeConstruction;

      PROCEDURE TypeDeclaration;
	 VAR
	    typeid: Identifier;
	    typeconstr: Type;
	    typepos: Pos;
      BEGIN
	 WHILE sy = identSY DO
	    typeid := ident;
	    Mark; typepos := pos;
	    GetSy;
	    TestGetSy(eql);
	    TypeConstruction(typeconstr);
	    previdpos := typepos;
	    DeclType(typeid,typeconstr);
	    TestGetSy(semicolon);
	 END;
      END TypeDeclaration;

      PROCEDURE VarDeclaration;
	 VAR
	    idlist: IdList;
	    vartype: Type;
      BEGIN
	 WHILE sy = identSY DO
	    IdentList(idlist);
	    TestGetSy(colon);
	    TypeConstruction(vartype);
	    WHILE idlist # NIL DO
	       previdpos := idlist^.idpos;
	       DeclVar(idlist^.ident,vartype);
	       NextIdent(idlist);
	    END;
	    TestGetSy(semicolon);
	 END;
      END VarDeclaration;

      PROCEDURE FormalParameters(VAR proctype: Type);
	 VAR
	    resulttype: Type;

	 PROCEDURE FPSection;		(* modifies proctype *)
	    VAR
	       isvar: BOOLEAN;
	       idlist: IdList;
	       parmtype: Type;
	 BEGIN
	    isvar := sy = varSY;
	    IF isvar THEN
	       GetSy;
	    END;
	    IdentList(idlist);
	    TestGetSy(colon);
	    FormalType(parmtype);
	    WHILE idlist # NIL DO
	       previdpos := idlist^.idpos;
	       AddParameter(isvar,idlist^.ident,parmtype, proctype);
	       NextIdent(idlist);
	    END;
	 END FPSection;

      BEGIN
	 ProcedureType(proctype);
	 IF sy = lparen THEN
	    GetSy;
	    IF sy IN SymSet{ varSY, identSY } THEN
	       FPSection;
	       WHILE sy = semicolon DO
		  GetSy;
		  FPSection;
	       END;
	    END;
	    TestGetSy(rparen);
	    IF sy = colon THEN
	       GetSy;
	       IF sy = identSY THEN
		  Mark;
		  QualIdentType(resulttype);
		  Function(resulttype, proctype);
	       ELSE
		  Expected(identSY);
	       END;
	    END;
	 END;
      END FormalParameters;

      PROCEDURE Designator(VAR desat: Attribute);
	 VAR
	    desid: Identifier;

	 PROCEDURE IndexConstruction;			(* modifies desat *)
	    VAR
	       indexat: Attribute;
	 BEGIN
	    Expression(indexat);
	    Index(indexat, desat);
	    WHILE sy = comma DO
	       GetSy;
	       Expression(indexat);
	       Index(indexat, desat);
	    END;
	    TestGetSy(rbracket);
	 END IndexConstruction;

	 PROCEDURE TypeGuard;				(* modifies desat *)
	    VAR
	       typeat: Attribute;
	 BEGIN
	    GetQualIdent(typeat);
	    Typeguard(typeat, desat);
	    TestGetSy(rparen);
	 END TypeGuard;

      BEGIN
	 Assert( sy = identSY );
	 desid := ident;
	 Mark;
	 InitIdentAt(ident, desat);
	 GetSy;
	 IF (sy = period) & ~IsVar(desid) THEN
	    GetSy;
	    IF sy = identSY THEN
	       Mark;
	       Qualified(ident, desat);
	       GetSy;
	    ELSE
	       Expected(identSY);
	    END;
	 END;
	 WHILE sy IN SymSet{ period, lbracket, lparen, arrow } DO
	    CASE sy OF
	    |  period:
		     GetSy;
		     IF sy = identSY THEN
			Mark;
			FieldSelection(ident, desat);
			GetSy;
		     ELSE
			Insert;
			Error("field selector expected");
		     END;
	    |  lbracket:
		     GetSy;
		     IndexConstruction;
	    |  lparen:
		     IF IsVarNoProcvar(desat) THEN
			GetSy;
			TypeGuard;
		     ELSE
			RETURN	(* lparen belongs to actual parameters *)
		     END;
	    |  arrow:
		     GetSy;
		     Dereference(desat);
	    |  ELSE  Assert( FALSE );
	    END;
	 END;
      END Designator;

      PROCEDURE ActualParameters(VAR desat: Attribute);
	 (* transforming the designator desat into a procedure designator *)
	 (* and appending actual parameters, if there are any; no call!   *)

	 VAR
	    parmat: Attribute;
      BEGIN
	 ProcDesignator(desat);
	 IF sy = lparen THEN
	    GetSy;
	    IF sy = rparen THEN
	       GetSy;
	    ELSE
	       Expression(parmat);
	       Parameter(parmat,desat);
	       WHILE sy = comma DO
		  GetSy;
		  Expression(parmat);
		  Parameter(parmat,desat);
	       END;
	       TestGetSy(rparen);
	    END;
	 END;
      END ActualParameters;

      PROCEDURE Expression(VAR expat: Attribute);
	 VAR
	    left, right: Attribute;
	    operation: Symbol;

	 PROCEDURE SimpleExpression(VAR simpat: Attribute);
	    VAR
	       left, right: Attribute;
	       operation: Symbol;

	    PROCEDURE Term(VAR termat: Attribute);
	       VAR
		  left, right: Attribute;
		  operation: Symbol;

	       PROCEDURE Factor(VAR factat: Attribute);

		  PROCEDURE SetConstruction;		(* modifies factat *)
		     VAR
			left, right: Attribute;

		     PROCEDURE Element(VAR elemat: Attribute);
			VAR
			   left, right: Attribute;
		     BEGIN
			Expression(elemat);
			IF sy = range THEN
			   GetSy;
			   left := elemat;
			   Expression(right);
			   BinaryOp(range,left,right, elemat);
			END;
		     END Element;

		  BEGIN
		     IF sy = rbrace THEN
			InitEmptyAt(factat);
		     ELSE
			Element(factat);
			WHILE sy = comma DO
			   GetSy;
			   left := factat;
			   Element(right);
			   BinaryOp(comma,left,right, factat);
			END;
		     END;
		     TestGetSy(rbrace);
		     Set(factat);
		  END SetConstruction;

	       BEGIN
		  CASE sy OF
		  |  stringcon..longrealcon:
			Mark;
			InitConstAt(cval,factat);
			GetSy;
		  |  setcon,boolcon:
			Assert( FALSE );
		  |  nilSY:
			GetSy;
			InitNILAt(factat);
		  |  lbrace:
			GetSy;
			SetConstruction;
		  |  identSY:
			Designator(factat);
			IF sy = lparen THEN
			   ActualParameters(factat);
			   FunctionCall(factat);
			END;
		  |  lparen:
			GetSy;
			Expression(factat);
			TestGetSy(rparen);
		  |  tilde:
			GetSy;
			Factor(factat);
			UnaryOp(tilde, factat);
		  |  ELSE
			InitEmptyAt(factat);
			Error("error in expression");
		  END;
	       END Factor;

	    BEGIN
	       Factor(termat);
	       WHILE sy IN muloperators DO
		  operation := sy;
		  GetSy;
		  left := termat;
		  Factor(right);
		  BinaryOp(operation,left,right, termat);
	       END;
	    END Term;

	 BEGIN
	    IF sy IN SymSet{ plus, minus } THEN
	       operation := sy;
	       GetSy;
	       Term(simpat);
	       UnaryOp(operation, simpat);
	    ELSE
	       Term(simpat);
	    END;
	    WHILE sy IN addoperators DO
	       operation := sy;
	       GetSy;
	       left := simpat;
	       Term(right);
	       BinaryOp(operation,left,right, simpat);
	    END;
	 END SimpleExpression;

      BEGIN
	 SimpleExpression(expat);
	 IF sy IN reloperators THEN
	    operation := sy;
	    GetSy;
	    left := expat;
	    SimpleExpression(right);
	    BinaryOp(operation,left,right, expat);
	 END;
      END Expression;

      PROCEDURE ConstExpression(VAR val: Constval);
	 VAR
	    constat: Attribute;
      BEGIN
	 Expression(constat);
	 EvalConst(constat,val);
      END ConstExpression;

      PROCEDURE EndIdent(name: Identifier);
      BEGIN
	 TestGetSy(endSY);
	 IF sy = identSY THEN
	    IF (ident # name) & (name # dummyid) THEN
	       Mark;
	       Error2("%I does not match to %I",ident,name);
	    END;
	    GetSy;
	 ELSE
	    Expected(identSY);
	 END;
      END EndIdent;

      PROCEDURE Definition;
	 VAR
	    modulename: Identifier;
	    definitionsyms: SymSet;

	 PROCEDURE DefSequence;
	    VAR
	       lastdef: DefinitionType;

	    PROCEDURE ProcedureDeclaration;	(* within definition file! *)
	       VAR
		  procname: Identifier;
		  proctype: Type;
		  procpos: Pos;
	    BEGIN
	       IF sy = times THEN	(* the asterisk may be ignored *)
		  GetSy;
	       END;
	       IF sy = identSY THEN
		  procname := ident;
		  Mark; procpos := pos;
		  GetSy;
		  FormalParameters(proctype);
		  previdpos := pos;
		  DeclProc(procname, (*forward=*) FALSE, proctype);
	       ELSE
		  Expected(identSY);
	       END;
	       TestGetSy(semicolon);
	    END ProcedureDeclaration;

	 BEGIN
	    lastdef := none;
	    WHILE sy IN SymSet{ constSY, varSY, typeSY, procedureSY } DO
	       CASE sy OF
	       |  constSY:
		     TestOrder(lastdef,const);
		     GetSy;
		     ConstDeclaration;
	       |  typeSY:
		     TestOrder(lastdef,type);
		     GetSy;
		     TypeDeclaration;
	       |  varSY:
		     TestOrder(lastdef,var);
		     GetSy;
		     VarDeclaration;
	       |  procedureSY:
		     lastdef := procedure;
		     GetSy;
		     ProcedureDeclaration;
	       END;
	       Test(definitionsyms, defM);
	    END;
	 END DefSequence;

      BEGIN (* Definition *)
	 IF sy = definitionSY THEN
	    GetSy;
	 ELSIF sy = moduleSY THEN
	    Message1(fatal,
	       'suffix "%s" only for definition files', definitionSX)
	 ELSE
	    Expected(definitionSY);
	 END;
	 IF sy = identSY THEN
	    modulename := ident;
	    GetSy;
	 ELSE
	    Expected(identSY);
	    modulename := dummyid;
	 END;
	 TestGetSy(semicolon);
	 definitionsyms :=
	    SymSet{importSY,constSY,typeSY,varSY,procedureSY,endSY,eop};
	 Test(definitionsyms, defM);
	 EXCL(definitionsyms, importSY);
	 IF sy = importSY THEN
	    GetSy;
	    ImportList;
	    Test(definitionsyms, defM);
	 END;
	 Insert;
	 OpenScope(modulename);
	 DefSequence;
	 EndIdent(modulename);
	 TestGetSy(period);
	 Mark;
	 IF sy # eop THEN
	    Expected(eop);
	    REPEAT
	       GetSy;
	    UNTIL sy = eop;
	 END;
	 CloseScope;
      END Definition;

      PROCEDURE Module;
	 VAR
	    modulename: Identifier;
	    modulesyms: SymSet;

	 PROCEDURE StatementSequence;

	    PROCEDURE Statement;

	       PROCEDURE CallOrAssignment;
		  VAR
		     desat: Attribute;
		     expat: Attribute;
	       BEGIN
		  Designator(desat);
		  IF sy = becomes THEN
		     GetSy;
		     Expression(expat);
		     Assignment(desat,expat);
		  ELSE
		     ActualParameters(desat);
		     ProcedureCall(desat);
		  END;
	       END CallOrAssignment;

	       PROCEDURE IfStatement;
		  VAR
		     conditionat: Attribute;
	       BEGIN
		  Expression(conditionat);
		  TestGetSy(thenSY);
		  IfThen(conditionat);
		  StatementSequence;
		  WHILE sy = elsifSY DO
		     GetSy;
		     Expression(conditionat);
		     TestGetSy(thenSY);
		     ElsifThen(conditionat);
		     StatementSequence;
		  END;
		  IF sy = elseSY THEN
		     GetSy;
		     Else;
		     StatementSequence;
		  END;
		  TestGetSy(endSY);
		  EndIf;
	       END IfStatement;

	       PROCEDURE CaseStatement;
		  VAR
		     expat: Attribute;
		     labels: Labels;

		  PROCEDURE Case;

		     PROCEDURE CaseLabelList;		(* creates labels *)

			PROCEDURE CLabels;		(* modifies labels *)
			   VAR
			      left, right: Attribute;
			BEGIN
			   Expression(left);
			   IF sy = range THEN
			      GetSy;
			      Expression(right);
			      AddRange(left,right, labels);
			   ELSE
			      AddLabel(left, labels);
			   END;
			END CLabels;

		     BEGIN
			InitLabel(labels);
			CLabels;
			WHILE sy = comma DO
			   GetSy;
			   CLabels;
			END;
		     END CaseLabelList;

		  BEGIN
		     IF ~(sy IN SymSet{bar, elseSY, endSY} ) THEN
			CaseLabelList;
			TestGetSy(colon);
			CaseLabels(labels);
			StatementSequence;
		     END;
		  END Case;

	       BEGIN
		  Expression(expat);
		  TestGetSy(ofSY);
		  CaseOf(expat, labels);
		  Case;
		  WHILE sy = bar DO
		     GetSy;
		     Case;
		  END;
		  IF sy = elseSY THEN
		     GetSy;
		     CaseElse(labels);
		     StatementSequence;
		  END;
		  TestGetSy(endSY);
		  EndCase(labels);
	       END CaseStatement;

	       PROCEDURE WhileStatement;
		  VAR
		     condat: Attribute;
	       BEGIN
		  Expression(condat);
		  TestGetSy(doSY);
		  WhileDo(condat);
		  StatementSequence;
		  TestGetSy(endSY);
		  EndWhile;
	       END WhileStatement;

	       PROCEDURE RepeatStatement;
		  VAR
		     condat: Attribute;
	       BEGIN
		  Repeat;
		  StatementSequence;
		  TestGetSy(untilSY);
		  Expression(condat);
		  Until(condat);
	       END RepeatStatement;

	       PROCEDURE LoopStatement;
	       BEGIN
		  Loop;
		  StatementSequence;
		  TestGetSy(endSY);
		  EndLoop;
	       END LoopStatement;

	       PROCEDURE ReturnStatement;
		  VAR
		     expat: Attribute;
	       BEGIN
		  IF sy IN expstartsyms THEN
		     Expression(expat);
		     ReturnExpr(expat);
		  ELSE
		     Return;
		  END;
	       END ReturnStatement;

	       PROCEDURE WithStatement;
		  VAR
		     varat, typeat: Attribute;
	       BEGIN
		  GetQualIdent(varat);
		  TestGetSy(colon);
		  GetQualIdent(typeat);
		  TestGetSy(doSY);
		  WithDo(varat,typeat);
		  StatementSequence;
		  EndWith(varat);
		  TestGetSy(endSY);
	       END WithStatement;

	    BEGIN
	       PosStatement;
	       CASE sy OF
	       |  identSY:
		     CallOrAssignment;
	       |  ifSY:
		     GetSy;
		     IfStatement;
	       |  caseSY:
		     GetSy;
		     CaseStatement;
	       |  whileSY:
		     GetSy;
		     WhileStatement;
	       |  repeatSY:
		     GetSy;
		     RepeatStatement;
	       |  loopSY:
		     GetSy;
		     LoopStatement;
	       |  withSY:
		     GetSy;
		     WithStatement;
	       |  exitSY:
		     GetSy;
		     ExitLoop;
	       |  returnSY:
		     GetSy;
		     ReturnStatement;
	       |  ELSE
		     (* empty statement *)
	       END;
	    END Statement;

	 BEGIN
	    Statement;
	    WHILE sy = semicolon DO
	       GetSy;
	       Statement;
	    END;
	 END StatementSequence;

	 PROCEDURE DeclarationSequence;
	    VAR
	       lastdef: DefinitionType;

	    PROCEDURE ProcedureDeclaration;	(* within module file *)
	       (* includes procedure heading and forward declaration *)

	       VAR
		  procname: Identifier;
		  procnameat: Attribute;
		  proctype: Type;
		  procpos: Pos;
		  forward: BOOLEAN;

	       PROCEDURE ProcedureBody;
	       BEGIN
		  DeclarationSequence;
		  ProcedureBegin(procnameat);
		  IF sy = beginSY THEN
		     GetSy;
		     StatementSequence;
		  END;
		  ProcedureEnd(procnameat);
	       END ProcedureBody;

	    BEGIN
	       forward := sy = arrow;
	       IF forward THEN
		  GetSy;
	       ELSIF sy = times THEN
		  GetSy;			(* ignore asterisk! *)
	       END;
	       IF sy = identSY THEN
		  procname := ident;
		  Mark; procpos := pos;
		  GetSy;
	       ELSE
		  Expected(identSY);
		  procpos := pos;
		  procname := dummyid;
	       END;
	       FormalParameters(proctype);
	       previdpos := procpos;
	       IF forward THEN
		  IF procname # dummyid THEN
		     DeclProc(procname,forward,proctype);
		  END;
	       ELSE
		  DeclProc(procname,forward,proctype);
		  TestGetSy(semicolon);
		  Insert;
		  InitIdentAt(procname, procnameat);
		  OpenScope(procname);
		  ProcedureBody;
		  CloseScope;
		  EndIdent(procname);
	       END;
	       TestGetSy(semicolon);
	    END ProcedureDeclaration;

	 BEGIN
	    lastdef := none;
	    WHILE sy IN SymSet{ constSY, varSY, typeSY, procedureSY } DO
	       CASE sy OF
	       |  constSY:
		     TestOrder(lastdef,const);
		     GetSy;
		     ConstDeclaration;
	       |  typeSY:
		     TestOrder(lastdef,type);
		     GetSy;
		     TypeDeclaration;
	       |  varSY:
		     TestOrder(lastdef,var);
		     GetSy;
		     VarDeclaration;
	       |  procedureSY:
		     lastdef := procedure;
		     GetSy;
		     ProcedureDeclaration;
	       END;
	       Test(modulesyms, modM);
	    END;
	 END DeclarationSequence;

      BEGIN
	 IF sy = moduleSY THEN
	    GetSy;
	 ELSIF sy = definitionSY THEN
	    Message1(fatal, 'suffix "%s" only for module files', moduleSX)
	 ELSE
	    Expected(moduleSY);
	 END;
	 IF sy = identSY THEN
	    modulename := ident;
	    ImportOurDefinition(modulename);
	    GetSy;
	 ELSE
	    Expected(identSY);
	    modulename := dummyid;
	    Fatal("no module name specified")
	 END;
	 TestGetSy(semicolon);
	 modulesyms :=
	    SymSet{importSY,constSY,typeSY,varSY,procedureSY,beginSY,endSY,eop};
	 Test(modulesyms, modM);
	 EXCL(modulesyms, importSY);
	 IF sy = importSY THEN
	    GetSy;
	    ImportList;
	    Test(modulesyms, modM);
	 END;
	 Insert;
	 OpenScope(modulename);
	 ModuleHeader;
	 DeclarationSequence;
	 ModuleBegin;
	 IF sy = beginSY THEN
	    GetSy;
	    StatementSequence;
	 END;
	 ModuleEnd;
	 EndIdent(modulename);
	 TestGetSy(period);
	 Mark;
	 IF sy # eop THEN
	    Expected(eop);
	    REPEAT
	       GetSy;
	    UNTIL sy = eop;
	 END;
	 CloseScope;
      END Module;

   BEGIN
      GetSy;
      IF defunit THEN
	 Definition;
      ELSE
	 Module;
      END;
   END Parse;

BEGIN
   AddLegalOptions( CompilerOptions{ revisionOpt, rev90Opt } );
   noaliasmesg := FALSE;
END Parser.
