(* Ulm's Oberon Compiler
   Copyright (c) 1989 by University of Ulm, SAI, D-W-7900 Ulm, Germany
   ----------------------------------------------------------------------------
   $Id: RegMan.m2,v 0.1 1992/07/30 10:49:11 borchert Exp $
   ----------------------------------------------------------------------------
   $Log: RegMan.m2,v $
   Revision 0.1  1992/07/30  10:49:11  borchert
   Initial revision

   ----------------------------------------------------------------------------
*)

IMPLEMENTATION MODULE RegMan; (* AFB 4/89 *)

   FROM Attributes IMPORT top, Reg, RegSet;
   FROM EmitCode IMPORT StrEmit2, Emit2;
   FROM Exception IMPORT Assert;
   FROM Machine IMPORT bitsperword;
   FROM Memory IMPORT ALLOCATE, DEALLOCATE;
   FROM Mnemonics IMPORT Mnemonic;
   FROM Scan IMPORT Error;
   FROM SymTab IMPORT Ident;

   (* (* imported from Attributes *)
      Reg = (d0, d1, d2, d3, d4, d5, d6, d7,
	     a0, a1, a2, a3, a4, a5, a6, a7,
	     illegal, pc, ccr,
	     (* MC68881 *)
	     fp0, fp1, fp2, fp3, fp4, fp5, fp6, fp7,
	     fpcr, fpsr);
      
      available registers:

      data registers:	 d1..d7
      address registers: a0..a5
      fp registers:	 fp1..fp7

      d0 and fp0 are for temporary use (incl. return)
      a6 and a7 are reserved for stack management
   *)

   TYPE
      RegMode = (freeR, releasedR, lockedR, staticR, addrR, valueR);
	 (* freeR:	free register with unknown contents
	    releasedR:  free but saved register
	    lockedR:    locked register
	    staticR:    free register containing a static link
	    addrR:      free register containing an address
	    valueR:     free register containing the value of a variable
	 *)
      RegUse =
	 RECORD
	    CASE mode: RegMode OF
	    | staticR:        level: CARDINAL;
	    | addrR, valueR:  varp: Ident;
	    END;
	 END;
      RegTab = ARRAY Reg OF RegUse;
      RegTabList = POINTER TO RegTabNode;
      RegTabNode =
	 RECORD
	    regtab: RegTab;
	    link: RegTabList;
	 END;
   VAR
      nestlevel: CARDINAL;
      regtabs: RegTabList;
      regtab: RegTab;

   PROCEDURE AllocReg(VAR reg: Reg; regset: RegSet);
      VAR
	 r: Reg;		(* index variable for regtab[] *)
	 otherreg: Reg;		(* take this register if there is no free one *)
	 prio: RegMode;		(* = regtab[otherreg].mode *)
	 mode: RegMode;		(* = regtab[r].mode *)
	 member: Reg;		(* one member of regset *)
   BEGIN
      otherreg := illegal;
      FOR r := MIN(Reg) TO MAX(Reg) DO
	 IF r IN regset THEN
	    mode := regtab[r].mode;
	    IF mode = freeR THEN
	       reg := r; regtab[r].mode := lockedR; RETURN
	    ELSIF (mode # lockedR) &
		  ((otherreg = illegal) OR (mode < prio)) THEN
	       otherreg := r; prio := mode;
	    END;
	    member := r;
	 END;
      END;
      IF otherreg # illegal THEN
	 reg := otherreg; regtab[r].mode := lockedR;
      ELSE
	 reg := member;	(* return valid register for error recovery *)
	 Error("%R: too many registers needed");
      END;
   END AllocReg;

   PROCEDURE GetReg(VAR r: Reg);
      (* return data register *)
   BEGIN
      AllocReg(r, RegSet{d1..d7});
   END GetReg;

   PROCEDURE GetAddrReg(VAR r: Reg);
      (* return address register *)
   BEGIN
      AllocReg(r, RegSet{a0..a5});
   END GetAddrReg;

   PROCEDURE GetAnyReg(VAR r: Reg);
      (* get data or address register *)
   BEGIN
      AllocReg(r, RegSet{d1..d7, a0..a5});
   END GetAnyReg;

   PROCEDURE GetFloatReg(VAR r: Reg);
      (* get floating point register *)
   BEGIN
      AllocReg(r, RegSet{fp1..fp7});
   END GetFloatReg;

   PROCEDURE RequestReg(r: Reg) : BOOLEAN;
      (* request register `r'; if `r' is already locked FALSE is returned *)
   BEGIN
      IF regtab[r].mode = lockedR THEN
	 RETURN FALSE
      ELSE
	 regtab[r].mode := lockedR;
	 RETURN TRUE
      END;
   END RequestReg;

   PROCEDURE FreeReg(r: Reg);
      (* release `r' in innermost level *)
      VAR
	 ptr: RegTabList;
   BEGIN
      IF regtab[r].mode = lockedR THEN
	 regtab[r].mode := freeR;
      ELSE
	 (* special case:
	    we permit the release of saved registers;
	    this is needed for `hand-made' procedure calls
	    if attributes from the outer level are modified
	    in the inner level
	 *)
	 Assert(regtab[r].mode = freeR);
	 ptr := regtabs;
	 WHILE (ptr # NIL) & (ptr^.regtab[r].mode # lockedR) DO
	    ptr := ptr^.link;
	 END;
	 Assert(ptr # NIL);
	 ptr^.regtab[r].mode := releasedR;
      END;
   END FreeReg;

   (* static links and addresses of variables remains unchanged
      during the execution of a procedure;
      so if they have been loaded into a register and
      the register has not been used otherwise
      we needn't to load them again.

      ReturnStaticLink and ReturnAddress returns registers
      (like FreeReg) to the register manager.
      GetStaticLink and GetAddress returns the
      (previously returned) registers if they are still valid.
   *)

   PROCEDURE ReturnStaticLink(r: Reg; slevel: CARDINAL);
   BEGIN
      WITH regtab[r] DO
	 Assert(mode = lockedR);
	 mode := staticR;
	 level := slevel;
      END;
   END ReturnStaticLink;

   PROCEDURE ReturnAddress(r: Reg; ip: Ident);
   BEGIN
      WITH regtab[r] DO
	 Assert(mode = lockedR);
	 mode := addrR;
	 varp := ip;
      END;
   END ReturnAddress;

   PROCEDURE GetStaticLink(VAR reg: Reg; slevel: CARDINAL) : BOOLEAN;
      VAR r: Reg;
   BEGIN
      FOR r := MIN(Reg) TO MAX(Reg) DO
	 WITH regtab[r] DO
	    IF (mode = staticR) & (level = slevel) THEN
	       StrEmit2("%*RegMan: static link of level %c found in %r",
		  slevel, r);
	       reg := r; mode := lockedR; RETURN TRUE
	    END;
	 END;
      END;
      RETURN FALSE
   END GetStaticLink;

   PROCEDURE GetAddress(VAR reg: Reg; ip: Ident) : BOOLEAN;
      VAR r: Reg;
   BEGIN
      FOR r := MIN(Reg) TO MAX(Reg) DO
	 WITH regtab[r] DO
	    IF (mode = addrR) & (varp = ip) THEN
	       StrEmit2("%*RegMan: address of %n found in %r", ip^.name, r);
	       reg := r; mode := lockedR; RETURN TRUE
	    END;
	 END;
      END;
      RETURN FALSE
   END GetAddress;

   (* values of variables which has been loaded into a register
      can be released by ReturnVariable.
      If the value of a variable is changed (due to assignments
      or procedure calls) TouchVariable must be called.
      TouchAllVariables causes any information to be forgotten.
   *)

   PROCEDURE ReturnVariable(r: Reg; ip: Ident);
   BEGIN
      WITH regtab[r] DO
	 Assert(mode = lockedR);
	 mode := valueR;
	 varp := ip;
      END;
   END ReturnVariable;

   PROCEDURE TouchVariable(ip: Ident);
      VAR r: Reg;
   BEGIN
      FOR r := MIN(Reg) TO MAX(Reg) DO
	 WITH regtab[r] DO
	    IF (mode = valueR) & (ip = varp) THEN
	       mode := freeR;
	    END;
	 END;
      END;
   END TouchVariable;

   PROCEDURE TouchAllVariables;
      VAR r: Reg;
   BEGIN
      FOR r := MIN(Reg) TO MAX(Reg) DO
	 WITH regtab[r] DO
	    IF mode = valueR THEN
	       mode := freeR;
	    END;
	 END;
      END;
   END TouchAllVariables;

   PROCEDURE GetVariable(VAR reg: Reg; ip: Ident) : BOOLEAN;
      VAR r: Reg;
   BEGIN
      FOR r := MIN(Reg) TO MAX(Reg) DO
	 WITH regtab[r] DO
	    IF (mode = valueR) & (ip = varp) THEN
	       StrEmit2("%*RegMan: value of %n found in %r", ip^.name, r);
	       reg := r; mode := lockedR; RETURN TRUE
	    END;
	 END;
      END;
      RETURN FALSE
   END GetVariable;

   PROCEDURE OpenRegTab;
      VAR r: Reg;
   BEGIN
      FOR r := MIN(Reg) TO MAX(Reg) DO
	 regtab[r].mode := freeR;
      END;
   END OpenRegTab;

   PROCEDURE CloseRegTab;
      VAR
	 r: Reg;
   BEGIN
      FOR r := MIN(Reg) TO MAX(Reg) DO
	 Assert(regtab[r].mode # lockedR);
      END;
   END CloseRegTab;

   PROCEDURE BeginBlock;
      (* forget any information about returned registers *)
   BEGIN
      Assert(nestlevel = 0);
      OpenRegTab;
   END BeginBlock;

   PROCEDURE EndBlock;
      (* check that all registers has been released *)
   BEGIN
      Assert(nestlevel = 0);
      CloseRegTab;
   END EndBlock;

   PROCEDURE SaveRegs;
      (* save currently used registers *)
      VAR
	 new: RegTabList;

      PROCEDURE SaveGenRegs;
	 (* save general registers; assumptions:
	    1) BITSET {0}: bit with highest significance is set
	    2) Reg = (d0,..,d7, a0,..,a7, ...)
	       ORD(d0) = 0, ORD(a0) = 8, ORD(a7) = 15
	    register mask in bitset:

	      31         15          8    7          0     bit number
	    +----+-----+----+-----+----+----+-----+----+
	    |    | ... | d0 | ... | d7 | a0 | ... | a7 |
	    +----+-----+----+-----+----+----+-----+----+
	       0         16         23   24         31     BITSET elements
	 *)
	 CONST
	    shift = 16;
	 VAR
	    bitset: BITSET;	(* register mask of MOVEM-instruction *)
	    card: CARDINAL;	(* cardinality of bitset *)
	    reg: Reg;		(* (one) register to be saved *)
	    r: Reg;
      BEGIN
	 bitset := {}; card := 0;
	 FOR r := d0 TO a7 DO
	    IF regtab[r].mode = lockedR THEN
	       INCL(bitset, ORD(r) + shift);
	       INC(card);
	       reg := r;
	    END;
	 END;
	 IF bitset # {} THEN
	    IF card = 1 THEN
	       (* MOVE is more efficient than MOVEM *)
	       Emit2(MOVE, "%L%r,%-r", reg, top);
	    ELSE
	       Emit2(MOVEM, "%L%C,%-r", CARDINAL(bitset), top);
	    END;
	 END;
      END SaveGenRegs;

      PROCEDURE SaveFloatRegs;
	 (* save floating point registers; assumptions:
	    1) BITSET {0}: bit with highest significance is set
	    2) Reg = (..., fp0..fp7, ...)
	       ORD(fp7) = ORD(fp0) + 7
	    register mask in bitset:

	       31           7           0     bit number
	    +-----+-----+-----+-----+-----+
	    |     | ... | fp7 | ... | fp0 |
	    +-----+-----+-----+-----+-----+
	        0          24          31     BITSET elements
	 *)
	 VAR
	    bitset: BITSET;	(* register mask of FMOVEM-instruction *)
	    card: CARDINAL;	(* cardinality of bitset *)
	    reg: Reg;		(* (one) register to be saved *)
	    r: Reg;
      BEGIN
	 bitset := {}; card := 0;
	 FOR r := fp0 TO fp7 DO
	    IF regtab[r].mode = lockedR THEN
	       INCL(bitset, bitsperword-1 - (ORD(r)-ORD(fp0)));
	       INC(card);
	       reg := r;
	    END;
	 END;
	 IF bitset # {} THEN
	    IF card = 1 THEN
	       (* FMOVE is more efficient than FMOVEM *)
	       Emit2(FMOVE, "%X%r,%-r", reg, top);
	    ELSE
	       Emit2(FMOVEM, "%X%C,%-r", CARDINAL(bitset), top);
	    END;
	 END;
      END SaveFloatRegs;

   BEGIN
      SaveGenRegs;
      SaveFloatRegs;
      INC(nestlevel);
      NEW(new);
      new^.regtab := regtab;
      new^.link := regtabs;
      regtabs := new;
      OpenRegTab;
   END SaveRegs;

   PROCEDURE RestoreRegs;
      (* restore registers saved previously *)
      VAR
	 old: RegTabList;
	 r: Reg;

      PROCEDURE RestoreGenRegs;
	 (* save general registers; assumptions:
	    1) BITSET {0}: bit with highest significance is set
	    2) Reg = (d0,..,d7, a0,..,a7, ...)
	       ORD(d0) = 0, ORD(a0) = 8, ORD(a7) = 15
	    register mask in bitset:

	      31         15          8    7          0     bit number
	    +----+-----+----+-----+----+----+-----+----+
	    |    | ... | a7 | ... | a0 | d7 | ... | d0 |
	    +----+-----+----+-----+----+----+-----+----+
	       0         16         23   24         31     BITSET elements
	 *)
	 VAR
	    bitset: BITSET;	(* register mask of MOVEM-instruction *)
	    card: CARDINAL;	(* cardinality of bitset *)
	    reg: Reg;		(* (one) register to be saved *)
	    r: Reg;
      BEGIN
	 bitset := {}; card := 0;
	 FOR r := d0 TO a7 DO
	    IF (regtab[r].mode = lockedR) OR (regtab[r].mode = releasedR) THEN
	       INCL(bitset, bitsperword-1 - ORD(r));
	       INC(card);
	       reg := r;
	    END;
	 END;
	 IF bitset # {} THEN
	    IF card = 1 THEN
	       (* MOVE is more efficient than MOVEM *)
	       Emit2(MOVE, "%L%+r,%r", top, reg);
	    ELSE
	       Emit2(MOVEM, "%L%+r,%C", top, CARDINAL(bitset));
	    END;
	 END;
      END RestoreGenRegs;

      PROCEDURE RestoreFloatRegs;
	 (* save floating point registers; assumptions:
	    1) BITSET {0}: bit with highest significance is set
	    2) Reg = (..., fp0..fp7, ...)
	       ORD(fp7) = ORD(fp0) + 7
	    register mask in bitset:

	       31           7           0     bit number
	    +-----+-----+-----+-----+-----+
	    |     | ... | fp0 | ... | fp7 |
	    +-----+-----+-----+-----+-----+
	        0          24          31     BITSET elements
	 *)
	 CONST
	    shift = 24;
	 VAR
	    bitset: BITSET;	(* register mask of FMOVEM-instruction *)
	    card: CARDINAL;	(* cardinality of bitset *)
	    reg: Reg;		(* (one) register to be saved *)
	    r: Reg;
      BEGIN
	 bitset := {}; card := 0;
	 FOR r := fp0 TO fp7 DO
	    IF (regtab[r].mode = lockedR) OR (regtab[r].mode = releasedR) THEN
	       INCL(bitset, ORD(r)-ORD(fp0) + shift);
	       INC(card);
	       reg := r;
	    END;
	 END;
	 IF bitset # {} THEN
	    IF card = 1 THEN
	       (* FMOVE is more efficient than FMOVEM *)
	       Emit2(FMOVE, "%X%+r,%r", top, reg);
	    ELSE
	       Emit2(FMOVEM, "%X%+r,%C", top, CARDINAL(bitset));
	    END;
	 END;
      END RestoreFloatRegs;

   BEGIN
      Assert(nestlevel >= 1); DEC(nestlevel);
      CloseRegTab;
      regtab := regtabs^.regtab;
      old := regtabs;
      regtabs := regtabs^.link;
      DISPOSE(old);
      RestoreFloatRegs;
      RestoreGenRegs;
      FOR r := MIN(Reg) TO MAX(Reg) DO
	 IF regtab[r].mode # lockedR THEN
	    regtab[r].mode := freeR;
	 END;
      END;
   END RestoreRegs;

BEGIN
   nestlevel := 0;
   regtabs := NIL;
END RegMan.
