(* Ulm's Oberon Compiler
   Copyright (c) 1989 by University of Ulm, SAI, D-W-7900 Ulm, Germany
   ----------------------------------------------------------------------------
   $Id: ScanArgs.m2,v 0.4 1993/04/19 14:37:57 borchert Exp $
   ----------------------------------------------------------------------------
   $Log: ScanArgs.m2,v $
   Revision 0.4  1993/04/19  14:37:57  borchert
   the current version of the Oberon System may now be given as option

   Revision 0.3  1993/02/03  12:43:20  borchert
   ``output'' may be assembly output or symbol file (comment changed)

   Revision 0.2  1992/08/19  13:51:07  borchert
   C flag added to usage

   Revision 0.1  1992/07/30  10:49:20  borchert
   Initial revision

   ----------------------------------------------------------------------------
*)

IMPLEMENTATION MODULE ScanArgs;		(* Martin Hasch, Dec 1988 *)

   FROM Arguments IMPORT
      InitArgs, GetFlag, GetArg, Usage, FetchString, FetchInt;
   FROM Strings	  IMPORT StrCpy;
   FROM Memory	  IMPORT ALLOCATE;
   FROM Suffixes  IMPORT
      TestSuffix, MakeName,
      definitionSX, moduleSX, deflistSX, modlistSX, symfileSX, asfileSX;
   FROM Exception IMPORT Fatal;
   FROM Scan IMPORT Message1, fatal;
   FROM Lex IMPORT SetDefault;
   FROM Calendar  IMPORT CurrentTime;

(***** EXPORTED:
   TYPE
      FileName = ARRAY [0..511] OF CHAR;
      Path = POINTER TO PathRec;
      PathRec =
	 RECORD
	    symfile: FileName;	(* symbol file or archive with symbolfiles *)
	    link: Path;
	 END;
      VersionString = ARRAY [0..31] OF CHAR;

   TYPE
      UpdateMode =
	 (noUpdate,		(* no authorization for updates *)
	 softUpdate,		(* update only if necessary *)
	 hardUpdate);		(* update if sizes are different *)
   VAR
      input: FileName;		(* source file *)
      output: FileName;		(* assembly output or symbol file *)
      dolist: BOOLEAN;		(* create a listing ? *)
      listing: FileName;	(* listing file *)
      updatesym: UpdateMode;	(* authorized to update symbol file? *)
      commands: BOOLEAN;	(* parameterless global procs = commands? *)
      sympath: Path;		(* search path for symbol files *)
      defunit: BOOLEAN;		(* source file with definition suffix? *)
      currenttime,
      lastrevision: Time;
      writeref: BOOLEAN;	(* writing a reference file? *)
      reffile: FileName;	(* name of reference file *)
      version: VersionString;   (* version to be printed for listings *)
*****)

   PROCEDURE ScanArguments;				(* EXPORTED *)
      VAR
	 flag: CHAR;
	 pathptr: Path;
	 filename: FileName;
	 intarg: INTEGER;

      PROCEDURE NewPath( (*read*)VAR filename: FileName): Path;
	 VAR
	    result: Path;
      BEGIN
	 ALLOCATE(result, SIZE(PathRec));
	 WITH result^ DO
	    StrCpy(symfile,filename);
	    link := NIL;
	 END;
	 RETURN result
      END NewPath;

      PROCEDURE OptionLetters;
	 VAR
	    index: CARDINAL;
	    letters: ARRAY [0..199] OF CHAR;
      BEGIN
	 FetchString(letters);
	 index := 0;
	 WHILE (index <= HIGH(letters)) & (letters[index] # 0C) DO
	    IF ~SetDefault(letters[index],FALSE) THEN
	       Message1(fatal,
		  "%Q: illegal compiler option character", letters[index])
	    END;
	    INC(index);
	 END;
      END OptionLetters;

   BEGIN
      dolist := FALSE; updatesym := noUpdate; writeref := FALSE;
      commands := FALSE; output[0] := 0C; listing[0] := 0C;
      currenttime := CurrentTime();
      lastrevision := currenttime - 24*60*60;	(* yesterday, same time *)
      version := "0.1";
      InitArgs(
"[-CLuU] [-D letters] [-l listing] [-o output] [-r reffile] [-T timestamp] [-v version] sourcefile [symfile|symarchive]...");
      WHILE GetFlag(flag) DO
	 CASE flag OF
	 |  "C":  commands := TRUE;
	 |  "D":  OptionLetters;
	 |  "l":  dolist := TRUE; FetchString(listing);
	 |  "L":  dolist := TRUE;
	 |  "o":  FetchString(output);
	 |  "r":  writeref := TRUE;
		  FetchString(reffile);
	 |  "T":  FetchInt(intarg);
		  lastrevision := intarg;
	 |  "u":  updatesym := softUpdate;
	 |  "U":  updatesym := hardUpdate;
	 |  "v":  FetchString(version);
	    ELSE  Usage
	 END;
      END;

      IF ~GetArg(input) THEN
	 Usage
      END;
      defunit := TestSuffix(input, definitionSX);
      IF ~defunit & ~TestSuffix(input, moduleSX) THEN
	 Fatal("bad suffix of source file name");
      END;
      IF output[0] = 0C THEN
	 IF defunit THEN
	    MakeName(output, input,definitionSX,symfileSX);
	    writeref := FALSE;
	 ELSE
	    MakeName(output, input,definitionSX,asfileSX);
	 END;
      END;
      IF dolist & (listing[0] = 0C) THEN
	 IF defunit THEN
	    MakeName(listing,input,definitionSX,deflistSX);
	 ELSE
	    MakeName(listing,input,moduleSX,modlistSX);
	 END;
      END;

      sympath := NIL;
      WHILE GetArg(filename) DO
	 IF sympath = NIL THEN
	    pathptr := NewPath(filename);
	    sympath := pathptr;
	 ELSE
	    pathptr^.link := NewPath(filename);
	    pathptr := pathptr^.link;
	 END;
      END;
   END ScanArguments;

END ScanArgs.
