(* Ulm's Oberon Compiler
   Copyright (c) 1989 by University of Ulm, SAI, D-W-7900 Ulm, Germany
   ----------------------------------------------------------------------------
   $Id: SymDef.m2,v 0.13 1994/05/25 11:32:23 borchert Exp $
   ----------------------------------------------------------------------------
   $Log: SymDef.m2,v $
   Revision 0.13  1994/05/25  11:32:23  borchert
   some bug fixes due to recent GC support:
   (1) Usually, we don't need the parameter offsets at the time of
       generating the code of a procedure call. This has been changed
       since parameter passing may be delayed in case of temporary
       pointers. Consequently, SymDef has to assure that all procedure types
       have valid parameter offsets. Regrettably, the interface of SymDef
       hasn't been designed with this in mind by not providing a
       procedure which terminates the definition of a procedure type.
       Thus, we are enforced to check for the correctness of the
       parameter offsets at any time we may use them. This is done by
       CheckProcedureType.
   (2) Due to the GC we are enforced to change from paramV to
       copyparamV in case where records or arrays contain pointers.
       If the definition part was compiled with paramV, the module part
       may reveal that we have to use copyparamV instead. In this case
       we have to update the symbol file. The check for this is now
       being done in CompareProc.

   Revision 0.12  1994/04/02  12:18:03  borchert
   bug fix: short arrays and records which do contain pointers are
            now treated as copy parameters (in case of call by value)
            to avoid trouble with the GC

   Revision 0.11  1994/02/18  11:14:54  borchert
   bug fix: PrivatePart didn't set public^.containsptr --
            this caused pointer list entries to be lost

   Revision 0.10  1993/09/27  12:48:39  borchert
   offset calculations of parameters shifted
   bug fix: integer constants can now be of SYSTEM.INT16 type

   Revision 0.9  1993/06/25  12:34:48  borchert
   compiler option $O implemented: pointer may now be untagged

   Revision 0.8  1993/06/18  15:30:53  borchert
   InRange: INTEGER replaced by Integer

   Revision 0.7  1993/06/16  09:49:28  borchert
   InitSYSTEM added to assure that int16ptr is initialized even
   when SYSTEM is not imported

   Revision 0.6  1993/06/09  14:51:17  borchert
   ImportOurDefinition has been added to assure that our own
   symbolfile gets read before the others.
   This is necessary in case of reference cycles where our symbolfile
   might be read indirectly without ourmodule set to TRUE.

   Revision 0.5  1993/05/03  11:04:37  borchert
   bug fix: RefCloseBlock was called after SymTab.CloseBlock and
            tried to emit types in its worklist which were already disposed

   Revision 0.4  1993/04/13  15:19:19  borchert
   bug fixes:
   (1) forward declarations of procedures lead to wrong reference files
   (2) basetypes which were imported by the definition but not the
       module lead to a failed assertion

   Revision 0.3  1993/02/03  12:45:01  borchert
   new compiler option: $P
   PrivatePart: bug fixed

   Revision 0.2  1992/10/22  14:42:52  borchert
   new checks for correct order of record fields

   Revision 0.1  1992/07/30  10:49:37  borchert
   Initial revision

   ----------------------------------------------------------------------------
*)

IMPLEMENTATION MODULE SymDef; (* AFB 1/89 *)

   (* symbol table interface to the parser *)
   (* the main goal of this module is to achieve maximum independency *)
   (* between symbol table management and parsing *)

   FROM ConstExpr IMPORT Binary;
   FROM Exception IMPORT Assert;
   FROM Lex IMPORT Constval, Symbol, StringLen, previdpos,
      options, AddLegalOptions, CompilerOptions;
   FROM IdentSys IMPORT Identifier, PutIdentChar, PutIdent;
   FROM Machine IMPORT minshort, maxshort, minint, maxint, minreal,
      maxreal, Align, oneword, Direction, stackdirection, procmarkspace,
      alignparam, align, onebyte, recspace, minint16, maxint16;
   FROM Memory IMPORT ALLOCATE, DEALLOCATE;
   FROM Scan IMPORT Error, Error1, Error2, ErrorList, AddToList, EndOfList,
      At, errorflag;
   FROM ScanArgs IMPORT defunit, updatesym, UpdateMode;
   FROM Standard IMPORT sysmodid, InitSysModule,
      charptr, shortptr, intptr, int16ptr, longptr,
      realptr, longrealptr, setptr, InitStd, boolptr, niltype;
   FROM Strings IMPORT StrCat;
   FROM SymFile IMPORT InitType, LinkType, ReadSymFile, GenSymFile,
      UpdateSymFile;
   FROM SymRef IMPORT RefImport, RefIdent, RefOpenBlock, RefCloseBlock;
   FROM SymTab IMPORT IdentClass, Enter, FieldList, Ident, ParamList, Level,
      import, global, Search, Size, Form, IdentList, Include,
      SearchFieldList, SearchList, Find, SearchParamList, SearchAndEnter,
      mainmod, globalvars, VarKind, FormSet;
   FROM Types IMPORT Integer;
   IMPORT Exception, IdentSys, Lex, Memory, Scan, SymRef, SymTab;

   CONST
      taggedPointerOpt = "O"; (* pointer type reference tagged records *)
      tracePointerOpt = "P"; (* garbage collection may trace pointers *)

   TYPE
      Type = SymTab.Type;

   VAR
      paramtail: ParamList;	(* used by Procedure and AddParameter *)
      openproc: Ident;		(* last parameter of OpenScope *)
      noident: Identifier;	(* NIL reference for identifiers *)
      nextprocno: CARDINAL;	(* next (unique) procedure number *)
      lastproctype: Type;	(* type of last procedure (DeclProc) *)
      lastprocid: Identifier;	(* identifier of last procedure (DeclProc) *)

   PROCEDURE CondError(msg: ARRAY OF CHAR; ip: Ident);
      (* print error message which is related to `ip' only *)
      (* if ip^.error is FALSE *)
   BEGIN
      Assert(ip # NIL);
      WITH ip^ DO
	 IF ~error THEN
	    error := TRUE;
	    Error1(msg, name);
	 END;
      END;
   END CondError;

   PROCEDURE QualSearchAndEnter(qualid: QualIdent; VAR ip: Ident);
      VAR
	 module: Ident;
	 list: IdentList;
   BEGIN
      WITH qualid DO
	 IF qualified THEN
	    SearchAndEnter(modname, module);
	    WITH module^ DO
	       IF class # badclass THEN
		  IF class = moduleC THEN
		     IF ~SearchList(idname, module^.export, ip) THEN
			Error2("%I not exported from qualifying module %I",
			   idname, modname);
			NEW(ip);
			WITH ip^ DO (* error recovery *)
			   name := idname;
			   error := TRUE;
			   class := badclass;
			END;
			NEW(list);
			WITH list^ DO
			   ident := ip;
			   link := module^.export;
			END;
			module^.export := list;
		     END;
		  ELSE
		     Error1("%I must be a module name", modname);
		     ip := module; (* error recovery *)
		  END;
	       ELSE (* class = badclass *)
		  ip := module; (* error recovery *)
	       END;
	    END;
	 ELSE
	    SearchAndEnter(idname, ip);
	 END;
      END;
   END QualSearchAndEnter;

   PROCEDURE InitIdent(ip: Ident; t: Type);
   BEGIN
      WITH ip^ DO
	 type := t;
	 mod := mainmod;
      END;
   END InitIdent;

   PROCEDURE Allocate(size: Size; VAR next, offset: Size; dir: Direction);
   BEGIN
      Assert(size >= 0);
      IF size >= align THEN
	 Align(next);
      END;
      IF dir = forwardDir THEN
	 offset := next;
	 IF next <= maxint-size THEN (* size >= 0! *)
	    INC(next, size);
	 ELSE
	    Error("size or offset exceeds MAX(INTEGER)");
	    next := 0; (* avoid additional error messages *)
	 END;
      ELSE (* dir = backwardDir *)
	 IF next >= minint+size THEN (* size >= 0! *)
	    DEC(next, size);
	 ELSE
	    Error("offset exceeds MIN(INTEGER)");
	    next := 0; (* avoid additional error messages *)
	 END;
	 offset := next;
      END;
   END Allocate;

   PROCEDURE AllocateType(type: Type; VAR next, offset: Size; dir: Direction);
   BEGIN
      IF type = NIL THEN
	 Allocate(oneword, next, offset, dir);
      ELSE
	 Allocate(type^.size, next, offset, dir);
      END;
   END AllocateType;

   PROCEDURE CheckExport(ip: Ident);
      VAR
	 new: IdentList;
   BEGIN
      IF defunit & (Level() = global) THEN
	 NEW(new);
	 WITH new^ DO
	    ident := ip;
	    link := mainmod^.export;
	 END;
	 mainmod^.export := new;
      END;
   END CheckExport;

   (* ---------------------------------------------------------	*)
   (* imports							*)
   (* ---------------------------------------------------------	*)

   PROCEDURE DoImport(ip: Ident; modname: Identifier; ourmodule: BOOLEAN);
      VAR
	 modip: Ident; (* returned by ReadSymFile *)
   BEGIN
      Assert(Level() = import);
      ReadSymFile(modname, modip, ourmodule);
      WITH ip^ DO
	 export := modip^.export;
	 key := modip^.key;
	 origname := modname;
      END;
      IF ~ourmodule THEN
	 RefImport(ip);
      END;
   END DoImport;

   PROCEDURE ImportOurDefinition(id: Identifier);
      (* to be called BEFORE the other imports *)
      VAR
	 ip: Ident;
	 il: IdentList;
   BEGIN
      Assert(Level() = import);
      ReadSymFile(id, ip, (* ourmodule = *) TRUE);
      (* look for highest procedure number *)
      il := ip^.export;
      WHILE il # NIL DO
	 WITH il^.ident^ DO
	    IF (class = procedureC) & (procno >= nextprocno) THEN
	       nextprocno := procno + 1;
	    END;
	 END;
	 il := il^.link;
      END;
   END ImportOurDefinition;

   PROCEDURE InitSYSTEM;
      VAR
	 ip: Ident;
   BEGIN
      ReadSymFile(sysmodid, ip, (* ourmodule = *) FALSE);
   END InitSYSTEM;

   PROCEDURE Import(id: Identifier);
      VAR
	 ip: Ident;
   BEGIN
      IF Enter(id, moduleC, ip) THEN
	 DoImport(ip, id, (* ourmodule = *) FALSE);
      END;
   END Import;

   PROCEDURE AliasImport(alias, id: Identifier);
      VAR
	 ip: Ident;
   BEGIN
      IF Enter(alias, moduleC, ip) THEN
	 DoImport(ip, id, (* ourmodule = *) FALSE);
      END;
   END AliasImport;


   (* ---------------------------------------------------------	*)
   (* scopes							*)
   (* ---------------------------------------------------------	*)

   PROCEDURE OpenScope(id: Identifier);
      (* the first call opens the module; *)
      (* all further calls are for procedures only *)
      (* module: to be called AFTER the imports *)
      (* procedures: to be called AFTER DeclProc *)
      VAR
	 ip: Ident;
	 pl: ParamList;
	 paramip: Ident;
	 il: IdentList;

   BEGIN
      IF Level() = import THEN
	 IF ~Enter(id, moduleC, mainmod) THEN
	    (* mainmod must be # NIL *)
	    NEW(mainmod);
	    WITH mainmod^ DO
	       name := id;
	       error := TRUE;
	       class := moduleC;
	    END;
	 END;
	 WITH mainmod^ DO
	    origname := id;
	    export := NIL;
	 END;
	 IF ~defunit THEN
	    DoImport(mainmod, id, (* ourmodule = *) TRUE);
	 END;
      END;
      CheckForwards((* printErrors = *) TRUE); (* of pointer declarations *)
      Assert(Search(id, ip));
      SymTab.OpenScope;
      (* we need a valid `ip' *)
      IF (ip^.class # procedureC) & (ip^.class # moduleC) THEN
	 NEW(ip);
	 WITH ip^ DO
	    name := id;
	    class := procedureC;
	    error := TRUE;
	    plevel := Level();
	    mod := mainmod;
	    IF (lastproctype # NIL) & (id = lastprocid) THEN
	       type := lastproctype;
	    ELSE
	       ProcedureType(type);
	    END;
	    exported := FALSE;
	    forward := FALSE;
	    local := NIL;
	    params := NIL;
	    varoffset := 0;
	    parmoffset := 0;
	 END;
      END;
      RefOpenBlock;
      WITH ip^ DO
	 IF class = procedureC THEN
	    Assert(type # NIL);
	    pl := type^.param;
	    WHILE pl # NIL DO
	       IF Enter(pl^.id, varC, paramip) THEN
		  paramip^.offset := pl^.offset;
		  paramip^.type := pl^.type;
		  WITH paramip^ DO
		     mod := mainmod;
		     varkind := pl^.varkind;
		     indirect := (varkind = varparamV) OR IndirectType(type);
		     IF (varkind = copyparamV) & ~indirect THEN
			(* indirect parameters are allocated dynamically *)
			ptroffset := offset;
			(* allocate copy parameters twice:
			   (1) as pointer (which is pushed onto stack)
			   (2) as copied value (allocated as variable)
			*)
			AllocateType(type,
				     ip^.varoffset, offset, stackdirection);
		     END;
		  END;
		  RefIdent(paramip, previdpos);
		  (* enter `paramip' into `params'-list *)
		  NEW(il);
		  WITH il^ DO
		     ident := paramip;
		     link := params;
		  END;
		  params := il;
	       ELSE
		  (* error message already printed *)
	       END;
	       pl := pl^.link;
	    END;
	    openproc := ip;
	 ELSE
	    Assert(class = moduleC);
	    (* too hard, if module name has been misused
	       Assert((class = moduleC) & (ip = mainmod));
	    *)
	    openproc := NIL;
	    RefIdent(ip, previdpos);
	 END;
      END;
      lastprocid := noident; lastproctype := NIL;
   END OpenScope;

   PROCEDURE CloseScope;
      (* to be called at END (ProcedureName|ModuleName) *)

      PROCEDURE CheckExports;
	 VAR
	    il: IdentList;
	    ip: Ident;
	    first: BOOLEAN;
	    error: BOOLEAN;

	 PROCEDURE NotDeclared(name: Identifier);
	 BEGIN
	    error := TRUE;
	    IF first THEN
	       ErrorList("exported but not declared: ");
	       AddToList("%I", name);
	       first := FALSE;
	    ELSE
	       AddToList(", %I", name);
	    END;
	 END NotDeclared;

      BEGIN
	 first := TRUE; error := FALSE;
	 il := mainmod^.export;
	 WHILE il # NIL DO
	    WITH il^.ident^ DO
	       IF ~Find(Level(), name, ip) OR (ip^.class = moduleC) THEN
		  (* if class = moduleC then `name' is a imported module *)
		  NotDeclared(name);
	       END;
	    END;
	    il := il^.link;
	 END;
	 IF error THEN
	    EndOfList;
	 END;
      END CheckExports;

   BEGIN
      CheckForwards((* printErrors = *) TRUE); (* of pointer types *)
      CheckForwardDeclarations; (* of procedures *)
      IF Level() = global THEN
	 IF defunit THEN
	    GenSymFile(mainmod);
	 ELSE
	    CheckExports;
	 END;
      END;
      RefCloseBlock;
      SymTab.CloseScope;
      openproc := NIL;
      lastprocid := noident; lastproctype := NIL;
   END CloseScope;

   PROCEDURE Exported(id: Identifier; class: IdentClass;
		      VAR ip: Ident) : BOOLEAN;
      (* return TRUE
	    if `id' is exported and of class `class'
	 an error message is printed if `id' is of different class
      *)
      VAR
	 s: ARRAY [0..15] OF CHAR;
   BEGIN
      IF ~defunit & (Level() = global) &
	 SearchList(id, mainmod^.export, ip) THEN
	 IF ip^.class # class THEN
	    CASE ip^.class OF
	    | constC:      s := "const";
	    | typeC:       s := "type";
	    | varC:        s := "variable";
	    | procedureC:  s := "procedure";
	    END;
	    Error2("%I is exported as %s", id, s);
	    RETURN FALSE
	 END;
	 RETURN TRUE
      ELSE
	 RETURN FALSE
      END;
   END Exported;


   (* ---------------------------------------------------------	*)
   (* declaration and retrieval of constant values		*)
   (* ---------------------------------------------------------	*)

   PROCEDURE GetConstVal(qualid: QualIdent; VAR cval: Constval);
      (* retrieval of a constant value *)
      VAR
	 ip: Ident;
   BEGIN
      QualSearchAndEnter(qualid, ip);
      WITH ip^ DO
	 IF class = constC THEN
	    cval := constval;
	 ELSE
	    CondError("%I must be a constant", ip);
	 END;
      END;
   END GetConstVal;

   PROCEDURE DeclConst(id: Identifier; cval: Constval);
      (* CONST id = constval *)
      VAR
	 ip: Ident;
	 typeip: Ident;
	 exptype: Type;

      PROCEDURE InRange(i: Integer; min, max: Integer) : BOOLEAN;
      BEGIN
	 RETURN (i >= min) & (i <= max)
      END InRange;

      PROCEDURE StringType(len: CARDINAL; VAR type: Type);
      BEGIN
	 IF len = 2 THEN (* len = StringLen() + 1 *)
	    type := charptr;
	 ELSE
	    InitType(type);
	    WITH type^ DO
	       ident := NIL;
	       size := len * onebyte; Align(size);
	       form := array;
	       dyn := FALSE;
	       length := len;
	       element := charptr;
	       LinkType(charptr);
	    END;
	 END;
      END StringType;

      PROCEDURE CompareConsts(cval1, cval2: Constval) : BOOLEAN;
      BEGIN
	 IF cval1.sy # cval2.sy THEN
	    RETURN FALSE
	 END;
	 Binary(eql, cval1, cval2);
	 RETURN cval1.boolval
      END CompareConsts;

   BEGIN
      At(previdpos);
      exptype := NIL;
      IF Exported(id, constC, ip) THEN
	 IF ~CompareConsts(ip^.constval, cval) THEN
	    Error1("constant %I differs from export declaration", id);
	 ELSE
	    exptype := ip^.type;
	 END;
      END;
      IF Enter(id, constC, ip) THEN
	 WITH ip^ DO
	    mod := mainmod;
	    constval := cval;
	    exported := defunit;
	    WITH constval DO
	       CASE sy OF
	       | stringcon:   StringType(StringLen(string)+1, type);
	       | charcon:     type := charptr;
	       | intcon:      IF InRange(intval, minshort, maxshort) THEN
				 type := shortptr;
			      ELSIF InRange(intval, minint16, maxint16) THEN
				 type := int16ptr;
			      ELSIF InRange(intval, minint, maxint) THEN
				 type := intptr;
			      ELSE
				 type := longptr;
			      END;
	       | realcon:     type := realptr;
	       | longrealcon: type := longrealptr;
	       | setcon:      type := setptr;
	       | boolcon:     type := boolptr;
	       | nilSY:       type := niltype;
	       END;
	    END;
	    LinkType(type); (* necessary for `SymFile' *)
	    IF exptype # NIL THEN
	       CompareTypes(id, exptype, type); 
	    END;
	 END;
	 CheckExport(ip);
      END;
   END DeclConst;


   (* ---------------------------------------------------------	*)
   (* type constructors						*)
   (* ---------------------------------------------------------	*)

   MODULE ForwardDeclarationsOfPointers;

      FROM IdentSys IMPORT Identifier;
      FROM Lex IMPORT previdpos;
      FROM Memory IMPORT ALLOCATE, DEALLOCATE;
      FROM Scan IMPORT Error2, Error3, Error4;
      FROM SymRef IMPORT RefIdent;
      FROM SymTab IMPORT Type, Ident, IdentClass, Search, Form;

      EXPORT EnterForward, CheckForwards;

      TYPE
	 ForwardList = POINTER TO ForwardListRec;
	 ForwardListRec =
	    RECORD
	       ptrtype: Type;
	       refid: Identifier;
	       defip: Ident; (* if not NIL check refid against this *)
	       link: ForwardList;
	    END;
      VAR
	 list: ForwardList;

      PROCEDURE EnterForward(t: Type; id: Identifier);
	 VAR
	    new: ForwardList;
      BEGIN
	 NEW(new);
	 WITH new^ DO
	    ptrtype := t;
	    refid := id;
	    defip := NIL;
	    link := list;
	 END;
	 list := new;
      END EnterForward;

      PROCEDURE CheckForwards(printError: BOOLEAN);
	 CONST
	    bad = "bad forward declaration";
	    notatype = "must be a type identifier";
	    badtype = "must be a record or array";
	 VAR
	    old: ForwardList;
	    refip: Ident;
	    lp: ForwardList;
	    keep: BOOLEAN; (* keep entry alive *)
	    head: ForwardList; (* new head of list *)
	    tail: ForwardList; (* new tail of list *)
      BEGIN
	 lp := list; tail := NIL; head := NIL;
	 WHILE lp # NIL DO
	    WITH lp^ DO
	       keep := FALSE;
	       IF Search(refid, refip) THEN
		  WITH refip^ DO
		     IF class = typeC THEN
			(* don't print error messages if type = NIL *)
			IF type # NIL THEN
			   IF (type^.form = record) OR (type^.form = array) THEN
			      ptrtype^.reftype := type;
			      IF ptrtype^.ident # NIL THEN
				 RefIdent(ptrtype^.ident, previdpos);
			      END;
			   ELSE
			      IF ptrtype^.ident # NIL THEN
				 Error4("%s of type %I: %I %s",
				    bad, ptrtype^.ident^.name,
				    refid, badtype);
			      ELSE
				 Error3("%s: %I %s", bad, refid, badtype);
			      END;
			   END;
			END;
		     ELSIF ptrtype^.ident # NIL THEN
			Error4("%s of type %I: %I %s",
			   bad, ptrtype^.ident^.name,
			   refid, notatype);
		     ELSE
			Error3("%s: %I %s", bad, refid, notatype);
		     END;
		  END;
	       ELSIF ~printError THEN
		  keep := TRUE;
	       ELSIF ptrtype^.ident # NIL THEN
		  Error3("%s of %I: pointer-referenced type %I not declared",
		     bad, ptrtype^.ident^.name, refid);
	       ELSE
		  Error2("%s: pointer-referenced type %I not declared",
		     bad, refid);
	       END;
	    END; (* WITH *)
	    IF keep THEN
	       IF head = NIL THEN
		  head := lp;
	       ELSE
		  tail^.link := lp;
	       END;
	       tail := lp;
	       lp := lp^.link;
	    ELSE
	       old := lp;
	       lp := lp^.link;
	       DISPOSE(old);
	    END;
	 END;
	 IF tail # NIL THEN
	    tail^.link := NIL;
	 END;
	 list := head;
      END CheckForwards;

   BEGIN
      list := NIL;
   END ForwardDeclarationsOfPointers;

   PROCEDURE CheckPointerRefType(VAR ptrtype: Type);
      CONST
	 msg = "pointer-referenced types must be an array or a record";
   BEGIN
      WITH ptrtype^ DO
	 Assert(form = pointer);
	 IF reftype # NIL THEN
	    WITH reftype^ DO
	       IF (form # array) & (form # record) THEN
		  IF ptrtype^.ident # NIL THEN
		     Error2("bad definition of type %I: %s",
			ptrtype^.ident^.name, msg);
		  ELSE
		     Error1("%s", msg);
		  END;
		  reftype := NIL;
	       END;
	    END;
	 END;
      END;
   END CheckPointerRefType;

   PROCEDURE UseType(qualid: QualIdent; VAR idtype: Type);
      (* type-identifier 'qualid' is used; return type-structure *)
      (* if 'id' is unknown or of other id-class an error message *)
      (* will be printed and a dummy-type returned *)
      VAR
	 ip: Ident;
   BEGIN
      QualSearchAndEnter(qualid, ip);
      IF ip^.class = typeC THEN
	 idtype := ip^.type;
	 LinkType(idtype); (* needed for `SymFile' *)
      ELSE
	 CondError("%I must be a type", ip);
	 idtype := NIL;
      END;
   END UseType;

   PROCEDURE NoType() : Type;
   BEGIN
      RETURN NIL
   END NoType;

   PROCEDURE CheckForPrivateParts(t1, t2: Type);
      (* inherits `privateparts' from `t2' to `t1';
	 `t1': container type
	 `t2': member of `t1'
	 `t2' may be NIL
      *)
   BEGIN
      Assert(t1 # NIL);
      IF t2 # NIL THEN
	 WITH t1^ DO
	    privateparts := privateparts OR t2^.privateparts;
	 END;
      END;
   END CheckForPrivateParts;

   PROCEDURE ArrayOf(dim: Constval; elemtype: Type; VAR type: Type);
      VAR
	 dimension: Size;
   BEGIN
      IF (dim.sy = intcon) & (dim.intval > 0) THEN
	 dimension := dim.intval;
      ELSE
	 dimension := 10;
	 Error("array dimension must be a positive integer");
      END;
      InitType(type); CheckProcedureType(elemtype);
      WITH type^ DO
	 ident := NIL;
	 IF elemtype = NIL THEN
	    size := dimension;
	 ELSIF elemtype^.size <= maxint DIV dimension THEN
	    size := elemtype^.size * dimension;
	    Align(size);
	 ELSE
	    Error("size of array exceeds MAX(INTEGER)");
	 END;
	 form := array;
	 dyn := FALSE;
	 length := dimension;
	 element := elemtype;
	 IF tracePointerOpt IN options THEN
	    containsptr := (element # NIL) & element^.containsptr;
	 ELSE
	    containsptr := FALSE;
	 END;
      END;
      CheckForPrivateParts(type, elemtype);
   END ArrayOf;

   PROCEDURE DynArrayOf(elemtype: Type; VAR type: Type);
   BEGIN
      InitType(type); CheckProcedureType(elemtype);
      WITH type^ DO
	 ident := NIL;
	 IF (elemtype # NIL) & (elemtype^.form = array) & elemtype^.dyn THEN
	    size := elemtype^.size + oneword;
	 ELSE
	    size := 2 * oneword;
	 END;
	 form := array;
	 element := elemtype;
	 IF tracePointerOpt IN options THEN
	    containsptr := (element # NIL) & element^.containsptr;
	 ELSE
	    containsptr := FALSE;
	 END;
	 dyn := TRUE;
      END;
      CheckForPrivateParts(type, elemtype);
   END DynArrayOf;

   PROCEDURE PointerTo(rtype: Type; VAR type: Type);
   BEGIN
      InitType(type); CheckProcedureType(rtype);
      WITH type^ DO
	 ident := NIL;
	 size := oneword;
	 form := pointer;
	 reftype := rtype;
	 containsptr := (CompilerOptions{tracePointerOpt,
					 taggedPointerOpt} <= options);
	 taggedptr := (taggedPointerOpt IN options);
      END;
      CheckPointerRefType(type);
   END PointerTo;

   PROCEDURE PointerToIdent(rtype: Identifier; VAR t: Type);
      VAR
	 typeip: Ident;
   BEGIN
      IF Exported(rtype, typeC, typeip) THEN
	 WITH typeip^ DO
	    IF class = typeC THEN
	       PointerTo(type, t);
	    ELSE
	       Error1("%I must be a type identifier", rtype);
	       PointerTo(NIL, t);
	    END;
	 END;
      ELSE
	 PointerTo(NIL, t);
	 EnterForward(t, rtype);
      END;
   END PointerToIdent;

   PROCEDURE Record(btype: Type; VAR type: Type);
      (* basetype may be NIL *)
   BEGIN
      InitType(type);
      WITH type^ DO
	 ident := NIL;
	 size := 0;
	 form := record;
	 IF (btype = NIL) OR (btype^.form = record) THEN
	    basetype := btype;
	 ELSE
	    basetype := NIL; (* error message is printed below *)
	 END;
	 IF tracePointerOpt IN options THEN
	    containsptr := (basetype # NIL) & basetype^.containsptr;
	 ELSE
	    containsptr := FALSE;
	 END;
	 IF btype # NIL THEN
	    WITH btype^ DO
	       IF form # record THEN
		  Assert(ident # NIL);
		  Error1("%I is not a record type", ident^.name);
		  type^.fields := NIL;
	       ELSE
		  (* following trick assumes that new field elements *)
		  (* are inserted at the beginning of the list and   *)
		  (* not appended *)
		  type^.fields := btype^.fields;
	       END;
	    END;
	    size := btype^.size;
	 ELSE
	    fields := NIL;
	 END;
	 projection := FALSE;
      END;
      CheckForPrivateParts(type, btype);
   END Record;

   PROCEDURE AddField(ident: Identifier; fieldtype: Type; VAR t: Type);
      VAR
	 fp: FieldList;
   BEGIN
      Assert(t # NIL);
      IF SearchFieldList(ident, t^.fields, fp) THEN
	 Error1("%I declared twice", ident);
      ELSE
	 CheckProcedureType(fieldtype);
	 IF (fieldtype # NIL) & (fieldtype^.ident = NIL) THEN
	    INC(fieldtype^.refcnt);
	 END;
	 NEW(fp);
	 WITH fp^ DO
	    id := ident;
	    type := fieldtype;
	    AllocateType(fieldtype, t^.size, offset, forwardDir);
	    link := t^.fields;
	    seekpos := 0;
	 END;
	 t^.fields := fp;
	 CheckForPrivateParts(t, fieldtype);
	 IF tracePointerOpt IN options THEN
	    t^.containsptr := t^.containsptr OR
			      (fieldtype # NIL) & fieldtype^.containsptr;
	 END;
      END;
   END AddField;

   PROCEDURE EndRecord(VAR type: Type);
   BEGIN
      Assert(type # NIL);
      Align(type^.size);
   END EndRecord;

   PROCEDURE CompareProc(name: Identifier; p1, p2: Type;
			 export, update: BOOLEAN) : BOOLEAN;

      (* compare two procedure(types)
	 and print error messages for any difference
	 `name' and `export' are needed for readable error messages;
	 if `update' is set, CompareProc is authorized to
	 call UpdateSymFile for p1 if paramV has been changed to
	 copyparamV or vice versa
      *)
      VAR
	 first: BOOLEAN; (* first error? *)
	 pl1, pl2: ParamList;
	 doUpdate: BOOLEAN;

      PROCEDURE ErrorHead;
      BEGIN
	 IF first THEN
	    first := FALSE;
	    IF export THEN
	       Error1("%I differs from export declaration:", name);
	    ELSE
	       Error1("%I differs from forward declaration:", name);
	    END;
	 END;
      END ErrorHead;

      PROCEDURE ErrorMsg(msg: ARRAY OF CHAR);
      BEGIN
	 ErrorHead; Error(msg);
      END ErrorMsg;

      PROCEDURE ErrorMsg1(msg: ARRAY OF CHAR);
      BEGIN
	 ErrorHead; Error1(msg, pl1^.id);
      END ErrorMsg1;

      PROCEDURE ErrorMsg2(msg: ARRAY OF CHAR);
      BEGIN
	 ErrorHead; Error2(msg, pl1^.id, pl2^.id);
      END ErrorMsg2;

      PROCEDURE CompareTypes(t1, t2: Type) : BOOLEAN;
	 (* return TRUE if types are equal *)
      BEGIN
	 RETURN (t1 = t2) OR (t1 = NIL) OR (t2 = NIL) OR
		(t1^.form = t2^.form) & (t1^.form = array) &
		t1^.dyn & t2^.dyn & CompareTypes(t1^.element, t2^.element)
      END CompareTypes;

   BEGIN
      doUpdate := FALSE;
      first := TRUE;
      IF p1^.function # p2^.function THEN
	 IF p1^.function THEN
	    ErrorMsg("as function declared");
	 ELSE
	    ErrorMsg("as procedure declared");
	 END;
      ELSIF p1^.function & (p1^.restype # p2^.restype) THEN
	 ErrorMsg("result types different");
      END;
      pl1 := p1^.param; pl2 := p2^.param;
      WHILE (pl1 # NIL) & (pl2 # NIL) DO
	 IF pl1^.id # pl2^.id THEN
	    ErrorMsg2("different parameter names: %I and %I");
	 ELSIF pl1^.varkind # pl2^.varkind THEN
	    IF (pl1^.varkind = varparamV) = (pl2^.varkind = varparamV) THEN
	       (* copyparamV/paramV clash *)
	       Assert(update);
	       pl1^.varkind := pl2^.varkind; doUpdate := TRUE;
	    ELSIF pl1^.varkind = varparamV THEN
	       Assert((pl2^.varkind = copyparamV) OR (pl2^.varkind = paramV));
	       ErrorMsg1("%I has been declared as VAR-parameter");
	    ELSE
	       Assert(pl2^.varkind = varparamV);
	       ErrorMsg1("%I has not been declared as VAR-parameter");
	    END;
	 ELSIF ~CompareTypes(pl1^.type, pl2^.type) THEN
	    ErrorMsg1("type of %I differs");
	 END;
	 pl1 := pl1^.link; pl2 := pl2^.link;
      END;
      IF pl1 # NIL THEN
	 WHILE pl1 # NIL DO
	    ErrorMsg1("missing parameter: %I");
	    pl1 := pl1^.link;
	 END;
      ELSIF pl2 # NIL THEN
	 WHILE pl2 # NIL DO
	    pl1 := pl2;
	    IF export THEN
	       ErrorMsg1("not found in export declaration: %I");
	    ELSE
	       ErrorMsg1("not found in forward declaration: %I");
	    END;
	    pl2 := pl2^.link;
	 END;
      END;
      IF doUpdate & ~errorflag THEN
	 UpdateSymFile(p1);
      END;
      (* first is TRUE if no errors occured *)
      RETURN first
   END CompareProc;

   PROCEDURE CompareFields(f1, f2: FieldList; missing: BOOLEAN) : BOOLEAN;
      (* compare `f1' against `f2':
	 check for every field of `f1' to be present in `f2'
	 return TRUE if ok
	 `missing' indicates the appropiate error message if record
	    fields of `f1' are not found in `f2'
      *)
      VAR
	 error: BOOLEAN;
	 field: FieldList; (* of `f2' *)
	 maxDiffOfOffsets: Size;
	 offsetError: BOOLEAN;
   BEGIN
      error := FALSE; maxDiffOfOffsets := 0;
      WHILE f1 # NIL DO
	 WITH f1^ DO
	    IF ~SearchFieldList(id, f2, field) THEN
	       IF missing THEN
		  Error1("missing record field: %I", id);
	       ELSE
		  Error1("added record field: %I", id);
	       END;
	       error := TRUE;
	    ELSIF ~TypesEqual(id, f1^.type, field^.type) THEN
	       (* error message printed by TypesEqual *)
	       error := TRUE;
	    ELSIF ~error & (f1^.offset # field^.offset) THEN
	       (* offsets may differ in case of extended record types;
		  so we check here if offset differences result
		  from different field orders or not
	       *)
	       offsetError := FALSE;
	       IF maxDiffOfOffsets = 0 THEN
		  maxDiffOfOffsets := f1^.offset - field^.offset;
	       ELSIF (maxDiffOfOffsets > 0) #
		     (f1^.offset - field^.offset > 0) THEN
		  offsetError := TRUE;
	       ELSIF ABS(f1^.offset - field^.offset) >
		     ABS(maxDiffOfOffsets) THEN
		  offsetError := TRUE;
	       END;
	       IF offsetError THEN
		  Error1("different order of record fields: %I", id);
		  error := TRUE;
	       END;
	    END;
	 END;
	 f1 := f1^.link;
      END;
      RETURN ~error
   END CompareFields;

   PROCEDURE TypesEqual(name: Identifier; t1, t2: Type) : BOOLEAN;
      (* compare two types for equality
	 two types are equal if they have
	 1) the same form
	 2) IF (t1^.ident = NIL) & (t2^.ident = NIL) THEN
	       TypesEqual(of the component types)
	    ELSE
	       type identity (i.e. type pointers are identical)
	    END;
	 error messages are printed for `name'
	 TypesEqual does not return FALSE on refcnt-errors but
	 prints an error message
      *)
      CONST
	 msg1 = "%I differs from export declaration: %s";
      VAR
	 ok: BOOLEAN;

      PROCEDURE Direct(t: Type) : BOOLEAN;
	 (* return TRUE
	    if `t' is NOT declared as t = t';
	 *)
      BEGIN
	 RETURN (t^.ident = NIL) OR (t^.ident^.name = name)
      END Direct;

   BEGIN
      IF (t1 = NIL) OR (t2 = NIL) OR (t1^.form # t2^.form) THEN
	 Error2(msg1, name, "different types");
	 RETURN FALSE
      END;
      IF t1 = t2 THEN RETURN TRUE END;
      IF Direct(t1) & (t2^.ident = NIL) THEN
	 IF (t1^.ident = NIL) & (t2^.ident = NIL) &
	    (t1^.refcnt # t2^.refcnt) THEN
	    Error2(msg1, name, "different lists");
	    (* for better recovery and avoidance of unnessary messages
	       no return of FALSE here
	    *)
	 END;
	 CASE t1^.form OF
	 | array:          IF t1^.length # t2^.length THEN
			      Error2(msg1, name, "array length");
			      RETURN FALSE
			   END;
			   RETURN TypesEqual(name, t1^.element, t2^.element)
	 | record:         IF t1^.basetype # t2^.basetype THEN
			      Error2(msg1, name, "different base types");
			      ok := FALSE;
			   ELSE
			      ok := TRUE;
			   END;
			   (* t1^.projection = t2^.projection *)
			   ok := CompareFields(t1^.fields, t2^.fields, TRUE) &
				 CompareFields(t2^.fields, t1^.fields, FALSE) &
				 ok;
			   RETURN ok
	 | pointer:        RETURN TypesEqual(name, t1^.reftype, t2^.reftype)
	 | proceduretype:  RETURN CompareProc(name, t1, t2, FALSE, FALSE)
	 (* ELSE -- should not happen because base types are identical *)
	 END;
      ELSE
	 Error2(msg1, name, "different types");
	 RETURN FALSE
      END;
   END TypesEqual;

   PROCEDURE CompareTypes(name: Identifier; deft, modt: Type);
      (* `name': name of both types
	 `deft': type in definition
	 `modt': type in module
	 both types must be identical in their declaration
	 print error messages about any difference (but size differences)
      *)
   BEGIN
      IF (deft = NIL) OR (modt = NIL) OR (deft = modt) THEN
	 RETURN
      END;
      IF ~TypesEqual(name, deft, modt) THEN
	 (* error messages already printed *)
      END;
   END CompareTypes;

   PROCEDURE DeclType(id: Identifier; t: Type);
      VAR
	 ip: Ident;
	 field: FieldList;

      PROCEDURE IsRec(ip: Ident) : BOOLEAN;
      BEGIN
	 WITH ip^ DO
	    RETURN (class = typeC) & (type # NIL) & (type^.form = record) &
		   ~type^.extended & (type^.ident = ip)
	 END;
      END IsRec;

      PROCEDURE PrivatePart(public, private: Type);
	 VAR
	    error: BOOLEAN;
	    termfield: FieldList;

	 PROCEDURE CopySeekPositions(private, public, termfield: FieldList);
	    VAR
	       field, fp: FieldList;
	 BEGIN
	    field := private;
	    WHILE field # termfield DO
	       Assert(field # NIL);
	       WITH field^ DO
		  IF SearchFieldList(id, public, fp) THEN
		     seekpos := fp^.seekpos;
		     Assert(seekpos # 0);
		  END;
	       END;
	       field := field^.link;
	    END;
	 END CopySeekPositions;

	 PROCEDURE CheckFieldOrder(basefields, private, public: FieldList);
	    VAR
	       field, fp: FieldList;
	       maxPublicOffset: Size;
	 BEGIN
	    Assert(public # NIL);

	    (* at first determine the maximal offset of a public component
	       in private
	    *)
	    maxPublicOffset := 0;
	    field := private;
	    WHILE field # basefields DO
	       Assert(field # NIL);
	       WITH field^ DO
		  IF SearchFieldList(id, public, fp) &
			(offset > maxPublicOffset) THEN
		     maxPublicOffset := offset;
		  END;
	       END;
	       field := field^.link;
	    END;

	    (* now check if one of the private components has an offset
	       which is less than maxPublicOffset
	    *)
	    field := private;
	    WHILE field # basefields DO
	       WITH field^ DO
		  IF ~SearchFieldList(id, public, fp) &
			(offset < maxPublicOffset) THEN
		     Error("public fields must not be preceded by private fields");
		  END;
	       END;
	       field := field^.link;
	    END;
	 END CheckFieldOrder;

      BEGIN (* PrivatePart *)
	 (* check for correct extension *)
	 error := FALSE;
	 IF public^.basetype # private^.basetype THEN
	    Error1("base types of %I are different", id);
	    error := TRUE;
	 END;
	 error := CompareFields(public^.fields, private^.fields, TRUE) OR error;

	 (* check that public fields are followed by private fields *)
	 IF ~error & (public^.fields # NIL) THEN
	    IF public^.basetype # NIL THEN
	       CheckFieldOrder(public^.basetype^.fields,
			       private^.fields, public^.fields);
	    ELSE
	       CheckFieldOrder(NIL, private^.fields, public^.fields);
	    END;
	 END;

	 IF ~errorflag THEN (* CopySeekPositions assumes errorfree structures *)
	    IF public^.basetype # NIL THEN
	       termfield := public^.basetype^.fields;
	    ELSE
	       termfield := NIL;
	    END;
	    CopySeekPositions(private^.fields, public^.fields, termfield);
	 END;
	 public^.fields := private^.fields;
	 public^.extended := TRUE;
	 public^.containsptr := private^.containsptr;
      END PrivatePart;

   BEGIN (* DeclType *)
      At(previdpos);
      CheckProcedureType(t);
      IF Exported(id, typeC, ip) & (t # NIL) THEN
	 IF (t^.form = record) & (t^.ident = NIL) & IsRec(ip) THEN
	    WITH ip^ DO
	       Assert(type^.projection);
	       IF ~type^.extended THEN
		  PrivatePart(ip^.type, t);
	       (* ELSE -- error message will be printed by Enter *)
	       END;
	    END;
	 ELSE
	    CompareTypes(id, ip^.type, t);
	 END;
	 (* update symbolfile iff
	    (1) the private part needs more space, or
	    (2) the private part needs less space
		and we are authorized to update the symbol file
	 *)
	 IF ~errorflag & (t^.size # ip^.type^.size) &
	    ((updatesym = hardUpdate) OR (t^.size > ip^.type^.size)) THEN
	    ip^.type^.size := t^.size;
	    UpdateSymFile(ip^.type);
	 END;
	 (* DisposeType(t); --- TO BE DONE ---- *)
	 t := ip^.type;
      END;
      IF Enter(id, typeC, ip) THEN
	 IF (t # NIL) & (t^.ident = NIL) THEN
	    WITH t^ DO
	       ident := ip;
	       IF (form = record) & defunit THEN
		  projection := TRUE;
		  extended := FALSE;
		  extmod := mainmod;
	       END;
	    END;
	 END;
	 WITH ip^ DO
	    type := t;
	    mod := mainmod;
	    exported := defunit;
	 END;
	 CheckExport(ip);
	 IF (ip^.type # NIL) &
	       ((ip^.type^.form # pointer) OR (ip^.type^.reftype # NIL)) THEN
	    (* else reference will be generated by CheckForwards *)
	    RefIdent(ip, previdpos);
	 END;
      ELSE
	 (* DisposeType(t); -- TO BE DONE *)
      END;
   END DeclType;


   (* ---------------------------------------------------------	*)
   (* variable declarations					*)
   (* ---------------------------------------------------------	*)

   PROCEDURE IndirectType(t: Type) : BOOLEAN;
      (* return TRUE on types which must be represented indirectly *)
   BEGIN
      RETURN (t # NIL) & (t^.form = array) & t^.dyn
   END IndirectType;

   PROCEDURE DeclVar(id: Identifier; t: Type);
      (* VAR id: type *)
      VAR
	 ip: Ident;
	 il: IdentList;
	 indefmod: BOOLEAN;
   BEGIN
      CheckForwards((* printErrors = *) FALSE);
      CheckProcedureType(t);
      IF (t # NIL) & (t^.ident = NIL) THEN
	 INC(t^.refcnt);
      END;
      IF Exported(id, varC, ip) THEN
	 CompareTypes(id, ip^.type, t);
	 IF (t # NIL) & (t^.size # ip^.type^.size) THEN
	    ip^.type^.size := t^.size;
	    UpdateSymFile(ip^.type);
	 END;
	 indefmod := TRUE;
      ELSE
	 indefmod := FALSE;
      END;
      IF Enter(id, varC, ip) THEN
	 WITH ip^ DO
	    type := t;
	    mod := mainmod;
	    exported := defunit OR indefmod;
	    (* plevel initialized by `Enter' *)
	    IF openproc # NIL THEN
	       AllocateType(t, openproc^.varoffset, offset, stackdirection);
	       WITH openproc^ DO
		  NEW(il);
		  WITH il^ DO
		     ident := ip;
		     link := local;
		  END;
		  local := il;
	       END;
	    ELSE (* global variable *)
	       offset := 0;
	       NEW(il);
	       WITH il^ DO
		  ident := ip;
		  link := globalvars;
	       END;
	       globalvars := il;
	    END;
	    indirect := IndirectType(t);
	    varkind := noparamV;
	 END;
	 CheckExport(ip);
	 RefIdent(ip, previdpos);
      END;
   END DeclVar;

   (* ---------------------------------------------------------	*)
   (* construction of procedures and procedure types:		*)
   (* procedure types:						*)
   (*    ProcedureType { AddParameterType } [ Function ]	*)
   (* procedure declaration:					*)
   (*    ProcedureType { AddParameter } [ Function ] DeclProc	*)
   (* ---------------------------------------------------------	*)

   MODULE ForwardDeclarations; (* of procedures *)

      FROM Exception IMPORT Assert;
      FROM Memory IMPORT ALLOCATE, DEALLOCATE;
      FROM Scan IMPORT Error1;
      FROM SymTab IMPORT Level, Ident, IdentClass;

      EXPORT EnterForwardDeclaration, CheckForwardDeclarations;

      TYPE
	 List = POINTER TO ForwardRec;
	 ForwardRec =
	    RECORD
	       ip: Ident;
	       link: List;
	    END;
      VAR
	 list: List;

      PROCEDURE EnterForwardDeclaration(procip: Ident);
	 VAR new: List;
      BEGIN
	 Assert((procip # NIL) & (procip^.class = procedureC));
	 NEW(new);
	 WITH new^ DO
	    ip := procip;
	    link := list;
	 END;
	 list := new;
      END EnterForwardDeclaration;

      PROCEDURE CheckForwardDeclarations;
	 (* to be called before SymTab.CloseScope *)
	 VAR
	    old: List;
      BEGIN
	 WHILE list # NIL DO
	    WITH list^.ip^ DO
	       IF forward & (Level() = plevel) THEN
		  Error1("actual declaration of %I missing", name);
	       END;
	    END;
	    old := list;
	    list := list^.link;
	    DISPOSE(old);
	 END;
      END CheckForwardDeclarations;

   BEGIN
      list := NIL;
   END ForwardDeclarations;

   PROCEDURE SetParamOffsets(t: Type; plevel: CARDINAL; VAR parmoffset: Size);
      VAR
	 pl: ParamList;
	 dummyoffs: Size;
   BEGIN
      parmoffset := 0;
      pl := t^.param;
      WHILE pl # NIL DO
	 WITH pl^ DO
	    IF ((varkind = varparamV) OR (varkind = copyparamV)) &
		  (type # NIL) & ((type^.form # array) OR ~type^.dyn) THEN
	       IF (varkind = varparamV) &
		     ((type^.form = record) OR
		      (type^.form = pointer) & (type^.reftype # NIL) &
		      (type^.reftype^.form = record)) THEN
		  Allocate(2*oneword, parmoffset, offset, stackdirection);
	       ELSE
		  Allocate(oneword, parmoffset, offset, stackdirection);
	       END;
	    ELSE (* varkind = paramV *)
	       AllocateType(type, parmoffset, offset, stackdirection);
	       IF alignparam THEN
		  Align(parmoffset); (* force alignment *)
	       END;
	    END;
	 END;
	 pl := pl^.link;
      END;
      IF plevel > global THEN
	 (* allocate static link *)
	 Allocate(oneword, parmoffset, dummyoffs, stackdirection);
      END;

      (* now calculate correct offset after
	 we know the total length of the parameter block
      *)
      pl := t^.param;
      WHILE pl # NIL DO
	 DEC(pl^.offset, parmoffset);
	 IF stackdirection = forwardDir THEN
	    DEC(pl^.offset, procmarkspace);
	 ELSE
	    INC(pl^.offset, procmarkspace);
	 END;
	 pl := pl^.link;
      END;
   END SetParamOffsets;

   PROCEDURE CheckProcedureType(t: Type);
      VAR
	 parmoffset: Size;
   BEGIN
      IF t # NIL THEN
	 WITH t^ DO
	    IF (form = proceduretype) & (param # NIL) & (param^.offset = 0) THEN
	       SetParamOffsets(t, global, parmoffset);
	    END;
	 END;
      END;
   END CheckProcedureType;

   PROCEDURE ProcedureType(VAR type: Type);
      (* 1st call for procedure types *)
   BEGIN
      InitType(type);
      WITH type^ DO
	 ident := NIL;
	 size := oneword;
	 form := proceduretype;
	 function := FALSE;
	 std := FALSE;
	 param := NIL;
      END;
      paramtail := NIL;
   END ProcedureType;

   PROCEDURE AddParameter(varparam: BOOLEAN; paramid: Identifier;
			  paramtype: Type; VAR type: Type);
      (* to be called in the order of parameters *)
      VAR
	 pl: ParamList;
   BEGIN
      Assert(type # NIL);
      IF (paramid # noident) & SearchParamList(paramid, type^.param, pl) THEN
	 Error1("%I twice declared", paramid);
      ELSE
	 NEW(pl);
	 WITH pl^ DO
	    id := paramid;
	    type := paramtype; CheckProcedureType(paramtype);
	    IF varparam THEN
	       varkind := varparamV;
	    ELSIF (paramtype # NIL) &
		  ((paramtype^.size > 3 * oneword) OR
		    IndirectType(paramtype) OR
		    paramtype^.containsptr &
		    (paramtype^.form IN FormSet{array, record})) THEN
	       (* be sure that LONGREAL's are not copy parameters! *)
	       (* we enforce even short pointer containing arrays
		  or records to be copy parameters to avoid trouble with the GC;
		  simple pointers are catched by GenExpr/LoadParam
	       *)
	       varkind := copyparamV;
	    ELSE
	       varkind := paramV;
	    END;
	    offset := 0;
	    link := NIL;
	 END;
	 IF paramtail = NIL THEN
	    type^.param := pl;
	 ELSE
	    paramtail^.link := pl;
	 END;
	 paramtail := pl;
      END;
   END AddParameter;

   PROCEDURE AddParameterType(varparam: BOOLEAN; paramtype: Type;
			      VAR type: Type);
      (* like AddParameter but for procedure types (with nameless parameters) *)
   BEGIN
      AddParameter(varparam, noident, paramtype, type);
   END AddParameterType;

   PROCEDURE Function(rtype: Type; VAR type: Type);
   BEGIN
      Assert(type # NIL);
      WITH type^ DO
	 function := TRUE;
	 restype := rtype;
	 IF restype # NIL THEN
	    WITH restype^ DO
	       IF (form = record) OR (form = array) THEN
		  Error("functions must not return arrays nor records");
		  restype := NIL;
	       END;
	    END;
	 END;
      END;
   END Function;

   PROCEDURE DeclProc(id: Identifier; forw: BOOLEAN; t: Type);
      VAR
	 ip: Ident;
	 isexported: BOOLEAN;

   BEGIN
      lastprocid := id; lastproctype := t;
      Assert(t # NIL);
      IF Exported(id, procedureC, ip) THEN
	 IF ~CompareProc(id, ip^.type, t, (* export = *) TRUE,
			 (* update = *) TRUE) THEN
	    (* error messages already printed *)
	 END;
	 isexported := TRUE;
      ELSE
	 isexported := FALSE;
      END;
      At(previdpos);
      CheckForwards((* printErrors = *) TRUE);
      IF ~forw & Find(Level(), id, ip) &
	 (ip^.class = procedureC) & ip^.forward & (ip^.plevel = Level()) THEN
	 IF ~CompareProc(id, ip^.type, t, (* export = *) FALSE,
			 (* update = *) FALSE) THEN
	    (* error messages already printed *)
	    (* DisposeType(ip^.type) *)
	    (* take this type to avoid additional error messages;
	       this must not be done if code is generated because
	       `t' does not have correct offsets
	    *)
	    ip^.type := t;
	 END;
	 ip^.forward := FALSE;
	 RefIdent(ip, previdpos);
      ELSIF Enter(id, procedureC, ip) THEN
	 WITH ip^ DO
	    type := t;
	    mod := mainmod;
	    exported := defunit OR isexported;
	    IF forw THEN
	       IF defunit THEN
		  Error1("%I cannot be forward declared in a definition", id);
		  forward := FALSE;
	       ELSE
		  forward := TRUE;
		  EnterForwardDeclaration(ip);
	       END;
	    ELSE
	       forward := FALSE;
	    END;
	    procno := nextprocno; INC(nextprocno);
	    local := NIL;
	    params := NIL;
	    varoffset := 0;
	    SetParamOffsets(t, plevel, parmoffset);
	 END;
	 CheckExport(ip);
	 IF ~forw THEN
	    RefIdent(ip, previdpos);
	 END;
      END;
   END DeclProc;


   (* ---------------------------------------------------------	*)
   (* resolution of grammar conflicts:				*)
   (* (1)	ident "(" ident ")"				*)
   (*		   can be a procedure call			*)
   (* 		   or a type guard				*)
   (* (2)	ident "." ident					*)
   (*		   can be a qualident				*)
   (*		   or qualident "." ident (* field selection *)	*)
   (* ---------------------------------------------------------	*)

   (*****
   PROCEDURE IsVarNoProcvar(qualid: QualIdent) : BOOLEAN;
      (* returns TRUE if 'qualid' is a variable in the current scope *)
      (*                 and no procedure variable		 *)
      (* to be called on 'qualid' "(" to decide whether a type guard *)
      (* or a procedure call is coming now;			 *)
      (* This test does not print any error messages.		 *)

      VAR
	 ip: Ident;
	 modip: Ident;
   BEGIN
      WITH qualid DO
	 IF qualified THEN
	    IF ~Search(modname, modip) OR
	       ~SearchList(idname, modip^.export, ip) THEN
	       RETURN FALSE
	    END;
	 ELSE
	    IF ~Search(idname, ip) THEN RETURN FALSE END;
	 END;
	 RETURN (ip^.class = varC) &
		((ip^.type = NIL) OR (ip^.type^.form # proceduretype))
      END;
   END IsVarNoProcvar;
   *****)

   PROCEDURE IsVar(id: Identifier) : BOOLEAN;
      (* case (2) *)
      VAR
	 ip: Ident;
   BEGIN
      RETURN Search(id, ip) & (ip^.class = varC)
   END IsVar;

BEGIN
   InitStd;
   InitSYSTEM;
   PutIdentChar(' '); PutIdent(noident);
   globalvars := NIL;
   mainmod := NIL;
   nextprocno := 1;
   lastprocid := noident;
   lastproctype := NIL;
   AddLegalOptions(CompilerOptions{taggedPointerOpt, tracePointerOpt});
END SymDef.
