(* Ulm's Oberon Compiler
   Copyright (c) 1989 by University of Ulm, SAI, D-W-7900 Ulm, Germany
   ----------------------------------------------------------------------------
   $Id: SymRef.m2,v 0.4 1993/06/18 15:31:10 borchert Exp $
   ----------------------------------------------------------------------------
   $Log: SymRef.m2,v $
   Revision 0.4  1993/06/18  15:31:10  borchert
   WriteNumber accepts now LONGINTs

   Revision 0.3  1993/06/16  09:49:05  borchert
   int16SY added for SYSTEM.INT16

   Revision 0.2  1993/05/03  11:05:52  borchert
   minor fix: multiple emits of types are now avoided

   Revision 0.1  1992/07/30  10:49:46  borchert
   Initial revision

   ----------------------------------------------------------------------------
*)

IMPLEMENTATION MODULE SymRef; (* AFB 7/90 *)

   (* generation of reference files *)

   FROM Exception IMPORT Assert, IOFault;
   FROM IdentSys IMPORT GetIdentChar, Identifier;
   FROM Scan IMPORT errorflag, Pos;
   FROM ScanArgs IMPORT FileName;
   FROM StdIO IMPORT FILE, Fopen, read, write, Fclose, Fgetc, Fputc, Fwrite;
   FROM Storage IMPORT ALLOCATE, DEALLOCATE;
   FROM SymTab IMPORT FieldList, Form, Key, Ident, IdentList, IdentClass,
      Type, mainmod;
   FROM SYSTEM IMPORT ADR;

   (* version changes:

      3: SYSTEM.INT16 has been added to the standard types
   *)

   CONST
      version = 3; (* reference file version *)
      magic = 200AFB00H + version; (* magic number of reference file *)

   TYPE
      Symbol =
	 (dummySY,
	 arraySY, booleanSY, byteSY, charSY, dirSY,		(* 1..5 *)
	 endSY, fieldSY, importSY, indirSY, integerSY,		(* 6..10 *)
	 int16SY, longintSY, longrealSY, pointerSY, procSY,	(* 11..15 *)
	 procedureSY, realSY, recordSY, setSY,			(* 16..19 *)
	 shortintSY, typeSY, varSY);				(* 20..22 *)
      TypeList = POINTER TO TypeListRec;
      TypeListRec =
	 RECORD
	    type: Type;
	    link: TypeList;
	 END;

   VAR
      open: BOOLEAN; (* writing a reference file? *)
      ref: FILE; (* valid only if open is TRUE *)
      reffile: FileName; (* name of reference file *)
      nestlevel: CARDINAL; (* of RefOpenBlock and RefCloseBlock *)
      nextrtypeno: CARDINAL; (* next rtypeno *)
      workup: TypeList; (* referenced but not dumped *)
      imports: IdentList; (* list of imports *)

   PROCEDURE OpenRef(filename: FileName);
      (* open reference file for writing and
	 enable reference file generation
	 (else all other procedures are noops)
      *)
   BEGIN
      Assert(~open);
      IF ~Fopen(ref, filename, write, (* buffered = *) TRUE) THEN
	 IOFault(filename);
      END;
      reffile := filename;
      open := TRUE;
      nestlevel := 0;
      nextrtypeno := 1;
      workup := NIL;
      imports := NIL;
   END OpenRef;

   PROCEDURE CloseRef;
      (* close reference file *)
   BEGIN
      Assert(open);
      IF ~Fclose(ref) THEN
	 IOFault(reffile);
      END;
      Assert(errorflag OR (nestlevel = 0));
      open := FALSE;
   END CloseRef;

   PROCEDURE WriteSy(sy: Symbol);
      VAR
	 ch: CHAR;
   BEGIN
      ch := CHR(ORD(sy));
      IF ~Fputc(ch, ref) THEN
	 IOFault(reffile);
      END;
   END WriteSy;

   PROCEDURE WriteChar(ch: CHAR);
   BEGIN
      IF ~Fputc(ch, ref) THEN
	 IOFault(reffile);
      END;
   END WriteChar;

   PROCEDURE WriteNumber(number: LONGINT);
      VAR count: CARDINAL;
   BEGIN
      count := 1;
      IF ~Fwrite(ADR(number), SIZE(number), count, ref) OR (count # 1) THEN
	 IOFault(reffile);
      END;
   END WriteNumber;

   PROCEDURE WriteKey(key: Key);
      VAR count: CARDINAL;
   BEGIN
      count := 1;
      IF ~Fwrite(ADR(key), SIZE(key), count, ref) OR (count # 1) THEN
	 IOFault(reffile);
      END;
   END WriteKey;

   PROCEDURE WriteIdentifier(name: Identifier);
      VAR
	 ch: CHAR;
   BEGIN
      REPEAT
	 GetIdentChar(name, ch);
	 WriteChar(ch);
      UNTIL ch = 0C;
   END WriteIdentifier;

   PROCEDURE WriteName(ident: Ident);

      PROCEDURE WriteIdentifier(name: Identifier);
	 VAR
	    ch: CHAR;
      BEGIN
	 LOOP
	    GetIdentChar(name, ch);
	    IF ch = 0C THEN EXIT END;
	    WriteChar(ch);
	 END;
      END WriteIdentifier;

   BEGIN
      IF ident # NIL THEN
	 WITH ident^ DO
	    IF (class # moduleC) & (mod # NIL) & (mod # mainmod) THEN
	       WriteIdentifier(mod^.origname); WriteChar(".");
	    END;
	    IF class = moduleC THEN
	       WriteIdentifier(origname);
	    ELSE
	       WriteIdentifier(name);
	    END;
	 END;
      END;
      WriteChar(0C);
   END WriteName;

   PROCEDURE RefImport(ident: Ident);
      (* write reference record for import *)
      VAR
	 import: IdentList;
   BEGIN
      IF open & (ident # NIL) THEN
	 Assert(ident^.class = moduleC);
	 NEW(import);
	 import^.ident := ident;
	 import^.link := imports;
	 imports := import;
      END;
   END RefImport;

   PROCEDURE WriteImports;
      VAR
	 import: IdentList;
	 old: IdentList;
   BEGIN
      import := imports;
      WHILE import # NIL DO
	 WriteSy(importSY);
	 WITH import^ DO
	    WriteName(ident);
	    WriteKey(ident^.key);
	 END;
	 old := import;
	 import := import^.link;
	 DISPOSE(old);
      END;
      imports := NIL;
   END WriteImports;

   PROCEDURE TypeNo(type: Type) : CARDINAL;
      VAR
	 new: TypeList;
   BEGIN
      Assert(type # NIL);
      WITH type^ DO
	 IF rtypeno = 0 THEN
	    rtypeno := nextrtypeno; INC(nextrtypeno);
	    NEW(new);
	    new^.type := type;
	    new^.link := workup;
	    workup := new;
	 END;
	 RETURN rtypeno
      END;
   END TypeNo;

   PROCEDURE WriteType(type: Type);

      PROCEDURE WriteRecord(type: Type);
	 VAR
	    fp: FieldList;
      BEGIN
	 WITH type^ DO
	    Assert(form = record);
	    WriteSy(recordSY);
	    IF basetype = NIL THEN
	       WriteNumber(0);
	    ELSE
	       WriteNumber(TypeNo(basetype));
	    END;
	    fp := fields;
	    WHILE fp # NIL DO
	       WITH fp^ DO
		  WriteSy(fieldSY);
		  WriteIdentifier(id);
		  WriteNumber(offset);
		  WriteNumber(TypeNo(type));
	       END;
	       fp := fp^.link;
	    END;
	    WriteSy(endSY);
	 END;
      END WriteRecord;

   BEGIN
      WITH type^ DO
	 WriteSy(typeSY);
	 IF rtypeno = 0 THEN
	    rtypeno := nextrtypeno; INC(nextrtypeno);
	 END;
	 WriteNumber(rtypeno);
	 IF (ident # NIL) & (ident^.mod # NIL) THEN
	    WriteName(ident);
	 ELSE
	    (* don't define standard identifiers here *)
	    WriteChar(0C);
	 END;
	 WriteNumber(size);
	 CASE form OF
	 | shortint:       WriteSy(shortintSY);
	 | int16:          WriteSy(int16SY);
	 | integer:        WriteSy(integerSY);
	 | longint:        WriteSy(longintSY);
	 | real:           WriteSy(realSY);
	 | longreal:       WriteSy(longrealSY);
	 | boolean:        WriteSy(booleanSY);
	 | char:           WriteSy(charSY);
	 | set:            WriteSy(setSY);
	 | byte:           WriteSy(byteSY);
	 | array:          WriteSy(arraySY);
			   IF dyn THEN
			      WriteNumber(-1);
			   ELSE
			      WriteNumber(length);
			   END;
			   WriteNumber(TypeNo(element));
	 | record:         WriteRecord(type);
	 | pointer:        WriteSy(pointerSY); WriteNumber(TypeNo(reftype));
	 | proceduretype:  WriteSy(procedureSY);
	 ELSE
	    Assert(FALSE);
	 END;
      END;
   END WriteType;

   PROCEDURE Workup;
      VAR
	 list, old: TypeList;
   BEGIN
      list := workup;
      workup := NIL;
      WHILE list # NIL DO
	 WriteType(list^.type);
	 old := list;
	 list := list^.link;
	 DISPOSE(old);
      END;
   END Workup;

   PROCEDURE RefIdent(ident: Ident; pos: Pos);
      (* write reference record for modules, types, variables and procedures;
	 pos should point to the identifier
      *)
   BEGIN
      IF ~open OR (ident = NIL) OR errorflag THEN RETURN END;
      IF ident^.class = procedureC THEN
	 WHILE workup # NIL DO
	    Workup;
	 END;
      ELSIF (ident^.class = typeC) & (ident^.type^.rtypeno # 0) THEN
	 (* already written or in workup list *)
	 RETURN
      END;
      WITH ident^ DO
	 CASE class OF
	 | moduleC:     WriteNumber(magic); WriteName(ident); WriteKey(key);
			WriteImports;
	 | typeC:       WriteType(type);
	 | procedureC:  WriteSy(procSY); WriteName(ident);
			WriteNumber(pos.line);
			WriteNumber(procno);
			WriteNumber(parmoffset);
	 | varC:        WriteSy(varSY); WriteName(ident);
			WriteNumber(pos.line);
			WriteNumber(offset);
			IF indirect THEN
			   WriteSy(indirSY);
			ELSE
			   WriteSy(dirSY);
			END;
			WriteNumber(TypeNo(type));
	 ELSE
	    Assert(FALSE);
	 END;
      END;
   END RefIdent;

   PROCEDURE RefOpenBlock;
      (* blocks are to be opened after RefIdent of modules or procedures *)
   BEGIN
      IF ~open OR errorflag THEN RETURN END;
      INC(nestlevel);
   END RefOpenBlock;

   PROCEDURE RefCloseBlock;
      (* to be called at end of procedure/module *)
   BEGIN
      IF ~open OR errorflag THEN RETURN END;
      WHILE workup # NIL DO
	 Workup;
      END;
      Assert(nestlevel > 0);
      DEC(nestlevel);
      WriteSy(endSY);
   END RefCloseBlock;

BEGIN
   open := FALSE;
END SymRef.
