(* Ulm's Oberon Compiler
   Copyright (c) 1989 by University of Ulm, SAI, D-W-7900 Ulm, Germany
   ----------------------------------------------------------------------------
   $Id: SymTab.m2,v 0.1 1992/07/30 10:49:50 borchert Exp $
   ----------------------------------------------------------------------------
   $Log: SymTab.m2,v $
   Revision 0.1  1992/07/30  10:49:50  borchert
   Initial revision

   ----------------------------------------------------------------------------
*)

IMPLEMENTATION MODULE SymTab; (* AFB 1/89 *)

   FROM Exception IMPORT Assert;
   FROM IdentSys IMPORT Identifier;
   FROM Scan IMPORT Error, Error1;
   FROM Memory IMPORT ALLOCATE, DEALLOCATE;

   CONST
      maxnestlevel = 64;		(* maximal scope nesting level *)
      hashtabsize = 4561;		(* MUST be prime number *)

   TYPE
      HiddenPart = POINTER TO HiddenPartRec;
      HiddenPartRec =		(* hidden part of IdentRec *)
	 RECORD			(* initialized during Enter *)
	    dlevel: CARDINAL;	(* declaration level *)
	    nlink: Ident;	(* name link (decreasing level) *)
	    dlink: Ident;	(* display link *)
	    lastuse: CARDINAL;	(* level of last use *)
	 END;

   VAR
      display: ARRAY [0..maxnestlevel-1] OF Ident;
	 (* level 0: standard names *)
	 (* level 1: names of imported modules and our module name *)
	 (* level 2: global names *)
	 (* level 3: names local to global procedures *)
      level: CARDINAL;		(* < maxnestlevel *)
      leveloverflow: CARDINAL;
      bucket: ARRAY [0..hashtabsize-1] OF Ident;

   (*
        design of display and hash-table (some fields are omitted):

                     bucket +---+---+---+---+---+---+---+---+
                            |   |   |   | o |   |   | o |   |
         display            +---+---+---+-|-+---+---+-|-+---+
                                          |           |
         +-----+                          |           |
         |  o------------>+-----+<-+      |           |     MODULE a;
         +-----+          |  a  |  |      |           |
         |  o--------+    +-----+  |      |           |        VAR a, b
         +-----+     |    | NIL |  |      |           |
         |     |     |    +-----+  |      |           |
         +-----+     |    | NIL |  |      |           |
         |     |     |    +-----+  |      |           |
         +-----+     |             |      |           |
         |     |     +----------------V   V           V
         +-----+                   | +-----+  +-->+-----+
         |     |                   | |  a  |  |   |  b  |   name
         +-----+                   | +-----+  |   +-----+
         |     |                   +----o  |  |   | NIL |   nlink
         +-----+                     +-----+  |   +-----+
				     |  o-----+   | NIL |   dlink
				     +-----+      +-----+
   *)

   (* ---------------------------------------------------------	*)
   (* scopes							*)
   (* ---------------------------------------------------------	*)

   PROCEDURE OpenScope;
      (* the first call opens the module scope *)
      (* the second call opens the module; *)
      (* all further calls are for procedures only *)
      (* module: to be called AFTER the imports *)
      (* procedures: to be called AFTER DeclProc *)
   BEGIN
      IF level < HIGH(display) THEN
	 INC(level);
	 display[level] := NIL;
      ELSE
	 IF leveloverflow = 0 THEN
	    Error("%R: maximal nest level reached");
	 END;
	 INC(leveloverflow);
      END;
   END OpenScope;

   PROCEDURE CloseScope;
      (* to be called at END (ProcedureName|ModuleName) *)

      PROCEDURE ReleaseSymbols;
	 VAR
	    ptr, old: Ident;
	    index: CARDINAL;	(* index into bucket table *)
	    nptr: Ident;

	    PROCEDURE DisposeList(list: IdentList);
	       VAR
		  old: IdentList;
	    BEGIN
	       WHILE list # NIL DO
		  old := list;
		  list := list^.link;
		  DISPOSE(old);
	       END;
	    END DisposeList;

	    PROCEDURE DisposeType(type: Type);
	    BEGIN
	       IF (type # NIL) & (type^.ident = NIL) THEN
		  WITH type^ DO
		     IF refcnt <= 1 THEN
			(* dispose type *)
			(* TO BE DONE *)
		     ELSE
			DEC(refcnt);
		     END;
		  END;
	       END;
	    END DisposeType;

      BEGIN
	 ptr := display[level];
	 WHILE ptr # NIL DO
	    WITH ptr^ DO
	       WITH hidden^ DO
		  index := HashVal(name);
		  nptr := bucket[index];
		  WHILE (nptr # NIL) & (nptr^.hidden^.dlevel = level) DO
		     nptr := nptr^.hidden^.nlink;
		  END;
		  bucket[index] := nptr;
	       END;
	       IF class = moduleC THEN
	          DisposeList(export);
	       ELSIF class # badclass THEN
		  IF (type # NIL) & (type^.ident = NIL) THEN
		     DisposeType(type);
		  END;
		  IF class = procedureC THEN
		     DisposeList(local);
		     DisposeList(params);
		  END;
	       END;
	    END;
	    old := ptr;
	    ptr := ptr^.hidden^.dlink;
	    DISPOSE(old^.hidden);
	    DISPOSE(old);
	 END;
      END ReleaseSymbols;

      PROCEDURE UpdateLastUse;
	 VAR
	    ip: Ident;
	    index: CARDINAL;
      BEGIN
	 FOR index := 0 TO level DO
	    ip := display[index];
	    WHILE ip # NIL DO
	       IF ip^.hidden^.lastuse > level THEN
		  ip^.hidden^.lastuse := level;
	       END;
	       ip := ip^.hidden^.dlink;
	    END;
	 END;
      END UpdateLastUse;

   BEGIN
      IF leveloverflow > 0 THEN
	 DEC(leveloverflow);
      ELSE
	 Assert(level > 0);
	 ReleaseSymbols;
	 DEC(level);
	 UpdateLastUse;
      END;
   END CloseScope;

   PROCEDURE Used(ip: Ident) : BOOLEAN;
      (* return TRUE if `ip' has been used (i.e. Search called) *)
   BEGIN
      RETURN ip^.hidden^.lastuse > 0
   END Used;

   PROCEDURE Touch(ip: Ident; level: CARDINAL);
      (* mark `ip' as used *)
   BEGIN
      IF ip^.hidden^.lastuse < level THEN
	 ip^.hidden^.lastuse := level;
      END;
   END Touch;

   (* CheckId & Insert: common parts of Enter & Include *)

   PROCEDURE CheckId(name: Identifier) : BOOLEAN;
      VAR
	 other: Ident; (* twice declared? *)
   BEGIN
      IF Find(level, name, other) THEN
	 WITH other^.hidden^ DO
	    IF dlevel = level THEN
	       Error1("%I twice declared", name); RETURN FALSE
	    ELSIF (lastuse >= level) & (other^.class # badclass) THEN
	       Error1("%I used before declared (scope overlap)", name);
	       RETURN FALSE
	    END;
	 END;
      END;
      RETURN TRUE
   END CheckId;

   PROCEDURE Insert(ident: Ident);
      VAR
	 index: CARDINAL;	(* into bucket table *)
   BEGIN
      NEW(ident^.hidden);
      WITH ident^ DO
	 identno := 0;
	 IF (class # moduleC) & (class # badclass) THEN
	    exported := FALSE; (* initialization *)
	 END;
	 IF (class = varC) OR (class = procedureC) THEN
	    plevel := level;
	 END;
	 WITH hidden^ DO
	    dlevel := level;
	    lastuse := 0;
	    (* insert `ident' into hash table *)
	    index := HashVal(name);
	    nlink := bucket[index];
	    bucket[index] := ident;
	    (* insert `ident' into current scope *)
	    dlink := display[level];
	    display[level] := ident;
	 END;
      END;
   END Insert;

   PROCEDURE InsertBad(id: Identifier; VAR ident: Ident);
      CONST
	 badlevel = 0;		(* MUST be lowest level *)
      VAR
	 index: CARDINAL;	(* into bucket table *)
	 hashptr: Ident;
   BEGIN
      NEW(ident);
      NEW(ident^.hidden);
      WITH ident^ DO
	 name := id;
	 identno := 0;
	 class := badclass;
	 error := TRUE;
	 WITH hidden^ DO
	    dlevel := badlevel;
	    lastuse := 0;
	    (* insert `ident' into hash table (at tail of linked list) *)
	    nlink := NIL;
	    index := HashVal(name);
	    hashptr := bucket[index];
	    IF hashptr = NIL THEN
	       bucket[index] := ident;
	    ELSE
	       WHILE hashptr^.hidden^.nlink # NIL DO
		  hashptr := hashptr^.hidden^.nlink;
	       END;
	       hashptr^.hidden^.nlink := ident;
	    END;
	    (* insert `ident' into `badlevel' scope *)
	    dlink := display[badlevel];
	    display[badlevel] := ident;
	 END;
      END;
   END InsertBad;

   PROCEDURE Enter(sname: Identifier; idclass: IdentClass;
		   VAR ident: Ident) : BOOLEAN;
      (* enter `name' into the current scope; *)
      (* allocate `ident'; *)
      (* initialize hidden part of `ident'; *)
      (* initialize class, name, and plevel of `ident' *)
      (* print error messages on multiple occurences *)
      (* return TRUE if successful *)
   BEGIN
      IF ~CheckId(sname) THEN
	 RETURN FALSE
      END;
      NEW(ident);
      WITH ident^ DO
	 class := idclass;
	 name := sname;
	 error := class = badclass;
      END;
      Insert(ident);
      RETURN TRUE
   END Enter;

   PROCEDURE Include(ident: Ident);
      (* enter `ident' (already initialized) into current scope; *)
      (* aborts if not successful *)
   BEGIN
      Assert(CheckId(ident^.name));
      Insert(ident);
   END Include;

   PROCEDURE Find(level: CARDINAL;
		  name: Identifier; VAR ident: Ident) : BOOLEAN;
      (* starting from level `level' *)
      (* search for `name' starting in the current scope *)
      (* return TRUE if `name' has been found *)
      VAR
	 index: CARDINAL;
	 ptr: Ident;
	 found: Ident;
   BEGIN
      index := HashVal(name);
      ptr := bucket[index];
      ident := NIL;
      WHILE (ptr # NIL) & (ident = NIL) DO
	 IF (ptr^.name = name) & (ptr^.hidden^.dlevel <= level) THEN
	    ident := ptr;
	 END;
	 ptr := ptr^.hidden^.nlink;
      END;
      RETURN ident # NIL
   END Find;

   PROCEDURE Search(name: Identifier; VAR ident: Ident) : BOOLEAN;
      (* search for `name' starting in the current scope *)
      (* return TRUE if `name' has been found *)
   BEGIN
      IF Find(level, name, ident) THEN
	 ident^.hidden^.lastuse := level;
	 RETURN TRUE
      END;
      RETURN FALSE
   END Search;

   PROCEDURE SearchAndEnter(name: Identifier; VAR ident: Ident);
      (* print error message if `name' is unknown; *)
      (* to avoid multiple error messages enter unknown names *)
   BEGIN
      IF ~Search(name, ident) THEN
	 InsertBad(name, ident);
	 Error1("%I not declared", name);
      END;
   END SearchAndEnter;

   PROCEDURE SearchList(id: Identifier; idlist: IdentList;
			VAR ip: Ident) : BOOLEAN;
      (* search for `id' in `idlist' *)
   BEGIN
      WHILE idlist # NIL DO
	 WITH idlist^ DO
	    IF id = ident^.name THEN
	       ip := ident;
	       RETURN TRUE
	    END;
	 END;
	 idlist := idlist^.link;
      END;
      RETURN FALSE
   END SearchList;

   PROCEDURE SearchFieldList(id: Identifier; fields: FieldList;
			     VAR fp: FieldList) : BOOLEAN;
      (* search for `id' in `fields' *)
   BEGIN
      WHILE fields # NIL DO
	 IF id = fields^.id THEN
	    fp := fields;
	    RETURN TRUE
	 END;
	 fields := fields^.link;
      END;
      fp := NIL;
      RETURN FALSE
   END SearchFieldList;

   PROCEDURE SearchParamList(id: Identifier; params: ParamList;
			     VAR pp: ParamList) : BOOLEAN;
      (* search of `id' in `params' *)
   BEGIN
      WHILE params # NIL DO
	 IF id = params^.id THEN
	    pp := params;
	    RETURN TRUE
	 END;
	 params := params^.link;
      END;
      pp := NIL;
      RETURN FALSE
   END SearchParamList;

   PROCEDURE Init;

      PROCEDURE InitBucketTable;
	 VAR
	    index: CARDINAL;
      BEGIN
	 FOR index := 0 TO HIGH(bucket) DO
	    bucket[index] := NIL;
	 END;
      END InitBucketTable;

   BEGIN
      level := 0;
      leveloverflow := 0;
      InitBucketTable;
      display[level] := NIL;
   END Init;

   PROCEDURE HashVal(name: Identifier) : CARDINAL;
      (* implementation dependant: *)
   BEGIN
      RETURN CARDINAL(name) MOD hashtabsize
   END HashVal;

   PROCEDURE Level() : CARDINAL;
      (* returns the current level *)
   BEGIN
      RETURN level
   END Level;

   (* (* exported by definition module *)
   VAR
      extmods: IdentList;	(* managed by `GenSymFile.ExtMod' and used by *)
				(* `GenSymFile' *)
				(* includes "our" module *)
      mainmod: Ident;           (* "our" module; initialized by `GenSymFile' *)
      globalvars: IdentList;	(* list of global variables; *)
				(* managed by SymDef *)
   *)

BEGIN
   Init; (* initialization of internal variables *)
   extmods := NIL; mainmod := NIL; globalvars := NIL;
END SymTab.
