(* Ulm's Oberon Compiler
   Copyright (c) 1989 by University of Ulm, SAI, D-W-7900 Ulm, Germany
   ----------------------------------------------------------------------------
   $Id: CodeGen.m2,v 0.15 1994/05/25 11:48:34 borchert Exp $
   ----------------------------------------------------------------------------
   $Log: CodeGen.m2,v $
   Revision 0.15  1994/05/25  11:48:34  borchert
   SYSTEM.LSH & SYSTEM.ROT accept now SET as argument type

   Revision 0.14  1993/12/17  16:12:34  borchert
   bug fix: passing a module name as parameter caused compiler crash

   Revision 0.13  1993/10/02  15:12:43  borchert
   bug fix: string := 4X; was accepted by CodeGen but caused wrong code
            to be generated by GenStmts
   despite the fact that the Oberon report does not allow assignments
   of character constants which are not strings to character arrays,
   an ETHZ compatibility flag was added which supports this feature

   Revision 0.12  1993/09/27  12:53:04  borchert
   compatibility rule for SYSTEM.ADDRESS, SYSTEM.UNTRACEDADDRESS
      and LONGINT added
   SYSTEM.ADR returns now SYSTEM.ADDRESS

   Revision 0.11  1993/09/25  11:32:11  borchert
   bug fix: ptr[index] was not allowed as shortform for ptr^[index]

   Revision 0.10  1993/09/13  16:16:57  borchert
   bug fix: two pointer types are compatible if they are derived from
            the same record type because they extend each other --
            despite to this fact they were treated as incompatible types
   bug fix: AssignComp didn't differentiate between tagged and untagged
            pointers

   Revision 0.9  1993/07/28  10:59:09  borchert
   bug fix: tagged pointer check for NEW wasn't aware that types[1] may
            be NIL

   Revision 0.8  1993/07/06  08:02:38  borchert
   bug fix: ``bool_expr & ptr IS PtrType'' caused compiler crash

   Revision 0.7  1993/06/30  13:13:46  borchert
   bug fix: ``NIL IS PtrType'' caused compiler crash

   Revision 0.6  1993/06/25  12:34:12  borchert
   compiler option $O implemented: pointer may now be untagged

   Revision 0.5  1993/06/18  15:32:38  borchert
   CodeGen depends now on Types

   Revision 0.4  1993/06/16  09:48:10  borchert
   SYSTEM.INT16 added

   Revision 0.3  1993/05/06  16:14:51  borchert
   NIL is now compatible to procedure types
   bug fix: there was no check for assignments of local procedures

   Revision 0.2  1993/02/03  12:30:45  borchert
   new: GenPreamble called by ModuleEnd
   constant folding for SYSTEM.VAL added

   Revision 0.1  1992/07/30  10:47:56  borchert
   Initial revision

   ----------------------------------------------------------------------------
*)

IMPLEMENTATION MODULE CodeGen; (* AFB 1/89 *)
                               (* Oberon revision: AFB 11/89 *)

   (* code generation interface to the parser *)
   (*    procedures are to be called during the parse of statements *)

   FROM Attributes IMPORT IdModes, DesignatorModes, OpModes, AtMode,
      AtModeSet, CaseLabel, LabelList, DisposeAttribute,
      procedureAt, badprocedure;
   FROM ConstExpr IMPORT Unary, Binary;
   FROM EmitCode IMPORT InitEmitCode;
   FROM Exception IMPORT Assert;
   FROM GenBlocks IMPORT GenModuleHeader, GenProcEntry, GenProcExit,
      GenPreamble, GenModuleEntry, GenModuleExit;
   FROM GenExpr IMPORT GenCall;
   FROM GenStmts IMPORT GenAssign, GenIfThen, GenElsifThen, GenElse,
      GenEndIf, GenWhileDo, GenEndWhile, GenRepeat, GenUntil,
      GenCaseOf, GenCase, GenCaseElse, GenEndCase,
      GenReturn, GenReturnExpr, GenWith;
   FROM Lex IMPORT String, Constval, Symbol, StringLen, GetStringChar,
      previdpos, AddLegalOptions, CompilerOptions, options;
   FROM IdentSys IMPORT Identifier;
   FROM Machine IMPORT Align, minshort, maxshort, minint, maxint, onebyte,
      minlong, maxlong, bitsperword, minreal, maxreal, minlreal, maxlreal,
      oneword, nilvalue, maxint16, minint16;
   FROM Memory IMPORT ALLOCATE, DEALLOCATE;
   FROM Scan IMPORT Error, Error1, Error2, ErrorList, AddToList, EndOfList,
      Message, warning, errorflag, At;
   FROM ScanArgs IMPORT defunit;
   FROM Standard IMPORT realptr, boolptr, niltype, charptr, longrealptr,
      setptr, shortptr, longptr, intptr, int16ptr, byteptr, sigprocptr, crptr,
      addrptr;
   FROM SymFile IMPORT InitType, LinkType;
   FROM SymTab IMPORT Type, Ident, IdentClass, Search, Enter, FieldList,
      IdentList, SearchList, SearchFieldList, Form, FormSet, Numeric,
      ParamList, StdProc, SearchAndEnter, VarKind, Basic, global,
      address;
   FROM Types IMPORT Integer, Real;
   IMPORT Attributes, Exception, GenStmts, Scan, Storage, Types;

   CONST
      charConstantsAreETHZCompatibleOpt = "S";
	 (* if switched on, not only strings of length 1 are
	    treated as possibly strings but also all other
	    character constants, e.g. 4X, MAX(CHAR), CAP("A"), CHR(4);
	    while the report doesn't encourage this interpretation,
	    the ETHZ compilers treat all these character constants
	    as possible strings of the length 1
	 *)

   TYPE
      SymSet = SET OF Symbol;
      Attribute = Attributes.Attribute;
      Labels = Attributes.Labels;

   (* ---------------------------------------------------------	*)
   (* initialize attributes					*)
   (* ---------------------------------------------------------	*)

   PROCEDURE SearchListAndEnter(id: Identifier;
				modip: Ident;
				VAR ip: Ident);
      VAR
	 il: IdentList;

   BEGIN
      IF ~SearchList(id, modip^.export, ip) THEN
	 Error2("%I not exported from module %I", id, modip^.name);
	 NEW(ip);
	 WITH ip^ DO
	    name := id;
	    error := TRUE;
	    class := badclass;
	 END;
	 NEW(il);
	 WITH il^ DO
	    ident := ip;
	    link := modip^.export;
	 END;
	 modip^.export := il;
      END;
   END SearchListAndEnter;

   PROCEDURE CondError(msg: ARRAY OF CHAR; ip: Ident);
      (* print error message which is related to `ip' only *)
      (* if ip^.error is FALSE *)
   BEGIN
      Assert(ip # NIL);
      WITH ip^ DO
	 IF NOT error THEN
	    error := TRUE;
	    Error1(msg, name);
	 END;
      END;
   END CondError;

   PROCEDURE InitIpAt(ip: Ident; VAR at: Attribute);
      (* get attribute for any identifer: *)
      (* modules, constants, types, variables, and procedures *)
   BEGIN
      IF ip^.class = badclass THEN
	 at := NIL;
      ELSE
	 NEW(at);
	 WITH at^ DO
	    atip := ip;
	    IF ip^.class = moduleC THEN
	       mode := moduleAt;
	    ELSE
	       attype := ip^.type;
	       IF ip^.class = constC THEN
	          mode := constAt; cval := ip^.constval;
	       ELSE
		  CASE ip^.class OF
		  | typeC:      mode := typeAt;
		  | varC:       mode := varAt;
		  | procedureC: mode := procAt;
		  END;
	       END;
	    END;
	 END;
      END;
   END InitIpAt;

   PROCEDURE InitConstAt(constval: Constval; VAR at: Attribute);

      PROCEDURE InRange(i: Integer; min, max: Integer) : BOOLEAN;
      BEGIN
	 RETURN (i >= min) & (i <= max)
      END InRange;

      PROCEDURE StringType(len: CARDINAL; VAR type: Type);
      BEGIN
	 IF len = 2 THEN (* len = StringLen() + 1 *)
	    type := charptr;
	 ELSE
	    InitType(type);
	    WITH type^ DO
	       ident := NIL;
	       size := len * onebyte; Align(size);
	       form := array;
	       dyn := FALSE;
	       length := len;
	       element := charptr;
	       LinkType(charptr);
	    END;
	 END;
      END StringType;

   BEGIN
      NEW(at);
      WITH at^ DO
	 mode := constAt;
	 atip := NIL;
	 cval := constval;
	 WITH cval DO
	    CASE sy OF
	    | stringcon:   StringType(StringLen(string)+1, attype);
	    | charcon:     attype := charptr;
	    | intcon:      IF InRange(intval, minshort, maxshort) THEN
			      attype := shortptr;
			   ELSIF InRange(intval, minint16, maxint16) THEN
			      attype := int16ptr;
			   ELSIF InRange(intval, minint, maxint) THEN
			      attype := intptr;
			   ELSE
			      attype := longptr;
			   END;
	    | realcon:     attype := realptr;
	    | longrealcon: attype := longrealptr;
	    | setcon:      attype := setptr;
	    | boolcon:     attype := boolptr;
	    | nilSY:       attype := niltype;
	    END;
	 END;
      END;
   END InitConstAt;

   PROCEDURE InitIdentAt(id: Identifier; VAR at: Attribute);
      (* get attribute for any identifer: *)
      (* modules, constants, types, variables, and procedures *)
      VAR
	 ip: Ident;
   BEGIN
      SearchAndEnter(id, ip);
      InitIpAt(ip, at);
   END InitIdentAt;

   PROCEDURE InitNILAt(VAR at: Attribute);
      (* get attribute for NIL *)
   BEGIN
      NEW(at);
      WITH at^ DO
	 mode := constAt;
	 atip := NIL;
	 attype := niltype;
	 cval.sy := nilSY;
      END;
   END InitNILAt;

   PROCEDURE InitEmptyAt(VAR at: Attribute);
      (* usable for errorneous attributes or empty element list *)
   BEGIN
      at := NIL;
   END InitEmptyAt;

   PROCEDURE RestOfString(s: String);
      VAR
	 ch: CHAR;
   BEGIN
      REPEAT
	 GetStringChar(s, ch);
      UNTIL ch = 0C;
   END RestOfString;


   (* ---------------------------------------------------------	*)
   (* designators						*)
   (* ---------------------------------------------------------	*)

   PROCEDURE Qualified(ident: Identifier; VAR at: Attribute);
      (* at.ident *)
      VAR
	 ip: Ident;
   BEGIN
      IF at # NIL THEN
	 WITH at^ DO
	    IF mode # moduleAt THEN
	       Assert(mode IN IdModes);
	       At(previdpos);
	       CondError("%I must be a module name", atip);
	       DisposeAttribute(at);
	    ELSE
	       SearchListAndEnter(ident, atip, ip);
	       IF ip^.class = badclass THEN
		  DisposeAttribute(at);
	       ELSE
		  InitIpAt(ip, at);
	       END;
	    END;
	 END;
      END;
   END Qualified;

   PROCEDURE Index(index: Attribute; VAR at: Attribute);
      (* at[index] *)
      (* `at' remains not-NIL only if everything is OK *)
      (*                           exceptions: element type may be NIL *)
      (*                                       constant index out of range *)
      VAR
	 indextype: Type; (* type of `index' *)
	 newat: Attribute;
   BEGIN
      newat := NIL;
      IF (index # NIL) & ~(index^.mode IN OpModes) THEN
	 Error("index must be an expression");
	 DisposeAttribute(index);
      END;
      IF (index # NIL) & (index^.attype # NIL) &
	 CheckForm(index^.attype^.form, FormSet{shortint..longint}) THEN
	 indextype := index^.attype;
	 IF (index^.mode = constAt) & (index^.cval.intval < 0) THEN
	    Error("index must not be negative");
	 END;
      ELSE
	 indextype := NIL;
      END;
      IF at # NIL THEN
	 IF at^.mode IN DesignatorModes THEN
	    IF at^.attype # NIL THEN
	       IF (at^.attype^.form = pointer) &
		     (at^.attype^.reftype # NIL) &
		     (at^.attype^.reftype^.form = array) THEN
		  Dereference(at);
	       END;
	       WITH at^ DO
		  WITH attype^ DO
		     IF form = array THEN
			(* check index *)
			IF (indextype # NIL) & (index^.mode = constAt) THEN
			   IF ~dyn & (index^.cval.intval >= length) THEN
			      Error("index out of range");
			   END;
			END;
			NEW(newat);
			WITH newat^ DO
			   link := NIL;
			   mode := indexAt;
			   attype := at^.attype^.element; (* may be NIL *)
			   desat := at;
			   indexat := index;
			END;
		     ELSE
			IF mode IN IdModes THEN
			   CondError("%I must be an array", atip);
			ELSE
			   Error("array expected");
			END;
		     END;
		  END;
	       END;
	    END;
	 ELSE
	    IF at^.mode IN IdModes THEN
	       CondError("%I must be a designator", at^.atip);
	    ELSE
	       Error("must be a designator");
	    END;
	 END;
      END;
      IF newat = NIL THEN
	 DisposeAttribute(at);
      ELSE
	 at := newat;
      END;
   END Index;

   PROCEDURE FieldSelection(field: Identifier; VAR at: Attribute);
      (* at.field *)
      VAR
	 newat: Attribute;
	 rectype: Type;
	 ref: BOOLEAN;
	 fp: FieldList;

      PROCEDURE RecordType(attype: Type;
			   VAR rectype: Type; VAR ref: BOOLEAN) : BOOLEAN;
      BEGIN
	 rectype := NIL; ref := FALSE;
	 IF attype # NIL THEN
	    WITH attype^ DO
	       IF form = pointer THEN
		  IF reftype # NIL THEN
		     IF reftype^.form = record THEN
			rectype := reftype; ref := TRUE;
		     ELSE
			IF at^.mode IN IdModes THEN
			   CondError("%I must be a designator", at^.atip);
			ELSE
			   Error("must be a designator");
			END;
		     END;
		  END;
	       ELSIF form # record THEN
		  IF at^.mode IN IdModes THEN
		     CondError("%I must be a record or pointer", at^.atip);
		  ELSE
		     Error("record or pointer expected");
		  END;
	       ELSE (* form = record *)
		  rectype := attype;
	       END;
	    END; (* WITH attype^ *)
	 END;
	 RETURN rectype # NIL
      END RecordType;

      PROCEDURE SearchFieldListAndEnter(name: Identifier; VAR fields: FieldList;
					VAR fp: FieldList);
      BEGIN
	 IF ~SearchFieldList(name, fields, fp) THEN
	    Error1("unknown field name: %I", name);
	    NEW(fp);
	    WITH fp^ DO
	       id := name;
	       type := NIL;
	       offset := 0;
	    END;
	    IF fields = NIL THEN
	       (* empty record:
		  initialize the field-list with `fp'
		  BUT: `fp' is not known by the extensions
	       *)
	       fp^.link := fields;
	       fields := fp;
	    ELSE
	       (* make `name' visible for records which are an
		  extension of this record:
		  insert `fp' after `fields'
	       *)
	       fp^.link := fields^.link;
	       fields^.link := fp;
	    END;
	 END;
      END SearchFieldListAndEnter;

   BEGIN
      IF at # NIL THEN
	 newat := NIL;
	 IF at^.mode IN DesignatorModes THEN
	    IF RecordType(at^.attype, rectype, ref) THEN
	       IF ref THEN
		  Dereference(at);
	       END;
	       SearchFieldListAndEnter(field, rectype^.fields, fp);
	       NEW(newat);
	       WITH newat^ DO
		  link := NIL;
		  mode := selectAt;
		  attype := fp^.type;
		  desat := at;
		  field := fp;
	       END;
	    ELSE (* error message printed *)
	    END;
	 ELSE
	    WITH at^ DO
	       IF mode IN IdModes THEN
		  CondError("%I must be a designator", atip);
	       ELSE
		  Error("must be a designator");
	       END;
	    END;
	 END;
	 IF newat = NIL THEN
	    DisposeAttribute(at);
	 ELSE
	    at := newat;
	 END;
      END;
   END FieldSelection;

   PROCEDURE Dereference(VAR at: Attribute);
      (* at^ *)
      VAR
	 newat: Attribute;
   BEGIN
      IF at # NIL THEN
	 newat := NIL;
	 WITH at^ DO
	    IF mode IN DesignatorModes THEN
	       IF attype # NIL THEN
		  WITH attype^ DO
		     IF form = pointer THEN
			NEW(newat);
			WITH newat^ DO
			   link := NIL;
			   mode := refAt;
			   attype := reftype; (* may be NIL *)
			   desat := at;
			END;
		     ELSE
			IF mode IN IdModes THEN
			   CondError("%I must be a pointer", atip);
			ELSE
			   Error("pointer expected");
			END;
		     END;
		  END;
	       END;
	    ELSIF mode IN IdModes THEN
	       CondError("%I must be a designator", atip);
	    ELSE
	       Error("designator expected");
	    END;
	 END;
	 IF newat = NIL THEN
	    DisposeAttribute(at);
	 ELSE
	    at := newat;
	 END;
      END;
   END Dereference;

   PROCEDURE Typeguard(type: Attribute; VAR at: Attribute);
      (* at(type) *)
      VAR
	 newat: Attribute;
	 base: Type;
	 guard: Type;
   BEGIN
      guard := NIL;
      IF type # NIL THEN
	 WITH type^ DO
	    IF mode = typeAt THEN
	       guard := attype;
	    ELSE
	       IF mode IN IdModes THEN
		  Error1("%I must be a type identifier", atip^.name);
	       ELSE
		  Error("type identifier expected");
	       END;
	    END;
	 END;
      END;
      IF at # NIL THEN
	 newat := NIL;
	 WITH at^ DO
	    IF mode IN DesignatorModes THEN
	       IF attype # NIL THEN
		  WITH attype^ DO
		     IF (form = record) OR (form = pointer) THEN
			IF guard # NIL THEN
			   IF ~BaseType(guard, attype, base) THEN
			      IF Compatible(guard, attype) THEN
				 IF mode IN IdModes THEN
				    Error1("type guard not applicable to %I",
				       atip^.name);
				 ELSE
				    Error("type guard not applicable");
				 END;
			      ELSE
				 Assert(guard^.ident # NIL);
				 IF mode IN IdModes THEN
				    Error2("%I cannot be of type %I",
				       atip^.name, guard^.ident^.name);
				 ELSE
				    Error1("cannot be of type %I",
				       guard^.ident^.name);
				 END;
			      END;
			   ELSIF (form = record) &
				    ((mode # varAt) OR
				       (atip^.varkind # varparamV)) THEN
			      IF mode IN IdModes THEN
				 Error1("%I must be a VAR-parameter",
				    atip^.name);
			      ELSE
				 Error("VAR-parameter expected");
			      END;
			   ELSE
			      IF base = guard THEN
				 (* satisfied by definition *)
				 Message(warning, "type guard satisfied by definition");
				 newat := at;
			      ELSE (* base = attype *)
				 NEW(newat);
				 WITH newat^ DO
				    link := NIL;
				    mode := guardAt;
				    attype := guard;
				    desat := at;
				 END;
			      END;
			   END;
			END;
		     ELSE
			IF mode IN IdModes THEN
			   CondError("%I must be record or pointer", atip);
			ELSE
			   Error("record or pointer expected");
			END;
		     END;
		  END;
	       END;
	    ELSE
	       IF mode IN IdModes THEN
		  CondError("%I must be a designator", atip);
	       ELSE
		  Error("designator expected");
	       END;
	    END;
	 END;
	 IF newat = NIL THEN
	    DisposeAttribute(at);
	 ELSE
	    at := newat;
	 END;
      END;
   END Typeguard;

   (* ---------------------------------------------------------	*)
   (* resolution of following grammar conflict:			*)
   (*    	ident "(" ident ")"				*)
   (*		   can be a procedure call			*)
   (* 		   or a type guard				*)
   (* ---------------------------------------------------------	*)

   PROCEDURE IsVarNoProcvar(at: Attribute) : BOOLEAN;
      (* returns TRUE if 'qualid' is a variable in the current scope *)
      (*                 and no procedure variable		 *)
      (* to be called on designator "(" to decide whether a type guard *)
      (* or a procedure call is coming now;			 *)
      (* This test does not print any error messages.		 *)
   BEGIN
      IF at = NIL THEN
	 RETURN FALSE
      END;
      WITH at^ DO
	 RETURN (mode IN DesignatorModes) &
		((attype = NIL) OR (attype^.form # proceduretype))
      END;
   END IsVarNoProcvar;

   (* ---------------------------------------------------------	*)
   (* construction of expressions				*)
   (* ---------------------------------------------------------	*)

   PROCEDURE ExpectedForm(fset: FormSet);
      (* print error message: set of expected types *)
      VAR
	 form: Form;
	 tp: Type;
	 first: BOOLEAN;
	 s: ARRAY [0..15] OF CHAR;
   BEGIN
      ErrorList("bad type of operand; possible operand types are: ");
      first := TRUE;
      FOR form := MIN(Form) TO MAX(Form) DO
	 IF form IN fset THEN
	    tp := NIL;
	    CASE form OF
	    | shortint:       tp := shortptr;
	    | int16:          tp := int16ptr;
	    | integer:        tp := intptr;
	    | longint:        tp := longptr;
	    | real:           tp := realptr;
	    | longreal:       tp := longrealptr;
	    | boolean:        tp := boolptr;
	    | char:           tp := charptr;
	    | set:            tp := setptr;
	    | byte:           tp := byteptr;
	    | array:          s := "arrays";
	    | record:         s := "records";
	    | pointer:        s := "pointers";
	    | proceduretype:  s := "proceduretypes";
	    END;
	    EXCL(fset, form);
	    IF first THEN
	       IF tp # NIL THEN
		  AddToList("%I", tp^.ident^.name);
	       ELSE
		  AddToList("%s", s);
	       END;
	       first := FALSE;
	    ELSIF fset = FormSet{} THEN
	       IF tp # NIL THEN
		  AddToList(" and %I", tp^.ident^.name);
	       ELSE
		  AddToList(" and %s", s);
	       END;
	    ELSE
	       IF tp # NIL THEN
		  AddToList(", %I", tp^.ident^.name);
	       ELSE
		  AddToList(", %s", s);
	       END;
	    END;
	 END;
      END;
      EndOfList;
   END ExpectedForm;

   PROCEDURE CheckForm(form: Form; fset: FormSet) : BOOLEAN;
   BEGIN
      IF form IN fset THEN
	 RETURN TRUE
      ELSE
	 ExpectedForm(fset);
	 RETURN FALSE
      END;
   END CheckForm;

   PROCEDURE CheckType(t: Type; set: FormSet) : BOOLEAN;
   BEGIN
      RETURN (t # NIL) & (t^.form IN set)
   END CheckType;

   PROCEDURE IsString(t: Type) : BOOLEAN;
      (* not TRUE for charptr *)
   BEGIN
      RETURN (t # NIL) &
	     (t^.form = array) & (t^.element = charptr)
   END IsString;

   PROCEDURE IsStringAt(at: Attribute) : BOOLEAN;
      (* TRUE for string constants or if IsString(attype) *)
   BEGIN
      IF at = NIL THEN RETURN FALSE END;
      WITH at^ DO
	 RETURN (mode = constAt) & (cval.sy = stringcon) OR
		IsString(attype) OR
		(charConstantsAreETHZCompatibleOpt IN options) &
		  (cval.sy = charcon)
      END;
   END IsStringAt;

   PROCEDURE CheckAtType(at: Attribute; set: FormSet) : BOOLEAN;
   BEGIN
      IF (at = NIL) OR ~(at^.mode IN OpModes) THEN RETURN FALSE END;
      RETURN CheckType(at^.attype, set)
   END CheckAtType;

   PROCEDURE CheckOpMode(at: Attribute) : BOOLEAN;
   BEGIN
      WITH at^ DO
	 IF mode IN OpModes THEN
	    RETURN TRUE
	 ELSIF (mode IN IdModes) & (atip # NIL) THEN
	    Error1("%I misused in this context", atip^.name);
	 ELSE
	    Error("identifier misused in this context");
	 END;
	 RETURN FALSE
      END;
   END CheckOpMode;

   PROCEDURE ConstAt(t: Type; sy: Symbol; val: Integer; VAR at: Attribute);
   BEGIN
      WITH at^ DO
	 link := NIL;
	 mode := constAt;
	 attype := t;
	 atip := NIL;
	 cval.sy := sy;
	 IF sy = boolcon THEN
	    cval.boolval := VAL(BOOLEAN, val);
	 ELSIF sy = setcon THEN
	    cval.setval := BITSET(val);
	 ELSE
	    cval.intval := val;
	 END;
      END;
   END ConstAt;

   PROCEDURE BaseType(t1, t2: Type; VAR base: Type) : BOOLEAN;
      (* returns TRUE
	 if both types have the same form (record or pointer)
	 and one is the basetype of the other type
	 the base type is returned in `base'.

	 There may be a case where t1 is a basetype of t2 and
	 vice versa even if t1 # t2:

	 TYPE
	    Rec = RECORD END;
	    Ptr1 = POINTER TO Rec;
	    Ptr2 = POINTER TO Rec;

	 Ptr1 and Ptr2 are still assignment compatible due to
	 the rule that Ptr1 is an extension of Ptr2 and vice versa.
	 In this case, base is set to `t1'
      *)
      VAR
	 t: Type;
	 rt1, rt2: Type; (* record types *)
   BEGIN
      base := NIL;
      IF t1^.form # t2^.form THEN
	 RETURN FALSE
      END;
      IF t1^.form = pointer THEN
	 rt1 := t1^.reftype;
	 rt2 := t2^.reftype;
         IF (rt1 = NIL) OR (rt2 = NIL) THEN
            RETURN TRUE (* avoid additional error messages *)
         END;
      ELSE
	 rt1 := t1; rt2 := t2;
      END;
      IF (rt1^.form # record) OR (rt2^.form # record) THEN
	 RETURN FALSE
      END;

      (* check for `rt1' being a base type of `rt2' *)
      t := rt2;
      WHILE (t # NIL) & (t # rt1) DO
	 t := t^.basetype;
      END;
      IF t = rt1 THEN
	 base := t1; RETURN TRUE
      END;

      (* check for `rt2' being a base type of `rt1' *)
      t := rt1;
      WHILE (t # NIL) & (t # rt2) DO
	 t := t^.basetype;
      END;
      IF t = rt2 THEN
	 base := t2; RETURN TRUE
      END;

      RETURN FALSE
   END BaseType;

   PROCEDURE ProcedureComp(t1, t2: Type) : BOOLEAN;
      VAR
	 pl1, pl2: ParamList;

      PROCEDURE DynComp(ft, actt: Type) : BOOLEAN;
      BEGIN
	 WHILE (ft^.form = array) & (actt^.form = array) & ft^.dyn DO
	    ft := ft^.element;
	    actt := actt^.element;
	 END;
	 RETURN ft = actt
      END DynComp;

   BEGIN
      IF (t1^.form = t2^.form) & (t1^.form = proceduretype) &
	 (t1^.function = t2^.function) THEN
	 IF t1^.function &
	    (t1^.restype # t2^.restype) &
	    (t1^.restype # NIL) & (t2^.restype # NIL) THEN
	    RETURN FALSE
	 END;
	 pl1 := t1^.param; pl2 := t2^.param;
	 WHILE (pl1 # NIL) & (pl2 # NIL) DO
	    IF (pl1^.varkind # pl2^.varkind) OR
	       (pl1^.type # pl2^.type) &
	       (pl1^.type # NIL) & (pl2^.type # NIL) &
	       ~DynComp(pl1^.type, pl2^.type) THEN
	       RETURN FALSE
	    END;
	    pl1 := pl1^.link;
	    pl2 := pl2^.link;
	 END;
	 RETURN pl1 = pl2
      ELSIF (t1 = niltype) & (t2^.form = proceduretype) OR
	    (t2 = niltype) & (t1^.form = proceduretype) THEN
	    (* NIL is compatible and assignment compatible to procedure types
	       (Oberon revision 1991)
	    *)
	 RETURN TRUE
      ELSE
	 RETURN FALSE
      END;
   END ProcedureComp;

   PROCEDURE ProcedureDiff(t1, t2: Type);
      (* print differences about procedures (as error messages) *)
      VAR
	 pl1, pl2: ParamList;

      PROCEDURE DynComp(ft, actt: Type) : BOOLEAN;
      BEGIN
	 WHILE (ft^.form = array) & (actt^.form = array) & ft^.dyn DO
	    ft := ft^.element;
	    actt := actt^.element;
	 END;
	 RETURN ft = actt
      END DynComp;

   BEGIN
      IF (t1^.form = t2^.form) & (t1^.form = proceduretype) &
	 (t1^.function = t2^.function) THEN
	 IF t1^.function & (t1^.restype # t2^.restype) &
	    (t1^.restype # NIL) & (t2^.restype # NIL) THEN
	    IF (t1^.restype^.ident # NIL) & (t1^.restype^.ident # NIL) THEN
	       Error2("result types are not identical: %I and %I",
		  t1^.restype^.ident^.name, t2^.restype^.ident^.name);
	    ELSE
	       Error("result types are not identical");
	    END;
	 END;
	 pl1 := t1^.param; pl2 := t2^.param;
	 WHILE (pl1 # NIL) & (pl2 # NIL) DO
	    IF pl1^.varkind # pl2^.varkind THEN
	       Error("different kinds of parameter: VAR versus value");
	    ELSIF (pl1^.type # pl2^.type) &
		  (pl1^.type # NIL) & (pl2^.type # NIL) &
		  ~DynComp(pl1^.type, pl2^.type) THEN
	       IF (pl1^.type^.ident # NIL) & (pl2^.type^.ident # NIL) THEN
		  Error2("parameter types are not identical: %I and %I",
		     pl1^.type^.ident^.name, pl2^.type^.ident^.name);
	       ELSE
		  Error("parameter types are not identical");
	       END;
	    END;
	    pl1 := pl1^.link;
	    pl2 := pl2^.link;
	 END;
	 IF pl1 # pl2 THEN
	    Error("different number of parameters");
	 END;
      END;
   END ProcedureDiff;

   PROCEDURE Compatible(t1, t2: Type) : BOOLEAN;
      VAR base: Type;
   BEGIN
      RETURN (t1 = t2) OR (t1 = NIL) OR (t2 = NIL) OR
	     (t1^.form IN Numeric) & (t2^.form IN Numeric) OR
	     (t1^.form = t2^.form) & (t1^.form = pointer) &
	     ((t1 = niltype) OR (t2 = niltype)) OR
	     BaseType(t1, t2, base) OR ProcedureComp(t1, t2) OR
	     IsString(t1) & IsString(t2)
   END Compatible;

   PROCEDURE AssignComp(left, right: Type; const: BOOLEAN) : BOOLEAN;
      (* test compatibility for left := right *)
      (* `const' is true if `right' is a constant *)
      VAR
	 base: Type;
   BEGIN
      IF (left # NIL) & left^.privateparts THEN
	 (*
	 Error("values of types designated as public parts cannot be assigned");
	 *)
	 Error("unknown private parts must not be assigned to");
	 RETURN FALSE
      ELSIF (left # NIL) & (left^.form = array) & left^.dyn THEN
	 Error("lhs of an assignment must not be a dynamic array");
	 RETURN FALSE
      ELSIF (left = right) OR (left = NIL) OR (right = NIL) OR
	 (left = byteptr) & ((right = charptr) OR (right = shortptr)) THEN
	 RETURN TRUE
      END;
      IF (left^.form IN Numeric) & (right^.form IN Numeric) THEN
	 (* the type of the expression must be included by the type
	    of the variable
	 *)
	 IF (left^.form = right^.form) & (left^.form = address) &
	       ~left^.treatAsAddress & right^.treatAsAddress THEN
	    Message(warning, "addresses should be assigned to SYSTEM.ADDRESS but not to LONGINT");
	 END;
	 RETURN left^.form >= right^.form
      ELSIF (left^.form = pointer) & (right = niltype) THEN
         (* NIL may be assigned to any pointer type *)
         RETURN TRUE
      ELSIF BaseType(left, right, base) THEN
	 (* return TRUE if the expression extends the type of the variable *)
	 IF base = left THEN
	    IF (left^.form = pointer) &
		  (left^.taggedptr # right^.taggedptr) THEN
	       Error("pointer types are incompatible due to different tag options");
	       RETURN FALSE
	    END;
	    RETURN TRUE
	 ELSE
	    RETURN FALSE
	 END;
      ELSIF const & (left^.form = array) & (right^.form = array) THEN
	 (* strings can be assigned to any variable whose type is an array
	    of characters, provided the length of the string is less
	    than that of the array.
	    right^.length includes the null-byte here
	 *)
	 RETURN ~left^.dyn &
		(left^.element = charptr) & (right^.element = charptr) &
		(left^.length >= right^.length)
      ELSIF const & (right = charptr) THEN
	 (* assignment of a string constant of length 1, e.g. "x" *)
	 RETURN ~left^.dyn &
		(left^.form = array) & (left^.element = charptr) &
		(left^.length > 1)
      ELSE
	 RETURN ProcedureComp(left, right)
      END;
   END AssignComp;

   PROCEDURE TypeIdentity(t1, t2: Type) : BOOLEAN;
   BEGIN
      RETURN (t1 = t2) OR (t1 = NIL) OR (t2 = NIL)
   END TypeIdentity;

   PROCEDURE UnaryOp(op: Symbol; VAR at: Attribute);
      (* op may be: plus, minus, tilde *)
      VAR
	 newat: Attribute;
	 t: Type;
   BEGIN
      IF at # NIL THEN
	 newat := NIL;
	 IF CheckOpMode(at) THEN
	    t := at^.attype;
	    IF (t # NIL) &
	       ((op = minus) & CheckForm(t^.form, Numeric + FormSet{set}) OR
		(op = plus)  & CheckForm(t^.form, Numeric) OR
		(op = tilde) & CheckForm(t^.form, FormSet{boolean})) THEN
	       IF at^.mode = constAt THEN
		  Unary(op, at^.cval);
		  newat := at;
	       ELSE
		  NEW(newat);
		  WITH newat^ DO
		     mode := unaryAt;
		     opsy := op;
		     attype := t;
		     rightop := at;
		  END;
	       END;
	    ELSE
	       DisposeAttribute(at);
	    END;
	 END;
	 at := newat;
      END;
   END UnaryOp;

   PROCEDURE BinaryOp(op: Symbol; left, right: Attribute; VAR at: Attribute);
      (* op may be: inSY, isSY, divSY, modSY, orSY *)
      (*	    ampersand, *)
      (*            plus, minus, times, slash, *)
      (*            eql, neq, lst, grt, leq, geq *)
      (* incl. set construction: comma, range *)

      CONST
	 CompareOp = SymSet{eql, neq, lst, grt, leq, geq};
	 (* operations with boolean result *)
	 BooleanOp = SymSet{inSY, isSY, orSY, ampersand} + CompareOp;
	 LogOp = SymSet{orSY, ampersand};
	 SetOp = SymSet{plus, minus, times, slash};
	 SetConstrOp = SymSet{comma, range};
	 ArithOp = SetOp + SymSet{divSY, modSY};
      VAR
	 ok: BOOLEAN;
	 forms: FormSet; (* set of legal forms for both operands *)

      PROCEDURE BadOp;
      BEGIN
	 DisposeAttribute(left); DisposeAttribute(right);
	 IF op IN BooleanOp THEN
	    NEW(at); ConstAt(boolptr, boolcon, ORD(TRUE), at);
	 ELSE
	    at := NIL;
	 END;
      END BadOp;

      PROCEDURE Fold(t: Type; sy: Symbol; val: Integer);
      BEGIN
	 DisposeAttribute(left); DisposeAttribute(right);
	 NEW(at); ConstAt(t, sy, val, at);
      END Fold;

      PROCEDURE ArithmeticType(t1, t2: Type) : Type;
      BEGIN
	 IF (t1 = NIL) OR (t2 = NIL) THEN
	    RETURN NIL
	 END;
	 IF t1 = t2 THEN
	    RETURN t1
	 END;
	 IF (t1^.form IN Numeric) & (t2^.form IN Numeric) THEN
	    IF t1^.form > t2^.form THEN
	       RETURN t1
	    ELSE
	       RETURN t2
	    END;
	 END;
	 RETURN t1
      END ArithmeticType;

      PROCEDURE TypeTest;
	 VAR
	    ok: BOOLEAN; (* FALSE on errors: return constant boolean *)
	    t1, t2: Type; (* type of `left' and `right' *)
	    base: Type; (* base type of `t1' and `t2' *)

	 PROCEDURE ExpectedType(form: Form; name: Identifier);
	 BEGIN
	    IF form = record THEN
	       Error1("%I must be a record type", name);
	    ELSE
	       Assert(form = pointer);
	       Error1("%I must be a pointer type", name);
	    END;
	 END ExpectedType;

      BEGIN
	 ok := CheckOpMode(left);
	 WITH left^ DO
	    IF ok & (attype # NIL) THEN
	       t1 := attype;
	       WITH attype^ DO
		  (* check the left operand for being a variable parameter *)
		  (* of record type or a pointer *)
		  IF (form # record) & (form # pointer) THEN
		     IF (mode IN IdModes) & (atip # NIL) THEN
			Error1("%I must be a record or pointer", atip^.name);
		     ELSE
			Error("record or pointer expected");
		     END;
		     ok := FALSE;
		  ELSIF form = record THEN
		     (* the left operand must be a designator because *)
		     (* functions cannot return records *)
		     Assert((mode = varAt) &
			    (atip # NIL) & (atip^.class = varC));
		     IF atip^.varkind # varparamV THEN
			Error1("%I must be a VAR-parameter", atip^.name);
			ok := FALSE;
		     END;
		  ELSE (* form = pointer *)
		     IF t1 = niltype THEN
			Error("NIL must not be subject of a type test");
		     ELSIF ~taggedptr THEN
			IF atip # NIL THEN
			   Error1("%I is an untagged pointer", atip^.name);
			ELSE
			   Error("left operand must be an untagged pointer");
			END;
			ok := FALSE;
		     END;
		  END;
	       END;
	    ELSE
	       ok := FALSE;
	    END;
	 END;
	 WITH right^ DO
	    IF mode # typeAt THEN
	       IF (mode IN IdModes) & (atip # NIL) THEN
		  Error1("%I must be a type identifier", atip^.name);
	       ELSE
		  Error("type identifier expected");
	       END;
	       ok := FALSE;
	    ELSE
	       t2 := attype;
	       IF t2 # NIL THEN
		  WITH t2^ DO
		     IF (form # record) & (form # pointer) THEN
			IF ok THEN
			   ExpectedType(t1^.form, atip^.name);
			ELSE
			   Error1("%I must be a record or pointer type",
			      atip^.name);
			END;
			ok := FALSE;
		     ELSIF (form = pointer) & ~taggedptr THEN
			Error1("%I is an untagged pointer type", atip^.name);
		     END;
		  END;
	       ELSE
		  (* error message printed *)
		  ok := FALSE;
	       END;
	    END;
	 END;
	 (* base type check *)
	 IF ok THEN
	    IF t1^.form # t2^.form THEN
	       ExpectedType(t1^.form, right^.atip^.name); ok := FALSE;
	    ELSIF ~BaseType(t1, t2, base) THEN
	       IF left^.atip # NIL THEN
		  Error2("%I cannot be of type %I", left^.atip^.name,
		     right^.atip^.name);
	       ELSE
		  Error1("operand cannot be of type %I", right^.atip^.name);
	       END;
	       ok := FALSE;
	    END;
	 END;

	 IF ok THEN
	    IF (t2 = base) OR (t2^.form = pointer) & (t2^.reftype = base) THEN
	       Fold(boolptr, boolcon, ORD(TRUE));
	    ELSE
	       NEW(at);
	       WITH at^ DO
		  mode := binaryAt;
		  attype := boolptr;
		  opsy := isSY;
		  leftop := left;
		  rightop := right;
	       END;
	    END;
	 ELSE
	    Fold(boolptr, boolcon, ORD(FALSE));
	 END;
      END TypeTest;

   BEGIN (* BinaryOp *)
      IF (left = NIL) & (op = comma) THEN
	 (* set construction: first element *)
	 NEW(left); ConstAt(setptr, setcon, Integer({}), left);
      ELSIF (right = NIL) & (op = comma) THEN
	 NEW(right); ConstAt(setptr, setcon, Integer({}), right);
      END;
      IF (left = NIL) OR (right = NIL) THEN
	 BadOp; RETURN
      END;

      IF op = isSY THEN
	 TypeTest;
      ELSE
	 ok := CheckOpMode(left);
	 IF ~CheckOpMode(right) OR ~ok THEN
	    BadOp; RETURN
	 END;

	 IF (op = inSY) THEN
	    IF ~CheckType(left^.attype, FormSet{shortint..longint}) THEN
	       Error("integer type expected");
	       ok := FALSE;
	    END;
	    IF ~CheckType(right^.attype, FormSet{set}) THEN
	       Error("SET type expected");
	       ok := FALSE;
	    END;
	 ELSIF IsStringAt(left) & IsStringAt(right) THEN
	    (* ok := TRUE; *)
	 ELSIF (op # comma) & ~Compatible(left^.attype, right^.attype) THEN
	    Error("type incompatible operands");
	    ok := FALSE;
	 END;
	 IF ~ok THEN
	    BadOp; RETURN
	 END;

	 (* check that operator is applyable *)
	 IF op IN (SetOp + ArithOp) THEN
	    forms := Numeric;
	    IF op IN SetOp THEN
	       INCL(forms, set);
	    END;
	 ELSIF op IN LogOp THEN
	    forms := FormSet{boolean};
	 ELSIF op = comma THEN
	    forms := FormSet{shortint..longint};
	    IF CheckAtType(left, FormSet{set, shortint..longint}) &
	       CheckAtType(right, FormSet{set, shortint..longint}) THEN
	       INCL(forms, set);
	    END;
	 ELSIF op = range THEN
	    forms := FormSet{shortint..longint};
	 ELSIF op = inSY THEN
	    forms := FormSet{shortint..longint, set};
	 ELSE (* op IN CompareOp *)
	    forms := Numeric + FormSet{char};
	    IF IsStringAt(left) THEN
	       INCL(forms, array);
	    END;
	    IF (op = eql) OR (op = neq) THEN
	       forms := forms + FormSet{boolean, set, pointer, proceduretype};
	    END;
	 END;
	 IF ~CheckAtType(left, forms) OR ~CheckAtType(right, forms) THEN
	    IF (left # NIL) & (left^.attype # NIL) &
	       (right # NIL) & (right^.attype # NIL) THEN
	       ExpectedForm(forms);
	    END;
	    BadOp;
	    RETURN
	 END;

	 IF (left^.mode = constAt) & (right^.mode = constAt) THEN
	    (* constant folding *)
	    Binary(op, left^.cval, right^.cval);
	    InitConstAt(left^.cval, at);
	    DisposeAttribute(left);
	    DisposeAttribute(right);
	 ELSIF (left^.mode = constAt) &
		  ((op = ampersand) & ~left^.cval.boolval OR
		   (op = orSY)      &  left^.cval.boolval) THEN
	    (* legal short circuit folding:
	       ONLY done if the left operand is constant,
	       e.g. `Fct' is called in evaluating "Fct() & FALSE"
		    `Fct' is not called in evaluating "FALSE & Fct()"
	    *)
	    NEW(at); ConstAt(boolptr, boolcon, ORD(left^.cval.boolval), at);
	    DisposeAttribute(left); DisposeAttribute(right);
	 ELSE
	    NEW(at);
	    WITH at^ DO
	       mode := binaryAt;
	       opsy := op;
	       IF opsy IN BooleanOp THEN
		  attype := boolptr;
	       ELSIF opsy IN SetConstrOp THEN
		  attype := setptr;
	       ELSE
		  attype := ArithmeticType(left^.attype, right^.attype);
		  IF ((opsy = modSY) OR (opsy = divSY)) &
			(attype^.form >= real) THEN
		     ExpectedForm(FormSet{shortint..longint});
		     opsy := slash; (* error recovery *)
		  END;
		  IF (opsy = slash) & (attype^.form < real) THEN
		     attype := realptr;
		  END;
	       END;
	       leftop := left;
	       rightop := right;
	    END;
	 END;
      END;
   END BinaryOp;

   PROCEDURE Set(VAR at: Attribute);
      (* "{" at "}" *)
      VAR
	 newat: Attribute;
   BEGIN
      IF at = NIL THEN
	 (* empty set *)
	 NEW(at);
	 WITH at^ DO
	    mode := constAt;
	    attype := setptr;
	    atip := NIL;
	    cval.sy := setcon;
	    cval.setval := {};
	 END;
      ELSE
	 newat := at;
	 WITH at^ DO
	    IF mode IN OpModes THEN
	       IF attype # NIL THEN
		  IF CheckType(attype, FormSet{shortint..longint}) THEN
		     BinaryOp(comma, NIL, at, newat);
		  ELSIF attype # setptr THEN
		     Error("bad element type");
		  END;
	       END;
	    ELSE
	       Error("bad element");
	    END;
	 END;
	 at := newat;
      END;
   END Set;

   PROCEDURE EvalConst(at: Attribute; VAR constval: Constval);
   BEGIN
      IF at # NIL THEN
	 IF at^.mode # constAt THEN
	    Error("constant expected");
	    DisposeAttribute(at);
	 ELSE
	    constval := at^.cval;
	 END;
      END;
      IF at = NIL THEN
	 WITH constval DO
	    sy := intcon;
	    intval := 13;
	 END;
      ELSE
	 DisposeAttribute(at);
      END;
   END EvalConst;

   PROCEDURE AssertConst(at: Attribute);
      (* print error message if `at' is not a constant *)
   BEGIN
      IF (at # NIL) & (at^.mode # constAt) THEN
	 Error("constant expected");
      END;
   END AssertConst;


   (* ---------------------------------------------------------	*)
   (* procedure/function calls					*)
   (* ProcDesignator {Parameter} [ProcedureCall | FunctionCall] *)
   (* ---------------------------------------------------------	*)

   PROCEDURE ProcDesignator(VAR at: Attribute);
      VAR
	 newat: Attribute;
   BEGIN
      IF at # NIL THEN
	 newat := NIL;
	 WITH at^ DO
	    IF (mode IN DesignatorModes) &
	       ~CheckType(attype, FormSet{proceduretype}) THEN
	       Error("procedure type expected");
	       DisposeAttribute(at);
	    ELSIF (mode # procAt) & ~(mode IN DesignatorModes) THEN
	       Error("procedure expected");
	       DisposeAttribute(at);
	    ELSIF attype # NIL THEN
	       IF attype^.form = proceduretype THEN
		  NEW(newat);
		  WITH newat^ DO
		     mode := callAt;
		     IF at^.attype^.function THEN
			attype := at^.attype^.restype;
		     ELSE
			attype := NIL;
		     END;
		     procat := at;
		     IF at^.attype^.std THEN
			paramref := NIL;
		     ELSE
			paramref := at^.attype^.param;
		     END;
		     firstparam := NIL;
		     tail := NIL;
		     paramcnt := 0;
		     toomany := FALSE;
		  END;
	       ELSE
		  Error("procedure type expected");
		  DisposeAttribute(at);
	       END;
	    END;
	 END;
	 at := newat;
      END;
   END ProcDesignator;

   PROCEDURE Parameter(parm: Attribute; VAR at: Attribute);

      PROCEDURE ParamComp(var: BOOLEAN; ft, actt: Type);
	 (* ft: type of formal parameter *)
	 (* actt: type of actual parameter *)
	 VAR
	    base: Type;

	 PROCEDURE DynComp(ft, actt: Type) : BOOLEAN;
	 BEGIN
	    WHILE (ft^.form = array) & (actt^.form = array) & ft^.dyn DO
	       ft := ft^.element;
	       actt := actt^.element;
	    END;
	    RETURN ft = actt
	 END DynComp;

      BEGIN
	 IF (ft = NIL) OR (actt = NIL) THEN
	    RETURN
	 END;
	 (* no check necessary for passing public parts because
	    this is done by AssignComp
	 *)
	 IF (ft = actt) OR
	    (ft^.form = array) & ft^.dyn &
	       ((ft^.element = byteptr) OR DynComp(ft, actt) OR
		(actt = charptr) & (ft^.element = charptr) &
		   (parm^.mode = constAt) & (parm^.cval.sy = stringcon)) OR
	    (ft = byteptr) & ((actt = charptr) OR (actt = shortptr)) THEN
	    RETURN
	 END;
	 IF var THEN
	    IF ~BaseType(ft, actt, base) OR (base # ft) THEN
	       IF (ft^.form = record) OR (ft^.form = pointer) THEN
		  IF ft^.ident # NIL THEN
		     Error1("type identity or extension of %I required",
			ft^.ident^.name);
		  ELSE
		     Error("type identity or extension of base type required");
		  END;
	       ELSE
		  IF ft^.ident # NIL THEN
		     Error1("type identity to %I required", ft^.ident^.name);
		  ELSE
		     Error("type identity required");
		  END;
	       END;
	    END;
	 ELSIF (ft^.form = array) & ft^.dyn THEN
	    IF (ft^.element # NIL) & (ft^.element^.ident # NIL) THEN
	       Error1("type must be assignment compatible to ARRAY OF %I",
		  ft^.element^.ident^.name);
	    ELSE
	       Error("type must be assignment compatible");
	    END;
	 ELSIF ~AssignComp(ft, actt, parm^.mode = constAt) THEN
	    IF ft^.ident # NIL THEN
	       Error1("type must be assignment compatible to %I",
		  ft^.ident^.name);
	    ELSE
	       Error("type must be assignment compatible");
	    END;
	    IF (ft^.form = proceduretype) &
	       (actt^.form = proceduretype) THEN
	       ProcedureDiff(actt, ft);
	    END;
	 END;
      END ParamComp;

   BEGIN (* Parameter *)
      IF at = NIL THEN
	 DisposeAttribute(parm);
      ELSE
	 WITH at^ DO
	    Assert(mode = callAt);
	    IF ~procat^.attype^.std THEN
	       IF paramref = NIL THEN
		  IF ~toomany THEN
		     Error("too many actual parameters");
		     toomany := TRUE;
		  END;
	       ELSE
		  IF parm # NIL THEN
		     IF (paramref^.varkind = varparamV) &
			~(parm^.mode IN DesignatorModes) THEN
			Error("designator expected for VAR-parameter");
		     ELSIF ~(parm^.mode IN OpModes) THEN
			Error("expression expected");
		     ELSE
			ParamComp(paramref^.varkind = varparamV,
				  paramref^.type, parm^.attype);
			IF (parm^.mode = procAt) & (parm^.atip # NIL) &
			   (parm^.atip^.plevel # global) THEN
			   (* if a formal parameter specifies a procedure type,
			      then the corresponding actual parameter must be
			      either a procedure declared at level global or
			      a variable (or parameter) of that procedure type.
			   *)
			   Error("procedure parameters must not be local procedures");
			END;
		     END;
		  END;
		  paramref := paramref^.link;
	       END;
	    END;
	    INC(paramcnt);
	    IF parm # NIL THEN
	       parm^.link := NIL;
	       IF tail # NIL THEN
		  tail^.link := parm;
	       ELSE
		  at^.firstparam := parm;
	       END;
	       tail := parm;
	    END;
	 END;
      END;
   END Parameter;

   PROCEDURE CheckParamCnt(stdp: StdProc; paramcnt: CARDINAL) : BOOLEAN;
   BEGIN
      CASE stdp OF
      | absF, oddF, capF, adrF, sizeF, maxF, minF,
	ordF, chrF, shortF, longF, entierF, newP, haltP,
	unixforkF, tasF, crswitchP, syshaltP:
	    RETURN paramcnt = 1
      | ashF, inclP, exclP, copyP, bitF, lshF, rotF, valF, getP, putP, sysnewP,
	wclearP:
	    RETURN paramcnt = 2
      | incP, decP, lenF, crspawnP:
	    RETURN (paramcnt = 1) OR (paramcnt = 2)
      | moveP, wmoveP:
	    RETURN paramcnt = 3
      | unixcallF:
	    RETURN paramcnt >= 3
      | unixsignalF:
	    RETURN paramcnt = 4
      END;
   END CheckParamCnt;

   PROCEDURE ProcedureCall(at: Attribute);

      PROCEDURE Std(stdp: StdProc; params: Attribute; paramcnt: CARDINAL);
	 CONST
	    maxargs = 3; (* maximal number of args of standard procedures *)
	 TYPE
	    StdProcSet = SET OF StdProc;
	 VAR
	    ats: ARRAY [1..maxargs] OF Attribute;
	    types: ARRAY [1..maxargs] OF Type;
	    forms: ARRAY [1..maxargs] OF FormSet;
	    constval: Constval;
	    index: CARDINAL;

      BEGIN
	 IF ~CheckParamCnt(stdp, paramcnt) THEN
	    Error("bad number of arguments"); RETURN
	 END;
	 FOR index := 1 TO paramcnt DO
	    IF params = NIL THEN RETURN END;
	    ats[index] := params;
	    IF ats[index]^.mode IN OpModes THEN
	       types[index] := ats[index]^.attype;
	    ELSE
	       Error("bad argument"); RETURN
	    END;
	    params := params^.link;
	 END;
	 IF (stdp = haltP) OR (stdp = syshaltP) THEN
	    (* argument must be a constant *)
	    IF ats[1]^.mode # constAt THEN
	       Error("constant required for argument of HALT");
	    END;
	 ELSIF stdp IN StdProcSet{decP, incP, inclP, exclP, newP,
				  sysnewP, moveP, crspawnP, crswitchP} THEN
	    (* first argument must be a designator *)
	    IF ~(ats[1]^.mode IN DesignatorModes) THEN
	       Error1("designator required for first argument of %I",
		  at^.procat^.atip^.name);
	    ELSIF (stdp = newP) & (types[1] # NIL) &
		  (types[1]^.form = pointer) & ~types[1]^.taggedptr THEN
	       Error("SYSTEM.NEW is to be used instead of NEW for untagged pointers");
	    END;
	 END;
	 IF stdp IN StdProcSet{copyP, getP, moveP} THEN
	    (* second argument must be a designator *)
	    IF ~(ats[2]^.mode IN DesignatorModes) THEN
	       Error1("designator required for second argument of %I",
		  at^.procat^.atip^.name);
	    END;
	 END;
	 CASE stdp OF
	 | decP, incP:     (* INC(v)  INC(v,x)  DEC(v)  DEC(v,x) *)
			   forms[1] := FormSet{shortint..longint};
			   forms[2] := FormSet{shortint..longint};
	 | inclP, exclP:   (* INCL(v,x)  EXCL(v,x) *)
			   forms[1] := FormSet{set};
			   forms[2] := FormSet{shortint..longint};
	 | copyP:          (* COPY(x,v) *)
			   forms[1] := FormSet{array};
			   forms[2] := FormSet{array};
	 | newP:           (* NEW(v) *)
			   forms[1] := FormSet{pointer};
	 | haltP:          (* HALT(x) *)
			   forms[1] := FormSet{shortint..longint};
	 (* SYSTEM module *)
	 | getP:           (* SYSTEM.GET(a,v) *)
			   forms[1] := FormSet{longint};
			   forms[2] := Basic;
	 | putP:           (* SYSTEM.PUT(a,x) *)
			   forms[1] := FormSet{longint};
			   forms[2] := Basic;
	 | moveP:          (* SYSTEM.MOVE(v0,v1,n) *)
			   forms[1] := FormSet{MIN(Form)..MAX(Form)};
			   forms[2] := FormSet{MIN(Form)..MAX(Form)};
			   forms[3] := FormSet{shortint..longint};
	 | sysnewP:        (* SYSTEM.NEW(v,n) *)
			   forms[1] := FormSet{pointer};
			   forms[2] := FormSet{shortint..longint};
	 | wmoveP:         (* SYSTEM.WMOVE(from,to,n) *)
			   forms[1] := FormSet{shortint..longint};
			   forms[2] := FormSet{shortint..longint};
			   forms[3] := FormSet{shortint..longint};
	 | wclearP:        (* SYSTEM.WCLEAR(ptr,n) *)
			   forms[1] := FormSet{shortint..longint};
			   forms[2] := FormSet{shortint..longint};
	 | crspawnP:       (* SYSTEM.CRSPAWN(cr,size) *)
			   forms[1] := FormSet{pointer};
			   forms[2] := FormSet{shortint..longint};
	 | crswitchP:      (* SYSTEM.TRANSFER(src,dest) *)
			   forms[1] := FormSet{pointer};
	 | syshaltP:       (* SYSTEM.HALT(x) *)
			   forms[1] := FormSet{shortint..longint};
	 ELSE Assert(FALSE)
	 END;
	 FOR index := 1 TO paramcnt DO
	    IF (types[index] # NIL) & ~(types[index]^.form IN forms[index]) THEN
	       ExpectedForm(forms[index]);
	    END;
	 END;
	 IF (stdp = copyP) & (types[1] # NIL) & (types[2] # NIL) &
	    (types[1]^.form = array) & (types[2]^.form = array) THEN
	    (* check arguments of COPY to be character arrays *)
	    WITH types[1]^ DO
	       IF (element # NIL) & (element^.form # char) THEN
		  Error1("character array required for first argument of %I",
		     at^.procat^.atip^.name);
	       END;
	    END;
	    WITH types[2]^ DO
	       IF (element # NIL) & (element^.form # char) THEN
		  Error1("character array required for second argument of %I",
		     at^.procat^.atip^.name);
	       END;
	    END;
	 ELSIF (stdp = crspawnP) OR (stdp = crswitchP) THEN
	    IF (types[1] # NIL) & (types[1] # crptr) THEN
	       Error2("first parameter of %I must be of type %I",
		  at^.procat^.atip^.name, crptr^.ident^.name);
	    END;
	    IF stdp = crspawnP THEN
	       IF badprocedure THEN
		  Error1("%I must not be called inside a module body",
		     at^.procat^.atip^.name);
	       ELSE
		  IF procedureAt^.attype^.function THEN
		     Error("coroutines must not be functions");
		  ELSIF procedureAt^.atip^.plevel > global THEN
		     Error("coroutines must be global procedures");
		  END;
	       END;
	    END;
	 END;
      END Std;

   BEGIN
      IF at # NIL THEN
	 WITH at^ DO
	    Assert(mode = callAt);
	    IF paramref # NIL THEN
	       Error("missing actual parameters");
	    END;
	    IF procat^.attype^.function THEN
	       Error("procedure call expected");
	    ELSIF procat^.attype^.std THEN
	       (* standard procedure or function *)
	       Std(procat^.attype^.stdproc, firstparam, paramcnt);
	    END;
	 END;
	 IF ~errorflag THEN
	    GenCall(at);
	 END;
      END;
      DisposeAttribute(at);
   END ProcedureCall;
   
   PROCEDURE FunctionCall(VAR at: Attribute);

      PROCEDURE Std(stdp: StdProc; params: Attribute; paramcnt: CARDINAL);
	 CONST
	    maxargs = 32; (* maximal number of arguments *)
	    intlen = bitsperword; (* # number of bits per LONGINT *)
	 TYPE
	    StdProcSet = SET OF StdProc;
	 VAR
	    ats: ARRAY [1..maxargs] OF Attribute;
	    const: ARRAY [1..maxargs] OF BOOLEAN;
	    types: ARRAY [1..maxargs] OF Type;
	    index: CARDINAL;
	    ch: CHAR;
	    cnt: CARDINAL;
	    constval: Constval;
	    ok: BOOLEAN;
	    restype: Type;
	    fold: BOOLEAN; (* if true: result in `constval' *)

	 PROCEDURE Float(i: Integer) : Real;
	 BEGIN
	    RETURN FLOAT(i)
	 END Float;

	 PROCEDURE ArithShift(val, n: Integer) : Integer;
	 BEGIN
	    IF n >= intlen THEN
	       val := 0;
	    ELSIF n <= -intlen THEN
	       IF val < 0 THEN
		  val := -1;
	       ELSE
		  val := 0;
	       END;
	    ELSIF n > 0 THEN (* shift left *)
	       WHILE n > 0 DO
		  val := val * 2;
		  DEC(n);
	       END;
	    ELSIF n < 0 THEN (* shift right *)
	       WHILE n < 0 DO
		  val := val DIV 2;
		  INC(n);
	       END;
	    END;
	    RETURN val
	 END ArithShift;

	 PROCEDURE LogShift(val, n: Integer) : Integer;
	    VAR
	       neg: BOOLEAN; (* sign bit to be set? *)
	 BEGIN
	    IF (n >= intlen) OR (n <= -intlen) THEN
	       val := 0;
	    ELSIF n > 0 THEN (* shift left *)
	       WHILE n > 0 DO
		  neg := val >= MAX(Integer) DIV 2 + 1;
		  val := val * 2;
		  DEC(n);
	       END;
	       IF neg & (val >= 0) THEN
		  val := MIN(Integer) + val;
	       END;
	    ELSIF n < 0 THEN (* shift right *)
	       IF val < 0 THEN
		  val := (val + MAX(Integer) + 1) DIV 2 +
			 MAX(Integer) DIV 2 + 1;
		  INC(n);
	       END;
	       WHILE n < 0 DO
		  val := val DIV 2;
		  INC(n);
	       END;
	    END;
	    RETURN val
	 END LogShift;

	 PROCEDURE Rotate(val, n: Integer) : Integer;
	    VAR
	       neg: BOOLEAN;
	 BEGIN
	    n := n MOD intlen;
	    (* n in [0..intlen-1] --- rotate left *)
	    WHILE n > 0 DO
	       neg := val < 0;
	       IF neg THEN
		  val := val + MAX(Integer) + 1;
	       END;
	       val := val * 2;
	       IF neg THEN
		  INC(val);
	       END;
	       DEC(n);
	    END;
	    RETURN val
	 END Rotate;

      BEGIN (* Std *)
	 IF ~CheckParamCnt(stdp, paramcnt) THEN
	    Error("bad number of arguments"); RETURN
	 END;
	 FOR index := 1 TO paramcnt DO
	    ats[index] := params;
	    IF ats[index] = NIL THEN RETURN END;
	    const[index] := ats[index]^.mode = constAt;
	    IF (ats[index]^.mode IN OpModes) OR (ats[index]^.mode = typeAt) THEN
	       types[index] := ats[index]^.attype;
	       IF types[index] = NIL THEN RETURN END;
	    END;
	    params := params^.link;
	 END;
	 IF const[1] THEN
	    constval := ats[1]^.cval; (* default *)
	 END;
	 IF ~(ats[1]^.mode IN OpModes) &
	    ~(stdp IN StdProcSet{minF, maxF, sizeF, valF}) THEN
	    Error("bad argument"); RETURN
	 END;
	 FOR index := 2 TO paramcnt DO
	    IF ~(ats[index]^.mode IN OpModes) THEN
	       Error("bad argument"); RETURN
	    END;
	 END;
	 fold := FALSE;
	 WITH at^ DO
	    CASE stdp OF
	    | absF:  (* ABS(x) *)
		     IF CheckForm(types[1]^.form, Numeric) THEN
			attype := ats[1]^.attype;
			IF const[1] THEN
			   WITH ats[1]^.cval DO
			      IF sy = intcon THEN
				 IF intval = minlong THEN
				    Error("constant out of range");
				 ELSE
				    constval.intval := ABS(intval);
				 END;
			      ELSE (* realcon, longrealcon *)
				 constval.realval := ABS(realval);
			      END;
			      fold := TRUE;
			   END;
			END;
		     END;
	    | oddF:  (* ODD(x) *)
		     attype := boolptr;
		     IF CheckForm(types[1]^.form, FormSet{shortint..longint}) &
			const[1] THEN
			WITH ats[1]^.cval DO
			   Assert(sy = intcon);
			   constval.sy := boolcon;
			   constval.boolval := ODD(intval);
			   fold := TRUE;
			END;
		     END;
	    | capF:  (* CAP(x) *)
		     attype := charptr;
		     IF CheckForm(types[1]^.form, FormSet{char}) & const[1] THEN
			WITH ats[1]^.cval DO
			   Assert((sy = charcon) OR (sy = stringcon));
			   IF sy = charcon THEN
			      ch := charval;
			   ELSE
			      GetStringChar(string, ch);
			      RestOfString(string);
			   END;
			   IF (ch >= 'a') & (ch <= 'z') THEN
			      ch := CAP(ch);
			      constval.sy := charcon;
			      constval.charval := ch;
			      fold := TRUE;
			   ELSE
			      Error("not a lower case letter");
			   END;
			END;
		     END;
	    | ashF:  (* ASH(x, n) *)
		     attype := longptr;
		     IF CheckForm(types[1]^.form, FormSet{shortint..longint}) &
			CheckForm(types[2]^.form, FormSet{shortint..longint}) &
			(const[1] OR const[2]) THEN
			IF const[1] THEN
			   WITH ats[1]^.cval DO
			      IF intval = 0 THEN
				 constval.sy := intcon;
				 constval.intval := 0;
				 fold := TRUE;
			      ELSIF const[2] THEN
				 cnt := ats[2]^.cval.intval;
				 constval.sy := intcon;
				 constval.intval := ArithShift(intval, cnt);
				 fold := TRUE;
			      END;
			   END;
			END;
		     END;
	    | lenF:  (* LEN(v), LEN(v, n), v ranges from 0 to #dims-1 *)
		     attype := longptr;
		     IF CheckForm(types[1]^.form, FormSet{array}) &
			(  (paramcnt = 1) OR
			   CheckForm(types[2]^.form,
				  FormSet{shortint..longint})) THEN
			IF ats[1]^.mode # varAt THEN
			   Error("first argument of LEN must be a variable");
			END;
			IF (paramcnt = 2) & const[2] THEN
			   WITH ats[2]^.cval DO
			      IF intval < 0 THEN
				 Error("dimension must not be negative");
			      ELSE
				 cnt := intval;
				 WHILE cnt > 0 DO
				    types[1] := types[1]^.element;
				    IF (types[1] = NIL) OR
				       (types[1]^.form # array) THEN
				       IF types[1] # NIL THEN
					  Error("dimension out of range");
				       END;
				       cnt := 1;
				    END;
				    DEC(cnt);
				 END;
			      END;
			   END;
			END;
			IF (types[1] # NIL) & (types[1]^.form = array) &
			   ~types[1]^.dyn &
			   ((paramcnt = 1) OR const[2]) THEN
			   fold := TRUE;
			   constval.sy := intcon;
			   constval.intval := types[1]^.length;
			END;
		     END;
	    | maxF:  (* MAX(T) *)
		     IF ats[1]^.mode = typeAt THEN
			attype := ats[1]^.attype; (* default *)
			fold := TRUE; constval.sy := intcon; (* default *)
			WITH constval DO
			   CASE types[1]^.form OF
			   | shortint: intval := maxshort;
			   | int16:    intval := maxint16;
			   | integer:  intval := maxint;
			   | longint:  intval := maxlong;
			   | real:     sy := realcon; realval := maxreal;
			   | longreal: sy := longrealcon; realval := maxlreal;
			   | boolean:  sy := boolcon; boolval := TRUE;
			   | char:     sy := charcon; charval := MAX(CHAR);
			   | set:      attype := shortptr;
				       intval := bitsperword-1;
			   | byte:     intval := ORD(MAX(CHAR));
			   ELSE
			      Error("basic type required");
			      intval := 0;
			   END;
			END;
		     ELSE
			Error("type identifier required");
		     END;
	    | minF:  (* MIN(T) *)
		     IF ats[1]^.mode = typeAt THEN
			attype := ats[1]^.attype; (* default *)
			fold := TRUE; constval.sy := intcon; (* default *)
			WITH constval DO
			   CASE types[1]^.form OF
			   | shortint: intval := minshort;
			   | int16:    intval := minint16;
			   | integer:  intval := minint;
			   | longint:  intval := minlong;
			   | real:     sy := realcon; realval := minreal;
			   | longreal: sy := longrealcon; realval := minlreal;
			   | boolean:  sy := boolcon; boolval := FALSE;
			   | char:     sy := charcon; charval := MIN(CHAR);
			   | set:      attype := shortptr;
				       intval := 0;
			   | byte:     intval := 0;
			   ELSE
			      Error("basic type required");
			      intval := 0;
			   END;
			END;
		     ELSE
			Error("type identifier required");
		     END;
	    | ordF:  (* ORD(x) *)
		     attype := intptr;
		     IF CheckForm(types[1]^.form, FormSet{char, byte}) &
			const[1] THEN
			fold := TRUE;
			constval.sy := intcon;
			WITH ats[1]^.cval DO
			   CASE sy OF
			   | intcon:      constval.intval := intval;
			   | charcon:     constval.intval := ORD(charval);
			   | stringcon:   GetStringChar(string, ch);
			                  RestOfString(string);
					  constval.intval := ORD(ch);
			   END;
			END;
		     END;
	    | chrF:  (* CHR(x) *)
		     attype := charptr;
		     IF CheckForm(types[1]^.form,
				  FormSet{shortint..longint, byte}) &
			const[1] THEN
			fold := TRUE;
			constval.sy := charcon;
			WITH ats[1]^.cval DO
			   Assert(sy = intcon);
			   IF (intval < 0) OR
			      (intval > VAL(Integer, ORD(MAX(CHAR)))) THEN
			      Error("argument out of range");
			      constval.charval := 0C;
			   ELSE
			      constval.charval := CHR(intval);
			   END;
			END;
		     END;
	    | shortF:(* SHORT(x) *)
		     IF CheckForm(types[1]^.form,
				  FormSet{int16, integer..longint,
					  longreal}) THEN
			CASE types[1]^.form OF
			| int16:    attype := shortptr;
			| integer:  attype := shortptr;
			| longint:  attype := intptr;
			| longreal: attype := realptr;
			END;
			IF const[1] THEN
			   fold := TRUE;
			   WITH ats[1]^.cval DO
			      CASE types[1]^.form OF
			      | integer,
				int16:    ok := (intval >= minshort) &
						(intval <= maxshort);
			      | longint:  ok := (intval >= minint) &
						(intval <= maxint);
			      | longreal: ok := (realval >= minreal) &
						(realval <= maxreal);
			      END;
			      IF ~ok THEN
				 Error("argument out of range");
			      END;
			   END;
			END;
		     ELSE
			attype := NIL;
		     END;
	    | longF: (* LONG(x) *)
		     IF CheckForm(types[1]^.form,
				  FormSet{shortint..integer, real}) THEN
			CASE types[1]^.form OF
			| shortint: attype := intptr;
			| int16:    attype := intptr;
			| integer:  attype := longptr;
			| real:     attype := longrealptr;
			END;
			IF const[1] THEN
			   fold := TRUE;
			END;
		     ELSE
			attype := NIL;
		     END;
	    | entierF: (* ENTIER(x) *)
		     attype := longptr;
		     IF CheckForm(types[1]^.form, FormSet{real, longreal}) &
			const[1] THEN
			fold := TRUE;
			constval.sy := intcon;
			WITH ats[1]^.cval DO
			   IF (realval >= 0.0) &
			      (realval <= Float(maxlong)) THEN
			      constval.intval := TRUNC(realval);
			   ELSIF (realval < 0.0) &
				 (realval >= Float(minlong)) THEN
			      constval.intval := TRUNC(realval-1.0);
			   ELSE
			      Error("argument out of range");
			   END;
			END;
		     END;

	    (* SYSTEM module *)
	    | adrF:  (* SYSTEM.ADR(v) *)
		     attype := addrptr;
		     IF ~(ats[1]^.mode IN DesignatorModes) THEN
			Error("not a variable");
		     END;
	    | bitF:  (* SYSTEM.BIT(a, n) *)
		     attype := boolptr;
		     IF CheckForm(types[1]^.form, FormSet{longint}) &
			CheckForm(types[1]^.form,
				  FormSet{shortint..longint}) THEN
		     END;
	    | lshF:  (* SYSTEM.LSH(x, n) *)
		     IF types[1]^.form = set THEN
			attype := setptr;
		     ELSE
			attype := longptr;
		     END;
		     IF CheckForm(types[1]^.form,
				  FormSet{shortint..longint, set}) &
			CheckForm(types[2]^.form,
				  FormSet{shortint..longint}) &
			(const[1] OR const[2]) THEN
			IF const[1] THEN
			   WITH ats[1]^.cval DO
			      IF sy = intcon THEN
				 IF intval = 0 THEN
				    constval.sy := intcon;
				    constval.intval := 0;
				    fold := TRUE;
				 ELSIF const[2] THEN
				    cnt := ats[2]^.cval.intval;
				    constval.sy := intcon;
				    constval.intval := LogShift(intval, cnt);
				    fold := TRUE;
				 END;
			      ELSIF sy = setcon THEN
				 IF setval = Types.Set{} THEN
				    constval.sy := setcon;
				    constval.setval := Types.Set{};
				    fold := TRUE;
				 ELSIF const[2] THEN
				    cnt := ats[2]^.cval.intval;
				    constval.sy := setcon;
				    constval.setval := Types.Set(
				       LogShift(Integer(setval), cnt));
				    fold := TRUE;
				 END;
			      END;
			   END;
			END;
		     END;
	    | rotF:  (* SYSTEM.ROT(x, n) *)
		     IF types[1]^.form = set THEN
			attype := setptr;
		     ELSE
			attype := longptr;
		     END;
		     IF CheckForm(types[1]^.form,
				  FormSet{shortint..longint, set}) &
			CheckForm(types[2]^.form, FormSet{shortint..longint}) &
			(const[1] OR const[2]) THEN
			IF const[1] THEN
			   WITH ats[1]^.cval DO
			      IF sy = intcon THEN
				 IF intval = 0 THEN
				    constval.sy := intcon;
				    constval.intval := 0;
				    fold := TRUE;
				 ELSIF const[2] THEN
				    cnt := ats[2]^.cval.intval;
				    constval.sy := intcon;
				    constval.intval := Rotate(intval, cnt);
				    fold := TRUE;
				 END;
			      ELSIF sy = setcon THEN
				 IF setval = Types.Set{} THEN
				    constval.sy := setcon;
				    constval.setval := Types.Set{};
				    fold := TRUE;
				 ELSIF const[2] THEN
				    cnt := ats[2]^.cval.intval;
				    constval.sy := setcon;
				    constval.setval := Types.Set(
				       Rotate(Integer(setval), cnt));
				    fold := TRUE;
				 END;
			      END;
			   END;
			END;
		     END;
	    | sizeF: (* SYSTEM.SIZE(T) *)
		     attype := intptr;
		     IF ats[1]^.mode = typeAt THEN
			WITH ats[1]^.attype^ DO
			   IF ~defunit OR
				 (form # record) OR ~projection OR extended THEN
			      fold := TRUE;
			      constval.sy := intcon;
			      constval.intval := size;
			   END;
			END;
		     ELSE
			Error("type identifier required");
		     END;
	    | valF:  (* SYSTEM.VAL(T,x) *)
		     IF ats[1]^.mode = typeAt THEN
			attype := types[1];
			IF const[2] THEN
			   IF types[1]^.form IN FormSet{shortint..longint} THEN
			      WITH ats[2]^.cval DO
				 IF sy = nilSY THEN
				    constval.sy := intcon;
				    constval.intval := nilvalue;
				    fold := TRUE;
				 ELSIF sy = charcon THEN
				    constval.sy := intcon;
				    constval.intval := ORD(charval);
				    fold := TRUE;
				 ELSIF sy = setcon THEN
				    constval.sy := intcon;
				    constval.intval := Integer(setval);
				    fold := TRUE;
				 ELSIF sy = boolcon THEN
				    constval.sy := intcon;
				    constval.intval := ORD(boolval);
				    fold := TRUE;
				 ELSIF sy = intcon THEN
				    constval.sy := intcon;
				    constval.intval := intval;
				    fold := TRUE;
				 END;
			      END;
			   ELSIF types[1]^.form = set THEN
			      WITH ats[2]^.cval DO
				 IF sy = nilSY THEN
				    constval.sy := setcon;
				    constval.setval := Types.Set(nilvalue);
				    fold := TRUE;
				 ELSIF sy = charcon THEN
				    constval.sy := setcon;
				    constval.setval := Types.Set(ORD(charval));
				    fold := TRUE;
				 ELSIF sy = setcon THEN
				    constval.sy := setcon;
				    constval.setval := setval;
				    fold := TRUE;
				 ELSIF sy = boolcon THEN
				    constval.sy := setcon;
				    constval.setval := Types.Set(ORD(boolval));
				    fold := TRUE;
				 ELSIF sy = intcon THEN
				    constval.sy := setcon;
				    constval.setval := Types.Set(intval);
				    fold := TRUE;
				 END;
			      END;
			   END;
			END;
		     ELSE
			Error("type identifier required");
			attype := NIL;
		     END;
	    | unixcallF: (* SYSTEM.UNIXCALL(call,d0,d1, p1,p2,...) *)
		     attype := boolptr;
		     IF ~const[1] THEN
			Error("first argument of UNIXCALL must be a constant");
		     END;
		     FOR index := 2 TO 3 DO
			IF CheckForm(types[index]^.form, FormSet{integer}) THEN
			   IF ~(ats[index]^.mode IN DesignatorModes) THEN
			      Error("designator expected");
			   END;
			END;
		     END;
		     FOR index := 4 TO paramcnt DO
			IF (types[index] # NIL) &
			   (types[index]^.size > oneword) THEN
			   Error("bad type of argument");
			END;
		     END;
	    | unixforkF: (* SYSTEM.UNIXFORK(pid) *)
		     attype := boolptr;
		     IF CheckForm(types[1]^.form, FormSet{integer}) THEN
			IF ~(ats[1]^.mode IN DesignatorModes) THEN
			   Error("designator expected");
			END;
		     END;
	    | unixsignalF: (* SYSTEM.UNIXSIGNAL(sig,p,old,error) *)
		     attype := boolptr;
		     IF ~CheckForm(types[1]^.form,
				   FormSet{shortint..integer}) THEN
		     END;
		     FOR index := 2 TO 3 DO
			IF ~(types[index]^.form IN FormSet{shortint..longint}) &
			   CheckForm(types[index]^.form,
			       FormSet{proceduretype, shortint..longint}) THEN
			   IF ~ProcedureComp(types[index], sigprocptr) THEN
			      Error("bad procedure type; PROCEDURE (INTEGER, ARRAY OF BYTE) expected");
			   END;
			END;
		     END;
		     FOR index := 3 TO 4 DO
			IF ~(ats[index]^.mode IN DesignatorModes) THEN
			   Error("designator expected");
			END;
		     END;
		     IF ~CheckForm(types[4]^.form, FormSet{integer}) THEN END;
	    | tasF:  (* SYSTEM.TAS(b) *)
		     attype := boolptr;
		     IF CheckForm(types[1]^.form, FormSet{boolean}) THEN
			IF ~(ats[1]^.mode IN DesignatorModes) THEN
			   Error("designator expected");
			END;
		     END;
	    END;
	 END;
	 IF fold THEN
	    restype := at^.attype;
	    DisposeAttribute(at);
	    NEW(at);
	    WITH at^ DO
	       mode := constAt;
	       atip := NIL;
	       attype := restype;
	       cval := constval;
	    END;
	 END;
      END Std;

   BEGIN (* FunctionCall *)
      IF at # NIL THEN
	 WITH at^ DO
	    IF paramref # NIL THEN
	       Error("missing actual parameters");
	    END;
	    IF ~procat^.attype^.function THEN
	       Error("function call expected");
	       DisposeAttribute(at);
	    ELSIF procat^.attype^.std THEN
	       (* standard procedure or function *)
	       Std(procat^.attype^.stdproc, firstparam, paramcnt);
	    END;
	 END;
      END;
   END FunctionCall;


   (* ---------------------------------------------------------	*)
   (* statements -- procedure calls see above			*)
   (* ---------------------------------------------------------	*)

   PROCEDURE Assignment(desat, exprat: Attribute);
      (* desat := exprat *)
   BEGIN
      IF (desat # NIL) & (exprat # NIL) &
	 (desat^.mode IN DesignatorModes) THEN
	 IF exprat^.mode IN OpModes THEN
	    IF ~AssignComp(desat^.attype, exprat^.attype,
			   exprat^.mode = constAt) THEN
	       IF desat^.attype^.ident # NIL THEN
		  Error1("expression not assignment compatible to %I",
		     desat^.attype^.ident^.name);
	       ELSE
		  Error("expression not assignment compatible to designator");
	       END;
	       IF (desat^.attype^.form = proceduretype) &
		  (exprat^.attype^.form = proceduretype) THEN
		  ProcedureDiff(desat^.attype, exprat^.attype);
	       END;
	    ELSE
	       IF ~errorflag THEN
		  GenAssign(desat, exprat);
	       END;
	    END;
	    IF (exprat^.mode = procAt) & (exprat^.atip # NIL) &
		  (exprat^.atip^.plevel # global) THEN
	       Error("local procedures must not be assigned to");
	    END;
	 ELSE
	    Error("expression expected");
	 END;
      END;
   END Assignment;

   PROCEDURE IfThen(condat: Attribute);
      (* IfThen { ElsifThen } [ Else ] EndIf *)
   BEGIN
      IF (condat # NIL) & CheckOpMode(condat) THEN
	 WITH condat^ DO
	    IF (attype # NIL) & (attype # boolptr) THEN
	       Error("boolean type expected");
	    END;
	 END;
	 IF ~errorflag THEN
	    GenIfThen(condat);
	 END;
      END;
   END IfThen;

   PROCEDURE ElsifThen(condat: Attribute);
   BEGIN
      IF (condat # NIL) & CheckOpMode(condat) THEN
	 WITH condat^ DO
	    IF (attype # NIL) & (attype # boolptr) THEN
	       Error("boolean type expected");
	    END;
	 END;
	 IF ~errorflag THEN
	    GenElsifThen(condat);
	 END;
      END;
   END ElsifThen;

   PROCEDURE Else;
   BEGIN
      IF ~errorflag THEN
	 GenElse;
      END;
   END Else;

   PROCEDURE EndIf;
   BEGIN
      IF ~errorflag THEN
	 GenEndIf;
      END;
   END EndIf;

   PROCEDURE CaseOf(at: Attribute; VAR labels: Labels);
      (* CaseOf { CaseLabels } [ CaseElse ] EndCase *)
   BEGIN
      NEW(labels);
      WITH labels^ DO
	 ltype := NIL;
	 IF at # NIL THEN
	    IF at^.mode IN OpModes THEN
	       ltype := at^.attype;
	       IF ltype # NIL THEN
		  IF NOT (ltype^.form IN FormSet{shortint..longint, char}) THEN
		     Error("must be of integer type or CHAR");
		  END;
	       END;
	    ELSE
	       Error("expression expected");
	    END;
	 END;
	 count := 0;
	 head := NIL; tail := NIL;
	 IF ~errorflag THEN
	    GenCaseOf(at);
	 END;
      END;
   END CaseOf;

   PROCEDURE InitLabel(labels: Labels);
      (* InitLabel { AddLabel | AddRange }+ CaseLabels *)
   BEGIN
      Assert(labels # NIL);
   END InitLabel;

   PROCEDURE CaseConst(ltype: Type; at: Attribute;
		       VAR caseval: CaseLabel) : BOOLEAN;
      VAR
	 typeok: BOOLEAN;
	 ch: CHAR;

      PROCEDURE SameType(t1, t2: Type) : BOOLEAN;
      BEGIN
	 RETURN (t1 = NIL) OR (t2 = NIL) OR (t1 = t2) OR
		(t1^.form IN FormSet{shortint..longint}) &
		(t2^.form IN FormSet{shortint..longint})
      END SameType;

   BEGIN
      IF at # NIL THEN
	 WITH at^ DO
	    IF (mode IN OpModes) & (attype # NIL) THEN
	       typeok := attype^.form IN FormSet{shortint..longint, char};
	       IF ~typeok & (ltype = NIL) THEN
		  Error("type of case label must be an integer type or CHAR");
	       END;
	       IF ~SameType(ltype, attype) THEN
		  Error("type of case label different from type of case expression");
		  typeok := FALSE;
	       END;
	    END;
	    IF mode = constAt THEN
	       IF typeok THEN
		  CASE cval.sy OF
		  | charcon:     caseval := ORD(cval.charval);
		  | intcon:      caseval := cval.intval;
		  | stringcon:   GetStringChar(cval.string, ch);
				 RestOfString(cval.string);
			         caseval := ORD(ch);
		  END;
		  RETURN TRUE
	       END;
	    ELSE
	       Error("constant expected");
	    END;
	 END;
      END;
      RETURN FALSE
   END CaseConst;

   PROCEDURE AddCaseLabelRange(VAR labels: Labels; lbl1, lbl2: CaseLabel);
      (* lbl1 <= lbl2 *)
      VAR
	 range: LabelList;
	 prev: LabelList;
	 ok: BOOLEAN;
   BEGIN
      ok := TRUE;
      NEW(range);
      WITH range^ DO
	 low := lbl1;
	 high := lbl2;
	 case := labels^.count;
	 link := NIL;
      END;
      WITH labels^ DO
	 IF head = NIL THEN
	    head := range; tail := range;
	    min := lbl1;
	    max := lbl2;
	 ELSE
	    IF lbl2 < min THEN
	       range^.link := head;
	       head := range;
	    ELSIF lbl1 > max THEN
	       tail^.link := range;
	       tail := range;
	    ELSE
	       prev := head;
	       WHILE (prev^.link # NIL) & (lbl1 > prev^.link^.high) DO
		  prev := prev^.link;
	       END;
	       IF (prev^.high >= lbl1) OR
		  (prev^.link # NIL) & (lbl2 >= prev^.link^.low) THEN
		  Error("case label defined twice");
		  ok := FALSE;
	       ELSE
		  range^.link := prev^.link;
		  prev^.link := range;
	       END;
	    END;
	    IF ok THEN
	       IF lbl1 < min THEN
		  min := lbl1;
	       END;
	       IF lbl2 > max THEN
		  max := lbl2;
	       END;
	    END;
	 END;
      END;
      IF ~ok THEN DISPOSE(range) END;
   END AddCaseLabelRange;

   PROCEDURE AddLabel(at: Attribute; VAR labels: Labels);
      VAR caselbl: CaseLabel;
   BEGIN
      IF CaseConst(labels^.ltype, at, caselbl) THEN
	 AddCaseLabelRange(labels, caselbl, caselbl);
      END;
   END AddLabel;

   PROCEDURE AddRange(at1, at2: Attribute; VAR labels: Labels);
      VAR caselbl1, caselbl2: CaseLabel;
   BEGIN
      IF CaseConst(labels^.ltype, at1, caselbl1) &
	 CaseConst(labels^.ltype, at2, caselbl2) THEN
	 IF caselbl1 <= caselbl2 THEN
	    AddCaseLabelRange(labels, caselbl1, caselbl2);
	 ELSE
	    Error("bad case label range");
	 END;
      END;
   END AddRange;

   PROCEDURE CaseLabels(labels: Labels);
      (* to be called at ":" *)
   BEGIN
      IF ~errorflag THEN
	 GenCase(labels);
      END;
      INC(labels^.count);
   END CaseLabels;

   PROCEDURE CaseElse(labels: Labels);
   BEGIN
      IF ~errorflag THEN
	 GenCaseElse(labels);
      END;
   END CaseElse;

   PROCEDURE EndCase(labels: Labels);
   BEGIN
      IF ~errorflag THEN
	 GenEndCase(labels);
      END;
   END EndCase;

   PROCEDURE WhileDo(condat: Attribute);
   BEGIN
      IF (condat # NIL) & CheckOpMode(condat) THEN
	 WITH condat^ DO
	    IF (attype # NIL) & (attype # boolptr) THEN
	       Error("boolean type expected");
	    END;
	 END;
	 IF ~errorflag THEN
	    GenWhileDo(condat);
	 END;
      END;
   END WhileDo;

   PROCEDURE EndWhile;
   BEGIN
      IF ~errorflag THEN
	 GenEndWhile;
      END;
   END EndWhile;

   PROCEDURE Repeat;
   BEGIN
      IF ~errorflag THEN
	 GenRepeat;
      END;
   END Repeat;

   PROCEDURE Until(condat: Attribute);
   BEGIN
      IF (condat # NIL) & CheckOpMode(condat) THEN
	 WITH condat^ DO
	    IF (attype # NIL) & (attype # boolptr) THEN
	       Error("boolean type expected");
	    END;
	 END;
	 IF ~errorflag THEN
	    GenUntil(condat);
	 END;
      END;
   END Until;

   MODULE Loops;

      FROM Exception IMPORT Assert;
      FROM GenStmts IMPORT GenLoop, GenExitLoop, GenEndLoop;
      FROM Scan IMPORT errorflag, Error;
      EXPORT Loop, ExitLoop, EndLoop;

      VAR
	 nestlevel: CARDINAL;

      PROCEDURE Loop;
      BEGIN
	 INC(nestlevel);
	 IF ~errorflag THEN
	    GenLoop;
	 END;
      END Loop;

      PROCEDURE ExitLoop;
	 (* correct nesting is checked by CodeGen *)
      BEGIN
	 IF nestlevel = 0 THEN
	    Error("EXIT only in LOOP");
	 ELSIF ~errorflag THEN
	    GenExitLoop;
	 END;
      END ExitLoop;

      PROCEDURE EndLoop;
      BEGIN
	 Assert(nestlevel > 0);
	 DEC(nestlevel);
	 IF ~errorflag THEN
	    GenEndLoop;
	 END;
      END EndLoop;

   BEGIN
      nestlevel := 0;
   END Loops;

   PROCEDURE Return;
   BEGIN
      IF ~badprocedure THEN
	 IF procedureAt = NIL THEN
	    Error("RETURN must not be used inside of module bodies");
	 ELSIF procedureAt^.attype^.function THEN
	    Error("expression of return value expected");
	 ELSIF ~errorflag THEN
	    GenReturn;
	 END;
      END;
   END Return;

   PROCEDURE ReturnExpr(at: Attribute);
   BEGIN
      IF ~badprocedure THEN
	 IF procedureAt = NIL THEN
	    Error("RETURN must not be used inside of module bodies");
	 ELSIF ~procedureAt^.attype^.function THEN
	    Error1("%I is not a function procedure", procedureAt^.atip^.name);
	 ELSIF (at # NIL) & CheckOpMode(at) THEN
	    WITH at^ DO
	       IF ~AssignComp(procedureAt^.attype^.restype, attype,
			      at^.mode = constAt) THEN
		  Error("type must be identical to result type");
	       ELSIF ~errorflag THEN
		  GenReturnExpr(procedureAt^.attype^.restype, at);
	       END;
	    END;
	 END;
      END;
   END ReturnExpr;

   PROCEDURE WithDo(VAR at, type: Attribute);
      VAR
	 guard: Type;
	 nameofguard: Identifier;
	 base: Type;
	 ok: BOOLEAN;
   BEGIN
      guard := NIL; ok := FALSE;
      IF type # NIL THEN
	 WITH type^ DO
	    IF mode = typeAt THEN
	       IF attype # NIL THEN
		  WITH attype^ DO
		     IF (form # record) & (form # pointer) THEN
			Error1("%I must be a record or pointer type",
			   atip^.name);
		     ELSIF (form = pointer) & ~taggedptr THEN
			Error1("%I is an untagged pointer type", atip^.name);
		     ELSE
			guard := attype;
			nameofguard := atip^.name;
		     END;
		  END;
	       END;
	    ELSE
	       IF mode IN IdModes THEN
		  CondError("%I must be a type identifier", atip);
	       ELSE
		  Error("type identifier expected");
	       END;
	    END;
	 END;
	 DisposeAttribute(type);
      END;
      IF at # NIL THEN
	 WITH at^ DO
	    IF mode = varAt THEN
	       IF attype # NIL THEN
		  WITH attype^ DO
		     IF (form = record) OR (form = pointer) THEN
			IF (form = record) & (atip^.varkind # varparamV) THEN
			   Error1("%I must be a VAR-parameter", atip^.name);
			ELSIF (form = pointer) & ~taggedptr THEN
			   Error1("%I is an untagged pointer", atip^.name);
			END;
		     ELSE
			Error1("%I must be a pointer or record", atip^.name);
		     END;
		     IF guard # NIL THEN
			IF ~BaseType(guard, attype, base) THEN
			   Error2("%I cannot be of type %I",
			      atip^.name, nameofguard);
			ELSIF base = guard THEN
			   Message(warning,
			      "type guard satisfied by definition");
			ELSE
			   ok := TRUE;
			   atip^.type := guard;
			   IF ~errorflag THEN
			      GenWith(at, guard);
			   END;
			END;
		     END;
		  END;
	       END;
	    ELSE
	       IF mode IN IdModes THEN
		  CondError("%I must be a variable", atip);
	       ELSE
		  Error("variable expected");
	       END;
	    END;
	 END;
	 IF ~ok THEN DisposeAttribute(at) END;
      END;
   END WithDo;

   PROCEDURE EndWith(at: Attribute);
   BEGIN
      IF at # NIL THEN
	 WITH at^ DO
	    Assert((mode = varAt) & (attype # NIL));
	    atip^.type := attype;
	 END;
	 DisposeAttribute(at);
      END;
   END EndWith;


   (* ---------------------------------------------------------	*)
   (* procedures						*)
   (*    o InitIdentAt(procedure name) just before OpenScope	*)
   (*    o ProcedureBegin at BEGIN				*)
   (*    o ProcedureEnd at END procedure name			*)
   (* ---------------------------------------------------------	*)

   PROCEDURE ProcedureBegin(at: Attribute);
   BEGIN
      (* valid `at' not guaranteed by SymDef.OpenScope *)
      badprocedure := (at = NIL) OR (at^.mode # procAt) OR (at^.attype = NIL);
      IF ~badprocedure THEN
	 procedureAt := at;
	 GenProcEntry(at^.atip);
      END;
   END ProcedureBegin;

   PROCEDURE ProcedureEnd(at: Attribute);
   BEGIN
      IF ~badprocedure THEN
	 GenProcExit(at^.atip);
	 procedureAt := NIL;
	 badprocedure := TRUE;
      END;
   END ProcedureEnd;

   PROCEDURE ModuleHeader;
   BEGIN
      GenModuleHeader;
   END ModuleHeader;

   PROCEDURE ModuleBegin;
   BEGIN
      GenModuleEntry;
   END ModuleBegin;

   PROCEDURE ModuleEnd;
   BEGIN
      GenModuleExit;
      GenPreamble;
   END ModuleEnd;

   PROCEDURE InitCodeGen;
      (* to be called after argument scanning *)
   BEGIN
      IF ~defunit THEN
	 InitEmitCode;
      END;
   END InitCodeGen;

BEGIN
   procedureAt := NIL;
   badprocedure := TRUE;
   AddLegalOptions(CompilerOptions{charConstantsAreETHZCompatibleOpt});
END CodeGen.
