(* Ulm's Oberon Compiler
   Copyright (c) 1989 by University of Ulm, SAI, D-W-7900 Ulm, Germany
   ----------------------------------------------------------------------------
   $Id: ConstExpr.m2,v 0.3 1993/10/07 11:01:22 borchert Exp $
   ----------------------------------------------------------------------------
   $Log: ConstExpr.m2,v $
   Revision 0.3  1993/10/07  11:01:22  borchert
   bug fix: set constants like {a, b..c} were not processed correctly

   Revision 0.2  1993/06/18  15:30:16  borchert
   ConstExpr depends now on Types

   Revision 0.1  1992/07/30  10:48:01  borchert
   Initial revision

   ----------------------------------------------------------------------------
*)

IMPLEMENTATION MODULE ConstExpr; (* AFB 1/89 *)

   (* evaluation of operations with constants *)

   FROM Exception IMPORT Assert;
   FROM Lex IMPORT Symbol, Constval, String, StringLen, GetStringChar;
   FROM Machine IMPORT minlong, maxlong, minlreal, maxlreal, bitsperword;
   FROM Memory IMPORT ALLOCATE, DEALLOCATE;
   FROM Scan IMPORT Error;
   FROM Types IMPORT Cardinal, Integer, Real, Set;

   TYPE
      SymSet = SET OF Symbol;

   (* to be called after semantic checks:
      1) operands must be compatible (except IN-operator)
      2) operator must be applyable to the operands

      error messages can result from overflow
      and division by zero
   *)

   PROCEDURE Unary(opsy: Symbol; VAR cval: Constval);
      (* apply `opsy' to `cval'; result in `cval' *)
   BEGIN
      IF opsy = plus THEN RETURN END; (* identity operation *)
      WITH cval DO
	 CASE sy OF
	 | intcon:   IF intval = minlong THEN
			Error("constant out of range");
		     ELSE
			intval := -intval;
		     END;
	 | realcon, longrealcon:
		     realval := -realval;
	 | boolcon:  boolval := NOT boolval;
	 | setcon:   setval := Set{0..bitsperword-1} - setval;
	 END;
      END;
   END Unary;

   PROCEDURE Binary(opsy: Symbol; VAR left: Constval; right: Constval);
      (* apply `opsy' to `left' and `right'; result in `left' *)

      CONST
	 CmpOp = SymSet{eql..geq}; (* relational operators except IN and IS *)
	 SetConstrOp = SymSet{comma, range};

      VAR
	 result: Constval;
	 err: BOOLEAN;

      PROCEDURE StringCompare(relop: Symbol; s1, s2: String) : BOOLEAN;
	 TYPE
	    CharPointer = POINTER TO ARRAY [0..0] OF CHAR;
	 VAR
	    index: CARDINAL;		(* position in strings *)
	    cp1, cp2: CharPointer;
	    ch1, ch2: CHAR;
	    len1, len2: CARDINAL;

	 PROCEDURE ReadStrings;

	    PROCEDURE ReadString(VAR cp: CharPointer; s: String);
	       VAR
		  i: CARDINAL;
		  len: CARDINAL; (* length of `s'; > 0 *)
		  null: CHAR;
	    BEGIN
	       len := StringLen(s);
	       ALLOCATE(cp, len * SIZE(CHAR));
	       FOR i := 0 TO len-1 DO
		  (* $T- *)
		  GetStringChar(s, cp^[i]);
		  (* $T= *)
	       END;
	       (* read null byte to avoid return of 0C on following
		  GetStringChar-call
	       *)
	       GetStringChar(s, null);
	       Assert(null = 0C);
	    END ReadString;

	 BEGIN
	    ReadString(cp1, s1);
	    ReadString(cp2, s2);
	 END ReadStrings;

	 PROCEDURE DropStrings;
	 BEGIN
	    DEALLOCATE(cp1, len1 * SIZE(CHAR));
	    DEALLOCATE(cp2, len2 * SIZE(CHAR));
	 END DropStrings;

      BEGIN
	 (* fast checks and special cases *)
	 IF (relop = eql) & (s1 = s2) THEN RETURN TRUE END;
	 len1 := StringLen(s1); len2 := StringLen(s2);
	 IF len1 # len2 THEN
	    IF relop = eql THEN
	       RETURN FALSE
	    ELSIF relop = neq THEN
	       RETURN TRUE
	    END;
	 END;
	 IF (len1 = 0) & (len2 = 0) THEN
	    RETURN relop IN SymSet{eql, leq, geq};
	 ELSIF len1 = 0 THEN
	    RETURN relop IN SymSet{lst, leq, neq};
	 ELSIF len2 = 0 THEN
	    RETURN relop IN SymSet{grt, geq, neq};
	 END;

	 ReadStrings;
	 index := 0;
	 REPEAT
	    (* $T- *)
	    ch1 := cp1^[index];
	    ch2 := cp2^[index];
	    (* $T= *)
	    INC(index);
	 UNTIL (ch1 # ch2) OR (index >= len1) OR (index >= len2);
	 DropStrings;

	 IF ch1 # ch2 THEN
	    CASE relop OF
	    | lst: RETURN ch1 < ch2
	    | leq: RETURN ch1 <= ch2
	    | eql: RETURN ch1 = ch2
	    | neq: RETURN ch1 # ch2
	    | geq: RETURN ch1 >= ch2
	    | grt: RETURN ch1 > ch2
	    END;
	 ELSIF len1 = len2 THEN
	    RETURN relop IN SymSet{leq, eql, geq};
	 ELSIF len1 > len2 THEN
	    RETURN relop IN SymSet{geq, grt, neq};
	 ELSE (* len1 < len2 *)
	    RETURN relop IN SymSet{lst, leq, neq};
	 END;
      END StringCompare;

      PROCEDURE IntegerCompare(relop: Symbol; i1, i2: Integer) : BOOLEAN;
      BEGIN
	 CASE relop OF
	 | lst: RETURN i1 < i2
	 | leq: RETURN i1 <= i2
	 | eql: RETURN i1 = i2
	 | neq: RETURN i1 # i2
	 | geq: RETURN i1 >= i2
	 | grt: RETURN i1 > i2
	 END;
      END IntegerCompare;

      PROCEDURE IntegerArithmetic(op: Symbol; i1, i2: Integer) : Integer;

	 PROCEDURE AddOp(i1, i2: Integer) : Integer;
	 BEGIN
	    err := (i1 >= 0) & (i2 > 0) & (maxlong - i1 < i2) OR
		   (i1 < 0) & (i2 < 0) & (i1 < minlong-i2);
	    RETURN i1 + i2
	 END AddOp;

	 PROCEDURE SubOp(i1, i2: Integer) : Integer;
	 BEGIN
	    err := (i1 >= 0) & (i2 < 0) & (i2 < i1 - maxlong) OR
		   (i1 < -1) & (i2 > 0) & (i1 < minlong+i2);
	    RETURN i1 - i2
	 END SubOp;

	 PROCEDURE MulOp(i1, i2: Integer) : Integer;
	    VAR
	       i: Integer;
	       minus: BOOLEAN;
	       prod, c1, c2: Cardinal;
	 BEGIN
	    IF i1 > i2 THEN
	       i := i1; i1 := i2; i2 := i;
	    END;
	    IF (i1 = minlong) & (i2 # 1) THEN
	       err := TRUE; RETURN minlong
	    END;
	    minus := (i1 < 0) # (i2 < 0);

	    (* multiplication in Cardinal arithmetic *)
	    c1 := ABS(i1); c2 := ABS(i2);
	    prod := 0;
	    WHILE (c1 > 0) & ~err DO
	       IF ODD(c1) THEN
		  IF prod <= maxlong-c2 THEN
		     INC(prod, c2);
		  ELSE
		     err := TRUE;
		  END;
	       END;
	       c1 := c1 DIV 2;
	       IF c1 > 0 THEN
		  IF c2 <= maxlong THEN
		     c2 := 2 * c2;
		  ELSE
		     err := TRUE;
		  END;
	       END;
	    END;

	    IF minus THEN
	       RETURN - VAL(Integer, prod)
	    ELSE
	       RETURN prod
	    END;
	 END MulOp;

	 PROCEDURE DivOp(i1, i2: Integer) : Integer;
	    (* assumption: DIV and MOD are performed in Euler-arithmetic *)
	    (* result is in Modulo-arithmetic *)
	    VAR
	       q, r: Integer;
	 BEGIN
	    IF i2 = 0 THEN err := TRUE; RETURN 1 END;
	    q := i1 DIV i2;
	    r := i1 MOD i2;
	    IF ((i1 >= 0) = (i2 >= 0)) OR (r = 0) THEN
	       RETURN q
	    ELSE
	       RETURN q-1
	    END;
	 END DivOp;

	 PROCEDURE ModOp(i1, i2: Integer) : Integer;
	    (* assumption: DIV and MOD are performed in Euler-arithmetic *)
	    (* result is in Modulo-arithmetic *)
	    VAR
	       q, r: Integer;
	 BEGIN
	    IF i2 = 0 THEN err := TRUE; RETURN 1 END;
	    q := i1 DIV i2;
	    r := i1 MOD i2;
	    IF ((i1 >= 0) = (i2 >= 0)) OR (r = 0) THEN
	       RETURN r
	    ELSE
	       RETURN i2+r
	    END;
	 END ModOp;

      BEGIN
	 CASE op OF
	 | plus:           RETURN AddOp(i1, i2)
	 | minus:          RETURN SubOp(i1, i2)
	 | times:          RETURN MulOp(i1, i2)
	 | divSY:          RETURN DivOp(i1, i2)
	 | modSY:          RETURN ModOp(i1, i2)
	 END;
      END IntegerArithmetic;

      PROCEDURE RealCompare(relop: Symbol; r1, r2: Real) : BOOLEAN;
      BEGIN
	 CASE relop OF
	 | lst: RETURN r1 < r2
	 | leq: RETURN r1 <= r2
	 | eql: RETURN r1 = r2
	 | neq: RETURN r1 # r2
	 | geq: RETURN r1 >= r2
	 | grt: RETURN r1 > r2
	 END;
      END RealCompare;

      PROCEDURE RealArithmetic(op: Symbol; r1, r2: Real) : Real;

	 (* implementation could be more secure in
	    using the MC68861 exceptions
	 *)

	 PROCEDURE AddOp(r1, r2: Real) : Real;
	    (* assume maxlreal = - minlreal *)
	 BEGIN
	    err := (r1 >= 0.0) & (r2 > 0.0) & (maxlreal - r1 < r2) OR
		   (r1 < 0.0) & (r2 < 0.0) & (r1 < minlreal-r2);
	    RETURN r1 + r2
	 END AddOp;

	 PROCEDURE MulOp(r1, r2: Real) : Real;
	    (* NO SECURITY *)
	 BEGIN
	    RETURN r1 * r2
	 END MulOp;

	 PROCEDURE DivOp(r1, r2: Real) : Real;
	    (* NO SECURITY *)
	 BEGIN
	    IF r2 = 0.0 THEN
	       err := TRUE;
	       RETURN 1.0
	    ELSE
	       RETURN r1 / r2
	    END;
	 END DivOp;

      BEGIN
	 CASE op OF
	 | plus:  RETURN AddOp(r1, r2)
	 | minus: RETURN AddOp(r1, -r2)
	 | times: RETURN MulOp(r1, r2)
	 | slash: RETURN DivOp(r1, r2)
	 END;
      END RealArithmetic;

      PROCEDURE SetOperation(sy: Symbol; s1, s2: Set) : Set;
      BEGIN
	 CASE sy OF
	 | plus:  RETURN s1 + s2
	 | minus: RETURN s1 - s2
	 | times: RETURN s1 * s2
	 | slash: RETURN s1 / s2
	 END;
      END SetOperation;

      PROCEDURE SetInclusion(s: Set; i: Integer) : Set;
      BEGIN
	 IF (i >= 0) & (i < bitsperword) THEN
	    RETURN s + Set{ORD(i)}
	 ELSE
	    err := TRUE;
	    RETURN s
	 END;
      END SetInclusion;

      PROCEDURE SetRange(i1, i2: Integer) : Set;
      BEGIN
	 IF (i1 >= 0) & (i2 >= 0) & (i1 < bitsperword) & (i2 < bitsperword) THEN
	    RETURN Set{ORD(i1)..ORD(i2)}
	 ELSE
	    err := TRUE;
	    RETURN Set{}
	 END;
      END SetRange;

      PROCEDURE IsReal(sy: Symbol) : BOOLEAN;
      BEGIN
	 RETURN (sy = realcon) OR (sy = longrealcon)
      END IsReal;

      PROCEDURE Real(VAR cval: Constval);
	 VAR val: Integer;
      BEGIN
	 WITH cval DO
	    IF sy = intcon THEN
	       val := intval;
	       sy := realcon;
	       realval := FLOAT(val);
	    END;
	 END;
      END Real;

   BEGIN
      err := FALSE;
      IF (left.sy = intcon) & IsReal(right.sy) OR (opsy = slash) THEN
	 Real(left);
      END;
      IF IsReal(left.sy) & (right.sy = intcon) OR (opsy = slash) THEN
	 Real(right);
      END;

      CASE left.sy OF
      | stringcon:   (* any relational operator *)
		     result.sy := boolcon;
		     result.boolval := StringCompare(opsy, left.string,
							   right.string);
      | charcon:     (* any relational operator *)
		     result.sy := boolcon;
		     result.boolval := IntegerCompare(opsy, ORD(left.charval),
							    ORD(right.charval));
      | intcon:      (* arithmetic, relational operator or setconstruction *)
		     IF (opsy IN CmpOp) OR (opsy = inSY) THEN
			result.sy := boolcon;
			IF opsy = inSY THEN
			   IF (left.intval < 0) OR
			      (left.intval >= bitsperword) THEN
			      result.boolval := TRUE;
			      err := TRUE;
			   ELSE
			      result.boolval := ORD(left.intval) IN
						right.setval;
			   END;
			ELSE
			   result.boolval := IntegerCompare(opsy, left.intval,
								  right.intval);
			END;
		     ELSIF opsy IN SetConstrOp THEN
			result.sy := setcon;
			IF opsy = range THEN
			   result.setval := SetRange(left.intval, right.intval);
			ELSIF right.sy = setcon THEN
			   result.setval :=
			      SetInclusion(right.setval, left.intval);
			ELSE (* right.sy = intcon *)
			   result.setval :=
			      SetInclusion(SetInclusion(Set{}, left.intval),
					   right.intval);
			END;
		     ELSE
			result.sy := intcon;
			result.intval := IntegerArithmetic(opsy, left.intval,
								 right.intval);
		     END;
      | realcon, longrealcon: (* arithmetic or relational operator *)
		     IF opsy IN CmpOp THEN
			result.sy := boolcon;
			result.boolval := RealCompare(opsy, left.realval,
							    right.realval);
		     ELSE
			result.sy := realcon;
			result.realval := RealArithmetic(opsy, left.realval,
							       right.realval);
		     END;
      | setcon:      (* set-operator, set comparisons or
			set construction (comma) *)
		     IF (opsy = eql) OR (opsy = neq) THEN
			result.sy := boolcon;
			result.boolval :=
			   (left.setval = right.setval) = (opsy = eql);
		     ELSE
			result.sy := setcon;
			IF (opsy = comma) & (right.sy = setcon) THEN
			   opsy := plus;
			END;
			IF opsy = comma THEN
			   result.setval := SetInclusion(left.setval,
							 right.intval);
			ELSE
			   result.setval := SetOperation(opsy, left.setval,
							       right.setval);
			END;
		     END;
      | boolcon:     (* logic operator or `eql' or `neq' *)
		     result.sy := boolcon;
		     IF opsy = ampersand THEN
			result.boolval := left.boolval & right.boolval;
		     ELSIF opsy = orSY THEN
			result.boolval := left.boolval OR right.boolval;
		     ELSE
			result.boolval := (left.boolval = right.boolval) =
					  (opsy = eql);
		     END;
      | nilSY:       (* relational operator: `eql' or `neq' *)
		     result.sy := boolcon;
		     result.boolval := opsy = eql;
      END;
      left := result;
      IF err THEN
	 Error("constant out of range");
      END;
   END Binary;

END ConstExpr.
