(* Ulm's Oberon Compiler
   Copyright (c) 1989 by University of Ulm, SAI, D-W-7900 Ulm, Germany
   ----------------------------------------------------------------------------
   $Id: GenBasicOps.m2,v 0.13 1994/05/25 11:52:38 borchert Exp $
   ----------------------------------------------------------------------------
   $Log: GenBasicOps.m2,v $
   Revision 0.13  1994/05/25  11:52:38  borchert
   Convert has been extended to handle some rather odd cases correctly

   Revision 0.12  1994/03/17  10:39:29  borchert
   reg is now guaranteed and required to be an address register in
   case of addrMode

   Revision 0.11  1993/10/21  09:40:44  borchert
   bug fix: Convert generated wrong code for conversions from INTEGER
            to SYSTEM.INT16

   Revision 0.10  1993/10/03  14:55:58  borchert
   MoveBytesAt added
   Address prepares now string constants for OffsetAt

   Revision 0.9  1993/10/02  15:11:50  borchert
   SHORTINT to SYSTEM.INT16 conversion enhanced

   Revision 0.8  1993/09/27  12:44:15  borchert
   Min and Max added

   Revision 0.7  1993/06/18  15:33:52  borchert
   GenBasicOps depends now on Types

   Revision 0.6  1993/06/16  09:47:28  borchert
   SYSTEM.INT16 added

   Revision 0.5  1993/01/14  14:02:24  borchert
   ArithType fixed: ArithType wasn't able to handle procedure types

   Revision 0.4  1992/10/15  13:21:19  borchert
   bug in range check generation for dynamic arrays removed

   Revision 0.3  1992/07/31  16:03:25  borchert
   ConversionCheck added

   Revision 0.2  1992/07/31  14:38:27  borchert
   RangeCheck & DynArrayCheck added

   Revision 0.1  1992/07/30  10:48:15  borchert
   Initial revision

   ----------------------------------------------------------------------------
*)

IMPLEMENTATION MODULE GenBasicOps; (* AFB 4/89 *)

   (* generate basic operations for common use of Gen-modules;
      every procedure may generate code and is authorized to
      modify `at';
      `at' and everything in connection with `at' must be valid
      (i.e. Scan.errorflag must be FALSE)
   *)

   FROM Attributes IMPORT ArithmeticType, Attribute, Reg, AtMode, base, RegSet,
      top, GetLabel, Label, NewAttribute, DisposeAttribute, TestType, AtModeSet;
   FROM EmitCode IMPORT EmitLabel, StrEmit, StrEmit1, StrEmit2,
      Emit, Emit1, Emit2, Emit3;
   FROM Exception IMPORT Assert;
   FROM Lex IMPORT Symbol, String, GetStringNumber, GetStringChar;
   FROM Machine IMPORT bitsperword, procmarkspace, oneword, minshort, maxshort,
      oneshort, oneint16;
   FROM Mnemonics IMPORT Mnemonic;
   FROM RegMan IMPORT GetReg, GetFloatReg, GetAddrReg, FreeReg;
   FROM Scan IMPORT ErrorType, Error, Message;
   FROM StackMan IMPORT StackFree;
   FROM Standard IMPORT charptr, int16ptr, intptr, realptr, longrealptr;
   FROM SymTab IMPORT Numeric, Form, FormSet, Type, Size, global, Level, Ident;
   FROM Types IMPORT Cardinal, Integer;

   VAR
      conversionWarningGiven: BOOLEAN;

   PROCEDURE Address(at: Attribute);
      (* convert `at' and generate code such that
	 `at' is a valid addressing mode for the MOVE-instruction
	 and can be given as %a-argument to Emit;
	 `at.mode' must be a leaf-mode, i.e. not unaryAt,...
      *)
      VAR
	 indexReg: Reg;
	 leveldiff: CARDINAL;
	 ip: Ident;
	 string: String;

   BEGIN
      WITH at^ DO
	 IF mode = varAt THEN
	    ip := atip;
	    WITH ip^ DO
	       Assert(Level() >= plevel);
	       IF plevel = global THEN
		  mode := indexMode;
		  reg := illegal;
		  tagged := FALSE;
		  labelip := ip;
		  rlabel.ok := FALSE;
		  addr := 0;
		  xreg := illegal;
		  tmp := FALSE;
	       ELSE
		  leveldiff := Level() - plevel;
		  IF leveldiff > 0 THEN
		     IF leveldiff > 1 THEN
			GetAddrReg(indexReg);
			Emit3(MOVE, "%L%(d,r),%r",
				    procmarkspace, base, indexReg);
			WHILE leveldiff > 2 DO
			   Emit3(MOVE, "%L%(d,r),%r",
				       procmarkspace, indexReg, indexReg);
			   DEC(leveldiff);
			END;
			ccat := NIL;
		     ELSE
			indexReg := base;
		     END;
		     mode := memIndexMode;
		     tagged := FALSE;
		     reg := indexReg;
		     labelip := NIL;
		     rlabel.ok := FALSE;
		     addr := procmarkspace;
		     od := offset;
		     xreg := illegal;
		     tmp := FALSE;
		     IF indirect THEN
			DereferenceAt(at);
		     END;
		  ELSE
		     IF indirect THEN
			mode := memIndexMode;
			post := TRUE;
			od := 0;
		     ELSE
			mode := indexMode;
		     END;
		     tagged := FALSE;
		     reg := base;
		     labelip := NIL;
		     rlabel.ok := FALSE;
		     addr := offset;
		     xreg := illegal;
		     scale := 0;
		     tmp := FALSE;
		  END;
	       END;
	    END;
	 ELSIF (mode = constAt) & (cval.sy = stringcon) THEN
	    string := cval.string;
	    mode := indexMode;
	    reg := illegal;
	    tagged := FALSE;
	    labelip := NIL;
	    addr := 0;
	    xreg := illegal;
	    scale := 0;
	    tmp := FALSE;
	    rlabel.ok := TRUE;
	    rlabel.head := 'S';
	    rlabel.n1 := 0;
	    rlabel.n2 := GetStringNumber(string);
	 END;
      END;
   END Address;

   PROCEDURE DynArray(at: Attribute);
      (* prepare `at' for access of the array AND the dope vector
	 at^.mode must be varAt and is modified to indexMode or memIndexMode;
	 then Dereference(at)            accesses the array
	      Offset(at, dim * oneword)  accesses LEN(array, dim)
      *)
   BEGIN
      WITH at^ DO
	 WITH atip^ DO
	    Assert((mode = varAt) & indirect);
	    indirect := FALSE;
	    Address(at);
	    indirect := TRUE;
	 END;
      END;
   END DynArray;

   PROCEDURE LoadDynArraySize(varp: Ident; sizeReg: Reg);
      (* load size of `varp' into `sizeReg' (data register);
	 condition codes are set
      *)
      VAR
	 at: Attribute;

   BEGIN
      StrEmit1("%*calculate size of %n", varp^.name);
      NewAttribute(at);
      WITH at^ DO
	 mode := varAt;
	 atip := varp;
	 attype := varp^.type;
      END;
      DynArray(at);
      OffsetAt(at, oneword);
      CalcDynArraySize(varp^.type, at, sizeReg);
      ReturnAt(at);
      ccat := NIL;
   END LoadDynArraySize;

   PROCEDURE CalcDynArraySize(type: Type; VAR lenat: Attribute; sizeReg: Reg);
      (* `type' is the type of an dynamic array;
	 `lenat' addresses the next length component;
		 (not released by CalcDynArraySize)
	 the size is calculated and stored into `sizeReg';
	 the condition codes are set according to the result
	 -- common part of LoadDynArraySize and procedures which
	    want to calculate the element size of a multi-dimensional
	    dynamic array
      *)
      VAR
	 dim: INTEGER;
	 align: BOOLEAN;
	 elem: Type;
   BEGIN
      Emit2(MOVE, "%L%a,%r", lenat, sizeReg);
      elem := type^.element;
      dim := 1;
      WHILE (elem^.form = array) & (elem^.dyn) DO
	 INC(dim);
	 elem := elem^.element;
      END;
      align := elem^.size < oneword;
      IF dim = 1 THEN
	 IF align THEN
	    AlignReg(sizeReg);
	 END;
      ELSE
	 IF align THEN
	    DEC(dim); (* different code for last dimension *)
	 END;
	 WHILE dim > 1 DO
	    OffsetAt(lenat, oneword);
	    Emit2(MULS, "%L%a,%r", lenat, sizeReg);
	    DEC(dim);
	 END;
	 IF align THEN
	    (* multiplication with last dimension *)
	    OffsetAt(lenat, oneword);
	    Emit2(MOVE, "%L%a,%r", lenat, d0);
	    AlignReg(d0);
	    Emit2(MULS, "%L%r,%r", d0, sizeReg);
	 END;
      END;
      (* multiplication with size of non-dynamic element type *)
      ConstMulReg(longAT, sizeReg, elem^.size);
      ccat := NIL;
   END CalcDynArraySize;

   PROCEDURE AlignReg(reg: Reg);
      (* align contents of `reg' which must be a data register *)
      CONST
	 mask = -4; (* 11..1100 *)
   BEGIN
      Emit2(ADD, "%L%C,%r", oneword-1, reg);
      Emit2(ANDop, "%L%C,%r", mask, reg);
      ccat := NIL;
   END AlignReg;

   PROCEDURE Min(reg1, reg2: Reg);
      (* return minimum of reg1 and reg2 in reg1;
	 algorithm taken from Henry Massalin
      *)
   BEGIN
      Emit2(SUB, "%L%r,%r", reg2, reg1);
      Emit2(SUBX, "%L%r,%r", d0, d0);
      Emit2(ANDop, "%L%r,%r", d0, reg1);
      Emit2(ADD, "%L%r,%r", reg2, reg1);
   END Min;

   PROCEDURE Max(reg1, reg2: Reg);
      (* return unsigned maximum of reg1 and reg2 in reg1 *)
   BEGIN
      Emit2(SUB, "%L%r,%r", reg1, reg2);
      Emit2(SUBX, "%L%r,%r", d0, d0);
      Emit2(ORop, "%L%r,%r", reg2, d0);
      Emit2(ADDX, "%L%r,%r", d0, reg1);
   END Max;

   PROCEDURE Load(at: Attribute);
      (* load the corresponding value of `at' into a data
	 (or floating point) register.
	 `at' must not be a record or an array
      *)
      VAR
	 loadReg: Reg;
   BEGIN
      WITH at^ DO
	 IF (mode # regMode) & (mode # floatRegMode) OR
	    (mode = regMode) & ~(reg IN RegSet{d0..d7}) THEN
	    IF (attype = realptr) OR (attype = longrealptr) THEN
	       GetFloatReg(loadReg);
	    ELSE
	       GetReg(loadReg);
	    END;
	    LoadReg(at, loadReg);
	 END;
      END;
   END Load;

   PROCEDURE LoadA(at: Attribute);
      (* like Load; but destination is an address register *)
      VAR
	 loadReg: Reg;
   BEGIN
      WITH at^ DO
	 IF (mode # regMode) & (mode # floatRegMode) OR
	    (mode = regMode) & ~(reg IN RegSet{a0..a7}) THEN
	    GetAddrReg(loadReg);
	    LoadReg(at, loadReg);
	 END;
      END;
   END LoadA;

   PROCEDURE LoadReg(at: Attribute; r: Reg);
      (* like Load with destination register `r' *)
      VAR
	 label: Label;
	 const: BOOLEAN;
   BEGIN
      WITH at^ DO
	 IF mode = condMode THEN
	    Assert(at = ccat);
	    IF ~tlabel.ok & ~flabel.ok THEN
	       SetBool(at, r);
	    ELSIF tlabel.ok THEN
	       GetLabel(label);
	       GenTest(test, atype, tlabel);
	       IF flabel.ok THEN
		  EmitLabel(flabel);
	       END;
	       Emit1(CLR, "%B%r", r);
	       Emit1(BRA, "%l", label);
	       EmitLabel(tlabel);
	       Emit2(MOVE, "%B%I,%r", 1, r);
	       EmitLabel(label);
	    ELSE
	       GetLabel(label);
	       InvertTest(test);
	       GenTest(test, atype, flabel);
	       IF tlabel.ok THEN
		  EmitLabel(tlabel);
	       END;
	       Emit2(MOVE, "%B%I,%r", 1, r);
	       Emit1(BRA, "%l", label);
	       EmitLabel(flabel);
	       Emit1(CLR, "%B%r", r);
	       EmitLabel(label);
	    END;
	    mode := regMode;
	    reg := r;
	    extended := FALSE;
	 ELSE
	    IF (mode # regMode) & (mode # floatRegMode) OR (reg # r) THEN
	       Address(at);
	       const := mode = constAt;
	       IF r IN RegSet{fp0..fp7} THEN
		  Emit2(FMOVE, "%A%a,%r", at, r); ccat := NIL;
	       ELSIF r IN RegSet{a0..a7} THEN
		  Emit2(MOVE, "%L%a,%r", at, r); ccat := NIL;
	       ELSIF const THEN
		  Emit2(MOVE, "%L%a,%r", at, r); ccat := at;
	       ELSE
		  Assert(attype^.size <= oneword);
		  Emit2(MOVE, "%A%a,%r", at, r); ccat := at;
	       END;
	       ReleaseAt(at);
	       IF r IN RegSet{fp0..fp7} THEN
		  mode := floatRegMode;
	       ELSE
		  mode := regMode;
		  extended := (attype^.size = oneword) OR const;
	       END;
	       reg := r;
	    END;
	 END;
      END;
   END LoadReg;

   PROCEDURE LoadAddr(at: Attribute);
      (* load address of `at' into a address register and
	 convert mode of `at' into addrMode
      *)
      VAR
	 addrReg: Reg;
   BEGIN
      WITH at^ DO
	 IF mode # addrMode THEN
	    addrReg := illegal;
	    IF (mode = indexMode) OR (mode = memIndexMode) THEN
	       IF reg IN RegSet{a0..a5} THEN
		  addrReg := reg;
	       ELSIF xreg IN RegSet{a0..a5} THEN
		  addrReg := xreg;
	       END;
	    END;
	    IF addrReg = illegal THEN
	       GetAddrReg(addrReg);
	    END;
	    LoadAddrReg(at, addrReg);
	 END;
      END;
   END LoadAddr;

   PROCEDURE LoadAddrReg(at: Attribute; r: Reg);
      (* like LoadAddr with destination register `r';
	 `r' must be an address register
      *)
   BEGIN
      Assert(r IN RegSet{a0..a7});
      WITH at^ DO
	 IF (mode # addrMode) OR (reg # r) THEN
	    Address(at);
	    Emit2(LEA, "%L%a,%r", at, r); ccat := NIL;
	    RestrictedRelease(at, r);
	    mode := addrMode;
	    reg := r;
	 END;
      END;
   END LoadAddrReg;

   PROCEDURE LoadAndExtend(at: Attribute);
      (* Load(at); if `at' is byte-sized or int16-sized
	 it is extended to an INTEGER
      *)
   BEGIN
      Load(at);
      WITH at^ DO
	 Assert(mode = regMode);
	 IF ~extended THEN
	    IF at^.attype = int16ptr THEN
	       Emit1(EXT, "%L%r", reg);
	    ELSE
	       IF at^.attype = charptr THEN
		  Emit2(ANDop, "%L%C,%a", 0FFH, at);
	       ELSE
		  Emit1(EXTB, "%L%r", reg);
	       END;
	    END;
	    ccat := at;
	    extended := TRUE;
	 END;
      END;
   END LoadAndExtend;

   PROCEDURE LoadCond(at: Attribute);
      (* set condition codes of `at';
	 at is in condMode afterwards and ccat equals at
      *)
   BEGIN
      WITH at^ DO
	 Assert((mode # condMode) OR (ccat = at));
	 IF mode # condMode THEN
	    IF ccat # at THEN
	       Emit1(TST, "%A%a", at);
	       ccat := at;
	    END;
	    ReleaseAt(at);
	    mode := condMode;
	    test := ne;
	    atype := ArithType(at^.attype);
	    tlabel.ok := FALSE;
	    flabel.ok := FALSE;
	 END;
      END;
   END LoadCond;

   PROCEDURE Convert(at: Attribute; dtype: Type);
      (* convert `at' to type `dtype'; at^.attype becomes dtype;
	 `at' may be loaded
      *)
      CONST
	 Integers = FormSet{shortint..longint};
      VAR
	 form1, form2: Form;
	 loadReg: Reg;
	 ch, null: CHAR;
	 setConversion: BOOLEAN;
   BEGIN
      WITH at^ DO
	 form1 := attype^.form;
	 form2 := dtype^.form;
	 IF form1 = longint THEN
	    form1 := integer;
	 END;
	 IF form2 = longint THEN
	    form2 := integer;
	 END;

	 IF form1 IN FormSet{char, byte} THEN
	    form1 := shortint;
	 END;
	 IF form2 IN FormSet{char, byte} THEN
	    form2 := shortint;
	 END;

	 (* sets have the same size like integers;
	    they may be converted down to smaller types --
	    in this case the effect is fairly system-dependent
	 *)
	 setConversion := FALSE;
	 IF (form1 IN Numeric) & (form2 = set) THEN
	    form2 := integer; setConversion := TRUE;
	 END;
	 IF (form1 = set) & (form2 IN Numeric) THEN
	    form1 := integer; setConversion := TRUE;
	 END;
	 IF setConversion & (form1 > form2) & ~conversionWarningGiven THEN
	    Message(warning, "this is an unportable conversion down to a smaller type");
	    Message(warning, "this compiler takes the less significant bits");
	    Message(warning, "this warning is only given once");
	    conversionWarningGiven := TRUE;
	 END;

	 IF (form2 = shortint) & (mode = constAt) & (cval.sy = stringcon) THEN
	    WITH cval DO
	       GetStringChar(string, ch);
	       GetStringChar(string, null); (* null-byte *) Assert(null = 0C);
	       IF dtype^.form IN FormSet{char} THEN
		  sy := charcon;
		  charval := ch;
	       ELSE
		  sy := intcon;
		  intval := ORD(ch);
	       END;
	    END;
	 ELSIF form1 # form2 THEN
	    IF (form1 IN Numeric) & (form2 IN Numeric) THEN
	       IF (form1 IN FormSet{shortint, int16, integer}) &
		  (form2 IN FormSet{real, longreal}) THEN
		  GetFloatReg(loadReg);
		  Emit2(FMOVE, "%A%a,%r", at, loadReg); ccat := NIL;
		  ReleaseAt(at);
		  mode := floatRegMode;
		  reg := loadReg;
	       ELSIF (form2 IN FormSet{shortint, int16, integer}) &
		     (form1 IN FormSet{real, longreal}) THEN
		  GetReg(loadReg);
		  Load(at);
		  Emit2(FMOVE, "%L%a,%r", at, loadReg); ccat := NIL;
		  ReleaseAt(at);
		  mode := regMode;
		  reg := loadReg;
		  extended := TRUE;
	       ELSIF ((form1 = shortint) OR (form1 = int16)) &
		     (form2 = integer) THEN
		  IF mode # constAt THEN
		     LoadAndExtend(at);
		  END;
	       ELSIF (form1 = shortint) & (form2 = int16) THEN
		  IF mode # constAt THEN
		     Load(at);
		     IF ~extended THEN
			Emit1(EXT, "%W%a", at);
		     END;
		  END;
	       ELSIF (form1 IN FormSet{integer, longint}) & (form2 = int16) THEN
		  IF mode # constAt THEN
		     IF mode IN AtModeSet{varAt, addrMode,
					  indexMode, memIndexMode} THEN
			(* we assume big endian here *)
			OffsetAt(at, oneword - oneint16);
		     ELSE
			Load(at);
		     END;
		  END;
	       ELSIF (form1 IN FormSet{int16, integer}) &
		     (form2 = shortint) THEN
		  IF mode # constAt THEN
		     IF mode IN AtModeSet{varAt, addrMode,
					  indexMode, memIndexMode} THEN
			(* we assume big endian here *)
			IF form1 = integer THEN
			   OffsetAt(at, oneword - oneshort);
			ELSE
			   OffsetAt(at, oneint16 - oneshort);
			END;
		     ELSE
			Load(at);
		     END;
		  END;
	       ELSIF FormSet{form1, form2} <= FormSet{real, longreal} THEN
		  Load(at);
	       END;
	    END;
	 END;
	 attype := dtype;
      END;
   END Convert;

   PROCEDURE DereferenceAt(at: Attribute);
      (* dereferencing of `at'; like '^'-operator *)
   BEGIN
      WITH at^ DO
	 Address(at);
	 CASE mode OF
	 | regMode:        IF reg IN RegSet{d0..d7} THEN
			      mode := indexMode;
			      addr := 0; labelip := NIL; rlabel.ok := FALSE;
			      tmp := FALSE; tagged := FALSE;
			      xreg := reg; scale := 1; reg := illegal;
			   ELSE
			      mode := addrMode;
			   END;
	 | addrMode:       mode := memIndexMode;
			   addr := 0; labelip := NIL; od := 0; tmp := FALSE;
			   rlabel.ok := FALSE; tagged := FALSE;
			   IF reg IN RegSet{d0..d7} THEN
			      xreg := reg; scale := 1;
			      reg := illegal;
			      post := FALSE;
			   ELSE
			      xreg := illegal;
			      post := TRUE;
			   END;
	 | indexMode:      mode := memIndexMode; od := 0;
			   post := xreg = illegal;
	 | memIndexMode:   LoadA(at); mode := addrMode;
	 ELSE
	    Assert(FALSE);
	 END;
      END;
   END DereferenceAt;

   PROCEDURE OffsetAt(at: Attribute; offset: Size);
      (* `at' must have an addressable addressing mode (i.e. not regMode);
	 the address described by `at' is then incremented by `offset';
	 `offset' may be negative
      *)
   BEGIN
      WITH at^ DO
	 Address(at);
	 CASE mode OF
	 | addrMode:       mode := indexMode;
			   labelip := NIL; addr := offset; tmp := FALSE;
			   rlabel.ok := FALSE;
			   IF reg IN RegSet{a0..a7} THEN
			      xreg := illegal;
			   ELSE
			      xreg := reg; scale := 1;
			      reg := illegal;
			   END;
	 | indexMode:      Assert(~tmp); INC(addr, offset);
	 | memIndexMode:   INC(od, offset);
	 ELSE
	    Assert(FALSE);
	 END;
      END;
   END OffsetAt;

   PROCEDURE IndexAtReg(at: Attribute; indexreg: Reg; scalefactor: CARDINAL);
      (* generate []-operation for `at' with index in `indexreg'
	 `indexreg' is released afterwards
      *)
      CONST
	 scales = {1, 2, 4, 8};
	 maxscale = 8;
      VAR
	 addrReg: Reg;

      PROCEDURE AddIndex;
	 (* add `indexreg' to `xreg' and release `indexreg' *)
      BEGIN
	 WITH at^ DO
	    IF scale # scalefactor THEN
	       IF (scale >= scalefactor) & (scale MOD scalefactor = 0) THEN
		  ConstMulReg(intAT, xreg, scale DIV scalefactor);
		  scale := scalefactor;
	       ELSE
		  ConstMulReg(intAT, xreg, scale);
		  ConstMulReg(intAT, indexreg, scalefactor);
		  scale := 1;
	       END;
	    END;
	    Emit2(ADD, "%L%r,%r", indexreg, xreg); ccat := NIL;
	    FreeReg(indexreg);
	 END;
      END AddIndex;

   BEGIN
      WITH at^ DO
	 Address(at);
	 IF (mode = memIndexMode) & ~post THEN
	    LoadAddr(at);
	 END;
	 CASE mode OF
	 | addrMode:       mode := indexMode;
			   labelip := NIL; addr := 0; tmp := FALSE;
			   rlabel.ok := FALSE;
			   xreg := indexreg;
			   IF (xreg IN RegSet{d0..d7}) &
			      (reg IN RegSet{d0..d7}) THEN
			      GetAddrReg(addrReg);
			      Emit2(MOVE, "%L%r,%r", reg, addrReg); ccat := NIL;
			      FreeReg(reg); reg := addrReg;
			   ELSIF reg IN RegSet{d0..d7} THEN
			      IF scalefactor = 1 THEN
				 xreg := reg;
				 reg := indexreg;
			      ELSE
				 GetAddrReg(addrReg);
				 Emit2(MOVE, "%L%r,%r", reg, addrReg);
				 ccat := NIL;
				 FreeReg(reg); reg := addrReg;
			      END;
			   END;
			   scale := scalefactor;
	 | indexMode:      IF xreg = illegal THEN
			      scale := scalefactor;
			      xreg := indexreg;
			   ELSE
			      AddIndex;
			   END;
	 | memIndexMode:   (* post = TRUE *)
			   IF xreg = illegal THEN
			      scale := scalefactor;
			      xreg := indexreg;
			   ELSE
			      AddIndex;
			   END;
	 ELSE
	    Assert(FALSE);
	 END;
	 Assert(xreg # illegal);
	 IF (scale > maxscale) OR ~(scale IN scales) THEN
	    ConstMulReg(intAT, xreg, scalefactor);
	    scale := 1;
	 END;
      END;
   END IndexAtReg;

   PROCEDURE ReleaseAt(at: Attribute);
      (* release anything (registers and stack reservations) of `at' *)
   BEGIN
      RestrictedRelease(at, illegal);
   END ReleaseAt;

   PROCEDURE ReleaseStackRes(at: Attribute);
      (* release stack reservation of `at' if there is one *)
   BEGIN
      WITH at^ DO
	 IF (mode = indexMode) OR (mode = memIndexMode) THEN
	    IF tmp THEN
	       Assert(reg = base);
	       IF (attype = realptr) OR
		  (attype = longrealptr) THEN
		  StackFree(addr, attype^.size);
	       ELSE
		  StackFree(addr, oneword);
	       END;
	       tmp := FALSE;
	    END;
	 END;
      END;
   END ReleaseStackRes;

   PROCEDURE RestrictedRelease(at: Attribute; r: Reg);
      (* release `at' but not `r' *)

      PROCEDURE ReleaseReg(VAR reg: Reg);
      BEGIN
	 IF ~(reg IN RegSet{r, base, top, d0, fp0, illegal}) THEN
	    FreeReg(reg); reg := illegal;
	 END;
      END ReleaseReg;

   BEGIN
      WITH at^ DO
	 CASE mode OF
	 | floatRegMode,
	   regMode, addrMode:       ReleaseReg(reg);
	 | indexMode, memIndexMode: ReleaseStackRes(at);
	                            ReleaseReg(reg); ReleaseReg(xreg);
	 ELSE
	 END;
      END;
   END RestrictedRelease;

   PROCEDURE ReturnAt(VAR at: Attribute);
      (* release anything (registers and stack reservations) of `at' and
	 dispose `at'
      *)
   BEGIN
      RestrictedRelease(at, illegal);
      DisposeAttribute(at);
   END ReturnAt;

   PROCEDURE Shift(at: ArithmeticType;
		   shiftinstr: Mnemonic; dx: Reg; count: CARDINAL);
      CONST
	 maxshift = 8; (* maximal shift count for [AL]S[LR] &data,dx *)
      VAR
	 ax, shiftreg: Reg;
   BEGIN
      IF dx IN RegSet{a0..a7} THEN
	 ax := dx;
	 dx := d0;
	 Emit2(EXG, "%r,%r", dx, ax);
      ELSE
	 ax := illegal;
      END;
      IF count <= maxshift THEN
	 IF at = shortAT THEN
	    Emit2(shiftinstr, "%B%C,%r", count, dx);
	 ELSIF at = int16AT THEN
	    Emit2(shiftinstr, "%W%C,%r", count, dx);
	 ELSE
	    Emit2(shiftinstr, "%L%C,%r", count, dx);
	 END;
      ELSIF count <= 2*maxshift THEN
	 IF at = int16AT THEN
	    Emit2(shiftinstr, "%W%C,%r", maxshift, dx);
	    Emit2(shiftinstr, "%W%C,%r", count-maxshift, dx);
	 ELSE
	    Emit2(shiftinstr, "%L%C,%r", maxshift, dx);
	    Emit2(shiftinstr, "%L%C,%r", count-maxshift, dx);
	 END;
      ELSE
	 IF dx # d0 THEN
	    shiftreg := d0;
	 ELSE
	    GetReg(shiftreg);
	 END;
	 Emit2(MOVE, "%L%C,%r", count, shiftreg);
	 Emit2(shiftinstr, "%L%r,%r", shiftreg, dx);
	 IF shiftreg # d0 THEN
	    FreeReg(shiftreg);
	 END;
      END;
      IF ax # illegal THEN
	 Emit2(EXG, "%r,%r", dx, ax);
      END;
   END Shift;

   PROCEDURE GetPower(VAR power: CARDINAL; VAR mask: Cardinal;
		      val: Cardinal) : BOOLEAN;
   BEGIN
      mask := 1;
      FOR power := 1 TO bitsperword-1 DO
         mask := mask * 2;
         IF mask = val THEN RETURN TRUE END;
      END;
      RETURN FALSE
   END GetPower;

   PROCEDURE InitAttributes(VAR at1, at2: Attribute;
			    r: Reg; value: Integer);
   BEGIN
      NewAttribute(at1);
      WITH at1^ DO
	 mode := regMode;
	 reg := r;
	 attype := intptr;
	 extended := TRUE;
      END;
      NewAttribute(at2);
      WITH at2^ DO
	 mode := constAt;
	 attype := intptr;
	 atip := NIL;
	 cval.sy := intcon;
	 cval.intval := value;
      END;
   END InitAttributes;

   PROCEDURE ConstMulReg(at: ArithmeticType; r: Reg; value: Integer);
      VAR
	 power: CARDINAL;
	 mask: Cardinal;
	 at1, at2: Attribute;

      PROCEDURE Neg;
      BEGIN
	 IF at = shortAT THEN
	    Emit1(NEG, "%B%r", r);
	 ELSIF at = int16AT THEN
	    Emit1(NEG, "%W%r", r);
	 ELSE
	    Emit1(NEG, "%L%r", r);
	 END;
      END Neg;

   BEGIN
      IF value = 0 THEN
	 Emit1(CLR, "%L%r", r);
      ELSIF value = -1 THEN
	 Neg;
      ELSIF value # 1 THEN
	 IF GetPower(power, mask, ABS(value)) THEN
	    IF value < 0 THEN
	       Neg;
	    END;
	    Shift(at, ASL, r, power);
	 ELSE
	    InitAttributes(at1, at2, r, value);
	    Mult(at, at1, at2);
	    DisposeAttribute(at1); DisposeAttribute(at2);
	 END;
      END;
      ccat := NIL;
   END ConstMulReg;

   PROCEDURE ConstDivReg(at: ArithmeticType; r: Reg; value: Integer);
      VAR
	 power: CARDINAL;
	 mask: Cardinal;
	 at1, at2: Attribute;

      PROCEDURE Neg;
      BEGIN
	 IF at = shortAT THEN
	    Emit1(NEG, "%B%r", r);
	 ELSIF at = int16AT THEN
	    Emit1(NEG, "%W%r", r);
	 ELSE
	    Emit1(NEG, "%L%r", r);
	 END;
      END Neg;

   BEGIN
      IF value = 0 THEN
	 Error("division by zero");
      ELSIF value = -1 THEN
	 Neg;
      ELSIF value # 1 THEN
	 IF GetPower(power, mask, ABS(value)) THEN
	    IF value < 0 THEN
	       Neg;
	    END;
	    Shift(at, ASR, r, power);
	 ELSE
	    InitAttributes(at1, at2, r, value);
	    Div(at, at1, at2);
	    DisposeAttribute(at1); DisposeAttribute(at2);
	 END;
      END;
      ccat := NIL;
   END ConstDivReg;

   PROCEDURE ConstModReg(at: ArithmeticType; r: Reg; value: Integer);
      VAR
	 power: CARDINAL;
	 mask: Cardinal;
	 at1, at2: Attribute;

      PROCEDURE Neg;
      BEGIN
	 IF at = shortAT THEN
	    Emit1(NEG, "%B%r", r);
	 ELSIF at = int16AT THEN
	    Emit1(NEG, "%W%r", r);
	 ELSE
	    Emit1(NEG, "%L%r", r);
	 END;
      END Neg;

   BEGIN
      IF value = 0 THEN
	 Error("division by zero");
      ELSIF ABS(value) = 1 THEN
	 Emit1(CLR, "%L%r", r);
      ELSE
	 IF GetPower(power, mask, ABS(value)) THEN
	    IF value < 0 THEN
	       Neg;
	    END;
	    Emit2(ANDop, "%L%C,%r", mask-1, r);
	    IF value < 0 THEN
	       Neg;
	    END;
	 ELSE
	    InitAttributes(at1, at2, r, value);
	    Mod(at, at1, at2);
	    DisposeAttribute(at1); DisposeAttribute(at2);
	 END;
      END;
      ccat := NIL;
   END ConstModReg;

   PROCEDURE Mult(at: ArithmeticType; desat, opat: Attribute);
      (* generate code for `desat' := `desat' * `opat'
	 and release `opat'
	 `desat' is in regMode afterwards
      *)
      VAR
	 desreg: Reg;
   BEGIN
      LoadAndExtend(desat); desreg := desat^.reg;
      WITH opat^ DO
	 IF (at = shortAT) OR (at = int16AT) OR
	       (mode = regMode) & (reg IN RegSet{a0..a7}) THEN
	    LoadAndExtend(opat);
	 END;
	 Emit2(MULS, "%L%a,%r", opat, desreg);
	 ReturnAt(opat);
      END;
      ccat := desat;
   END Mult;

   PROCEDURE DivMod(at: ArithmeticType; div: BOOLEAN; desat, opat: Attribute);
      (* generate code for `desat' := `desat' DIV `opat'
	 and release `opat'
	 desat^.reg is not changed if desat is in regMode
	 `desat' is in regMode afterwards

	 the Motorola 68020 hardware implements DIV and MOD
	 in Euler arithmetic:

	    ((-x) DIV y) = (x DIV (-y)) = -(x DIV y);

	 but Oberon requires Modulo arithmetic:

	    (0 <= (x MOD y) < y) OR (y < (x MOD y) <= 0)

	 If Q, R are the quotient and the remainder in Euler arithmetic
	 then q, r of Modulo arithmetic can be calculated efficiently
	 as follows:

	    IF ((x >= 0) = (y >= 0)) OR (R = 0) THEN
	       (* no differences *)
	       q := Q;
	       r := R;
	    ELSE
	       q := Q-1;
	       r := R+y; (* y: right operand *)
	    END;
      *)
      VAR
	 desreg: Reg;
	 quotient, remainder: Reg;
	 label: Label;
   BEGIN
      LoadAndExtend(desat); desreg := desat^.reg;
      WITH opat^ DO
	 LoadAndExtend(opat);
	 quotient := desreg;
	 Emit2(MOVE, "%L%r,%r", desreg, d0);
	 GetReg(remainder);
	 Emit3(DIVSL, "%L%a,%:r", opat, remainder, quotient);

	 GetLabel(label);
	 Emit2(EOR, "%L%a,%r", opat, d0);
	 Emit1(BPL, "%l", label);
	 Emit1(TST, "%L%r", remainder);
	 Emit1(BEQ, "%l", label);
	 IF div THEN
	    Emit2(SUB, "%L%C,%r", 1, desreg);
	 ELSE
	    Emit2(ADD, "%L%a,%r", opat, remainder);
	 END;
	 EmitLabel(label);
	 IF ~div THEN
	    Emit2(MOVE, "%L%r,%r", remainder, desreg);
	 END;
	 FreeReg(remainder);

	 ReturnAt(opat);
      END;
      ccat := desat;
   END DivMod;

   PROCEDURE Div(at: ArithmeticType; desat, opat: Attribute);
      (* generate code for `desat' := `desat' DIV `opat'
	 and release `opat'
	 `desat' is in regMode afterwards
      *)
   BEGIN
      DivMod(at, (* div = *) TRUE, desat, opat);
   END Div;

   PROCEDURE Mod(at: ArithmeticType; desat, opat: Attribute);
      (* generate code for `desat' := `desat' MOD `opat'
	 and release `opat'
	 `desat' is in regMode afterwards
      *)
   BEGIN
      DivMod(at, (* div = *) FALSE, desat, opat);
   END Mod;


   PROCEDURE ArithType(type: Type) : ArithmeticType;
   BEGIN
      WITH type^ DO
	 CASE form OF
	 | shortint:       RETURN shortAT
	 | int16:          RETURN int16AT
	 | integer:        RETURN intAT
	 | longint:        RETURN longAT
	 | real:           RETURN floatAT
	 | longreal:       RETURN longfloatAT
	 | boolean:        RETURN logAT
	 | char:           RETURN logAT
	 | set:            RETURN bitAT
	 | byte:           RETURN shortAT
	 | array:          RETURN logAT
	 | pointer:        RETURN intAT
	 | proceduretype:  RETURN longAT
	 END;
      END;
   END ArithType;

   PROCEDURE InvertTest(VAR t: TestType);
      (* invert test type (NOT), e.g. `le' becomes `gt' *)
   BEGIN
      CASE t OF
      | lt: t := ge;
      | le: t := gt;
      | eq: t := ne;
      | ne: t := eq;
      | ge: t := lt;
      | gt: t := le;
      END;
   END InvertTest;

   PROCEDURE ReverseTest(VAR t: TestType);
      (* reverse test type (exchange of operands), e.g. `le' becomes `ge' *)
   BEGIN
      CASE t OF
      | lt: t := gt;
      | le: t := ge;
      | eq, ne: (* ok *)
      | ge: t := le;
      | gt: t := lt;
      END;
   END ReverseTest;

   PROCEDURE GenTest(t: TestType; atype: ArithmeticType; dest: Label);
      (* condition codes are set; generate code for branching
	 to `dest' if `t' is true, i.e. "bcc test,dest"
      *)
      VAR
	 bm: Mnemonic;
   BEGIN
      IF (atype = floatAT) OR (atype = longfloatAT) THEN
	 (* exception on unordered! *)
	 CASE t OF
	 | lt: bm := FBLT;
	 | le: bm := FBLE;
	 | eq: bm := FBEQ;
	 | ne: bm := FBNE;
	 | ge: bm := FBGE;
	 | gt: bm := FBGT;
	 END;
      ELSIF atype = logAT THEN
	 (* unsigned *)
	 CASE t OF
	 | lt: bm := BCS;
	 | le: bm := BLS;
	 | eq: bm := BEQ;
	 | ne: bm := BNE;
	 | ge: bm := BCC;
	 | gt: bm := BHI;
	 END;
      ELSE
	 (* signed *)
	 CASE t OF
	 | lt: bm := BLT;
	 | le: bm := BLE;
	 | eq: bm := BEQ;
	 | ne: bm := BNE;
	 | ge: bm := BGE;
	 | gt: bm := BGT;
	 END;
      END;
      Emit1(bm, "%l", dest);
   END GenTest;

   PROCEDURE SetBool(at: Attribute; destreg: Reg);
      (* set destreg according to condition codes *)
      VAR
	 mnem: Mnemonic;
   BEGIN
      WITH at^ DO
	 IF mode # condMode THEN
	    LoadCond(at);
	 END;
	 Assert(at = ccat);
	 IF (atype = floatAT) OR (atype = longfloatAT) THEN
	    (* exception on unordered! *)
	    CASE test OF
	    | lt: mnem := FSLT;
	    | le: mnem := FSLE;
	    | eq: mnem := FSEQ;
	    | ne: mnem := FSNE;
	    | ge: mnem := FSGE;
	    | gt: mnem := FSGT;
	    END;
	 ELSIF atype = logAT THEN
	    (* unsigned *)
	    CASE test OF
	    | lt: mnem := SCS;
	    | le: mnem := SLS;
	    | eq: mnem := SEQ;
	    | ne: mnem := SNE;
	    | ge: mnem := SCC;
	    | gt: mnem := SHI;
	    END;
	 ELSE
	    (* signed *)
	    CASE test OF
	    | lt: mnem := SLT;
	    | le: mnem := SLE;
	    | eq: mnem := SEQ;
	    | ne: mnem := SNE;
	    | ge: mnem := SGE;
	    | gt: mnem := SGT;
	    END;
	 END;
      END;
      Emit1(mnem, "%B%r", destreg);
      Emit1(NEG, "%B%r", destreg);
   END SetBool;

   PROCEDURE MoveBytes(from, to: Reg; nbytes: Size);
      (* addresses are in `from' and `to'; both registers
	 are released afterwards
      *)
      CONST
	 maxdb = 7FFFH; (* maximal count for DBcc instruction *)
      VAR
	 word, words: Size;
	 loopLabel: Label;
   BEGIN
      Assert((nbytes = 1) OR (nbytes = oneint16) OR (nbytes MOD oneword = 0));
      IF nbytes < oneword THEN
	 Emit2(MOVE, "%B%(r),%(r)", from, to);
      ELSIF nbytes = oneint16 THEN
	 Emit2(MOVE, "%W%(r),%(r)", from, to);
      ELSIF nbytes = oneword THEN
	 Emit2(MOVE, "%L%(r),%(r)", from, to);
      ELSIF nbytes <= 3 * oneword THEN
	 words := nbytes DIV oneword;
	 FOR word := 1 TO words DO
	    Emit2(MOVE, "%L%+r,%+r", from, to);
	 END;
      ELSIF nbytes DIV oneword <= maxdb THEN
	 GetLabel(loopLabel);
	 Emit2(MOVE, "%L%C,%r", nbytes DIV oneword - 1, d0);
	 EmitLabel(loopLabel);
	 Emit2(MOVE, "%L%+r,%+r", from, to);
	 Emit2(DBF, "%r,%l", d0, loopLabel);
      ELSE
	 GetLabel(loopLabel);
	 Emit2(MOVE, "%L%C,%r", nbytes DIV oneword, d0);
	 EmitLabel(loopLabel);
	 Emit2(MOVE, "%L%+r,%+r", from, to);
	 Emit2(SUB, "%L%C,%r", 1, d0);
	 Emit1(BNE, "%B%l", loopLabel);
      END;
      FreeReg(from); FreeReg(to);
      ccat := NIL;
   END MoveBytes;

   PROCEDURE MoveBytesAt(from, to: Attribute; nbytes: Size);
   BEGIN
      IF (nbytes MOD oneword = 0) & (nbytes <= 3 * oneword) THEN
	 WHILE nbytes > 0 DO
	    Emit2(MOVE, "%L%a,%a", from, to);
	    OffsetAt(from, oneword); OffsetAt(to, oneword);
	    DEC(nbytes, oneword);
	 END;
	 ReleaseAt(from); ReleaseAt(to);
      ELSE
	 LoadAddr(from); LoadAddr(to);
	 MoveBytes(from^.reg, to^.reg, nbytes);
	 (* don't call ReleaseAt here -- registers are released by MoveBytes *)
      END;
      ccat := NIL;
   END MoveBytesAt;

   PROCEDURE RangeCheck(at: Attribute; upperBound: Size);
      (* at^.mode = regMode;
	 check at^.reg for being inside [0..upperBound]
      *)
      VAR
	 failure, continue: Label;
   BEGIN
      StrEmit2("%*range check: %r in [0..%i]?", at^.reg, upperBound);
      GetLabel(continue); GetLabel(failure);
      IF ccat # at THEN
	 Emit1(TST, "%A%a", at);
      END;
      GenTest(lt, longAT, failure);
      Emit2(CMP, "%L%r,%C", at^.reg, upperBound);
      GenTest(le, longAT, continue);
      EmitLabel(failure);
      Emit2(MOVE, "%L%C,%-r", upperBound, top);
      Emit2(MOVE, "%L%a,%-r", at, top);
      Emit(JSR, "%_RTErrors_RangeError");
      EmitLabel(continue);
      ccat := NIL;
   END RangeCheck;

   PROCEDURE DynArrayCheck(indexat, lenat: Attribute);
      (* check indexat for being inside [0..lenat-1] *)
      VAR
	 failure, continue: Label;
   BEGIN
      GetLabel(continue); GetLabel(failure);
      LoadAndExtend(indexat);
      StrEmit1("%*range check for dynamic array: %r", indexat^.reg);
      IF ccat # indexat THEN
	 Emit1(TST, "%A%a", indexat);
      END;
      GenTest(lt, longAT, failure);
      Emit2(CMP, "%L%a,%a", indexat, lenat);
      GenTest(lt, longAT, continue);
      EmitLabel(failure);
      Emit2(MOVE, "%L%a,%r", lenat, d0);
      Emit2(SUB, "%L%C,%r", 1, d0);
      Emit2(MOVE, "%L%r,%-r", d0, top);
      Emit2(MOVE, "%L%a,%-r", indexat, top);
      Emit(JSR, "%_RTErrors_RangeError");
      EmitLabel(continue);
      ccat := NIL;
   END DynArrayCheck;

   PROCEDURE ConversionCheck(at: Attribute);
      (* at^.attype = intptr or int16ptr;
	 check at for being representable as SHORTINT
      *)
      VAR
	 failure, continue: Label;
	 arithType: ArithmeticType;
   BEGIN
      StrEmit("%*conversion check");
      GetLabel(failure); GetLabel(continue);
      Load(at);
      arithType := ArithType(at^.attype);
      Emit2(CMP, "%A%a,%C", at, maxshort);
      GenTest(gt, arithType, failure);
      Emit2(CMP, "%A%a,%C", at, minshort);
      GenTest(ge, arithType, continue);
      EmitLabel(failure);
      Emit2(MOVE, "%L%a,%-r", at, top);
      Emit(JSR, "%_RTErrors_ConversionError");
      EmitLabel(continue);
      ccat := NIL;
   END ConversionCheck;

BEGIN
   ccat := NIL;
   conversionWarningGiven := FALSE;
END GenBasicOps.
