(* Ulm's Oberon Compiler
   Copyright (c) 1989 by University of Ulm, SAI, D-W-7900 Ulm, Germany
   ----------------------------------------------------------------------------
   $Id: GenExpr.m2,v 0.15 1994/06/22 08:30:10 borchert Exp $
   ----------------------------------------------------------------------------
   $Log: GenExpr.m2,v $
   Revision 0.15  1994/06/22  08:30:10  borchert
   range checks for SET operations added

   Revision 0.14  1994/05/25  11:54:18  borchert
   some bug fixes for the handling of temporary pointers
   SYSTEM.VAL is now more permissive

   Revision 0.13  1994/04/04  11:00:59  borchert
   bug fix: assertion of PushTmpPointers failed because ptrIndex was
            increased even on excess of cntptrs

   Revision 0.12  1994/04/02  12:15:55  borchert
   bug fix: the check for temporary pointers didn't catch all cases

   Revision 0.11  1994/03/15  13:10:52  borchert
   bug fix: SYSTEM.PUT and SYSTEM.GET didn't work correctly for
            REALs because %A was not able to catch the correct size

   Revision 0.10  1994/03/15  12:17:53  borchert
   bug fix: desat^.tagged wasn't set properly for untagged pointers to
            records on dereferencing

   Revision 0.9  1993/10/02  15:17:30  borchert
   StringCompare accepts now character constants due to the changes
   in CodeGen (from release 0.12 to 0.13)

   Revision 0.8  1993/09/27  12:44:47  borchert
   temporary pointers are now pushed in a tracable way for the GC
   CRSPAWN pushes now the value of Coroutines_tag instead of
   the address to the stack of a newly created coroutine
   interrupts field added to coroutine structure

   Revision 0.7  1993/06/16  09:46:29  borchert
   SYSTEM.INT16 added

   Revision 0.6  1993/04/13  15:13:56  borchert
   SYSTEM.NEW is now done by GenTypes
   coroutines have now a tag field (Coroutines.tag)

   Revision 0.5  1993/02/03  12:39:16  borchert
   code for type tests changed

   Revision 0.3  1992/07/31  16:02:42  borchert
   conversion check (INTEGER to SHORTINT) added

   Revision 0.2  1992/07/31  14:37:18  borchert
   range check for array indices added

   Revision 0.1  1992/07/30  10:48:24  borchert
   Initial revision

   ----------------------------------------------------------------------------
*)

IMPLEMENTATION MODULE GenExpr; (* AFB 4/89 *)
                               (* Oberon revision: AFB 11/89 *)

   (* generate code for expressions
      `at' and everything in connection with `at' must be valid
      (i.e. Scan.errorflag must be FALSE)
   *)

   FROM Attributes IMPORT Attribute, Reg, ArithmeticType, AtMode, Label,
      AtModeSet, GetLabel, TestType, RegSet, base, top, NewAttribute,
      badprocedure, procedureAt, returnLabel;
   FROM EmitCode IMPORT EmitLabel, Emit, Emit1, Emit2, Emit3, Emit4, StrEmit,
      StrEmit1, StrEmit2, EmitAlign;
   FROM Exception IMPORT Assert;
   FROM GenBasicOps IMPORT Address, InvertTest, ccat, LoadCond,
      ArithType, GenTest, Load, ReverseTest, ReturnAt, ConstMulReg,
      ConstDivReg, ConstModReg, Mult, Div, Mod, IndexAtReg, OffsetAt,
      DereferenceAt, ReleaseAt, LoadDynArraySize, DynArray, LoadReg,
      LoadAndExtend, LoadAddr, Convert, AlignReg, CalcDynArraySize, LoadA,
      MoveBytes, RangeCheck, DynArrayCheck, ConversionCheck, Min;
   FROM GenTypes IMPORT Allocate, GetRecSize, PushTag, TypeGuard, TypeTest,
      SysAllocate;
   FROM Lex IMPORT GetStringChar, Symbol, AddLegalOptions, options,
      CompilerOptions;
   FROM Machine IMPORT procmarkspace, oneword, onebyte, bitsperword, Align,
      Environments, environment, stackdirection, Direction;
   FROM Memory IMPORT ALLOCATE;
   FROM Mnemonics IMPORT Mnemonic;
   FROM RegMan IMPORT GetReg, FreeReg, GetFloatReg, SaveRegs, RestoreRegs,
      GetAddrReg, RequestReg;
   FROM Scan IMPORT Error;
   FROM StackMan IMPORT StackAlloc, StackAllocPtr, StackFree;
   FROM Standard IMPORT charptr, byteptr, realptr, longrealptr, setptr, intptr,
      shortptr, int16ptr, longptr;
   FROM SymTab IMPORT StdProc, VarKind, ParamList, Form, Type, Size, mainmod,
      global, Level, FormSet, Numeric;
   FROM Sys IMPORT exit, fork;
   IMPORT Memory, StackMan, SymTab;

   CONST
      indexCheckOpt = "T";
	 (* this letter is in use for index checks since the first
	    Modula-2 compilers
	 *)
      conversionCheckOpt = "C"; (* check arg of SHORT for being small enough *)

   TYPE
      SymSet = SET OF Symbol;

   PROCEDURE GetTestType(sy: Symbol; VAR testtype: TestType);
   BEGIN
      CASE sy OF
      | lst: testtype := lt;
      | leq: testtype := le;
      | eql: testtype := eq;
      | neq: testtype := ne;
      | geq: testtype := ge;
      | grt: testtype := gt;
      END;
   END GetTestType;

   PROCEDURE DynArrayAndLen(VAR at: Attribute; VAR lenat: Attribute);
      (* at is an array and dyn is TRUE;
	 at^.mode is varAt (one dimension) or
		     indexAt (all dimensions)
	 in case of indexAt all indices are evaluated and
	    `at' addresses the indexed element
	    `lenat' addresses the remaining dimensions
	 `lenat' is NIL if all dimensions are indexed

	 example:
	    a: ARRAY 5, 4, 3 OF INTEGER;

	    PROCEDURE P(a: ARRAY OF ARRAY OF ARRAY OF INTEGER);
	       (* dope vector: 5 4 3 *)
	    BEGIN
	       a[i, j]
	       (* at:     start address of a[i, j]
		  lenat:  points to 3
	       *)
	       a[i]
	       (* at:     start address of a[i]
		  lenat:  points to 4
	       *)
	    END P;
      *)
      VAR
	 dynat: Attribute;		(* attribute of the whole array *)
	 dim: INTEGER;			(* # of dimensions *)
	 dimindex: INTEGER;             (* current dimension *)
	 btype: Type;			(* base type of array *)
	 indexReg: Reg;

      PROCEDURE MultIndex;
      BEGIN
	 OffsetAt(dynat, oneword); INC(dimindex);
	 IF (dimindex = dim) & (btype^.size < oneword) THEN
	    (* alignment necessary *)
	    Emit2(MOVE, "%L%a,%r", dynat, d0);
	    AlignReg(d0);
	    Emit2(MULU, "%L%r,%r", d0, indexReg);
	 ELSIF dimindex <= dim THEN
	    Emit2(MULU, "%L%a,%r", dynat, indexReg);
	 END;
      END MultIndex;

      PROCEDURE Index(at: Attribute);
      BEGIN
	 WITH at^ DO
	    IF mode = indexAt THEN
	       Index(desat);
	       GenExpr(indexat);
	       IF (indexCheckOpt IN options) &
		  ((indexat^.mode # constAt) OR (indexat^.cval.intval # 0)) THEN
		  DynArrayCheck(indexat, dynat);
	       END;
	       IF indexReg = illegal THEN
		  LoadAndExtend(indexat); indexReg := indexat^.reg;
	       ELSE
		  Emit2(ADD, "%L%a,%r", indexat, indexReg);
		  ReleaseAt(indexat);
	       END;
	       MultIndex;
	    END;
	 END;
      END Index;

   BEGIN (* DynArrayAndLen *)
      dynat := at;
      WHILE dynat^.mode = indexAt DO
	 dynat := dynat^.desat;
      END;
      WITH dynat^.attype^ DO
	 Assert((form = array) & dyn);
	 dim := 1;
	 btype := element;
	 WHILE (btype^.form = array) & btype^.dyn DO
	    INC(dim);
	    btype := btype^.element;
	 END;
      END;
      DynArray(dynat); LoadAddr(dynat);
      indexReg := illegal;
      OffsetAt(dynat, oneword); dimindex := 1;
      Index(at);
      IF dimindex <= dim THEN
	 NewAttribute(lenat);
	 WITH lenat^ DO
	    mode := addrMode;
	    attype := intptr;
	    GetAddrReg(reg);
	    Emit2(LEA, "%L%a,%r", dynat, reg);
	 END;
	 WHILE dimindex < dim DO
	    MultIndex;
	 END;
      ELSE
	 lenat := NIL;
      END;
      OffsetAt(dynat, - dimindex * oneword);
      at := dynat;
      DereferenceAt(at);
      IF indexReg # illegal THEN
	 IndexAtReg(at, indexReg, btype^.size);
      END;
   END DynArrayAndLen;

   PROCEDURE SizeOfDynArray(VAR at: Attribute; sizeReg: Reg);
      (* like DynArrayAndLen but the size is returned instead of the length *)
      VAR
	 lenat: Attribute;
   BEGIN
      DynArrayAndLen(at, lenat);
      CalcDynArraySize(at^.attype, lenat, sizeReg);
      ReleaseAt(lenat);
   END SizeOfDynArray;

   PROCEDURE LogOp(VAR at: Attribute; VAR trueLab, falseLab: Label);
      (* at^.mode is unaryAt or binaryAt and
	     opsy is one of orSY, ampersand, tilde

	 this procedure allows to avoid assembler pseudo-ops
	 which declares a label to be equal to another one;
	 this is necessary due to an assembler bug on the Nixdorf Targon/31

	 post conditions:
	     trueLab and falseLab remain unchanged if ok is set
	     at^.mode = condMode
		 tlabel = trueLab
		 flabel = falseLab
	 remark:
	     both labels are possibly used
      *)
      TYPE
	 SymSet = SET OF Symbol;
      CONST
	 LogOps = SymSet{orSY, ampersand, tilde};
      VAR
	 op1, op2: Attribute;
	 operator: Symbol;
	 label: Label; (* label between evaluation of op1 and op2 *)

      PROCEDURE LogAt(at: Attribute) : BOOLEAN;
      BEGIN
	 WITH at^ DO
	    RETURN ((mode = unaryAt) OR (mode = binaryAt)) &
		   (opsy IN LogOps)
	 END;
      END LogAt;

   BEGIN
      operator := at^.opsy;
      IF operator = tilde THEN
	 op1 := at^.rightop;
	 IF LogAt(op1) THEN
	    LogOp(op1, falseLab, trueLab);
	 ELSE
	    GenLogExpr(op1, falseLab, trueLab);
	    LoadCond(op1);
	 END;
	 InvertTest(op1^.test);
	 at := op1;
      ELSE
	 WITH at^ DO
	    op1 := leftop;
	    op2 := rightop;
	 END;
	 IF (operator = ampersand) & ~falseLab.ok THEN
	    GetLabel(falseLab);
	 ELSIF (operator = orSY) & ~trueLab.ok THEN
	    GetLabel(trueLab);
	 END;
	 label.ok := FALSE;
	 IF LogAt(op1) THEN
	    IF operator = ampersand THEN
	       LogOp(op1, label, falseLab);
	    ELSE
	       LogOp(op1, trueLab, label);
	    END;
	 ELSE
	    IF operator = ampersand THEN
	       GenLogExpr(op1, label, falseLab);
	    ELSE
	       GenLogExpr(op1, trueLab, label);
	    END;
	    LoadCond(op1);
	 END;
	 WITH op1^ DO
	    IF operator = ampersand THEN
	       InvertTest(test);
	       GenTest(test, atype, falseLab);
	    ELSE
	       GenTest(test, atype, trueLab);
	    END;
	    IF label.ok THEN
	       EmitLabel(label);
	    END;
	 END;
	 IF LogAt(op2) THEN
	    LogOp(op2, trueLab, falseLab);
	 ELSE
	    GenLogExpr(op2, trueLab, falseLab); LoadCond(op2);
	 END;
	 at := op2;
      END;
      WITH at^ DO
	 (* mode = condMode *)
	 tlabel := trueLab;
	 flabel := falseLab;
      END;
   END LogOp;

   PROCEDURE AddrAndLen(VAR at: Attribute; lenReg: Reg);
      (* at is a dynamic array;
	 store length in lenReg and load address of at
      *)
      VAR
	 lenat: Attribute;
   BEGIN
      IF at^.mode = varAt THEN
	 DynArray(at);
	 OffsetAt(at, oneword);
	 Emit2(MOVE, "%L%a,%r", at, lenReg);
	 OffsetAt(at, -oneword);
	 DereferenceAt(at);
      ELSE
	 DynArrayAndLen(at, lenat);
	 Assert(lenat # NIL);
	 LoadReg(lenat, lenReg);
      END;
      LoadAddr(at);
   END AddrAndLen;

   PROCEDURE AddrOfDynArray(VAR at: Attribute);
      (* load the address of a dynamic array;
	 the reference to the dope vector is being lost
      *)
      VAR
	 lenat: Attribute;
   BEGIN
      IF at^.mode = varAt THEN
	 DynArray(at); DereferenceAt(at);
      ELSE
	 DynArrayAndLen(at, lenat);
	 Assert(lenat # NIL);
	 ReleaseAt(lenat);
      END;
      LoadAddr(at);
   END AddrOfDynArray;

   PROCEDURE StringCompare(testtype: TestType;
			   at1, at2: Attribute; VAR at: Attribute;
			   VAR trueLab, falseLab: Label);
      (* generate code for string comparisons:
	 0X-termination is not needed

	 design of loop:

	 # calculate minimum length of both strings
		bra entry
	 loop:
		tst.b	d0
		beq	# both strings are equal
	 entry:
		mov.b	(a1)+,d0	# a1: address of at1
		cmp.b	d0,(a2)+	# a2: address of at2
		bne	# strings are not equal
		sub.l	&1,length
		bgt	loop
	 # check next byte of longer string

	 some optimizations are done for known string lengths
      *)

      CONST
	 maxdbcount = 32767; (* 2^15-1 *)
      VAR
	 t1, t2: Type;		(* type of at1 and at2 *)
	 len1, len2: CARDINAL;	(* =0 if dyn, else t?^.length *)
	 len: CARDINAL;  	(* minimum of len1 and len2 *)
	 usedb: BOOLEAN; 	(* use DBcc-instruction *)
	 cntReg, cnt2Reg: Reg;	(* string length of at1 and at2 *)
	 ccReg: Reg;		(* save cc of length comparison *)
	 loopLabel, elseLabel, entryLabel, exitLabel: Label;
	 helpat: Attribute;	(* swap variable *)
	 charval: CHAR;		(* set for one-character string constants *)
	 null: CHAR;

      PROCEDURE SetLab(VAR atLab, extLab: Label);
      BEGIN
	 IF extLab.ok THEN
	    atLab := extLab;
	 ELSE
	    GetLabel(atLab);
	    extLab := atLab;
	 END;
      END SetLab;

   BEGIN
      WITH at^ DO
	 mode := condMode;
	 atype := logAT;
	 test := testtype;
	 tlabel.ok := FALSE;
	 flabel.ok := FALSE;
      END;
      t1 := at1^.attype; t2 := at2^.attype;
      IF (t1 = charptr) OR (t2 = charptr) THEN
	 IF t1 = charptr THEN
	    ReverseTest(testtype);
	    helpat := at1; at1 := at2; at2 := helpat;
	    t1 := at1^.attype; t2 := at2^.attype;
	 END;
	 IF t1^.dyn THEN
	    AddrOfDynArray(at1);
	 ELSE
	    GenExpr(at1);
	 END;
	 (* at1: string; at2: string constant of length 1 *)
	 IF at2^.cval.sy = stringcon THEN
	    GetStringChar(at2^.cval.string, charval);
	    GetStringChar(at2^.cval.string, null); Assert(null = 0C);
	 ELSE
	    Assert(at2^.cval.sy = charcon);
	    charval := at2^.cval.charval;
	 END;
	 Emit2(CMP, "%B%a,%C", at1, ORD(charval));
	 IF t1^.dyn OR (t1^.length > 1) THEN
	    GetLabel(exitLabel);
	    Emit1(BNE, "%l", exitLabel);
	    OffsetAt(at1, onebyte);
	    Emit1(TST, "%B%a", at1);
	    EmitLabel(exitLabel);
	 END;
      ELSE
	 (* t1^.form = t2^.form = array *)
	 GetReg(cntReg);
	 IF t1^.dyn OR t2^.dyn THEN
	    len1 := 0; len2 := 0;
	    usedb := FALSE;
	    IF t1^.dyn THEN
	       AddrAndLen(at1, cntReg);
	    ELSE
	       GenExpr(at1);
	    END;
	    IF ~t2^.dyn THEN
	       GenExpr(at2);
	    END;
	    GetReg(ccReg);
	    IF t1^.dyn & t2^.dyn THEN
	       GetReg(cnt2Reg);
	       AddrAndLen(at2, cnt2Reg);
	       Emit2(CMP, "%L%r,%r", cntReg, cnt2Reg);
	       Emit2(MOVE, "%W%r,%r", ccr, ccReg);
	       Min(cntReg, cnt2Reg);
	       FreeReg(cnt2Reg);
	    ELSE
	       IF t2^.dyn THEN
		  AddrAndLen(at2, cntReg);
		  len := t1^.length; len1 := len;
	       ELSE
		  len := t2^.length; len2 := len;
	       END;
	       GetReg(cnt2Reg);
	       Emit2(MOVE, "%L%C,%r", len, cnt2Reg);
	       IF t1^.dyn THEN
		  Emit2(CMP, "%L%r,%r", cntReg, cnt2Reg);
	       ELSE
		  Emit2(CMP, "%L%r,%r", cnt2Reg, cntReg);
	       END;
	       Emit2(MOVE, "%W%r,%r", ccr, ccReg);
	       Min(cntReg, cnt2Reg);
	       FreeReg(cnt2Reg);
	    END;
	    LoadAddr(at1); LoadAddr(at2);
	 ELSE
	    GenExpr(at1); LoadAddr(at1);
	    GenExpr(at2); LoadAddr(at2);
	    len1 := t1^.length; len2 := t2^.length;
	    IF len1 > len2 THEN
	       len := len2;
	    ELSE
	       len := len1;
	    END;
	    usedb := len-1 <= maxdbcount;
	    IF usedb THEN
	       Emit2(MOVE, "%L%C,%r", len-1, cntReg);
	    ELSE
	       Emit2(MOVE, "%L%C,%r", len, cntReg);
	    END;
	 END;
	 GetLabel(entryLabel);
	 Emit1(BRA, "%l", entryLabel);
	 GetLabel(loopLabel);
	 EmitLabel(loopLabel);
	 exitLabel.ok := FALSE;
	 Emit1(TST, "%B%r", d0);
	 CASE testtype OF
	 | eq, ge, le:  SetLab(at^.tlabel, trueLab);
			Emit1(BEQ, "%l", trueLab);
	 | ne, lt, gt:  SetLab(at^.flabel, falseLab);
			Emit1(BEQ, "%l", falseLab);
	 END;
	 EmitLabel(entryLabel);
	 Emit2(MOVE, "%B%+r,%r", at1^.reg, d0);
	 Emit2(CMP, "%B%r,%+r", d0, at2^.reg);
	 IF usedb THEN
	    Emit2(DBNE, "%r,%l", cntReg, loopLabel);
	    IF len1 # len2 THEN
	       GetLabel(exitLabel);
	       Emit1(BNE, "%l", exitLabel);
	       (* check for simultaneous 0X-termination *)
	       Emit1(TST, "%B%r", d0);
	       Emit1(BEQ, "%l", exitLabel);
	       IF len1 > len THEN
		  Emit1(TST, "%B%(r)", at1^.reg);
	       ELSE (* len2 > len *)
		  Emit1(TST, "%B%(r)", at2^.reg);
		  (* XOR carry-bit in condition codes *)
		  Emit2(EOR, "%B%C,%r", 1, ccr);
	       END;
	       EmitLabel(exitLabel);
	    END;
	 ELSE
	    IF testtype = eq THEN
	       SetLab(at^.flabel, falseLab);
	       Emit1(BNE, "%l", at^.flabel);
	    ELSIF testtype = ne THEN
	       SetLab(at^.tlabel, trueLab);
	       Emit1(BNE, "%l", at^.tlabel);
	    ELSE
	       GetLabel(exitLabel); Emit1(BNE, "%l", exitLabel);
	    END;
	    Emit2(SUB, "%L%C,%r", 1, cntReg);
	    Emit1(BGT, "%l", loopLabel);
	    (* check for simultaneous 0X-termination *)
	    Emit1(TST, "%B%r", d0);
	    IF ~exitLabel.ok THEN
	       GetLabel(exitLabel);
	    END;
	    Emit1(BEQ, "%l", exitLabel);
	    IF (len1 > 0) & (len2 > 0) THEN
	       IF len1 = len2 THEN
		  Emit1(CLR, "%L%r", d0);
	       ELSIF len1 > len THEN
		  Emit1(TST, "%B%(r)", at1^.reg);
	       ELSE (* len2 > len *)
		  Emit1(TST, "%B%(r)", at2^.reg);
		  (* XOR carry-bit in condition codes *)
		  Emit2(EOR, "%B%C,%r", 1, ccr);
	       END;
	    ELSE
	       Emit2(MOVE, "%W%r,%r", ccReg, ccr); FreeReg(ccReg);
	       IF ~exitLabel.ok THEN
		  GetLabel(exitLabel);
	       END;
	       Emit1(BEQ, "%l", exitLabel);
	       GetLabel(elseLabel);
	       Emit1(BLS, "%l", elseLabel);
	       Emit1(TST, "%B%(r)", at1^.reg);
	       Emit1(BRA, "%l", exitLabel);
	       EmitLabel(elseLabel);
	       Emit1(TST, "%B%(r)", at2^.reg);
	       (* XOR carry-bit in condition codes *)
	       Emit2(EOR, "%B%C,%r", 1, ccr);
	    END;
	    IF exitLabel.ok THEN
	       EmitLabel(exitLabel);
	    END;
	 END;
	 FreeReg(cntReg);
      END;
      ReleaseAt(at1); ReleaseAt(at2);
      ccat := at;
   END StringCompare;

   PROCEDURE GenExpr(VAR at: Attribute);
      (* `at' is an expression tree of CodeGen;
	 the result attribute is returned in `at'

	 take care of condition codes:
	 if any attribute is returned with condMode
	 then the condition codes are set and must not
	 be destroyed by following instructions
      *)
      VAR
	 resultType: Type;

      PROCEDURE Unary(VAR at: Attribute);
	 (* minus:      numerics, sets
	    plus:       numerics
	    tilde:      booleans
	 *)
	 VAR
	    arith: ArithmeticType;
	    freg: Reg;
	    label1, label2: Label;
      BEGIN
	 arith := ArithType(at^.attype);
	 WITH at^ DO
	    IF opsy = tilde THEN
	       label1.ok := FALSE; label2.ok := FALSE;
	       LogOp(at, label1, label2);
	    ELSE
	       GenExpr(rightop);
	       CASE opsy OF
	       | minus: CASE arith OF
			| bitAT:       Load(rightop);
				       Emit1(NOTop, "%A%a", rightop);
			| shortAT,
			  int16AT,
			  intAT,
			  longAT:      Load(rightop);
				       Emit1(NEG, "%A%a", rightop);
			| floatAT,
			  longfloatAT: IF rightop^.mode = floatRegMode THEN
					  Emit1(FNEG, "%A%a", rightop);
				       ELSE
					  GetFloatReg(freg);
					  Emit2(FNEG, "%A%a,%r", rightop, freg);
					  ReleaseAt(rightop);
					  WITH rightop^ DO
					     mode := floatRegMode; reg := freg;
					  END;
				       END;
			END;
			ccat := rightop;
	       | plus:
	       END;
	       at := at^.rightop;
	    END;
	 END;
      END Unary;

      PROCEDURE Binary(VAR at: Attribute);
	 (* op may be: inSY, isSY, divSY, modSY, orSY *)
	 (*	    ampersand, *)
	 (*            plus, minus, times, slash, *)
	 (*            eql, neq, lst, grt, leq, geq *)
	 (* incl. set construction: comma, range *)
	 TYPE
	    SymSet = SET OF Symbol;
	 VAR
	    arith: ArithmeticType;
	    float: BOOLEAN;
	    label1, label2: Label;
	    helpat: Attribute;
	    testtype: TestType;
	    released: BOOLEAN;
	    dtype: Type;

	    PROCEDURE SetConstructor(VAR at: Attribute);

	       PROCEDURE CommaList(VAR at: Attribute);
		  (* at^.opsy = comma not required;
		     returns `at' as constAt or in regMode
		  *)
		  VAR
		     setpattern: BITSET;
		     setReg: Reg;

		  PROCEDURE Element(at: Attribute);
		  BEGIN
		     WITH at^ DO
			GenExpr(at);
			IF setReg = illegal THEN
			   GetReg(setReg);
			   Emit1(CLR, "%L%r", setReg);
			END;
			IF (mode = constAt) & (cval.sy = intcon) THEN
			   INCL(setpattern, cval.intval);
			ELSIF (mode = constAt) & (cval.sy = setcon) THEN
			   setpattern := setpattern + cval.setval;
			ELSE
			   LoadAndExtend(at);
			   IF indexCheckOpt IN options THEN
			      RangeCheck(at, bitsperword-1);
			   END;
			   Emit3(BFSET, "%r%{r:d}", setReg, reg, 1);
			   ReleaseAt(at);
			END;
		     END;
		  END Element;

	       BEGIN
		  setpattern := {};
		  setReg := illegal;
		  WHILE (at^.mode = binaryAt) & (at^.opsy = comma) DO
		     WITH at^ DO
			WITH rightop^ DO
			   IF (mode = binaryAt) & (opsy = range) THEN
			      Range(at^.rightop, setReg);
			   ELSE
			      Element(at^.rightop);
			   END;
			END;
		     END;
		     at := at^.leftop;
		  END;
		  Element(at);
		  WITH at^ DO
		     attype := setptr;
		     IF setReg = illegal THEN
			mode := constAt;
			cval.sy := setcon;
			cval.setval := setpattern;
		     ELSE
			mode := regMode;
			reg := setReg;
			extended := TRUE;
			IF setpattern # {} THEN
			   Emit2(ORop, "%L%C,%r", setpattern, reg);
			END;
		     END;
		  END;
	       END CommaList;

	       PROCEDURE Range(VAR at: Attribute; destReg: Reg);
		  (* at^.opsy = range;
		     return at in regMode
		  *)
		  VAR
		     emptyLab: Label;
		     tmpReg: Reg;
	       BEGIN
		  WITH at^ DO
		     GenExpr(leftop); GenExpr(rightop);
		     IF leftop^.mode # constAt THEN
			LoadAndExtend(leftop);
			IF indexCheckOpt IN options THEN
			   RangeCheck(leftop, bitsperword-1);
			END;
		     END;
		     IF rightop^.mode # constAt THEN
			LoadAndExtend(rightop);
			IF indexCheckOpt IN options THEN
			   RangeCheck(rightop, bitsperword-1);
			END;
		     END;
		     GetLabel(emptyLab);
		     IF destReg = illegal THEN
			GetReg(destReg);
			Emit1(CLR, "%L%r", destReg);
		     END;
		     IF (leftop^.mode = constAt) &
			(rightop^.mode = regMode) THEN
			Emit2(SUB, "%L%C,%r", leftop^.cval.intval-1,
					      rightop^.reg);
			Emit1(BLE, "%l", emptyLab);
			Emit3(BFSET, "%r%{d:r}", destReg, leftop^.cval.intval,
						 rightop^.reg);
			ReleaseAt(rightop);
		     ELSIF (leftop^.mode = regMode) &
			   (rightop^.mode = constAt) THEN
			GetReg(tmpReg);
			Emit2(MOVE, "%L%C,%r", rightop^.cval.intval+1, tmpReg);
			Emit2(SUB, "%L%r,%r", leftop^.reg, tmpReg);
			Emit1(BLE, "%l", emptyLab);
			Emit3(BFSET, "%r%{r:r}", destReg, leftop^.reg, tmpReg);
			ReleaseAt(leftop);
			FreeReg(tmpReg);
		     ELSIF (leftop^.mode = regMode) &
			   (rightop^.mode = regMode) THEN
			Emit2(SUB, "%L%r,%r", leftop^.reg, rightop^.reg);
			Emit1(BLT, "%l", emptyLab);
			Emit2(ADD, "%L%C,%r", 1, rightop^.reg);
			Emit3(BFSET, "%r%{r:r}", destReg, leftop^.reg,
						 rightop^.reg);
			ReleaseAt(leftop); ReleaseAt(rightop);
		     ELSE
			(* constAt..constAt should be folded by CodeGen *)
			Assert(FALSE);
		     END;
		     EmitLabel(emptyLab);
		     attype := setptr;
		     mode := regMode;
		     reg := destReg;
		  END;
	       END Range;

	    BEGIN
	       WITH at^ DO
		  IF (mode = binaryAt) & (opsy IN SymSet{comma, range}) THEN
		     IF opsy = comma THEN
			CommaList(at);
		     ELSE
			Range(at, illegal);
		     END;
		  ELSE
		     CommaList(at);
		  END;
	       END;
	    END SetConstructor;

      BEGIN
	 IF at^.opsy IN SymSet{orSY, ampersand} THEN
	    label1.ok := FALSE; label2.ok := FALSE;
	    LogOp(at, label1, label2);
	 ELSIF (at^.opsy IN SymSet{eql..geq}) &
	       ((at^.leftop^.attype^.form = array) OR
		(at^.rightop^.attype^.form = array)) THEN
	    WITH at^ DO
	       GetTestType(opsy, testtype);
	       label1.ok := FALSE; label2.ok := FALSE;
	       StringCompare(testtype, leftop, rightop, at, label1, label2);
	    END;
	 ELSIF at^.opsy = isSY THEN
	    WITH at^ DO
	       IF leftop^.mode # varAt THEN
		  GenExpr(leftop);
	       END;
	       label1.ok := FALSE; label2.ok := FALSE;
	       TypeTest(leftop, rightop^.attype, label1, label2);
	    END;
	    at := at^.leftop;
	 ELSIF at^.opsy IN SymSet{comma, range} THEN
	    WITH at^ DO
	       IF (opsy = comma) & (leftop = NIL) THEN
		  SetConstructor(at^.rightop);
		  at := at^.rightop;
	       ELSE
		  SetConstructor(at);
	       END;
	    END;
	 ELSIF at^.opsy = inSY THEN
	    WITH at^ DO
	       GenExpr(leftop); GenExpr(rightop);
	       IF leftop^.mode # constAt THEN
		  LoadAndExtend(leftop);
	       END;
	       IF rightop^.mode = constAt THEN
		  LoadAndExtend(rightop);
	       END;
	       IF leftop^.mode = constAt THEN
		  Emit3(BFTST, "%a%{d:d}", rightop, leftop^.cval.intval, 1);
	       ELSE
		  Emit3(BFTST, "%a%{r:d}", rightop, leftop^.reg, 1);
	       END;
	       ReleaseAt(leftop); ReleaseAt(rightop);
	       mode := condMode;
	       test := ne;
	       atype := intAT;
	       tlabel.ok := FALSE;
	       flabel.ok := FALSE;
	       ccat := at;
	    END;
	 ELSE
	    WITH at^ DO
	       IF opsy IN SymSet{eql..geq} THEN
		  IF leftop^.attype^.form > rightop^.attype^.form THEN
		     dtype := leftop^.attype;
		  ELSE
		     dtype := rightop^.attype;
		  END;
	       ELSE
		  dtype := attype;
	       END;
	       GenExpr(leftop);
	       IF leftop^.mode = condMode THEN
		  Load(leftop);
	       END;
	       Convert(leftop, dtype);
	       GenExpr(rightop);
	       IF rightop^.mode = condMode THEN
		  Load(rightop);
	       END;
	       Convert(rightop, dtype);
	       arith := ArithType(dtype);
	       float := (dtype = realptr) OR (dtype = longrealptr);
	       IF opsy IN SymSet{eql..geq} THEN
		  GetTestType(opsy, testtype);
		  IF rightop^.mode IN
		     AtModeSet{regMode, floatRegMode} THEN
		     helpat := rightop; rightop := leftop; leftop := helpat;
		     ReverseTest(testtype);
		  END;
		  IF ~(leftop^.mode IN AtModeSet{regMode, floatRegMode}) &
		     ~(rightop^.mode IN AtModeSet{constAt}) THEN
		     Load(leftop);
		  END;
		  IF float THEN
		     Load(leftop);
		     Emit2(FCMP, "%A%a,%a", leftop, rightop);
		  ELSE
		     Emit2(CMP, "%A%a,%a", leftop, rightop);
		  END;
		  ReturnAt(leftop); ReturnAt(rightop);
		  mode := condMode;
		  test := testtype;
		  atype := arith;
		  tlabel.ok := FALSE;
		  flabel.ok := FALSE;
		  ccat := at;
	       ELSE
		  IF opsy IN SymSet{times, divSY, modSY} THEN
		     IF leftop^.mode = constAt THEN
			Load(rightop);
		     ELSIF rightop^.mode = constAt THEN
			Load(leftop);
		     END;
		  END;
		  IF (rightop^.mode IN AtModeSet{regMode, floatRegMode}) &
		     (rightop^.reg IN RegSet{d0..d7}) &
		     (opsy IN SymSet{plus, times}) THEN
		     (* commutative operators *)
		     helpat := rightop; rightop := leftop; leftop := helpat;
		  END;
		  IF ~(leftop^.mode IN AtModeSet{regMode, floatRegMode}) OR
		     (leftop^.reg IN RegSet{a0..a7}) THEN
		     Load(leftop);
		  END;
		  released := FALSE;
		  CASE opsy OF
		  | plus:  IF float THEN
			      Emit2(FADD, "%A%a,%a", rightop, leftop);
			   ELSIF arith = bitAT THEN
			      Emit2(ORop, "%A%a,%a", rightop, leftop);
			   ELSE
			      Emit2(ADD, "%A%a,%a", rightop, leftop);
			   END;
		  | minus: IF float THEN
			      Emit2(FSUB, "%A%a,%a", rightop, leftop);
			   ELSIF arith = bitAT THEN
			      Load(rightop);
			      Emit1(NOTop, "%A%a", rightop);
			      Emit2(ANDop, "%A%a,%a", rightop, leftop);
			   ELSE
			      Emit2(SUB, "%A%a,%a", rightop, leftop);
			   END;
		  | times: IF float THEN
			      Emit2(FMUL, "%A%a,%a", rightop, leftop);
			   ELSIF arith = bitAT THEN
			      Emit2(ANDop, "%A%a,%a", rightop, leftop);
			   ELSIF rightop^.mode = constAt THEN
			      ConstMulReg(arith,
					  leftop^.reg, rightop^.cval.intval);
			   ELSE
			      Mult(arith, leftop, rightop);
			      released := TRUE;
			   END;
		  | slash: IF float THEN
			      Emit2(FDIV, "%A%a,%a", rightop, leftop);
			   ELSE (* arith = bitAT *)
			      IF rightop^.mode # constAt THEN
				 Load(rightop);
			      END;
			      Emit2(EOR, "%A%a,%a", rightop, leftop);
			   END;
		  | divSY: IF rightop^.mode = constAt THEN
			      ConstDivReg(arith,
					  leftop^.reg, rightop^.cval.intval);
			   ELSE
			      Div(arith, leftop, rightop);
			      released := TRUE;
			   END;
		  | modSY: IF rightop^.mode = constAt THEN
			      ConstModReg(arith,
					  leftop^.reg, rightop^.cval.intval);
			   ELSE
			      Mod(arith, leftop, rightop);
			      released := TRUE;
			   END;
		  END;
		  IF ~released THEN
		     ReturnAt(rightop);
		  END;
		  at := at^.leftop;
	       END;
	    END;
	 END;
      END Binary;

      PROCEDURE Designator(VAR at: Attribute);
	 (* mode one of refAt, selectAt, indexAt, or guardAt *)
	 VAR
	    lenat: Attribute;
	    taggedptr: BOOLEAN;
      BEGIN
	 WITH at^ DO
	    IF (mode = indexAt) & desat^.attype^.dyn THEN
	       DynArrayAndLen(at, lenat);
	       (* check for elementwise access *)
	       Assert(lenat = NIL);
	    ELSIF mode = guardAt THEN
	       IF desat^.mode # varAt THEN
		  GenExpr(desat);
	       END;
	       TypeGuard(desat, attype);
	       at := desat;
	    ELSE
	       GenExpr(desat);
	       CASE mode OF
	       | refAt:    taggedptr := desat^.attype^.taggedptr;
			   DereferenceAt(desat);
			   desat^.tagged := taggedptr &
					    (at^.attype^.form = record);
	       | selectAt: OffsetAt(desat, field^.offset);
			   desat^.tagged := FALSE;
	       | indexAt:  GenExpr(indexat);
			   IF indexat^.mode = constAt THEN
			      Assert(indexat^.cval.sy = intcon);
			      (* index check performed by CodeGen *)
			      OffsetAt(desat, indexat^.cval.intval *
					      desat^.attype^.element^.size);
			   ELSE
			      LoadAndExtend(indexat);
			      IF indexCheckOpt IN options THEN
				 RangeCheck(indexat, desat^.attype^.length-1);
			      END;
			      IndexAtReg(desat, indexat^.reg,
						desat^.attype^.element^.size);
			      (* indexat returned by IndexAtReg *)
			   END;
			   desat^.tagged := FALSE;
	       END;
	       at := at^.desat;
	    END;
	 END;
      END Designator;

   BEGIN
      resultType := at^.attype;
      CASE at^.mode OF
      | binaryAt: Binary(at);
      | unaryAt:  Unary(at);
      | refAt,
	selectAt,
	indexAt,
	guardAt:  Designator(at);
      | callAt:   GenCall(at);
      ELSE
      END;
      at^.attype := resultType;
      Address(at);
   END GenExpr;

   PROCEDURE GenLogExpr(VAR at: Attribute; VAR trueLab, falseLab: Label);
      (* like GenExpr; result is in condMode;
	 if tlabel or flabel are set they equal trueLab and falseLab
      *)
      VAR
	 testtype: TestType;
	 resultType: Type;
   BEGIN
      WITH at^ DO
	 resultType := attype;
	 IF ((mode = unaryAt) OR (mode = binaryAt)) &
	    (opsy IN SymSet{orSY, ampersand, tilde}) THEN
	    LogOp(at, trueLab, falseLab);
	 ELSIF (mode = binaryAt) & (opsy IN SymSet{eql..geq}) &
	       ((leftop^.attype^.form = array) OR
		(rightop^.attype^.form = array)) THEN
	    GetTestType(opsy, testtype);
	    StringCompare(testtype, leftop, rightop, at, trueLab, falseLab);
	 ELSIF (mode = binaryAt) & (opsy = isSY) THEN
	    IF leftop^.mode # varAt THEN
	       GenExpr(leftop);
	    END;
	    TypeTest(leftop, rightop^.attype, trueLab, falseLab);
	    at := leftop;
	 ELSE
	    GenExpr(at);
	    LoadCond(at);
	 END;
      END;
      at^.attype := resultType;
   END GenLogExpr;

   PROCEDURE GenCall(VAR at: Attribute);
      (* in case of functions the function value attribute is
	 returned in `at'
	 else `at' is NIL
      *)
      VAR
	 ptype: Type;

      MODULE StackAllocations;

	 (* collect stack reservations:
	       allocated during parameter loading
	       deallocated after procedure/function call
	 *)

	 FROM Memory IMPORT ALLOCATE, DEALLOCATE;
	 FROM StackMan IMPORT StackFree;
	 FROM SymTab IMPORT Size;

	 EXPORT StackRes, ReleaseAll;

	 TYPE
	    List = POINTER TO Res;
	    Res =
	       RECORD
		  offset, size: Size;
		  link: List;
	       END;
	 VAR
	    list: List;

	 PROCEDURE StackRes(offset, size: Size);
	    VAR new: List;
	 BEGIN
	    NEW(new);
	    new^.offset := offset;
	    new^.size := size;
	    new^.link := list;
	    list := new;
	 END StackRes;

	 PROCEDURE ReleaseAll;
	    VAR old: List;
	 BEGIN
	    WHILE list # NIL DO
	       StackFree(list^.offset, list^.size);
	       old := list;
	       list := list^.link;
	       DISPOSE(old);
	    END;
	 END ReleaseAll;

      BEGIN
	 list := NIL;
      END StackAllocations;

      PROCEDURE Standard(params: Attribute);

	 TYPE
	    ShiftKind = (arithShift, logShift, rotate);
	 VAR
	    param: Attribute;
	    atype: Type; (* type of argument *)

	 PROCEDURE NextParam;
	 BEGIN
	    Assert(params # NIL);
	    param := params;
	    atype := param^.attype;
	    params := params^.link;
	 END NextParam;

	 PROCEDURE Next;
	 BEGIN
	    NextParam;
	    GenExpr(param);
	 END Next;

	 PROCEDURE AbsAt(param: Attribute);
	    (* param is in regMode *)
	 BEGIN
	    (* code sequence from Henry Massalin *)
	    Emit2(MOVE, "%A%a,%r", param, d0);
	    IF atype^.size = oneword THEN
	       Emit2(ADD, "%L%r,%r", d0, d0);
	       Emit2(SUBX, "%L%r,%r", d0, d0);
	    ELSE
	       Emit2(ADD, "%B%r,%r", d0, d0);
	       Emit2(SUBX, "%B%r,%r", d0, d0);
	    END;
	    Emit2(EOR, "%A%r,%a", d0, param);
	    Emit2(SUB, "%A%r,%a", d0, param);
	    ccat := param;
	 END AbsAt;

	 PROCEDURE Shift(shiftkind: ShiftKind);
	    VAR
	       shiftcount: INTEGER;
	       left, right: Mnemonic;
	       shiftinstr: Mnemonic;
	       elseLab, endifLab: Label;
	 BEGIN
	    CASE shiftkind OF
	    | arithShift:  StrEmit("%* ASH(x,n)");
			   left := ASL; right := ASR;
	    | logShift:    StrEmit("%* LSH(x,n)");
			   left := LSL; right := LSR;
	    | rotate:      StrEmit("%* ROT(x,n)");
			   left := ROL; right := ROR;
	    END;
	    Next; at := param; Next;
	    LoadAndExtend(at);
	    IF param^.mode = constAt THEN
	       shiftcount := param^.cval.intval;
	       IF shiftcount > 0 THEN
		  shiftinstr := left;
	       ELSE
		  shiftinstr := right;
	       END;
	       shiftcount := ABS(shiftcount);
	       IF shiftcount # 0 THEN
		  IF shiftcount <= 8 THEN
		     Emit2(shiftinstr, "%L%C,%a", shiftcount, at);
		  ELSE
		     Emit2(MOVE, "%L%C,%r", shiftcount, d0);
		     Emit2(shiftinstr, "%L%r,%a", d0, at);
		  END;
	       END;
	    ELSE
	       LoadAndExtend(param);
	       IF ccat # param THEN
		  Emit1(TST, "%A%a", param);
	       END;
	       GetLabel(elseLab); GetLabel(endifLab);
	       GenTest(gt, intAT, elseLab);
	       GenTest(eq, intAT, endifLab);
	       AbsAt(param);
	       Emit2(right, "%L%r,%a", param^.reg, at);
	       Emit1(BRA, "%l", endifLab);
	       EmitLabel(elseLab);
	       Emit2(left, "%L%r,%a", param^.reg, at);
	       EmitLabel(endifLab);
	    END;
	    ReturnAt(param);
	    ccat := at;
	 END Shift;

	 PROCEDURE Entier;
	    CONST
	       (* the bits 27 and 28 determine the rounding mode *)
	       roundingMode = { 27 };   (* 10: round towards minus infinity *)
	       roundingMask = { 0..31 } - { 27, 28 };
	    VAR
	       controlReg: Reg; (* save old fp-control register *)
	       loadReg: Reg; (* integer result *)
	 BEGIN
	    StrEmit("%* ENTIER: rounding mode towards minus infinity");
	    Next; Load(param);
	    GetReg(controlReg);
	    Emit2(FMOVE, "%L%r,%r", fpcr, controlReg);
	    Emit2(MOVE, "%L%r,%r", controlReg, d0);
	    Emit2(ANDop, "%L%C,%r", roundingMask, d0);
	    Emit2(ORop, "%L%C,%r", roundingMode, d0);
	    Emit2(FMOVE, "%L%r,%r", d0, fpcr);
	    GetReg(loadReg);
	    WITH param^ DO
	       Emit2(FMOVE, "%L%r,%r", reg, loadReg);
	       FreeReg(reg);
	       mode := regMode;
	       extended := TRUE;
	       reg := loadReg;
	    END;
	    Emit2(FMOVE, "%L%r,%r", controlReg, fpcr);
	    FreeReg(controlReg);
	    ccat := NIL;
	    at := param;
	 END Entier;

	 PROCEDURE Len;
	    VAR
	       arrayType: Type;
	       endLab, errorLab, testLab: Label;
	       tabLab: Label;
	       dimensions: CARDINAL;

	    PROCEDURE RangeError;
	    BEGIN
	       Emit2(MOVE, "%L%C,%-r", dimensions-1, top);
	       Emit2(MOVE, "%L%a,%-r", param, top);
	       Emit(JSR, "%_RTErrors_RangeError");
	    END RangeError;

	 BEGIN
	    at := params; Assert(at # NIL); params := params^.link;
	    IF params # NIL THEN
	       Next;
	    ELSE
	       param := NIL;
	    END;
	    IF at^.attype^.dyn THEN
	       DynArray(at);
	       IF param = NIL THEN
		  OffsetAt(at, oneword);
	       ELSIF param^.mode = constAt THEN
		  OffsetAt(at, (param^.cval.intval + 1) * oneword);
	       ELSE
		  Load(param);
		  OffsetAt(at, oneword);
		  IndexAtReg(at, param^.reg, oneword);
		  param^.reg := illegal; (* don't release param^.reg twice *)
	       END;
	       ccat := NIL;
	    ELSE (* not a dynamic array *)
	       Load(param); Convert(param, longptr);
	       IF ccat # param THEN
		  Emit1(TST, "%L%a", param);
	       END;
	       (* count number of dimensions *)
	       arrayType := at^.attype;
	       dimensions := 0;
	       WHILE arrayType^.form = array DO
		  INC(dimensions);
		  arrayType := arrayType^.element;
	       END;
	       arrayType := at^.attype;
	       WITH at^ DO
		  mode := regMode;
		  extended := TRUE;
		  GetReg(reg);
	       END;
	       IF dimensions = 1 THEN (* one legal value only *)
		  GetLabel(endLab);
		  GenTest(ne, longAT, endLab);
		  RangeError;
		  EmitLabel(endLab);
		  Emit2(MOVE, "%L%C,%a", arrayType^.length, at);
	       ELSE (* table of dimensions needed *)
		  GetLabel(testLab);
		  Emit1(BRA, "%l", testLab);
		  EmitAlign;
		  GetLabel(tabLab);
		  EmitLabel(tabLab);
		  WHILE arrayType^.form = array DO
		     StrEmit1("%:L%c", arrayType^.length);
		     arrayType := arrayType^.element;
		  END;
		  EmitLabel(testLab);
		  GetLabel(endLab); GetLabel(errorLab);
		  GenTest(lt, longAT, errorLab);
		  Emit2(CMP, "%L%a,%C", param, dimensions);
		  GenTest(lt, longAT, endLab);
		  EmitLabel(errorLab);
		  RangeError;
		  EmitLabel(endLab);
		  Emit3(MOVE, "%L%(%l,r4),%a", tabLab, param^.reg, at);
		  ReturnAt(param);
	       END;
	       ccat := at;
	    END;
	    IF param # NIL THEN
	       ReturnAt(param);
	    END;
	 END Len;

	 PROCEDURE CopyString;
	    (* COPY(x,v)  string assignment;
	       guarantee 0X-termination of `v' afterwards
	    *)
	    CONST
	       maxdb = 7FFFH; (* maximal count for DBcc instruction *)
	    VAR
	       stringAt: Attribute;
	       desAt: Attribute;
	       termReg,			(* index reg for 0X *)
	       len1Reg,			(* length of first argument: x *)
	       len2Reg: Reg;		(* length of 2nd argument: v *)
	       loopLab: Label;
	       nbytes,
	       minlen: Size;		(* minimum length *)
	 BEGIN
	    StrEmit("%*  string copy");
	    NextParam; stringAt := param;
	    NextParam; desAt := param;
	    IF stringAt^.attype^.dyn OR desAt^.attype^.dyn THEN
	       GetReg(len1Reg); GetReg(len2Reg); GetReg(termReg);
	       IF stringAt^.attype^.dyn THEN
		  AddrAndLen(stringAt, len1Reg);
	       ELSE
		  GenExpr(stringAt);
		  Emit2(MOVE, "%L%C,%r", stringAt^.attype^.length, len1Reg);
		  LoadAddr(stringAt);
	       END;
	       IF desAt^.attype^.dyn THEN
		  AddrAndLen(desAt, len2Reg);
	       ELSE
		  GenExpr(desAt);
		  Emit2(MOVE, "%L%C,%r", desAt^.attype^.length, len2Reg);
		  LoadAddr(desAt);
	       END;
	       Min(len1Reg, len2Reg); (* minimum now in len1Reg *)
	       (* convert number of bytes to number of words *)
	       Emit2(MOVE, "%L%r,%r", len1Reg, termReg);
	       Emit2(ADD, "%L%r,%r", desAt^.reg, termReg);
	       Emit2(ADD, "%L%C,%r", oneword-1 - 1, len1Reg); (* reserve 0X! *)
	       Emit2(LSR, "%L%C,%r", 2, len1Reg); (* len1Reg := len1Reg DIV 4 *)
	       (* copy loop *)
	       GetLabel(loopLab); EmitLabel(loopLab);
	       Emit2(MOVE, "%L%+r,%+r", stringAt^.reg, desAt^.reg);
	       Emit2(SUB, "%L%C,%r", 1, len1Reg);
	       (* don't use eq for the test because len1Reg may
		  be negative (if one of the arrays has a length of 1)
	       *)
	       GenTest(gt, longAT, loopLab);
	       Emit2(CLR, "%B%(d,r)", -1, termReg);
	       FreeReg(len1Reg); FreeReg(len2Reg); FreeReg(termReg);
	    ELSE (* no dynamic arrays *)
	       GenExpr(stringAt); GenExpr(desAt);
	       IF stringAt^.attype^.length > desAt^.attype^.length THEN
		  minlen := desAt^.attype^.length;
	       ELSE
		  minlen := stringAt^.attype^.length;
	       END;
	       LoadAddr(stringAt); LoadAddr(desAt);
	       GetReg(termReg); Emit2(MOVE, "%L%r,%r", desAt^.reg, termReg);
	       nbytes := minlen-1; Align(nbytes);
	       MoveBytes(stringAt^.reg, desAt^.reg, nbytes);
	       stringAt^.reg := illegal; desAt^.reg := illegal;
	       Emit2(CLR, "%B%(d,r)", minlen-1, termReg);
	       FreeReg(termReg);
	    END;
	    ReturnAt(stringAt); ReturnAt(desAt);
	    at := NIL; ccat := NIL;
	 END CopyString;

	 PROCEDURE BitAccess;
	    VAR
	       addrAt: Attribute;
	 BEGIN
	    Next; LoadA(param); addrAt := param;
	    Next;
	    IF param^.mode = constAt THEN
	       Emit3(BFTST, "%(r)%{d:d}", addrAt^.reg, param^.cval.intval, 1);
	    ELSE
	       Load(param);
	       Emit3(BFTST, "%(r)%{r:d}", addrAt^.reg, param^.reg, 1);
	    END;
	    ReturnAt(addrAt); ReleaseAt(param);
	    at := param;
	    WITH at^ DO
	       mode := condMode;
	       test := ne;
	       atype := logAT;
	       tlabel.ok := FALSE;
	       flabel.ok := FALSE;
	    END;
	    ccat := at;
	 END BitAccess;

	 PROCEDURE Val;
	    (* convert expression of type `stype' to `dtype'

	       possible forms

	       shortint, int16, integer, longint, real, longreal,
	       boolean, char, set, byte,
	       array, record, pointer, proceduretype
	    *)
	    CONST
	       byteTypes = FormSet{shortint, boolean, char, byte};
	       LikeNumeric = Numeric + FormSet{byte, set};
	    VAR
	       dtype,			(* target type *)
	       stype: Type;		(* operand type *)
	 BEGIN
	    ccat := NIL;
	    NextParam; dtype := param^.attype;
	    Next; at := param; stype := at^.attype;
	    IF (stype^.form = array) & stype^.dyn THEN
	       Error("argument of SYSTEM.VAL must not be a dynamic array");
	    ELSIF (dtype^.form IN LikeNumeric) &
		  (stype^.form IN LikeNumeric) THEN
	       Convert(param, dtype);
	    ELSIF (dtype^.size = stype^.size) &
		  (~(at^.mode IN AtModeSet{regMode, floatRegMode}) OR
		    (FormSet{dtype^.form, stype^.form} <=
			(* one word types *)
			FormSet{integer..longint,
				set, pointer, proceduretype}) OR
		    (FormSet{dtype^.form, stype^.form} <=
			(* one byte types *)
			FormSet{shortint, byte, char, boolean})) THEN
	       (* conversion w/o interpretation *)
	    ELSE
	       (* ERROR RECOVERY ???? *)
	       Error("conversion impossible or not supported");
	    END;
	 END Val;

	 PROCEDURE Get;
	    VAR
	       addrAt: Attribute;
	       floatreg: Reg;
	 BEGIN
	    Next; DereferenceAt(param); addrAt := param;
	    Next;
	    addrAt^.attype := param^.attype;
	       (* propagating the type assures that %A selects
		  the correct size attribute
	       *)
	    IF param^.attype^.form IN FormSet{real, longreal} THEN
	       GetFloatReg(floatreg);
	       Emit2(FMOVE, "%A%a,%r", addrAt, floatreg);
	       Emit2(FMOVE, "%A%r,%a", floatreg, param);
	       FreeReg(floatreg);
	    ELSE
	       Emit2(MOVE, "%A%a,%a", addrAt, param);
	    END;
	    ReturnAt(addrAt); ReturnAt(param);
	    ccat := NIL;
	 END Get;

	 PROCEDURE Put;
	    VAR
	       addrAt: Attribute;
	 BEGIN
	    Next; DereferenceAt(param); addrAt := param;
	    Next;
	    addrAt^.attype := param^.attype;
	       (* propagating the type assures that %A selects
		  the correct size attribute
	       *)
	    IF param^.attype^.form IN FormSet{real, longreal} THEN
	       Load(param);
	       Emit2(FMOVE, "%A%a,%a", param, addrAt);
	    ELSE
	       Emit2(MOVE, "%A%a,%a", param, addrAt);
	    END;
	    ReturnAt(addrAt); ReturnAt(param);
	    ccat := NIL;
	 END Put;

	 PROCEDURE Move;
	    (* MOVE(v0,v1,n): assign first n bytes of v0 to v1 *)
	    VAR
	       v0at, v1at: Attribute;
	       condLab, loopLab: Label;
	 BEGIN
	    Next; v0at := param; LoadAddr(v0at);
	    Next; v1at := param; LoadAddr(v1at);
	    Next;
	    IF (param^.mode = constAt) &
	       ((param^.cval.intval = 1) OR
	       (param^.cval.intval MOD oneword = 0)) THEN
	       MoveBytes(v0at^.reg, v1at^.reg, param^.cval.intval);
	    ELSE (* bytewise (and inefficient copy) *)
	       LoadAndExtend(param);
	       GetLabel(loopLab); GetLabel(condLab);
	       IF ccat # param THEN
		  Emit1(TST, "%L%a", param);
	       END;
	       Emit1(BRA, "%l", condLab);
	       EmitLabel(loopLab);
	       Emit2(MOVE, "%B%+r,%+r", v0at^.reg, v1at^.reg);
	       Emit2(SUB, "%L%C,%r", 1, param^.reg);
	       EmitLabel(condLab);
	       GenTest(gt, intAT, loopLab);
	       ReleaseAt(v0at); ReleaseAt(v1at); ReleaseAt(param);
	    END;
	    ccat := NIL;
	 END Move;

	 PROCEDURE SysNew;
	    (* NEW(v,n) allocate storage block of n bytes and
	       assign its address to v
	    *)
	    VAR
	       varAt: Attribute;
	 BEGIN
	    Next; varAt := param;
	    Next;
	    Assert(varAt^.attype^.form = pointer);
	    IF varAt^.attype^.reftype^.containsptr THEN
	       Error("SYSTEM.NEW must not be called for types which contain traced pointers");
	       ReleaseAt(varAt); ReleaseAt(param); ccat := NIL;
	    ELSE
	       SysAllocate(varAt, param);
	    END;
	 END SysNew;

	 PROCEDURE UnixCall;
	    VAR
	       sysCall: INTEGER;
	       d0at, d1at: Attribute;
	       parsize: Size; (* # of bytes used by parameters *)
	       d1saved: BOOLEAN; saveReg: Reg;

	    PROCEDURE PushParams(params: Attribute);
	       (* push parameters in reverse order *)
	    BEGIN
	       IF params # NIL THEN
		  PushParams(params^.link);
		  GenExpr(params);
		  WITH params^ DO
		     IF attype^.size < oneword THEN
			LoadAndExtend(params);
			INC(parsize, oneword);
			Emit2(MOVE, "%L%a,%-r", params, top);
		     ELSE
			Emit2(MOVE, "%A%a,%-r", params, top);
			INC(parsize, attype^.size);
		     END;
		  END;
		  ReleaseAt(params);
	       END;
	    END PushParams;

	 BEGIN
	    Next; Assert(param^.mode = constAt); sysCall := param^.cval.intval;
	    StrEmit1("%* SYSTEM.UNIXCALL call=%c", sysCall);
	    d0at := params; params := params^.link;
	    d1at := params; params := params^.link;
	    parsize := 0; PushParams(params);
	    Emit2(SUB, "%L%C,%r", oneword, top);
	    CASE environment OF
	    | targon31: Emit2(MOVE, "%L%C,%r", sysCall, d0);
	    | sun3:     Emit2(MOVE, "%L%C,%-r", sysCall, top);
			(* don't increment parsize; the sp is
			   adjusted by the kernel
			*)
			(* INC(parsize, oneword); *)
	    END;
	    d1saved := ~RequestReg(d1);
	    IF d1saved THEN
	       GetReg(saveReg); Emit2(MOVE, "%L%r,%r", d1, saveReg);
	    END;
	    Emit1(TRAP, "%C", 0);
	    WITH at^ DO
	       mode := regMode;
	       extended := FALSE;
	       GetReg(reg); (* cannot be d1 -- see above *)
	       Emit1(SCC, "%B%r", reg);
	       Emit1(NEG, "%B%r", reg);
	    END;
	    GenExpr(d0at); Emit2(MOVE, "%A%r,%a", d0, d0at);
	    GenExpr(d1at); Emit2(MOVE, "%A%r,%a", d1, d1at);
	    IF d1saved THEN
	       Emit2(MOVE, "%L%r,%r", saveReg, d1); FreeReg(saveReg);
	    ELSE
	       FreeReg(d1);
	    END;
	    ReturnAt(d0at); ReturnAt(d1at);
	    Emit2(ADD, "%L%C,%r", parsize + oneword, top);
	    ccat := NIL;
	 END UnixCall;

	 PROCEDURE UnixFork;
	    VAR
	       d1saved: BOOLEAN;
	       saveReg: Reg;
	       fatherLab, endifLab: Label;
	 BEGIN
	    StrEmit("%* SYSTEM.UNIXFORK");
	    d1saved := ~RequestReg(d1);
	    IF d1saved THEN
	       GetReg(saveReg); Emit2(MOVE, "%L%r,%r", d1, saveReg);
	    END;

	    Next;
	    CASE environment OF
	    | targon31: Emit2(MOVE, "%L%C,%r", fork, d0);
	    | sun3:     Emit2(MOVE, "%L%C,%-r", fork, top);
	    END;
	    Emit1(TRAP, "%C", 0);
	    WITH at^ DO
	       mode := regMode;
	       extended := FALSE;
	       GetReg(reg); (* cannot be d1 -- see above *)
	       Emit1(SCC, "%B%r", reg);
	       Emit1(NEG, "%B%r", reg);
	    END;
	    Emit1(TST, "%B%r", d1);
	    GetLabel(fatherLab); GetLabel(endifLab);
	    Emit1(BEQ, "%B%l", fatherLab);
	    StrEmit("%*  son");
	    Emit1(CLR, "%A%a", param); Emit1(BRA, "%B%l", endifLab);
	    StrEmit("%*  father");
	    EmitLabel(fatherLab); Emit2(MOVE, "%A%r,%a", d0, param);
	    EmitLabel(endifLab);
	    ReturnAt(param);

	    IF d1saved THEN
	       Emit2(MOVE, "%L%r,%r", saveReg, d1); FreeReg(saveReg);
	    ELSE
	       FreeReg(d1);
	    END;
	    ccat := NIL;
	 END UnixFork;

	 PROCEDURE UnixSignal;
	 BEGIN
	    StrEmit("%* SYSTEM.UNIXSIGNAL");
	    Next; Convert(param, intptr); Emit2(MOVE, "%L%a,%-r", param, top);
	    ReturnAt(param);
	    Next; Emit2(MOVE, "%L%a,%-r", param, top); ReturnAt(param);
	    Next; Emit1(PEA, "%L%a", param); ReturnAt(param);
	    Next; Emit1(PEA, "%L%a", param); ReturnAt(param);
	    Emit(JSR, "_signal_");
	    WITH at^ DO
	       mode := regMode;
	       extended := FALSE;
	       GetReg(reg);
	       Emit2(MOVE, "%B%r,%r", d0, reg);
	    END;
	    ccat := at;
	 END UnixSignal;

	 PROCEDURE TestAndSet;
	    (* we do not use TAS because TAS sets the high-order-bit;
	       like TAS, CAS (compare and set) is undividable;
	       return TRUE if successful

	       Bug of SunOS 4.x:

	       call of fork() causes the data pages to be write-protected
	       and to be copied on write accesses;
	       regrettably, the system does not recognize instructions
	       with indivisible read-modify-write memory cycles as
	       write accesses;
	       thus tas or cas lead to bus errors after fork() if the
	       associated page has not been touched previously
	    *)
	    VAR
	       reg: Reg;
	 BEGIN
	    StrEmit("%* SYSTEM.TAS");
	    Next;
	    Emit1(CLR, "%B%r", d0);
	    GetReg(reg); Emit2(MOVE, "%B%C,%r", ORD(TRUE), reg);
	    IF environment = sun3 THEN
	       LoadAddr(param);
	       StrEmit("%* bug fix for SunOS 4.x");
	       (* write access to enforce page copying:
		  param is a boolean value: the first bit is always clear
	       *)
	       Emit3(BFCLR, "%a%{d:d}", param, 0, 1);
	    END;
	    Emit3(CAS, "%B%r,%r,%a", d0, reg, param);
	    ReturnAt(param); FreeReg(reg);
	    WITH at^ DO
	       mode := condMode;
	       test := ne;
	       atype := logAT;
	       tlabel.ok := FALSE;
	       flabel.ok := FALSE;
	    END;
	    ccat := at;
	 END TestAndSet;

	 PROCEDURE WordMove;
	    VAR
	       fromReg, toReg: Reg;
	       loopLabel: Label;
	 BEGIN
	    Next; LoadA(param); fromReg := param^.reg;
	    Next; LoadA(param); toReg := param^.reg;
	    Next;
	    IF param^.mode = constAt THEN
	       MoveBytes(fromReg, toReg, param^.cval.intval * oneword);
	    ELSE
	       Emit2(MOVE, "%L%a,%r", param, d0); ReleaseAt(param);
	       GetLabel(loopLabel);
	       EmitLabel(loopLabel);
	       Emit2(MOVE, "%L%+r,%+r", fromReg, toReg);
	       Emit2(SUB, "%L%C,%r", 1, d0);
	       Emit1(BNE, "%B%l", loopLabel);
	       FreeReg(fromReg); FreeReg(toReg);
	    END;
	    ccat := NIL;
	 END WordMove;

	 PROCEDURE WordClear;
	    CONST
	       maxdb = 7FFFH; (* maximal count for DBcc instruction *)
	    VAR
	       addrReg: Reg;
	       loopLabel: Label;
	 BEGIN
	    Next; LoadA(param); addrReg := param^.reg;
	    Next;
	    IF (param^.mode = constAt) & (param^.cval.intval <= maxdb) OR
	       (param^.attype = shortptr) THEN
	       IF param^.mode = constAt THEN
		  Emit2(MOVE, "%L%C,%r", param^.cval.intval - 1, d0);
	       ELSE
		  Emit2(MOVE, "%L%a,%r", param, d0);
		  Emit2(SUB, "%L%C,%r", 1, d0);
	       END;
	       GetLabel(loopLabel);
	       EmitLabel(loopLabel);
	       Emit1(CLR, "%L%+r", addrReg);
	       Emit2(DBF, "%r,%l", d0, loopLabel);
	       FreeReg(addrReg);
	    ELSE
	       Emit2(MOVE, "%L%a,%r", param, d0); ReleaseAt(param);
	       GetLabel(loopLabel);
	       EmitLabel(loopLabel);
	       Emit1(CLR, "%L%+r", addrReg);
	       Emit2(SUB, "%L%C,%r", 1, d0);
	       Emit1(BNE, "%B%l", loopLabel);
	       FreeReg(addrReg);
	    END;
	 END WordClear;

	 PROCEDURE CrSpawn;
	    (* CRSPAWN(VAR cr: SYSTEM.COROUTINE [; size: LONGINT]); *)
	    VAR
	       cr: Attribute;
	       sizereg: Reg;
	       arlen: Reg; (* register containing length of AR *)
	       source, dest: Reg;
	       crbase, crtop: Reg;
	       continueLab: Label;
	       paramsize: Size;
	       copyLoop: Label;
	 BEGIN
	    StrEmit("%* SYSTEM.CRSPAWN");
	    GetLabel(continueLab);
	    Assert(~badprocedure);
	    Next; cr := param;

	    (* calculate size of activation record *)
	    GetReg(arlen);
	    Emit2(MOVE, "%L%r,%r", base, arlen);
	    Emit2(SUB, "%L%r,%r", top, arlen);
	    paramsize := ABS(procedureAt^.atip^.parmoffset) + procmarkspace;

	    (* determine size of stack *)
	    IF params # NIL THEN
	       Next; Load(param); sizereg := param^.reg;
	       (* check that sizereg exceeds arlen *)
	    ELSE
	       GetReg(sizereg);
	       Emit2(MOVE, "%L%r,%r", arlen, sizereg);
	       Emit1(ADD, "%L%_Coroutines_defaultsize,%r", sizereg);
	    END;

	    (* allocate stack *)
	    SaveRegs;
	    Emit1(PEA, "%L%a", cr); (* VAR ptr: LONGINT *)
	    Emit2(MOVE, "%L%r,%-r", sizereg, top); (* size: LONGINT *)
	    Emit2(MOVE, "%L%C,%-r", -1, top); (* mode: SHORTINT -- backward *)
	    Emit(JSR, "%_Storage_AllocateStack");
	    (* ptr points now to the end address of the
	       backward growing stack
	    *)
	    RestoreRegs;
	    FreeReg(sizereg);

	    (* write coroutine record at beginning of new stack *)
	    GetAddrReg(dest); Emit2(MOVE, "%L%a,%r", cr, dest);
	    Emit1(CLR, "%L%-r", dest); (* dest.interrupts *)
	    Emit2(MOVE, "%L%#%l,%-r", continueLab, dest);        (* dest.pc *)
	    GetReg(crbase); Emit2(MOVE, "%L%r,%r", dest, crbase);
	    Emit2(SUB, "%L%C,%r", 3*oneword + paramsize, crbase);
	    GetReg(crtop);
	    Emit2(MOVE, "%L%r,%r", crbase, crtop);
	    Emit2(SUB, "%L%r,%r", arlen, crtop);
	    Emit2(MOVE, "%L%r,%-r", crtop, dest);                (* dest.top *)
	    FreeReg(crtop);
	    Emit2(MOVE, "%L%r,%-r", crbase, dest);               (* dest.base *)
	    Emit2(MOVE, "%L%r,%a", dest, cr);
	    Emit1(MOVE, "%L%_Coroutines_tag,%-r", dest); (* tag field *)

	    (* copy activation record to new stack *)
	    Emit2(ADD, "%L%C,%r", paramsize, arlen);
	    GetAddrReg(source);
	    Emit2(MOVE, "%L%r,%r", base, source);
	    Emit2(ADD, "%L%C,%r", paramsize, source);
	    GetLabel(copyLoop); EmitLabel(copyLoop);
	    Emit2(MOVE, "%L%-r,%-r", source, dest);
	    Emit2(SUB, "%L%C,%r", oneword, arlen);
	    Emit1(BGT, "%l", copyLoop);
	    FreeReg(source); FreeReg(dest); FreeReg(arlen);

	    (* set dynamic link to 0;
	       needed by garbage collectors and debuggers
	    *)
	    Emit1(CLR, "%L%(r)", crbase);
	    (* set return address to RTErrors.CoroutineReturn *)
	    Emit2(MOVE, "%L%#%_RTErrors_CoroutineReturn,%(d,r)",
	       oneword, crbase);
	    FreeReg(crbase);

	    (* return *)
	    Emit1(BRA, "%l", returnLabel);

	    EmitLabel(continueLab); (* new coroutine continues here *)

	    ReleaseAt(cr);
	 END CrSpawn;

	 PROCEDURE CrSwitch;
	    CONST
	       regmask = {16, 17}; (* base(a6) and top(a7) *)
	    VAR
	       source, dest: Reg;
	       continueLab: Label;
	 BEGIN
	    StrEmit("%* SYSTEM.CRSWITCH");
	    SaveRegs;

	    GetAddrReg(source);
	    Emit1(MOVE, "%L%_Coroutines_current,%r", source);
	    Next; LoadA(param); dest := param^.reg;

	    GetLabel(continueLab);
	    Emit2(MOVE, "%L%r,%+r", base, source);
	    Emit2(MOVE, "%L%r,%+r", top, source);
	    Emit2(MOVE, "%L%#%l,%+r", continueLab, source);
	    Emit(MOVE, "%L%_Coroutines_current,%_Coroutines_source");
	    Emit1(MOVE, "%L%r,%_Coroutines_current", dest);
	    Emit2(MOVEM, "%L%+r,%C", dest, regmask);
	    Emit1(JMP, "%([r])", dest);
	    EmitLabel(continueLab);
	    FreeReg(source); FreeReg(dest);

	    RestoreRegs;
	 END CrSwitch;

      BEGIN
	 CASE ptype^.stdproc OF

	 (* standard functions *)
         | absF:     Next; Load(param);
		     IF (atype = realptr) OR (atype = longrealptr) THEN
			Emit1(FABS, "%A%a", param);
		     ELSE
			AbsAt(param);
		     END;
		     at := param; ccat := at;
	 | ashF:     Shift(arithShift);
         | capF:     Next; Load(param);
		     Emit2(ANDop, "%A%C,%a", 137B, param);
		     at := param; ccat := at;
         | chrF:     Next; Load(param); at := param;
         | entierF:  Entier;
         | lenF:     Len;
         | longF:    Next; LoadAndExtend(param); at := param;
         | maxF:     Assert(FALSE);
         | minF:     Assert(FALSE);
         | oddF:     Next; Load(param);
		     Emit2(ANDop, "%A%C,%a", 1, param); ccat := param;
		     LoadCond(param); at := param;
         | ordF:     Next; Load(param);
		     Emit2(ANDop, "%L%C,%a", 0FFH, param);
		     at := param;
         | shortF:   Next; Load(param); at := param;
		     IF (conversionCheckOpt IN options) &
			   ((atype = intptr) OR (atype = int16ptr)) THEN
			ConversionCheck(at);
		     END;

	 (* standard procedures *)
         | decP:     Next; at := param;
		     IF params # NIL THEN
			Next; Load(param);
			Emit2(SUB, "%A%a,%a", param, at);
			ReturnAt(param); ReturnAt(at);
		     ELSE
			Emit2(SUB, "%A%C,%a", 1, at);
			ReturnAt(at);
		     END;
         | exclP:    Next; at := param; Next;
		     IF param^.mode = constAt THEN
			Emit3(BFCLR, "%a%{d:d}", at, param^.cval.intval, 1);
		     ELSE
			LoadAndExtend(param);
			IF indexCheckOpt IN options THEN
			   RangeCheck(param, bitsperword-1);
			END;
			Emit3(BFCLR, "%a%{r:d}", at, param^.reg, 1);
		     END;
		     ReturnAt(at); ReturnAt(param);
         | haltP:    Next;
		     LoadAndExtend(param);
	             Emit2(MOVE, "%L%a,%-r", param, top);
		     (* pure exit is supported by SYSTEM.HALT
			Emit2(SUB, "%L%C,%r", oneword, top);
			CASE environment OF
			| targon31: Emit2(MOVE, "%L%C,%r", exit, d0);
			| sun3:     Emit2(MOVE, "%L%C,%-r", exit, top);
			END;
			Emit1(TRAP, "%C", 0);
		     *)
		     (* library version;
			the Process module is initialized by ort0.s
		     *)
		     Emit(JSR, "%_Process_Exit");
		     ReturnAt(param);
         | incP:     Next; at := param;
		     IF params # NIL THEN
			Next; Load(param);
			Emit2(ADD, "%A%a,%a", param, at);
			ReturnAt(param); ReturnAt(at);
		     ELSE
			Emit2(ADD, "%A%C,%a", 1, at);
			ReturnAt(at);
		     END;
         | inclP:    Next; at := param; Next;
		     IF param^.mode = constAt THEN
			Emit3(BFSET, "%a%{d:d}", at, param^.cval.intval, 1);
		     ELSE
			LoadAndExtend(param);
			IF indexCheckOpt IN options THEN
			   RangeCheck(param, bitsperword-1);
			END;
			Emit3(BFSET, "%a%{r:d}", at, param^.reg, 1);
		     END;
		     ReturnAt(at); ReturnAt(param);
	 | copyP:    CopyString;
         | newP:     NextParam;
		     IF param^.mode # varAt THEN
			GenExpr(param);
		     END;
		     Allocate(param);

	 (* SYSTEM module *)
         | adrF:     Next; LoadAddr(param); at := param;
		     WITH at^ DO
			mode := regMode;
			extended := TRUE;
		     END;
	 | bitF:     BitAccess;
	 | lshF:     Shift(logShift);
	 | rotF:     Shift(rotate);
         | sizeF:    Next;
		     Assert(atype^.dyn);
		     WITH at^ DO
			mode := regMode; GetReg(reg); extended := TRUE;
		     END;
		     LoadDynArraySize(param^.atip, at^.reg);
		     ReturnAt(param);
	 | valF:     Val;

	 | crspawnP: CrSpawn;
	 | getP:     Get;
	 | putP:     Put;
	 | moveP:    Move;
	 | sysnewP:  SysNew;
	 | crswitchP: CrSwitch;

         | syshaltP: Next;
		     LoadAndExtend(param);
	             Emit2(MOVE, "%L%a,%-r", param, top);
		     Emit2(SUB, "%L%C,%r", oneword, top);
		     CASE environment OF
		     | targon31: Emit2(MOVE, "%L%C,%r", exit, d0);
		     | sun3:     Emit2(MOVE, "%L%C,%-r", exit, top);
		     END;
		     Emit1(TRAP, "%C", 0);
		     ReturnAt(param);
	 | tasF:     TestAndSet;
	 | unixcallF: UnixCall;
	 | unixsignalF: UnixSignal;
	 | unixforkF: UnixFork;
	 | wmoveP:   WordMove;
	 | wclearP:  WordClear;
	 END;
      END Standard;

      PROCEDURE LoadParams(params: Attribute; plist: ParamList);

	 TYPE
	    TmpPointerList = POINTER TO TmpPointerListRec;
	    TmpPointerListRec =
	       RECORD
		  offset: Size; (* stack offset *)
		  paramOffset: Size; (* where to store the parameter? *)
		  next: TmpPointerList;
	       END;
	 VAR
	    param: Attribute;
	    ptype: Type; (* type of parameter *)
	    word: Size;
	    cntptrs: CARDINAL;
	       (* number of temporary pointers which are to 
		  be checked by the GC
	       *)
	    tmpptrs: BOOLEAN;  (* do we have temporary pointers? *)
	    ptrIndex: CARDINAL; tmpptr: BOOLEAN;
	    tmpptrStartLabel, tmpptrEndLabel: Label;
	    tmpptrlist: TmpPointerList;

	 PROCEDURE PointerParam(at: Attribute; param: ParamList) : BOOLEAN;
	    (* only parameters where a pointer value get passed
	       are counted as pointer params here
	    *)
	 BEGIN
	    RETURN (param^.varkind = paramV) &
		   (at^.attype^.form = pointer) &
		   (param^.type^.form = pointer)
	 END PointerParam;

	 PROCEDURE CheckParamListForTmpPointers(params: Attribute;
						plist: ParamList) : CARDINAL;
	    (* returns number of temporary pointers:
	       we assume here that the GC is only called non-preemptive,
	       consequently the GC can only be called directly or
	       indirectly (via NEW) by one of the functions we need
	       to call for our parameter list;
	       thus temporary pointers are all parameters of pointer
	       type which are evaluated before calling a function
	    *)
	    VAR
	       param: Attribute;
	       pl: ParamList;
	       count: CARDINAL;
		  (* this counts all pointers of the parameter list *)
	       committedCount: CARDINAL;
		  (* pointer counts are committed if we find a
		     function call
		  *)
	 BEGIN
	    count := 0; committedCount := 0;
	    param := params; pl := plist;
	    WHILE param # NIL DO
	       WITH param^ DO
		  IF (mode = callAt) & ~procat^.attype^.std THEN
		     committedCount := count;
		  END;
		  IF PointerParam(param, pl) THEN
		     INC(count);
		  END;
	       END;
	       param := param^.link; pl := pl^.link;
	    END;
	    RETURN committedCount
	 END CheckParamListForTmpPointers;

	 PROCEDURE StoreTmpPointer(ptrat: Attribute);
	    VAR
	       offset: Size;
	       tmpptr: TmpPointerList;
	       staticLinkSpace: Size;
	 BEGIN
	    Assert(at^.mode = callAt);
	    WITH at^.procat^ DO
	       IF (mode = procAt) & (atip^.plevel > global) THEN
		  staticLinkSpace := oneword;
	       ELSE
		  staticLinkSpace := 0;
	       END;
	    END;
	    StackAllocPtr(offset, tmpptrStartLabel, tmpptrEndLabel);
	    Emit3(MOVE, "%L%a,%(d,r)", ptrat, offset, base);
	    NEW(tmpptr);
	    tmpptr^.offset := offset;
	    tmpptr^.paramOffset := plist^.offset;
	    IF stackdirection = backwardDir THEN
	       Assert(tmpptr^.paramOffset > 0);
	       DEC(tmpptr^.paramOffset, procmarkspace + staticLinkSpace);
	    ELSE
	       Assert(tmpptr^.paramOffset < 0);
	       INC(tmpptr^.paramOffset, procmarkspace + staticLinkSpace);
	    END;
	    tmpptr^.next := tmpptrlist;
	    tmpptrlist := tmpptr;
	 END StoreTmpPointer;

	 PROCEDURE PushTmpPointers;
	    VAR
	       tmpptr: TmpPointerList;
	       cnt: CARDINAL;
	 BEGIN
	    Assert(ptrIndex = cntptrs);
	    cnt := 0;
	    tmpptr := tmpptrlist;
	    WHILE tmpptr # NIL DO
	       WITH tmpptr^ DO
		  Emit4(MOVE, "%L%(d,r),%(d,r)",
		     offset, base, paramOffset, top);
		  StackFree(offset, oneword);
	       END;
	       tmpptr := tmpptr^.next;
	       INC(cnt);
	    END;
	    Assert(cnt = cntptrs);
	    EmitLabel(tmpptrEndLabel);
	    tmpptrlist := NIL;
	 END PushTmpPointers;

	 PROCEDURE LoadDynParam(VAR at: Attribute; ptype: Type);
	    (* push address of parameter AND dope vector
	       things has to be pushed in reverse order
	       (because stackdirection = backwardDir)
	    *)
	    VAR
	       actdyn: BOOLEAN; (* actual parameter is a dynamic array *)
	       sizeReg: Reg;
	       etype: Type;
	       dimensions: INTEGER; (* number of dynamic dimensions *)
	       dim: INTEGER;
	       lenat: Attribute;
	       stackOffset: Size;  	(* result of StackAlloc *)
	       pushSize: BOOLEAN; (* push of size delayed? *)
	       sizeat: Attribute;
	       startlab: Label;

	    PROCEDURE PushLength(t: Type; dim: INTEGER);
	       (* push dope vector of non-dynamic array in reverse order *)
	    BEGIN
	       WITH t^ DO
		  IF dim > 0 THEN
		     Assert(form = array);
		     PushLength(element, dim-1);
		     Emit2(MOVE, "%L%C,%-r", length, top);
		  END;
	       END;
	    END PushLength;

	 BEGIN (* LoadDynParam *)
	    WITH at^.attype^ DO
	       actdyn := (form = array) & dyn;
	    END;
	    dimensions := 0;
	    etype := ptype;
	    WHILE (etype^.form = array) & etype^.dyn DO
	       INC(dimensions);
	       etype := etype^.element;
	    END;
	    StrEmit1("%*   push dope vector of length %c", dimensions);
	    WITH ptype^ DO
	       IF element = byteptr THEN
		  pushSize := FALSE;
		  IF actdyn THEN
		     IF (at^.mode = varAt) &
			(at^.attype^.element^.size = onebyte) THEN
			DynArray(at);
			OffsetAt(at, oneword);
			Emit2(MOVE, "%L%a,%-r", at, top);
			OffsetAt(at, -oneword);
			DereferenceAt(at);
		     ELSE
			GetReg(sizeReg);
			IF at^.mode = varAt THEN
			   Assert(at^.atip # NIL);
			   LoadDynArraySize(at^.atip, sizeReg);
			ELSE
			   SizeOfDynArray(at, sizeReg);
			END;
			Emit2(MOVE, "%L%r,%-r", sizeReg, top);
			FreeReg(sizeReg);
		     END;
		  ELSE
		     WITH at^.attype^ DO
			IF (form = array) & (element^.size = onebyte) THEN
			   Emit2(MOVE, "%L%C,%-r", length, top);
			ELSE
			   IF form = record THEN
			      (* the record may be larger than expected *)
			      IF at^.mode = varAt THEN
				 GetRecSize(at, sizeat);
				 Emit2(MOVE, "%L%a,%-r", sizeat, top);
				 ReleaseAt(sizeat);
			      ELSE
				 pushSize := TRUE;
			      END;
			   ELSE
			      Emit2(MOVE, "%L%C,%-r", size, top);
			   END;
			END;
		     END;
		  END;
		  GenExpr(at);
		  IF pushSize THEN
		     GetRecSize(at, sizeat);
		     Emit2(MOVE, "%A%a,%-r", sizeat, top);
		     ReleaseAt(sizeat);
		  END;
		  WITH at^ DO
		     IF (mode = regMode) OR (mode = floatRegMode) OR
			(mode = procAt) OR (mode = condMode) OR
			(mode = constAt) & (cval.sy # stringcon) THEN
			StackAlloc(stackOffset, attype^.size);
			StackRes(stackOffset, attype^.size);
			IF (mode = constAt) &
			   (attype^.form IN FormSet{real, longreal}) OR
			   (mode = condMode) THEN
			   Load(at);
			END;
			IF mode = floatRegMode THEN
			   IF attype^.form = real THEN
			      Emit3(FMOVE, "%D%a,%(d,r)",
				    at, stackOffset, base);
			   ELSE
			      Emit3(FMOVE, "%X%a,%(d,r)",
				    at, stackOffset, base);
			   END;
			ELSE
			   Emit3(MOVE, "%A%a,%(d,r)", at, stackOffset, base);
			END;
			Emit2(PEA, "%L%(d,r)", stackOffset, base);
		     ELSE
			Emit1(PEA, "%L%a", at);
		     END;
		  END;
	       ELSE
		  IF actdyn THEN
		     IF at^.mode = varAt THEN
			DynArray(at);
			OffsetAt(at, dimensions * oneword);
			FOR dim := 1 TO dimensions DO
			   Emit2(MOVE, "%L%a,%-r", at, top);
			   OffsetAt(at, - oneword);
			END;
			Emit2(MOVE, "%L%a,%-r", at, top);
		     ELSE
			DynArrayAndLen(at, lenat);
			Assert(lenat # NIL);
			OffsetAt(lenat, dimensions * oneword);
			FOR dim := 1 TO dimensions DO
			   OffsetAt(lenat, - oneword);
			   Emit2(MOVE, "%L%a,%-r", lenat, top);
			END;
			ReleaseAt(lenat);
			Emit1(PEA, "%L%a", at);
		     END;
		  ELSE
		     IF at^.attype = charptr THEN
			(* string of length 1 *)
			Emit2(MOVE, "%L%C,%-r", 2, top);
		     ELSE
			PushLength(at^.attype, dimensions);
		     END;
		     GenExpr(at);
		     Emit1(PEA, "%L%a", at);
		  END;
	       END;
	    END;
	 END LoadDynParam;

      BEGIN (* LoadParams *)
	 cntptrs := CheckParamListForTmpPointers(params, plist);
	 tmpptrs := (cntptrs > 0); tmpptr := FALSE;
	 IF tmpptrs THEN
	    ptrIndex := 0;
	    tmpptrlist := NIL;
	    GetLabel(tmpptrStartLabel); EmitLabel(tmpptrStartLabel);
	    GetLabel(tmpptrEndLabel);
	 END;
	 WHILE params # NIL DO
	    param := params; params := params^.link;
	    StrEmit1("%*  parameter %n", plist^.id);
	    ptype := param^.attype;
	    IF tmpptrs THEN
	       tmpptr := FALSE;
	       IF PointerParam(param, plist) THEN
		  IF ptrIndex < cntptrs THEN
		     tmpptr := TRUE;
		     INC(ptrIndex);
		     StrEmit2("%*   temporary pointer (%c/%c)", ptrIndex,
			cntptrs);
		  END;
	       END;
	    END;
	    WITH plist^ DO
	       IF (type^.form = array) & type^.dyn THEN
		  Assert(~tmpptr);
		  LoadDynParam(param, type);
	       ELSIF varkind = paramV THEN
		  IF type^.size > 0 THEN
		     GenExpr(param);
		     Convert(param, type); ptype := param^.attype;
		     IF ptype^.size < oneword THEN
			(* take care of alignment *)
			IF param^.mode # constAt THEN
			   IF type^.size = oneword THEN
			      LoadAndExtend(param);
			   ELSE
			      Load(param);
			   END;
			END;
			IF (param^.mode = constAt) &
			   (param^.cval.sy = intcon) &
			   (param^.cval.intval = 0) THEN
			   Emit1(CLR, "%L%-r", top);
			ELSE
			   Emit2(MOVE, "%L%a,%-r", param, top);
			END;
		     ELSIF ptype^.size > oneword THEN
			IF (ptype = realptr) OR (ptype = longrealptr) THEN
			   Load(param);
			   (* %A does not work here *)
			   IF ptype = realptr THEN
			      Emit2(FMOVE, "%D%a,%-r", param, top);
			   ELSE
			      Emit2(FMOVE, "%X%a,%-r", param, top);
			   END;
			ELSE
			   OffsetAt(param, ptype^.size);
			   FOR word := 1 TO ptype^.size DIV oneword DO
			      OffsetAt(param, -oneword);
			      Emit2(MOVE, "%L%a,%-r", param, top);
			   END;
			END;
		     ELSIF ptype^.size > 0 THEN
			IF tmpptr THEN
			   StoreTmpPointer(param);
			   Emit2(SUB, "%L%C,%r", oneword, top);
			ELSE
			   Emit2(MOVE, "%A%a,%-r", param, top);
			END;
		     END;
		  END;
	       ELSE (* varparamV or copyparamV *)
		  Assert(~tmpptr);
		  IF (varkind = varparamV) &
			((ptype^.form = record) OR
			 (ptype^.form = pointer) &
			 (ptype^.reftype^.form = record)) THEN
		     IF param^.mode # varAt THEN
			GenExpr(param);
		     END;
		     PushTag(param);
		  END;
		  GenExpr(param);
		  Emit1(PEA, "%L%a", param);
	       END;
	       ReleaseAt(param);
	    END;
	    plist := plist^.link;
	 END;
	 IF tmpptrs THEN
	    PushTmpPointers;
	 END;
      END LoadParams;

      PROCEDURE StaticLink(plevel: CARDINAL);
	 (* plevel > global *)
	 VAR
	    indexReg: Reg;
	    leveldiff: CARDINAL;
	    level: CARDINAL;
      BEGIN
	 StrEmit("%*  static link");
	 level := Level()-1; (* current plevel *)
	 IF level >= plevel THEN
	    leveldiff := level - plevel;
	    IF leveldiff = 0 THEN
	       Emit3(MOVE, "%L%(d,r),%-r", procmarkspace, base, top);
	    ELSIF leveldiff = 1 THEN
	       Emit4(MOVE, "%L%([d,r],d),%-r",
		  procmarkspace, base, procmarkspace, top);
	    ELSE
	       GetReg(indexReg);
	       Emit3(MOVE, "%L%(d,r),%r", procmarkspace, base, indexReg);
	       WHILE leveldiff > 2 DO
		  Emit3(MOVE, "%L%(d,r),%r", procmarkspace, indexReg, indexReg);
		  DEC(leveldiff);
	       END;
	       Emit4(MOVE, "%L%([d,r],d),%-r",
		  procmarkspace, indexReg, procmarkspace, top);
	       FreeReg(indexReg);
	    END;
	 ELSE
	    Assert(level+1 = plevel);
	    Emit2(MOVE, "%L%r,%-r", base, top);
	 END;
      END StaticLink;

   BEGIN (* GenCall *)
      WITH at^ DO
	 Assert(mode = callAt);
	 ptype := procat^.attype;
	 WITH ptype^ DO
	    Assert(form = proceduretype);
	 END;
	 (* check for standard procedure or function *)
	 IF ptype^.std THEN
	    Standard(firstparam);
	 ELSE
	    IF procat^.mode = procAt THEN
	       StrEmit1("%* call of %n", procat^.atip^.name);
	    ELSIF ptype^.function THEN
	       StrEmit("%* function variable call");
	    ELSE
	       StrEmit("%* procedure variable call");
	    END;
	    SaveRegs;
	    LoadParams(firstparam, ptype^.param);
	    IF procat^.mode = procAt THEN
	       WITH procat^.atip^ DO
		  IF mod = mainmod THEN
		     IF plevel > global THEN
			StaticLink(plevel);
		     END;
		     Emit2(JSR, "%_%n_%c", mod^.origname, procno);
		  ELSE
		     Emit2(JSR, "%_%n_%n", mod^.origname, name);
		  END;
	       END;
	    ELSE
	       GenExpr(procat); DereferenceAt(procat);
	       Emit1(JSR, "%a", procat);
	       ReleaseAt(procat);
	    END;
	    RestoreRegs;
	    WITH ptype^ DO
	       IF function THEN
		  IF (restype = realptr) OR (restype = longrealptr) THEN
		     (* result in fp0 *)
		     mode := floatRegMode;
		     GetFloatReg(reg);
		     Emit2(FMOVE, "%A%r,%a", fp0, at);
		  ELSE
		     (* result in d0 *)
		     mode := regMode;
		     GetReg(reg);
		     (* beware, extended is also a component of ptype^ *)
		     at^.extended := at^.attype^.size = oneword;
		     Emit2(MOVE, "%A%r,%a", d0, at);
		  END;
	       END;
	    END;
	 END;
      END;
      IF ~ptype^.function THEN
	 at := NIL; ccat := NIL;
      END;
      ReleaseAll; (* stack reservations *)
   END GenCall;

BEGIN
   AddLegalOptions(CompilerOptions{conversionCheckOpt, indexCheckOpt});
END GenExpr.
