(* Ulm's Oberon Compiler
   Copyright (c) 1989 by University of Ulm, SAI, D-W-7900 Ulm, Germany
   ----------------------------------------------------------------------------
   $Id: GenTypes.m2,v 0.10 1994/06/22 08:30:51 borchert Exp $
   ----------------------------------------------------------------------------
   $Log: GenTypes.m2,v $
   Revision 0.10  1994/06/22  08:30:51  borchert
   some IF-statements added to avoid code generation when minAllocLen = 0

   Revision 0.9  1994/03/19  12:29:16  borchert
   bug fix: GenTypes was walking thru a list which was modified by GenType;
            this caused type tag records to be generated twice in some odd cases

   Revision 0.8  1994/02/21  12:44:36  borchert
   bug fix: InitLocalPointers didn't work correctly for arrays with pointers

   Revision 0.7  1993/10/03  14:59:16  borchert
   some bug fixes due to last revision and some minor optimizations

   Revision 0.6  1993/10/02  15:09:22  borchert
   AssignRecs produces now better code if the dynamic type of expr
   is statically known and desat^.attype = exprat^.attype

   Revision 0.5  1993/09/27  12:57:58  borchert
   support of GC completed:
   (1) compiler option $M added: on default, the two least significant
       bits of type tags may be used for marks
   (2) to assure that copying GCs can store a forward pointer, a
       minimal storage size has been defined for allocated objects
   (3) the structure of pointer lists has been extended with support
       for temporary pointers and addresses which may point inside
       an object
   bug fix: Align4 assures now that the label of the tag record
            points to the tag record

   Revision 0.4  1993/06/11  12:34:17  borchert
   Modules renamed to SysModules

   Revision 0.3  1993/04/13  15:15:45  borchert
   new memory management:
   (1) use interface of Storage instead of SysStorage
   (2) use the free region [Storage.next, Storage.next + Storage.left)
       for inline code
   (3) generation of type tags for arrays, too
   (4) SYSTEM.NEW is now handled here instead by GenExpr

   Revision 0.2  1993/02/03  12:40:39  borchert
   new type tags
   type tests now in constant time
   generation of pointer lists for garbage collection

   Revision 0.1  1992/07/30  10:48:33  borchert
   Initial revision

   ----------------------------------------------------------------------------
*)

IMPLEMENTATION MODULE GenTypes; (* AFB 5/89 *)

   (* generate code for
      1) type guards
      2) allocation
      3) type tags

      interface to the Oberon module Storage:
				     ~~~~~~~~

	 TYPE
	    Address = LONGINT;
	    Size = LONGINT;

	 VAR
	    next: Address; left: Size;
	       (* the interval [next, next+left) marks an interval
		  which may be consumed by the inline code of NEW
		  without calling any of the procedures below
	       *)

	 PROCEDURE AllocateWithTag(VAR ptr: Address; tag: Address; size: Size);
	    (* allocation with tag field;
	       the size includes the tag field
	    *)

      
      type tags:

      O_module_type_TAG:
		long	size
		long	module		# pointer to module record
		long	typeno		# of corresponding reference file
		long	hierlen		# number of base types
		long	basetype_1
		long	basetype_2	# extends basetype_1
		...
		# pointer list
		long	sentinel	# marks end of pointer list

      don't change this structure without updating the associated
      modules in the Oberon system library
   *)

   FROM Attributes IMPORT Attribute, Reg, Label, top, NewAttribute, AtMode,
      GetLabel, TestType, ArithmeticType, base;
   FROM EmitCode IMPORT SetSegment, text, StrEmit, StrEmit1, StrEmit2,
      Emit, Emit1, Emit2, Emit3, Emit4, Emit5, Emit6, EmitLabel, EmitAlign4;
   FROM Exception IMPORT Assert;
   FROM GenBasicOps IMPORT Address, OffsetAt, DereferenceAt, ReturnAt, ccat,
      ReleaseAt, MoveBytesAt, Load, LoadAddr, ConstDivReg, GenTest, AlignReg,
      Max;
   FROM Lex IMPORT Symbol, AddLegalOptions, CompilerOptions, options;
   FROM Machine IMPORT nilvalue, oneword, Align, stackdirection, Direction;
   FROM Memory IMPORT ALLOCATE, DEALLOCATE;
   FROM Mnemonics IMPORT Mnemonic;
   FROM RegMan IMPORT GetReg, FreeReg, SaveRegs,
      RestoreRegs, GetAddrReg, GetAnyReg;
   FROM StackMan IMPORT PointerList;
   FROM Standard IMPORT crptr, longptr;
   FROM SymTab IMPORT mainmod, VarKind, Type, Form, FieldList, Ident, Size,
      IdentList, IdentClass, globalvars, address;

   CONST
      maskTagsOpt = "M";
	 (* if set, the two low order bits of tag references will be ignored;
	    some garbage collectors require this option to be applied to
	    the whole program
	 *)
      tagMask = {0..29};

   CONST
      minAllocLen = 0;
	 (* minimal length to be allocated (beside the tag field);
	    this is larger than the actual size in case of empty records;
	    some garbage collectors need this space (e.g. copying
	    collectors need it to store the new address of the object)

	    this miminal length does not enlarge the size of the type
	    because this additional storage is only needed for
	    dynamic allocation
	    --
	    set to 0, because nobody needs this feature in the moment afb 3/94
	 *)

   CONST
      tagOffset = -oneword;
      sizeOffset = 0;
      moduleOffset = oneword;
      typenoOffset = 2 * oneword;
      hierlenOffset = 3 * oneword;
      basetypeOffset = 4 * oneword;
      sentinel = MIN(Size); (* marks end of pointer list *)
	 (* don't change sentinel without changing the other copies of
	    this constant in Coroutines, SysStorage and ...
	 *)
      dummyTagRecLen = basetypeOffset + oneword;
   TYPE
      TypeList = POINTER TO TypeInList;
      TypeInList =
	 RECORD
	    type: Type;
	    link: TypeList;
	 END;
   VAR
      (* to avoid generation of unnecessary type tags
	 we maintain a list of types for which a type tag is needed;
	 the tags are generated by GenTypes
      *)
      nexttagno: CARDINAL;
      typelist: TypeList;

   PROCEDURE Prepare(VAR at: Attribute; VAR varparam: BOOLEAN);
      (* prepare `at' in a way which allows us to access the type tag
	 and later the pointer or record itself:
	 + GetTag allows to access the type tag
	 + after "IF varparam THEN DereferenceAt(at) END" the
	   pointer or record may be accessed
      *)
   BEGIN
      WITH at^ DO
	 varparam := (mode = varAt) & (atip^.varkind = varparamV);
      END;
      IF varparam THEN
	 IndirectAddress(at);
      ELSE
	 Address(at);
      END;
   END Prepare;

   PROCEDURE ApplyMaskToTagReg(tagReg: Reg);
      TYPE
	 RegSet = SET OF Reg;
   BEGIN
      IF maskTagsOpt IN options THEN
	 IF tagReg IN RegSet{d0..d7} THEN
	    Emit2(ANDop, "%L%C,%r", tagMask, tagReg);
	 ELSE
	    Emit2(MOVE, "%L%r,%r", tagReg, d0);
	    Emit2(ANDop, "%L%C,%r", tagMask, d0);
	    Emit2(MOVE, "%L%r,%r", d0, tagReg);
	 END;
      END;
   END ApplyMaskToTagReg;

   PROCEDURE GetTag(VAR at: Attribute; varparam: BOOLEAN; tagReg: Reg);
      (* PRE: Prepare has been called for `at';
	 load type tag of `at' into given register
      *)
      VAR
	 applyMask: BOOLEAN;
   BEGIN
      applyMask := TRUE;
      IF varparam THEN
	 OffsetAt(at, oneword);
	 Emit2(MOVE, "%L%a,%r", at, tagReg);
	 OffsetAt(at, -oneword);
	 applyMask := FALSE;
      ELSIF at^.attype^.form = record THEN
	 IF at^.tagged THEN
	    OffsetAt(at, tagOffset);
	    Emit2(MOVE, "%L%a,%r", at, tagReg);
	    OffsetAt(at, -tagOffset);
	 ELSE (* take static type *)
	    AddTypeToList(at^.attype);
	    Emit2(MOVE, "%L%#%t,%r", at^.attype, tagReg);
	    applyMask := FALSE;
	 END;
      ELSE (* pointer # NIL *)
	 Emit2(MOVE, "%L%a,%r", at, tagReg);
	 Emit3(MOVE, "%L%(d,r),%r", tagOffset, tagReg, tagReg);
      END;
      IF applyMask THEN
	 ApplyMaskToTagReg(tagReg);
      END;
   END GetTag;

   PROCEDURE GenAssignmentTest(tagReg, tag2Reg: Reg; copyLoop: Label);
      (* generate type test for assignments of records and pointers;
	 tagReg and tag2Reg must be address registers;
	 if tag2Reg extends tagReg then continue at copyLoop;
	 the code which handles the failure is to be appended

	 tagReg ---> +------+      tag2Reg ---> +------+
		     | size |                   | size |
		     +------+                   +------+
		     | i    |                   | >= i |
		     +------+                   +------+
		     | bt1  |                   | bt1  |
		     +------+                   +------+
		     | ...  |                   | ...  |
		     +------+                   +------+
		     | bti-1|                   | bti-1|
		     +------+                   +------+
						| bti  | equal to tagReg?
						+------+
      *)
      VAR
	 failure: Label;
   BEGIN
      GetLabel(failure);

      (* are both tags equal? *)
      Emit2(CMP, "%L%r,%r", tagReg, tag2Reg);
      Emit1(BEQ, "%l", copyLoop);

      (* assure that the hierlen of tag2Reg exceeds that of tagReg *)
      Emit3(MOVE, "%L%(d,r),%r", hierlenOffset, tagReg, d0);
      Emit3(CMP, "%L%r,%(d,r)", d0, hierlenOffset, tag2Reg);
      Emit1(BGE, "%l", failure);

      (* check tagReg to be equal to tagReg2.bt[tagReg.hierlen] *)
      Emit4(CMP, "%L%r,%(d,r,r4)", tagReg, basetypeOffset, tag2Reg, d0);
      Emit1(BEQ, "%l", copyLoop);

      EmitLabel(failure);
   END GenAssignmentTest;

   PROCEDURE GetRecSize(at: Attribute; VAR sizeat: Attribute);
      (* get allocated size *)
      VAR
	 sizeReg: Reg;
	 varparam: BOOLEAN;
   BEGIN
      Assert(at^.attype^.form = record);
      Prepare(at, varparam);
      NewAttribute(sizeat);
      IF varparam OR at^.tagged THEN
	 GetReg(sizeReg); GetTag(at, varparam, sizeReg);
	 IF varparam THEN DereferenceAt(at) END;
	 Emit3(MOVE, "%L%(d,r),%r", sizeOffset, sizeReg, sizeReg);
	 sizeat^.mode := regMode;
	 sizeat^.reg := sizeReg;
	 ccat := sizeat;
      ELSE
	 (* take at^.attype^.size *)
	 sizeat^.mode := constAt;
	 sizeat^.cval.sy := intcon;
	 sizeat^.cval.intval := at^.attype^.size;
	 ccat := NIL;
      END;
      sizeat^.attype := longptr;
   END GetRecSize;

   PROCEDURE AssignRecs(desat, exprat: Attribute);
      (* assign `exprat' to `desat'; varAt permitted *)
      VAR
	 varparamL, varparamR: BOOLEAN;
	 sizeReg, tagReg, tag2Reg: Reg;
	 copyLoop, loopLabel, endOfLoop: Label;
	 tagRegAllocated: BOOLEAN;
   BEGIN
      Assert((desat^.attype^.form = record) &
	     (exprat^.attype^.form = record));
      Prepare(desat, varparamL); Prepare(exprat, varparamR);
      tagRegAllocated := FALSE;
      IF varparamL OR desat^.tagged THEN
	 (* in this case the designator could be an extension
	    of the expression
	 *)
	 GetAddrReg(tagReg); GetTag(desat, varparamL, tagReg);
	 tagRegAllocated := TRUE;
	 GetLabel(copyLoop);
	 (* type test necessary *)
	 GetAddrReg(tag2Reg); GetTag(exprat, varparamR, tag2Reg);
	 GenAssignmentTest(tagReg, tag2Reg, copyLoop);
	 FreeReg(tag2Reg);
	 Emit(JSR, "%_RTErrors_TypeGuardFailure");
	 StrEmit("%* NOT REACHED");
	 EmitLabel(copyLoop);
      END;
      IF varparamL THEN DereferenceAt(desat) END;
      IF varparamR THEN DereferenceAt(exprat) END;

      (* we know the # of bytes to be copied if
	 (1) the type of designator is statically known
	     (the type guard guarantees that expr is extended enough), or if
	 (2) (desat^.attype = exprat^.attype) &
	     type of expr is statically known
	     (in this case static type of des = static & dyn type of expr
	     and the type guarantees that dynamic type of des equals the
	     static type of des)
      *)
      IF (varparamL OR desat^.tagged) &
	    ((desat^.attype # exprat^.attype) OR
	     varparamR OR exprat^.tagged) THEN
	 (* # of bytes to be copied is not known in advance *)
	 GetLabel(endOfLoop);
	 GetReg(sizeReg);
	 Assert(tagRegAllocated);
	 Emit3(MOVE, "%L%(d,r),%r", sizeOffset, tagReg, sizeReg);
	 FreeReg(tagReg);

	 (* check for size = 0 *)
	 IF desat^.attype^.size = 0 THEN
	    Emit1(TST, "%L%r", sizeReg);
	    Emit1(BEQ, "%l", endOfLoop);
	 END;

	 LoadAddr(desat); LoadAddr(exprat);
	 ConstDivReg(intAT, sizeReg, oneword);
	 GetLabel(loopLabel);
	 EmitLabel(loopLabel);
	 Emit2(MOVE, "%L%+r,%+r", exprat^.reg, desat^.reg);
	 Emit2(SUB, "%L%C,%r", 1, sizeReg);
	 Emit1(BNE, "%l", loopLabel);
	 EmitLabel(endOfLoop);
	 ReleaseAt(exprat); ReleaseAt(desat);
	 FreeReg(sizeReg);
      ELSE
	 IF tagRegAllocated THEN
	    FreeReg(tagReg);
	 END;
	 MoveBytesAt(exprat, desat, desat^.attype^.size);
      END;
      ccat := NIL;
   END AssignRecs;

   PROCEDURE AssignPtrs(desat, exprat: Attribute);
      (* assign `exprat' to `desat'; varAt permitted *)
      VAR
	 varparamL, varparamR: BOOLEAN;
	 tagReg, tag2Reg: Reg;
	 copyLabel: Label;
   BEGIN
      Prepare(desat, varparamL); Prepare(exprat, varparamR);
      IF varparamL & (desat^.attype^.reftype^.form = record) &
	    exprat^.attype^.taggedptr THEN
	 GetAddrReg(tagReg); GetTag(desat, varparamL, tagReg);
	 GetLabel(copyLabel);
	 IF ~varparamR THEN
	    (* check for NIL *)
	    Emit1(TST, "%L%a", exprat);
	    Emit1(BEQ, "%l", copyLabel);
	 END;
	 GetAddrReg(tag2Reg); GetTag(exprat, varparamR, tag2Reg);
	 GenAssignmentTest(tagReg, tag2Reg, copyLabel);
	 StrEmit("%* right hand pointer is not an extension of left hand pointer");
	 Emit(JSR, "%_RTErrors_TypeGuardFailure");
	 StrEmit("%* NOT REACHED");
	 EmitLabel(copyLabel);
	 FreeReg(tagReg); FreeReg(tag2Reg);
      END;
      IF varparamL THEN DereferenceAt(desat) END;
      IF varparamR THEN DereferenceAt(exprat) END;
      Emit2(MOVE, "%L%a,%a", exprat, desat);
      ReleaseAt(exprat); ReleaseAt(desat);
   END AssignPtrs;

   PROCEDURE IndirectAddress(at: Attribute);
      (* mode = varAt *)
      VAR ip: Ident;
   BEGIN
      ip := at^.atip;
      Assert(ip^.indirect);
      ip^.indirect := FALSE;
      Address(at);
      ip^.indirect := TRUE;
   END IndirectAddress;

   PROCEDURE PushTag(VAR at: Attribute);
      (* push tag of `at' onto stack; `at' is not modified if mode # varAt;
	 `mode' one of DesignatorModes
      *)
      VAR
	 varparam: BOOLEAN;
	 rectype: Type;
   BEGIN
      WITH at^ DO
	 WITH attype^ DO
	    IF form = pointer THEN
	       rectype := reftype;
	    ELSE
	       rectype := attype;
	    END;
	 END;
	 varparam := (mode = varAt) & (atip^.varkind = varparamV) &
		     (rectype^.form = record);
	 IF varparam THEN
	    IndirectAddress(at);
	    OffsetAt(at, oneword);
	    Emit2(MOVE, "%L%a,%-r", at, top);
	    OffsetAt(at, -oneword);
	    DereferenceAt(at);
	 ELSE
	    AddTypeToList(rectype);
	    Address(at);
	    IF at^.tagged THEN
	       OffsetAt(at, -oneword);
	       IF maskTagsOpt IN options THEN
		  Emit2(MOVE, "%L%a,%r", at, d0);
		  ApplyMaskToTagReg(d0);
	       ELSE
		  Emit2(MOVE, "%L%a,%-r", at, top);
	       END;
	       OffsetAt(at, oneword);
	    ELSE
	       Emit2(MOVE, "%L%#%t,%-r", rectype, top);
	    END;
	 END;
      END;
   END PushTag;

   PROCEDURE GenTypeTest(VAR at: Attribute; type: Type;
			 trueLab, falseLab: Label;
			 readonly: BOOLEAN);
      (* common part of TypeGuard and TypeTest;
	 readonly specifies whether at may be modified or not;
	 jump to trueLab on success and
	 jump to falseLab on failure or fall thru (if readonly = TRUE)

	 tag of at --->  +------+
			 | size |
			 +------+
			 | hlen |
			 +------+
			 | bt1  |
			 +------+
			 | ...  |
			 +------+
			 | bti  | <--- static type of at
			 +------+
			 | ...  |
			 +------+
			 | btj  | <--- test type?
			 +------+

	 summary:
	    we need to check if the dynamic type of at equals the
	       test type because in this case we don't have btj
	       (the table ends just before)
	    the type test fails if hlen is less than that of the test type 

	    only after this two preliminary tests we can check
	    btj for being equal to test type
      *)
      VAR
	 varparam: BOOLEAN; (* tagged VAR-parameter? *)
	 rectype: Type;
	 tagReg: Reg;

	 hierlen: CARDINAL;
	 parent: Type;
   BEGIN
      IF type^.form = pointer THEN
	 rectype := type^.reftype;
      ELSE
	 rectype := type;
      END;
      Assert(rectype^.form = record);
      AddTypeToList(rectype); AddTypeToList(at^.attype);

      WITH at^ DO
	 varparam := (mode = varAt) & (atip^.varkind = varparamV) &
		     (rectype^.form = record);
	 GetReg(tagReg);
	 IF varparam & (type^.form = record) THEN
	    IndirectAddress(at);
	    OffsetAt(at, oneword);
	    Emit2(MOVE, "%L%a,%r", at, tagReg);
	    IF readonly THEN
	       OffsetAt(at, -oneword);
	       DereferenceAt(at);
	    END;
	 ELSE
	    Assert(attype^.form = pointer);
	    Address(at);
	    IF readonly THEN
	       Emit2(MOVE, "%L%a,%r", at, tagReg);
	       Emit3(MOVE, "%L%(d,r),%r", tagOffset, tagReg, tagReg);
	    ELSE
	       DereferenceAt(at);
	       OffsetAt(at, tagOffset);
	       Emit2(MOVE, "%L%a,%r", at, tagReg);
	    END;
	    ApplyMaskToTagReg(tagReg);
	 END;

	 (* check the type tag for being equal to the test type *)
	 WITH rectype^ DO
	    IF (ident # NIL) & ident^.exported THEN
	       Emit3(CMP, "%L%r,%#%_%n_%n_TAG",
			  tagReg, ident^.mod^.name, ident^.name);
	    ELSE
	       Emit3(CMP, "%L%r,%#%_%n_%c_TAG",
			  tagReg, mainmod^.name, tagno);
	    END;
	    Emit1(BEQ, "%l", trueLab);
	 END;

	 (* assure that tagReg.hierlen exceeds that of hierlen *)
	 parent := rectype^.basetype; hierlen := 0;
	 WHILE parent # NIL DO
	    INC(hierlen); parent := parent^.basetype;
	 END;
	 Emit3(CMP, "%L%(d,r),%C", hierlenOffset, tagReg, hierlen);
	 Emit1(BLE, "%l", falseLab);

	 (* check tagReg.bt[hierlen] against the test type *)
	 Emit3(CMP, "%L%(d,r),%#%t",
	      basetypeOffset + hierlen * oneword, tagReg,
	      rectype);
	 IF readonly THEN
	    Emit1(BEQ, "%l", trueLab);
	 END;
	 FreeReg(tagReg);
      END;
      IF readonly THEN
	 EmitLabel(falseLab);
      ELSE
	 ReleaseAt(at);
	 WITH at^ DO
	    mode := condMode;
	    test := eq;
	    atype := intAT;
	    tlabel := trueLab;
	    flabel := falseLab;
	 END;
	 ccat := at;
      END;
   END GenTypeTest;

   PROCEDURE TypeGuard(VAR at: Attribute; type: Type);
      (* generate code for type guard; `at' is not modified if mode # varAt *)
      VAR
	 trueLab, falseLab: Label;
   BEGIN
      GetLabel(trueLab); GetLabel(falseLab);
      GenTypeTest(at, type, trueLab, falseLab, TRUE);
      StrEmit("%* type guard failed");
      Emit(JSR, "%_RTErrors_TypeGuardFailure");
      StrEmit("%* NOT REACHED");
      EmitLabel(trueLab);
   END TypeGuard;

   PROCEDURE TypeTest(VAR at: Attribute; type: Type;
		      VAR trueLab, falseLab: Label);
      (* set condition codes according to `at' IS `type';
	 at^.mode = varAt permitted
	 the execution continues at `trueLab' in case of TRUE;
	 GetLabel is called for trueLab and/or falseLab if
	 trueLab/falseLab.ok is not TRUE
      *)
   BEGIN
      IF ~trueLab.ok THEN
	 GetLabel(trueLab);
      END;
      IF ~falseLab.ok THEN
	 GetLabel(falseLab);
      END;
      GenTypeTest(at, type, trueLab, falseLab, FALSE);
   END TypeTest;

   PROCEDURE Allocate(VAR at: Attribute);
      (* PRE: at^.mode IN GenModes + AtModeSet{varAt}
	      at^.attype must be a pointer type
	 NEW is executed for at^.attype (GenModes or untagged varAt)
			     type tag   (tagged varAt)
	 `at' is released
      *)
      VAR
	 varparam: BOOLEAN; (* tagged VAR-parameter? *)

      PROCEDURE AllocFixed(VAR at: Attribute);
	 (* allocate sth with known size *)
	 VAR
	    length: Size;
	    loopLabel, allocLabel, endLabel: Label;
	    oldNextReg, newNextReg: Reg;
	    oldLeftReg, newLeftReg: Reg;
	    addrNextReg, addrLeftReg: Reg;
	    ptrReg: Reg;
      BEGIN
	 Address(at);

	 length := at^.attype^.reftype^.size;
	 IF length < minAllocLen THEN
	    length := minAllocLen;
	 END;
	 INC(length, oneword); (* for tag field *)
	 Align(length);
	 GetLabel(loopLabel); GetLabel(allocLabel); GetLabel(endLabel);
	 EmitLabel(loopLabel);
	 Emit1(CMP, "%L%_Storage_left,%C", length);
	 GenTest(lt, logAT, allocLabel);

	 GetReg(oldNextReg); GetReg(newNextReg);
	 GetReg(oldLeftReg); GetReg(newLeftReg);
	 GetAnyReg(addrNextReg); GetAnyReg(addrLeftReg);
	 Emit1(MOVE, "%L%#%_Storage_next,%r", addrNextReg);
	 Emit1(MOVE, "%L%_Storage_next,%r", oldNextReg);
	 Emit2(MOVE, "%L%r,%r", oldNextReg, newNextReg);
	 Emit2(ADD, "%L%C,%r", length, newNextReg);
	 Emit1(MOVE, "%L%#%_Storage_left,%r", addrLeftReg);
	 Emit1(MOVE, "%L%_Storage_left,%r", oldLeftReg);
	 Emit2(MOVE, "%L%r,%r", oldLeftReg, newLeftReg);
	 Emit2(SUB, "%L%C,%r", length, newLeftReg);

	 Emit6(CAS2, "%L%:r,%:r,%:i",
	    oldNextReg, oldLeftReg,
	    newNextReg, newLeftReg,
	    addrNextReg, addrLeftReg);
	 GenTest(ne, logAT, loopLabel);
	 FreeReg(oldLeftReg);
	 FreeReg(newNextReg); FreeReg(newLeftReg);
	 FreeReg(addrNextReg); FreeReg(addrLeftReg);

	 GetAddrReg(ptrReg);
	 Emit2(MOVE, "%L%r,%r", oldNextReg, ptrReg); FreeReg(oldNextReg);
	 Emit2(MOVE, "%L%#%t,%+r", at^.attype^.reftype, ptrReg);
	 Emit2(MOVE, "%L%r,%a", ptrReg, at);
	 FreeReg(ptrReg);
	 Emit1(BRA, "%l", endLabel);

	 EmitLabel(allocLabel);
	 SaveRegs;
	 Emit1(PEA, "%L%a", at); (* VAR ptr: Address *)
	 Emit2(MOVE, "%L%#%t,%-r", at^.attype^.reftype, top); (* tag: Address *)
	 Emit2(MOVE, "%L%C,%-r", length, top); (* size: Size *)
	 Emit(JSR, "%_Storage_AllocateWithTag");
	 RestoreRegs;
	 EmitLabel(endLabel);
      END AllocFixed;

      PROCEDURE AllocVarParam(VAR at: Attribute);
	 (* allocate by using a ptr which is a VAR parameter *)
	 VAR
	    loopLabel, allocLabel, endLabel: Label;
	    oldNextReg, newNextReg: Reg;
	    oldLeftReg, newLeftReg: Reg;
	    addrNextReg, addrLeftReg: Reg;
	    ptrReg: Reg;
	    addrOfPtrReg: Reg; tagReg: Reg; sizeReg: Reg;
	    minlenReg: Reg;
      BEGIN
	 IndirectAddress(at);
	 GetAddrReg(addrOfPtrReg); GetReg(tagReg); GetReg(sizeReg);
	 Emit2(MOVE, "%L%a,%r", at, addrOfPtrReg);
	 OffsetAt(at, oneword);
	 Emit2(MOVE, "%L%a,%r", at, tagReg);
	 DereferenceAt(at);
	 OffsetAt(at, sizeOffset);
	 Emit2(MOVE, "%L%a,%r", at, sizeReg);
	 IF minAllocLen > 0 THEN
	    IF at^.attype^.reftype^.size < minAllocLen THEN
	       GetReg(minlenReg);
	       Emit2(MOVE, "%L%C,%r", minAllocLen, minlenReg);
	       Max(sizeReg, minlenReg);
	       FreeReg(minlenReg);
	    END;
	 END;
	 Emit2(ADD, "%L%C,%r", oneword, sizeReg); (* add tag field to size *)
	 AlignReg(sizeReg);

	 GetLabel(loopLabel); GetLabel(allocLabel); GetLabel(endLabel);
	 EmitLabel(loopLabel);
	 Emit1(CMP, "%L%r,%_Storage_left", sizeReg);
	 GenTest(gt, logAT, allocLabel);

	 GetReg(oldNextReg); GetReg(newNextReg);
	 GetReg(oldLeftReg); GetReg(newLeftReg);
	 GetAnyReg(addrNextReg); GetAnyReg(addrLeftReg);
	 Emit1(MOVE, "%L%#%_Storage_next,%r", addrNextReg);
	 Emit1(MOVE, "%L%_Storage_next,%r", oldNextReg);
	 Emit2(MOVE, "%L%r,%r", oldNextReg, newNextReg);
	 Emit2(ADD, "%L%r,%r", sizeReg, newNextReg);
	 Emit1(MOVE, "%L%#%_Storage_left,%r", addrLeftReg);
	 Emit1(MOVE, "%L%_Storage_left,%r", oldLeftReg);
	 Emit2(MOVE, "%L%r,%r", oldLeftReg, newLeftReg);
	 Emit2(SUB, "%L%r,%r", sizeReg, newLeftReg);

	 Emit6(CAS2, "%L%:r,%:r,%:i",
	    oldNextReg, oldLeftReg,
	    newNextReg, newLeftReg,
	    addrNextReg, addrLeftReg);
	 GenTest(ne, logAT, loopLabel);
	 FreeReg(oldLeftReg);
	 FreeReg(newNextReg); FreeReg(newLeftReg);
	 FreeReg(addrNextReg); FreeReg(addrLeftReg);

	 GetAddrReg(ptrReg);
	 Emit2(MOVE, "%L%r,%r", oldNextReg, ptrReg); FreeReg(oldNextReg);
	 Emit2(MOVE, "%L%r,%+r", tagReg, ptrReg);
	 Emit2(MOVE, "%L%r,%(r)", ptrReg, addrOfPtrReg);
	 FreeReg(ptrReg);
	 Emit1(BRA, "%l", endLabel);

	 EmitLabel(allocLabel);
	 SaveRegs;
	 Emit2(MOVE, "%L%r,%-r", addrOfPtrReg, top); (* VAR ptr: Address *)
	 Emit2(MOVE, "%L%r,%-r", tagReg, top); (* tag: Address *)
	 Emit2(MOVE, "%L%r,%-r", sizeReg, top); (* size: Size *)
	 Emit(JSR, "%_Storage_AllocateWithTag");
	 RestoreRegs;
	 FreeReg(addrOfPtrReg); FreeReg(tagReg); FreeReg(sizeReg);
	 EmitLabel(endLabel);
      END AllocVarParam;

   BEGIN (* Allocate *)
      WITH at^ DO
	 Assert(attype^.form = pointer);
	 AddTypeToList(attype);
	 varparam := (mode = varAt) & (atip^.varkind = varparamV) &
		     (attype^.form = pointer) &
		     (attype^.reftype^.form = record);
	 IF varparam THEN
	    AllocVarParam(at);
	 ELSE
	    AllocFixed(at);
	 END;
	 ReturnAt(at);
      END;
      ccat := NIL;
   END Allocate;

   PROCEDURE SysAllocate(VAR at, size: Attribute);
      (* PRE: at^.mode IN GenModes + AtModeSet{varAt}
	      at^.attype must be a pointer type
	      size^.mode IN GetModes
	 SYS.NEW is executed for at
	 `at' and `size' are released
      *)
      VAR
	 loopLabel, allocLabel, endLabel: Label;
	 oldNextReg, newNextReg: Reg;
	 oldLeftReg, newLeftReg: Reg;
	 addrNextReg, addrLeftReg: Reg;
	 ptrReg: Reg;
	 tagReg: Reg;
	 sizeReg: Reg;
	 minlenReg: Reg;
   BEGIN
      WITH at^ DO
	 Assert(attype^.form = pointer);
	 Address(at);
	 Load(size);

	 (* assure minimal length, add tag field and tag record to size *)
	 IF minAllocLen > 0 THEN
	    GetReg(minlenReg); Emit2(MOVE, "%L%C,%r", minAllocLen, minlenReg);
	    Max(size^.reg, minlenReg); FreeReg(minlenReg);
	 END;
	 AlignReg(size^.reg);
	 GetAddrReg(tagReg); Emit2(MOVE, "%A%a,%r", size, tagReg);
	 Emit2(ADD, "%A%C,%a", oneword + dummyTagRecLen, size);

	 GetLabel(loopLabel); GetLabel(allocLabel); GetLabel(endLabel);
	 EmitLabel(loopLabel);
	 Emit1(CMP, "%A%a,%_Storage_left", size);
	 GenTest(gt, logAT, allocLabel);

	 GetReg(oldNextReg); GetReg(newNextReg);
	 GetReg(oldLeftReg); GetReg(newLeftReg);
	 GetAnyReg(addrNextReg); GetAnyReg(addrLeftReg);
	 Emit1(MOVE, "%L%#%_Storage_next,%r", addrNextReg);
	 Emit1(MOVE, "%L%_Storage_next,%r", oldNextReg);
	 Emit2(MOVE, "%L%r,%r", oldNextReg, newNextReg);
	 Emit2(ADD, "%A%a,%r", size, newNextReg);
	 Emit1(MOVE, "%L%#%_Storage_left,%r", addrLeftReg);
	 Emit1(MOVE, "%L%_Storage_left,%r", oldLeftReg);
	 Emit2(MOVE, "%L%r,%r", oldLeftReg, newLeftReg);
	 Emit2(SUB, "%A%a,%r", size, newLeftReg);

	 Emit6(CAS2, "%L%:r,%:r,%:i",
	    oldNextReg, oldLeftReg,
	    newNextReg, newLeftReg,
	    addrNextReg, addrLeftReg);
	 GenTest(ne, logAT, loopLabel);
	 FreeReg(oldLeftReg);
	 FreeReg(newNextReg); FreeReg(newLeftReg);
	 FreeReg(addrNextReg); FreeReg(addrLeftReg);

	 GetAddrReg(ptrReg);
	 Emit2(MOVE, "%L%r,%r", oldNextReg, ptrReg); FreeReg(oldNextReg);
	 GetReg(sizeReg); Emit2(MOVE, "%L%r,%r", tagReg, sizeReg);
	 Emit2(ADD, "%L%r,%r", ptrReg, tagReg);
	 Emit2(ADD, "%L%C,%r", oneword, tagReg);
	 Emit2(MOVE, "%L%r,%+r", tagReg, ptrReg);
	 (* fill dummy tag record *)
	 Emit2(ADD, "%L%C,%r", dummyTagRecLen, sizeReg);
	 Emit2(MOVE, "%L%r,%+r", sizeReg, tagReg); (* size *)
	 Emit2(MOVE, "%L%C,%+r", nilvalue, tagReg); (* module *)
	 Emit2(MOVE, "%L%C,%+r", 0, tagReg); (* typeno *)
	 Emit2(MOVE, "%L%C,%+r", 0, tagReg); (* hierlen *)
	 Emit2(MOVE, "%L%C,%+r", sentinel, tagReg); (* end of ptr list *)
	 FreeReg(tagReg); FreeReg(sizeReg);
	 (* return pointer value *)
	 Emit2(MOVE, "%L%r,%a", ptrReg, at);
	 FreeReg(ptrReg);
	 Emit1(BRA, "%l", endLabel);

	 EmitLabel(allocLabel);
	 SaveRegs;
	 Emit1(PEA, "%L%a", at); (* VAR ptr: Address *)
	 Emit2(MOVE, "%L%C,%-r", nilvalue, top); (* tag: Address *)
	 Emit2(MOVE, "%A%a,%-r", size, top); (* size: Size *)
	 Emit(JSR, "%_Storage_AllocateWithTag");
	 RestoreRegs;
	 EmitLabel(endLabel);
      END;
      ReturnAt(at); ReturnAt(size);
      ccat := NIL;
   END SysAllocate;

   PROCEDURE AddTypeToList(type: Type);
      VAR new: TypeList;
   BEGIN
      IF (type # NIL) & (type^.form = pointer) THEN
	 type := type^.reftype;
      END;
      IF (type # NIL) &
	    ((type^.form = record) OR (type^.form = array)) THEN
	 WITH type^ DO
	    IF (tagno = 0) &
	       ((ident = NIL) OR (ident^.mod^.name = mainmod^.name)) THEN
	       tagno := nexttagno; INC(nexttagno);
	       NEW(new);
	       new^.type := type;
	       new^.link := typelist;
	       typelist := new;
	    END;
	    IF (type^.form = record) & (basetype # NIL) &
		  ((basetype^.ident = NIL) OR ~basetype^.ident^.exported) &
		  (basetype^.tagno = 0) THEN
	       AddTypeToList(basetype);
	    END;
	 END;
      END;
   END AddTypeToList;

   PROCEDURE GenTagReference(type: Type);
   BEGIN
      WITH type^ DO
	 IF (ident # NIL) & ident^.exported THEN
	    StrEmit2("%:L%_%n_%n_TAG", ident^.mod^.origname, ident^.name);
	 ELSE
	    IF tagno = 0 THEN
	       AddTypeToList(type);
	       Assert(tagno # 0);
	    END;
	    StrEmit2("%:L%_%n_%c_TAG", mainmod^.origname, tagno);
	 END;
      END;
   END GenTagReference;

   CONST
      (* see comments below in GenPointerList *)
      tagSimplePointer = {};
      tagPointerArray = { 31 };
      tagOther = { 30 };
      tagRecordArray = { 30, 31 };

      (* selector for tagOther *)
      TmpPointerSY = 0;
      SimpleAddressSY = 1;
      AddressArraySY = 2;

   PROCEDURE EmitOffset(offset: Size; tag: BITSET);
   BEGIN
      StrEmit1("%:L%c", Size(BITSET(offset) + tag));
   END EmitOffset;

   PROCEDURE EmitSelector(selector: CARDINAL);
   BEGIN
      StrEmit1("%:L%c", selector);
   END EmitSelector;

   PROCEDURE GenPointerList(type: Type; offset: Size);
      (* generate the pointer list for `type' without sentinel:

	 PointerList = { PtrDescriptor } .
	 PtrDescriptor = SimplePointer | PointerArray |
			 RecordArray | Other .
	 SimplePointer = offset .
	 PointerArray = offset ElementCount .
	 RecordArray = offset ElementCount typeTag .
	 ElementCount = [ numberOfDynDimensions ] count .
	 Other = offset ( TmpPointer | SimpleAddress | AddressArray ) .
	 TmpPointer = TmpPointerSY StartAddr EndAddr .
	 SimpleAddress = SimpleAddressSY
	 AddressArray = AddressArraySY ElementCount .
	 StartAddr = addr .
	 EndAddr = addr .

	 bit0 = least significant bit

	 bit0 \ bit1 | 0             | 1
	 ------------+---------------+-------------
	 0           | SimplePointer | Other
	 1           | PointerArray  | RecordArray

	 selector for other pointers (4 bytes each):

	 TmpPointerSY = 0
	 SimpleAddressSY = 1
	 AddressArraySY = 2

	 temporary pointers are only to be considered if the associated
	 program counter ranges inside [StartAddr..EndAddr)

	 while pointers are guaranteed to point to the beginning
	 of a structure, addresses are free to point inside a structure
	 and they may reference other areas (eg global or local data).

	 basetypes are encoded as RecordArray with a length of 1
	 if a record contains private parts

	 dynamic arrays are encoded with numberOfDynDimensions = - n for
	 n-dimensional arrays

	 some ideas have been stolen from:
	       C. A. Lins, Generational Garbage Collection in Oberon,
	       Proceedings of the 2nd International Modula-2 Conference,
	       1991 Loughborough University, UK
      *)

      PROCEDURE GenRecord(type: Type);
	 VAR
	    field: FieldList;
      BEGIN
	 WITH type^ DO
	    IF (basetype # NIL) & basetype^.privateparts THEN
	       (* there may be some pointers we don't know about:
		  generate a pointer list for our components only
		  and a RecordArray-entry for the basetype
	       *)
	       field := fields;
	       WHILE field # basetype^.fields DO
		  Assert(field # NIL);
		  GenPointerList(field^.type, offset + field^.offset);
		  field := field^.link;
	       END;
	       EmitOffset(offset, tagRecordArray);
	       StrEmit1("%:L%c", 1);
	       GenTagReference(basetype);
	    ELSE
	       (* we know of all record fields *)
	       field := fields;
	       WHILE field # NIL DO
		  GenPointerList(field^.type, offset + field^.offset);
		  field := field^.link;
	       END;
	    END;
	 END;
      END GenRecord;

      PROCEDURE GenArray(type: Type);
	 VAR
	    elementCount: Size; (* number of elements incl. all subarrays *)
	    dimsOfDynArr: Size; (* number of dynamic array dimensions *)
	    eltype: Type;       (* element type *)
      BEGIN
	 WITH type^ DO
	    eltype := element;
	    IF dyn THEN
	       elementCount := 1;
	       dimsOfDynArr := 1;
	       WHILE (eltype^.form = array) & eltype^.dyn DO
		  INC(dimsOfDynArr);
		  eltype := eltype^.element;
	       END;
	    ELSE
	       elementCount := length;
	    END;
	    WHILE eltype^.form = array DO
	       (* calculate total number of elements
		  for multidimensional arrays
	       *)
	       elementCount := elementCount * eltype^.length;
	       eltype := eltype^.element;
	    END;
	    IF eltype^.form = pointer THEN
	       EmitOffset(offset, tagPointerArray);
	       IF dyn THEN
		  StrEmit1("%:L%c", -dimsOfDynArr);
	       END;
	       StrEmit1("%:L%c", elementCount);
	    ELSIF eltype^.form = address THEN
	       EmitOffset(offset, tagOther);
	       EmitSelector(AddressArraySY);
	       IF dyn THEN
		  StrEmit1("%:L%c", -dimsOfDynArr);
	       END;
	       StrEmit1("%:L%c", elementCount);
	    ELSE
	       EmitOffset(offset, tagRecordArray);
	       IF dyn THEN
		  StrEmit1("%:L%c", -dimsOfDynArr);
	       END;
	       StrEmit1("%:L%c", elementCount);
	       GenTagReference(eltype);
	    END;
	 END;
      END GenArray;

   BEGIN (* GenPointerList *)
      WITH type^ DO
	 IF containsptr THEN
	    Assert(offset MOD oneword = 0);
	    CASE form OF
	    | array:    GenArray(type);
	    | record:   GenRecord(type);
	    | pointer:  EmitOffset(offset, tagSimplePointer);
	    | address:  EmitOffset(offset, tagOther);
			EmitSelector(SimpleAddressSY);
	    ELSE
	       Assert(FALSE);
	    END;
	 END;
      END;
   END GenPointerList;

   PROCEDURE GenType(type: Type);
      (* generate type tag for `type'
	 type is the referenced type

	 TYPE
	    Name = POINTER TO ARRAY ... OF CHAR;
	    TypeTag = POINTER TO TypeTagRec;
	    TypeTagRec =
	       RECORD
		  size: LONGINT;    (* size of the type *)
		  module: SysModules.Module;
		  typeno: INTEGER; (* of corresponding reference file *)
		  hierlen: LONGINT; (* number of base types *)
		  basetype: ARRAY hierlen OF TypeTagRec;
		     (* the 2nd basetype extends the 1st etc. *)
		  (* pointer list *)
		  sentinel: LONGINT;
	       END;

	 module, name and typeno are needed for debugging purposes only
      *)

      PROCEDURE GenBaseTypes(type: Type);
	 (* generate hierlen and basetype array *)
	 VAR
	    hierlen: CARDINAL; (* number of base types *)
	    parent: Type;

	 PROCEDURE GenBaseType(type: Type);
	 BEGIN
	    WITH type^ DO
	       IF basetype # NIL THEN
		  GenBaseType(basetype);
	       END;
	    END;
	    GenTagReference(type);
	 END GenBaseType;

      BEGIN
	 (* determine the number of base types *)
	 hierlen := 0; parent := type^.basetype;
	 WHILE parent # NIL DO
	    INC(hierlen);
	    Assert(parent^.form = record);
	    parent := parent^.basetype;
	 END;
	 StrEmit1("%:L%c", hierlen);

	 (* generate array of type tags for all base types *)
	 IF hierlen > 0 THEN
	    GenBaseType(type^.basetype);
	 END;
      END GenBaseTypes;

   BEGIN (* GenType *)
      Assert((type # NIL) &
	     ((type^.form = record) OR (type^.form = array)));
      WITH type^ DO
	 Assert(tagno # 0);
	 SetSegment(text);
	 EmitAlign4;
	 StrEmit("%*");
	 IF ident # NIL THEN
	    StrEmit1("%*TYPE %n", ident^.name);
	 END;
	 StrEmit2("%:l%_%n_%c_TAG", mainmod^.name, tagno);
	 IF (ident # NIL) & ident^.exported THEN
	    StrEmit2("%g%_%n_%n_TAG", mainmod^.name, ident^.name);
	    StrEmit2("%:l%_%n_%n_TAG", mainmod^.name, ident^.name);
	 END;

	 StrEmit1("%:L%c", size);
	 StrEmit1("%:L%_%n__MODS", mainmod^.origname);
	 StrEmit1("%:L%c", rtypeno);
	 IF type^.form = record THEN
	    GenBaseTypes(type);
	 ELSE
	    StrEmit1("%:L%c", (* hierlen = *) 0);
	 END;

	 (* generate the pointer list for the garbage collector *)
	 StrEmit("%* pointer list");
	 GenPointerList(type, 0);
	 StrEmit1("%:L%c", sentinel); (* end of pointer list *)

	 IF ident # NIL THEN
	    StrEmit1("%*END TYPE %n", ident^.name);
	 END;
      END;
   END GenType;

   PROCEDURE GenTypeTagForGlobals(globalVarSize: Size);
      (* generate type tag for all global variables together;
	 the list of global variables must have been processed by
	 GenVariables which assigns the offsets to the variables;
	 globalVarSize specifies the space occupied by the globals
      *)
      VAR
	 varp: IdentList;
   BEGIN
      SetSegment(text);
      EmitAlign4;
      StrEmit("%*type tag for the global variables");
      StrEmit1("%:l%_%n__GTAG", mainmod^.name);

      StrEmit1("%:L%c", globalVarSize);
      StrEmit1("%:L%_%n__MODS", mainmod^.origname);
      StrEmit1("%:L%c", 0); (* typeno *)
      StrEmit1("%:L%c", 0); (* hierlen *)

      varp := globalvars;
      WHILE varp # NIL DO
	 WITH varp^.ident^ DO
	    StrEmit1("%* pointer list of %n", name);
	    GenPointerList(type, offset);
	 END;
	 varp := varp^.link;
      END;
      StrEmit1("%:L%c", sentinel); (* end of pointer list *)
      StrEmit("%*END OF GTAG");
   END GenTypeTagForGlobals;

   PROCEDURE GenTypeTagForBlock(label: Label; procp: Ident;
				tmpptrs: PointerList);
      (* emit the label and generate pointer list for the given procedure *)
      VAR
	 varp: IdentList;
	 tmpptr: PointerList;
   BEGIN
      WITH procp^ DO
	 SetSegment(text);
	 EmitAlign4;
	 EmitLabel(label);
	 IF class = moduleC THEN
	    StrEmit1("%*type tag of MODULE %n", name);
	 ELSE
	    Assert(class = procedureC);
	    StrEmit1("%*type tag of PROCEDURE %n", name);

	    varp := params;
	    WHILE varp # NIL DO
	       WITH varp^.ident^ DO
		  StrEmit1("%* pointer list of %n", name);
		  IF varkind = varparamV THEN
		     EmitOffset(offset, tagOther);
		     EmitSelector(SimpleAddressSY);
		  ELSE
		     Assert(~indirect OR (type^.form = array) & type^.dyn);
		     GenPointerList(type, offset);
		  END;
	       END;
	       varp := varp^.link;
	    END;

	    varp := local;
	    WHILE varp # NIL DO
	       WITH varp^.ident^ DO
		  StrEmit1("%* pointer list of %n", name);
		  Assert(~indirect);
		  GenPointerList(type, offset);
	       END;
	       varp := varp^.link;
	    END;
	 END;
	 IF tmpptrs # NIL THEN
	    tmpptr := tmpptrs;
	    WHILE tmpptr # NIL DO
	       EmitOffset(tmpptr^.offset, tagOther);
	       EmitSelector(TmpPointerSY);
	       StrEmit1("%:L%l", tmpptr^.startlabel);
	       StrEmit1("%:L%l", tmpptr^.endlabel);
	       tmpptr := tmpptr^.next;
	    END;
	 END;
	 StrEmit1("%:L%c", sentinel); (* end of pointer list *)
	 StrEmit("%*END OF TAG");
      END;
   END GenTypeTagForBlock;

   PROCEDURE InitLocalPointers(vars: IdentList);
      (* to be called at procedure begin for initialization of
	 all pointers (NIL)
      *)
      VAR
	 (* statistics *)
	 cntPtrs: CARDINAL;
	 containers: BOOLEAN;
	 totalSize: Size;	(* of areas containing pointers *)
	 localSize: Size;	(* size of all local variables *)
	 minOffset: Size;	(* pointers are from minOffset to maxOffset *)
	 maxOffset: Size;	(* exclusive *)

      PROCEDURE Statistics;
	 VAR varp: IdentList;
      BEGIN
	 cntPtrs := 0; containers := FALSE; totalSize := 0; localSize := 0;
	 minOffset := MAX(Size); maxOffset := MIN(Size);
	 varp := vars;
	 WHILE varp # NIL DO
	    WITH varp^.ident^ DO
	       WITH type^ DO
		  INC(localSize, size);
		  IF containsptr THEN
		     INC(totalSize, size);
		     IF offset < minOffset THEN
			minOffset := offset;
		     END;
		     IF offset+size > maxOffset THEN
			maxOffset := offset+size;
		     END;
		     IF (form = pointer) OR (form = address) THEN
			INC(cntPtrs);
		     ELSE
			containers := TRUE;
		     END;
		  END;
	       END;
	    END;
	    varp := varp^.link;
	 END;
      END Statistics;

      PROCEDURE InitPointers;
	 VAR
	    varp: IdentList;
      BEGIN
	 varp := vars;
	 WHILE varp # NIL DO
	    WITH varp^.ident^ DO
	       IF (type^.form = pointer) OR (type^.form = address) THEN
		  Emit2(CLR, "%L%(d,r)", offset, base);
	       END;
	    END;
	    varp := varp^.link;
	 END;
      END InitPointers;

      PROCEDURE InitArea(from, to: Size);
	 (* init [from..to) *)
	 VAR
	    words: Size;
	    offset: Size;
	    addrReg: Reg;
	    loopLabel: Label;
	    tmp: Size;
      BEGIN
	 Assert(from < to);
	 Align(from); Align(to); words := ABS(to-from) DIV oneword;
	 IF words < 4 THEN
	    FOR offset := from TO to - oneword BY oneword DO
	       Emit2(CLR, "%L%(d,r)", offset, base);
	    END;
	 ELSE
	    Emit2(MOVE, "%L%C,%r", words - 1, d0);
	    GetAddrReg(addrReg);
	    Emit3(LEA, "%L%(d,r),%r", from, base, addrReg);
	    GetLabel(loopLabel);
	    EmitLabel(loopLabel);
	    Emit1(CLR, "%L%+r", addrReg);
	    Emit2(DBF, "%r,%l", d0, loopLabel);
	    FreeReg(addrReg);
	 END;
      END InitArea;

   BEGIN
      Statistics;
      IF containers OR (cntPtrs > 0) THEN
	 StrEmit("%*initialization of local pointers");
	 IF containers THEN
	    InitArea(minOffset, maxOffset);
	 ELSE
	    InitPointers;
	 END;
      END;
   END InitLocalPointers;

   PROCEDURE GenTypes;
      VAR
	 list, old: TypeList;
   BEGIN
      list := typelist; typelist := NIL;
      WHILE list # NIL DO
	 GenType(list^.type);
	 old := list;
	 list := list^.link;
	 DISPOSE(old);
      END;
   END GenTypes;

   PROCEDURE GenExportedTypes;
      VAR
	 exp: IdentList;
   BEGIN
      exp := mainmod^.export;
      WHILE exp # NIL DO
	 WITH exp^.ident^ DO
	    IF class = typeC THEN
	       AddTypeToList(type);
	    END;
	 END;
	 exp := exp^.link;
      END;
      GenTypes;
   END GenExportedTypes;

BEGIN
   nexttagno := 1;
   typelist := NIL;
   AddLegalOptions(CompilerOptions{maskTagsOpt});
END GenTypes.
