(* Ulm's Oberon Compiler
   Copyright (c) 1989 by University of Ulm, SAI, D-W-7900 Ulm, Germany
   ----------------------------------------------------------------------------
   $Id: Lex.m2,v 0.2 1993/06/18 15:29:31 borchert Exp $
   ----------------------------------------------------------------------------
   $Log: Lex.m2,v $
   Revision 0.2  1993/06/18  15:29:31  borchert
   Lex depends now on Types.Integer etc.

   Revision 0.1  1992/07/30  10:48:41  borchert
   Initial revision

   ----------------------------------------------------------------------------
*)

IMPLEMENTATION MODULE Lex;		(* Martin Hasch, Jan/Oct 1989 *)
					(* rev: nested comments Jul 1990 *)

   FROM Scan IMPORT
      GetCh, ch, At, Pos, Error, Error1, Message, Message1,
      warning, symbolcount;
   FROM Exception IMPORT Assert, Fatal;
   FROM IdentSys IMPORT Identifier, PutIdentChar, PutIdent;
   FROM Machine IMPORT ordmaxchar;
   FROM RealConsts IMPORT
      InitRealConst, ConvertToFraction, ConvertToExponent, TermRealConst;
   FROM Types IMPORT Cardinal, Integer, Real;
   IMPORT IdentSys, Exception, Memory, ASCII, Strings;
   IMPORT Scan; (* Scan.pos *)

   TYPE
      CharSet = SET OF CHAR;

   CONST
      eopchar = 0C;
      eolchar = ASCII.nl;
      optionprefix = "$";
      whitespace = CharSet{ ASCII.nl, ASCII.tab, ASCII.ff, " " };
      alphanumeric = CharSet{ "A".."Z", "a".."z", "0".."9" };
      optioncharset =
	 CharSet{ MIN(CompilerOptionChar)..MAX(CompilerOptionChar) };
      optionsuffixes = CharSet{ "+", "-", "=" };
      rev90Opt = "I";

   VAR
      rangeperiod: BOOLEAN;	(* if ch is "." but must be considered ".." *)

      cpos: Pos;			(* start position of last token *)
      lpos: Pos;		(* start position of token before last token *)
      epos,			(* end position + 1 of token at cpos *)
      lepos: Pos;		(* end position + 1 of token at lpos *)
      ipos: Pos;			(* position of current identifier *)
      illegalbytes: CARDINAL;		(* counts illegal characters *)
      legaloptions,
      defaultoptions: CompilerOptions;
      initcalled: BOOLEAN;
      noquotemesg: BOOLEAN;

   MODULE ReservedWords;

      IMPORT Symbol, dummyid;
      FROM IdentSys IMPORT Identifier, PutIdentChar, PutIdent, GetIdent;
      FROM Memory IMPORT ALLOCATE;
      FROM Exception IMPORT Assert;
      FROM Strings IMPORT StrCpy;

      EXPORT InitResWords, IsResWord, GetSymString;

      CONST
	 hashtabsize = 53;

      TYPE
	 HashIndex = [0..hashtabsize];

	 ResWord = POINTER TO ResWordRec;
	 ResWordRec =
	    RECORD
	       word: Identifier;
	       symbol: Symbol;
	       nextrw: ResWord;
	    END;

	 SymSet = SET OF Symbol;

      CONST
	 minkeyword = arraySY;
	 maxkeyword = withSY;
	 minsingle = plus;
	 maxsingle = rbrace;
	 keywords = SymSet{ minkeyword..maxkeyword };
	 doublechars = SymSet{ becomes, range, leq, geq };
	 singlechars = SymSet{minsingle..maxsingle} - doublechars;
	 others = SymSet{ identSY, stringcon..boolcon, eop };

      VAR
	 reswordtab: ARRAY HashIndex OF ResWord;
	 symstringtab: ARRAY [minkeyword..maxkeyword] OF Identifier;
	 symchartab: ARRAY [minsingle..maxsingle] OF CHAR;

      PROCEDURE HashWord(str: Identifier): HashIndex;
      BEGIN
	 RETURN CARDINAL(str) MOD hashtabsize
      END HashWord;

      PROCEDURE InitResWords;

	 PROCEDURE EnterWord(str: ARRAY OF CHAR; sym: Symbol);
	    VAR
	       sindex: CARDINAL;
	       hindex: HashIndex;
	       id: Identifier;
	       rw: ResWord;
	 BEGIN
	    Assert( sym IN keywords );
	    sindex := 0;
	    WHILE (sindex <= HIGH(str)) & (str[sindex] # 0C) DO
	       PutIdentChar(str[sindex]);
	       INC(sindex);
	    END;
	    PutIdent(id);
	    hindex := HashWord(id);
	    ALLOCATE(rw,SIZE(ResWordRec));
	    WITH rw^ DO
	       word := id;
	       symbol := sym;
	       nextrw := reswordtab[hindex];
	    END;
	    reswordtab[hindex] := rw;
	    symstringtab[sym] := id;
	 END EnterWord;

      BEGIN
	 PutIdent(dummyid);
	 EnterWord("ARRAY",arraySY);
	 EnterWord("BEGIN",beginSY);
	 EnterWord("CASE",caseSY);
	 EnterWord("CONST",constSY);
	 EnterWord("DEFINITION",definitionSY);
	 EnterWord("DIV",divSY);
	 EnterWord("DO",doSY);
	 EnterWord("ELSE",elseSY);
	 EnterWord("ELSIF",elsifSY);
	 EnterWord("END",endSY);
	 EnterWord("EXIT",exitSY);
	 EnterWord("IF",ifSY);
	 EnterWord("IMPORT",importSY);
	 EnterWord("IN",inSY);
	 EnterWord("IS",isSY);
	 EnterWord("LOOP",loopSY);
	 EnterWord("MOD",modSY);
	 EnterWord("MODULE",moduleSY);
	 EnterWord("NIL",nilSY);
	 EnterWord("OF",ofSY);
	 EnterWord("OR",orSY);
	 EnterWord("POINTER",pointerSY);
	 EnterWord("PROCEDURE",procedureSY);
	 EnterWord("RECORD",recordSY);
	 EnterWord("REPEAT",repeatSY);
	 EnterWord("RETURN",returnSY);
	 EnterWord("THEN",thenSY);
	 EnterWord("TO",toSY);
	 EnterWord("TYPE",typeSY);
	 EnterWord("UNTIL",untilSY);
	 EnterWord("VAR",varSY);
	 EnterWord("WHILE",whileSY);
	 EnterWord("WITH",withSY);
      END InitResWords;

      PROCEDURE IsResWord(str: Identifier; VAR sym: Symbol): BOOLEAN;
	 VAR
	    resword: ResWord;
      BEGIN
	 resword := reswordtab[ HashWord(str) ];
	 WHILE resword # NIL DO
	    WITH resword^ DO
	       IF str = word THEN
		  sym := symbol;
		  RETURN TRUE
	       END;
	       resword := nextrw;
	    END;
	 END;
	 RETURN FALSE
      END IsResWord;

      PROCEDURE InitHashtab;
	 VAR index: HashIndex;
      BEGIN
	 FOR index := MIN(HashIndex) TO MAX(HashIndex) DO
	    reswordtab[index] := NIL;
	 END;
      END InitHashtab;

      PROCEDURE GetSymString(sym: Symbol; VAR str: ARRAY OF CHAR);
      (* EXPORTED *)
	 (* useful for error messages *)
      BEGIN
	 IF sym IN keywords THEN
	    GetIdent(symstringtab[sym],str);
	 ELSIF sym IN singlechars THEN
	    StrCpy(str,'"x"');
	    IF HIGH(str) >= 1 THEN
	       str[1] := symchartab[sym];
	    END;
	 ELSE
	    CASE sym OF
	    |  range:	StrCpy(str,'".."');
	    |  becomes:	StrCpy(str,'":="');
	    |  leq:	StrCpy(str,'"<="');
	    |  geq:	StrCpy(str,'">="');
	    |  identSY:
			StrCpy(str,"identifier");
	    |  stringcon..boolcon:
			StrCpy(str,"constant");
	    |  eop:
			StrCpy(str,"end of file");
	    |  ELSE	Assert( FALSE )
	    END;
	 END;
      END GetSymString;

   BEGIN
      Assert( (keywords / singlechars / doublechars / others) =
	       SymSet{ MIN(Symbol)..MAX(Symbol) } );
      InitHashtab;
      symchartab[plus]	   := "+";
      symchartab[minus]	   := "-";
      symchartab[times]	   := "*";
      symchartab[slash]	   := "/";
      symchartab[tilde]	   := "~";
      symchartab[ampersand]:= "&";
      symchartab[period]   := ".";
      symchartab[comma]	   := ",";
      symchartab[semicolon]:= ";";
      symchartab[bar]	   := "|";
      symchartab[lparen]   := "(";
      symchartab[lbracket] := "[";
      symchartab[lbrace]   := "{";
      symchartab[becomes]  := " ";
      symchartab[arrow]	   := "^";
      symchartab[eql]	   := "=";
      symchartab[neq]	   := "#";
      symchartab[lst]	   := "<";
      symchartab[grt]	   := ">";
      symchartab[leq]	   := " ";
      symchartab[geq]	   := " ";
      symchartab[range]	   := " ";
      symchartab[colon]	   := ":";
      symchartab[rparen]   := ")";
      symchartab[rbracket] := "]";
      symchartab[rbrace]   := "}";
   END ReservedWords;

   MODULE StringConstants;

      FROM Exception IMPORT Assert, Fatal;
      FROM Memory    IMPORT ALLOCATE;

      EXPORT
	 String, PutStringChar, PutString, GetStringChar, StringLen,
	 GetStringNumber, FirstString, NextString;

   (*
    *	   string table structure:
    *
    *	      +-----+			    +-----+
    * table-->|  o------------------------->| NIL |
    *	      +---+-+-+---+---+---+---+     +---+-+-+---+---+---+---+---+
    *	      | s | t | r | 1 | s | t |     | r | 2 | s | t | 3 |   |	|
    *	      +---+---+---+---+---+---+     +---+---+---+---+---+---+---+
    *		^		^		      ^		  ^
    *	     +--|--+	     +--|--+	     +-----+  |        +--|--+
    * list-->|	o  |	  +->|	o  |	+--->|	o-----+   +--->|  o  |
    *	     +-----+	  |  +-----+	|    +-----+	  | +->+-----+
    *	     |len=4|	  |  |	4  |	|    |	3  |	  | |  |     |
    *	     +-----+	  |  +-----+	|    +-----+	  | |  +-----+
    *	     |	o---------+  |	o-------+    |	o---------+ |  | NIL |
    *	     +-----+	     +-----+	     +-----+	    |  +-----+
    *	     |No. 1|	     |	2  |	     |	3  |	    |  |  4  |
    *	     +-----+	     +-----+	     +-----+	    |  +-----+
    *							    |
    *							endoflist
    *)

      CONST
	 tablepartsize = 512;		(* table space allocated at one time *)

      TYPE
	 LargeTable = POINTER TO TablePart;

	 PartIndex = [0..tablepartsize-SIZE(LargeTable)-1];
	 TablePart =
	    RECORD
	       nextpart: LargeTable;
	       content:  ARRAY PartIndex OF CHAR;
	    END;
	 TablePosition =
	    RECORD
	       part: LargeTable;
	       pos:  PartIndex;
	    END;
	 String = POINTER TO StringRec;		(* opaque EXPORTED *)
	 StringRec =
	    RECORD
	       tablepos: TablePosition;
	       size:	 CARDINAL;
	       nextstr:  String;
	       number:   CARDINAL;
	    END;

      VAR
	 table: LargeTable;
	 list,
	 endoflist: String;
	 currentpos: TablePosition;	(* current write position; equal ...*)
					(*... or beyond endoflist^.tablepos *)
	 currentstring: String;		(* set by First-/NextString *)
	 laststring: String;			(* set by GetStringChar *)
	 currentchar: TablePosition;		(* set by GetStringChar *)
	 getcharcount: CARDINAL;

      PROCEDURE IncPos(VAR tpos: TablePosition);
      BEGIN
	 WITH tpos DO
	    IF pos < MAX(PartIndex) THEN
	       INC(pos);
	    ELSE
	       Assert( part # NIL );
	       part := part^.nextpart;
	       pos := 0;
	    END;
	 END;
      END IncPos;

      PROCEDURE NewString(VAR str: String;
			      tpos: TablePosition; siz: CARDINAL;
			      nxt: String; num: CARDINAL);
      BEGIN
	 ALLOCATE(str,SIZE(StringRec));
	 WITH str^ DO
	    tablepos := tpos;
	    size := siz;
	    nextstr := nxt;
	    number := num;
	 END;
      END NewString;

      PROCEDURE NewTablepart(VAR part: LargeTable);
      BEGIN
	 ALLOCATE(part,SIZE(TablePart));
	 part^.nextpart := NIL;
      END NewTablepart;

      PROCEDURE PutStringChar(ch: CHAR);			(* EXPORTED *)
      BEGIN
	 INC(endoflist^.size);
	 WITH currentpos DO
	    part^.content[pos] := ch;
	    IF pos < MAX(PartIndex) THEN
	       INC(pos);
	    ELSE
	       IF part^.nextpart = NIL THEN
		  NewTablepart(part^.nextpart);
	       END;
	       part := part^.nextpart;
	       pos := 0;
	    END;
	 END;
      END PutStringChar;

      PROCEDURE PutString(VAR str:  String);			(* EXPORTED *)
      BEGIN
	 str := endoflist;
	 WITH endoflist^ DO
	    NewString(nextstr, currentpos, 0, NIL, number+1);
	    endoflist := nextstr;
	 END;
      END PutString;

      PROCEDURE GetStringChar(str: String; VAR ch: CHAR);	(* EXPORTED *)
	 (* last value of ch: 0C *)
      BEGIN
	 IF str # laststring THEN
	    Assert( getcharcount = 0 );	(* was last string read completely? *)
	    laststring := str;
	 END;
	 IF getcharcount = 0 THEN
	    currentchar := str^.tablepos;
	 ELSE
	    IncPos(currentchar);
	 END;
	 INC(getcharcount);
	 IF getcharcount <= str^.size THEN
	    ch := currentchar.part^.content[currentchar.pos];
	 ELSE
	    ch := 0C;
	    getcharcount := 0;
	 END;
      END GetStringChar;

      PROCEDURE StringLen(str: String) : CARDINAL;		(* EXPORTED *)
	 (* the terminal 0C is not counted *)
      BEGIN
	 RETURN str^.size
      END StringLen;

      PROCEDURE GetStringNumber(str: String) : CARDINAL;	(* EXPORTED *)
      BEGIN
	 RETURN str^.number
      END GetStringNumber;

      PROCEDURE FirstString;					(* EXPORTED *)
	 (* FirstString; WHILE NextString(s) DO ... END; *)
      BEGIN
	 currentstring := list;
      END FirstString;

      PROCEDURE NextString(VAR str: String) : BOOLEAN;		(* EXPORTED *)
	 VAR
	    index,
	    strsize: CARDINAL;
	    tpos: TablePosition;
      BEGIN
	 Assert( currentstring # NIL );
	 IF currentstring = endoflist THEN
	    RETURN FALSE
	 END;
	 str := currentstring;
	 currentstring := currentstring^.nextstr;
	 RETURN TRUE
      END NextString;

   BEGIN
      NewTablepart(table);
      currentpos.part := table;
      currentpos.pos := 0;
      NewString(list, currentpos, 0, NIL, 1);
      endoflist := list;
      currentstring := NIL;
      laststring := NIL;
      getcharcount := 0;
   END StringConstants;

   MODULE OptionStack;

      FROM Memory IMPORT ALLOCATE, DEALLOCATE;
      IMPORT CompilerOptionChar, CompilerOptions, options, defaultoptions;

      EXPORT PushOption, PopOption;

      TYPE
	 OptionList = POINTER TO OptionRec;
	 OptionRec =
	    RECORD
	       value:  BOOLEAN;
	       link: OptionList;
	    END;

      VAR
	 och: CompilerOptionChar;
	 stack: ARRAY CompilerOptionChar OF OptionList;

      PROCEDURE PushOption(opt: CompilerOptionChar; val: BOOLEAN);
	 VAR
	    new: OptionList;
      BEGIN
	 NEW(new);
	 WITH new^ DO
	    value := opt IN options;
	    link := stack[opt];
	 END;
	 stack[opt] := new;
	 IF val THEN
	    INCL(options,opt);
	 ELSE
	    EXCL(options,opt);
	 END;
      END PushOption;

      PROCEDURE PopOption(opt: CompilerOptionChar);
	 VAR
	    old: OptionList;
      BEGIN
	 old := stack[opt];
	 IF old # NIL THEN
	    WITH old^ DO
	       IF value THEN
		  INCL(options,opt);
	       ELSE
		  EXCL(options,opt);
	       END;
	       stack[opt] := link;
	    END;
	    DISPOSE(old);
	 ELSE (* old = NIL *)
	    IF opt IN defaultoptions THEN
	       INCL(options,opt);
	    ELSE
	       EXCL(options,opt);
	    END;
	 END;
      END PopOption;

   BEGIN
      FOR och := MIN(CompilerOptionChar) TO MAX(CompilerOptionChar) DO
	 stack[och] := NIL;
      END;
   END OptionStack;

   PROCEDURE GetSy;					(* EXPORTED *)
      VAR
	 prevch: CHAR;			(* previous char *)
	 space: BOOLEAN;		(* TRUE while skipping white space *)

      PROCEDURE Comment;
	 VAR
	    ppos,
	    startpos: Pos;
	    atbegin: BOOLEAN;		(* TRUE while options may occur *)
	    copen: INTEGER;
	    allownested: BOOLEAN;

	 PROCEDURE Option;
	 BEGIN
	    IF ~(prevch IN legaloptions) THEN
	       At(ppos);
	       Message1(warning, "%Q: illegal compiler option", prevch);
	       At(pos);
	    ELSE
	       CASE ch OF
	       |  "+":  PushOption(prevch, TRUE);
	       |  "-":  PushOption(prevch, FALSE);
	       |  "=":  PopOption(prevch);
	       |  ELSE  Assert( FALSE )
	       END;
	    END;
	 END Option;

      BEGIN
	 startpos := cpos;
	 copen := 1;
	 allownested := rev90Opt IN options;
	 atbegin := TRUE;
	 WHILE (copen > 0) & (ch # eopchar) DO
	    prevch := ch;
	    ppos := Scan.pos;
	    GetCh;
	    IF (prevch = "*") & (ch = ")") THEN
	       DEC(copen);
	       GetCh;
	    ELSIF (prevch = "(") & (ch = "*") THEN
	       IF allownested THEN
		  INC(copen);
		  GetCh;
	       ELSE
		  At(ppos);
		  Message(warning,'"(*" within comment ignored');
		  At(pos);
		  (* not GetCh, as "(" + "*" + ")" must close a comment *)
	       END;
	    END;
	    IF atbegin THEN
	       IF prevch = optionprefix THEN
		  prevch := ch;
		  GetCh;
		  IF (prevch IN optioncharset) & (ch IN optionsuffixes) THEN
		     Option;
		     GetCh;
		  ELSE
		     atbegin := FALSE;
		  END;
	       ELSIF ~(prevch IN whitespace) THEN
		  atbegin := FALSE;
	       END;
	    END;
	 END; (* WHILE (copen > 0) & ... *)
	 IF copen > 0 THEN
	    At(startpos);
	    Error("comment not closed");
	    At(pos);
	 END;
      END Comment;

      PROCEDURE Number;
	 VAR
	    decimal: Integer;
	    hex: Cardinal;
	    digit: Cardinal;
	    decimalok,
	    hexok,
	    realok: BOOLEAN;
	    ishex,		(* TRUE after reading digits "A".."F" *)
	    rangeerr,		(* causes error message when TRUE *)
	    islong: BOOLEAN;	(* TRUE after scale factor with "D" *)

	 PROCEDURE ConvToDecimal(digit: Integer);
	 BEGIN
	    IF (decimal < MAX(Integer) DIV 10) OR
		  (decimal = MAX(Integer) DIV 10) &
		  (digit <= MAX(Integer) MOD 10) THEN
	       decimal := decimal * 10 + digit;
	    ELSE
	       decimalok := FALSE;
	       decimal := MAX(Integer);
	    END;
	 END ConvToDecimal;

	 PROCEDURE ConvToHex(digit: Cardinal);
	 BEGIN
	    IF (hex < MAX(Cardinal) DIV 10H) OR
		  (hex = MAX(Cardinal) DIV 10H) &
		  (digit <= MAX(Cardinal) MOD 10H) THEN
	       hex := hex * 10H + digit;
	    ELSE
	       hexok := FALSE;
	       hex := VAL(Cardinal, MAX(Integer));
	    END;
	 END ConvToHex;

      BEGIN
	 decimalok := TRUE;
	 hexok := TRUE;
	 decimal := ORD(prevch) - ORD("0");
	 hex := decimal;
	 InitRealConst;
	 ConvertToFraction(prevch);
	 WHILE ("0" <= ch) & (ch <= "9") DO
	    digit := ORD(ch) - ORD("0");
	    IF decimalok THEN
	       ConvToDecimal(digit);
	       IF hexok THEN
		  ConvToHex(digit);
	       END;
	    END;
	    ConvertToFraction(ch);
	    GetCh;
	 END;
	 realok := ch = ".";
	 IF realok THEN
	    GetCh;
	    IF ch = "." THEN
	       rangeperiod := TRUE;
	       realok := FALSE;
	    END;
	 END;
	 IF realok THEN			(* now there must be a real constant *)
	    ConvertToFraction(".");
	    WHILE ("0" <= ch) & (ch <= "9") DO
	       ConvertToFraction(ch);
	       GetCh;
	    END;
	    islong := FALSE;
	    IF (ch = "D") OR (ch = "E") THEN
	       islong := ch = "D";
	       GetCh;
	       IF (ch = "+") THEN
		  (* ignore it *)
		  GetCh;
	       ELSIF (ch = "-") THEN
		  ConvertToExponent(ch);
		  GetCh;
	       END;
	       IF (ch < "0") OR ("9" < ch) THEN
		  realok := FALSE;
		  At(Scan.pos);
		  Error("digits of scale factor expected");
		  At(pos);
	       ELSE
		  REPEAT
		     ConvertToExponent(ch);
		     GetCh;
		  UNTIL (ch < "0") OR ("9" < ch);
	       END;
	    END;
	    IF realok THEN
	       TermRealConst(cval,islong,realok);
	       sy := cval.sy;
	       IF ~realok THEN
		  At(cpos);
		  IF islong THEN
		     Error("longreal constant out of range");
		  ELSE
		     Error("real constant out of range");
		  END;
		  At(pos);
	       END;
	    ELSE
	       sy := realcon;
	       cval.sy := realcon;
	       cval.realval := 0.0;
	    END;
	 ELSE		(* now there must be a char or integer constant *)
	    ishex := FALSE;
	    rangeerr := FALSE;
	    LOOP	(* continues while input chars look like a number *)
	       CASE ch OF
	       |  "0".."9":
			IF hexok THEN
			   ConvToHex( ORD(ch)-ORD("0") );
			END;
			GetCh;
	       |  "A".."F":
			ishex := TRUE;
			IF hexok THEN
			   ConvToHex( ORD(ch)-ORD("A")+0AH );
			END;
			GetCh;
	       |  "H":
			GetCh;
			rangeerr := ~hexok;
			cval.sy := intcon;
			cval.intval := INTEGER(hex);
			EXIT
	       |  "X":
			GetCh;
			rangeerr := ~hexok OR (hex > ordmaxchar);
			cval.sy := charcon;
			IF rangeerr THEN
			   cval.charval := CHR(ordmaxchar);
			ELSE
			   cval.charval := CHR(hex);
			END;
			EXIT
	       |  ELSE
			IF ishex THEN
			   At(Scan.pos);
			   Error('"H" or "X" after hex digits expected');
			   At(pos);
			ELSE
			   rangeerr := ~decimalok;
			END;
			cval.sy := intcon;
			cval.intval := decimal;
			EXIT
	       END;
	    END; (*LOOP*)
	    sy := cval.sy;
	    IF rangeerr THEN
	       At(cpos);
	       Error("constant out of range");
	       At(pos);
	    END;
	 END;
      END Number;

   BEGIN
      lpos := cpos;
      REPEAT
	 IF ch = eopchar THEN
	    sy := eop;
	    pos := lpos;
	    At(lpos);
	    lepos := epos;
	    epos := Scan.pos;
	    RETURN			(* do not read after end of file *)
	 END;
	 cpos := Scan.pos;
	 space := FALSE;
	 prevch := ch;
	 GetCh;
	 CASE prevch OF
	 |  "+":  sy := plus;
	 |  "-":  sy := minus;
	 |  "*":  IF ch = ")" THEN
		     GetCh;
		     At(cpos);
		     Error('"*)" not expected');
		     At(pos);
		     space := TRUE;
		  ELSE
		     sy := times;
		  END;
	 |  "/":  sy := slash;
	 |  "~":  sy := tilde;
	 |  "&":  sy := ampersand;
	 |  ".":  IF rangeperiod THEN
		     rangeperiod := FALSE;
		     sy := range;
		  ELSIF ch = "." THEN
		     GetCh;
		     sy := range;
		  ELSE
		     sy := period;
		  END;
	 |  ",":  sy := comma;
	 |  ";":  sy := semicolon;
	 |  "|":  sy := bar;
	 |  "(":  IF ch = "*" THEN
		     GetCh;
		     Comment;
		     space := TRUE;
		  ELSE
		     sy := lparen;
		  END;
	 |  "[":  sy := lbracket;
	 |  "{":  sy := lbrace;
	 |  ":":  IF ch = "=" THEN
		     GetCh;
		     sy := becomes;
		  ELSE
		     sy := colon;
		  END;
	 |  "^":  sy := arrow;
	 |  "=":  sy := eql;
	 |  "#":  sy := neq;
	 |  "<":  IF ch = "=" THEN
		     GetCh;
		     sy := leq;
		  ELSE
		     sy := lst;
		  END;
	 |  ">":  IF ch = "=" THEN
		     GetCh;
		     sy := geq;
		  ELSE
		     sy := grt;
		  END;
	 |  ")":  sy := rparen;
	 |  "]":  sy := rbracket;
	 |  "}":  sy := rbrace;
	 |  "A".."Z","a".."z":
		  PutIdentChar(prevch);
		  WHILE ch IN alphanumeric DO
		     PutIdentChar(ch);
		     GetCh;
		  END;
		  PutIdent(ident);
		  IF ~IsResWord(ident,sy) THEN
		     sy := identSY;
		     previdpos := ipos;
		     ipos := cpos;
		  END;
	 |  '"',"'":
		  IF (prevch = "'") & ~noquotemesg THEN
		     At(cpos);
		     Message1(warning, "%r: illegal string quote %Q", prevch);
		     At(pos);
		     noquotemesg := TRUE;
		  END;
		  WHILE (ch # prevch) & (ch # eolchar) & (ch # eopchar) DO
		     PutStringChar(ch);
		     GetCh;
		  END;
		  IF ch = prevch THEN
		     GetCh;
		  ELSE
		     At(Scan.pos);
		     Error1("string quote %Q on this line expected",prevch);
		     At(pos);
		  END;
		  sy := stringcon;
		  cval.sy := stringcon;
		  PutString(cval.string);
	 |  "0".."9":
		  Number;
	 |  ELSE  space := TRUE;
		  IF ~(prevch IN whitespace) THEN
		     INC(illegalbytes);
		     IF illegalbytes > Scan.pos.line + 10 THEN
			(* avoid parsing non-ascii-files *)
			Fatal("too many illegal characters in source file")
		     END;
		     At(cpos);
		     Error("illegal character");
		     At(pos);
		  END;
	 END;
      UNTIL ~space;
      pos := lpos;
      At(lpos);
      lepos := epos;
      epos := Scan.pos;
      INC(symbolcount);
   END GetSy;

   PROCEDURE Mark;						(* EXPORTED *)
      (* set default error position to current token instead of last one *)
   BEGIN
      pos := cpos;
      At(cpos);
   END Mark;

   PROCEDURE Insert;						(* EXPORTED *)
   BEGIN
      pos := lepos;
      At(lepos);
   END Insert;

   PROCEDURE InsertId;						(* EXPORTED *)
      (* set current identifier position after last token *)
   BEGIN
      previdpos := ipos;
      ipos := lepos;
   END InsertId;

   PROCEDURE Skip;						(* EXPORTED *)
      (* leave current symbol out of error positioning *)
   BEGIN
      Assert( sy # eop );
      epos := lepos;
      cpos := lpos;
      IF sy = identSY THEN
	 ipos := previdpos;
      END;
   END Skip;

   PROCEDURE PosStatement;					(* EXPORTED *)
      (* set statementpos to position of last token *)
   BEGIN
      statementpos := lpos;
   END PosStatement;

   PROCEDURE InitLex;					(* EXPORTED *)
      (* to be called after argument scanning *)
   BEGIN
      Assert( ORD("F")-ORD("A") = 0FH-0AH );	(* for hex constant eval. *)
      Assert( VAL(CARDINAL,MAX(INTEGER)) < MAX(CARDINAL) );	(* dito *)
      Assert( ~initcalled );
      initcalled := TRUE;
      InitResWords;
      lpos := Scan.pos;
      GetCh;
      cpos := Scan.pos;
      epos := cpos;
      previdpos := cpos;
      pos := cpos;
      ipos := cpos;
      previdpos := cpos;
      statementpos := cpos;
      lepos := Scan.pos;
      rangeperiod := FALSE;
      options := defaultoptions;
   END InitLex;

   PROCEDURE AddLegalOptions(co: CompilerOptions);		(* EXPORTED *)
      (* includes co into the set of legal compiler option characters *)
      (* and sets default values TRUE *)
   BEGIN
      Assert( ~initcalled & (legaloptions * co = CompilerOptions{}) );
      legaloptions := legaloptions + co;
      defaultoptions := defaultoptions + co;
   END AddLegalOptions;

   PROCEDURE SetDefault(ochar: CHAR; val: BOOLEAN): BOOLEAN;	(* EXPORTED *)
      (* changes default value of ochar according to val *)
   BEGIN
      Assert( ~initcalled );
      IF (ochar < MIN(CompilerOptionChar)) OR
	    (ochar > MAX(CompilerOptionChar)) OR
	    ~(ochar IN legaloptions) THEN
	 RETURN FALSE
      END;
      IF val THEN
	 INCL(defaultoptions, ochar);
      ELSE
	 EXCL(defaultoptions, ochar);
      END;
      RETURN TRUE
   END SetDefault;

BEGIN
   illegalbytes := 0;
   symbolcount := 0;
   legaloptions := CompilerOptions{};
   defaultoptions := CompilerOptions{};
   options := CompilerOptions{};
   initcalled := FALSE;
   noquotemesg := FALSE;
END Lex.
