(* Ulm's Oberon Compiler
   Copyright (c) 1989 by University of Ulm, SAI, D-W-7900 Ulm, Germany
   ----------------------------------------------------------------------------
   $Id: Scan.m2,v 0.2 1993/04/19 14:36:44 borchert Exp $
   ----------------------------------------------------------------------------
   $Log: Scan.m2,v $
   Revision 0.2  1993/04/19  14:36:44  borchert
   the version number in the header depends now from ScanArgs.version
   the layout of the import list in the listing has been slightly changed

   Revision 0.1  1992/07/30  10:49:16  borchert
   Initial revision

   ----------------------------------------------------------------------------
*)

IMPLEMENTATION MODULE Scan;		(* Martin Hasch, Jan/Feb 1989 *)

   FROM SYSTEM	  IMPORT BYTE;
   FROM ASCII	  IMPORT nl, tab;
   FROM Clock	  IMPORT UnitsPerSecond, CPUTime;
   FROM Conversions IMPORT ConvertHex;
   FROM Errno	  IMPORT errno;
   FROM FtdIO	  IMPORT
      FwriteString, FwriteLn, Done, FwriteInt, FwriteChar, FwriteCard;
   FROM Files	  IMPORT Delete;
   FROM Memory	  IMPORT MaxSpace;
   FROM Messages  IMPORT
      PushArgument, PrintFormat, writeproc, ClearTabs, SetTab;
   FROM ScanArgs  IMPORT
      input, dolist, listing, currenttime, lastrevision, version;
   FROM StdIO	  IMPORT
      FILE, Fopen, MODE, Fclose, stderr, Fgetc, Fputc, CloseAll;
   FROM Strings   IMPORT StrLen;
   FROM SysExit	  IMPORT Exit;
   FROM SysPerror IMPORT GetErrorString;
   FROM TimeIO	  IMPORT FwriteTime; IMPORT TimeIO (*.Done*);
   IMPORT ASCII, Conversions;

   MODULE ToASCII;				(* Martin Hasch, Feb 1989 *)
      (* machine dependent: "knows" about character representation and order *)

      FROM ASCII IMPORT nl;
      FROM Conversions IMPORT ConvertOctal;

      EXPORT CharSet, visible, Substitute;

      TYPE
	 CharSet = SET OF CHAR;

      CONST
	 visible = CharSet{ nl, " ".."~" };
	    (* set of characters to be substituted by themselves *)

      PROCEDURE Substitute(ch: CHAR; VAR s: ARRAY OF CHAR);
	 (* assigns s an ASCII substitution for an arbitrary character ch *)

	 PROCEDURE Control(name: ARRAY OF CHAR);
	    VAR
	       index: CARDINAL;
	 BEGIN
	    s[0] := "<";
	    index := 0;
	    WHILE (index < HIGH(s)) & (index <= HIGH(name)) &
		  (name[index] # 0C) DO
	       s[index+1] := name[index];
	       INC(index);
	    END;
	    INC(index);
	    IF index <= HIGH(s) THEN
	       s[index] := ">";
	       IF index < HIGH(s) THEN
		  s[index+1] := 0C;
	       END;
	    END;
	 END Control;

	 PROCEDURE Octal;
	    VAR
	       octstring: ARRAY [0..3] OF CHAR;
	 BEGIN
	    ConvertOctal(ORD(ch),0,octstring);
	    Control(octstring);
	 END Octal;

      BEGIN
	 IF ch IN visible THEN
	    s[0] := ch;
	    IF HIGH(s) >= 1 THEN
	       s[1] := 0C;
	    END;
	 ELSE
	    CASE ch OF
	    |  0C: Control("NUL");
	    |  1C: Control("SOH");
	    |  2C: Control("STX");
	    |  3C: Control("ETX");
	    |  4C: Control("EOT");
	    |  5C: Control("ENQ");
	    |  6C: Control("ACK");
	    |  7C: Control("BEL");
	    |  10C: Control("BS");
	    |  11C: Control("HT");
	    |  12C: Control("LF");
	    |  13C: Control("VT");
	    |  14C: Control("FF");
	    |  15C: Control("CR");
	    |  16C: Control("SO");
	    |  17C: Control("SI");
	    |  20C: Control("DLE");
	    |  21C: Control("DC1");
	    |  22C: Control("DC2");
	    |  23C: Control("DC3");
	    |  24C: Control("DC4");
	    |  25C: Control("NAK");
	    |  26C: Control("SYN");
	    |  27C: Control("ETB");
	    |  30C: Control("CAN");
	    |  31C: Control("EM");
	    |  32C: Control("SUB");
	    |  33C: Control("ESC");
	    |  34C: Control("FS");
	    |  35C: Control("GS");
	    |  36C: Control("RS");
	    |  37C: Control("US");
	    |  40C: Control("SP");
	    |  177C: Control("DEL");
	    |  ELSE Octal;
	    END;
	 END;
      END Substitute;

   END ToASCII;

(***** EXPORTED:
   TYPE
      Pos = (* position in source *)
	 RECORD
	    line, pos: CARDINAL;
	 END;
      ErrorType = (headerinfo, warning, error, fatal, bug);

   VAR
      pos: Pos;			(* position of last ch read by GetCh *)
      ch: CHAR;			(* character read by GetCh *)
				(* ch = 0C at end of file  *)
      errorflag: BOOLEAN;	(* FALSE as long as no error/fatal/bug *)
				(* occured, otherwise TRUE	       *)
      symbolcount: CARDINAL;	(* incremented by Lex, for statistics *)
      outputcount: CARDINAL;	(* incremented by CodeGen, for statistics *)
*****)

   (* miscellaneous declarations *)

   CONST
      linenumberwidth = 4;
      indentwidth = 8;		(* must be greater than linenumberwidth *)
      perrfmt = "%s: %E";
      assertfmt = "assertion failed";

   VAR
      infile: FILE;
      outfile: FILE;
      writing: BOOLEAN;		(* TRUE when output is a listing file *)
      outfilebad: BOOLEAN;	(* TRUE if outfile is to be scratched *)
      fatalflag: BOOLEAN;	(* TRUE if compilation ended with fatal error*)
      inendscan: BOOLEAN;	(* normally FALSE; prevents recursion *)
      inerrorlist: BOOLEAN;	(* between calls of ErrorList and EndofList *)
      nullbytepos: Pos;			(* position of last null-byte *)

   (* input buffering *)

   CONST
      buffersize = 32768;
      linetablesize = 1024;

   TYPE
      BufferIndex = [0..buffersize-1];
      LineIndex = [0..linetablesize-1];
      LineInfo =
	 RECORD
	    linelength: CARDINAL;
	 END;

   VAR
      leastline: CARDINAL;		(* no. of least line still in buffer *)
      currentline: CARDINAL;		(* no. of last complete line + 1 *)
      buffer: ARRAY BufferIndex OF CHAR;
	 (* ring buffer for output characters *)
      bufferstart,		(* = bufferindex[leastline MOD linetablesize]*)
      bufferpos: BufferIndex;	(* index of next char to be buffered *)
      bufferfill: CARDINAL;	(* number of valid characters in buffer *)
      linetable: ARRAY LineIndex OF LineInfo;
	 (* ring buffer for line start indices in buffer *)

   (* error messages buffering *)

   CONST
      mesgbuffersize = 4096;
      mesgtablesize = 128;

   TYPE
      MesgBufferIndex = [0..mesgbuffersize-1];
      MesgIndex = [0..mesgtablesize-1];
      MesgInfo =
	 RECORD
	    mesgline: CARDINAL;		(* line where message is printed *)
	    mesgpos: Pos;		(* real error position *)
	       (* mesgpos.pos = 0 stands for "end of line mesgpos.line" *)
	    mesglength: CARDINAL;
	 END;

   VAR
      mesgbuffer: ARRAY MesgBufferIndex OF CHAR;
	 (* ring buffer for error messages *)
      mesgbufferstart,
      mesgbufferfill: CARDINAL;
      mesgtable: ARRAY [0..mesgtablesize-1] OF MesgInfo;
	 (* sorted list of not yet output error messages *)
	 (* mesgpos.line is always >= leastline *)
      mesgtablefill: CARDINAL;

   (* current error message buffering *)

   CONST
      messagesize = 1024;
	 (* max. length of current error message, MUST be <= mesgbuffersize *)

   VAR
      errpos: Pos;			(* message position in source file *)
	 (* errpos.pos = 0 stands for "end of line errpos.line - 1" *)
      message: ARRAY [0..messagesize-1] OF CHAR;
	 (* buffer collecting chars of current message *)
      messagefill: CARDINAL;	(* index of next char to write into message *)

   (* ----- *)

   PROCEDURE OutfileGood(done: BOOLEAN; newlines: CARDINAL);
      (* writes newlines newlines into listing file and checks done *)
   BEGIN
      WHILE done DO
	 IF newlines = 0 THEN
	    RETURN
	 END;
	 done := Fputc(nl,outfile);
	 DEC(newlines);
      END;
      outfilebad := TRUE;
      Message2(fatal, perrfmt, listing, errno)
   END OutfileGood;

   PROCEDURE PrintHeader;
      CONST
	 (*
	 headline =
	    "**** Oberon-Compiler      Version  1.0      %d.%m.%Y ****%n";
				  1234567890123456789012 = 22 chars
	 *)
	 headline1 = "**** Oberon-Compiler ";
	 headline2len = 22;
	 headline3 = " %d.%m.%Y ****%n";
	 timedate =
	    "**** time: %H:%M:%S                   date: %d.%m.%Y ****%n";

      PROCEDURE PrintVersion(width: CARDINAL);
	 (* print centered version number *)
	 CONST
	    prefix = "Version ";
	 VAR
	    len: CARDINAL; (* length of version string *)
	    prefixlen: CARDINAL; (* length of prefix *)
	    printprefix: BOOLEAN;
	    left, right: CARDINAL; (* # of white space left and right *)
	    cnt: CARDINAL;
      BEGIN
	 len := StrLen(version);
	 prefixlen := StrLen(prefix);
	 IF len >= width THEN
	    FwriteString(outfile, version);
	    OutfileGood(Done,0);
	 ELSE
	    (* if possible, insert prefix before version string *)
	    IF len + prefixlen < width THEN
	       printprefix := TRUE; INC(len, prefixlen);
	    ELSE
	       printprefix := FALSE;
	    END;
	    (* calculate the # of white spaces to be printed left & right *)
	    left := (width - len) DIV 2;
	    IF (width - len) MOD 2 = 0 THEN
	       right := left;
	    ELSE
	       right := left + 1;
	    END;
	    FOR cnt := 1 TO left DO
	       FwriteChar(outfile, " ");
	       OutfileGood(Done,0);
	    END;
	    IF printprefix THEN
	       FwriteString(outfile, prefix);
	       OutfileGood(Done,0);
	    END;
	    FwriteString(outfile, version);
	    OutfileGood(Done,0);
	    FOR cnt := 1 TO right DO
	       FwriteChar(outfile, " ");
	       OutfileGood(Done,0);
	    END;
	 END;
      END PrintVersion;

   BEGIN
      IF writing & ~outfilebad THEN
	 FwriteString(outfile, headline1);
	 OutfileGood(Done,0);
	 PrintVersion(headline2len);
	 FwriteTime(outfile, headline3, lastrevision);
	 OutfileGood(TimeIO.Done,1);
	 FwriteTime(outfile, timedate, currenttime);
	 OutfileGood(TimeIO.Done,1);
	 FwriteString(outfile,"source: ");
	 OutfileGood(Done,0);
	 FwriteString(outfile,input);
	 OutfileGood(Done,2);
      END;
   END PrintHeader;

   PROCEDURE PrintStatistics;
      VAR
	 cputime: CARDINAL;
   BEGIN
      IF writing & ~outfilebad THEN
	 OutfileGood(TRUE,1);
	 FwriteString(outfile,"compilation statistics:");
	 OutfileGood(Done,1);
	 FwriteString(outfile,"   dynamically allocated space: ");
	 OutfileGood(Done,0);
	 FwriteCard(outfile, MaxSpace(), 7);
	 OutfileGood(Done,1);
	 FwriteString(outfile,"   number of parsed symbols: ");
	 OutfileGood(Done,0);
	 FwriteCard(outfile, symbolcount, 3+7);
	 OutfileGood(Done,1);
	 FwriteString(outfile,"   input lines per minute: ");
	 OutfileGood(Done,0);
	 cputime := CPUTime(FALSE);
	 FwriteCard(outfile,
	    (currentline-1) * 60*UnitsPerSecond DIV cputime, 5+7);
	 OutfileGood(Done,1);
	 FwriteString(outfile,"   output lines per minute: ");
	 OutfileGood(Done,0);
	 FwriteCard(outfile, outputcount * 60*UnitsPerSecond DIV cputime, 4+7);
	 OutfileGood(Done,1);
      END;
   END PrintStatistics;

   PROCEDURE NotEqual(pos1,pos2: Pos): BOOLEAN;
   BEGIN
      IF pos1.line # pos2.line THEN
	 RETURN TRUE
      END;
      RETURN pos1.pos # pos2.pos
   END NotEqual;

   PROCEDURE FlushOneLine;
      VAR
	 index: CARDINAL;
	 length: CARDINAL;		(* no. of flushed characters *)
	 mustprint: BOOLEAN;

      PROCEDURE ErrorMessages(upto: CARDINAL; lineprinted: BOOLEAN);
	 (* precondition: at least one error message is to flush *)
	 (* if lineprinted, position can be indicated by an arrow *)

	 VAR
	    oldpos: Pos;
	    mesgcount: CARDINAL;
	    mtindex: CARDINAL;

	 PROCEDURE PrintMessage(mi: MesgInfo; firstmesg: BOOLEAN);
	    VAR
	       showpos: BOOLEAN;
	       count: CARDINAL;

	    PROCEDURE PrintChar(ch: CHAR);
	    BEGIN
	       IF ~writing THEN
		  FwriteChar(stderr, ch);
	       ELSIF ~outfilebad THEN
		  OutfileGood( Fputc(ch,outfile), 0);
	       END;
	    END PrintChar;

	    PROCEDURE PrintString(str: ARRAY OF CHAR);
	    BEGIN
	       IF writing THEN
		  IF ~outfilebad THEN
		     FwriteString(outfile,str);
		     OutfileGood(Done,0);
		  END;
	       ELSE
		  FwriteString(stderr,str);
	       END;
	    END PrintString;

	    PROCEDURE PrintCard(number: CARDINAL);
	    BEGIN
	       IF writing THEN
		  IF ~outfilebad THEN
		     FwriteCard(outfile,number,0);
		     OutfileGood(Done,0);
		  END;
	       ELSE
		  FwriteCard(stderr,number,0);
	       END;
	    END PrintCard;

	    PROCEDURE PrintStars;
	       VAR
		  count: CARDINAL;
		  char: CHAR;
	    BEGIN
	       IF writing THEN
		  IF ~outfilebad THEN
		     FOR count := 1 TO linenumberwidth DO
			OutfileGood( Fputc("*",outfile), 0);
		     END;
		  END;
	       ELSE
		  FOR count := 1 TO linenumberwidth DO
		     FwriteChar(stderr," ");
		  END;
	       END;
	    END PrintStars;

	 BEGIN
	    showpos := firstmesg OR NotEqual(mi.mesgpos,oldpos);
	    oldpos := mi.mesgpos;
	    IF showpos THEN
	       IF lineprinted & (mi.mesgline = mi.mesgpos.line) THEN
		  INC(mi.mesgpos.pos,indentwidth);
		  IF (mi.mesglength + linenumberwidth+1 >= mi.mesgpos.pos) THEN
		     PrintStars;
		     FOR count := linenumberwidth+1 TO mi.mesgpos.pos-1 DO
			PrintChar(" ");
		     END;
		     PrintChar("^");
		     PrintChar(nl);
		     showpos := FALSE;
		  END;
	       ELSE
		  PrintStars;
		  PrintString(" error at ");
		  IF mi.mesgpos.pos = 0 THEN
		     PrintString("end");
		  ELSE
		     PrintString("column ");
		     PrintCard(mi.mesgpos.pos);
		  END;
		  PrintString(" of line ");
		  PrintCard(mi.mesgpos.line);
		  PrintChar(":"); PrintChar(nl);
		  showpos := FALSE;
	       END;
	    END;
	    PrintStars; PrintChar(" ");
	    count := 0;
	    WHILE count < mi.mesglength DO
	       PrintChar(
		  mesgbuffer[(mesgbufferstart + count) MOD mesgbuffersize] );
	       INC(count);
	    END;
	    IF showpos THEN
	       FOR count := linenumberwidth+2 + mi.mesglength TO
		     mi.mesgpos.pos-1 DO
		  PrintChar(" ");
	       END;
	       PrintChar("^");
	    END;
	    PrintChar(nl);
	    mesgbufferstart :=
	       (mesgbufferstart + mi.mesglength) MOD mesgbuffersize;
	    DEC(mesgbufferfill,mi.mesglength);
	 END PrintMessage;

      BEGIN
	 mesgcount := 0;
	 PrintMessage(mesgtable[0],TRUE);
	 mesgcount := 1;
	 WHILE (mesgcount < mesgtablefill) &
	       (mesgtable[mesgcount].mesgline <= upto) DO
	    PrintMessage(mesgtable[mesgcount], FALSE);
	    INC(mesgcount);
	 END;
	 FOR mtindex := mesgcount TO mesgtablefill-1 DO
	    mesgtable[mtindex-mesgcount] := mesgtable[mtindex];
	 END;
	 DEC(mesgtablefill,mesgcount);
      END ErrorMessages;

   BEGIN (*FlushOneLine*)
      IF leastline >= currentline THEN
	 Message(bug,assertfmt)
      END;
      IF writing & ~outfilebad & (leastline = 1) THEN
	 Message(headerinfo,"%nlisting:");
      END;
      IF (mesgtablefill > 0) & (mesgtable[0].mesgline < leastline) THEN
	 (* this should not be expected, because "effective line"s are *)
	 (* chosen (by this module) greater or equal leastline - some  *)
	 (* day, however, "effective line"s might be specified via     *)
	 (* At(...), so it seems considerable to keep this test here.  *)
	 ErrorMessages(leastline-1, FALSE);
      END;
      length := linetable[leastline MOD linetablesize].linelength;
      IF length > bufferfill THEN
	 Message(bug,assertfmt)
      END;
      IF writing & ~outfilebad & (length # 0) THEN
	 FwriteCard(outfile, leastline, linenumberwidth);
	 OutfileGood(Done,0);
	 FOR index := linenumberwidth + 1 TO indentwidth DO
	    OutfileGood( Fputc(" ",outfile), 0);
	 END;
	 FOR index := 0 TO length-1 DO
	    OutfileGood( Fputc(buffer[(bufferstart + index) MOD buffersize],
		  outfile), 0);
	 END;
      END;
      IF (mesgtablefill > 0) & (leastline >= mesgtable[0].mesgline) THEN
	 mustprint := mesgtable[0].mesgpos.line = leastline;
	 IF ~writing & (length # 0) & mustprint THEN
	    FwriteCard(stderr, leastline, linenumberwidth);
	    FwriteString(stderr,"    ");
	    FOR index := 0 TO length-1 DO
	       FwriteChar(stderr,
		  buffer[(bufferstart + index) MOD buffersize] );
	    END;
	 END;
	 ErrorMessages(leastline, mustprint);
      END;
      bufferstart := (bufferstart + length) MOD buffersize;
      DEC(bufferfill,length);
      INC(leastline);
   END FlushOneLine;

   PROCEDURE PutChar(ch: CHAR);
      (* put a character into input buffer *)

      CONST
	 goodchars = visible - CharSet{ tab };
      VAR
	 substitute: ARRAY [0..7] OF CHAR;
   BEGIN
      IF bufferfill >= buffersize THEN
	 IF leastline < currentline THEN
	    FlushOneLine;
	 ELSE
	    Message(fatal, "huge line in source file")
	 END;
      END;
      IF ch IN goodchars THEN
	 buffer[ bufferpos ] := ch;
	 INC(bufferfill);
	 bufferpos := (bufferpos + 1) MOD buffersize;
	 INC(linetable[currentline MOD linetablesize].linelength);
      ELSIF ~fatalflag THEN
	 Substitute(ch, substitute);
	 (* substitute contains only "good" characters *)
	 PutString(substitute);
      END;
   END PutChar;

   PROCEDURE PutBlanks(blanks: CARDINAL);
      VAR
	 count: CARDINAL;
   BEGIN
      FOR count := 1 TO blanks DO
	 PutChar(" ");
      END;
   END PutBlanks;

   PROCEDURE PutString(str: ARRAY OF CHAR);
      VAR
	 index: CARDINAL;
   BEGIN
      index := 0;
      WHILE (index <= HIGH(str)) & (str[index] # 0C) DO
	 PutChar(str[index]);
	 INC(index);
      END;
   END PutString;

   PROCEDURE Newline;
      (* to be called after PutChar(nl) *)
      (* - does not modify pos - *)
   BEGIN
      IF (linetable[currentline MOD linetablesize].linelength = 0) THEN
	 RETURN
      END;
      IF currentline + 1 - leastline >= linetablesize THEN
	 FlushOneLine;
      END;
      INC(currentline);
      WITH linetable[currentline MOD linetablesize] DO
	 linelength := 0;
      END;
   END Newline;

   PROCEDURE GetCh;					(* EXPORTED *)
      VAR
	 blanks: CARDINAL;			(* for tab to blank mapping *)
   BEGIN
      LOOP
	 IF ~Fgetc(ch,infile) THEN
	    IF (currentline = 1) OR (linetable[currentline MOD
		  linetablesize].linelength # 0) THEN
	       (* incomplete last line - no warning *)
	       PutChar(nl);
	       Newline;
	       INC(pos.line);
	       pos.pos := 0;
	    END;
	    ch := 0C;
	    RETURN
	 END;
	 IF ch = nl THEN
	    PutChar(nl);
	    Newline;
	    INC(pos.line);
	    pos.pos := 0;
	    RETURN
	 END;
	 IF ch = tab THEN
	    blanks := 8 - pos.pos MOD 8;
	    PutBlanks(blanks);
	    INC(pos.pos, blanks);
	    RETURN
	 END;
	 IF ch # 0C THEN
	    PutChar(ch);
	    INC(pos.pos);
	    RETURN
	 END;
	 IF NotEqual(nullbytepos,pos) THEN
	    nullbytepos := errpos;
	    errpos := pos;
	    Error("null-byte(s) in source file");
	    errpos := nullbytepos;
	    nullbytepos := pos;
	 END;
      END; (*LOOP*)
   END GetCh;

   PROCEDURE At(pos: Pos);				(* EXPORTED *)
      (* override default error position for next Error-call *)
   BEGIN
      IF pos.line > currentline THEN
	 Message(bug,assertfmt)
      END;
      errpos := pos;
   END At;

   PROCEDURE WriteToStderr(ch: CHAR);
   BEGIN
      IF ~Fputc(ch,stderr) THEN
      END;
   END WriteToStderr;

   PROCEDURE IgnoreChar(ch: CHAR);
   END IgnoreChar;

   PROCEDURE PutMesgChar(ch: CHAR);
   BEGIN
      IF messagefill <= HIGH(message) THEN
	 message[messagefill] := ch;
	 INC(messagefill);
      END;
   END PutMesgChar;

   PROCEDURE PutMesgString(str: ARRAY OF CHAR);
      VAR
	 index: CARDINAL;
   BEGIN
      index := 0;
      WHILE (index <= HIGH(str)) & (str[index] # 0C) DO
	 PutMesgChar(str[index]);
	 INC(index);
      END;
   END PutMesgString;

   PROCEDURE InitMessage(et: ErrorType);
   BEGIN
      IF (et > warning) THEN
	 errorflag := TRUE;
      END;
      IF et = bug THEN
	 messagefill := 0;
	 writeproc := PutMesgChar;
      ELSIF fatalflag THEN
	 writeproc := IgnoreChar;
      ELSE
	 writeproc := PutMesgChar;
      END;
   END InitMessage;

   PROCEDURE EndMessage(et: ErrorType);

      PROCEDURE InsertMesgIntoTable;
	 (* precondition: enough space in mesgbuffer and mesgtable *)

	 VAR
	    mtinsertpos,
	    mtindex: MesgIndex;		(* index into mesgtable *)
	    notshiftedchars: CARDINAL;	(* no. of chars not to be shifted *)
	    mbufinsertpos: CARDINAL;
	    index: CARDINAL;		(* index into message *)
	    errline: CARDINAL;		(* line where message is printed *)

	 PROCEDURE LessEqual( mi: MesgInfo;
			      line2: CARDINAL; pos2: Pos): BOOLEAN;
	    (* message order:
	     *	1.) effective line (where message should be printed)
	     *	2.) "good" messages (effective line = real error line) first
	     *	3.) real error line
	     *	4.) real error column
	     *)
	 BEGIN
	    IF mi.mesgline # line2 THEN
	       RETURN mi.mesgline <= line2
	    END;
	    IF (pos2.line = line2) # (mi.mesgpos.line = line2) THEN
	       RETURN pos2.line # line2
	    END;
	    IF mi.mesgpos.line # pos2.line THEN
	       RETURN mi.mesgpos.line <= pos2.line
	    END;
	    RETURN (pos2.pos = 0) OR (mi.mesgpos.pos <= pos2.pos)
	 END LessEqual;

      BEGIN
	 mtinsertpos := 0;
	 notshiftedchars := 0;
	 IF errpos.line >= leastline THEN
	    errline := errpos.line;
	 ELSIF (linetable[currentline MOD linetablesize].linelength = 0) &
	       (leastline < currentline) THEN
	    errline := currentline - 1;
	 ELSE
	    errline := currentline;
	 END;
	 WHILE (mtinsertpos < mesgtablefill) &
	       LessEqual(mesgtable[mtinsertpos],errline,errpos) DO
	    INC(notshiftedchars,mesgtable[mtinsertpos].mesglength);
	    INC(mtinsertpos);
	 END;
	 mtindex := mesgtablefill;
	 WHILE mtindex > mtinsertpos DO
	    mesgtable[mtindex] := mesgtable[mtindex-1];
	    DEC(mtindex);
	 END;
	 WITH mesgtable[mtinsertpos] DO
	    mesgline := errline;
	    mesgpos := errpos;
	    mesglength := messagefill;
	 END;
	 INC(mesgtablefill);
	 IF messagefill # 0 THEN
	    IF notshiftedchars < mesgbufferfill THEN
	       FOR index := mesgbufferstart + mesgbufferfill - 1 TO
		     mesgbufferstart + notshiftedchars BY -1 DO
		  mesgbuffer[(index + messagefill) MOD mesgbuffersize] :=
		     mesgbuffer[index MOD mesgbuffersize];
	       END;
	    END;
	    mbufinsertpos :=
	       (mesgbufferstart + notshiftedchars) MOD mesgbuffersize;
	    FOR index := 0 TO messagefill - 1 DO
	       mesgbuffer[mbufinsertpos] := message[index];
	       mbufinsertpos := (mbufinsertpos + 1) MOD mesgbuffersize;
	    END;
	    INC(mesgbufferfill, messagefill);
	    messagefill := 0;
	 END;
      END InsertMesgIntoTable;

   BEGIN
      IF fatalflag & (et # bug) THEN
	 RETURN
      END;
      IF messagefill <= HIGH(message) THEN
	 message[messagefill] := 0C;
      END;
      CASE et OF
      |  headerinfo:
	    IF writing & ~outfilebad THEN
	       IF leastline # 1 THEN
		  FwriteString(outfile,
		     "(continuing list of symbol files ...)");
		  OutfileGood(Done,1);
	       END;
	       FwriteString(outfile,message);
	       OutfileGood(Done,1);
	    END;
	    messagefill := 0;
      |  warning, error:
	    IF et = warning THEN
	       PutMesgString(" (warning only)");
	    END;
	    IF (errpos.line = 0) OR (errpos.line > currentline) THEN
	       Message(bug,"illegal error position")
	    END;
	    IF errpos.pos = 0 THEN
	       DEC(errpos.line);
	       IF (errpos.line >= leastline) THEN
		  errpos.pos :=
		     linetable[errpos.line MOD linetablesize].linelength;
	       END;
	    END;
	    WHILE ((messagefill > mesgbuffersize - mesgbufferfill) OR
		  (mesgtablefill >= mesgtablesize)) DO
	       IF (leastline >= currentline) THEN
		  (* too many messages in one line: ignore this one *)
		  messagefill := 0;
		  RETURN
	       END;
	       FlushOneLine;
	    END;
	    InsertMesgIntoTable;
      |  fatal:
	    fatalflag := TRUE;
	    IF writing THEN
	       WHILE ch # 0C DO
		  GetCh;
	       END;
	    END;
	    EndScan;
	    FwriteString(stderr,message);
	    FwriteString(stderr, " (compilation stopped)");
	    FwriteLn(stderr);
	    Exit(1)
      |  bug:
	    FwriteString(stderr,"---- compiler error");
	    FwriteLn(stderr);
	    FwriteString(stderr,"---- ");
	    FwriteString(stderr,message);
	    FwriteLn(stderr);
	    IF pos.line > 0 THEN
	       FwriteString(stderr,"---- line = ");
	       FwriteCard(stderr,pos.line,0);
	       FwriteString(stderr,", pos = ");
	       FwriteCard(stderr,pos.pos,0);
	       FwriteLn(stderr);
	    END;
	    IF ~CloseAll() THEN		(* flushes output files *)
	    END;
	    HALT
      END;
   END EndMessage;

   PROCEDURE Message(et: ErrorType; msg: ARRAY OF CHAR);	(* EXPORTED *)
   BEGIN
      InitMessage(et);
      PrintFormat(msg);
      EndMessage(et);
   END Message;

   PROCEDURE Message1(et: ErrorType; msg: ARRAY OF CHAR; p1: ARRAY OF BYTE);
   (* EXPORTED *)
   BEGIN
      InitMessage(et);
      PushArgument(p1);
      PrintFormat(msg);
      EndMessage(et);
   END Message1;

   PROCEDURE Message2(et: ErrorType; msg: ARRAY OF CHAR; p1,p2: ARRAY OF BYTE);
   (* EXPORTED *)
   BEGIN
      InitMessage(et);
      PushArgument(p2);
      PushArgument(p1);
      PrintFormat(msg);
      EndMessage(et);
   END Message2;

   PROCEDURE Message3(et: ErrorType; msg: ARRAY OF CHAR;
		     p1,p2,p3: ARRAY OF BYTE);			(* EXPORTED *)
   BEGIN
      InitMessage(et);
      PushArgument(p3);
      PushArgument(p2);
      PushArgument(p1);
      PrintFormat(msg);
      EndMessage(et);
   END Message3;

   PROCEDURE Message4(et: ErrorType; msg: ARRAY OF CHAR;
		     p1,p2,p3,p4: ARRAY OF BYTE);		(* EXPORTED *)
   BEGIN
      InitMessage(et);
      PushArgument(p4);
      PushArgument(p3);
      PushArgument(p2);
      PushArgument(p1);
      PrintFormat(msg);
      EndMessage(et);
   END Message4;

   PROCEDURE Error(msg: ARRAY OF CHAR);			(* EXPORTED *)
   BEGIN
      Message(error,msg);
   END Error;

   PROCEDURE Error1(msg: ARRAY OF CHAR; p1: ARRAY OF BYTE);	(* EXPORTED *)
   BEGIN
      Message1(error,msg,p1);
   END Error1;

   PROCEDURE Error2(msg: ARRAY OF CHAR; p1, p2: ARRAY OF BYTE);	(* EXPORTED *)
   BEGIN
      Message2(error,msg,p1,p2);
   END Error2;

   PROCEDURE Error3(msg: ARRAY OF CHAR;
		    p1, p2, p3: ARRAY OF BYTE); (* EXPORTED *)
   BEGIN
      Message3(error,msg,p1,p2,p3);
   END Error3;

   PROCEDURE Error4(msg: ARRAY OF CHAR;
		    p1, p2, p3, p4: ARRAY OF BYTE); (* EXPORTED *)
   BEGIN
      Message4(error,msg,p1,p2,p3,p4);
   END Error4;

   PROCEDURE ErrorList(msg: ARRAY OF CHAR);		(* EXPORTED *)
      (* produce long error messages, e.g. error message about *)
      (* exported but undefined procedures *)
      (* ErrorList { AddToList } EndOfList *)
   BEGIN
      IF inerrorlist THEN
	 Message(bug,assertfmt)
      END;
      inerrorlist := TRUE;
      InitMessage(error);
      PrintFormat(msg);
   END ErrorList;

   PROCEDURE AddToList(msg: ARRAY OF CHAR; p1: ARRAY OF BYTE);	(* EXPORTED *)
   BEGIN
      IF ~inerrorlist THEN
	 Message(bug,assertfmt)
      END;
      PushArgument(p1);
      PrintFormat(msg);
   END AddToList;

   PROCEDURE EndOfList;					(* EXPORTED *)
   BEGIN
      IF ~inerrorlist THEN
	 Message(bug,assertfmt)
      END;
      EndMessage(error);
      inerrorlist := FALSE;
   END EndOfList;

   PROCEDURE InitScan;					(* EXPORTED *)
   BEGIN
      IF ~Fopen(infile,input,read,(*buff'd*)TRUE) THEN
	 Message2(fatal, perrfmt, input, errno)
      END;
      IF dolist THEN
	 IF ~Fopen(outfile,listing,write,(*buff'd*)TRUE) THEN
	    Message2(fatal, perrfmt, listing, errno)
	 END;
	 writing := TRUE;
	 outfilebad := FALSE;
      END;
      PrintHeader;
      ClearTabs;
      SetTab(20); SetTab(68);
   END InitScan;

   PROCEDURE EndScan;					(* EXPORTED *)

      PROCEDURE WriteMessage;
	 VAR
	    count: CARDINAL;
      BEGIN
	 FOR count := 1 TO linenumberwidth DO
	    OutfileGood( Fputc("*",outfile), 0);
	 END;
	 OutfileGood( Fputc(" ",outfile), 0);
	 FwriteString(outfile,message);
	 OutfileGood(Done,1);
      END WriteMessage;

   BEGIN
      IF inendscan THEN
	 RETURN
      END;
      inendscan := TRUE;
      Newline;		(* so that even an incomplete line can be flushed *)
      WHILE (bufferfill > 0) DO
	 FlushOneLine;
      END;
      IF mesgtablefill > 0 THEN
	 Message(bug,assertfmt)
      END;
      IF writing THEN
	 IF ~outfilebad THEN
	    IF fatalflag THEN
	       FwriteString(outfile,"compilation stopped");
	       OutfileGood(Done,1);
	       WriteMessage;
	    ELSE
	       FwriteString(outfile,"end of compilation");
	    END;
	    OutfileGood(Done,1);
	    PrintStatistics;
	    IF ~Fclose(outfile) THEN
	       outfilebad := TRUE;
	       writing := FALSE;
	       Message2(fatal, perrfmt, listing, errno)
	    END;
	 END;
	 IF outfilebad THEN
	    Delete(listing);
	 END;
      END;
      IF ~Fclose(infile) THEN END;
   END EndScan;

BEGIN
   writing := FALSE;
   outfilebad := FALSE;
   fatalflag := FALSE;
   inendscan := FALSE;
   inerrorlist := FALSE;
   errorflag := FALSE;
   writeproc := WriteToStderr;			(* just for safety ... *)

   leastline := 1;			(* only modified by FlushOneLine *)
   currentline := 1;			(* only modified by Newline *)
   linetable[1].linelength := 0;
   bufferstart := 0;
   bufferpos := 0;
   bufferfill := 0;
   mesgbufferstart := 0;
   mesgbufferfill := 0;
   mesgtablefill := 0;
   messagefill := 0;			(* only modified by EndMessage *)
   pos.line := 1;
   pos.pos := 0;
   errpos := pos;
   ch := nl;					(* anything but 0C *)
   symbolcount := 0;
   outputcount := 0;
END Scan.
