MODULE ShowSym;

   FROM Archive IMPORT AFILE, ArchiveOpen, ArchiveRead, ArchiveClose;
   FROM Calendar IMPORT Time;
   FROM FtdIO IMPORT Fread, Fwrite, Done;
   FROM InOut IMPORT Write, WriteString, WriteLn, WriteCard, WriteInt;
   FROM StdIO IMPORT FILE, read, write, Fopen, Fclose, Fputc, Ftell, Fseek,
      Fgetc;
   FROM SystemTypes IMPORT ProcessId, OFF, DirSize;
   FROM SysExit IMPORT Exit;
   FROM Strings IMPORT StrLen, StrCat, StrCpy;
   FROM SYSTEM IMPORT BYTE;

   CONST
      version = 14;
      magic = 100AFB00H + version;
      NilTypeNo = 0;

   TYPE
      Header =
	 RECORD
	    nmodules: CARDINAL; (* number of modules *)
	    ntypes: CARDINAL;	(* number of types *)
	    nidents: CARDINAL;  (* number of ident-recs *)
	 END;

   TYPE
      Identifier = POINTER TO INTEGER;
      String = POINTER TO INTEGER;
      Symbol = (

	 (* keywords *)
	 arraySY, beginSY, caseSY, constSY, definitionSY, divSY, doSY,
	 elseSY, elsifSY, endSY, exitSY, ifSY, importSY, inSY, isSY, loopSY,
	 modSY, moduleSY, nilSY, ofSY, orSY, pointerSY, procedureSY,
	 recordSY, repeatSY, returnSY, thenSY, toSY, typeSY, untilSY,
	 varSY, whileSY, withSY,

	 (* operators and delimiters *)
	 plus, minus, times, slash, tilde, ampersand, period,
	 comma, semicolon, bar,
	 lparen (* "(" *), lbracket (* "[" *), lbrace (* "{" *),
	 becomes, arrow,
	 eql, neq, lst, grt, leq, geq,
	 range, colon,
	 rparen (* ")" *), rbracket (* "]" *), rbrace (* "}" *),

	 (* miscellaneous symbols *)
	 identSY, stringcon,
	 charcon, intcon, realcon, longrealcon,
	 (* not returned by GetSy: *) setcon, boolcon,
	 eop
	 );
      Constval =
	 RECORD
	    CASE sy: Symbol OF
	    | stringcon:   string: String;
	    | charcon:     charval: CHAR;
	    | intcon:      intval: INTEGER;
	    | realcon,
	      longrealcon: realval: LONGREAL;
	    | setcon:	   setval: BITSET;
	    | boolcon:     boolval: BOOLEAN;
	    | nilSY:
	    END;
	 END;

   (* copied from SymTab... *)
   TYPE
      Form = (
	 shortint, int16, integer, longint, real, longreal, (* numeric types *)
	 boolean, char, set, byte,			(* non-numeric types *)
	 array, record, pointer, proceduretype);	(* structured types *)
      FormSet = SET OF Form;
   CONST
      address = longint;

   TYPE
      StdProc = (absF, oddF, capF, ashF, lenF, adrF, sizeF, maxF, minF,
		 ordF, chrF, shortF, longF, entierF,
		 incP, decP, inclP, exclP, newP, haltP);
      Size = INTEGER; (* type for size and offsets in bytes *)
      Ident = POINTER TO IdentRec;
      Type = POINTER TO TypeRec;
      FieldList = POINTER TO FieldListRec;
      FieldListRec =
	 RECORD
	    id: Identifier;
	    CASE : BOOLEAN OF
	    | TRUE:  typeno: CARDINAL;
	    | FALSE: type: Type;
	    END;
	    offset: Size;
	    link: FieldList;
	    seekpos: OFF;
	 END;
      VarKind = (paramV, varparamV, copyparamV, noparamV);
	 (* paramV:	call by value; value is copied by calling procedure
	    varparmV:	call by reference; address is pushed on stack
	    copyparamV:	call by value; address is pushed on stack
	    noparamV:	no parameter
	 *)
      ParamList = POINTER TO ParamListRec;
      ParamListRec =
	 RECORD
	    id: Identifier;	(* for error messages only *)
	    CASE : BOOLEAN OF
	    | TRUE:  typeno: CARDINAL;
	    | FALSE: type: Type;
	    END;
	    varkind: VarKind;	(* anything but noparamV *)
	    offset: Size;	(* 1st parameter has offset 0 *)
	    link: ParamList;
	    seekpos: OFF;
	 END;
      TypeRec =
	 RECORD
	    CASE : BOOLEAN OF
	    | TRUE:  identno: CARDINAL;	(* for SymFile *)
	    | FALSE: ident: Ident;	(* defining identifier *)
	    END;
	    refcnt: CARDINAL; (* reference count -- used if ident = NIL *)
	    typeno: CARDINAL; (* used by `SymFile' *)
	    rtypeno: CARDINAL;(* used by `SymRef' *)
	    tagno: CARDINAL;  (* used by `GenTypes' *)
	    containsptr: BOOLEAN; (* type containing pointers? *)
	    size: Size;	      (* in bytes *)
	    link: Type;       (* chain of types -- used by `SymFile' *)
	    seekpos: OFF;     (* seek position in symbol file; ident # NIL *)
	    privateparts: BOOLEAN; (* any private parts we don't know about? *)
	    sizemodified: BOOLEAN; (* size modified by SymFile.Update *)
	    CASE form: Form OF
	    | address:   treatAsAddress: BOOLEAN;
	    | array:     CASE dyn: BOOLEAN OF
			 | FALSE: length: Size;
			 END;
			 CASE : BOOLEAN OF
			 | TRUE:  elementtypeno: CARDINAL;
			 | FALSE: element: Type;
			 END;
	    | record:    CASE : BOOLEAN OF
			 | TRUE:  basetypeno: CARDINAL;
			 | FALSE: basetype: Type;    (* may be NIL *)
			 END;
			 fields: FieldList; (* list of known fields *)
			 CASE projection: BOOLEAN OF (* hidden parts possible *)
			 | TRUE: extmod: Ident;   (* quick reference *)
				 extended: BOOLEAN;
			 ELSE
			 END;
	    | pointer:   CASE : BOOLEAN OF
			 | TRUE:  reftypeno: CARDINAL;
			 | FALSE: reftype: Type;    (* may be NIL: forward! *)
			 END;
			 taggedptr: BOOLEAN; (* IS and WITH legal? *)
	    | proceduretype:
			 function: BOOLEAN;
			 CASE std: BOOLEAN OF
			 | TRUE:  stdproc: StdProc;
			 | FALSE: param: ParamList;
				  CASE (* function *) : BOOLEAN OF
				  | TRUE: CASE : BOOLEAN OF
					  | TRUE:  restypeno: CARDINAL;
					  | FALSE: restype: Type;
					  END;
				  END;
			 END;
	    END;
	 END;

   TYPE
      IdentList = POINTER TO IdentListRec;
      IdentListRec =
	 RECORD
	    ident: Ident;
	    link: IdentList;
	 END;
      IdentClass = (moduleC, constC, typeC, varC, procedureC, badclass);
      Key = RECORD time: Time; pid: ProcessId; END; (* set by SymFile *)
      IdentRec =
	 RECORD
	    hidden: POINTER TO INTEGER;
	    name: Identifier;
	    identno: CARDINAL;		(* used by `SymFile'; >0 if set *)
	    error: BOOLEAN;
	    CASE class: IdentClass OF
	    | badclass:
	    | moduleC:  export: IdentList;
			key: Key;
			origname: Identifier;	(* original name *)
			modid: CARDINAL;	(* used by `SymFile' *)
	    ELSE
	       CASE (* symbolfile *) : BOOLEAN OF
	       | TRUE:  typeno: CARDINAL;
	       | FALSE: type: Type;
	       END;
	       CASE : BOOLEAN OF
	       | TRUE:  modno: CARDINAL;	(* used by `SymFile' *)
	       | FALSE: mod: Ident;		(* defining module *)
	       END;
	       exported: BOOLEAN;
	       CASE (* class *) : IdentClass OF
	       | constC:            constval: Constval;
	       | typeC:
	       | varC, procedureC:  plevel: CARDINAL; (* procedure nest level *)
				    CASE : IdentClass OF
				    | varC:        offset: Size;
						   indirect: BOOLEAN;
						   CASE varkind: VarKind OF
						   | copyparamV:  ptroffset:
								     Size;
						   END;
				    | procedureC:  local: IdentList; (* vars *)
						   params: IdentList;
						   parmoffset: Size;
						   varoffset: Size;
						   forward: BOOLEAN;
						   procno: CARDINAL;
				    END;
	       END;
	    END;
	 END;

   PROCEDURE Line(ch: CHAR);
      CONST length = 79;
      VAR i: CARDINAL;
   BEGIN
      FOR i := 1 TO length DO
	 Write(ch);
      END;
      WriteLn;
   END Line;

   PROCEDURE ReadSymFile(modulename: ARRAY OF CHAR);
      (* read symbol file; *)
      (* `modip' serves as reference to the export list and *)
      (* the module key but is not entered into the symbol *)
      (* table *)

      VAR
	 archive: BOOLEAN;
	 in: FILE;
	 ain: AFILE;
	 eof: BOOLEAN;
	 header: Header;

      PROCEDURE Read(VAR buf: ARRAY OF BYTE);
	 VAR
	    index: CARDINAL;
      BEGIN
	 IF archive THEN
	    FOR index := 0 TO HIGH(buf) DO
	       ReadByte(buf[index]);
	    END;
	 ELSE
	    Fread(in, buf);
	    eof := ~Done;
	 END;
      END Read;

      PROCEDURE ReadByte(VAR byte: BYTE);
	 VAR
	    ch: CHAR;
      BEGIN
	 IF archive & ArchiveRead(ain, ch) OR
	    ~archive & Fgetc(ch, in) THEN
	    byte := BYTE(ch);
	 ELSE
	    byte := BYTE(0C);
	    eof := TRUE;
	 END;
      END ReadByte;

      PROCEDURE ReadString() : BOOLEAN;
	 (* returns TRUE if identifier has non-zero length *)
	 VAR
	    ch: CHAR;
	    i: CARDINAL;
      BEGIN
	 ReadByte(ch);
	 IF ch = 0C THEN RETURN FALSE END;
	 WriteString("   `"); i := 0;
	 REPEAT
	    Write(ch); INC(i);
	    ReadByte(ch);
	 UNTIL ch = 0C;
	 Write("'");
	 WHILE i < 20 DO
	    Write(" ");
	    INC(i);
	 END;
	 RETURN TRUE
      END ReadString;

      PROCEDURE Open;
	 (* on return the symbolfile has been successfully opened; *)
	 (* the file position is left beyond the identifier of the *)
	 (* module name *)

	 CONST
	    SYMArchive = "SYM";
	 TYPE
	    BaseName = ARRAY [0..DirSize-1] OF CHAR;
	 VAR
	    symfile: BaseName;

	 PROCEDURE GetBasename(VAR basename: BaseName);
	    CONST
	       suffixlen = 3;
	       symfileSX = ".sy";
	    VAR
	       modname: ARRAY [0..DirSize-suffixlen-1] OF CHAR;
	 BEGIN
	    StrCpy(modname, modulename);
	    StrCpy(basename, modname); StrCat(basename, symfileSX);
	 END GetBasename;

	 PROCEDURE Try(filename: ARRAY OF CHAR) : BOOLEAN;

	    PROCEDURE SymFileOK() : BOOLEAN;
	       VAR
		  magicno: CARDINAL;
		  symmodid: Identifier;
	    BEGIN
	       Read(magicno);
	       IF magicno # magic THEN RETURN FALSE END;
	       Read(header); IF eof THEN RETURN FALSE END;
	       WriteString("module name: ");
	       RETURN ReadString()
	    END SymFileOK;

	 BEGIN
	    IF ArchiveOpen(ain, filename, symfile) THEN
	       archive := TRUE;
	    ELSIF Fopen(in, filename, read, (* buffered = *) TRUE) THEN
	       archive := FALSE;
	    ELSE
	       RETURN FALSE
	    END;
	    eof := FALSE;
	    IF ~SymFileOK() THEN
	       Close;
	       RETURN FALSE
	    END;
	    WriteLn;
	    RETURN TRUE
	 END Try;

      BEGIN (* Open *)
	 GetBasename(symfile);
	 IF Try(symfile) OR Try(SYMArchive) THEN RETURN END;
	 WriteLn;
	 WriteString("symbol file not found"); WriteLn;
	 Exit(1);
      END Open;

      PROCEDURE Close;
      BEGIN
	 IF archive THEN
	    ArchiveClose(ain);
	 ELSE
	    IF ~Fclose(in) THEN END;
	 END;
      END Close;

      PROCEDURE ReadModules;
	 VAR
	    cnt: CARDINAL;
	    k: Key;

	 PROCEDURE WriteKey;
	 BEGIN
	    WITH k DO
	       WriteString("Key: [");
	       WriteInt(time, 1); WriteString(", ");
	       WriteInt(pid, 1); Write("]"); WriteLn;
	    END;
	 END WriteKey;

      BEGIN
	 Read(k);
	 WriteKey;
	 WriteString("imported modules: "); WriteLn;
	 WHILE ReadString() DO
	    Read(k); WriteString("  "); WriteKey;
	    WriteLn;
	 END;
      END ReadModules;

      PROCEDURE ReadTypes;

	 PROCEDURE ReadFields;
	    VAR
	       field: FieldListRec;
	 BEGIN
	    WriteString("field names: "); WriteLn;
	    WHILE ReadString() DO
	       Read(field);
	       WITH field DO
		  WriteString(" type="); WriteCard(typeno, 1);
		  WriteString(" offset="); WriteInt(offset, 1); WriteLn;
	       END;
	    END;
	 END ReadFields;

	 PROCEDURE ReadParams;
	    VAR
	       param: ParamListRec;
	 BEGIN
	    WriteString("parameters: "); WriteLn;
	    WHILE ReadString() DO
	       Read(param);
	       WITH param DO
		  IF varkind = varparamV THEN
		     WriteString(" VAR ");
		  END;
		  WriteString(" type="); WriteCard(typeno, 1);
		  WriteString(" offset="); WriteInt(offset, 1);
		  WriteLn;
	       END;
	    END;
	 END ReadParams;

	 VAR
	    t: TypeRec;
	    index: CARDINAL;

      BEGIN
	 Line("=");
	 FOR index := 1 TO header.ntypes DO
	    Line("-");
	    WriteString("type "); WriteCard(index, 1); WriteLn;
	    Read(t);
	    WITH t DO
	       WriteString("size="); WriteInt(size, 1);
	       IF identno # 0 THEN
		  WriteString(" ident-#="); WriteCard(identno, 1);
	       END;
	       IF privateparts THEN
		  WriteString(" PRV");
	       END;
	       IF sizemodified THEN
		  WriteString(" UPD");
	       END;
	       WriteString(" form=");
	       CASE form OF
	       | shortint: WriteString("shortint"); WriteLn;
	       | int16: WriteString("int16"); WriteLn;
	       | integer: WriteString("integer"); WriteLn;
	       | longint:
		  IF treatAsAddress THEN
		     IF containsptr THEN
			WriteString("traced ");
		     ELSE
			WriteString("untraced ");
		     END;
		     WriteString("address");
		  ELSE
		     WriteString("longint");
		  END;
		  WriteLn;
	       | real: WriteString("real"); WriteLn;
	       | longreal: WriteString("longreal"); WriteLn;
	       | boolean: WriteString("boolean"); WriteLn;
	       | char: WriteString("char"); WriteLn;
	       | set: WriteString("set"); WriteLn;
	       | byte: WriteString("byte"); WriteLn;
	       | array: WriteString("array"); WriteLn;
		  IF dyn THEN WriteString("dynamic array")
		  ELSE WriteString("length="); WriteCard(length, 1);
		  END;
		  WriteString(" elementtype=");
		  WriteCard(elementtypeno, 1); WriteLn;
	       | record: WriteString("record");
		  IF basetypeno # 0 THEN
		     WriteString(" basetypeno=");
		     WriteCard(basetypeno, 1);
		  END;
		  WriteLn;
		  ReadFields;
	       | pointer:
		  IF ~taggedptr THEN
		     WriteString("untagged ");
		  END;
		  IF ~containsptr THEN
		     WriteString("untraced ");
		  END;
		  WriteString("pointer"); WriteLn;
		  WriteString(" referenced type: ");
		  WriteCard(reftypeno, 1); WriteLn;
	       | proceduretype: WriteString("proceduretype"); WriteLn;
		  ReadParams;
		  IF function THEN
		     WriteString(" returns ");
		     WriteCard(restypeno, 1);
		     WriteLn;
		  END;
	       ELSE
		  WriteString("???");
	       END;
	    END;
	 END;
      END ReadTypes;

      PROCEDURE WriteIdentRec(idrec: IdentRec);
      BEGIN
	 WITH idrec DO
	    WriteString("identno = "); WriteCard(identno, 1);
	    IF class = moduleC THEN
	       WriteString("  class=");
               WriteString("module"); WriteLn;
	    ELSE
	       WriteString("  type="); WriteCard(typeno, 1);
	       WriteString("  module="); WriteCard(modno, 1);
	       WriteString("  class=");
	       CASE class OF
	       | constC: WriteString("const");
			 WriteString(" constval=");
			 CASE constval.sy OF
			 | boolcon: IF constval.boolval THEN WriteString("TRUE")
				    ELSE WriteString("FALSE") END;
			 | intcon: WriteInt(constval.intval, 1);
			 ELSE
			 END;
			 WriteLn;
	       | typeC: WriteString("type"); WriteLn;
	       | varC: WriteString("var"); WriteLn;
	       | procedureC: WriteString("procedure");
			WriteString(" procno="); WriteCard(procno, 1);
			WriteLn;
	       ELSE
		  WriteString("???"); WriteLn;
	       END;
	    END;
	 END;
      END WriteIdentRec;

      PROCEDURE ReadIdent(VAR idrec: IdentRec);

	 PROCEDURE ReadStringVal;
	    VAR
	       ch: CHAR;
	 BEGIN
	    WriteString('constval="');
	    ReadByte(ch);
	    WHILE ch # 0C DO
	       Write(ch);
	       ReadByte(ch);
	    END;
	    WriteString('" ');
	 END ReadStringVal;

      BEGIN
	 Read(idrec);
	 WITH idrec DO
	    IF (class = constC) & (constval.sy = stringcon) THEN
	       ReadStringVal;
	    END;
	 END;
      END ReadIdent;

      PROCEDURE ReadExportList;
	 VAR
	    idrec: IdentRec;
      BEGIN
	 Line("=");
	 WriteString("export list:"); WriteLn;
	 WHILE ReadString() DO
	    ReadIdent(idrec);
	    WriteIdentRec(idrec);
	 END;
      END ReadExportList;

      PROCEDURE ReadIdents;
	 VAR
	    idrec: IdentRec;
      BEGIN
	 Line("=");
	 WriteString("other identifiers:"); WriteLn;
	 WHILE ReadString() DO
	    ReadIdent(idrec);
	    WriteIdentRec(idrec);
	 END;
      END ReadIdents;

   BEGIN (* ReadSymFile *)
      Open;
      ReadModules;
      ReadTypes;
      ReadExportList;
      ReadIdents;
      Close;
      Line("=");
   END ReadSymFile;

   VAR
      modulename: ARRAY [0..13] OF CHAR;

BEGIN
   ARGV(modulename, 1);
   ReadSymFile(modulename);
END ShowSym.
