(* Ulm's Oberon Compiler
   Copyright (c) 1989 by University of Ulm, SAI, D-W-7900 Ulm, Germany
   ----------------------------------------------------------------------------
   $Id: StackMan.m2,v 0.2 1993/09/27 12:41:28 borchert Exp $
   ----------------------------------------------------------------------------
   $Log: StackMan.m2,v $
   Revision 0.2  1993/09/27  12:41:28  borchert
   extended support for garbage collector:
      StackAllocPtr added and StackUse returns now a list of
      temporary pointers

   Revision 0.1  1992/07/30  10:49:26  borchert
   Initial revision

   ----------------------------------------------------------------------------
*)

IMPLEMENTATION MODULE StackMan; (* AFB 4/89 *)

   (* allocation and deallocation of temporary stack space *)

   FROM Attributes IMPORT Label;
   FROM Exception IMPORT Assert;
   FROM Machine IMPORT Align, oneword, Direction, stackdirection;
   FROM Memory IMPORT ALLOCATE;
   FROM SymTab IMPORT Size;

   CONST
      blocksize = 32;
   TYPE
      Bitmap = SET OF Size [0..blocksize-1];
      StackList = POINTER TO StackBlock;
      StackBlock =
	 RECORD
	    used: Bitmap;	(* used 4-byte-words *)
	    link: StackList;
	 END;
   VAR
      inblock: BOOLEAN;		(* check calling order *)
      startoffset: Size;	(* set by StackOffset *)
      topoffset: Size;		(* next free position *)
      head, tail: StackList;	(* list of bitmaps *)
      maplen: Size;		(* length of bitmap in words *)
      ptrlist: PointerList;

   PROCEDURE Set(offset, count: Size; used: BOOLEAN);
      VAR
	 sp: StackList;
	 base: Size;
	 mask, range: Bitmap;
	 low, high: Size;
	 invmask: Bitmap;
   BEGIN
      IF used THEN
	 mask := Bitmap{0..blocksize-1};
	 invmask := Bitmap{};
      ELSE
	 mask := Bitmap{};
	 invmask := Bitmap{0..blocksize-1};
      END;
      sp := head; base := 0;
      WHILE base+blocksize <= offset DO
	 sp := sp^.link; INC(base, blocksize);
      END;
      WHILE count > 0 DO
	 IF offset > 0 THEN
	    low := offset MOD blocksize; offset := 0;
	 ELSE
	    low := 0;
	 END;
	 IF low+count >= blocksize THEN
	    high := blocksize - 1;
	 ELSE
	    high := low + count - 1;
	 END;
	 DEC(count, high + 1 - low);
	 range := Bitmap{low..high};
	 Assert(sp^.used * range = invmask * range);
	 sp^.used := sp^.used - range + mask * range;
	 sp := sp^.link;
      END;
   END Set;

   PROCEDURE StackAlloc(VAR offset: Size; size: Size);
      (* `offset' is aligned *)
      VAR
	 sp: StackList;		(* from head to tail *)
	 base: Size;		(* corresponds to `sp' *)
	 count: Size;		(* of consecutive free words *)
	 offs: Size;		(* current offset in bitmap-list *)

      PROCEDURE New;
      BEGIN
	 IF head = NIL THEN
	    NEW(head);
	    tail := head;
	 ELSE
	    NEW(tail^.link);
	    tail := tail^.link;
	 END;
	 WITH tail^ DO
	    used := Bitmap{};
	    link := NIL;
	 END;
	 INC(maplen, blocksize);
      END New;

   BEGIN
      Align(size); size := size DIV oneword;

      (* make sure that we have at least one element in list *)
      WHILE size > maplen DO
	 New;
      END;

      (* look for `size' consecutive free words *)
      sp := head; base := 0;
      offs := 0; count := 0;
      WHILE count < size DO
	 IF base+blocksize <= offs THEN
	    IF sp^.link = NIL THEN New END;
	    sp := sp^.link; INC(base, blocksize);
	 END;
	 IF offs MOD blocksize IN sp^.used THEN
	    count := 0;
	 ELSE
	    INC(count);
	 END;
	 INC(offs);
      END;

      (* `offs' points now to the first word after the free area of
	 `size' words
      *)
      DEC(offs, size);
      Set(offs, size, (* used = *) TRUE);
      IF stackdirection = forwardDir THEN
	 offset := startoffset + offs * oneword;
	 IF offset + size > topoffset THEN
	    topoffset := offset + size * oneword;
	 END;
      ELSE
	 offset := startoffset - (size+offs) * oneword;
	 IF offset < topoffset THEN
	    topoffset := offset;
	 END;
      END;
   END StackAlloc;

   PROCEDURE StackFree(offset: Size; size: Size);
      VAR
	 offs: Size;
   BEGIN
      Align(size); size := size DIV oneword;
      IF stackdirection = forwardDir THEN
	 offs := (offset - startoffset) DIV oneword;
      ELSE
	 offs := (startoffset - offset) DIV oneword - size;
      END;
      Set(offs, size, (* used = *) FALSE);
   END StackFree;

   PROCEDURE StackAllocPtr(VAR offset: Size;
			   startlabel, endlabel: Label);
      (* like StackAlloc but for pointers (=> size = oneword)
	 with additional bookkeeping for the garbage collection:
	 startLabel and endlabel define the pc-range where the
	 pointer has a defined value, i.e. is to be checked for
	 during the garbage collection
      *)
      VAR
	 ptr: PointerList;
   BEGIN
      StackAlloc(offset, oneword);
      NEW(ptr);
      ptr^.offset := offset;
      ptr^.startlabel := startlabel;
      ptr^.endlabel := endlabel;
      ptr^.next := ptrlist;
      ptrlist := ptr;
   END StackAllocPtr;

   PROCEDURE StackOffset(offset: Size);
      (* called at block entry; `offset' is the offset for temporaries *)
   BEGIN
      Assert(~inblock); inblock := TRUE;
      startoffset := offset; topoffset := startoffset;
      ptrlist := NIL;
   END StackOffset;

   PROCEDURE StackUse(VAR size: Size; VAR pointerlist: PointerList);
      (* called at block exit; *)
      (* returns the (aligned) number of bytes used for temporaries *)
      VAR
	 sp: StackList;
   BEGIN
      Assert(inblock); inblock := FALSE;
      sp := head;
      WHILE sp # NIL DO
	 Assert(sp^.used = Bitmap{});
	 sp := sp^.link;
      END;
      size := ABS(topoffset-startoffset);
      pointerlist := ptrlist;
   END StackUse;

BEGIN
   inblock := FALSE;
   head := NIL; tail := NIL; maplen := 0;
END StackMan.
