(* Ulm's Oberon Compiler
   Copyright (c) 1989 by University of Ulm, SAI, D-W-7900 Ulm, Germany
   ----------------------------------------------------------------------------
   $Id: SymFile.m2,v 0.9 1994/05/25 11:51:25 borchert Exp $
   ----------------------------------------------------------------------------
   $Log: SymFile.m2,v $
   Revision 0.9  1994/05/25  11:51:25  borchert
   bug fix: assertion in Update was to strong: ident = NIL is pretty legal
   procedure types and their parameter lists may now be updated

   Revision 0.8  1993/09/27  12:56:40  borchert
   new symbolfile version 13: treatAsAddress has been added to SymTab.TypeRec

   Revision 0.7  1993/06/25  12:33:38  borchert
   new symbolfile version 12: SymTab.TypeRec has been changed

   Revision 0.6  1993/06/18  15:34:23  borchert
   several instances of CARDINAL replaced by LONGCARD to
   assure 32-bit values

   Revision 0.5  1993/06/16  09:48:45  borchert
   new symbol file version 11 due to SYSTEM.INT16

   Revision 0.4  1993/04/19  14:38:27  borchert
   the extra spaces of the import list for the headerinfo has been removed

   Revision 0.3  1993/02/03  12:46:52  borchert
   take output filename from ScanArgs
   new symbol file version: 9
   better error message on version mismatches
   field lists of base types are now longer included

   Revision 0.2  1992/10/22  14:43:10  borchert
   now, field offsets may be patched, too
   a new symbol file format was necessary for this to work
   (seekpos component was added to FieldListRec)

   Revision 0.1  1992/07/30  10:49:42  borchert
   Initial revision

   ----------------------------------------------------------------------------
*)

IMPLEMENTATION MODULE SymFile; (* AFB 1/89 *)

   (* interface to symbol files *)

   FROM Archive IMPORT AFILE, AStat, ArchiveOpen, ArchiveRead, ArchiveClose,
      NameLength, ArchiveStat;
   FROM Calendar IMPORT CurrentTime;
   FROM Exception IMPORT IOFault, Assert;
   FROM FtdIO IMPORT Fread, Fwrite, Done;
   FROM IdentSys IMPORT Identifier, GetIdentChar, GetIdent, PutIdentChar,
      PutIdent;
   FROM Lex IMPORT String, stringcon, PutStringChar, PutString, GetStringChar;
   FROM Memory IMPORT ALLOCATE, DEALLOCATE;
   FROM Scan IMPORT headerinfo, fatal, warning, Message, Message1, Message2,
      Message3, Message4, errorflag;
   FROM ScanArgs IMPORT FileName, input, output, sympath, Path;
   FROM Standard IMPORT InitSysModule, sysmodid;
   FROM StdIO IMPORT FILE, read, write, Fopen, Fclose, Fputc, Ftell, Fseek,
      Fgetc;
   FROM Strings IMPORT StrLen, StrCat, StrCpy;
   FROM Suffixes IMPORT definitionSX, symfileSX;
   FROM SymTab IMPORT Type, Key, Ident, IdentList, Form, IdentClass, FieldList,
      ParamList, TypeRec, IdentRec, SearchList, Find, import, standard,
      FieldListRec, ParamListRec, Used, extmods, mainmod, address;
   FROM SysGetpid IMPORT Getpid;
   FROM SystemTypes IMPORT OFF, DirSize;
   FROM SYSTEM IMPORT BYTE;
   IMPORT Calendar, Conversions, Exception, FtdIO, IdentSys, Memory, Scan,
      ScanArgs, StdIO, SymTab, SysExit, SysGetpid, SystemTypes, SYSTEM;

   (*
	symbol file organisation:

		Principles: a symbol file contains the export list of
	a module. This includes constants, types, variables, and procedures.
	Any imported module is named but not copied into the symbol file.
	Tables are sequences of binary records. Tables of IdentRec,
	ParamRec, and FieldRec are preceded by their identifier.
	Identifiers are 0C-terminated sequences of non-null-characters.
	An empty identifier marks the end of a table.

		Header: Magic number followed by the header structure.

		Modtab: Table of modules. Every module imported and the
	compiled definition-module is included. Modules are later referred
	by their module number.  Numbering starts at 1
	(module number 0 is reserved for standard names). Module number 1
	represents the compiled module. For every module the identifier
	and the key are given.

		Typetab: Table of types. Every type record is included.
	Types are optionally followed by field names and parameter lists.
	The number of types is found in the header.

		Identtab: Table of Ident-records. Every identifier exported
	or used (by types) is included. References between identifiers and
	types are given by numbers. Zero values represent NIL.
	The list of Ident-records is splitted into two parts
	(separated by a zero-length identifier): export list and other
	identifiers.
	IdentRec's of string constants are followed by their 0C-terminated
	string contents.

   *)

   (*   notice about compiler dependency:
	some procedures relay on the $T compiler directive which
	allows to switch off range checks; if the compiler doesn't
	support such directives, the arrays modtab, typetab and idtab
	have to be extended to a sufficient length
   *)

   (*   REV AFB 4/91

	Some hacks have been added to update symbol file in archives.
	The current way of doing this depends on the ar-format.
	On the other side, mmo & make work with this solution.

	Calling ar(1) by this module would be a violation of the
	politics of oc(1): oc knows everything about the environment,
	oberon knows how to compile Oberon.
	A better solution needs an Archive module which supports
	write accesses.
   *)

   CONST
      (* we accept versions which range from oldversion to version;
	 version history:

	 10: field lists include no longer the fields of the base type
	 11: int16 has been added to SymTab.Form
	 12: taggedptr added to SymTab.TypeRec (form = pointer)
	 13: treatAsAddress added to SymTab.TypeRec (form = address)
	 14: seekpos added to ParamListRec
      *)
      version = 14;             (* current version *)
      oldversion = 14;          (* last supported version *)
      prefix = 100AFB00H;
      magic = prefix + version;
      NilTypeNo = 0;

   TYPE
      Header =
	 RECORD
	    nmodules: LONGCARD; (* number of modules *)
	    ntypes: LONGCARD;	(* number of types *)
	    nidents: LONGCARD;  (* number of ident-recs *)
	 END;

   VAR
      typelist: Type;		(* list of all needed types *)
      firstheader: BOOLEAN;
      oursymfile: FileName;	(* symbolfile name of our module *)
				(* set by `ReadSymFile' *)
      oursyminarchive: BOOLEAN; (* "our" symbol file archived in SYM? *)
      symoffset: OFF;		(* offset position to beginning of
				   the symbol file in the archive
				*)
      keypos: OFF;		(* offset position to key in symbol file
				   of "our" module *)

   PROCEDURE InitType(VAR t: Type);
      (* all types must be linked into a chain for symbol file generation *)
   BEGIN
      NEW(t);
      WITH t^ DO
	 link := typelist;
	 typeno := 1;
	 rtypeno := 0;
	 tagno := 0;
	 containsptr := FALSE;
	 refcnt := 0;
	 privateparts := FALSE;
	 sizemodified := FALSE;
      END;
      typelist := t;
   END InitType;

   PROCEDURE LinkType(t: Type);
   BEGIN
      IF t # NIL THEN
	 WITH t^ DO
	    IF typeno = 0 THEN
	       link := typelist;
	       typeno := 1;
	       typelist := t;
	    END;
	 END;
      END;
   END LinkType;

   PROCEDURE ExtMod(VAR ip: Ident);
      VAR new: IdentList;
   BEGIN
      NEW(new);
      WITH new^ DO
	 ident := ip;
	 link := extmods;
      END;
      extmods := new;
   END ExtMod;

   PROCEDURE GenSymFile(modip: Ident);
      (* generate symbol file; file names are taken from `ScanArgs' *)

      VAR
	 out: FILE;
	 headerpos: OFF;
	 header: Header;
	 symfile: FileName;
	 identlist: IdentList; nextidno: CARDINAL;
	 typelist2, typelist2t: Type; (* type list for temporary use *)
	 typecnt: CARDINAL;

      PROCEDURE Open;
      BEGIN
	 symfile := output;
	 IF ~Fopen(out, symfile, write, (* buffered = *) TRUE) THEN
	    IOFault(symfile);
	 END;
      END Open;

      PROCEDURE Close;
      BEGIN
	 IF ~Fclose(out) THEN
	    IOFault(symfile);
	 END;
      END Close;

      PROCEDURE Write(buf: ARRAY OF BYTE);
      BEGIN
	 Fwrite(out, buf);
	 IF ~Done THEN IOFault(symfile) END;
      END Write;

      PROCEDURE WriteByte(b: BYTE);
      BEGIN
	 IF ~Fputc(CHAR(b), out) THEN IOFault(symfile) END;
      END WriteByte;

      PROCEDURE GetPos(VAR pos: OFF);
      BEGIN
	 IF ~Ftell(out, pos) THEN IOFault(symfile) END;
      END GetPos;

      PROCEDURE Seek(pos: OFF);
      BEGIN
	 IF ~Fseek(out, pos, 0) THEN IOFault(symfile) END;
      END Seek;

      PROCEDURE WriteString(name: Identifier);
	 VAR
	    ch: CHAR;
      BEGIN
	 REPEAT
	    GetIdentChar(name, ch);
	    WriteByte(ch);
	 UNTIL ch = 0C;
      END WriteString;

      PROCEDURE TerminateList;
      BEGIN
	 WriteByte(0C);
      END TerminateList;

      PROCEDURE ModuleTab(VAR nmodules: LONGCARD);
	 VAR
	    mod: IdentList;
	    cnt: CARDINAL;

	 PROCEDURE Module(modip: Ident);
	 BEGIN
	    WITH modip^ DO
	       INC(cnt); modid := cnt;
	       WriteString(origname);
	       Write(key);
	    END;
	 END Module;

      BEGIN
	 WITH modip^.key DO
	    time := CurrentTime();
	    pid := Getpid();
	 END;
	 cnt := 0;
	 Module(modip); (* "our" module *)
	 mod := extmods;
	 WHILE mod # NIL DO
	    IF mod^.ident # modip THEN
	       (* imported modules; even indirect imports *)
	       Module(mod^.ident);
	    END;
	    mod := mod^.link;
	 END;
	 TerminateList;
	 nmodules := cnt;
      END ModuleTab;

      PROCEDURE TypeNo(t: Type) : CARDINAL;
      BEGIN
	 IF t = NIL THEN
	    RETURN NilTypeNo
	 ELSIF t^.typeno = 0 THEN
	    WITH t^ DO
	       INC(typecnt); typeno := typecnt;
	       link := NIL;
	       IF typelist2t = NIL THEN
		  typelist2 := t;
	       ELSE
		  typelist2t^.link := t;
	       END;
	       typelist2t := t;
	       RETURN typeno
	    END;
	 ELSE
	    RETURN t^.typeno
	 END;
      END TypeNo;

      PROCEDURE IdentNo(ip: Ident) : CARDINAL;
	 VAR
	    il: IdentList;
      BEGIN
	 IF ip = NIL THEN
	    RETURN 0
	 ELSE
	    WITH ip^ DO
	       IF identno = 0 THEN
		  identno := nextidno; INC(nextidno);
		  IF mod # modip THEN
		     (* other identifier *)
		     NEW(il);
		     WITH il^ DO
			ident := ip;
			link := identlist;
		     END;
		     identlist := il;
		  END;
	       END;
	       RETURN identno
	    END;
	 END;
      END IdentNo;

      PROCEDURE WriteTypes(VAR cnt: LONGCARD);
	 VAR
	    ptr: Type;
	    trec: TypeRec;
	    il: IdentList;

	 PROCEDURE WriteFields(fields, termfield: FieldList);
	    (* revision 10: don't include field list of base type *)
	    VAR
	       fp: FieldList;
	       frec: FieldListRec;
	 BEGIN
	    fp := fields;
	    WHILE fp # termfield DO
	       Assert(fp # NIL);
	       frec := fp^;
	       WITH frec DO
		  WriteString(id);
		  typeno := TypeNo(type);
		  link := NIL;
	       END;
	       Write(frec);
	       fp := fp^.link;
	    END;
	    TerminateList;
	 END WriteFields;

	 PROCEDURE WriteParams(params: ParamList);
	    VAR
	       pp: ParamList;
	       prec: ParamListRec;
	 BEGIN
	    pp := params;
	    WHILE pp # NIL DO
	       prec := pp^;
	       WITH prec DO
		  WriteString(id);
		  typeno := TypeNo(type);
		  link := NIL;
	       END;
	       Write(prec);
	       pp := pp^.link;
	    END;
	    TerminateList;
	 END WriteParams;

      BEGIN (* WriteTypes *)
	 (* complete list of needed types *)
	 il := modip^.export;
	 WHILE il # NIL DO
	    WITH il^.ident^ DO
	       IF (class # moduleC) & (class # badclass) THEN
		  LinkType(type);
	       END;
	    END;
	    il := il^.link;
	 END;

	 cnt := 0;
	 ptr := typelist;
	 WHILE ptr # NIL DO
	    INC(cnt);
	    ptr^.typeno := cnt;
	    ptr := ptr^.link;
	 END;
	 typecnt := cnt;

	 ptr := typelist;
	 REPEAT (* until all referenced types are written *)
	    typelist2 := NIL; typelist2t := NIL;
	    WHILE ptr # NIL DO
	       trec := ptr^;
	       WITH trec DO
		  identno := IdentNo(ident);
		  CASE form OF
		  | array:         elementtypeno := TypeNo(element);
		  | record:        basetypeno := TypeNo(basetype);
		  | pointer:       reftypeno := TypeNo(reftype);
		  | proceduretype: IF function THEN
				      restypeno := TypeNo(restype);
				   END;
		  ELSE
		  END;
		  Write(trec);
		  CASE form OF
		  | record:        IF ptr^.basetype # NIL THEN
				      (* beware of using basetype of trec *)
				      WriteFields(fields,
					    ptr^.basetype^.fields);
				   ELSE
				      WriteFields(fields, NIL);
				   END;
		  | proceduretype: WriteParams(param);
		  ELSE
		  END;
	       END;
	       ptr := ptr^.link;
	    END;
	    ptr := typelist2;
	 UNTIL ptr = NIL;

	 cnt := typecnt;
      END WriteTypes;

      PROCEDURE WriteIdents;
	 VAR
	    il: IdentList;

	 PROCEDURE WriteIdent(ip: Ident);
	    VAR
	       ch: CHAR;
	       idrec: IdentRec;

	    PROCEDURE WriteStringVal(s: String);
	       VAR
		  ch: CHAR;
	    BEGIN
	       REPEAT
		  GetStringChar(s, ch);
		  WriteByte(ch);
	       UNTIL ch = 0C;
	    END WriteStringVal;

	 BEGIN
	    idrec := ip^;
	    WITH idrec DO
	       IF (class # moduleC) & (class # badclass) THEN
		  typeno := TypeNo(type);
		  IF mod # NIL THEN
		     modno := mod^.modid;
		  ELSE
		     modno := 0; (* standard name *)
		  END;
	       END;
	       WriteString(name);
	       Write(idrec);
	       IF (class = constC) & (constval.sy = stringcon) THEN
		  WriteStringVal(constval.string);
	       END;
	    END;
	 END WriteIdent;

	 PROCEDURE WriteList(il: IdentList);
	 BEGIN
	    WHILE il # NIL DO
	       WriteIdent(il^.ident);
	       il := il^.link;
	    END;
	    TerminateList;
	 END WriteList;

      BEGIN
	 WriteList(modip^.export); (* export list *)
	 WriteList(identlist); (* other ident-records *)
      END WriteIdents;

   BEGIN (* GenSymFile *)
      IF errorflag THEN RETURN END;
      identlist := NIL; nextidno := 1;
      Open;
      Write(magic);
      GetPos(headerpos); Write(header);
      WITH header DO
	 ModuleTab(nmodules);
	 WriteTypes(ntypes);
	 WriteIdents;
	 nidents := nextidno-1;
      END;
      Seek(headerpos); Write(header);
      Close;
   END GenSymFile;

   MODULE ModuleNames;

      (* manage list of module names; *)
      (* this cannot be done in use of the symbol table because *)
      (* we look for the original names and not for the aliases *)

      FROM IdentSys IMPORT Identifier;
      FROM Memory IMPORT ALLOCATE;
      FROM SymTab IMPORT Ident, IdentClass;

      EXPORT SearchModule, AddModule;

      TYPE
	 ModuleList = POINTER TO ModuleListRec;
	 ModuleListRec =
	    RECORD
	       id: Identifier;
	       ip: Ident;
	       link: ModuleList;
	    END;

      VAR
	 list: ModuleList;

      PROCEDURE SearchModule(modid: Identifier; VAR modip: Ident) : BOOLEAN;
	 VAR
	    ptr: ModuleList;
      BEGIN
	 ptr := list;
	 WHILE ptr # NIL DO
	    WITH ptr^ DO
	       IF id = modid THEN
		  modip := ip;
		  RETURN TRUE
	       END;
	    END;
	    ptr := ptr^.link;
	 END;
	 RETURN FALSE
      END SearchModule;

      PROCEDURE AddModule(modname: Identifier; VAR modip: Ident);
	 VAR
	    new: ModuleList;
      BEGIN
	 NEW(modip);
	 WITH modip^ DO
	    name := modname;
	    origname := modname;
	    identno := 0;
	    error := FALSE;
	    class := moduleC;
	    export := NIL;
	    (* key defined later *)
	 END;
	 NEW(new);
	 WITH new^ DO
	    id := modname;
	    ip := modip;
	    link := list;
	 END;
	 list := new;
      END AddModule;

   BEGIN
      list := NIL;
   END ModuleNames;

   PROCEDURE ReadSymFile(modid: Identifier; VAR modip: Ident;
			 ourmodule: BOOLEAN);
      (* read symbol file; *)
      (* `modip' serves as reference to the export list and *)
      (* the module key but is not entered into the symbol *)
      (* table *)
      (* `ourmodule' (only if ~defunit): indicates if the *)
      (* corresponding definition is to be read; this information *)
      (* is necessary to remember the filename for symbol file updates *)

      VAR
	 symfileversion: LONGCARD;
	 archive: BOOLEAN; archiveOffset: OFF; archivePos: OFF;
	 in: FILE;
	 ain: AFILE;
	 eof: BOOLEAN;
	 inputfile: FileName;
	 header: Header;
	 modtab: POINTER TO ARRAY [0..0] OF Ident;
	 typetab: POINTER TO ARRAY [0..0] OF Type;
	 idtab: POINTER TO ARRAY [0..0] OF Ident;

      PROCEDURE Read(VAR buf: ARRAY OF BYTE);
	 VAR
	    index: CARDINAL;
      BEGIN
	 IF archive THEN
	    FOR index := 0 TO HIGH(buf) DO
	       ReadByte(buf[index]);
	    END;
	 ELSE
	    Fread(in, buf);
	    eof := ~Done;
	 END;
      END Read;

      PROCEDURE Tell(VAR pos: OFF);
      BEGIN
	 IF ~ourmodule THEN
	    pos := 0;
	 ELSIF archive THEN
	    pos := archiveOffset + archivePos;
	 ELSE
	    IF ~Ftell(in, pos) THEN
	       IOFault(inputfile);
	    END;
	 END;
      END Tell;

      PROCEDURE ReadByte(VAR byte: BYTE);
	 VAR
	    ch: CHAR;
      BEGIN
	 IF archive & ArchiveRead(ain, ch) OR ~archive & Fgetc(ch, in) THEN
	    INC(archivePos);
	    byte := BYTE(ch);
	 ELSE
	    byte := BYTE(0C);
	    eof := TRUE;
	 END;
      END ReadByte;

      PROCEDURE ReadString(VAR id: Identifier) : BOOLEAN;
	 (* returns TRUE if identifier has non-zero length *)
	 VAR ch: CHAR;
      BEGIN
	 ReadByte(ch);
	 IF ch = 0C THEN RETURN FALSE END;
	 REPEAT
	    PutIdentChar(ch);
	    ReadByte(ch);
	 UNTIL ch = 0C;
	 PutIdent(id);
	 RETURN TRUE
      END ReadString;

      PROCEDURE Open;
	 (* on return the symbolfile has been successfully opened; *)
	 (* the file position is left beyond the first member of   *)
	 (* module table                                           *)

	 CONST
	    SYMArchive = "SYM";
	 TYPE
	    BaseName = ARRAY [0..DirSize-1] OF CHAR;
	 VAR
	    symp: Path;
	    symfile: BaseName;
	    membername: BaseName;
	    warningPrinted: BOOLEAN;

	 PROCEDURE GetBasename(VAR basename: BaseName);
	    CONST
	       suffixlen = 3;
	    VAR
	       modname: ARRAY [0..DirSize-suffixlen-1] OF CHAR;
	 BEGIN
	    Assert(StrLen(symfileSX) = suffixlen);
	    GetIdent(modid, modname);
	    StrCpy(basename, modname); StrCat(basename, symfileSX);
	 END GetBasename;

	 PROCEDURE GetMembername(VAR membername: BaseName);
	    CONST
	       suffixlen = 3;
	    VAR
	       modname: ARRAY [0..NameLength-suffixlen-1] OF CHAR;
	 BEGIN
	    Assert(StrLen(symfileSX) = suffixlen);
	    GetIdent(modid, modname);
	    StrCpy(membername, modname); StrCat(membername, symfileSX);
	 END GetMembername;

	 PROCEDURE Try(filename: ARRAY OF CHAR) : BOOLEAN;

	    VAR
	       arstat: AStat;

	    PROCEDURE SymFileOK() : BOOLEAN;
	       VAR
		  magicno: LONGCARD;
		  symmodid: Identifier;
	    BEGIN
	       Read(magicno);
	       IF magicno # magic THEN
		  IF magicno >= prefix THEN
		     symfileversion := magicno - prefix;
		     IF (symfileversion < oldversion) OR
			   (symfileversion > version) THEN
			IF symfileversion < oldversion THEN
			   (* probably an old version *)
			   IF ~warningPrinted THEN
			      Message2(warning,
				 "%s contains an old-fashioned symbol file of version %c",
				 filename, symfileversion);
			      warningPrinted := TRUE;
			   END;
			END;
			RETURN FALSE
		     END;
		     (* now we have an oldfashioned symbol file
			which is still supported
		     *)
		  ELSE
		     RETURN FALSE
		  END;
	       ELSE
		  symfileversion := version;
	       END;
	       Read(header); IF eof THEN RETURN FALSE END;
	       RETURN ReadString(symmodid) & (symmodid = modid)
	    END SymFileOK;

	 BEGIN (* Try *)
	    IF ArchiveOpen(ain, filename, membername) THEN
	       archive := TRUE;
	       ArchiveStat(ain, arstat);
	       archiveOffset := arstat.offset;
	       archivePos := 0;
	    ELSIF Fopen(in, filename, read, (* buffered = *) TRUE) THEN
	       archive := FALSE;
	    ELSE
	       RETURN FALSE
	    END;
	    eof := FALSE;
	    IF ~SymFileOK() THEN
	       Close;
	       RETURN FALSE
	    END;
	    IF firstheader THEN
	       Message(headerinfo, "symbol files:");
	       firstheader := FALSE;
	    END;
	    IF ourmodule THEN
	       Tell(keypos); (* needed for updates *)
	    END;
	    Read(modip^.key);
	    IF archive THEN
	       Message4(headerinfo, "   %I %T%s(%s) %T%d",
		  modid, filename, symfile, modip^.key.time);
	    ELSE
	       Message3(headerinfo, "   %I %T%s %T%d",
		  modid, filename, modip^.key.time);
	    END;
	    IF ourmodule THEN
	       StrCpy(oursymfile, filename);
	       oursyminarchive := archive;
	       IF archive THEN
		  symoffset := archiveOffset;
	       END;
	    END;
	    StrCpy(inputfile, filename);
	    RETURN TRUE
	 END Try;

      BEGIN (* Open *)
	 GetBasename(symfile);
	 GetMembername(membername);
	 warningPrinted := FALSE;
	 symp := sympath;
	 WHILE symp # NIL DO
	    IF Try(symp^.symfile) THEN RETURN END;
	    symp := symp^.link;
	 END;
	 IF Try(symfile) OR Try(membername) OR Try(SYMArchive) THEN RETURN END;
	 Message1(fatal, "symbolfile for module %I not found", modid);
      END Open;

      PROCEDURE Close;
      BEGIN
	 IF archive THEN
	    ArchiveClose(ain);
	 ELSE
	    IF ~Fclose(in) THEN END;
	 END;
      END Close;

      PROCEDURE InitTabs;
      BEGIN
	 WITH header DO
	    ALLOCATE(modtab, nmodules * SIZE(Ident));
	    IF ntypes > 0 THEN
	       ALLOCATE(typetab, ntypes * SIZE(Type));
	    END;
	    IF nidents > 0 THEN
	       ALLOCATE(idtab, nidents * SIZE(Ident));
	    END;
	 END;
      END InitTabs;

      PROCEDURE DisposeTabs;
      BEGIN
	 WITH header DO
	    DEALLOCATE(modtab, nmodules * SIZE(Ident));
	    IF ntypes > 0 THEN
	       DEALLOCATE(typetab, ntypes * SIZE(Type));
	    END;
	    IF nidents > 0 THEN
	       DEALLOCATE(idtab, nidents * SIZE(Ident));
	    END;
	 END;
      END DisposeTabs;

      PROCEDURE GetModule(modno: LONGCARD) : Ident;
      BEGIN
	 (* $T- *)
	 Assert(modno <= header.nmodules);
	 IF modno = 0 THEN
	    RETURN NIL
	 ELSE
	    RETURN modtab^[modno-1]
	 END;
	 (* $T= *)
      END GetModule;

      PROCEDURE SetModule(modno: LONGCARD; modip: Ident);
      BEGIN
	 (* $T- *)
	 Assert((modno > 0) & (modno <= header.nmodules));
	 modtab^[modno-1] := modip;
	 (* $T= *)
      END SetModule;

      PROCEDURE GetIdentPtr(identno: LONGCARD) : Ident;
      BEGIN
	 (* $T- *)
	 Assert(identno <= header.nidents);
	 IF identno = 0 THEN
	    RETURN NIL
	 ELSE
	    RETURN idtab^[identno-1]
	 END;
	 (* $T= *)
      END GetIdentPtr;

      PROCEDURE SetIdent(identno: LONGCARD; ident: Ident);
      BEGIN
	 (* $T- *)
	 Assert((identno > 0) & (identno <= header.nidents));
	 idtab^[identno-1] := ident;
	 (* $T= *)
      END SetIdent;

      PROCEDURE GetType(typeno: LONGCARD) : Type;
      BEGIN
	 (* $T- *)
	 Assert(typeno <= header.ntypes);
	 IF typeno = 0 THEN
	    RETURN NIL
	 ELSE
	    RETURN typetab^[typeno-1]
	 END;
	 (* $T= *)
      END GetType;

      PROCEDURE SetType(typeno: LONGCARD; type: Type);
      BEGIN
	 (* $T- *)
	 Assert((typeno > 0) & (typeno <= header.ntypes));
	 typetab^[typeno-1] := type;
	 (* $T= *)
      END SetType;

      PROCEDURE ReadModules;
	 CONST
	    keymismatch =
	    "module %I is imported in different versions dated from %t and %t";
	 VAR
	    modname: Identifier;
	    modip: Ident;
	    cnt: CARDINAL;
	    k: Key;
      BEGIN
	 cnt := 2;
	 WHILE ReadString(modname) DO
	    IF ~SearchModule(modname, modip) THEN
	       ReadSymFile(modname, modip, (* oursymfile = *) FALSE);
	    END;
	    Read(k);
	    WITH modip^ DO
	       IF (k.time # key.time) OR (k.pid # key.pid) THEN
		  Message3(fatal, keymismatch, modname, k.time, key.time);
	       END;
	    END;
	    SetModule(cnt, modip); INC(cnt);
	 END;
      END ReadModules;

      PROCEDURE ReadTypes;
	 VAR
	    index: LONGCARD;
	    t: Type;
	    pos: OFF;

	 PROCEDURE ReadFields(VAR fields: FieldList);
	    VAR
	       fieldid: Identifier;
	       field: FieldList;
	       seekpos: OFF;
	 BEGIN
	    fields := NIL;
	    WHILE ReadString(fieldid) DO
	       NEW(field);
	       Tell(seekpos);
	       Read(field^);
	       field^.seekpos := seekpos;
	       WITH field^ DO
		  id := fieldid;
		  link := fields;
	       END;
	       fields := field;
	    END;
	 END ReadFields;

	 PROCEDURE ReadParams(VAR params: ParamList);
	    VAR
	       paramid: Identifier;
	       param: ParamList;
	       tail: ParamList;	(* order of parameters is significant! *)
	       seekpos: OFF;
	 BEGIN
	    params := NIL; tail := NIL;
	    WHILE ReadString(paramid) DO
	       NEW(param);
	       Tell(seekpos);
	       Read(param^);
	       param^.seekpos := seekpos;
	       WITH param^ DO
		  id := paramid;
		  link := NIL;
	       END;
	       IF tail = NIL THEN
		  params := param;
	       ELSE
		  tail^.link := param;
	       END;
	       tail := param;
	    END;
	 END ReadParams;

      BEGIN (* ReadTypes *)
	 FOR index := 1 TO header.ntypes DO
	    NEW(t);
	    Tell(pos); Read(t^);
	    WITH t^ DO
	       link := NIL;
	       typeno := 0;
	       rtypeno := 0;
	       seekpos := pos;
	       IF ~ourmodule & sizemodified THEN
		  privateparts := TRUE;
	       END;
	       IF form = record THEN
		  ReadFields(fields);
	       ELSIF form = proceduretype THEN
		  ReadParams(param);
	       END;
	    END;
	    SetType(index, t);
	 END;
      END ReadTypes;

      PROCEDURE ReadIdent(VAR idrec: IdentRec);

	 PROCEDURE ReadStringVal(VAR s: String);
	    VAR
	       ch: CHAR;
	 BEGIN
	    ReadByte(ch);
	    WHILE ch # 0C DO
	       PutStringChar(ch);
	       ReadByte(ch);
	    END;
	    PutString(s);
	 END ReadStringVal;

      BEGIN
	 Read(idrec);
	 WITH idrec DO
	    IF (class = constC) & (constval.sy = stringcon) THEN
	       ReadStringVal(constval.string);
	    END;
	 END;
      END ReadIdent;

      PROCEDURE ReadExportList(VAR explist: IdentList);
	 VAR
	    id: Identifier;
	    expid: Ident;
	    element: IdentList;
      BEGIN
	 explist := NIL;
	 WHILE ReadString(id) DO
	    NEW(expid);
	    ReadIdent(expid^);
	    WITH expid^ DO
	       name := id;
	       IF identno # 0 THEN
		  SetIdent(identno, expid);
	       END;
	       Assert((class # badclass) & (class # moduleC));
	       (* `type' is set by `FixIdents' *)
	       mod := GetModule(1); (* exporting module *)
	    END;
	    NEW(element);
	    WITH element^ DO
	       ident := expid;
	       link := explist;
	    END;
	    explist := element;
	 END;
      END ReadExportList;

      PROCEDURE ReadIdents;
	 VAR
	    id: Identifier;
	    ip: Ident;
      BEGIN
	 WHILE ReadString(id) DO
	    NEW(ip);
	    ReadIdent(ip^);
	    WITH ip^ DO
	       name := id;
	       Assert(identno # 0);
	       SetIdent(identno, ip);
	       Assert((class # badclass) & (class # moduleC));
	       (* type is set in FixIdents *)
	       mod := GetModule(modno);
	    END;
	 END;
      END ReadIdents;

      PROCEDURE FixTypeTable;
	 (* types and identifiers are read;			*)
	 (* before resolving identifier references to types	*)
	 (* we have to check the types which are standard names	*)
	 (* or 3rd-party types.					*)
	 (* In these cases the type references in the type	*)
	 (* table must be replaced by the correct ones.		*)
	 VAR
	    index: LONGCARD;	(* index into type-table *)
	    oldt, t: Type;	(* old and corrected type reference *)
	    knownip: Ident;	(* standard name or 3rd party ref *)
      BEGIN
	 FOR index := 1 TO header.ntypes DO
	    t := GetType(index);
	    oldt := t;
	    WITH oldt^ DO
	       ident := GetIdentPtr(identno);
	       IF ident # NIL THEN
		  WITH ident^ DO
		     Assert((class # badclass) & (class # moduleC));
		     IF mod = NIL THEN (* standard names *)
			Assert(Find(standard, name, knownip) &
			       (knownip^.class = typeC));
			t := knownip^.type;
		     ELSIF mod # modip THEN (* 3rd-party identifier *)
			Assert(SearchList(name, mod^.export, knownip));
			t := knownip^.type;
		     END;
		  END;
	       END;
	    END;
	    SetType(index, t);
	    (* LinkType(t); *)
	    t^.typeno := 0;
	 END;
      END FixTypeTable;

      PROCEDURE FixTypes;
	 VAR
	    index: LONGCARD;
	    t: Type;

	 PROCEDURE FixFields(type: Type);
	    VAR
	       field, prev: FieldList;
	 BEGIN
	    WITH type^ DO
	       field := fields; prev := NIL;
	       WHILE field # NIL DO
		  WITH field^ DO
		     type := GetType(typeno);
		  END;
		  prev := field; field := field^.link;
	       END;
	       IF basetype # NIL THEN
		  IF prev = NIL THEN
		     fields := basetype^.fields;
		  ELSE
		     prev^.link := basetype^.fields;
		  END;
	       END;
	    END;
	 END FixFields;

	 PROCEDURE FixParams(params: ParamList);
	 BEGIN
	    WHILE params # NIL DO
	       WITH params^ DO
		  type := GetType(typeno);
	       END;
	       params := params^.link;
	    END;
	 END FixParams;

      BEGIN
	 FOR index := 1 TO header.ntypes DO
	    t := GetType(index);
	    WITH t^ DO
	       (* `ident' set in FixTypeTable *)
	       IF (ident = NIL) OR (ident^.mod = modip) THEN
		  CASE form OF
		  | array:         element := GetType(elementtypeno);
		  | record:        basetype := GetType(basetypeno);
				   IF projection THEN
				      extmod := ident^.mod;
				   END;
				   FixFields(t);
		  | pointer:       reftype := GetType(reftypeno);
		  | proceduretype: IF function THEN
				      restype := GetType(restypeno);
				   END;
				   FixParams(param);
		  ELSE
		  END;
	       (* else: references resolved in FixTypeTable *)
	       END;
	    END;
	 END;
      END FixTypes;

      PROCEDURE FixIdents;
	 (* resolve type references of identifiers *)
	 VAR
	    index: LONGCARD;
	    ip: Ident;
	    list: IdentList;

      BEGIN
	 list := modip^.export;
	 WHILE list # NIL DO
	    WITH list^.ident^ DO
	       IF identno = 0 THEN
		  type := GetType(typeno);
	       ELSE
		  (* identifier in both lists: identifier table and	*)
		  (* export list -- so "type := GetType(typeno)"	*)
		  (* will be done below. 				*)
	       END;
	    END;
	    list := list^.link;
	 END;
	 FOR index := 1 TO header.nidents DO
	    ip := GetIdentPtr(index);
	    WITH ip^ DO
	       type := GetType(typeno);
	       identno := 0;
	    END;
	 END;
      END FixIdents;

   BEGIN (* ReadSymFile *)
      IF ~SearchModule(modid, modip) THEN
	 AddModule(modid, modip);
	 ExtMod(modip);
	 IF ~ourmodule & (modid = sysmodid) THEN
	    InitSysModule(modip);
	 ELSE
	    Open;
	    InitTabs;
	    WITH modip^ DO
	       export := NIL;
	    END;
	    SetModule(1, modip);
	    ReadModules;
	    ReadTypes;
	    ReadExportList(modip^.export);
	    ReadIdents;
	    FixTypeTable;
	    FixTypes;
	    FixIdents;
	    DisposeTabs;
	    Close;
	 END;
      END;
   END ReadSymFile;

   MODULE Updates;

      FROM Calendar IMPORT CurrentTime;
      FROM Conversions IMPORT ConvertInteger;
      FROM Exception IMPORT Assert, IOFault;
      FROM FtdIO IMPORT Done, Fread, Fwrite, FwriteInt;
      FROM Scan IMPORT Error, errorflag;
      FROM ScanArgs IMPORT FileName, updatesym, UpdateMode, defunit, input;
      FROM StdIO IMPORT FILE, Fopen, Fclose, Fseek, read, append;
      FROM SymTab IMPORT Form, Ident, Type, TypeRec,
	 ParamList, ParamListRec, FieldList, FieldListRec, mainmod;
      FROM SysExit IMPORT EnterCleanup;
      FROM SysGetpid IMPORT Getpid;
      FROM SystemTypes IMPORT TIME, OFF;
      FROM SYSTEM IMPORT BYTE;
      IMPORT keypos, oursymfile, oursyminarchive, symoffset;

      EXPORT UpdateSymFile;

      VAR
	 first: BOOLEAN; (* first call of UpdateSymFile? *)
	 infp, outfp: FILE;
	 opened: BOOLEAN; (* opened for writing *)

      PROCEDURE Cleanup;
      BEGIN
	 IF ~first THEN
	    first := TRUE; (* avoid recursion *)
	    IF ~Fclose(infp) THEN END;
	    IF opened & ~Fclose(outfp) THEN
	       IOFault(oursymfile); (* exits again *)
	    END;
	    opened := FALSE;
	 END;
      END Cleanup;

      PROCEDURE UpdateSymFile(t: Type);

	 PROCEDURE Open;
	 BEGIN
	    Assert(oursymfile[0] # 0C);
	    IF ~Fopen(infp, oursymfile, read, (* buffered = *) TRUE) THEN
	       IOFault(oursymfile);
	    END;
	 END Open;

	 PROCEDURE Update;

	    VAR
	       trec: TypeRec;

	    PROCEDURE Seek(pos: OFF);
	    BEGIN
	       IF ~Fseek(infp, pos, 0) THEN
		  IOFault(oursymfile);
	       END;
	       IF opened & ~Fseek(outfp, pos, 0) THEN
		  IOFault(oursymfile);
	       END;
	    END Seek;

	    PROCEDURE Read(VAR buf: ARRAY OF BYTE);
	    BEGIN
	       Fread(infp, buf);
	       IF ~Done THEN IOFault(oursymfile) END;
	    END Read;

	    PROCEDURE Write(buf: ARRAY OF BYTE);
	    BEGIN
	       IF ~errorflag THEN
		  Fwrite(outfp, buf);
		  IF ~Done THEN IOFault(oursymfile) END;
	       END;
	    END Write;

	    PROCEDURE WriteStamp(time: TIME);
	       VAR
		  buf: ARRAY [0..11] OF CHAR;
		  index: CARDINAL;
	    BEGIN
	       ConvertInteger(time, 1, buf);
	       index := 0;
	       WHILE (index <= HIGH(buf)) & (buf[index] # 0C) DO
		  INC(index);
	       END;
	       WHILE index <= HIGH(buf) DO
		  buf[index] := " ";
		  INC(index);
	       END;
	       Write(buf);
	    END WriteStamp;

	    PROCEDURE UpdateFields(fields: FieldList);
	       (* update field offsets in symbolfile;
		  they may change if the basetype has been changed
	       *)
	       VAR
		  fieldrec: FieldListRec;
		  field: FieldList;
	    BEGIN
	       field := fields;
	       WHILE field # NIL DO
		  WITH field^ DO
		     IF seekpos # 0 THEN
			Seek(seekpos);
			Read(fieldrec);
			fieldrec.offset := offset;
			Write(fieldrec);
		     END;
		  END;
		  field := field^.link;
	       END;
	    END UpdateFields;

	    PROCEDURE UpdateParams(params: ParamList);
	       VAR
		  pp: ParamList;
		  prec: ParamListRec;
	    BEGIN
	       pp := params;
	       WHILE pp # NIL DO
		  WITH pp^ DO
		     IF seekpos # 0 THEN
			Seek(seekpos);
			Read(prec);
			prec.offset := offset;
			prec.varkind := varkind;
			Write(prec);
		     END;
		  END;
		  pp := pp^.link;
	       END;
	    END UpdateParams;

	 BEGIN
	    WITH t^ DO
	       Seek(seekpos);
	       Read(trec);
	       IF ~opened THEN
		  IF ~Fopen(outfp, oursymfile, append, TRUE) THEN
		     IOFault(oursymfile);
		  END;
		  (* update symbol file key *)
		  Assert((keypos > 0) & (mainmod # NIL));
		  opened := TRUE;
		  Seek(keypos);
		  WITH mainmod^.key DO
		     time := CurrentTime();
		     pid := Getpid();
		  END;
		  Write(mainmod^.key);
		  (* update time stamp in case of archives *)
		  IF oursyminarchive THEN
		     (* an archive header is 60 bytes long
			the timestamp is in decimal form from
			byte 16 to 27 (12 bytes)
		     *)
		     Assert(symoffset >= 60);
		     Seek(symoffset - 44);
		     WriteStamp(mainmod^.key.time);
		  END;
		  Seek(seekpos);
	       END;
	       IF trec.size # size THEN
		  trec.size := size;
		  trec.sizemodified := TRUE;
	       END;
	       trec.containsptr := containsptr;
	       Write(trec);
	       IF form = record THEN
		  UpdateFields(fields);
	       ELSIF form = proceduretype THEN
		  UpdateParams(param);
	       END;
	    END;
	 END Update;

      BEGIN
	 Assert(~defunit);
	 IF updatesym >= softUpdate THEN
	    WITH t^ DO
	       IF ~errorflag THEN
		  Assert(seekpos # 0);
		  IF first THEN
		     Open;
		  END;
		  Update;
	       END;
	    END;
	 ELSE
	    IF first THEN
	       Error("required type extension is prohibited; use `-u' option");
	    END;
	 END;
	 first := FALSE;
      END UpdateSymFile;

   BEGIN
      first := TRUE; opened := FALSE; EnterCleanup(Cleanup);
   END Updates;

BEGIN
   typelist := NIL;
   extmods := NIL;
   firstheader := TRUE;
   oursymfile[0] := 0C;
   keypos := 0;
END SymFile.
