/*
**	Copyright (c) 1984 Piers Lauder, University of Sydney
**
**	Warning: Distribution of this software without written
**		 permission is prohibited.
*/

/*
**	Call a remote host and start a network daemon on it.
**
**	Needs a "call" script as argument.
*/

#ifndef	lint
static char	sccsid[]	= "@(#)call.c	1.51 89/10/30";
#endif	lint

char	Usage[]	= "\nUsage: %s [-[&][I][O]] [-a<call-arg>] [-d<daemon-arg>] \\\n\
	-h<host> [-l<logfile>] [-s<call-interpreter>] [-t<timeout>] \\\n\
	<call-program>\n";


#define	FILE_CONTROL
#define	STAT_CALL
#define	TERMIOCTL
#define	STDIO

#include	"global.h"
#include	"debug.h"

#include	<setjmp.h>
#include	<signal.h>
#include	<errno.h>

#include	"caller.h"


/*
**	Parameters set from arguments
*/

int		Background;		/* True if call should run in background */
VarArgs		DmnArgs			/* Passed to exec to invoke daemon */
=
{
	2,
	NNDAEMON,			/* Mode-node daemon */
	"-BF"				/* Batch mode, no fork */
}
;
FILE *		LogFd;
char *		Name;
int		Ondelay;		/* Set to O_NDELAY for line */
unsigned int	TimeOut		= 30;
VarArgs		CallArgs		/* Passed to exec to invoke interpreter */
#ifdef	BINSIFT
=
{
	2,
	BINSIFT,			/* Default call interpreter */
	"-bcp"				/* No input/output bufferring, no pre-processing */
}
#endif	BINSIFT
;
int		Traceflag;

/*
**	Interface to interpreter
*/

int		Ipipes[2];
#define		Callin		Ipipes[1]
#define		Call0		Ipipes[0]
int		Opipes[2];
#define		Callout		Opipes[0]
#define		Call1		Opipes[1]

char		TimeoutMsg[]	= TIMEOUT;
char		Eofstr[]	= EOFSTR;
char		NewLine[]	= "\n";
char		LockedMsg[]	= LOCKED;
char		OpenFailMsg[]	= OPENFAIL;
char		OpenedMsg[]	= OPENED;

/*
**	Commands from script
*/

typedef enum
{
	t_after, t_close, t_daemon, t_error, t_fail, t_flush,
	t_mode, t_open, t_opendial, t_ondelay, t_raw, t_read,
	t_retry, t_shell, t_sleep, t_speed, t_stty, t_succeed,
	t_timeout, t_trace, t_write, t_local, t_remote, t_break,
	t_offdelay, t_tellme
}
		Cmdtyp;

struct Command
{
	char *	c_name;
	Cmdtyp	c_type;
}
		Cmds[] =
{
	{"after",	t_after},
	{"break",	t_break},
	{"close",	t_close},
	{"daemon",	t_daemon},
	{"fail",	t_fail},
	{"flush",	t_flush},
	{"local",	t_local},
	{"mode",	t_mode},
	{"offdelay",	t_offdelay},
	{"ondelay",	t_ondelay},
	{"open",	t_open},
	{"opendial",	t_opendial},
	{"raw",		t_raw},
	{"read",	t_read},
	{"remote",	t_remote},
	{"retry",	t_retry},
	{"shell",	t_shell},
	{"sleep",	t_sleep},
	{"speed",	t_speed},
	{"stty",	t_stty},
	{"succeed",	t_succeed},
	{"tellme",	t_tellme},
	{"timeout",	t_timeout},
	{"trace",	t_trace},
	{"write",	t_write},
};

#define		CMDZ		(sizeof(struct Command))
#define 	NCMDS		((sizeof Cmds)/CMDZ)
typedef struct Command *	Cmdp;

struct Speed
{
	char *	s_name;
	int	s_val;
}
		Speeds[] =
{
	 {"110",	B110}
	,{"1200",	B1200}
#	ifdef	B19200
	,{"19200",	B19200}
#	endif
	,{"2400",	B2400}
	,{"300",	B300}
#	ifdef	B38400
	,{"38400",	B38400}
#	endif
	,{"4800",	B4800}
	,{"600",	B600}
	,{"9600",	B9600}
#	ifdef	EXTA
	,{"EXTA",	EXTA}
#	endif
#	ifdef	EXTB
	,{"EXTB",	EXTB}
#	endif
};

#define		SPDZ		(sizeof(struct Speed))
#define 	NSPDS		((sizeof Speeds)/SPDZ)
typedef struct Speed *		Spdp;

/*
**	List of commands to be executed after daemon has terminated.
*/

typedef struct after_list	AL;
struct after_list
{
	AL *	al_next;
	char *	al_command;
};

AL *		AfterList;
AL **		EAfterList	= &AfterList;

/*
**	Miscellaneous
*/

jmp_buf		Alrm_jmp;
char		Buf[BUFSIZ];
char *		CallScript;
int		Call_pid;
int		Daemon_pid;
char *		DevNull		= "/dev/null";
char		Errbuf[BUFSIZ];
int		Fd		= SYSERROR;
char *		Fn;
bool		HadArg;
bool		IgnoreOldPid	= false;
char *		KeepErrFile;
int		Pid;
int		Read_pid;
int		Retries;
unsigned int	Retry;
char *		Target;
int		TellMe		= 0;
Time_t		Time;
#if	CARR_ON_BUG == 1
char *		DeviceName;
#endif	CARR_ON_BUG == 1

/*
**	Routines
*/

bool
		command();

int
		alrmcatch(),
		compare(),
		catch();

void
		Daemon(),
		Shell(),
		Stty(),
		cleanup(),
		cleartty(),
		doafter(),
		lockalarm(),
		reader(),
		startcall(),
		usage();



int
main(argc, argv)
	int		argc;
	register char *	argv[];
{
	register int	c;
	register char *	cp;
	char *		path;

	if ( (Name = strrchr(*argv, '/')) != NULLSTR )
		Name++;
	else
		Name = *argv;

	Pid = getpid();
	Time = time((long *)0);

	while ( --argc > 0 )
	{
		if ( **++argv == '-' )
		{
			while ( c = *++*argv )
			{
				switch ( c )
				{
				case '&':
					Background++;
					continue;

				case 'I':
					IgnoreOldPid = true;
					continue;

				case 'O':
#					ifdef	O_NDELAY
					Ondelay = O_NDELAY;
					continue;
#					else	O_NDELAY
					Error("O_NDELAY not defined");
					exit(1);
#					endif	O_NDELAY

				case 'T':
					if ( (Traceflag = atoi(++*argv)) <= 0 )
						Traceflag = 1;
					break;

				case 'a':
					if ( NARGS(&CallArgs) == 0 )
					{
						usage("need '-s' before '-a'");
						exit(1);
					}
					NEXTARG(&CallArgs) = ++*argv;
					HadArg = true;
					goto break2;

				case 'd':
					NEXTARG(&DmnArgs) = ++*argv;
					goto break2;

				case 'h':
					Target = ++*argv;
					goto break2;

				case 'l':
					if ( (LogFd = freopen(++*argv, "a", stderr)) == NULL )
					{
						Syserror("can't open \"%s\"", *argv);
						exit(1);
					}
					goto break2;

				case 's':
					if ( HadArg )
					{
						usage("'-s' must come before '-a'");
						exit(1);
					}
					FIRSTARG(&CallArgs) = ++*argv;
					goto break2;

				case 't':
					if ( (TimeOut = atoi(++*argv)) == 0 )
					{
						usage("bad timeout %d", TimeOut);
						exit(1);
					}
					break;

				default:
					usage("unrecognised flag '%c'", c);
					exit(1);
				}

				while ( (c = **argv) <= '9' && c >= '0' )
					++*argv;
				--*argv;
			}

break2:			;
		}
		else
		{
			struct stat	statb;

			if ( CallScript != NULLSTR )
			{
				usage("only one call program needed");
				exit(1);
			}

			CallScript = *argv;

			if ( stat(CallScript, &statb) == SYSERROR )
			{
				Syserror(CallScript);
				exit(1);
			}

			if ( statb.st_mode & 0111 )
			{
				/*
				**	Executable program
				*/

				if ( HadArg )
				{
					usage("can't use '-a' with executable call program");
					exit(1);
				}

				FIRSTARG(&CallArgs) = CallScript;
			}
			else
			{
				if ( NARGS(&CallArgs) == 0 )
				{
					usage("need name of interpreter for \"%s\"", CallScript);
					exit(1);
				}

				NEXTARG(&CallArgs) = CallScript;
			}
		}
	}

	if ( CallScript == NULLSTR )
	{
		usage("need call program name");
		exit(1);
	}

	if ( Background )
	{
		switch ( fork() )
		{
		case SYSERROR:
			Syserror("Can't fork");
			exit(1);

		case 0:
			break;

		default:
			exit(0);
		}

#		if	BSD4 > 1 || (BSD4 == 1 && BSD4V >= 'c')
		(void)cleartty();
#		else	BSD4 > 1 || (BSD4 == 1 && BSD4V >= 'c')
#		if	PGRP == 1
		(void)setpgrp();
#		endif
#		endif	BSD4 > 1 || (BSD4 == 1 && BSD4V >= 'c')

		if ( LogFd == NULL )
			(void)freopen(DevNull, "w", stderr);

		(void)signal(SIGHUP, SIG_IGN);
		(void)signal(SIGINT, SIG_IGN);
		(void)signal(SIGQUIT, SIG_IGN);
	}
	else
		if ( signal(SIGINT, SIG_IGN) != SIG_IGN )
			(void)signal(SIGINT, catch);

	(void)signal(SIGTERM, catch);

#	if	KILL_0 != 1
	(void)signal(SIG0, SIG_IGN);
#	endif	KILL_0 != 1

	if ( (cp = strrchr(CallScript, '/')) != NULLSTR )
	{
		*cp = '\0';
		path = newstr(CallScript);
		*cp = '/';
	}
	else
		path = newstr(".");

	if ( DaemonActive(path, true) )
		exit(0);

	(void)SetDaemonActive(path, getpid());

#	if	SYSTEM > 0
	if ( ulimit(2, ULIMIT) == SYSERROR )
		Syserror("ulimit(%ld)", ULIMIT);
#	endif

	free(path);

#	if	FCNTL == 1 && O_APPEND != 0
#	if	O_APPEND_BUG
	if ( !isatty(fileno(stderr)) )
#	endif
	(void)fcntl
	(
		fileno(stderr),
		F_SETFL,
		fcntl(fileno(stderr), F_GETFL, 0) | O_APPEND
	);
#	endif

	setbuf(stderr, Errbuf);

	exit(call());
}



int
call()
{
	register char *	cp;
	register int	n;

	startcall();

	for ( cp = Buf ; (n = read(Callout, cp, 1)) == 1 ; )
	{
		if ( *cp != '\0' )
		{
			if ( ++cp >= &Buf[sizeof Buf] )
			{
				Error("callout too long");
				return 1;
			}
			continue;
		}

		if ( !command(Buf) )
			return 0;

		cp = Buf;
	}

	if ( n == SYSERROR )
		Syserror("read from callout");
	else
		Error("unexpected eof from call prog");

	return 1;
}



bool
command(comm)
	char *		comm;
{
	register char *	cp	= comm;
	register int	n;
	Cmdp		cmdp;
	Cmdtyp		cmdt;
	struct Command	cmd;
#	if	OPENDIAL
	bool		open_dial = false;
#	endif	OPENDIAL

	Trace1(2, cp);

	cmd.c_name = cp;

	if ( (cp = strchr(cp, ' ')) != NULLSTR )
	{
		*cp++ = '\0';
		while ( *cp == ' ' )	cp++;
		if ( *cp == '\0' )
			cp = NULLSTR;
	}

	if
	(
		(cmdp = (Cmdp)bsearch((char *)&cmd, (char *)Cmds, NCMDS, CMDZ, compare))
		!=
		(Cmdp)0
	)
		cmdt = cmdp->c_type;
	else
		cmdt = t_error;

	if ( comm != Buf )
		switch ( cmdt )
		{
		case t_after:
		case t_daemon:
		case t_mode:
		case t_succeed:
		case t_tellme:
			Error("Illegal after command");
			return false;
		}

	switch ( cmdt )
	{
	case t_succeed:
			if ( Fd == SYSERROR )
			{
				Error("succeed with no device");
				return false;
			}
			if ( cp == NULLSTR )
			{
				Error("succeed with no host name");
				return false;
			}
			cleanup();
#			if	UUCPLOCKS != 1
			if ( AfterList != (AL *)0 )
#			endif	UUCPLOCKS != 1
			for ( ;; )
			{
				register int	pid;
				int		stat;

				switch ( Daemon_pid = fork() )
				{
				default:
					signal(SIGINT, SIG_IGN);
					signal(SIGHUP, SIG_IGN);
					signal(SIGQUIT, SIG_IGN);
#					if	UUCPLOCKS == 1
					lockalarm();
#					endif	UUCPLOCKS == 1
					while
					(
						(pid = wait(&stat)) != Daemon_pid
						&&
						(pid != SYSERROR || errno == EINTR)
					)
						;
					Daemon_pid = 0;
#					if	UUCPLOCKS == 1
					(void)alarm(0);
#					endif	UUCPLOCKS == 1
					if ( AfterList != (AL *)0 )
						doafter();
#					if	UUCPLOCKS == 1
					rmlock(NULL);
#					endif	UUCPLOCKS == 1
					return false;

				case SYSERROR:	/* help */
					Syserror("Can't fork");
					continue;

				case 0:	;
				}
				break;
			}
			Daemon(cp);
			return false;

	case t_fail:
			Error("fail \"%s\"", cp==NULLSTR?"":cp);
			return false;

	case t_daemon:
			if ( cp == NULLSTR )
			{
				Error("daemon name?");
				return false;
			}
			FIRSTARG(&DmnArgs) = newstr(cp);
			break;

	case t_mode:
			if ( cp == NULLSTR )
			{
				Error("daemon mode?");
				return false;
			}
			(void)SplitArg(&DmnArgs, cp);
			if ( NARGS(&DmnArgs) > MAXVARARGS )
			{
				Error("too many mode params");
				return false;
			}
			break;

	case t_close:
			if ( Fd == SYSERROR )
			{
				Error("No file for close");
				return false;
			}
			if ( Read_pid )
			{
				int	status;

				(void)kill(Read_pid, SIGKILL);

				while
				(
					(n = wait(&status)) != Read_pid
					&&
					n != SYSERROR
				);

				Read_pid = 0;
			}
			(void)close(Fd);
			free(Fn);
#			if	CARR_ON_BUG == 1
			if (DeviceName != NULLSTR)
			{
				free(DeviceName);
				DeviceName = NULLSTR;
			}
#			endif	CARR_ON_BUG == 1
			Fd = SYSERROR;
#			if	UUCPLOCKS == 1
			rmlock(NULL);
#			endif	UUCPLOCKS == 1
			break;

	case t_offdelay:
#			ifdef	O_NDELAY
			Ondelay = 0;
			break;
#			else	O_NDELAY
			Error("O_NDELAY not defined");
			return false;
#			endif	O_NDELAY

	case t_ondelay:
#			ifdef	O_NDELAY
			Ondelay = O_NDELAY;
			break;
#			else	O_NDELAY
			Error("O_NDELAY not defined");
			return false;
#			endif	O_NDELAY

	case t_opendial:
#			if	OPENDIAL
			open_dial = true;
#			endif	OPENDIAL

	case t_open:
			if ( cp == NULLSTR )
			{
				Error("file name?");
				return false;
			}
			if ( Fd != SYSERROR )
			{
				Error("File \"%s\" already open", Fn);
				return false;
			}
			(void)signal(SIGALRM, alrmcatch);
			{
#				if	UUCPLOCKS == 1
				register char *	l;
#				endif	UUCPLOCKS == 1
				static char *	argp;	/* Preserve from "setjmp()" */
				static char *	argn;
				static int	retries;

				argp = cp;

				while ( Fd == SYSERROR )
				{
					if ( (argn = strchr(argp, ' ')) != NULLSTR )
						*argn++ = '\0';
					retries = Retries;

#					if	UUCPLOCKS == 1
					if ( (l = strrchr(argp, '/')) == NULLSTR )
						l = argp;
					else
						l++;

					if ( mlock(l) != 0 )
					{
						if ( argn == NULLSTR )
						{
							if ( TellMe )
							{
								(void) write(Callin, LockedMsg, sizeof LockedMsg - 1);
								(void) write(Callin, NewLine, sizeof NewLine - 1);
								break;
							}
							Error("\"%s\" locked (UUCP).", argp);
							return false;
						}
						argp = argn;
						continue;
					}
#					endif	UUCPLOCKS == 1

					while
					(
						setjmp(Alrm_jmp) && (retries = -1)
						||
						alarm(TimeOut) == SYSERROR
						||
						(
#							if	OPENDIAL
							open_dial ?
							(Fd = opendial(argp)) :
#							endif	OPENDIAL
							(Fd = open(argp, Ondelay|O_EXCL|O_RDWR))
						) == SYSERROR
					)
					{
						(void)alarm((unsigned)0);
						Trace2(1, "Can't open \"%s\"", argp);
						if ( Retry && --retries >= 0 )
						{
							(void)sleep(Retry);
							continue;
						}
#						if	UUCPLOCKS == 1
						rmlock(NULL);
#						endif
						if ( (argp = argn) == NULLSTR )
						{
							if ( TellMe )
							{
								(void) write(Callin, OpenFailMsg, sizeof OpenFailMsg - 1);
								(void) write(Callin, NewLine, sizeof NewLine - 1);
								break;
							}
							errno = EIO;
							Syserror("Can't open remote");
							return false;
						}
						break;
					}
				}
				if ( Fd != SYSERROR )
					Fn = newstr(argp);
			}
			(void)alarm((unsigned)0);
			Retry = 0;
			if ( Fd != SYSERROR )
			{
				(void)SetRaw(Fd, 0, 1, 0, false);
				if ( TellMe )
				{
					(void)write(Callin, OpenedMsg, sizeof OpenedMsg - 1);
					(void)write(Callin, NewLine, sizeof NewLine - 1);
				}
			}
			break;

	case t_raw:
			if ( cp != NULLSTR )
			{
				Error("unexpected arg \"%s\" for \"raw\"", cp);
				return false;
			}

			if ( Fd == SYSERROR )
			{
				Error("can't set raw mode without device");
				return false;
			}

			(void)SetRaw(Fd, 0, 1, 0, false);
			break;

	case t_retry:
			if ( cp == NULLSTR || (Retry = atoi(cp)) == 0 )
			{
				Error("bad retry value \"%s\"", cp);
				return false;
			}
			if
			(
				(cp = strchr(cp, ' ')) == NULLSTR
				||
				(Retries = atoi(cp+1)) == 0
			)
				Retries = 1;
			break;

	case t_shell:
			Shell(cp);
			break;

	case t_sleep:
			if
			(
				cp == NULLSTR
				||
				(n = atoi(cp)) == 0
			)
			{
				Error("bad sleep value \"%s\"", cp);
				return false;
			}

			if ( (Read_pid && n >= (TimeOut-2)) )
			{
				Error("sleep period %d conflicts with timeout set at %d", n, TimeOut);
				return false;
			}
			(void)sleep((unsigned)n);
			break;

	case t_stty:
			if ( Fd == SYSERROR )
			{
				Error("No file for stty");
				return false;
			}
			Stty(cp);
			break;

	case t_speed:
		{
			Spdp		spdp;
			struct Speed	speed;

			speed.s_name = cp;

			if
			(
				(spdp = (Spdp)bsearch((char *)&speed, (char *)Speeds, NSPDS, SPDZ, compare))
				==
				(Spdp)0
			)
			{
				Error("unrecognised speed \"%s\"", cp);
				return false;
			}

			if ( Fd == SYSERROR )
			{
				Error("can't set speed without device");
				return false;
			}

			(void)SetRaw(Fd, spdp->s_val, 1, 0, false);
			break;
		}

	case t_tellme:
			TellMe = 1;
			break;

	case t_timeout:
			if ( Read_pid != 0 )
			{
				Error("can't set timeout after reader started");
				return false;
			}

			if ( (TimeOut = atoi(cp)) == 0 )
			{
				Error("bad timeout value \"%s\"", cp);
				return false;
			}
			break;

	case t_write:
			if
			(
				cp == NULLSTR
				||
				(n = strlen(cp)) == 0
				||
				Fd == SYSERROR
				||
				write(Fd, cp, n) != n
			)
			{
				Syserror("cannot write to device");
				return false;
			}
			break;

	case t_read:
			if ( Fd == SYSERROR )
			{
				Error("reader with no device");
				return false;
			}
			if ( Read_pid != 0 )
			{
				Error("reader already active");
				return false;
			}
			reader();
			break;

	case t_trace:
			Mesg("trace", cp==NULLSTR?"":ExpandString(cp, strlen(cp)));
			putc('\n', stderr);
			(void)fflush(stderr);
			break;

	case t_after:
		{
			register AL *	alp;

			alp = Talloc(AL);
			alp->al_next = (AL *)0;
			alp->al_command = newstr(cp);
			*EAfterList = alp;
			EAfterList = &alp->al_next;
			break;
		}

#if	SYSTEM >= 3
	case t_flush:
			/* flush both input and output */
			if ( Fd == SYSERROR )
			{
				Error("no device for \"flush\"");
				return false;
			}
			if ( ioctl(Fd, TCXONC, 1) == SYSERROR )
			{
				Syserror("cannot ioctl onto device");
				return false;
			}
			if ( ioctl(Fd, TCFLSH, 2) == SYSERROR )
			{
				Syserror("cannot ioctl onto device");
				return false;
			}
			break;

	case t_local:
		{
			static struct termio	ttyi;

			if ( Fd == SYSERROR )
			{
				Error("no device for \"local\"");
				return false;
			}
			if ( ioctl(Fd, TCGETA, &ttyi) == SYSERROR )
			{
				Syserror("cannot ioctl from device");
				return false;
			}
			ttyi.c_cflag |= CLOCAL;
			if ( ioctl(Fd, TCSETA, &ttyi) == SYSERROR )
			{
				Syserror("cannot ioctl onto device");
				return false;
			}
#			if	CARR_ON_BUG == 1
			/*
			 * This fixes a bug on some SYSTEM xx.
			 * Driver sets CARR_ON on open or on modem DCD
			 * becoming high but not on above ioctl!
			 */
			(void)close(open(DeviceName, O_RDWR, 0));
#			endif	CARR_ON_BUG == 1
			break;
		}

	case t_remote:
		{
			static struct termio	ttyi;

			if ( Fd == SYSERROR )
			{
				Error("no device for \"remote\"");
				return false;
			}
			if ( ioctl(Fd, TCGETA, &ttyi) == SYSERROR )
			{
				Syserror("cannot ioctl from device");
				return false;
			}
			ttyi.c_cflag &= ~CLOCAL;
			if ( ioctl(Fd, TCSETA, &ttyi) == SYSERROR )
			{
				Syserror("cannot ioctl onto device");
				return false;
			}
			break;
		}

	case t_break:
			if ( Fd == SYSERROR )
			{
				Error("no device for \"break\"");
				return false;
			}
			if ( ioctl(Fd, TCSBRK, 0) == SYSERROR )
			{
				Syserror("cannot ioctl onto device");
				return false;
			}
			break;
#endif	SYSTEM >= 3

#if	BSD4 > 1 || (BSD4 == 1 && BSD4V >= 'c')

	case t_flush:
		{
			int	rw = 2;

			if ( Fd == SYSERROR )
			{
				Error("no device for \"flush\"");
				return false;
			}
			if ( ioctl(Fd, TIOCFLUSH, &rw) == SYSERROR )
			{
				Syserror("cannot ioctl onto device");
				return false;
			}
			break;
		}

	case t_local:
		{
			int	lflag = LNOHANG;

			if ( Fd == SYSERROR )
			{
				Error("no device for \"local\"");
				return false;
			}
			if ( ioctl(Fd, TIOCLBIS, &lflag) == SYSERROR )
			{
				Syserror("cannot ioctl onto device");
				return false;
			}
			break;
		}

	case t_remote:
		{
			int	lflag = LNOHANG;

			if ( Fd == SYSERROR )
			{
				Error("no device for \"remote\"");
				return false;
			}
			if ( ioctl(Fd, TIOCLBIC, &lflag) == SYSERROR )
			{
				Syserror("cannot ioctl onto device");
				return false;
			}
			break;
		}

	case t_break:
			if ( Fd == SYSERROR )
			{
				Error("no device for \"break\"");
				return false;
			}

			sleep(2);	/* let output drain */

			if ( ioctl(Fd, TIOCSBRK, (char *)0) == SYSERROR )
			{
				Syserror("cannot ioctl onto device");
				return false;
			}

			(void)sleep(MINSLEEP);	/* Big break */

			if ( ioctl(Fd, TIOCCBRK, (char *)0) == SYSERROR )
			{
				Syserror("cannot ioctl onto device");
				return false;
			}

			break;
#endif	BSD4 > 1 || (BSD4 == 1 && BSD4V >= 'c')

	default:
			Error("unrecognised command from call \"%s\"", Buf);
			return false;
	}

	return true;
}



void
Shell(cp)
	char *	cp;
{
	/**** This is one hell of a security hole -- disable for now

	VarArgs	va;
	char *	errs;

	if ( cp == NULLSTR || *cp == '\0' )
	{
		Error("No args for shell");
		return;
	}

	FIRSTARG(&va) = "/bin/sh";
	NEXTARG(&va) = "-c";
	NEXTARG(&va) = cp;

	if ( (errs = Execute(&va)) != NULLSTR )
	{
		Error(errs);
		free(errs);
	}
	****/
}



void
Stty(cp)
	char *	cp;
{
	VarArgs	va;
	char *	errs;
	int	i;
	int	pid;
	int	status;

	if ( cp == NULLSTR || *cp == '\0' )
	{
		Error("No args for stty");
		return;
	}

	FIRSTARG(&va) = STTY;

	if ( SplitArg(&va, cp) > MAXVARARGS )
	{
		Error("Too many args for stty");
		return;
	}

	NEXTARG(&va) = NULLSTR;

	switch ( pid = fork() )
	{
	case SYSERROR:	Syserror("Can't fork");	return;

	case 0:
		for ( i = 0 ; i <= 2 ; i++ )
			if ( i != Fd )
			{
				(void)close(i);
				(void)dup(Fd);
			}

		for ( i = 3 ; close(i) != SYSERROR ; i++ );

		for ( ;; )
		{
			(void)execve(ARG(&va, 0), &ARG(&va, 0), StripEnv());
			Syserror("Can't execve \"%s\"", ARG(&va, 0));
		}
	}

	while ( (i = wait(&status)) != pid )
		if ( i == SYSERROR )
		{
			Syserror("Lost child");
			return;
		}

	if ( status )
	{
		errs = GetErrFile(&va, status, open(DevNull, O_RDWR));
		Error(errs);
		free(errs);
	}
}




void
doafter()
{
	register AL *	alp;

	if ( (alp = AfterList) == (AL *)0 )
		return;

	AfterList = (AL *)0;
	EAfterList = &AfterList;

	do
	{
		if ( !command(alp->al_command) )
			return;
	}
	while
		( (alp = alp->al_next) != (AL *)0 );
}



int
pipecatch(sig)
	int	sig;
{
	(void)signal(sig, SIG_IGN);
	Error("SIGPIPE on callin");
}

void
startcall()
{
	if
	(
		pipe(Ipipes) == SYSERROR
		||
		pipe(Opipes) == SYSERROR
	)
		Syserror("cannot make pipes for call");

	switch ( Call_pid = fork() )
	{
	case SYSERROR:
			Syserror("cannot fork");
			break;

	case 0:
			(void)close(Callin);
			(void)close(Callout);
			(void)close(0);
			(void)dup(Call0);
			(void)close(Call0);
			(void)close(1);
			(void)dup(Call1);
			(void)close(Call1);

			if ( Traceflag )
			{
				NEXTARG(&CallArgs) = newstr("-T0");
				LASTARG(&CallArgs)[2] += Traceflag;
			}

			NEXTARG(&CallArgs) = Target;
			NEXTARG(&CallArgs) = NULLSTR;

			(void)execve(ARG(&CallArgs, 0), &ARG(&CallArgs, 0), StripEnv());

			Syserror("cannot exec %s", ARG(&CallArgs, 0));
	}

	(void)signal(SIGPIPE, pipecatch);
	(void)close(Call0);
	(void)close(Call1);
}

int
alrmcatch(sig)
	int	sig;
{
	(void)signal(sig, alrmcatch);
	(void)longjmp(Alrm_jmp, 1);
}

void
reader()
{
	register int	n;
	int		delay;

	switch ( Read_pid = fork() )
	{
	case SYSERROR:
		Syserror("cannot fork");
		return;

	case 0:
		DODEBUG(Name = "reader");

		Call_pid = 0;
		(void)close(Callout);
		delay = 0;

		for ( ;; )
		{
			if ( setjmp(Alrm_jmp) )
			{
				if ( !Ondelay || delay > TimeOut )
				{
					while
					(
						write(Callin, NewLine, sizeof NewLine -1)
						!=
						(sizeof NewLine -1)
						||
						write(Callin, TimeoutMsg, sizeof TimeoutMsg -1)
						!=
						(sizeof TimeoutMsg -1)
						||
						write(Callin, NewLine, sizeof NewLine -1)
						!=
						(sizeof NewLine -1)
					)
						Syserror("write to callin");

					delay = 0;
				}
			}

			(void)signal(SIGALRM, alrmcatch);
			(void)alarm(TimeOut);

			while ( (n = read(Fd, Buf, sizeof Buf -1)) > 0 )
			{
				register char *	cp;
				register int	i;
				register int	c;

				/*
				**	The string may have nulls in it,
				**	and other unprintables.  We clean
				**	all this up here.
				*/

				if ( Traceflag >= 2 )
				{
					(void)fflush(stdout);
					(void)fflush(stderr);
					(void)fseek(TraceFd, 0L, 2);
					for ( cp = Buf, i = n ; --i >= 0 ; )
					{
						c = *cp++ & 0x7f;
						if ( (c >= 040 && c <= 0176) || (c == '\n') )
							(void)fprintf(TraceFd, "%c", c);

						else
							(void)fprintf(TraceFd, "\\%03o", c);
					}
					(void)fflush(TraceFd);
				}

				for ( cp = Buf, i = n ; --i >= 0 ; cp++ )
				{
					c = *cp &= 0177;

					if ( c == '\r' )
						*cp = '\n';
				}

				while ( write(Callin, Buf, n) != n )
					Syserror("write to callin");

				delay = 0;

				(void)alarm(TimeOut);
			}

			if ( n == 0 )
			{
				if ( Ondelay )
				{
					delay++;
					(void)alarm(1);
					(void)pause();
					continue;
				}

				(void)write(Callin, NewLine, sizeof NewLine -1);
				(void)write(Callin, Eofstr, sizeof Eofstr -1);
				(void)write(Callin, NewLine, sizeof NewLine -1);

				Error("eof from device");
				exit(1);
			}

			(void)write(Callin, NewLine, sizeof NewLine -1);
			(void)write(Callin, Eofstr, sizeof Eofstr -1);
			(void)write(Callin, NewLine, sizeof NewLine -1);

			Syserror("cannot read device");
		}
	}
}



void
Daemon(args)
	register char *	args;
{
	(void)signal(SIGPIPE, SIG_IGN);

	if ( *args != '\0' )
		(void)SplitArg(&DmnArgs, args);

	if ( NARGS(&DmnArgs) > MAXVARARGS )
		Error("too many args for \"%s\"", ARG(&DmnArgs, 0));

	if ( NARGS(&DmnArgs) < 3 )
		Error
		(
			"too few args for daemon: \"%s %s\", need at least path name of daemon, flag \"-B\", and name of link",
			ARG(&DmnArgs, 0),
			ARG(&DmnArgs, 1)
		);

#	ifdef	O_NDELAY
	if ( Ondelay == 0 )
		(void)fcntl(Fd, F_SETFL, fcntl(Fd, F_GETFL, 0) & ~O_NDELAY);
#	endif

	(void)close(Callin);
	(void)close(Callout);
	(void)close(1);
	(void)dup(Fd);
	(void)close(Fd);

	Trace5
	(
		1,
		"%s %s %s %s",
		ARG(&DmnArgs, 0),
		ARG(&DmnArgs, 1),
		ARG(&DmnArgs, 2),
		ARG(&DmnArgs, 3) == NULLSTR ? "" : ARG(&DmnArgs, 3)
	);

	(void)fflush(stderr);

	(void)execve(ARG(&DmnArgs, 0), &ARG(&DmnArgs, 0), StripEnv());
	Syserror("cannot exec \"%s\"", ARG(&DmnArgs, 0));
}

void
cleanup()
{
	register int	pid;
	int		status;

	if ( Read_pid )
		(void)kill(Read_pid, SIGKILL);

	if ( Call_pid )
		(void)kill(Call_pid, SIGKILL);

	if ( Daemon_pid )
		(void)kill(Daemon_pid, SIGTERM);

	while
	(
		(Read_pid || Call_pid || Daemon_pid)
		&&
		(pid = wait(&status)) != SYSERROR
	)
	{
		if ( pid == Read_pid )
			Read_pid = 0;
		if ( pid == Call_pid )
			Call_pid = 0;
		if ( pid == Daemon_pid )
			Daemon_pid = 0;
	}
}

int
catch(sig)
	int	sig;
{
	(void)signal(sig, SIG_IGN);
	finish(0);
}

int
compare(p1, p2)
	char *	p1;
	char *	p2;
{
	return strcmp(((Cmdp)p1)->c_name, ((Cmdp)p2)->c_name);
}

finish(reason)
	int	reason;
{
	cleanup();
#	if	UUCPLOCKS == 1
	rmlock(NULL);
#	endif	UUCPLOCKS == 1
	exit(reason);
}

#if	BSD4 > 1 || (BSD4 == 1 && BSD4V >= 'c')

void
cleartty()
{
	register fd;

	(void)setpgrp(0, 0);

	if ( (fd = open("/dev/tty", 0)) < 0 )
		return;

	(void)ioctl(fd, TIOCNOTTY, (char *)0);
	(void)close(fd);
}
#endif	BSD4 > 1 || (BSD4 == 1 && BSD4V >= 'c')

#if	UUCPLOCKS == 1
void
lockalarm()
{
	ultouch();
	(void)signal(SIGALRM, lockalarm);
	(void)alarm(1800);
}
#endif	UUCPLOCKS == 1

#if	VARARGS
void
usage(va_alist)
	va_dcl
{
	va_list		ap;
	register char 	*s;

	va_start(ap);
	s = va_arg(ap, char *);
	VMesg("argument error", s, ap);
	va_end(ap);
	(void)fprintf(stderr, Usage, Name);
}
#else
/*VARARGS1*/
void
usage(s, a1, a2)
	char *	s;
	char *	a1;
	char *	a2;
{
	Mesg("argument error", s, a1, a2);
	(void)fprintf(stderr, Usage, Name);
}
#endif

#if	OPENDIAL
#if	SYSTEM >= 3

int
opendial(name)
	char *		name;
{
	register int	fd;
	static struct termio	ttyi;

	if ( (fd = open(name, O_NDELAY|O_EXCL|O_RDWR, 0)) < 0 )
		return fd;

	if ( ioctl(fd, TCGETA, &ttyi) < 0 )
	{
		close(fd);
		return -1;
	}

	ttyi.c_cflag = B1200 | CS8 | CREAD | HUPCL | CLOCAL;
	ttyi.c_iflag = IGNBRK;
	ttyi.c_oflag = 0;
	ttyi.c_lflag = NOFLSH;
	ttyi.c_cc[VMIN] = 1;
	ttyi.c_cc[VTIME] = 1;

	(void)ioctl(fd, TCSETA, &ttyi);

#	if	CARR_ON_BUG == 1
	/*
	 * This fixes a bug on some SYSTEM xx. Driver sets CARR_ON on open
	 * or on modem DCD becoming high but not on above ioctl!
	 */
	(void)close(open(name, O_RDWR, 0));
	DeviceName = newstr(name);
#	endif	CARR_ON_BUG == 1

	if ( Ondelay == 0 )
		(void)fcntl(fd, F_SETFL, fcntl(fd, F_GETFL, 0) & ~O_NDELAY);

	return fd;
}
#endif	SYSTEM >= 3

#if	BSD4 > 1 || (BSD4 == 1 && BSD4V >= 'c')
int
opendial(name)
	char *		name;
{
	register int	fd;
	struct sgttyb	ttyi;
	int		lflag;

	if ( (fd = open(name, O_NDELAY|O_EXCL|O_RDWR, 0)) < 0 )
		return fd;

	(void)ioctl(fd, TIOCEXCL, (char *)0);
	if ( ioctl(fd, TIOCGETP, &ttyi) < 0 )
	{
		close(fd);
		return -1;
	}

	ttyi.sg_ispeed = ttyi.sg_ospeed = B1200;
	ttyi.sg_flags = RAW;

	(void)ioctl(fd, TIOCSETP, &ttyi);

	lflag = OTTYDISC;
	(void)ioctl(fd, TIOCSETD, &lflag);
	(void)ioctl(fd, TIOCHPCL, (char *)0);
	lflag = LNOHANG;
	(void)ioctl(fd, TIOCLSET, &lflag);

	if ( Ondelay == 0 )
		(void)fcntl(fd, F_SETFL, fcntl(fd, F_GETFL, 0) & ~O_NDELAY);

	return fd;
}
#endif	BSD4 > 1 || (BSD4 == 1 && BSD4V >= 'c')
#endif	OPENDIAL
