/*
**	Copyright (c) 1984 Piers Lauder, University of Sydney
**
**	Warning: Distribution of this software without written
**		 permission is prohibited.
**
**	SCCSID @(#)FindCommand.c	1.18 89/02/20
*/

/*
**	Find lexically smallest file name for channel.
**	In the interests of speed, a sorted list of known command files
**	is maintained for each channel, and the directory is only searched
**	when a list is empty, and the modify time of the directory has changed.
*/

#define	STAT_CALL

#include	"global.h"

#include	"command.h"
#include	"debug.h"
#include	"spool.h"

#include	"Stream.h"
#include	"bad.h"
#include	"daemon.h"
#include	"driver.h"
#include	"AQ.h"

#include	<ndir.h>


/*
**	Command file queues.
*/

typedef struct MQel	MQel;
typedef MQel *		MQl_p;

typedef enum
{
	m_wait, m_active, m_sent
}
			Use_t;

struct MQel
{
	MQl_p	mql_next;
	Use_t	mql_use;
	char	mql_name[STREAMIDZ+1];
};

typedef struct
{
	MQl_p	cl_first;
	MQl_p *	cl_vec;
	short	cl_count;
	short	cl_index;
}
		CLhead;

typedef CLhead *CLh_p;

CLhead		ComLists[NSTREAMS];	/* The sorted lists for each channel */
MQl_p		MQfreelist;
MQl_p		MQactive[NSTREAMS];

DIR *		ComDirFd;
Time_t		ComDirMtime;
Time_t		LastLook;

int		clcomp();


bool
FindCommand(chan)
	int			chan;
{
	register CLh_p		clp;
	register MQl_p		mqlp;
	register char *		streamid = outStreams[chan].str_id;
	register struct direct *dp;
	register int		i;
	struct stat		statb;

	Trace3(1, "FindCommand for channel %d state %d", chan, outStreams[chan].str_state);

	if ( ComDirFd == NULL )
	{
#		ifdef	MULTIPATH
		while ( (ComDirFd = opendir(MultiPath!=NULLSTR?"..":".")) == NULL )
#		else	MULTIPATH
		while ( (ComDirFd = opendir(".")) == NULL )
#		endif	MULTIPATH
		{
			Syserror("commands directory");
			if ( BatchMode )
				finish(SYSERROR);
		}
	}

	/*
	**	Take last entry out of list, and unlink it
	*/

	if ( streamid[0] != '\0' )
	{
		Trace2(2, "unlink \"%s\"", streamid);

		(void)unlink(streamid);

		if ( (mqlp = MQactive[chan]) != (MQl_p)0 )
		{
#			if	DEBUG
			if ( mqlp->mql_use != m_active )
				Fatal1("!MQactive");
			if ( strcmp(mqlp->mql_name, streamid) != STREQUAL )
				Fatal1("MQactive unequal names");
#			endif	DEBUG

			mqlp->mql_use = m_sent;
		}

		streamid[0] = '\0';
	}

	DODEBUG(if((chan-Fstream)<0)Fatal("FindCommand chan < Fstream"));

loop:
	for ( clp = &ComLists[chan-Fstream], i = chan ; i-- >= 0 ; )
	{
		while ( clp->cl_index < clp->cl_count )
		{
			mqlp = clp->cl_vec[clp->cl_index++];

			if ( mqlp->mql_use != m_wait )
				continue;

			mqlp->mql_use = m_active;
			MQactive[chan] = mqlp;

			(void)strcpy(streamid, mqlp->mql_name);

#			ifdef	MULTIPATH
			if ( MultiPath != NULLSTR )
			{
				register char *	from = concat("../", streamid, NULLSTR);

				if
				(
					link(from, streamid) == SYSERROR
					||
					unlink(from) == SYSERROR
				)
				{
					if ( access(from, 0) != SYSERROR )
						Syserror("Can't link \"%s\" to \"%s\"", from, streamid);
					
					(void)unlink(streamid);
					free(from);
					continue;
				}

				free(from);
			}
#			endif	MULTIPATH

			Trace2(2, "found \"%s\"", streamid);

			return true;
		}

		if ( ++clp > &ComLists[chan] )
			clp = ComLists;
	}

	/*
	**	Nothing to do, if directory hasn't changed, give up.
	*/

	if ( LastLook == LastTime )
	{
		MQactive[chan] = (MQl_p)0;
		return false;
	}


	/*
	**	MIPS KLUDGE
	**
	**	Currently the MIPS directory update is broken.
	**	Scan the directory all the time until it's fixed.
	*/

#	ifndef	mips

	while ( fstat(ComDirFd->dd_fd, &statb) == SYSERROR )
	{
		Syserror("Can't stat directory");
		if ( BatchMode )
			finish(SYSERROR);
	}

	Trace2(2, "stat dir mtime %lu", statb.st_mtime);

#	else	/* mips */

	statb.st_mtime = time((long *)0);

#	endif	/* mips */

	/*
	**	The directory may have been written in the same period
	**	that we last looked at it, but just a bit later...
	*/

	if ( LastLook > ComDirMtime && statb.st_mtime == ComDirMtime )
	{
		LastLook = LastTime;
		MQactive[chan] = (MQl_p)0;
		return false;
	}

	LastLook = LastTime;

	ComDirMtime = statb.st_mtime;

	/*
	**	Free all elements from lists.
	*/

	for ( clp = ComLists ; clp < &ComLists[NSTREAMS] ; clp++ )
		if ( clp->cl_count != 0 )
		{
			Trace2(3, "free list %d", clp-ComLists);

			while ( (mqlp = clp->cl_first) != (MQl_p)0 )
			{
				clp->cl_first = mqlp->mql_next;
				mqlp->mql_next = MQfreelist;
				MQfreelist = mqlp;
			}

			free((char *)clp->cl_vec);
			clp->cl_index = 0;
			clp->cl_count = 0;
		}

	/*
	**	Read the directory, building lists.
	*/

	Trace1(2, "readdir");

	rewinddir(ComDirFd);

	while ( (dp = readdir(ComDirFd)) != NULL )
	{
		switch ( dp->d_name[0] )
		{
		case SMALL_ID:	clp = &ComLists[0];	break;
		case MEDIUM_ID:	clp = &ComLists[1];	break;
		case LARGE_ID:	clp = &ComLists[2];	break;
		default:	continue;
		}

		if ( strlen(dp->d_name) != STREAMIDZ )
		{
			(void)strcpy(streamid, Template);
			(void)UniqueName(streamid, (long)0, LastTime);
			(void)link(dp->d_name, streamid);
			(void)unlink(dp->d_name);

			Report2("Bad name for command file: \"%s\"", dp->d_name);
			BadMessage((AQarg)chan, bm_name);

			(void)unlink(streamid);
			streamid[0] = '\0';
			continue;
		}

		Trace2(3, "add \"%s\"", dp->d_name);

		if ( (mqlp = MQfreelist) != (MQl_p)0 )
			MQfreelist = mqlp->mql_next;
		else
			mqlp = Talloc(MQel);
		
		mqlp->mql_next = clp->cl_first;
		clp->cl_first = mqlp;
		clp->cl_count++;

		(void)strcpy(mqlp->mql_name, dp->d_name);
		mqlp->mql_use = m_wait;
	}

	/*
	**	Allocate and sort vectors for new lists.
	*/

	for ( clp = ComLists ; clp < &ComLists[NSTREAMS] ; clp++ )
	{
		register MQl_p *clvp;

		if ( clp->cl_count == 0 )
			continue;		/* Empty List */

		clp->cl_vec = clvp = (MQl_p *)Malloc(clp->cl_count * sizeof(MQl_p));

		for ( mqlp = clp->cl_first ; mqlp != (MQl_p)0 ; mqlp = mqlp->mql_next )
			*clvp++ = mqlp;

		Trace3(2, "new vector for channel %d, %d elements", clp-ComLists, clp->cl_count);

		if ( clp->cl_count == 1 )
			continue;

		qsort((char *)clp->cl_vec, clp->cl_count, sizeof(MQl_p), clcomp);
	}

	/*
	**	Take note of active messages.
	*/

	for ( i = 0 ; i < NSTREAMS ; i++ )
	{
		MQactive[i] = (MQl_p)0;

		switch ( outStreams[i].str_id[0] )
		{
		case SMALL_ID:	clp = &ComLists[0];	break;
		case MEDIUM_ID:	clp = &ComLists[1];	break;
		case LARGE_ID:	clp = &ComLists[2];	break;
		case '\0':	continue;
		default:	Fatal2("FindCommand bad activeid \"%s\"", outStreams[i].str_id);
		}

		for ( mqlp = clp->cl_first ; mqlp != (MQl_p)0 ; mqlp = mqlp->mql_next )
			if ( strcmp(mqlp->mql_name, outStreams[i].str_id) == STREQUAL )
			{
				Trace3(2, "\"%s\" already active on channel %d", outStreams[i].str_id, i);
				mqlp->mql_use = m_active;
				MQactive[i] = mqlp;
				break;
			}

		if ( mqlp == (MQl_p)0 )
		{
			if ( outStreams[i].str_state != STR_ERROR )
				Report3("Command file \"%s\" disappeared from channel %d", outStreams[i].str_id, i);

			outStreams[i].str_state = STR_ERROR;
			outStreams[i].str_id[0] = '\0';
		}
	}

	goto loop;
}



/*
**	Compare two command file names.
*/

int
clcomp(mqlpp1, mqlpp2)
	char *		mqlpp1;
	char *		mqlpp2;
{
	return strcmp((*(MQl_p *)mqlpp1)->mql_name, (*(MQl_p *)mqlpp2)->mql_name);
}
