/*
**	Copyright (c) 1984 Piers Lauder, University of Sydney
**
**	Warning: Distribution of this software without written
**		 permission is prohibited.
*/

static char	sccsid[]	= "@(#)rundaemon.c	1.11 84/10/09";

/*
**	Run a set of daemons, continuously restarting them, until SIGTERM.
*/

char *	Usage		= "[-F] [-T<level>] [-d<daemon>] [-w<delay>] \\\n\
	[-<daemon-args> ...] link ...";

#define	RECOVER

#include "global.h"
#include "debug.h"

#include <errno.h>
#include <signal.h>


#define	MAXWAIT		(5*60)	/* Longest delay allowed */
#define	MINWAIT		2	/* Shortest delay allowed */
#define	MINDELAY	5	/* Minimum delay between execs */
#define	WAITINCR	60	/* Increment to delay if quick termination */

/*
**	Parameters set from arguments
*/

int	Delay		= MINDELAY;
VarArgs	NNdArgs		=
{
	2,
	NNDAEMON,
	"-F"		/* No fork in daemon */
};
char *	Name;
bool	NoFork;		/* No fork in rundaemon */
int	Traceflag;

/*
**	Structure and list to hold active daemons.
*/

typedef struct Proc	Proc;
struct Proc
{
	Proc *	next;
	char *	link;
	Time_t	start;
	int	pid;
	int	delay;
}
	*Procs;

/*
**	Misc.
*/

jmp_buf	Alrm_buf;
int	DaemonsActive;
int	Nlinks;
Time_t	Time;
int	Pid;

bool	Run1Daemon();
void	RunDaemons();
int	catchalarm();
int	termin();



int
main(argc, argv)
	register int	argc;
	register char *	argv[];
{
	if (Name = strrchr(*argv, '/'))
		Name++;
	else
		Name = *argv;

	Recover(ert_finish);

	Time = time((long *)0);

	while ( --argc > 0 )
	{
		if ( **++argv == '-' )
		{
			register int	c;

			while ( c = *++*argv )
			{
				switch ( c )
				{
				case 'F':
					NoFork = true;
					continue;

				case 'T':
					if ( (Traceflag = atoi(++*argv)) <= 0 )
						Traceflag = 1;
					break;

				case 'd':
					ARG(&NNdArgs, 0) = ++*argv;
					goto break2;

				case 'w':
					if ( (Delay = atoi(++*argv)) < MINDELAY )
						Delay = MINDELAY;
					break;

				default:
					if ( NARGS(&NNdArgs) >= MAXVARARGS)
					{
						Error("Too many daemon args");
						return 1;
					}

					NEXTARG(&NNdArgs) = --*argv;
					goto break2;
				}

				while ( (c = **argv) <= '9' && c >= '0' )
					++*argv;
				--*argv;
			}
break2:			;
		}
		else
		{
			register Proc *	pp = Talloc(Proc);

			pp->next = Procs;
			Procs = pp;

			pp->pid = SYSERROR;
			pp->link = *argv;
			pp->delay = Delay;
			pp->start = Time + Delay*(++Nlinks);
		}
	}

	if ( Nlinks == 0 )
	{
		Error("Usage: \"%s %s\"", Name, Usage);
		return 1;
	}

	if ( !NoFork )
		switch ( fork() )
		{
		case SYSERROR:
			Syserror("Can't fork");
			return 1;

		case 0:
			break;

		default:
			return 0;
		}

	Pid = getpid();

#	if	NICEDAEMON
	nice(NICEDAEMON);
#	endif	NICEDAEMON

	SetUser(ACSNETUID, ACSNETGID);

	(void)signal(SIGINT, SIG_IGN);
	(void)signal(SIGHUP, SIG_IGN);
	(void)signal(SIGQUIT, SIG_IGN);

	(void)signal(SIGALRM, catchalarm);
	(void)signal(SIGTERM, termin);

	RunDaemons();
	return 0;
}



void
RunDaemons()
{
	register Proc *		pp;
	register unsigned	minwait;
	register int		pid;
	int			status;

	for ( ;; )
	{
		Time = time((long *)0);

		minwait = MAXWAIT;

		for ( pp = Procs ; pp != (Proc *)0 ; pp = pp->next )
		{
			register unsigned	mw;

			if ( pp->pid == SYSERROR && pp->start <= Time )
			{
				if ( Run1Daemon(pp) )
				{
					pp->start = Time;
					DaemonsActive++;
					(void)sleep(Delay);
					Time += Delay;
					continue;
				}

				pp->start = Time + pp->delay;
			}

			if ( (mw = pp->start - Time) < minwait )
				minwait = mw;
		}

		if ( minwait < MINWAIT )
			minwait = MINWAIT;

		if ( setjmp(Alrm_buf) )
			continue;

		(void)alarm(minwait);

		if ( DaemonsActive )
			pid = wait(&status);
		else
			(void)pause();

		(void)alarm(0);

		Time = time((long *)0);

		if ( pid == SYSERROR )
		{
			for ( pp = Procs ; pp != (Proc *)0 ; pp = pp->next )
			{
				if ( pp->pid != SYSERROR )
				{
					Warn
					(
						"\"%s\" (pid=%d) for link \"%s\" disappeared!",
						ARG(&NNdArgs, 0),
						pp->pid,
						pp->link
					);

					pp->pid = SYSERROR;
					pp->delay += WAITINCR;
					pp->start = Time + pp->delay;
				}
			}

			DaemonsActive = 0;
			continue;
		}

		DaemonsActive--;

		for ( pp = Procs ; pp != (Proc *)0 ; pp = pp->next )
		{
			if ( pp->pid == pid )
			{
				if ( status )
				{
					Warn
					(
						"\"%s\" for link \"%s\" bad exit status:%d signal:%d%s",
						ARG(&NNdArgs, 0),
						pp->link,
						(status>>8)&0xff,
						status&0x7f,
						(status&0x80)?" (core dumped)":""
					);
				}
				else
				{
					Warn
					(
						"\"%s\" for link \"%s\" terminated",
						ARG(&NNdArgs, 0),
						pp->link
					);
				}

				if ( (Time - pp->start) < pp->delay )
				{
					pp->delay += WAITINCR;
					pp->start = Time + pp->delay;
				}
				else
				{
					if ( (Time - pp->start) > MAXWAIT )
						pp->delay = Delay;

					pp->start = Time;
				}

				pp->pid = SYSERROR;
				break;
			}
		}
	}
}



bool
Run1Daemon(pp)
	register Proc *	pp;
{
	Trace3(1, "Starting \"%s %s\"", ARG(&NNdArgs, 0), pp->link);

	switch ( pp->pid = fork() )
	{
	case SYSERROR:
		Syserror("Can't fork");
		return false;

	case 0:
		break;

	default:
		return true;
	}

	NEXTARG(&NNdArgs) = pp->link;
	NEXTARG(&NNdArgs) = NULLSTR;

	execve(ARG(&NNdArgs, 0), &ARG(&NNdArgs, 0), (char **)0);
	Syserror("Can't execve \"%s\"", ARG(&NNdArgs, 0));

	return false;
}



int
termin(sig)
	int		sig;
{
	register Proc *	pp;

	(void)signal(sig, SIG_IGN);

	for ( pp = Procs ; pp != (Proc *)0 ; pp = pp->next )
		if ( pp->pid != SYSERROR )
			kill(pp->pid, SIGTERM);

	finish();
}



int
catchalarm(sig)
	int	sig;
{
	(void)signal(sig, catchalarm);
	longjmp(Alrm_buf, 1);
	return 0;
}



finish()
{
	if ( Pid != getpid() )
		exit(101);

	exit(1);
}
