/*
**	Copyright (c) 1984 Piers Lauder, University of Sydney
**
**	Warning: Distribution of this software without written
**		 permission is prohibited.
*/

static char	sccsid[]	= "@(#)purge.c	1.6 85/02/20";

/*
**	Purge the work directory by scanning the command directories.
*/

char *	Usage		= "[-[A][T<level>][W]] [file ...]";

#define	FILE_CONTROL
#define	STAT_CALL
#define	STDIO

#include	"global.h"

#include	"command.h"
#include	"debug.h"
#include	"spool.h"
#include	"Stream.h"

#include	<ndir.h>


/*
**	Parameters set from arguments
*/

bool	All;				/* remove all unreferenced files - no exceptions! */
char *	Name;				/* Program invoked name */
int	Traceflag;			/* Trace level */
bool	Warnings;			/* Whinge */

/*
**	Structure to count references to a file
*/

#define	WORKNAMLEN	14

typedef struct data_file * DF_p;

typedef struct data_file
{
	DF_p	df_great;		/* Branches of binary tree */
	DF_p	df_less;
	char	df_name[WORKNAMLEN+1];	/* Last componenent of path name in WORKDIR */
	short	df_refs;		/* References */
	short	df_flags;		/* For arg matches */
}
	DF;

#define	HASH_SIZE	257		/* Size of hash-table base of binary trees */
#if	CRC_HASH == 1
#define	hashname(A,L)	(((unsigned)acrc(0,A,L))%HASH_SIZE)
#else	CRC_HASH == 1
#define	HASH(H,C)	H+=H^(C-' ')	/* The hashing function */
#endif	CRC_HASH == 1

/*
**	Structure to remember timed-out files.
*/

typedef struct UnlinkN *UN_p;
typedef struct UnlinkN
{
	UN_p	un_next;
	DF_p	un_dfp;
}
		UnlinkN;

UN_p	Unfreelist;
UN_p	Unlist;

/*
**	Structure for WORKDIR info.
*/

char	Work1dir[]	= WORKDIR();	/* Ends in a '/' */
char	Work2dir[]	= WORK2DIR();

struct Workdir
{
	char *	name;
	int	length;
	DF_p	table[HASH_SIZE];
}
	Workdirs[] =
{
	{Work1dir, sizeof(Work1dir)-1},	/* Must be first */
	{Work2dir, sizeof(Work2dir)-1},
	NULLSTR
};

typedef struct Workdir *	WD_p;

/*
**	Miscellaneous info.
*/

char *	Cantopen	= "can't open \"%s\"";
int	Removed;			/* Count of files removed */
bool	SearchFiles;			/* Explicit file name arg used */
char *	Spooldir	= SPOOLDIR();
Time_t	Time;
long	Time_to_die;

#define	dots(A)		(A[0]=='.'&&(A[1]==0||(A[1]=='.'&&A[2]==0)))
#define	HOUR		(60*60)

/*
**	Routines
*/

void	purge(), remove(), searchcoms(), searchstatus(), usage();
bool	read_com();
DF_p	enter(), lookup();



int
main(argc, argv)
	int		argc;
	register char *	argv[];
{
	if ( (Name = strrchr(*argv, '/')) != NULLSTR )
		Name++;
	else
		Name = *argv;

	Time = time((long *)0);

	while ( --argc > 0 )
	{
		if ( **++argv == '-' )
		{
			register int	c;

			while ( c = *++*argv )
			{
				switch ( c )
				{
				case 'A':
					All = true;
					continue;

				case 'T':
					if ( (Traceflag = atoi(++*argv)) == 0 )
						Traceflag = 1;
					break;

				case 'W':
					Warnings = true;
					continue;

				default:
					usage("unrecognised flag '%c'", c);
					return 1;
				}

				while ( (c = **argv) <= '9' && c >= '0' )
					++*argv;
				--*argv;
			}

break2:			;
		}
		else
		{
			register WD_p	wdp;

			if ( strlen(*argv) > WORKNAMLEN )
			{
				usage("bad name for file \"%s\"", *argv);
				return 1;
			}

			for ( wdp = Workdirs ; wdp->name != NULLSTR ; wdp++ )
			{
				register DF_p	dfp;

				dfp = enter(*argv, wdp->table);
				dfp->df_flags = 1;
			}

			SearchFiles = true;
		}
	}

	purge();

	(void)fprintf(stdout, "%d file%s removed.\n", Removed, Removed==1?"":"s");

	return 0;
}



/*
**	Cleanup for error routines
*/

void
finish(error)
	int	error;
{
	(void)exit(error);
}



/*
**	Explain usage
*/

/*VARARGS1*/
void
usage(s, a1)
	char *	s;
	char *	a1;
{
	Mesg("bad arguments", s, a1);
	(void)fprintf(stderr, "\nUsage is \"%s %s\"\n", Name, Usage);
}



/*
**	Search all possible command directories
**	for references to files in WORKDIR,
**	then remove any file in WORKDIR for which there is no reference.
*/

void
purge()
{
	register WD_p		wdp;
	register struct direct *direp;
	DIR *			dirp;

	/*
	**	Find the link directories.
	*/

	if ( (dirp = opendir(Spooldir)) == NULL )
	{
		Syserror(Cantopen, Spooldir);
		return;
	}

	while ( (direp = readdir(dirp)) != NULL )
		if ( direp->d_name[0] != WORKFLAG && !dots(direp->d_name) )
		{
			register char *	fp;

			fp = concat(Spooldir, direp->d_name, "/", NULLSTR);

			searchcoms(fp, true);

			free(fp);
		}

	closedir(dirp);

	/*
	**	Search the holding directories.
	*/

	searchcoms(BADDIR(), false);
	searchcoms(REROUTEDIR(), true);

	/*
	**	Now remove any files unreferenced by the previous searches.
	*/

	for ( wdp = Workdirs ; wdp->name != NULLSTR ; wdp++ )
	{
		if ( (dirp = opendir(wdp->name)) == NULL )
		{
			Syserror(Cantopen, wdp->name);
			return;
		}

		while ( (direp = readdir(dirp)) != NULL )
		{
			register DF_p	dfp;

			if
			(
				!dots(direp->d_name)
				&&
				(
					(dfp = lookup(direp->d_name, wdp->table)) == (DF_p)0
					||
					dfp->df_refs == 0
				)
			)
			{
				register char *	fp;

				fp = concat(wdp->name, direp->d_name, NULLSTR);

				remove(fp);

				free(fp);
			}
		}

		closedir(dirp);
	}
}



/*
**	Search a directory for command files and enter names of data files.
*/

void
searchcoms(dir, timeouts)
	char *			dir;
	bool			timeouts;
{
	register DIR *		dirp;
	register struct direct *direp;

	Trace2(1, "searchcoms \"%s\"", dir);

	if ( (dirp = opendir(dir)) == NULL )
	{
		struct stat	statb;

		if
		(
			stat(dir, &statb) == SYSERROR
			||
			(statb.st_mode & S_IFMT) == S_IFDIR
		)
			Syserror(Cantopen, dir);

		return;
	}

	while ( (direp = readdir(dirp)) != NULL )
	{
		register UN_p	unp;
		register char *	fp;
		register int	fd;
		Time_t		mtime;

		/*
		**	Find valid command file.
		*/

		switch ( direp->d_name[0] )
		{
		default:
			/*
			**	Check for NNdaemon temporaries.
			*/

			if ( strcmp(direp->d_name, STATUSFILE) == STREQUAL )
			{
				fp = concat(dir, direp->d_name, NULLSTR);
				searchstatus(fp);
				free(fp);
			}

			continue;

		case SMALL_ID:
		case MEDIUM_ID:
		case LARGE_ID:
			break;
		}

		Trace2(2, "found \"%s\"", direp->d_name);

		if ( SearchFiles )
		{
			register WD_p	wdp;
			register DF_p	dfp;

			for ( wdp = Workdirs ; wdp->name != NULLSTR ; wdp++ )
				if
				(
					(dfp = lookup(direp->d_name, wdp->table)) != (DF_p)0
					&&
					dfp->df_flags
				)
				{
					Report("found \"%s%s\"", dir, direp->d_name);
					break;
				}
		}

		fp = concat(dir, direp->d_name, NULLSTR);

		if ( (fd = open(fp, O_READ)) == SYSERROR )
		{
			if ( Warnings )
				SysWarn("Can't read commands file \"%s\"", fp);

			free(fp);
			continue;
		}

		Time_to_die = 0;

		if ( !ReadCom(fd, &mtime, read_com) && Warnings )
			Warn("bad commands file \"%s\"", fp);
		
		(void)close(fd);

		if ( timeouts && Time_to_die > 0 && Time > (mtime + Time_to_die) )
		{
			Trace5(1, "\"%s\" timed out: time=%lu, mtime=%lu, ttd=%ld", fp, Time, mtime, Time_to_die);

			for ( unp = Unlist ; unp != (UN_p)0 ; unp = unp->un_next )
				unp->un_dfp->df_refs--;

			DODEBUG(if(!Traceflag))
			{
				(void)unlink(fp);
				Removed++;
			}
		}

		while ( (unp = Unlist) != (UN_p)0 )
		{
			Unlist = unp->un_next;
			unp->un_next = Unfreelist;
			Unfreelist = unp;
		}

		free(fp);
	}

	closedir(dirp);
}



/*
**	Search NNdaemon status file for receive temporaries,
**	(which are always made in WORKDIR).
*/

void
searchstatus(statusfile)
	char *		statusfile;
{
	register int	i;
	register int	fd;
	NN_state	NNstate;

	Trace2(1, "searchstatus \"%s\"", statusfile);

	for ( i = 0 ; i < 3 ; i++ )
	{
		if ( (fd = open(statusfile, O_READ)) != SYSERROR )
		{
			if ( read(fd, (char *)&NNstate, sizeof NNstate) == sizeof NNstate )
			{
				register Str_p	strp;

				if ( strcmp(NNstate.version, StreamSCCSID) != STREQUAL )
					Error("status file version, re-compile");

				for ( strp = inStreams ; strp < &inStreams[NSTREAMS] ; strp++ )
					if ( strp->str_recv.rh_id[0] != '\0' )
						(void)enter(strp->str_recv.rh_id, Workdirs[0].table);

				break;
			}

			(void)close(fd);
		}

		(void)sleep(1);
	}

	if ( fd != SYSERROR )
		(void)close(fd);
}



/*
**	Function called from "ReadCom" to process a command.
**	Enter any "unlink" file names that are in WORKDIR,
**	and remember them in temporary list in case they have timed-out.
*/

bool
read_com(name, base, range)
	char *	name;
	long	base;
	long	range;
{
	Trace4(3, "command \"%s %ld %ld\"", name, base, range);

	if ( range == 0 )
	{
		if ( base == 0 )
		{
			register WD_p	wdp;
			register UN_p	unp;
			register DF_p	dfp;

			for ( wdp = Workdirs ; wdp->name != NULLSTR ; wdp++ )
				if ( strncmp(wdp->name, name, wdp->length) == STREQUAL )
				{
					if ( strlen(name + wdp->length) <= WORKNAMLEN )
					{
						dfp = enter(name + wdp->length, wdp->table);

						if ( dfp->df_flags )
							Report("found \"%s\"", name);

						if ( (unp = Unfreelist) == (UN_p)0 )
							unp = Talloc(UnlinkN);
						else
							Unfreelist = unp->un_next;
						
						unp->un_next = Unlist;
						Unlist = unp;
						unp->un_dfp = dfp;

						free(name);
						return true;
					}
				}

			if ( Warnings )
				Warn
				(
					"strange name in command file: \"%.*s\"",
					wdp->length + 2*WORKNAMLEN,
					name
				);

			free(name);
			return false;
		}
		else
			Time_to_die = base;
	}

	free(name);
	return true;
}



/*
**	Enter a file name in hash table,
**	and increment reference count.
*/

DF_p
enter(namep, table)
	char *		namep;
	DF_p *		table;
{
	register DF_p *	dfpp;
	register char *	cp1;
	register char *	cp2;
	register int	c;
	register int	len = strlen(namep);

	Trace2(2, "enter \"%s\"", namep);

#	if	DEBUG == 2
	if ( Traceflag > 2 )
	{
		Traceflag--;
		if ( lookup(namep, table) == (DF_p)0 )
			Trace(2, "\"%s\" hash = %#x", namep, hashname(namep, len));
		Traceflag++;
	}
#	endif	DEBUG == 2

	for
	(
		dfpp = &table[hashname(namep, len)] ;
		*dfpp != (DF_p)0 ;
		dfpp = (c&1) ? &((*dfpp)->df_great) : &((*dfpp)->df_less)
	)
	{
#		if	DEBUG == 2
		if ( Traceflag > 2 )
		{
			Traceflag--;
			if ( lookup(namep, table) == (DF_p)0 )
				Trace(2, "hash duplication \"%s\"", (*dfpp)->df_name);
			Traceflag++;
		}
#		endif	DEBUG == 2

		for
		(
			cp1 = namep+len, cp2 = &(*dfpp)->df_name[len] ;
			cp1 != namep ;
		)
			if ( c = *--cp1 ^ *--cp2 )
				break;

		if ( c == 0 && cp1 == namep )
		{
			(*dfpp)->df_refs++;
			return *dfpp;
		}
	}

	*dfpp = Talloc(DF);

	(*dfpp)->df_great = (DF_p)0;
	(*dfpp)->df_less = (DF_p)0;
	(void)strcpy((*dfpp)->df_name, namep);
	(*dfpp)->df_refs = 1;
	(*dfpp)->df_flags = 0;

	return *dfpp;
}



/*
**	Look-up file name in hash table
*/

DF_p
lookup(namep, table)
	char *		namep;
	DF_p *		table;
{
	register DF_p	dfp;
	register char *	cp1;
	register char *	cp2;
	register int	c;
	register int	len = strlen(namep);

	for
	(
		dfp = table[hashname(namep, len)] ;
		dfp != (DF_p)0 ;
		dfp = (c&1) ? dfp->df_great : dfp->df_less
	)
	{
		for
		(
			cp1 = namep+len, cp2 = &dfp->df_name[len] ;
			cp1 != namep ;
		)
			if ( c = *--cp1 ^ *--cp2 )
				break;

		if ( c == 0 && cp1 == namep )
			return dfp;
	}

	return (DF_p)0;
}



#if	CRC_HASH == 0
/*
**	Calculate hash for a file name.
*/

int
hashname(namep, length)
	char *		namep;
	int		length;
{
	register char *	cp = namep;
	register int	hash = 0;
	register int	c;

	while ( c = *cp++ )
		HASH(hash, c);

	Trace3(3, "\"%s\" hash = %#x", namep, hash);

	return (int)((unsigned)hash % HASH_SIZE);
}
#endif	CRC_HASH == 0



/*
**	Remove file.
*/

void
remove(name)
	char *		name;
{
	struct stat	statb;

	if ( stat(name, &statb) == SYSERROR )
	{
		if ( Warnings )
			SysWarn("can't stat \"%s\"", name);
		return;
	}

	if ( !All && statb.st_mtime > (Time-HOUR) )
	{
		Trace2(1, "NOT removing \"%s\", less than 1 hour old", name);
		return;
	}
	else
		Trace2(1, "remove \"%s\"", name);

	if ( Warnings )
		Warn("removing \"%s\"", name);
	
	DODEBUG(if(!Traceflag))
	{
		if ( unlink(name) == SYSERROR )
			SysWarn("Can't unlink \"%s\"", name);
		else
			Removed++;
	}
}
