#
/*
 * ti.c: Texas Instruments 980B driver
 * The 980B minicomputer with magtape, RP04 disk and terminal multiplexer
 * is interfaced to the UNIBUS with an 8 register bidirectional I/O
 * channel, and an interprocessor DMA channel.
 * Commands are issued over the I/O channel, and may or may not 
 * start DMA activity, depending on the actual command.
 * The 908B may issue two interrupts to the PDP 11, at vectors
 * 0170 and 0174. An interrupt at 0170 indicates that the PDP 11 may
 * send the next command, and the 0174 interrupt indicates the completion of
 * of a command, requireing the examination of the I/O registers to
 * determine the outcome of the command.
 *
 * G. Ordy, CWRU, 6/79
 * Bill Shannon, CWRU, 9/3/79
*/

#include "../h/param.h"
#include "../h/buf.h"
#include "../h/dir.h"
#include "../h/user.h"
#include "../h/ti980.h"

struct buf ti_queue;			/* waiting for I/O channel queue */
struct buf ti_wait;			/* waiting for 980B queue */

struct device {
	int	ticsr;		/* control and status register */
	char	*tiaddr;	/* Unibus address */
	unsigned tiblkno;	/* virtual block number */
	int	tibcount;	/* byte count for I/O */
	struct buf *tibp;	/* address of buffer originating request */
	int	tierr;		/* error number */
	int	tiunused;
	int	tipanel;	/* alternate front panel */
};

tiopen(dev, flag)
{
	ti_wait.b_actf = ti_wait.b_actl = &ti_wait;
	ti_wait.b_active = ti_queue.b_active = 0;
	ti_free();
}


tistrategy(bp)
register struct buf *bp;
{
	spl5();
	bp->av_forw = 0;
	if(ti_queue.b_actf == 0)
		ti_queue.b_actf = bp;
	else
		ti_queue.b_actl->av_forw = bp;
	ti_queue.b_actl = bp;
	if(ti_queue.b_active == 0)
		tistart();
	spl0();
}

/*
 * It is the job of tistart to take an entry off of the ti_queue queue
 * and send it to the 980B through the I/O structure. The entry is removed
 * from the ti_queue and placed on the ti_wait queue, where it will wait
 * until finally serviced by the 980B.
 * The buffer is linked in using both pointers, as the order
 * of I/O completion is uncertain, and typically, the completed
 * command may be anywhere in the linked list.
*/

tistart()
{
	register struct buf *bp;

	if ((bp = ti_queue.b_actf) == 0)
		return;
	ti_queue.b_actf = bp->av_forw;
	ti_queue.b_active++;
	ti_wait.b_actf->av_back = bp;
	bp->av_forw = ti_wait.b_actf;
	bp->av_back = &ti_wait;
	ti_wait.b_actf = bp;
	ti_ioset(bp);
}
	
ti_ioset(bp)
register struct buf *bp;
{
	register int cmd;
	register struct device *addr;
	int func;

	addr = TIADDR;
	addr->tibp = bp;
	addr->tibcount = bp->b_bcount;
	addr->tiblkno = bp->b_blkno;
	addr->tiaddr = bp->b_un.b_addr;
	func  = bp->b_resid & 07;
	bp->b_resid = bp->b_bcount;
	if((bp->b_flags & B_CTRL) == 0)
		func = bp->b_flags & 01;
	cmd = (minor(bp->b_dev) << 8) |
	       ((bp->b_xmem & 03) << 4) | 
		((func & 07) << 1) | GO;
	if (bp->b_flags & B_PHYS)
		cmd |= TI_PHYS;
	ti_wait.b_active++;
	addr->ticsr = cmd;
}

tiintr(dev)
{
	register struct buf *bp;
	register struct device *addr;

	addr = TIADDR;
	if (minor(dev)) {
		for(bp = ti_wait.b_actf; bp != &ti_wait; bp = bp->av_forw)
			if (bp == addr->tibp) {
				ti_done(bp);
				return;
			}
		printf("Extra 174 980B interrupt\n");
	}
	ti_queue.b_active = 0;
	tistart();
}

ti_done(bp)
register struct buf *bp;
{
	register int *dp;
	register struct device *addr;

	addr = TIADDR;
	ti_wait.b_active--;
	bp->av_forw->av_back = bp->av_back;
	bp->av_back->av_forw = bp->av_forw;
	bp->b_resid = addr->tibcount;
	bp->b_blkno = addr->tiblkno;
	bp->b_un.b_addr = addr->tiaddr;
	if (addr->ticsr < 0) {
		bp->b_flags |= B_ERROR;
		bp->b_error = addr->tierr;
		deverror(bp, bp->b_error, addr->ticsr);
	}
	ti_free();
	iodone(bp);
}

ti_free()
{
	register struct device *addr;

	addr = TIADDR;
	addr->ticsr = TI_IGN | TI_ACK | GO;
}
