/* (lgl-
 *	The information contained herein is a trade secret of Mark Williams
 *	Company, and  is confidential information.  It is provided  under a
 *	license agreement,  and may be  copied or disclosed  only under the
 *	terms of  that agreement.  Any  reproduction or disclosure  of this
 *	material without the express written authorization of Mark Williams
 *	Company or persuant to the license agreement is unlawful.
 *
 *	COHERENT Version 4.2.12
 *	Copyright (c) 1994
 *	An unpublished work by Mark Williams Company, Chicago.
 *	All rights reserved.
 -lgl) */

/*
 * History:
 *
 * 1.0	Udo Munk	initial release created
 */

#include <sys/errno.h>
#include <sys/coherent.h>
#include <sys/devices.h>
#include <sys/con.h>
#include <sys/poll.h>
#include <sys/file.h>
#include <sys/sched.h>
#include <sys/stat.h>

#define VERSION	"v1.0"

#define DEBUG
/* #define DEBUG_IRQ */

/* -------------------------------------------------------------------- */
/* defines for the port and commands					*/
/* -------------------------------------------------------------------- */

/* aux controller ports */
#define AUX_INPUT_PORT	0x60	/* aux device output */
#define AUX_OUTPUT_PORT	0x60	/* aux device input */
#define AUX_COMMAND	0x64	/* aux device command */
#define AUX_STATUS	0x64	/* aux device status */

/* aux controller status bits */
#define AUX_OBUF_FULL	0x01	/* output buffer (from device) full */
#define AUX_IBUF_FULL	0x02	/* input buffer (to device) full */

/* aux controller commands */
#define AUX_CMD_WRITE	0x60	/* value to write to controller */
#define AUX_MAGIC_WRITE	0xd4	/* value to send to aux device data */
#define AUX_INTS_ON	0x47	/* enable controller interrupts */
#define AUX_INTS_OFF	0x65	/* disable controller interrupts */
#define AUX_DISABLE	0xa7	/* disable aux */
#define AUX_ENABLE	0xa8	/* enable aux */

/* aux device commands */
#define AUX_SET_RES	0xe8	/* set resolution */
#define AUX_SET_SCALE	0xe9	/* set scaling factor */
#define AUX_SET_STREAM	0xea	/* set stream mode */
#define AUX_SET_SAMPLE	0xf3	/* set sample rate */
#define AUX_ENABLE_DEV	0xf4	/* enable aux device */
#define AUX_DISABLE_DEV	0xf5	/* disable aux device */
#define AUX_RESET	0xff	/* reset aux device */

#define MAX_RETRIES	3
#define AUX_BUF_SIZE	2048

/* queue for device output */
struct aux_queue {
	unsigned int head;
	unsigned int tail;
	unsigned int state;
	unsigned char buf[AUX_BUF_SIZE];
};

#define OPEN	1		/* device is open */
#define ASLP	2		/* waiting for mouse data */

/* -------------------------------------------------------------------- */
/* Forward declarations							*/
/* -------------------------------------------------------------------- */

static void bmps_open(), bmps_close(), bmps_read();
static void bmps_load(), bmps_unload();
static int bmps_poll();
void bmpsintr();
static void aux_write_ack(), aux_write_dev(), aux_write_cmd();
static void poll_timeout();

/* -------------------------------------------------------------------- */
/* Variables								*/
/* -------------------------------------------------------------------- */

extern int BMPS_RES;			/* resolution from Space.c */
extern int BMPS_SCALE;			/* scaling from Space.c */
extern int BMPS_RATE;			/* sampling rate from Space.c */
static int initialized = 0;		/* device initialized ? */
static struct aux_queue queue;		/* a buffer queue for data from device */
static TIM timer;			/* timer for waiting */
static char *wait_text = "bmpswait";	/* text for ps while waiting */
static event_t eventp;			/* event for polling */

/* -------------------------------------------------------------------- */
/* CON structure for device						*/
/* -------------------------------------------------------------------- */

CON bmpscon = {
	DFCHR | DFPOL,		/* Flags */
	BM_MAJOR,		/* major device number */
	bmps_open,		/* open */
	bmps_close,		/* close */
	NULL,			/* block */
	bmps_read,		/* read */
	NULL,			/* write */
	NULL,			/* ioctl */
	NULL,			/* powerfail */
	NULL,			/* timeout */
	bmps_load,		/* load */
	bmps_unload,		/* unload */
	bmps_poll		/* poll */
};

/* -------------------------------------------------------------------- */
/* Device driver functions						*/
/* -------------------------------------------------------------------- */

static void bmps_open(dev, mode)
dev_t dev;
int mode;
{
	/*
	 * Normaly I would check for the mode here and if it's not
	 * read only terminate with an error, because it makes no
	 * sense to write something to a PS/2 mouse. Unfortunately
	 * existing software like XFree86 opens the device read/write,
	 * so we may not return an error here.
	 * Well, there's no write routine implemented, so processes
	 * trying to write will fail later there.
	 */

	if (minor(dev)) {
		/* we can only handle one devive with minor number 0 */
#ifdef DEBUG
		devmsg(dev, "bmps: minor device %d not supported\n", dev & 0xff);
#endif
		set_user_error(ENXIO);
		return;
	}

	/* check if driver or device already busy, we can serve one process only */
	if (queue.state & OPEN) {
		set_user_error(EBUSY);
		return;
	}
	if (!poll_status()) {
		set_user_error(EBUSY);
		return;
	}

	queue.state |= OPEN;		/* device is open now */
	queue.head = queue.tail = 0;	/* clear queue */
	aux_write_dev(AUX_ENABLE_DEV);	/* enable aux device */
	aux_write_cmd(AUX_INTS_ON);	/* enable controller interrupts */
	poll_status();
	outb(AUX_COMMAND, AUX_ENABLE);	/* enable aux */
}

static void bmps_close(dev, mode)
dev_t dev;
int mode;
{
	poll_status();
	outb(AUX_COMMAND, AUX_DISABLE);	/* disable aux */
	aux_write_dev(AUX_DISABLE_DEV);	/* disable aux device */
	aux_write_cmd(AUX_INTS_OFF);	/* disable controller interupts */
	queue.state &= ~OPEN;		/* device isn't open anymore */
}

static void bmps_read(dev, iop)
dev_t dev;
IO *iop;
{
	int s;
	register char c;

	s = splhi();

	/* block until some mouse activity occured */
	while (queue.head == queue.tail) {
		/* non blocking read, don't block */
		if (iop->io_flag & (IONDLY | IONONBLOCK)) {
			splx(s);
			set_user_error(EAGAIN);
			return;
		}
		/* set driver state to sleep and go sleeping until something happens */
		queue.state |= ASLP;
		x_sleep(&queue, pritty, slpriSigLjmp, wait_text);
	}

	/* we are awake again, get data from queue and send to process */
	while ((iop->io_ioc > 0) && (queue.head != queue.tail)) {
		c = queue.buf[queue.tail];
		queue.tail = (queue.tail +1) & (AUX_BUF_SIZE - 1);
		if (ioputc(c, iop) < 0)
			break;
	}

	splx(s);
}

static void bmps_load()
{
	printf("PS/2 InPort mouse driver %s\n", VERSION);

	queue.head = queue.tail = 0;	/* init queue */
	queue.state = 0;		/* init driver state */

	aux_write_ack(AUX_SET_RES);	/* set resolution */
	aux_write_ack(BMPS_RES);
	aux_write_ack(AUX_SET_SCALE);	/* set scaling */
	aux_write_ack(BMPS_SCALE);
	aux_write_ack(AUX_SET_SAMPLE);	/* set sampling rate */
	aux_write_ack(BMPS_RATE);
	aux_write_ack(AUX_SET_STREAM);	/* set stream mode */

	initialized++;			/* device initialized now */
}

static void bmps_unload()
{
	initialized = 0;
}

static int bmps_poll(dev, events, msec)
dev_t dev;
int events, msec;
{
	/* we only handle input polling, everything else isn't possible */
	events &= POLLIN;

	/* blocked polling with timeout ? */
	if (msec > 0) {
		timeout(&timer, (msec + (msec % 10) ? 10 : 0) / 10, poll_timeout, 0);
		if (events & POLLIN) {
			pollopen(&eventp);
			if (queue.tail == queue.head)
				events &= ~POLLIN;
		}
		timeout(&timer, 0, NULL, 0);
	/* no timeout */
	} else {
		if (queue.tail == queue.head)
			events &= ~POLLIN;
	}

	return(events);
}

/* -------------------------------------------------------------------- */
/* Interrupt handling							*/
/* -------------------------------------------------------------------- */

void bmpsintr()
{
	int head = queue.head;
	int maxhead = (queue.tail - 1) & (AUX_BUF_SIZE-1);
	int s;

	s = splhi();

#ifdef DEBUG_IRQ
	printf("bmps: interrupt received\n");
#endif

	/* read character from device and put it into queue */
	queue.buf[head] = inb(AUX_INPUT_PORT);
	if (head != maxhead) {
		head++;
		head &= (AUX_BUF_SIZE - 1);
	}

	queue.head = head;

	/* check if some process is waiting for data and awake it if yes */
	if (queue.state & ASLP); {
		queue.state &= ~ASLP;
		wakeup(&queue);
	}

	/* or when a process is polling it needs to become awake too */
	if (eventp.e_procp) {
		pollwake(&eventp);
	}

	splx(s);
}

/* -------------------------------------------------------------------- */
/* Timing functions							*/
/* -------------------------------------------------------------------- */

/*
 * sleep timeout, awake process again
 */
static void awake_bmps_driver()
{
	wakeup(&queue);
}

/*
 * poll timeout, awake process again
 */
static void poll_timeout()
{
	if (eventp.e_procp)
		pollwake(&eventp);
}

/* -------------------------------------------------------------------- */
/* Driver support functions						*/
/* -------------------------------------------------------------------- */

/*
 * write data to the device and wait for ack
 */
static void aux_write_ack(val)
int val;
{
	aux_write_dev(val);
	while ((inb(AUX_STATUS) & AUX_OBUF_FULL) == 0)
		;
	if ((inb(AUX_STATUS) & 0x20) == 0x20)
		inb(AUX_INPUT_PORT);
}

/*
 * write data to the device
 */
static void aux_write_dev(val)
int val;
{
	poll_status();
	outb(AUX_COMMAND, AUX_MAGIC_WRITE);	/* write magic cookie */
	poll_status();
	outb(AUX_OUTPUT_PORT, val);		/* write data */
}

/*
 * write command to the device
 */
static void aux_write_cmd(val)
int val;
{
	poll_status();
	outb(AUX_COMMAND, AUX_CMD_WRITE);
	poll_status();
	outb(AUX_OUTPUT_PORT, val);
}

/*
 * poll status of device
 */
static int poll_status()
{
	int retries = 0;

	while ((inb(AUX_STATUS) & 0x03) && (retries++ < MAX_RETRIES)) {
		if (inb(AUX_STATUS) & 0x01);
			inb(AUX_INPUT_PORT);
		if (!initialized)	/* busy wait when not initialized */
			busyWait(NULL, 5);
		else {			/* else sleep */
			timeout(&timer, 5, awake_bmps_driver, 0);
			x_sleep(&queue, pritty, slpriNoSig, wait_text);
		}
	}
#ifdef DEBUG
	if (retries >= MAX_RETRIES)
		printf("bmps: MAX_RETRIES exceeded\n");
#endif
	return(!(retries == MAX_RETRIES));
}
