MODULE systemtest;

(****************************************************************************

  Note: This test program assumes a KW11-L clock.
        Users with a KW11-P clock must modify the DEVICE
        MODULE clock accordingly.

****************************************************************************)
CONST idlength = 6;
TYPE id = ARRAY 1:idlength OF char;

{
	REAL-TIME SCHEDULER
	-------------------
A rewrite in MODULA of the system described in -
Brinch Hansen, P.  A real-time scheduler.
Information Science. California Institute of Technology.
November 1975
}

  MODULE system;
  DEFINE include, suspend, time, request, release,
         writec, writes, writen, nl;
  USE id, idlength;
  
  CONST maxtask = 10; { max user tasks in system }
  
  PROCEDURE same(s, t: id): boolean;
    VAR i: integer;
  BEGIN
    i := 1;
    WHILE (i<=idlength) AND (s[i]=t[i]) DO inc(i) END;
    same := i>idlength
  END same;


		{ TASKLIST }

  INTERFACE MODULE tasklist;
  DEFINE include, index, lasttask;
  USE    id, idlength, maxtask, same;
  
  VAR map: ARRAY 1:maxtask OF id;
      lasttask: integer;
  
  PROCEDURE include(task: id);
  BEGIN
    inc(lasttask);
    IF lasttask>maxtask THEN halt(0) END;
    map[lasttask] := task
  END include;
  
  PROCEDURE index(task: id): integer;
    VAR t: integer;
  BEGIN
    t := 1;
    WHILE (t<=lasttask) AND NOT same(task, map[t]) DO inc(t) END;
    index := t
  END index;
  
  BEGIN { tasklist }
    lasttask := 0
  END tasklist;


		{ TASKQUEUE }

  INTERFACE MODULE taskqueue;
  DEFINE suspend, resume;
  USE    id, maxtask, index;
  
  VAR queue: ARRAY 1:maxtask OF signal;
  
  PROCEDURE suspend(task: id);
  BEGIN
    wait(queue[index(task)])
  END suspend;
  
  PROCEDURE resume(task: integer);
  BEGIN
    send(queue[task])
  END resume;
  
  END taskqueue;


		{ CLOCK }

  DEVICE MODULE clock[6];
  DEFINE time, tick, settime;
  
  VAR time: integer;
      tick: signal;
  
  PROCEDURE settime(t: integer);
  BEGIN
    time := t
  END settime;
  
  PROCESS kw11l[100B];
    VAR lcs[177546B]: bits;
        ticks: integer;
  BEGIN
    ticks := 0;
    LOOP
      lcs[6] := true;
      doio;
      inc(ticks);
      IF ticks=50 THEN inc(time); ticks := 0; send(tick) END
    END
  END kw11l;
  
  BEGIN { clock }
    time := 0;
    kw11l
  END clock;


		{ SCHEDULE }

  INTERFACE MODULE schedule;
  DEFINE start, stop, period, examine;
  USE    maxtask, time, resume;
  
  TYPE taskstatus = RECORD
                      active: boolean;
                      start, period: integer
                    END;
  
  VAR status: ARRAY 1:maxtask OF taskstatus;
      task: integer;
  
  PROCEDURE start(task, s, p: integer);
  BEGIN
    WITH status[task] DO
      active := true;
      start := s;
      period := p
    END
  END start;
  
  PROCEDURE stop(task: integer);
  BEGIN
    status[task].active := false
  END stop;
  
  PROCEDURE period(task, p: integer);
  BEGIN
    status[task].period := p
  END period;
  
  PROCEDURE examine(task: integer; VAR due: boolean);
  BEGIN
    WITH status[task] DO
      IF active AND (time>=start)
      THEN due := true; start := start+period
      ELSE due := false
      END
    END
  END examine;
  
  BEGIN { schedule }
    task := 1;
    REPEAT stop(task); inc(task) UNTIL task>maxtask
  END schedule;


		{ SCHEDULER }

  PROCESS scheduler;
    VAR task: integer;
        due: boolean;
  BEGIN
    LOOP
      wait(tick);
      task := 1;
      WHILE task<=lasttask DO
        examine(task, due);
        IF due THEN resume(task) END;
        inc(task)
      END
    END
  END scheduler;


		{ KEYBOARD }

  DEVICE MODULE keyboard[4];
  DEFINE get, bell;
  
  VAR buffer: ARRAY 0:127 OF char;
      in, out, chars: integer;
      bell, nonempty, nonfull: signal;
  
  PROCEDURE get(VAR ch: char);
  BEGIN
    IF chars=0 THEN wait(nonempty) END;
    out := (out+1) MOD 128;
    ch := buffer[out];
    dec(chars);
    send(nonfull)
  END get;
  
  PROCESS driver[60B];
    VAR ksr[177560B]: bits;
        kbf[177562B]: char;
        ch: char;
  BEGIN
    LOOP
      IF chars=128 THEN wait(nonfull) END;
      ksr[6] := true;
      doio;
      ksr[6] := false;
      ch := char(integer(kbf) MOD 128);
      IF ch=7C
      THEN send(bell)
      ELSE
        in := (in+1) MOD 128;
        buffer[in] := ch;
        inc(chars);
        send(nonempty)
      END
    END
  END driver;
  
  BEGIN { keyboard }
    in := 0; out := 0;
    chars := 0; driver
  END keyboard;


		{ PRINTER }

  DEVICE MODULE printer[4];
  DEFINE put;
  
  VAR buffer: ARRAY 0:127 OF char;
      in, out, chars: integer;
      nonempty, nonfull: signal;
  
  PROCEDURE put(ch: char);
  BEGIN
    IF chars=128 THEN wait(nonfull) END;
    in := (in+1) MOD 128;
    buffer[in] := ch;
    inc(chars);
    send(nonempty)
  END put;
  
  PROCESS driver[64B];
    VAR psr[177564B]: bits;
        pbf[177566B]: char;
  BEGIN
    LOOP
      IF chars=0 THEN wait(nonempty) END;
      out := (out+1) MOD 128;
      pbf := buffer[out];
      dec(chars);
      send(nonfull);
      psr[6] := true;
      doio;
      psr[6] := false
    END
  END driver;
  
  BEGIN { printer }
    in := 0; out := 0;
    chars := 0; driver
  END printer;


		{ TERMINAL }

  INTERFACE MODULE terminal;
  DEFINE request, release, writec, writen, writes, readc, nl;
  USE    get, put;
  
  VAR free: signal;
      taken: boolean;
  CONST cr = 15C;
        nl = 12C;
  
  PROCEDURE request;
  BEGIN
    IF taken THEN wait(free) END;
    taken := true
  END request;
  
  PROCEDURE release;
  BEGIN
    taken := false;
    send(free)
  END release;
  
  PROCEDURE writec(c: char);
  BEGIN
    IF c=nl THEN put(cr) END;
    put(c)
  END writec;
  
  PROCEDURE writes(s: ARRAY integer OF char);
    VAR i: integer;
  BEGIN
    i := low(s);
    WHILE i<=high(s) DO
      writec(s[i]); inc(i)
    END
  END writes;
  
  PROCEDURE writen(n: integer);
    VAR digit: ARRAY 1:5 OF char;
        i, val: integer;
  BEGIN
    val := n;
    i := 0;
    REPEAT
      inc(i);
      digit[i] := char(val MOD 10+integer('0'));
      val := val/10
    UNTIL val=0;
    REPEAT
      put(digit[i]);
      dec(i)
    UNTIL i=0
  END writen;
  
  PROCEDURE readc(VAR ch: char);
  BEGIN
    REPEAT
      get(ch);
      IF ch=cr THEN ch := nl END
    UNTIL (ch>=' ') OR (ch=nl);
    IF ('a'<=ch) AND (ch<='z')
    THEN ch := char(integer(ch)-32)
    END;
    writec(ch)
  END readc;
  
  BEGIN { terminal }
    taken := false
  END terminal;


		{ OPERATOR }

  PROCESS operator;
  VAR ok: boolean;
      ch: char;
      command: id;
  
  PROCEDURE nextch;
  BEGIN
    IF ok THEN readc(ch) END
  END nextch;

  PROCEDURE moan;
  BEGIN
    IF ok
    THEN WHILE ch<>nl DO nextch END;
      writec(nl); writes('Command error'); writec(nl);
      ok := false
    END
  END moan;

  PROCEDURE readn(VAR n: integer);
  BEGIN
    WHILE ch=' ' DO nextch END;
    IF (ch<'0') OR (ch>'9') THEN moan END;
    n := 0;
    REPEAT
      n := 10*n+integer(ch)-integer('0'); nextch
    UNTIL (ch<'0') OR (ch>'9')
  END readn;
  
  
  PROCEDURE readid(VAR name: id);
    VAR i: integer;
  BEGIN
    WHILE ch=' ' DO nextch END;
    IF (ch<'A') OR (ch>'Z') THEN moan END;
    i := 1;
    REPEAT
      name[i] := ch; nextch; inc(i)
    UNTIL (ch<'A') OR (ch>'Z') OR (i>idlength);
    WHILE i<=idlength DO
      name[i] := ' '; inc(i)
    END
  END readid;
  
  PROCEDURE readtime(VAR time: integer);
    VAR hour, min, sec: integer;
  BEGIN
    min := 0; sec := 0;
    readn(hour);
    IF ch=':'
    THEN nextch; readn(min);
      IF ch=':' THEN nextch; readn(sec) END
    END;
    IF (hour>23) OR (min>59) OR (sec>59) THEN moan END;
    time := 60*(60*hour+min)+sec
  END readtime;
  
  PROCEDURE readtask(VAR task: integer);
    VAR name: id;
  BEGIN
    readid(name);
    task := index(name);
    IF task>lasttask THEN moan END
  END readtask;
  
  PROCEDURE startcom;
    VAR task, s, p: integer;
  BEGIN
    readtask(task); readtime(s); readtime(p);
    IF ok THEN start(task, s, p) END
  END startcom;
  
  PROCEDURE stopcom;
    VAR task: integer;
  BEGIN
    readtask(task);
    IF ok THEN stop(task) END
  END stopcom;
  
  PROCEDURE periodcom;
    VAR task, p: integer;
  BEGIN
    readtask(task); readtime(p);
    IF ok THEN period(task, p) END
  END periodcom;
  
  PROCEDURE timecom;
    VAR time: integer;
  BEGIN
    readtime(time);
    IF ok THEN settime(time) END
  END timecom;
  
  BEGIN { operator }
    LOOP
      wait(bell);
      request;
      writec(nl);
      writes('Type command: ');
      ok := true;
      nextch;
      readid(command);
      IF   same(command, 'START ') THEN startcom
      ELSIF same(command, 'STOP  ') THEN stopcom
      ELSIF same(command, 'PERIOD') THEN periodcom
      ELSIF same(command, 'TIME  ') THEN timecom
      ELSE moan
      END;
      WHILE ch<>nl DO nextch END;
      release
    END
  END operator;
  
  
  BEGIN { System }
    scheduler;
    operator
  END system;


		{ TASK PROCESS }

PROCESS task(me: id);

  PROCEDURE writeid(name: id);
    VAR i: integer;
  BEGIN
    i := 1;
    REPEAT writec(name[i]); inc(i) UNTIL i>idlength
  END writeid;

  PROCEDURE writetime(time: integer);
  BEGIN
    writen(time/3600); writec(':');
    writen(time MOD 3600/60); writec(':');
    writen(time MOD 60)
  END writetime;

BEGIN { task }
  include(me);
  LOOP
    suspend(me);
    request;
    writec(nl);
    writeid(me);
    writec(' ');
    writetime(time);
    writec(nl);
    release
  END
END task;

BEGIN { system test }
  task('JEREMY');
  task('IAN   ');
  task('DAVID ')
END systemtest.
{
.bp
}
