#define	BFI_GET	0
#define	BFI_SET	1

#include	<param.h>
#include	<dir.h>
#include	<signal.h>
#include	<user.h>
#include	<proc.h>
#include	<reg.h>
#include	<conf.h>
#include	<tty.h>
#include	<buf.h>
#include	<seg.h>
#include	<errno.h>

extern struct user u;
extern char b[];

/*
 * BFI-11 Micro Code Loader.
 */

#define NBFI	1
#define BFI0	0160010
#define BYTE	1
#define WORD	0


struct bfireg {		/* physical registers on UNIBUS */
	int	bfi_addr;
	int	bfi_data;
};

struct
{
	char	lobyte;
	char	hibyte;
};

/* Control structure for opening BFI-11 devices */
struct bfi {
	int	bfi_flags;
	struct bfireg	*bfi_maddr;
} bfi[NBFI] = {
	{0, BFI0},
};

/* bfi_flag bits */

#define LOADED	 01
#define RUNNING	 02
#define OPEN	 04
#define ACTIVE	 010

#define RESET	 04	/* not actually stored */

/* open routine used when examining internal BFI-11 memory */
bfiopen(dev, rw)
{
	/* open only if not currently open for data transfers
	 * and not already open for internal access */

	register d = minor(dev);

	if(d >= NBFI)
	{
		u.u_error = ENXIO;
		return;
	}

/*
	if(bfi[d].bfi_flags & (OPEN | ACTIVE))
	{
		u.u_error = EIO;
		return;
	}
*/
	bfi[d].bfi_flags |= OPEN;
}

bficlose(dev, rw)
{
	bfi[minor(dev)].bfi_flags &= ~OPEN;
}

/*
 * This stty function examines only the mode bits
 * passed by the system call. These bits allow reading
 * and writing of certain control bits:
 * bit 0	LOADED		the BFI-11 is loaded with a control program
 *     1	RUNNING		when read the BFI-11 processor is running
 *				when written start the BFI-11 processor
 *     2	RESET		when written reset the BFI-11 processor
 */
bfioctl(dev, cmd, addr, flag)
{
	register i;
	register d = minor(dev);
	int bfi_buf;

	if(cmd == BFI_GET)
	{
		bfi_buf = bfi[d].bfi_flags & (LOADED|RUNNING);
		if (copyout((caddr_t)&bfi_buf, addr, sizeof bfi_buf))
			u.u_error = EFAULT;
		return;
	}
	if (cmd != BFI_SET)
	{
		u.u_error = EINVAL;
		return;
	}
	if (copyin(addr, (caddr_t)&bfi_buf, sizeof bfi_buf))
	{
		u.u_error = EFAULT;
		return;
	}
	bfi[d].bfi_flags &= ~(LOADED|RUNNING);
	bfi[d].bfi_flags |= (bfi_buf & (LOADED|RUNNING));
	if(bfi_buf & RESET)
	{
		bfi[d].bfi_flags &= ~RUNNING;
		i = 0;		/* halt and reset */
	}
	else if(bfi_buf & RUNNING)
		i = 3;		/* go */
	else
		i = 2;		/* halt */
	spl5();
	bfi[d].bfi_maddr->bfi_addr = 0211; /* address of internal control register */
	bfi[d].bfi_maddr->bfi_data.hibyte = i;
	spl0();
}

/*
 * read internal memory
 * do not pass the top of memory and wrap around
 */

bfiread(dev)
{
	register c;
	register struct bfireg *bp = bfi[minor(dev)].bfi_maddr;

	if(u.u_offset & 0xffff0000) /* may only read to 0xffff */
	{
		u.u_error = EIO;
		return;
	}

	do {
		if(u.u_offset & 0xffff0000) /* end of memory ? */
			return;
		c = bfird(bp, 0, (int)u.u_offset, BYTE);
	} while ((passc(c) >= 0) && (u.u_error ==0));
}

/*
 * write into the BFI-11 memory
 * do not pass the end of the memory and
 * wrap around
 */
bfiwrite(dev)
{
	register c;
	register struct bfireg *bp = bfi[minor(dev)].bfi_maddr;
	register addr;

	if(u.u_offset & 0xffff0000) /* may only write to 0xffff */
	{
		u.u_error = EIO;
		return;
	}

	for(;;)
	{
		if(u.u_offset & 0xffff0000) /* end of memory ? */
			return;
		addr = (int)u.u_offset;
		if((c = cpass()) <0 || u.u_error)
			return;
		bfiwrt(bp, 0, addr, c, BYTE);
	}
}
/*
 * write a port register mapping to the
 * physical internal address
 */
bfiwrt(bfi, dev, reg, data, type)
struct bfireg *bfi;
{
	register sps;

	sps = spl5();
	bfi->bfi_addr = reg;
	if(type == WORD)
		bfi->bfi_data = data;
	else
	{
		if(reg & 01)
			bfi->bfi_data.hibyte = data;
		else
			bfi->bfi_data.lobyte = data;
	}
	splx(sps);
}

/*
 * read a port register mapping to the
 * physical internal address
 */

bfird(bfi, dev, reg, type)
struct bfireg *bfi;
{

	register sps;
	register data;

	sps = spl5();
	bfi->bfi_addr = reg;
	if(type == WORD)
	{
		data = bfi->bfi_data;
	}
	else
	{
		if(reg & 01)
			data = bfi->bfi_data.hibyte;
		else
			data = bfi->bfi_data.lobyte;
	}
	splx(sps);
	return(data);
}

#define NBF	2	/* number of buffers in the bfi buffer pool */
#define BFSZ	512	/* number of chars in each buffer */

#define TRANSCS		0170000		/* address of transmitter control and status register */
#define TRANSBADDR	0170002		/* transmitter bus address register */
#define TRANSBCOUNT	0170004		/* transmitter byte count register */
#define TRANSRETRY	0170006		/* transmitter retry counter */

#define RECCS		0170010		/* address of receiver control and status register */
#define RECBADDR	0170012		/* receiver bus address register */
#define RECBCOUNT	0170014		/* receiver byte count register */
#define RECCOUNT	0170016		/* receiver received byte count */

/* transmitter and receiver commands */
#define	GO		0000001
#define	IE		0000100

/* transmitter and receiver status */
#define	RDY		0000200
#define BFIERROR	0100000
#define BFIDMAERROR	0040000

/* transmitter commands */
#define	SENDEOF		0000010

/* transmitter status */

/* receiver commands */
#define	BFIRESET	0000010
#define IGNORE		0000020

/* receiver status */
#define RECEOF		0000400


#define RSLEEP	(PZERO+6)
#define TSLEEP	(PZERO+7)
#define CSLEEP	(PZERO+8)

struct bfi_cont {
	char	ractive; /* synchronizing flag for receiver start routine */
	char	tactive; /* synchronizing flag for transmitter start routine */
	int	bfi_cflag;	/* flag to indicate bfi port is closing */
	struct buf *bfi_rnready; /* pointer to next full receiver buffer */
	struct buf *bfi_rnfree;  /* pointer to next empty receiver buffer */

	int	bfi_tcount;	 /* count of number of transmitter buffers */
	struct buf *bfi_tfirst;	 /* next buffer to transmitt */
	struct buf *bfi_tlast;	 /* last buffer to transmitt */
} bfip[NBFI];


bfipopen(dev, flg)
{
	register struct bfi *p;
	register struct bfi_cont *q;
	register d = minor(dev);
	struct buf *bp1, *bp2;
	int i;

	if(d >= NBFI)
	{
		u.u_error = ENXIO;
		return;
	}

	p = &bfi[d];
	q = &bfip[d];
	if((p->bfi_flags & OPEN)
	  || ((p->bfi_flags & (RUNNING | LOADED)) != (RUNNING | LOADED)))
	{
		u.u_error = EACCES;
		return;
	}

	if( !(p->bfi_flags & ACTIVE))
	{
		p->bfi_flags |= ACTIVE;
	
		q->tactive = 0;
		q->ractive = 0;
		q->bfi_cflag = 0;
		q->bfi_tcount = 0;
		q->bfi_tfirst = 0;
		q->bfi_rnready = q->bfi_rnfree = bp1 = geteblk();
		for(i = 1;i < NBF; i++)
		{
			bp2 = geteblk();
			bp1->av_forw = bp2;
			bp1 = bp2;
		}
		bp1->av_forw = q->bfi_rnready;
	}


	spl5();
	bfirstart(bfi[d].bfi_maddr, q, IE+BFIRESET+GO);
	spl0();
}

bfipclose(dev)
{
	register struct bfi *p = &bfi[minor(dev)];
	register struct buf *bp1, *bp2;
	struct bfi_cont *q;
	int i;
	extern lbolt;

	q = &bfip[minor(dev)];

	q->bfi_cflag = 1;

	/* disable receiver */
	bfiwrt(p->bfi_maddr, 0, RECCS, IGNORE, WORD);

	/* wait for transmiter to drain */
	spl5();
	while(q->bfi_tcount != 0)
		sleep(&q->bfi_tcount, TSLEEP);
	spl0();

	/* wait for the above to work (micros are slow) */
	sleep(&lbolt, CSLEEP);
	sleep(&lbolt, CSLEEP);


	/* discard receiver buffers */
	for(i = 0, bp1 = q->bfi_rnready;i < NBF;i++)
	{
		bp2 = bp1->av_forw;
		brelse(bp1);
		bp1 = bp2;
	}

	/* send EOF to other end this will go immediately */
	bfiwrt(p->bfi_maddr, 0, TRANSCS, SENDEOF+GO, WORD);

	q->bfi_tfirst = 0;
	q->bfi_tcount = 0;
	q->bfi_cflag = 0;
	p->bfi_flags &= ~ ACTIVE;
}

bfipread(dev)
{
	register struct buf *p;
	struct bfi_cont *q;
	register char *cp;
	register size;
	int sz;

	q = &bfip[minor(dev)];
	p = q->bfi_rnready;
	spl5();
	while((p->b_flags & B_DONE) == 0)
		sleep(&q->ractive, RSLEEP);

	spl0();

	if(size = p->b_bcount)		/* end of file? */
	{
		if(size > u.u_count)
			size = u.u_count;
		u.u_segflg = USERD;
		ka5->r[0] = baddr(p);
		iomove(&b[p->b_resid], size, B_READ);
		if (u.u_error == 0)
		{
			p->b_resid += size;
			p->b_bcount -= size;
		}
		else
			p->b_bcount = 0;
	}

	if(p->b_bcount == 0)
	{
		spl5();
		p->b_flags &= ~ B_DONE;
		q->bfi_rnready = p->av_forw;
		bfirstart(bfi[minor(dev)].bfi_maddr, q, IE+GO);
		spl0();
	}
}

bfipwrite(dev)
{

	register char *cp;
	register struct buf *p;
	register struct bfi_cont *q;
	register size;

	while(u.u_count)
	{
		q = &bfip[minor(dev)];

		spl5();
		while(q->bfi_tcount == NBF)
			sleep(&q->bfi_tcount, TSLEEP);

		spl0();
		p = geteblk();
		p->av_forw = 0;

		size = u.u_count;
		if(size >= BFSZ)
			size = BFSZ;
		p->b_bcount = size;
		u.u_segflg = USERD;
		ka5->r[0] = baddr(p);
		iomove(&b, size, B_WRITE);
		if (u.u_error)
		{
			brelse(p);
			return;
		}
		spl5();
		if(q->bfi_tfirst == 0)
			q->bfi_tfirst = p;
		else
			q->bfi_tlast->av_forw = p;
		q->bfi_tlast = p;
		q->bfi_tcount++;
		bfixstart(bfi[minor(dev)].bfi_maddr, q);
		spl0();
	}
}

bfirint(dev)
{
	register struct bfi_cont *p = &bfip[minor(dev)];
	register struct bfireg *abfi = bfi[minor(dev)].bfi_maddr;


	if(!(bfird(abfi, 0, RECCS, WORD) & RECEOF))
		p->bfi_rnfree->b_bcount = bfird(abfi, 0, RECCOUNT, WORD);
	else
		p->bfi_rnfree->b_bcount = 0;
	p->bfi_rnfree->b_resid = 0;		/* start at &b[0] */
	p->bfi_rnfree->b_flags |= B_DONE;
	p->ractive = 0;

	wakeup(&p->ractive);
	p->bfi_rnfree = p->bfi_rnfree->av_forw;

	bfirstart(abfi, p, IE+GO);
}

bfixint(dev)
{

	register struct bfi_cont *p = &bfip[minor(dev)];
	register struct buf *bp;
	register struct bfireg *abfi = bfi[minor(dev)].bfi_maddr;


	p->bfi_tcount--;

	wakeup(&p->bfi_tcount);

	p->tactive = 0;
	if(p->bfi_tfirst == 0)
			return;
	bp = p->bfi_tfirst->av_forw;
	brelse(p->bfi_tfirst);
	p->bfi_tfirst = bp;
	if(bp == 0)
	{
		p->bfi_tlast = 0;
		return;
	}
	bfixstart(abfi, p);
}


bfirstart(abfi, p, cmd)
register struct bfireg *abfi;
register struct bfi_cont *p;
{
	register struct buf *q;
	extern unsigned short bufarea;

	if(p->bfi_cflag)
	{
		/* if this flag is set then the call is from an interrupt */
		bfiwrt(abfi, 0, RECCS, IGNORE, WORD);
		return;
	}
	q = p->bfi_rnfree;

	if((!p->ractive) && ((q->b_flags & B_DONE) == 0))
	{
		bfiwrt(abfi, 0, RECBADDR, q->b_addr - (bufarea << 6), WORD);
		bfiwrt(abfi, 0, RECBCOUNT, -BFSZ, WORD);
		bfiwrt(abfi, 0, RECCS, cmd, WORD);

		p->ractive = 1;
	}
}

bfixstart(abfi, p)
register struct bfireg *abfi;
register struct bfi_cont *p;
{
	register struct buf *q;
	extern unsigned short bufarea;

	q = p->bfi_tfirst;

	if((!p->tactive) && (p->bfi_tcount))
	{
		bfiwrt(abfi, 0, TRANSBADDR, q->b_addr - (bufarea << 6), WORD);
		bfiwrt(abfi, 0, TRANSBCOUNT, -q->b_bcount, WORD);
		bfiwrt(abfi, 0, TRANSCS, IE+GO, WORD);
		p->tactive = 1;
	}
}
