/*
 *	DZ11 driver
 */

#include <param.h>
#include <dir.h>
#include <signal.h>
#include <user.h>
#include <file.h>
#include <tty.h>
#include <sgtty.h>
#include <errno.h>
#include <fcntl.h>

extern struct user u;

#define	NDZ	5
#define	NPORT	(8*NDZ)

struct device *dz_addr[NDZ] =
{
	(struct device *)0160100,
	(struct device *)0160110,
	(struct device *)0160120,
	(struct device *)0160130,
	(struct device *)0160140
};

struct dz
{
	char dz_cc;	/* set bit if line is carrier-controlled */
	char dz_du;	/* set for dial-up lines */
	char dz_wb;	/* set if transmitting when carrier dropped */
}
dz[NDZ] =			/* this table should be set up in cf */
{
	{ 0377, 0000 },
	{ 0377, 0000 },
	{ 0377, 0000 },
	{ 0243, 0000 },
	{ 0236, 0200 },
};

int	dz_cnt = NPORT;
struct tty dz_tty[NPORT];
char	dz_scan;

char	dz_speeds[] =
{
	000, 020, 021, 022, 023, 024, 000, 025,
	026, 027, 030, 032, 034, 036, 037, 000,
};

#define	BITS7	020
#define	BITS8	030
#define	TWOSB	040
#define	PENABLE	0100
#define	OPAR	0200
#define	RCVENA	010000

#define	IE	050140	/* includes silo alarm enable */
#define	PERROR	010000
#define	FRERROR	020000
#define	OVERROR	040000
#define	SSPEED	B2400	/* standard speed */

#define	SCAN	2	/* ticks between receiver scans */
#define	CSCAN	(HZ/(5*SCAN))	/* SCANs between carrier signals */
#define	MSCAN	5	/* CSCANs between modem signals */
#define	RESPOND	30	/* MSCANs between ring and connect failure */

struct device
{
	short	dzcsr, dzrbuf;
	char	dztcr, dzdtr;
	char	dztbuf, dzbrk;
};
#define	dzlpr	dzrbuf
#define	dzcarr	dzbrk
#define	dzring	dztbuf

#define	ON	1
#define	OFF	0


dzopen(dev, flag)
{
	register struct tty *tp;
	register char carr_cntrl;
	int x;
	extern dzstart(), dzscan();

	x = dev;
	dev = minor(dev);
	if (dev >= dz_cnt)
	{
		u.u_error = ENXIO;
		return;
	}
	carr_cntrl = (dz[dev >> 3].dz_cc & (1 << (dev & 07)));
	tp = &dz_tty[dev];
	spl5();
	if ((tp->t_state&(ISOPEN|WOPEN)) == 0)
	{
		tp->t_oproc = dzstart;
		tp->t_erase = CERASE;
		tp->t_kill = CKILL;
		tp->t_ispeed = SSPEED;
		tp->t_ospeed = SSPEED;
		tp->t_flags = ODDP|EVENP|ECHO;
		dzparam(dev);
	}
	else if (tp->t_state & XCLUDE)
	{
		u.u_error = ENXIO;
		spl0();
		return;
	}
	if (carr_cntrl == 0)		/* line is not carrier-controlled */
		dzmodem(dev, ON);
	if (flag & O_EXCL)
		tp->t_state |= XCLUDE;
	if ((flag & O_NDELAY) == 0 && carr_cntrl)
	{
		tp->t_char = 0;
		while ((tp->t_state & CARR_ON) == 0)
		{
			tp->t_state |= WOPEN;
			sleep(&tp->t_canq, TTIPRI);
		}
	}
	spl6();
	ttyopen(x, tp);
	spl0();
}

dzclose(dev)
{
	register struct tty *tp;

	dev = minor(dev);
	tp = &dz_tty[dev];
	wflushtty(tp);
	spl5();
	if (tp->t_state&HUPCLS)
	{
		dzmodem(dev, OFF);
	}
	tp->t_state &= CARR_ON;
	spl0();
}

dzread(dev)
{
	dev = minor(dev);
	ttread(&dz_tty[dev]);
}

dzwrite(dev)
{
	dev = minor(dev);
	ttwrite(&dz_tty[dev]);
}

dzioctl(dev, cmd, addr, flag)
{
	register struct tty *tp;

	dev = minor(dev);
	tp = &dz_tty[dev];
	if (ttioccomm(cmd, tp, (caddr_t)addr, dev))
	{
		if (cmd==TIOCSETP||cmd==TIOCSETN)
		{
			spl5();
			dzparam(dev);
			spl0();
		}
	}
	else
	{
		u.u_error = ENOTTY;
	}
}

dzparam(dev)
{
	register struct tty *tp;
	register struct device *dzaddr;
	register lpr;

	tp = &dz_tty[dev];
	dzaddr = dz_addr[dev>>3];
	dzaddr->dzcsr = IE;
	if (dz_scan==0)
		dzscan();
	if (tp->t_ispeed==0)	/* Hang up line */
	{
		dzmodem(dev, OFF);
		return;
	}
	lpr = (dz_speeds[tp->t_ispeed]<<8)|(dev&07);
	if (tp->t_flags&RAW)
		lpr |= BITS8;
	else
		lpr |= BITS7|PENABLE;
	if ((tp->t_flags&EVENP)==0)
		lpr |= OPAR;
	if (tp->t_ispeed == 3)	/* 110 baud */
		lpr |= TWOSB;
	dzaddr->dzlpr = lpr;
}

dzrint(dev)
{
	register struct tty *tp;
	register c;
	register struct device *dzaddr;
	register char bit;

	dzaddr = dz_addr[dev];
	while ((c = dzaddr->dzrbuf) < 0)	/* char. present */
	{
		tp = &dz_tty[((c>>8)&07)|(dev<<3)];
		bit = (1 << ((c >> 8) & 07));
		if (tp >= &dz_tty[dz_cnt])
			continue;
		if (!(tp->t_state & (ISOPEN|WOPEN)))
			continue;
		if ((dz[dev].dz_cc & bit) && (dzaddr->dzcarr & bit) == 0)
			continue;
		if (c & (PERROR | FRERROR | OVERROR))
		{
			if (c & FRERROR)		/* break */
				if (tp->t_flags&(RAW|CBREAK))
					c = 0;		/* null (for getty) */
				else
					c = 0177;	/* DEL (intr) */
			if (c&PERROR)
				if ((tp->t_flags&(EVENP|ODDP))==EVENP
				 || (tp->t_flags&(EVENP|ODDP))==ODDP )
					continue;
		}
		ttyinput(c, tp);
	}
}

dzxint(dev)
{
	register struct tty *tp;
	register struct device *dzaddr;
	register unit;
	register char active;
	register bit;

	dzaddr = dz_addr[dev];
	while(dzaddr->dzcsr<0)	/* TX rdy */
	{
		unit = (dzaddr->dzcsr >> 8) & 07;
		bit = (1 << unit);
		active = (dz[dev].dz_cc & bit) == 0 || (dzaddr->dzcarr & bit);
		tp = &dz_tty[(dev<<3)|unit];
		if ((tp->t_state & BUSY) && active)
		{
			dzaddr->dztbuf = tp->t_char;
			tp->t_state &= ~BUSY;
			dzstart(tp);
			continue;
		}
		dzaddr->dztcr &= ~bit;
		if (! active && (dz[dev].dz_du & bit) == 0)
		{
			dzaddr->dzdtr &= ~bit;
			dz[dev].dz_wb |= bit;
		}
	}
}

dzstart(tp)
register struct tty *tp;
{
	register unit, c;
	int s;
	struct device *dzaddr;
	extern ttrstrt();

	unit = tp - dz_tty;
	dzaddr = dz_addr[unit>>3];
	unit = 1<<(unit&07);
	s = spl5();
	if (tp->t_state&(TIMEOUT|BUSY|TTSTOP))
	{
		splx(s);
		return;
	}
	if ((c=getc(&tp->t_outq)) >= 0)
	{
		if (c>=0200 && (tp->t_flags&RAW)==0)
		{
			/*
			 * t_char is used to store timeout period
			 * reducing the max. size req'd for
			 * the timeout list
			 */
			tp->t_state |= TIMEOUT;
			tp->t_char = c & 0177;
		}
		else
		{
			tp->t_char = c;
			tp->t_state |= BUSY;
			dzaddr->dztcr |= unit;
		}
	}
	if (tp->t_outq.c_cc<=TTLOWAT && tp->t_state&ASLEEP)
	{
		tp->t_state &= ~ASLEEP;
		wakeup((caddr_t)&tp->t_outq);
	}
	splx(s);
}

dzmodem(dev, flag)
{
	register struct device *dzaddr;
	register bit;

	dzaddr = dz_addr[dev>>3];
	bit = 1<<(dev&07);
	if (flag==OFF)
		dzaddr->dzdtr &= ~bit;
	else
		dzaddr->dzdtr |= bit;
}

dzscan()
{
	register i;
	register struct device *dzaddr;
	register struct tty *tp;
	char	bit;
	char	carr_cntrl;
	char	dialup;
	short	dznum;
	static	cscan, mscan;

	for (i = 0, tp = dz_tty; i < dz_cnt; i++, tp++)
	{
		if (tp->t_state & TIMEOUT)
			if ((tp->t_char -= SCAN) <= 0)
				ttrstrt(tp);
		if (cscan == 0)
		{
			bit = 1 << (i & 07);
			dznum = i >> 3;
			dzaddr = dz_addr[dznum];
			carr_cntrl = (dz[dznum].dz_cc & bit);
			dialup = (dz[dznum].dz_du & bit);
			if (carr_cntrl == 0 || (dzaddr->dzcarr & bit))
			{
				if ((tp->t_state&CARR_ON)==0)
				{
					wakeup((caddr_t)&tp->t_canq);
					tp->t_state |= CARR_ON;
				}
				if (carr_cntrl)
				{
					dzaddr->dzdtr |= bit;
					if (dz[dznum].dz_wb & bit)
					{
						dz[dznum].dz_wb &= ~bit;
						dzaddr->dztcr |= bit;
					}
				}
			}
			else		/* carrier controlled line - no carrier */
			{
				if (dialup == 0)
					dzaddr->dzdtr &= ~bit;
				else if (mscan == 0)
				{
					if (tp->t_state & CARR_ON)
					{
						if (tp->t_state & ISOPEN)
						{
							signal(tp->t_pgrp, SIGHUP);
							dzaddr->dzdtr &= ~bit;
							flushtty(tp);
						}
						tp->t_state &= ~CARR_ON;
					}
					else if (tp->t_state & WOPEN)
					{
						if (dzaddr->dzring & bit)
							tp->t_char = RESPOND;
						else if (tp->t_char && --tp->t_char == 0)
							dzaddr->dzdtr &= ~bit;
						else
							dzaddr->dzdtr |= bit;
					}
				}
			}
		}
	}
	if (--cscan < 0)
	{
		cscan = CSCAN - 1;
		if (--mscan < 0)
			mscan = MSCAN - 1;
	}
	for (i = dz_cnt >> 3; --i >= 0; )
		dzrint(i);
	dz_scan = timeout(dzscan, 0, SCAN) != NULL;
}
