/*
 * Each buffer in the pool is usually doubly linked into 2 lists:
 * one of a number of hash lists, depending on the device and
 * block number, with which it is currently associated (always)
 * and also on a list of blocks available for allocation
 * for other use (usually).
 * The latter list is kept in last-used order, and the two
 * lists are doubly linked to make it easy to remove
 * a buffer from one list when it was found by
 * looking through the other.
 * A buffer is on the available list, and is liable
 * to be reassigned to another disk block, if and only
 * if it is not marked BUSY or B_SUPER.  When a buffer is busy,
 * the available-list pointers can be used for other purposes.
 * Most drivers use the forward ptr as a link in their I/O
 * active queue.
 * A buffer header contains all the information required
 * to perform I/O.
 * Most of the routines which manipulate these things
 * are in bio.c.
 *
 * The buffers are all at the same virtual address (0120000).
 */

struct buf
{
	struct	buf *b_forw;		/* headed by bhash (below) */
	struct	buf *b_back;		/*  "  */
	int	b_flags;		/* see defines below */
	struct	buf *av_forw;		/* position on free list, */
	struct	buf *av_back;		/*	if not BUSY */
	dev_t	b_dev;			/* major+minor device name */
	unsigned b_bcount;		/* transfer count */
	caddr_t b_addr;			/* low order core address */
	daddr_t	b_blkno;		/* block # on device */
	char	b_xmem;			/* high order core address */
	char	b_error;		/* returned after I/O */
	unsigned int b_resid;		/* words not transferred after error */
	unsigned int b_scratch;		/* used by optimised disk drivers */
};

/*
 * These flags are kept in b_flags.
 */
#define	B_WRITE	     0	/* non-read pseudo-flag (CARE: no bit testing!) */
#define	B_READ	    01	/* read when I/O occurs */
#define	B_DONE	    02	/* transaction finished */
#define	B_ERROR	    04	/* transaction aborted */
#define	B_BUSY	   010	/* not on av_forw/back list */
#define	B_PHYS	   020	/* Physical IO potentially using UNIBUS map */
#define	B_MAP	   040	/* This block has the UNIBUS map allocated */
#define	B_WANTED  0100	/* issue wakeup when BUSY goes off */
#define	B_AGE	  0200	/* delayed write for correct aging */
#define	B_ASYNC	  0400	/* don't wait for I/O completion */
#define	B_DELWRI 01000	/* don't write till block leaves available list */
#define	B_SUPER	 02000	/* this is the superblock for a mounted file system */
#define	B_INVAL	 04000	/* contents to be considered out of date */

/*
 * special redeclarations for
 * the head of the queue per
 * device driver.
 */
#define	b_actf	av_forw
#define	b_actl	av_back
#define	b_active b_bcount
#define	b_errcnt b_resid

/*
 * hash table for quick look-up of blocks and devices
 * (replaces the old d_flags)
 */
struct bhash
{
	struct	buf *b_forw;		/* first buffer that hashes here */
	struct	buf *b_back;		/* last buffer that hashes here */
};

/*
 * the hashing algorithm itself - arguments are device and block number
 */
#define	BHASH(d, b)	(((unsigned)d + (unsigned)b) & (NBHASH-1))
