/* p11 - pdp11 emulator; Copyright (C) 1994 Hartmut Brandt, Joerg Micheel 
 * see the file LICENSE for further information */

# include <stdio.h>
# include <stdlib.h>
# include <sys/types.h>
# include <sys/socket.h>
# include <sys/ioctl.h>
# include <sys/time.h>
# include <errno.h>
# ifdef USE_TERMIOSH
#  include <termios.h>
# endif
# ifdef HAS_VARARGS
#  include <varargs.h>
# else
#  include <stdarg.h>
# endif
# include <signal.h>
# include <setjmp.h>
# include <fcntl.h>
# include <unistd.h>
# include <string.h>

# include "../system.h"
# include "../compat.h"

# include "tty.h"

# define CHECK_PARENT	10	/* seconds to next check if parent is alive */

# define IAC	0377		/* telnet IAC character */


int	rstate = 0;	/* \r seen 			*/
int	iacstate = 0;	/* telnet IAC seen 		*/
uchar	buf[2048];	/* i/o buffer 			*/
int	got;		/* number of chars		*/
int	ticks;		/* parent check tick count	*/
pid_t	ppid;		/* parent pid			*/

/*
 * is parent still alive?
 */
void
onalrm()
{
	if(kill(ppid, 0))
		Exit(0);
	ticks++;
}

/*
 * some signal occured - exit
 */
void
onsig()
{
	Exit(0);
}


/*
 * initialize timer
 */
void
tty_init()
{
	struct itimerval	tim;
	struct sigaction 	sigact;
	sigset_t 		set;

	ppid = getppid();

	sigemptyset(&set);

	bzero(&sigact, sizeof(sigact));
	sigact.sa_handler = onalrm;
	sigact.sa_mask = set;
	sigact.sa_flags = SA_RESTART;
	if(sigaction(SIGALRM, &sigact, NULL))	
		Exit(10);

	bzero(&tim, sizeof(tim));
	tim.it_interval.tv_sec = CHECK_PARENT;
	tim.it_value.tv_sec = CHECK_PARENT;
	if(setitimer(ITIMER_REAL, &tim, NULL))
		Exit(11);

	signal(SIGINT, onsig);
	signal(SIGTERM, onsig);
	signal(SIGHUP, onsig);
}


/*
 * fd 0 is socket to parent
 */
void
tty_loop(int fd)
{
	fd_set	iset, irset;

	rstate = iacstate = 0;

	FD_ZERO(&iset);
	FD_SET(0, &iset);
	FD_SET(fd, &iset);

	write( fd, "-----\r\n", 7);

	while(irset = iset, (got = select(FD_SETSIZE, &irset, 0, 0, 0)) >= 0 || errno == EINTR) {
		if(got < 0)
			continue;

		if(FD_ISSET(fd, &irset)) {
			if((got = read(fd, buf, sizeof(buf)/2)) <= 0)
				break;		/* connection closed */
			process_input();
			write(0, buf, got);
		}

		if(FD_ISSET(0, &irset)) {
			if((got = read(0, buf, sizeof(buf)/2)) <= 0)
					break;	/* parent died :-( */
			process_output();
			write(fd, buf, got);
		}
	}
	write( fd, "\r=====\r\n", 8);
}


/*
 * strip kl-generated parity on output, remove \377
 */
void
makecs7()
{
	uchar	*wp = buf;
	uchar	*rp = buf;
	int	i;

	while(rp < &buf[got])
		if((*wp++ = (*rp++ & 0177)) == '\177')
			wp--;
	got = wp - buf;
}


/*
 * remove NVT cr/lf mapping
 * \r -> \r\0
 * \n -> \r\n
 */
void
bin2nvt()
{
	uchar	*wp;
	uchar	*bp;
	int	extra;

	for(wp = buf, extra = 0; wp < &buf[got]; wp++)
		if(*wp == '\r' || *wp == '\n')
			extra++;
	bp = &buf[got];
	got += extra;
	wp = &buf[got];
	while(bp > buf) {
		if(*--bp == '\r') {
			*--wp = '\0';
			*--wp = '\r';
		} else if(*bp == '\n') {
			*--wp = '\n';
			*--wp = '\r';
		} else
			*--wp = *bp;
	}
}

/*
 * do NVT mapping
 * \r\0 -> \r
 * \r\n -> \n
 * \r*  -> *
 */
void
nvt2bin()
{
	uchar	*bp = buf;
	uchar	*wp = buf;

	while(bp < &buf[got]) {
		if(rstate) {
			if(*bp == '\0')
				*wp++ = '\r';
			else if(*bp == '\n')
				*wp++ = '\n';
			else
				*wp++ = *bp;
			rstate = 0;
		} else if(*bp == '\r')
			rstate = 1;
		else
			*wp++ = *bp;
		bp++;
	}
	got = wp - buf;
}

/*
 * insert IACs after IACs
 * \377 -> \377\377
 */
void
insiacs()
{
	uchar	*wp;
	uchar	*bp;
	int	extra;

	for(wp = buf, extra = 0; wp < &buf[got]; wp++)
		if(*wp == IAC)
			extra++;
	bp = &buf[got];
	got += extra;
	wp = &buf[got];
	while(bp > buf) {
		if(*--bp == IAC)
			*--wp = IAC;
		*--wp = *bp;
	}
}

/*
 * \377\377 -> \377
 */
void
deliacs()
{
	uchar	*bp = buf;
	uchar	*wp = buf;

	while(bp < &buf[got]) {
		if(iacstate) {
			if(*bp != IAC)
				*wp++ = *bp;
			iacstate = 0;
		} else if(*bp == IAC) {
			*wp++ = *bp;
			iacstate = 1;
		} else
			*wp++ = *bp;
		bp++;
	}
	got = wp - buf;
}



volatile void
panic(char *fmt, ...)
{
	va_list	ap;

	fprintf(stderr, "Panic: ");
	VA_START(ap, fmt);
	vfprintf(stderr, fmt, ap);
	va_end(ap);
	fprintf(stderr, "\n");
	Exit(27);
}
