/* p11 - pdp11 emulator; Copyright (C) 1994 Hartmut Brandt, Joerg Micheel 
 * see the file LICENSE for further information */

/*
 * processor trap, interrupt and DMA handling
 */

# include "proc.h"

# define CHECK_TRAP
# define D(C)

void Push(ushort);

volatile void
Trap10(void)
{
	Trap(010);
}

volatile void
Trap4(ushort err)
{
	proc.cpuerr |= err;
	Trap(4);
}

volatile void
Trap(ushort vec)
{
	SetupTrap(vec);
	longjmp(trap, 1);
}

void
SetupTrap(ushort vec)
{
	ushort psw, pc, prev, oldpsw, prevmode;

# ifdef CHECK_TRAP
	static int intrap = 0;
	if(intrap++) {
		if(proc.maint & 1)	/* power ok */
			printf("cyclic trap (%ho)\n", vec);
		monitor();
	}
# endif
	D(("Trap to %ho  ", vec));

	vec &= 0774;

again:
	/*
	 * fetch new ps and pc from kernel D-space
	 */
	psw = mmfetch(vec+2, 1, 0);
	pc  = mmfetch(vec,   1, 0);

	/*
	 * get old psw and implicite switch to new stack
	 */
	oldpsw = iofetch(017777776);
	prevmode = proc.curmode;
	iostore(017777776, M_Word, psw);
	proc.prevmode = prevmode;

	/*
	 * now push pc and psw, switch to new pc
	 */
	if(proc.curmode == 0 && proc.reg[6] < 0404 && proc.reg[6] != 4) {
		/*
	  	 * red stack violation
		 */
		proc.reg[6] = 4;
		proc.cpuerr |= 04;
		vec = 04;
		goto again;
	}

	D(("PC=%ho PSW=%ho -> PC=%ho PSW=%ho\n", proc.reg[7], oldpsw, pc, psw));
	Push(oldpsw);
	Push(proc.reg[7]);
	proc.reg[7] = pc;
# ifdef CHECK_TRAP
	intrap = 0;
# endif
}

/*
 * switch SP and/or general register set
 */
void
switchmode(int mode, int regs)
{
	if(regs && !proc.regs) {
		memcpy(proc.ureg, proc.reg, sizeof(proc.ureg));
		memcpy(proc.reg, proc.kreg, sizeof(proc.kreg));
		D(("switch to alternate registers\n"));
	} else if(!regs && proc.regs) {
		memcpy(proc.kreg, proc.reg, sizeof(proc.kreg));
		memcpy(proc.reg, proc.ureg, sizeof(proc.ureg));
		D(("switch to normal registers\n"));
	}
	proc.sp[proc.curmode] = proc.reg[6];
	proc.reg[6] = proc.sp[mode];
	D(("switchmode %d -> %d at %ho\n", proc.curmode, mode, proc.reg[7]));
}


/*
 * Process DMA and/or interrupt requests
 */
void
request(void)
{
	IODev *d, *f;
	int max, pri;
	ushort req;
	sigset_t oldmask, mask;

	sigemptyset(&mask);
	sigaddset(&mask, SIGIO);
	sigaddset(&mask, SIGALRM);
	sigprocmask(SIG_BLOCK, &mask, &oldmask);

again:
	/*
	 * find maximum request, nearest to processor
	 */
	for(d = proc.devices, max = 0, f = 0; d; d = d->next)
		if(d->reqpri > max) {
			f = d;
			max = d->reqpri;
		}
	if(max <= proc.pri) {
		proc.reqpri = max;
		sigprocmask(SIG_SETMASK, &oldmask, 0);
		return;
	}

	/*
	 * call dma routine or set up vectored interrupt
	 * routine should clear request
	 */
	if(max == DMAPRI)
		(*f->ioops->dma)(f);
	else
		SetupTrap((*f->ioops->vector)(f));

	/*
	 * recompute bus request word
	 */ 
	for(d = proc.devices, max = 0; d; d = d->next)
		if(d->reqpri > max)
			max = d->reqpri;
	proc.reqpri = max;
	
	goto again;
}

void
Push(ushort v)
{
	if((proc.reg[6] -= 2) & 1)
		Trap4(0100);

	mstore(proc.reg[6], M_Word, 1, v);
}
