/*
 * interact.c
 *
 * Routines necessary to provide our simple little interactive
 * interface.  "Shell" is the only entry point for these modules,
 * and its main action is to set up the extraction list (via
 * add_delete -- cd and ls are mainly for the benefit of humans).
 */

#include <stdio.h>
#include <strings.h>
#include <signal.h>
#include <setjmp.h>
#include <sys/types.h>
#include "rstr.h"

#define	LS	1
#define	CD	2
#define	QUIT	3
#define	ADD	4
#define	EXTRACT	5
#define	DELETE	6
#define	HELP	7
#define	NOTHING	8
#define	PWD	9
#define	BANG	10
#define UNKNOWN	99

/* a few global references */

extern uid_t	uid;

jmp_buf	env;

shell()
{
	char		buf[BUFSIZ], *s;
	extern char	*dirname();
	ushort		cur_dir = ROOTDIR;
	int		(*oldhandler)(),
			on_sigint();

	oldhandler = signal(SIGINT, on_sigint);

	(void) setjmp(env);
	printf("\n");

	for ( ;; ) {
		printf("%s", PROMPT);
		(void) gets(s = buf);

		switch (cmdtype(buf)) {
		case LS:
			s += 2;
			while (*s && (*s == ' ' || *s == '\t'))
				s++;
			do_ls(cur_dir, s);
			break;
		case CD:
			s += 2;
			while (*s && (*s == ' ' || *s == '\t'))
				s++;
			cur_dir = do_cd(cur_dir, s);
			break;
		case ADD:
			s += 3;
			while (*s && (*s == ' ' || *s == '\t'))
				s++;
			add_delete(cur_dir, s, 1);
			break;
		case DELETE:
			s += 3;
			while (*s && (*s == ' ' || *s == '\t'))
				s++;
			add_delete(cur_dir, s, 0);
			break;
		case PWD:
			printf("%s\n", dirname(cur_dir));
			break;
		case EXTRACT:
			(void) signal(SIGINT, oldhandler);
			return;
		case HELP:
			do_help();
			break;
		case BANG:
			if (!uid)
				(void) system(s+1);
			else
				printf("Shell escape disallowed.\n");
			break;
		case QUIT:
			clean_up(0);	/* doesn't return */
			break;
		case NOTHING:
			break;
		default:
			printf("\"%s\" unrecognized\n", buf);
			break;
		}
	}
}

add_delete(dir, s, flag)
ushort	dir;
char	*s;
{
	ushort	inode, nami();

	if ((inode = nami(dir, s, flag)) == 0) {
		printf("%s: not found\n", s);
		return;
	}
	if (flag)
		mark(inode);
	else
		clear(inode);
}

cmdtype(s)
char	*s;
{
	if (!*s)
		return NOTHING;
	if (!strncmp(s, "ls", 2))
		return LS;
	if (!strncmp(s, "cd", 2))
		return CD;
	if (!strncmp(s, "add", 3))
		return ADD;
	if (!strncmp(s, "del", 3))
		return DELETE;
	if (!strncmp(s, "pwd", 3))
		return PWD;
	if (!strncmp(s, "help", 4) || *s == '?')
		return HELP;
	if (!strncmp(s, "ex", 2) || !strncmp(s, "extract", 7))
		return EXTRACT;
	if (*s == '!')
		return BANG;
	if (!strncmp(s, "quit", 4))
		return QUIT;
	return UNKNOWN;
}

do_cd(cur, s)
ushort	cur;
char	*s;
{
	ushort	dir, nami();

	if (!*s)
		dir = ROOTDIR;
	else if ((dir = nami(cur, s, 0)) == 0)
		dir = cur;
	
	return dir;
}

do_help()
{
	extern char	*progname;

	printf("\n%s: interactive commands:\n\n", progname);
	printf("ls (-i): List contents of current directory ");
	printf("(objects followed\n");
	printf("\tby '/' are directories, objects preceded by '*' are\n");
	printf("\tmarked for extraction).\n");
	printf("cd:\tChange current directory to specified argument.\n");
	printf("\tWith no argument, change to root directory.\n");
	printf("quit:\tQuit immediately (perform no extractions)\n");
	printf("add:\tAdd the specified file or directory to extraction\n");
	printf("\tlist.  Adding a directory to the extraction list\n");
	printf("\timplies a recursive extraction of all files and\n");
	printf("\tdirectories under that directory.\n");
	printf("del:\tDelete the specified file or directory from the\n");
	printf("\textraction list.\n");
	printf("extract (ex): Extract all marked objects.\n");
	printf("help (?): Print this list.\n");
	printf("pwd:\tPrint the current directory.\n");
	if (!uid) {
		printf("!:\tExecute the command after the '!' as a Unix\n");
		printf("\tshell command.\n");
	}
	printf("\n");
}

do_ls(cur, s)
ushort	cur;
char	*s;
{
	struct dinode	di;
	struct dir	*d, *dbase, *get_contents();
	extern int	dir_compar();
	int		nitems,
			inode_also = 0,
			items_per_line = 4,
			i, j = 0;
	ushort		dir, nami();
	char		fn[15];

	if (!strncmp(s, "-i", 2)) {
		inode_also = 1;
		items_per_line = 3;
		s += 2;
		while (*s && *s == ' ')
			s++;
	}
	if (!*s)
		dir = cur;
	else {
		if ((dir = nami(cur, s, 0)) == 0)
		return;
	}

	d = dbase = get_contents(dir, &di, &nitems);
	qsort((char *)d, nitems, sizeof(struct dir), dir_compar);

	printf("%d files:\n", nitems);
	for (i = 0; i < nitems; i++,d++) {
		fn[14] = '\0';
		(void) strncpy(fn, d->d_name, 14);
		if (ino_type(d->d_inode) == DIRECT)
			(void) strcat(fn, "/");
		if (inode_also)
			printf("%5d %c%-18s", d->d_inode,
				marked(d->d_inode) ? '*' : ' ', fn);
		else
			printf("%c%-18s",
				marked(d->d_inode) ? '*' : ' ', fn);
		if (++j == items_per_line) {
			printf("\n");
			j = 0;
		}
	}
	if (j)
		printf("\n");
	free((char *)dbase);
}

on_sigint()
{
	longjmp(env, 0);
}
