/*
 *	@(#)process.c	1.1 14/12/13
 *
 *	process(), checksum(), control_packet(), send_end_packet()
 *	tape_boot(), tape_read(), tape_write(), tape_position()
 *
 *	Main tape and host handling routines.
 *
 */

#include <stdio.h>
#include <unistd.h>
#include <errno.h>
#include <string.h>
#include <malloc.h>
#include <sys/types.h>
#include <sys/param.h>
#include <sys/poll.h>
#include "globals.h"
#include "tu58.h"

#define POLL_FDS		1
#define POLL_TIMEOUT	1000

int tu_state = LINK_INIT;

static struct pollfd fdset[POLL_FDS];
static int last_fd;


/*
 *	Calculate the checksum of a packet.
 */
static int checksum(unsigned char *p, size_t n)
{
	int c, i;

	log_msg(20, "checksum(0x%8.8x, %d)", p, n);

	if (n & 1) {
		log_msg(0, "checksum: error - odd number of bytes");
		return (ERROR);
	}

	for (c = 0, i = 0; i < n; i+=2) {
		c += p[i] | (p[i+1] << 8);
		c -= (c > 0xffff ? 0xffff : 0);
	}

	log_msg(20, "checksum: 0x%4.4x", c);

	return (c);
}


/*
 *	Assemble and send an end packet.
 */
static void send_end_packet(int fd, int unit, int count, int code, int status)
{
	int c;
	ctrl_packet cp;

	log_msg(20, "send_end_packet(%d, %d, %d, %d, %d)", fd, unit, count,
		code, status);

	cp.flag = TU_F_CTRL;
	cp.count = TU_CMD_SIZE;
	cp.op_code = TU_OP_END;
	cp.modifier = code;
	cp.unit = unit;
	cp.switches = 0;
	cp.seq_num_low = 0;
	cp.seq_num_hi = 0;
	cp.byte_count_low = count & 0xff;
	cp.byte_count_hi = count >> 8;
	cp.block_num_low = 0;
	cp.block_num_hi = status;
	c = checksum((unsigned char *)&cp, 2 + TU_CMD_SIZE); 
	cp.checksum_low = c & 0xff;
	cp.checksum_hi = (c >> 8) & 0xff;

	write_serial(fd, (unsigned char *)&cp, sizeof(cp));

	return;
}


/*
 *	Quick & dirty boot from tape,
 */
static void tape_boot(int fd, FILE *tape[2], int unit)
{
	int n;
	unsigned char buffer[TU_BLK_SIZE];

	log_msg(10, "tape_boot(%d, 0x%8.8lx, %d)", fd, tape, unit);

	/* make sure unit is valid */
	if (unit != 0 && unit != 1) {
		log_msg(0, "tape_boot: bad unit %d.", unit);
		tu_state = LINK_INIT;	// protocol error, reset the link.

	/* see if tape mounted */
	} else if (tape[unit] == (FILE *)0) {
		log_msg(0, "tape_boot: no tape mounted on unit %d.", unit);

	/* seek to first block */
	} else if (fseek(tape[unit], 0, SEEK_SET) == ERROR) {
		log_msg(0, "tape_boot: fseek() -- %s", strerror(errno));

	/* read first block */
	} else if ((n = fread(buffer, sizeof(unsigned char), sizeof(buffer),
		tape[unit])) == ERROR) {
		log_msg(0, "tape_boot: fread() -- %s", strerror(errno));

	/* make sure entire block was read */
	} else if (n != TU_BLK_SIZE) {
		log_msg(0, "tape_boot: read %d/%d bytes", n, TU_BLK_SIZE);

	/* transmit block to the host */
	} else {
		log_msg(10, "tape_boot: booting from unit %d", unit);
		log_msg(0, "BOOTING FROM UNIT %d", unit);
		write_serial(fd, buffer, sizeof(buffer));
	}

	return;
}


/*
 *	Read from tape.
 */
static int tape_read(int fd, FILE *tape[2], int unit, int block, int *count)
{
	int bytes, c, i, n, result;
	data_packet dp;

	log_msg(10, "tape_read(%d, 0x%8.8x, %d, %d, %d)", fd, tape, unit, block,
		*count);

	result = TU_E_SUCCESS;
	dp.flag = TU_F_DATA;

	/*
	 *	If a tape is mounted...
	 */
	if (tape[unit]) {
		/*
		 *	Position tape at requested block.
		 */
		if (fseek(tape[unit], block * TU_BLK_SIZE, SEEK_SET) == ERROR) {
			log_msg(0, "tape_read: fseek() -- %s", strerror(errno));
			result = TU_E_ESEEK;;
		}
	} else {
		log_msg(0, "tape_read: tape not mounted on unit %d", unit);
		result = TU_E_ECART;
	}
	
	/*
	 *	Start reading from tape.
	 */
	for (n = 0, i = 0; i < *count && result == TU_E_SUCCESS; i+=TU_BUF_SIZE) {

		/* we're limited by the buffer size */
		bytes = min(*count - i, TU_BUF_SIZE);

		/*
		 *	Read from tape into buffer. We probably need to be
		 *	keeping track of the block number to make sure
		 *	it's still in range.
		 */
		if ((n = fread(dp.data, sizeof(unsigned char), bytes, tape[unit]))
			== ERROR) {
			log_msg(0, "tape_read: fread() -- %s", strerror(errno));
			result = TU_E_ESTOP;
			break;
		}

		/*
		 *	Fill in the rest of the data packet fields.
		 */
		dp.count = n;
		c = checksum(&dp.flag, n + 2);
		dp.checksum_low = c & 0xff;
		dp.checksum_hi = (c >> 8) & 0xff;

		/*
		 *	Transmit the data packet to the host.
		 */
		if (n == TU_BUF_SIZE)
			write_serial(fd, (unsigned char *)&dp, sizeof(dp));
		else {
			write_serial(fd, (unsigned char *)&dp, n + 2);
			write_serial(fd, (unsigned char *)&dp.checksum_low, 2);
		}

		/*
		 *	Check if we've reached the end of the medium.
		 */
		if (n != bytes) {
			log_msg(0, "tape_read: end of medium (%d/%d).", n, bytes);
			result = TU_E_EPART;
			break;
		}
	}

	/*
	 *	Save the number of bytes transferred..
	 */
	*count = i;

	/*
	 *	Return the result.
	 */
	return (result);
}


/*
 *	Write to tape.
 */
static int tape_write(int fd, FILE *tape[2], int unit, int block,
	int *count, char wp[2])
{
	int b, c, i, n, result;
	unsigned char flag;
	unsigned char buffer[TU_BLK_SIZE];
	data_packet dp;

	log_msg(10, "tape_write(%d, 0x%8.8lx, %d, %d, %d, 0x%8.8lx)",
		fd, tape, unit, block, *count, wp);

	result = TU_E_SUCCESS;
	flag = TU_F_CONT;
	i = 0;

	/*
	 *	Only if we're not write protected...
	 */
	if (!wp[unit]) {

		/*
		 *	...and a tape is mounted...
		 */
		if (tape[unit]) {
			/*
			 *	Position tape at requested block.
			 */
			if (fseek(tape[unit], block * TU_BLK_SIZE, SEEK_SET) == ERROR) {
				log_msg(0, "tape_write: fseek() -- %s", strerror(errno));
				result = TU_E_ESEEK;
			}
		} else {
			log_msg(0, "tape_write: tape not mounted on unit %d", unit);
			result = TU_E_ECART;
		}

		for (b = 0, i = 0; i < *count && result == TU_E_SUCCESS; i+= TU_BUF_SIZE) {
			/*
			 *	Check to see if the block is still valid.
			 */
			if (block > 511) {
				result = TU_E_EPART;
				break;
			}

			/*
			 *	Send the continuation flag.
			 */
			log_msg(20, "tape_write: sending continuation");
			write_serial(fd, &flag, 1);

			/*
			 *	Read back the next data packet.
			 */
			read_serial(fd, &dp.flag, sizeof(dp.flag));
			if (dp.flag == TU_F_DATA) {
				read_serial(fd, &dp.count, sizeof(dp.count));
				if (dp.count == 128) {
					read_serial(fd, dp.data, dp.count+2);
				} else if (dp.count <= 128) {
					read_serial(fd, dp.data, dp.count);
					read_serial(fd, &dp.checksum_low, sizeof(dp.checksum_low));
					read_serial(fd, &dp.checksum_hi, sizeof(dp.checksum_hi));
				} else {
					log_msg(0, "tape_write: invalid count %d", dp.count);
					result = TU_E_EDATA;
					break;
				}
			} else {
				log_msg(0, "tape_write: invalid flag 0x%2.2", dp.flag);
				tu_state = LINK_INIT;	// protocol error, reset the link.
				break;
			}

			/*
			 *	Validate the checksum.
			 */
			c = checksum(&dp.flag, dp.count + 2);
			if ((dp.checksum_low != (c & 0xff)) ||
				(dp.checksum_hi != ((c >> 8) & 0xff))) {
				log_msg(0, "tape_write: bad checksum");
				result = TU_E_EDATA;
				break;
			}

			/*
			 *	Copy the data to the block buffer, since we're
			 *	responsible for zeroing out the unused portion
			 *	of the last block.
			 */
			memcpy(&buffer[b], &dp.data, dp.count);
			b += dp.count;
			if (dp.count < TU_BUF_SIZE) {
				if (i < *count) {
					log_msg(0, "tape_write: we might have a problem"
						"i=%d, *count=%d, dp.count=%d, b=%d",
						i, *count, dp.count, b);
					result = TU_E_EDATA;
					break;
				}
				memset(&buffer[b], 0, TU_BUF_SIZE - b);
				b = TU_BUF_SIZE;
			}

			/*
			 *	Write the block to tape.
			 */
			if (b == TU_BUF_SIZE) {
				if ((n = fwrite(dp.data, sizeof(unsigned char), TU_BUF_SIZE,
					tape[unit])) == ERROR) {
					log_msg(0, "tape_write: fwrite() -- %s", strerror(errno));
					result = TU_E_ESTOP;
					break;
				}
				b = 0;
				block++;
			}
		}
	} else {
		result = TU_E_EWP;
		log_msg(0, "tape_write: tape write protected!");
	}

	*count = i;

	/*
	 *	Return the result.
	 */
	return (result);
}


/*
 *	Position tape at the requested block.
 */
static int tape_position(int fd, FILE *tape[2], int unit, int block)
{
	int result;

	log_msg(10, "tape_position(%d, 0x%8.8x, %d, %d)", fd, tape, unit, block);

	result = TU_E_SUCCESS;

	/*
	 *	Position tape at requested block.
	 */
	if (fseek(tape[unit], block * TU_BLK_SIZE, SEEK_SET) == ERROR) {
		log_msg(0, "tape_position: fseek() -- %s", strerror(errno));
		result = TU_E_ESEEK;
	}

	/*
	 *	Return the result.
	 */
	return (result);
}


/*
 *	Process control packets.
 */
static int control_packet(int fd, FILE *tape[2], char wp[2])
{
	int c, cs, count, result;
	ctrl_packet cp;

	result = TU_E_SUCCESS;
	count = 0;

	log_msg(10, "control_packet(%d, 0x%8.8lx, 0x%8.8lx)",
		fd, tape, wp);

	/*
	 * Read the rest of the packet.
	 */
	cp.flag = TU_F_CTRL;
	read_serial(fd, &(cp.count), sizeof(cp)-1);

	/*
	 * Validate the packet.
	 */
	if (cp.count != TU_CMD_SIZE) {
		log_msg(0, "control_packet: message is %d bytes", cp.count);
		result = TU_E_EDATA;
		tu_state = LINK_INIT;	// protocol error, reset the link.
	} else {
		c = CHECKSUM(cp);
		if ((cs = checksum((unsigned char *)&cp.flag, sizeof(cp) - 2)) != c) {
			log_msg(0, "control_packet: bad checksum, calculated 0x%4.4x", cs);
			result = TU_E_EDATA;
		} else if (BLOCK(cp) > 511)
			result = TU_E_EBLOCK;
	}

	/*
	 * If it looks good, then process it.
	 */
	if (result == TU_E_SUCCESS)  {
		log_msg(10, "control_packet: sequence number is %d", SEQ(cp));

		switch (cp.op_code) {
			case TU_OP_NOP:		/* No operation */
				log_msg(5, "control_packet: NOP");
				/* send a success end packet for the response */
				break;
			case TU_OP_INIT:	/* Reset TU58 controller to ready state */
				log_msg(5, "control_packet: INITIALIZE");
				/* send a success end packet for the response */
				break;
			case TU_OP_READ:	/* Read data from a tape */
				switch (cp.modifier) {
					case 0:
						log_msg(5, "control_packet: READ");
						break;
					case 1:
						log_msg(5, "control_packet: READ WITH INCREASED THRESHOLD");
						break;
					default:
						log_msg(0, "control_packet: READ with modifier %d",
							cp.modifier);
						result = TU_E_EBADOP;
						break;
				}
				if (result == TU_E_SUCCESS)  {
					count = COUNT(cp);
					log_msg(1, "READ: UNIT %d, BLOCK %d, BYTES %d",
						cp.unit, BLOCK(cp), count);
					result = tape_read(fd, tape, cp.unit, BLOCK(cp), &count);
				}
				break;
			case TU_OP_WRITE:	/* Write data to a tape. */
				switch (cp.modifier) {
					case 0:
						log_msg(5, "control_packet: WRITE");
						break;
					case 1:
						log_msg(5, "control_packet: WRITE AND READ VERIFY");
						break;
					default:
						log_msg(5, "control_packet: WRITE with modifier %d",
							cp.modifier);
						result = TU_E_EBADOP;
						break;
				}
				if (result == TU_E_SUCCESS)  {
					count = COUNT(cp);
					log_msg(1, "WRITE: UNIT %d, BLOCK %d, BYTES %d",
						cp.unit, BLOCK(cp), count);
					result = tape_write(fd, tape, cp.unit, BLOCK(cp),
						&count, wp);
				}
				break;
			case TU_OP_POS:		/* Position tape a the specified block */
				log_msg(5, "control_packet: POSITION");
				result = tape_position(fd, tape, cp.unit, BLOCK(cp));
				break;
			case TU_OP_DIAG:	/* Run TU58 internal diagnostics */
				log_msg(5, "control_packet: DIAGNOSE");
				/* send a success end packet for the response */
				break;
			case TU_OP_GSTAT:	/* Treated as a NOP */
				log_msg(5, "control_packet: GET STATUS");
				/* send a success end packet for the response */
				break;
			case TU_OP_SSTAT:	/* TU58 status cannot be set from host */
				log_msg(5, "control_packet: SET STATUS");
				/* send a success end packet for the response */
				break;
			default:
				log_msg(0, "control_packet: instruction %d received", cp.op_code);
				result = TU_E_EBADOP;
				break;
		}
	}

	if (tu_state == LINK_INIT) {
		log_msg(5, "control_packet: reinitializing the link");
		cp.flag = TU_F_INIT;
		write_serial(fd, &cp.flag, sizeof(cp.flag));
	} else
		send_end_packet(fd, cp.unit, count, result, TU_E_NOSUM);

	return 0;
}


void process(char *line, speed_t speed, FILE *tapeunit[2], char wp[2])
{
	int i, n, inits;
	unsigned char flag;

	inits = 0;

	log_msg(10, "process(\"%s\", %d, 0x%8.8lx)", line, speed, tapeunit);

	/*
	 *	Initialize the file descriptor set and open the serial line.
	 */
	last_fd = 0;
	for (i = 0; i < POLL_FDS; i++) {
		fdset[i].fd = -1;
		fdset[i].events = 0;
		fdset[i].revents = 0;
	}

	log_msg(5, "process: opening serial line");
	fdset[last_fd].fd = open_serial(line, speed);
	fdset[last_fd].events = POLLIN;
	last_fd++;

	done = FALSE;

	while (!done) {
		
		log_msg(10, "poll(0x%8.8lx, %d, %d)", fdset, last_fd, POLL_TIMEOUT);

		/*
		 *	Wait for something to happen.
		 */
		n = poll(fdset, last_fd, POLL_TIMEOUT);

		log_msg(25, "poll: n=%d", n);
		
		if (n == ERROR) {
			switch (errno) {
				case EINTR:
					log_msg(1, "EINTR");
					continue;
				default:
					log_msg(0, "%s: poll -- %s", program, strerror(errno));
					return;
			}
		}

		/*
		 *	Service any events.
		 */
		if (n > 0) {

			/*
			 *	Handle errors first.
			 */
			if (fdset[0].revents & POLLERR) {
				log_msg(0, "%s: poll -- %s", program, strerror(errno));
				continue;
			} else if (fdset[0].revents & POLLHUP) {
				log_msg(0, "%s: POLLHUP");
				done = TRUE;
			} else if (fdset[0].revents & POLLNVAL) {
				log_msg(0, "%s: POLLNVAL");
				done = TRUE;
				continue;
			}
			
			/*
			 * Service the serial port.
			 */
			if (fdset[0].revents & POLLIN) {

				log_msg(10, "process: data received");
				read_serial(fdset[0].fd, &flag, sizeof(flag));

				switch (tu_state) {
					case LINK_INIT:
						log_msg(5, "process: LINK_INIT");

						switch (flag) {
							case TU_F_BREAK:	// probably received break
								log_msg(5, "process: break received");
								tu_state = LINK_START;
								log_msg(1, "LINK START");
								inits = 0;
								break;
							case TU_F_INIT:
								flag = TU_F_CONT;
								write_serial(fdset[0].fd, &flag, sizeof(flag));
								tu_state = LINK_IDLE;
								log_msg(1, "LINK IDLE");
							default:
								log_msg(5, "process: flag=%d", flag);
								break;
						}
						break;
					case LINK_START:
						log_msg(5, "process: LINK_START");

						switch (flag) {
							case TU_F_BREAK:
								log_msg(5, "process: break received");
								break;
							case TU_F_INIT: // host sends 2 INITs
								inits++;
								if (inits >= 2) {
									log_msg(5, "process: sending continue");
									flag = TU_F_CONT;
									write_serial(fdset[0].fd, &flag, sizeof(flag));
									tu_state = LINK_IDLE;
									log_msg(5, "process: moving to LINK_IDLE");
								}
								break;
							case TU_F_BOOT:
								if (inits) {
									tu_state = LINK_BOOT;
									log_msg(5, "process: boot request");
									log_msg(1, "LINK BOOT");
								}
								break;
							default:
								log_msg(5, "process: flag=%d", flag);
								tu_state = LINK_INIT;
								log_msg(1, "LINK INIT");
								break;
						}
						break;
					case LINK_IDLE:
						log_msg(5, "process: LINK_IDLE");

						switch (flag) {
							case TU_F_BREAK:
								log_msg(5, "process: break received");
								tu_state = LINK_START;
								inits = 0;
								log_msg(5, "process: moving to LINK_START");
								break;
							case TU_F_INIT:
								log_msg(5, "process: INIT received");
								log_msg(5, "process: sending CONTINUE");
								flag = TU_F_CONT;
								write_serial(fdset[0].fd, &flag, sizeof(flag));
								break;
							case TU_F_CTRL:
								log_msg(5, "process: CONTROL received");
								control_packet(fdset[0].fd, tapeunit, wp);
								break;
							default:
								log_msg(0, "process: flag %2.2x received", flag);
								break;
						}
						break;
					case LINK_BOOT:
						log_msg(5, "process: LINK_BOOT");

						tape_boot(fdset[0].fd, tapeunit, flag);
						tu_state = LINK_IDLE;
						log_msg(1, "LINK IDLE");
					default:
						break;
				}
			}
		} else {

			/*
			 *	poll() timed out
			 */
			switch (tu_state) {
				case LINK_INIT:
					log_msg(5, "process: sending INIT");

					flag = TU_F_INIT;
					write_serial(fdset[0].fd, &flag, sizeof(flag));
					break;
				default:
					break;
			}
		}

	} // while


	/*
	 *	Shutdown
	 */
	log_msg(5, "process: shutting down");
	for (i = 0; i < last_fd; i++)
		if (fdset[i].fd != ERROR && close(fdset[i].fd) != 0)
			log_msg(0, "process: close(%d) -- %s", fdset[i].fd, strerror(errno), i);
	
	return;
}
