/* s3_tape.c: IBM 3411 Tape Units

   Copyright (c) 2006 Henk Stegeman
   Copyright (c) 2001 Charles E. Owen
   Copyright (c) 1993-2001, Robert M. Supnik

   Permission is hereby granted, free of charge, to any person obtaining a
   copy of this software and associated documentation files (the "Software"),
   to deal in the Software without restriction, including without limitation
   the rights to use, copy, modify, merge, publish, distribute, sublicense,
   and/or sell copies of the Software, and to permit persons to whom the
   Software is furnished to do so, subject to the following conditions:

   The above copyright notice and this permission notice shall be included in
   all copies or substantial portions of the Software.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
   ROBERT M SUPNIK BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
   IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
   CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

   Except as contained in this notice, the name of Robert M Supnik shall not be
   used in advertising or otherwise to promote the sale, use or other dealings
   in this Software without prior written authorization from Robert M Supnik.

   Adaptor --+--> Unit #0 (3411) t0
             +--> Unit #1 (3410) t1
             +--> Unit #2 (3410) t2
             +--> Unit #3 (3410) t3

*/

#include "s3_defs.h"
#include <ctype.h>

extern int32 IAR[], level, dev_int_req;
extern FILE *trace;
extern int32 debug_lvl;
extern int32 GetMem (int32 addr, int8 cycle);
extern int32 PutMem (int32 addr, int8 cycle, int32 data);
int32 read_block(UNIT *uptr, char *tbuf, int32 mod);
int32 write_block(UNIT *uptr, char *tbuf, int32 mod);
int32 tap (int32 tape, int32 op, int32 m, int32 n, int32 data);

t_stat t0_svc (UNIT *uptr);
t_stat t0_boot (int32 unitno);
t_stat t0_attach (UNIT *uptr, char *cptr);
t_stat t0_reset (DEVICE *dptr);
t_stat t1_svc (UNIT *uptr);
t_stat t1_boot (int32 unitno);
t_stat t1_attach (UNIT *uptr, char *cptr);
t_stat t1_reset (DEVICE *dptr);
t_stat t2_svc (UNIT *uptr);
t_stat t2_boot (int32 unitno);
t_stat t2_attach (UNIT *uptr, char *cptr);
t_stat t2_reset (DEVICE *dptr);
t_stat t3_svc (UNIT *uptr);
t_stat t3_boot (int32 unitno);
t_stat t3_attach (UNIT *uptr, char *cptr);
t_stat t3_reset (DEVICE *dptr);

int32 MTAR;                       /* Data Address Register */
int32 MBCR;                       /* Byte Count Register */
int32 t_nrdy[4] = { 0, 0, 0, 0 };   /* Not ready error */
int32 ss_busy = 0;                  /* Subsystem busy flag */
int32 blk_length = 0;               /* Block length (integer) */
unsigned char lcb[2] = { 0 };       /* Length current tape block. */
unsigned char lpb[2] = { 0 };       /* Length previous tape block. */
unsigned char tflags[2] = { 0 };    /* Tape flags */
char tbuf[MAX_TAPESIZE];        /* Tape buffer */

/* Disk data structures

   tx_dev   CDR descriptor
   tx_unit  CDR unit descriptor
            u3 is used for sense bytes 0 & 1.
            u4 is not used.
   tx_reg   CDR register list

   x = 0, 1, 2 or 3
*/

UNIT t0_unit = {
   UDATA (&t0_svc, UNIT_SEQ+UNIT_ATTABLE, 0), 100 };

REG t0_reg[] = {
   { FLDATA (NOTRDY, t_nrdy[0], 0) },
   { HRDATA (TAR, MTAR, 16) },
   { HRDATA (BCR, MBCR, 16) },
   { HRDATA (ERR, t0_unit.u3, 16) },
   { DRDATA (POS, t0_unit.pos, 31), PV_LEFT },
   { DRDATA (TIME, t0_unit.wait, 24), PV_LEFT },
   { BRDATA (BUF, tbuf, 8, 8, MAX_TAPESIZE) },
   { NULL } };

DEVICE t0_dev = {
   "T1", &t0_unit, t0_reg, NULL,
   1, 10, 31, 1, 8, 7,
   NULL, NULL, &t0_reset,
   &t0_boot, &t0_attach, NULL };

UNIT t1_unit = {
   UDATA (&t1_svc, UNIT_SEQ+UNIT_ATTABLE, 0), 500 };

REG t1_reg[] = {
   { FLDATA (NOTRDY, t_nrdy[1], 0) },
   { HRDATA (TAR, MTAR, 16) },
   { HRDATA (BCR, MBCR, 16) },
   { HRDATA (ERR, t1_unit.u3, 16) },
   { DRDATA (POS, t1_unit.pos, 31), PV_LEFT },
   { DRDATA (TIME, t1_unit.wait, 24), PV_LEFT },
   { BRDATA (BUF, tbuf, 8, 8, MAX_TAPESIZE) },
   { NULL } };

DEVICE t1_dev = {
   "T2", &t1_unit, t1_reg, NULL,
   1, 10, 31, 1, 8, 7,
   NULL, NULL, &t1_reset,
   &t1_boot, &t1_attach, NULL };

UNIT t2_unit = {
   UDATA (&t2_svc, UNIT_SEQ+UNIT_ATTABLE, 0), 100 };

REG t2_reg[] = {
   { FLDATA (NOTRDY, t_nrdy[2], 0) },
   { HRDATA (TAR, MTAR, 16) },
   { HRDATA (BCR, MBCR, 16) },
   { HRDATA (ERR, t2_unit.u3, 16) },
   { DRDATA (POS, t2_unit.pos, 31), PV_LEFT },
   { DRDATA (TIME, t2_unit.wait, 24), PV_LEFT },
   { BRDATA (BUF, tbuf, 8, 8, 256) },
   { NULL } };

DEVICE t2_dev = {
   "T3", &t2_unit, t2_reg, NULL,
   1, 10, 31, 1, 8, 7,
   NULL, NULL, &t2_reset,
   &t2_boot, &t2_attach, NULL };

UNIT t3_unit = {
   UDATA (&t3_svc, UNIT_SEQ+UNIT_ATTABLE, 0), 100 };

REG t3_reg[] = {
   { FLDATA (NOTRDY, t_nrdy[3], 0) },
   { HRDATA (TAR, MTAR, 16) },
   { HRDATA (BCR, MBCR, 16) },
   { HRDATA (ERR, t3_unit.u3, 16) },
   { DRDATA (POS, t3_unit.pos, 31), PV_LEFT },
   { DRDATA (TIME, t3_unit.wait, 24), PV_LEFT },
   { BRDATA (BUF, tbuf, 8, 8, 256) },
   { NULL } };

DEVICE t3_dev = {
   "T4", &t3_unit, t3_reg, NULL,
   1, 10, 31, 1, 8, 7,
   NULL, NULL, &t3_reset,
   &t3_boot, &t3_attach, NULL };


/* -------------------------------------------------------------------- */

/* 3411: master routines */

int32 tap1 (int32 op, int32 m, int32 n, int32 data)
{
   int32 r;
   if (m == 0)
      r = tap(0, op, m, n, data);
   else
      r = tap(1, op, m, n, data);
   return (r);
}

int32 tap2 (int32 op, int32 m, int32 n, int32 data)
{
   int32 r;
   if (m == 0)
      r = tap(2, op, m, n, data);
   else
      r = tap(3, op, m, n, data);
   return (r);
}

/* 3411: operational routine */

int32 tap (int32 tape, int32 op, int32 m, int32 n, int32 data)
{
   int32 iodata, i;
   char opstr[5][5] = { "SIO", "LIO", "TIO", "SNS", "APL" };
   UNIT *uptr = 0;

   switch (tape) {
      case 0:
         uptr = t0_dev.units;
         break;
      case 1:
         uptr = t1_dev.units;
         break;
      case 2:
         uptr = t2_dev.units;
         break;
      case 3:
         uptr = t3_dev.units;
         break;
      default:
         break;
   }

   if (debug_lvl & 0x04)
      fprintf(trace, "=T=> %04X %s %01X,%d,%04X MTAR=%04X MBCR=%04X \n",
         IAR[level],
         opstr[op],
         m, n, data,
         MTAR,
         MBCR);

   switch (op) {      
      case 0:     /* SIO 3411 */
         if ((uptr -> flags & UNIT_ATT) == 0)
            return SCPE_UNATT;
         uptr -> u3 = 0x0000; /* SIO resets all errors */
         iodata = 0;
         switch (n) {
            case 0x00:   /* Control */
               switch (data) {
                  case 0x07:   /* rewind */
                  case 0x0F:   /* rewind with unload */
                     fseek(uptr -> fileref, 0, 0);
                     uptr -> pos = 0;
                     sim_activate(uptr, uptr -> wait);
                     ss_busy = ON;
                     if (data == 0x0F)
                        detach_unit(uptr);   /* unload tape */
                     break;
                  case 0x1F:   /* write tape mark */
                     write_block(uptr, tbuf, 1);
                     sim_activate(uptr, uptr -> wait);
                     ss_busy = ON;
                     break;
                  case 0xC3:   /* Mode 2 set (9 track) */
                     break;
                  default:
                     return(STOP_INVDEV);
               }
               MBCR = 0;
               iodata = SCPE_OK;
               break;
            case 0x01:   /* Read Forward */
               read_block(uptr, tbuf, 0);
printf(" lcb = %02X%02X \n\r", lcb[1], lcb[0]);
printf(" lpb = %02X%02X \n\r", lpb[1], lpb[0]);
printf(" bl  = %08X \n\r", blk_length    );
printf(" flags = %02X%02X \n\r", tflags[0], tflags[1] );
printf(" block id = %02X%02X\n\r" , tbuf[0] & 0xFF, tbuf[1] & 0xFF);
               if (blk_length > MBCR)
                  blk_length = MBCR;
               if (blk_length != MBCR)
                  uptr -> u3 |= 0x4000;    /* wrong length record */
               for (i = 0; i < blk_length; i++) {
                  PutMem(MTAR, 0, tbuf[i]);
                  MTAR++;
               }
               sim_activate(uptr, uptr -> wait);
               ss_busy = ON;           /* subsystem is busy */
               MBCR = 0;               /* reset byte count */
               iodata = SCPE_OK;
               break;
            case 0x02:   /* Write */
               for (i = 0; i < MBCR; i++) {
                  tbuf[i] = GetMem(MTAR, 0);
                  MTAR++;
               }
               write_block(uptr, tbuf, 0);
               sim_activate(uptr, uptr -> wait);
               ss_busy = ON;
               iodata = SCPE_OK;
               break;
            case 0x03:   /* Read Backward */
               read_block(uptr, tbuf, 1);
            break;
            default:
               return STOP_INVDEV;
         }
         return iodata;

      case 1:    /* LIO 3411 */
         switch (n) {
            case 0x00:   /* Byte Count Register */
               MBCR = data;
               break;
            case 0x04:   /* Data Address Register */
               MTAR = data;
               break;
            case 0x06:   /* Interrupt Control Register */
               break;
            default:
               return STOP_INVDEV;
         }
         return SCPE_OK;

      case 2:     /* TIO 3411 */
      case 4:     /* APL 3411 */
         iodata = 0;
         switch (n) {
            case 0x00:   /* Not ready/check ? */
               if (uptr -> u3 != 0x0000)
                  iodata = TRUE;
               if ((uptr -> flags & UNIT_ATT) == 0)
                  iodata = TRUE;
               break;
            case 0x01:   /* Op-end pending ? */
               break;
            case 0x02:   /* Subsystem busy ? */
               if (ss_busy == ON)
                  iodata = TRUE;
               break;
            default:
               return (STOP_INVDEV << 16);
         }
         return ((SCPE_OK << 16) | iodata);

      case 3:     /* SNS 3411 */
//         if ((uptr -> flags & UNIT_ATT) == 0)
//            return SCPE_UNATT << 16;
         iodata = 0x0000;
         switch (n) {
            case 0x00:   /* subsystem bytes 0 & 1. */
               iodata = uptr -> u3;
               if (uptr -> pos == 0)
                  iodata |= 0x0020;   /* at load point */
               iodata |= 0x0100;      /* sense valid */
               break;
            case 0x01:   /* subsystem bytes 2 & 3. */
               break;
            case 0x02:   /* subsystem bytes 4 & 5. */
               break;
            case 0x03:   /* subsystem bytes 6 & 7. */
               break;
            case 0x04:   /* data address register. */
               iodata = MTAR;
               break;
            case 0x05:   /* attachment sense bytes. */
               break;
            case 0x06:   /* subsystem hw error sense. */
               break;
            case 0x07:   /* op-end sense byte */
               break;
            default:
               return (STOP_INVDEV << 16);
         }
         iodata |= ((SCPE_OK << 16) & 0xffff0000);
         return (iodata);

      default:
         break;
   }
   return SCPE_OK;
}


/*** Tape unit service. ***/

t_stat t0_svc (UNIT *uptr)
{
   ss_busy = OFF;
   return SCPE_OK;
}

t_stat t1_svc (UNIT *uptr)
{
   ss_busy = OFF;
   return SCPE_OK;
}

t_stat t2_svc (UNIT *uptr)
{
   ss_busy = OFF;
   return SCPE_OK;
}

t_stat t3_svc (UNIT *uptr)
{
   ss_busy = OFF;
   return SCPE_OK;
}


/*** Tape reset ***/

t_stat t0_reset (DEVICE *dptr)
{
   t_nrdy[0] = ss_busy = 0;         /* clear indicators */
   sim_cancel (&t0_unit);           /* clear event */
   t0_unit.u3 = 0x0000;             /* clear any errors */
   t0_unit.pos = 0;                 /* tape is at load point */
   return SCPE_OK;
}

t_stat t1_reset (DEVICE *dptr)
{
   t_nrdy[1] = ss_busy = 0;         /* clear indicators */
   sim_cancel (&t1_unit);           /* clear event */
   t1_unit.u3 = 0x0000;             /* clear any errors */
   t1_unit.pos = 0;                 /* tape is at load point */
   return SCPE_OK;
}

t_stat t2_reset (DEVICE *dptr)
{
   t_nrdy[2] = ss_busy = 0;         /* clear indicators */
   sim_cancel (&t2_unit);           /* clear event */
   t2_unit.u3 = 0x0000;             /* clear any errors */
   t2_unit.pos = 0;                 /* tape is at load point */
   return SCPE_OK;
}

t_stat t3_reset (DEVICE *dptr)
{
   t_nrdy[3] = ss_busy = 0;         /* clear indicators */
   sim_cancel (&t3_unit);           /* clear event */
   t3_unit.u3 = 0x0000;             /* clear any errors */
   t3_unit.pos = 0;                 /* tape is at load point */
   return SCPE_OK;
}


/*** Tape unit attach ***/

t_stat t0_attach (UNIT *uptr, char *cptr)
{
   t_nrdy[0] = 0;                   /* clear status */
   ss_busy = OFF;
   uptr -> pos = 0;                 /* tape is at load point */
   uptr -> u3 = 0x0000;             /* clear any error */
   return attach_unit (uptr, cptr);
}

t_stat t1_attach (UNIT *uptr, char *cptr)
{
   t_nrdy[1] = ss_busy = 0;         /* clear status */
   ss_busy = OFF;
   uptr -> pos = 0;                 /* tape is at load point */
   uptr -> u3 = 0x0000;             /* clear any error */
   return attach_unit (uptr, cptr);
}

t_stat t2_attach (UNIT *uptr, char *cptr)
{
   t_nrdy[2] = ss_busy = 0;         /* clear status */
   ss_busy = OFF;
   uptr -> pos = 0;                 /* tape is at load point */
   uptr -> u3 = 0x0000;             /* clear any error */
   return attach_unit (uptr, cptr);
}

t_stat t3_attach (UNIT *uptr, char *cptr)
{
   t_nrdy[3] = ss_busy = 0;         /* clear status */
   ss_busy = OFF;
   uptr -> pos = 0;                 /* tape is at load point */
   uptr -> u3 = 0x0000;             /* clear any error */
   return attach_unit (uptr, cptr);
}


/*** Bootstrap routine, not valid for 3411 ***/

t_stat t0_boot (int32 unitno)
{
   return STOP_INVDEV;
}

t_stat t1_boot (int32 unitno)
{
   return STOP_INVDEV;
}

t_stat t2_boot (int32 unitno)
{
   return STOP_INVDEV;
}

t_stat t3_boot (int32 unitno)
{
   return STOP_INVDEV;
}


/*** Raw Tape Read/Write ***/

int32 read_block(UNIT *uptr, char *tbuf, int32 mod)
{
   static int32 k, rtn;

   if (mod == 0) {   /* Read forward ? */
      /* Read 6 bytes of the AWS tape header */
      k = fread(lcb, sizeof(char), 2, uptr -> fileref);
      if (k != 2) 
         uptr -> u3 |= 0x0200;       /* equipment check */
      k = fread(lpb, sizeof(char), 2, uptr -> fileref);
      if (k != 2) 
         uptr -> u3 |= 0x0200;       /* equipment check */
      k = fread(tflags, sizeof(char), 2, uptr -> fileref);
      if (k != 2) 
         uptr -> u3 |= 0x0200;       /* equipment check */

      /* Read from AWS tape */
      blk_length = ((lcb[1] & 0xFF) << 8) | (lcb[0] & 0xFF);
      k = fread(tbuf, sizeof(char), blk_length, uptr -> fileref);
      if (k != blk_length) 
         uptr -> u3 |= 0x0200;       /* equipment check */
      uptr -> pos = uptr -> pos + k + 6;  /* update tape position. */

      if (tflags[0] == 0x40)
         uptr -> u3 |= 0x2000;       /* tapemark detected */
      return (rtn);
   }
   if (mod == 1) {                   /* Read backwards ? */
      printf("\n\rRead backward !!!\n\r");
   }
return SCPE_OK;
}

int32 write_block(UNIT *uptr, char *tbuf, int32 mod)
{
   static int32 rtn;
   static long pos;

   pos = uptr -> pos;
   if (mod == 0) {                   /* write data ? */
       lcb[0] = MBCR & 0x00FF;       /* set lcb */
       lcb[1] = (MBCR & 0xFF00) >> 8;
       if (pos == 0)
          lpb[0] = lpb[1] = 0x00;    /* if at LP set lpb */
       tflags[0] = 0xA0;             /* set flags */
   }
   if (mod == 1) {                   /* write tape mark ? */
       lcb[0] = lcb[1] = 0x00;       /* set lcb */
       tflags[0] = 0x40;             /* tape mak */
       MBCR = 0x00;
   }

   /* Write tape buffer to AWS tape */
   rtn = fseek(uptr -> fileref, pos, 0);  /* position tape */
   rtn = fwrite(lcb, sizeof(char), 2, uptr -> fileref);
   rtn = fwrite(lpb, sizeof(char), 2, uptr -> fileref);
   rtn = fwrite(tflags, sizeof(char), 2, uptr -> fileref);
   rtn = fwrite(tbuf, sizeof(char), MBCR, uptr -> fileref);
printf("\n\r Writing block %02X%02X...", tbuf[0] & 0xFF, tbuf[1] & 0xFF);
   uptr -> pos += MBCR + 6;    /* update tape position */
   lpb[0] = lcb[0]; lpb[1] = lcb[1];      /* for next write block */
   return (rtn);
}

