// Runstream (R) MicroChannel Port Driver, Windows NT 4.0 DDK
// Copyright (C) UnalZ, 2003  http://site.voila.fr/mcabase/index.htm
//---------------------------------------------------------------------
// writeled.c    Write to LED display. MCPOS Driver test routine
//
//               DOS / Windows NT User-Mode
//---------------------------------------------------------------------
// UZ changes
//    Sep 2002 -- created but not yet tested
//    Dec 2002 -- moved to DDK 4.0, tested
//    Feb 2003 -- ported to DOS (define _WLEDDOS)
//                DOS: Define _WLEDDOS for the compiler !!!
// ---------------------------------------------------------------------

// _WLEDDOS can be also set as a preprocessor definition

// DOS Mode
#define _WLEDDOS
// Windows NT
//#undef _WLEDDOS

#ifndef _WLEDDOS
#include <windows.h>
#include <winioctl.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <time.h>

#ifdef _WLEDDOS
#include <string.h>
#include "wdosdef.h"
#endif
#include "wmcpioc.h"

#define MCP_DEBUG
#undef  MCP_DEBUG

#ifndef _WLEDDOS
// user name
#include <lmcons.h>

#define USER_NAME_LEN    UNLEN
#define USER_NAME_SIZE   UNLEN + 1
#define MACHINE_NAME_LEN    MAX_COMPUTERNAME_LENGTH
#define MACHINE_NAME_SIZE   MACHINE_NAME_LEN + 1

// Win32 bad value for file operations
#define WIN_BAD_VAL 0xFFFFFFFF
#endif

// Command line options //////////////////////////////

#define RUN_OPTS    "osbfmudh"
#define RUN_TEXT    'o'
#define RUN_SCRL    's'
#define RUN_BLINK    'b'
#define RUN_FILE    'f'
#define RUN_MACH    'm'
#define RUN_USER    'u'
#define RUN_DEMO    'd'
#define RUN_HELP    'h'


// io success
#define    IO_SUCCESS    0
#define    IO_FAILURE    -1

// max scroll text
#define MAX_LED_TEXT    512

// Function prototypes /////////////////////////////

DWORD
WriteLED(
    IN HANDLE,
    IN USHORT,
    IN PCHAR,
    IN BOOL
    );

#ifdef _WLEDDOS

DWORD
McpWriteLED(
    IN PMCP_LED_DISPLAY
    );

#else

DWORD
WriteComputerName(
    IN HANDLE
    );

DWORD
WriteUserName(
    IN HANDLE
    );

#endif

VOID
clockSleep(
    IN clock_t
    );

DWORD
WriteLEDScroll(
    IN HANDLE,
    IN PCHAR,
    IN USHORT
    );


DWORD
WriteLEDBlink(
    IN HANDLE,
    IN PCHAR,
    IN USHORT,
    IN BOOL,
    IN USHORT
    );

DWORD
WriteLEDFile(
    IN HANDLE,
    IN PCHAR
    );

DWORD
WriteDemo(
    IN HANDLE
    );

VOID
printSyntax(
    PCHAR,
    BOOL
    );


/////////////////////////////////////////////////////////////////////////////

#ifdef _WLEDDOS

DWORD
McpWriteLED(
    IN PMCP_LED_DISPLAY  IoBuffer
    )

/*
   Strictly DOS mode !!! Replaces MCPOS driver call.
   Writes indicated number of characters to the LED display of Mod. 95

Arguments:
    IoBuffer   - pointer to LED structure

Return value:
    IO_SUCCESS - OK
    IO_FAILURE - Invalid sizes

*/

{
    USHORT    i;
    USHORT    ledBase;
    USHORT    position;
    USHORT    textLength;
    SHORT     outTextLength;

    position   = IoBuffer->Position;
    textLength = IoBuffer->TextLength;

    // ensure valid text write position
    if ( position < 1 || position > MCP_MAX_LED)
    {
        return IO_FAILURE;
    }

    // check if text length within LED limits
    outTextLength = textLength + position - 1;

    // trim if text too long
    if ( outTextLength > MCP_MAX_LED)
    {
        textLength = MCP_MAX_LED - position + 1;
    }

    // get mapped LED display base address. With origin at the leftmost
    // cell (LED ports decrease) move to desired position

    ledBase = MCP_LED_BASE + MCP_MAX_LED - position;

    for(i = 0; i < MCP_MAX_LED && i < textLength; i++)
    {
        WRITE_PORT_UCHAR((USHORT)(ledBase - i), IoBuffer->Text[i]);
        IO_DELAY;
    }

    return IO_SUCCESS;

} // DOS write LED

#endif

////////////////////////////////////////////////////////////////////

DWORD
WriteLED(
    IN HANDLE  hDriver,
    IN USHORT  cellNo,
    IN PCHAR   textOut,
    IN BOOL    clearScr
    )

/*
 * Write to LED display. The leftmost cell is numbered 1,
 * the rightmost cell is 8. The driver intercepts invalid values
 * and right trims the output text if necessary.
 *
 * in :   hDriver  = MCPOS driver handle
 *        cellNo   = starting cell (x-coord) for output
 *        textOut  = output text
 *        clearScr = set to true if display must be cleared
 *
 */

{
#ifndef _WLEDDOS
    DWORD            returnedCount;
    BOOL             ioStatus;
#endif
    SHORT            outTextLen;
    USHORT           textLen;
    USHORT           i;
    MCP_LED_DISPLAY  ledLine;

    // check cellNo
    if ( cellNo < 1 || cellNo > MCP_MAX_LED )
    {
        return IO_FAILURE;
    }

    // calculate text length
    textLen = strlen(textOut);

    // trim the text length if it is too long
    outTextLen = textLen + cellNo - 1;

    if ( outTextLen > MCP_MAX_LED )
    {
        textLen = MCP_MAX_LED - cellNo + 1;
    }

    // create input buffer to driver
    if ( clearScr )
    {
        strcpy(ledLine.Text, MCP_BLANK_LED);

        // copy text to input buffer
        for (i = 0; i < textLen; i++)
        {
            ledLine.Text[i + cellNo - 1] = textOut[i];
        }

        // full display update
        ledLine.Position = 1;
        ledLine.TextLength = MCP_MAX_LED;

    }
    else
    {
        strcpy(ledLine.Text, textOut);
        ledLine.Position = (UCHAR)cellNo;
        ledLine.TextLength = (UCHAR)textLen;
    }


#ifdef MCP_DEBUG
printf("\nDEBUG: Cell: %d  Len: %d   Text: [%s]\n", cellNo, textLen, ledLine.Text);
#endif

#ifndef _WLEDDOS
    // call the driver
    ioStatus = DeviceIoControl(
                   hDriver,                // Handle to device
                   IOCTL_MCP_WRITE_LED,    // IO Control code for write LED
                   &ledLine,               // Buffer to driver.
                   sizeof(ledLine),        // Length of buffer in bytes.
                   &ledLine,               // * not used: Buffer from driver.
                   sizeof(ledLine),        // * not used: Length of buffer in bytes.
                   &returnedCount,         // * not used: Bytes placed in output buffer
                   NULL                    // NULL means wait till op. completes.
                   );

    if (!ioStatus)
    {
        return    GetLastError();
    }

#else

    if ( McpWriteLED( &ledLine ) != IO_SUCCESS )
    {
        return    IO_FAILURE;
    }

#endif

    return IO_SUCCESS;

} // write LED

///////////////////////////////////////////////////////////////////////

#ifndef _WLEDDOS

DWORD
WriteUserName(
    IN HANDLE  hDriver
    )

/*
 * Write user name to the LED display
 *
 * in :    hDriver = MCPOS driver handle
 *
 */

{
    TCHAR   uName[USER_NAME_LEN];
    DWORD   unSize = USER_NAME_SIZE;

    GetUserName(uName, &unSize);

    return WriteLED(hDriver, 1, uName, 1);


} // write user name

#endif

///////////////////////////////////////////////////////////////////////

#ifndef _WLEDDOS

DWORD
WriteComputerName(
    IN HANDLE    hDriver
    )

/*
 * Write computer name to the LED display
 *
 * in :    hDriver  = MCPOS driver handle
 *
 */

{
    TCHAR   mName[MACHINE_NAME_LEN];
    DWORD   mnSize = MACHINE_NAME_SIZE;

    GetComputerName(mName, &mnSize);

    return WriteLED(hDriver, 1, mName, 1);


} // write computer name

#endif

///////////////////////////////////////////////////////////////

void
clockSleep(
    IN clock_t  wait
    )

/*
 * Sleep, do nothing
 *
 * in:  wait = wait specified amount of milliseconds
 *
 */

{
   clock_t goal;

   goal = wait + clock();
   while( goal > clock() );

} // end clock sleep

///////////////////////////////////////////////////////////////

DWORD
WriteLEDScroll(
    IN HANDLE   hDriver,
    IN PCHAR    outText,
    IN USHORT   repeatNumber
    )

/*
 * Write scrolling text to display
 *
 * in:    hDriver  = MCPOS driver handle
 *        outText  = Text to scroll
 *
 */

{

    USHORT    outLen;
    USHORT    curPos;
    USHORT    i, k;
    DWORD     ioStatus;

    curPos = 0;
    outLen = strlen(outText);

    for (k = 0; k < repeatNumber; k++ )
    {
        for(i = 0; i < MAX_LED_TEXT && i < outLen; i++)
        {
            ioStatus = WriteLED(hDriver, 1, &outText[i], 1);

            if ( ioStatus != IO_SUCCESS)
            {
                break;
            }
            clockSleep(CLOCKS_PER_SEC / 5);
        }
    }

    if ( ioStatus == IO_SUCCESS)
    {
        ioStatus = WriteLED(hDriver, 1, " ", 1);
    }

    return ioStatus;


} // write LED scroll

///////////////////////////////////////////////////////////////

DWORD
WriteLEDBlink(
    IN HANDLE    hDriver,
    IN PCHAR     outText,
    IN USHORT    cellNo,
    IN BOOL      clrScreen,
    IN USHORT    repeatNumber
    )

/*
 * Write blinking text to display
 *
 * in:    hDriver  = MCPOS driver handle
 *        outText  = Text to blink
 *
 */

{

    USHORT    outLen;
    USHORT    curPos;
    USHORT    k;
    DWORD     ioStatus;
    CHAR      blanks[MCP_MAX_LED];

    curPos = 0;
    outLen = strlen(outText);

    // to blank out nur the portion written
    strcpy(blanks, MCP_BLANK_LED);
    blanks[MCP_MAX_LED] = 0;

    if (outLen < MCP_MAX_LED)
    {
        blanks[outLen] = 0;
    }


    for (k = 0; k < repeatNumber; k++ )
    {
        ioStatus = WriteLED(hDriver, cellNo, outText, clrScreen);

        if ( ioStatus != IO_SUCCESS)
        {
            break;
        }
        clockSleep(CLOCKS_PER_SEC / 3);

        // clear now

        ioStatus = WriteLED(hDriver, cellNo, blanks, clrScreen);

        if ( ioStatus != IO_SUCCESS)
        {
            break;
        }
        clockSleep(CLOCKS_PER_SEC / 3);

    }

    if (!clrScreen)
    {
        ioStatus = WriteLED(hDriver, cellNo, outText, clrScreen);
    }

    return ioStatus;


} // write LED blink

///////////////////////////////////////////////////////////////

DWORD
WriteLEDFile(
    IN HANDLE     hDriver,
    IN PCHAR    outFile
    )

/*
 * Write file contents as scrolling text to display
 *
 * in:    hDriver  = MCPOS driver handle
 *        outFile  = File to scroll
 *
 */

{

    USHORT   curPos;
    USHORT   i, j, k;
    DWORD    ioStatus;
    DWORD    charCount;
    DWORD    fileSize, highSize = 0;
    HANDLE   hFile;
    CHAR     fileBuf[MAX_LED_TEXT];
    CHAR     outText[MAX_LED_TEXT];

#ifndef _WLEDDOS
    BOOL    readStatus;

    // open file

    hFile = CreateFile(
                outFile,
                GENERIC_READ,
                FILE_SHARE_READ,
                NULL,
                OPEN_EXISTING,
                FILE_FLAG_SEQUENTIAL_SCAN,
                NULL);

    if (hFile == INVALID_HANDLE_VALUE)
    {
        ioStatus = GetLastError();
        printf("\nError: file problem, could not open [%s]", outFile);
        return ioStatus;
    }

    // see if file has a valid size

    fileSize = GetFileSize(hFile, &highSize);
    ioStatus = GetLastError();

    if (fileSize == WIN_BAD_VAL && ioStatus != IO_SUCCESS) {
        printf("\nError: file problem, bad file size");
        CloseHandle(hFile);
        return ioStatus;
    }

    // skip zero size files

    if (fileSize == 0) {
        printf("\nAbort: file contains no characters (zero size)");
        CloseHandle(hFile);
        fclose(hFile);
        return (1);
    }

#else

    if ( (hFile = fopen(outFile, "r")) == NULL )
    {
        printf("\nError: file problem, could not open [%s]", outFile);
        return IO_FAILURE;
    }

    // file size check skipped for DOS

#endif

    printf("\nWorking... look at your display");

    k = 1;
    curPos = 0;
    charCount = 1;

    while (charCount > 0)
    {

#ifndef _WLEDDOS

        // read file

        readStatus = ReadFile(
                        hFile,
                        fileBuf,
                        MAX_LED_TEXT,
                        &charCount,
                        NULL);

        if ( !readStatus )
        {
            ioStatus = GetLastError();
            printf("\nError: file problem, cannot read");
            break;
        }

#else
        charCount = fread(fileBuf, sizeof(CHAR), MAX_LED_TEXT, hFile);
        ioStatus = IO_SUCCESS;

        // zero charCount will be intercepted below

#endif

        // clean and compress

        for (i = 0, k = 0; i < charCount; i++)
        {
            // strip controls chars and white space
            if (fileBuf[i] <= 0x20)
            {
                outText[k++] = 0x20;
                for(j = i+1; fileBuf[j] <= 0x20 && j < charCount; j++)
                    ;
                i = j;

            }
            if ( i < charCount )
            {
                outText[k++] = fileBuf[i];
            }
        }

        // the loop evals k but leave it so, better readable
        // for(i = 0; i < k; i++)

        if (k > 0)
        {
            for(i = 0; i < k; i++)
            {
                ioStatus = WriteLED(hDriver, 1, &outText[i], 1);

                if ( ioStatus != IO_SUCCESS)
                {
                    break;
                }
                clockSleep(CLOCKS_PER_SEC / 6);
            }
        }
    }


    // close file

#ifndef _WLEDDOS
    CloseHandle(hFile);
#else
    fclose(hFile);
#endif

    if ( ioStatus == IO_SUCCESS)
    {
        ioStatus = WriteLED(hDriver, 1, " ", 1);
    }

    return ioStatus;


} // write LED file

///////////////////////////////////////////////////////////////

DWORD
WriteDemo(
    IN HANDLE  hDriver
    )

/*
 * Write display demo
 *
 * in:    hDriver = MCPOS driver handle
 *
 */

{
    DWORD    ioStatus;

    // Zappa's Universe

    ioStatus = WriteLEDScroll(hDriver,
                "      ooo The way I see it Barry this should be a very dynamite show ... "
                "Ladies and Gentlemen, Elvis has just left the building those are his "
                "footprints right there ... (Frank Zappa) ", 1);

    if (ioStatus != IO_SUCCESS)
    {
        return ioStatus;
    }

    // blink

    ioStatus = WriteLED(hDriver, 1, "PS/2MCA", TRUE);

    if (ioStatus != IO_SUCCESS)
    {
        return ioStatus;
    }

    // blink first half

    ioStatus = WriteLEDBlink(hDriver, "PS/2", 1, FALSE, 3);

    if (ioStatus != IO_SUCCESS)
    {
        return ioStatus;
    }

    // blink second half

    ioStatus = WriteLEDBlink(hDriver, "MCA", 5, FALSE, 3);

    if (ioStatus != IO_SUCCESS)
    {
        return ioStatus;
    }

    // blink full

    ioStatus = WriteLEDBlink(hDriver, "PS/2MCA", 1, TRUE, 4);

    if (ioStatus != IO_SUCCESS)
    {
        return ioStatus;
    }

#ifndef _WLEDDOS

    // write computer name

    ioStatus = WriteComputerName(hDriver);

#endif

    return ioStatus;

} // write demo


///////////////////////////////////////////////////////////////////////
VOID
printSyntax(
    PCHAR    nprog,
    BOOL    full
    )

/*
 * Print syntax command line options
 *
 * in :   nprog  = program name (not used)
 *        full   = print full help
 *
 */

{

#ifndef _WLEDDOS
        printf("\nSyntax:\n"
               "\n    writeled [-%c |-%c |-%c] DisplayText | [-%c] FileName | [-%c |-%c |-%c]"
               "\n    writeled  -%c  for help\n"
               ,RUN_TEXT, RUN_BLINK, RUN_SCRL, RUN_FILE, RUN_MACH, RUN_USER, RUN_DEMO, RUN_HELP);

    if (full) {
        printf("\nOptions:\n"
               "\n    -%c  DisplayText   Write DisplayText, up to 8 characters"
               "\n    -%c  DisplayText   Blink DisplayText, up to 8 characters"
               "\n    -%c  DisplayText   Scroll DisplayText, any length"
               "\n    -%c  FileName      Dump file FileName to display, any length"
               "\n    -%c                Write machine name, first 8 characters"
               "\n    -%c                Write user name, first 8 characters"
               "\n    -%c                Run WriteLED demo\n"
               "\n    DisplayText: The text you wish to write to the LED display,"
               "\n                 enclosed in quotation marks, e.g. \"Hey you\"\n"
               "\n    Press Ctrl-C to abort any writing operation in progress.\n"
               , RUN_TEXT, RUN_BLINK, RUN_SCRL, RUN_FILE, RUN_MACH, RUN_USER, RUN_DEMO);
    }
#else
        printf("\nSyntax:\n"
               "\n    writeled [-%c |-%c |-%c] DisplayText | [-%c] FileName | [-%c]"
               "\n    writeled  -%c  for help\n"
               , RUN_TEXT, RUN_BLINK, RUN_SCRL, RUN_FILE, RUN_DEMO, RUN_HELP);

    if (full) {
        printf("\nOptions:\n"
               "\n    -%c  DisplayText   Write DisplayText, up to 8 characters"
               "\n    -%c  DisplayText   Blink DisplayText, up to 8 characters"
               "\n    -%c  DisplayText   Scroll DisplayText, any length"
               "\n    -%c  FileName      Dump file FileName to display, any length"
               "\n    -%c                Run WriteLED display demo\n"
               "\n    DisplayText: The text you wish to write to the LED display,"
               "\n                 enclosed in quotation marks, e.g. \"Hey you\"\n"
               "\n    Press Ctrl-C to abort any writing operation in progress.\n"
               , RUN_TEXT, RUN_BLINK, RUN_SCRL, RUN_FILE, RUN_DEMO);
    }
#endif

} // print Syntax

///////////////////////////////////////////////////////////////

void
main(
    IN int  argc,
    IN char *argv[]
    )

{

/*
 * (1) DOS trick: HANDLE  hDriver = NULL to preserve NT calling scheme
 * (2) goto endLabel can be replaced with exit(ExitMsg())
 *
 */

    HANDLE   hDriver;
    DWORD    ioStatus;

    // Command line arguments //////////////////////////////

    int runmode = 0;
    char *pargv = NULL;

    /// get run options ////////////////////////////////

    if (argc <= 1)
    {
        printSyntax(argv[0], FALSE);
        runmode = RUN_MACH;
    }
    else
    {
        runmode = argv[1][1];
        if (argv[1][0] != '-' || strchr(RUN_OPTS, runmode) == NULL)
        {
            printSyntax(argv[0], FALSE);
            goto endLabel;
        }

        if (runmode == RUN_HELP)
        {
            printSyntax(argv[0], TRUE);
            goto endLabel;
        }

        // argument expected now (this shorthand ok)
        pargv = (argc > 2 ? argv[2] : (argv[1][2] != 0 ? &argv[1][2] : NULL));

        if (pargv == NULL && (
            runmode == RUN_TEXT  ||
            runmode == RUN_BLINK ||
            runmode == RUN_FILE  ||
            runmode == RUN_SCRL ))
        {
            printf("\n[%s -%c]: Argument required.", argv[0], runmode);
            goto endLabel;
        }

    }

#ifdef _WLEDDOS

    hDriver = NULL;

#else
    // create device file

    hDriver = CreateFile(
                MCPOS_DEVICE_NAME,
                GENERIC_READ,
                FILE_SHARE_READ,
                NULL,
                OPEN_EXISTING,
                0,
                NULL
                );

    if (hDriver == INVALID_HANDLE_VALUE)
    {
        printf("\nI/O Error: unable to open MCPOS device.\n");
        goto endLabel;
    }

#endif

    // D I S P L A Y demo routines //////////////////////////////

    switch (runmode)
    {
    case RUN_TEXT:
        ioStatus = WriteLED(hDriver, 1, pargv, 1);
        break;

    case RUN_SCRL:
        // text
        printf("\nScrolling .... look at your LED display.\n");
        ioStatus = WriteLEDScroll(hDriver, pargv, 2);
        break;

    case RUN_BLINK:
        // text
        printf("\nBlinking .... look at your LED display.\n");
        ioStatus = WriteLEDBlink(hDriver, pargv, 1, TRUE, 6);
        break;

    case RUN_FILE:
        // text
        printf("\nAbout to dump file [%s] to the LED display...",
                pargv);
        ioStatus = WriteLEDFile(hDriver, pargv);
        break;

#ifndef _WLEDDOS

    case RUN_MACH:
        // machine name
        ioStatus = WriteComputerName(hDriver);
        break;

    case RUN_USER:
        // user name
        ioStatus = WriteUserName(hDriver);
        break;
#endif

    case RUN_DEMO:
        // demo
        printf("\nDemo running .... look at your LED display.\n");
        ioStatus = WriteDemo(hDriver);
        break;

    default:
        // do nothing
        ioStatus = IO_SUCCESS;
    }

    // close the device //////////////////////////////

#ifndef _WLEDDOS

    if (!CloseHandle(hDriver))
    {
        printf("I/O Error: Failed to close device.\n");
    }

#endif

    // handle possible io error

    if (ioStatus != IO_SUCCESS)
    {
        printf("\nError: write LED failed with code: %ld\n",
                ioStatus
                );
    }
    else
    {
        if (runmode == RUN_DEMO || runmode == RUN_MACH || runmode == RUN_USER)
        {
        #ifndef _WLEDDOS
            printf( (runmode == RUN_USER ?
                     "\nLED DISPLAY updated with the user name." :
                     "\nLED DISPLAY updated with the machine name."));
        #endif
        }
        else
        {
            printf("\nLED DISPLAY updated.");
        }
    }

      // exit ///////////////////////////////////////////////////

endLabel:

    printf("\n------------------------------------------------------"
        #ifdef _WLEDDOS
            "\nRunstream (R) WriteLED for DOS         Beta Rel. 00.09"
        #else
            "\nRunstream (R) WriteLED for Windows NT, Beta Rel. 00.09"
        #endif
            "\nCopyright (C) Unal Z, 2003. All rights reserved. #0302\n");

    exit(0);


} /* main */

///// End of code //////////////////////////////////////////////////
