#include <lib/libsa/stand.h>

#include <sys/cdefs.h>
#include <sys/types.h>
#if 0
#include <arc/arc/arcbios.h>
#else
#include "arcbios.h"
#endif

#include <stdio.h>
#include "dumpconf.h"

#define ARRAY_LENGTH(array)	(sizeof(array)/sizeof(array[0]))

void fput_indent __P((FILE *file, int indent));
void fput_string __P((FILE *file, char *string, int length));
void fput_name __P((FILE *file, const char **table, int table_size,
		    int index));
void fput_flags __P((FILE *file, const char **table, int table_size,
		     u_int32_t flags, int bits));
void fput_bytes __P((FILE *file, u_int8_t *p, int size));

int bios_dump_resource_data __P((FILE *file, arc_cm_partial_resource_t *rsrc, int indent));
void bios_dump_config_data __P((FILE *file, arc_config_t *cf, int data_len, int indent));
void bios_dump_config_component __P((FILE *file, arc_config_t *cf, int indent));
void bios_dump_config_subtree __P((FILE *file, arc_config_t *cf, int indent));

void
fput_indent(FILE *file, int indent)
{
	while (--indent >= 0)
		fputc(' ', file);
}

void
fput_string(FILE *file, char *string, int length)
{
	static char hex[] = "0123456789abcdef";

	while (--length >= 0) {
		int c = *string++ & 0xff;

		if (c < ' ' || 0x7f <= c) {
			fprintf(file, "\\x%c%c", hex[c >> 4], hex[c & 0xf]);
		} else {
			if (c == '\\')
				fputc(c, file);
			fputc(c, file);
		}
	}
}

void
fput_name(FILE *file, const char **table, int table_size, int index)
{
	if (index < 0 || table_size <= index) {
		fprintf(file, "#%d", index);
		return;
	}
	fprintf(file, "%s", table[index]);
}

void
fput_flags(FILE *file, const char **table, int table_size,
	   u_int32_t flags, int bits)
{
	int i, printed = 0;

	for (i = 0; i < bits; i++) {
		if (flags & (1 << i)) {
			if (printed)
				fprintf(file, "|");
			fput_name(file, table, table_size, i);
			printed = 1;
		}
	}
}

void
fput_bytes(FILE *file, u_int8_t *p, int size)
{
	int i;

	for (i = 0; i < size; i++) {
		if (i > 0)
			fprintf(file, ":");
		fprintf(file, "0x%x", p[i]);
	}
}

/*
 * Find out system type.
 */
int
bios_magic(void)
{
	switch (ArcBiosBase->magic) {
	case ARC_PARAM_BLK_MAGIC:
	case ARC_PARAM_BLK_MAGIC_BUG:
		return (1);
	default:
		return (0);	/* This is not an ARC system */
	}
}

void
bios_dump_param_blk(FILE *file)
{
	int i, j;

	fprintf(file, "magic:  \t%c%c%c%c\n",
		((char *)&ArcBiosBase->magic)[0],
		((char *)&ArcBiosBase->magic)[1],
		((char *)&ArcBiosBase->magic)[2],
		((char *)&ArcBiosBase->magic)[3]);
	fprintf(file, "length: \t0x%x\n", ArcBiosBase->length);
	fprintf(file, "version:\t%d\n", ArcBiosBase->version);
	fprintf(file, "revision:\t%d\n", ArcBiosBase->revision);
	fprintf(file, "restart_block:\t%p\n", ArcBiosBase->restart_block);
#if 0
	bios_dump_restart_blk(file);
#endif
	fprintf(file, "debug_block:\t%p\n", ArcBiosBase->debug_block);
	fprintf(file, "gen_excep_vec:\t%p\n", ArcBiosBase->general_exp_vect);
	fprintf(file, "utlbmiss_vec:\t%p\n", ArcBiosBase->tlb_miss_exp_vect);
	fprintf(file, "firm_vec_len:\t0x%x\n", ArcBiosBase->firmware_length);
	fprintf(file, "firmare_vector:\n");
	for (i = 0; i < ArcBiosBase->firmware_length; i += 4)
		fprintf(file, "\t0x%x\n",
			*(u_int32_t *)
			((u_int8_t *)ArcBiosBase->firmware_vect + i));
	fprintf(file, "vend_vec_len:\t0x%x\n", ArcBiosBase->vendor_length);
	fprintf(file, "vendor_vector:\n");
	for (i = 0; i < ArcBiosBase->vendor_length; i += 4)
		fprintf(file, "\t0x%x\n",
			*(u_int32_t *)
			((u_int8_t *)ArcBiosBase->vendor_vect + i));
	fprintf(file, "adapter_count:\t%d\n", ArcBiosBase->adapter_count);
	for (j = 0; j < ArcBiosBase->adapter_count; j++) {
		/*
		 * XXX
		 * access to ArcBiosBase->adapters[j].adapter_type
		 * makes egcs-2.91.66 19990314 (egcs-1.1.2 release)
		 * internal compiler error in function emit_move_insn_1
		 */
		struct arc_adapter_param *ap = &ArcBiosBase->adapters[j];
		
		fprintf(file, "adap%d_type:\t%d\n", j, ap->adapter_type);
		fprintf(file, "adap%d_vec_len:\t0x%x\n",
			j, ap->adapter_length);
		fprintf(file, "adap%d_vector:\n", j);
		for (i = 0; i < ap->adapter_length; i += 4)
			fprintf(file, "\t0x%x\n",
				*(u_int32_t *)
				((u_int8_t *)ap->adapter_vect + i));
	}
}

void
bios_dump_system_id(FILE *file)
{
	arc_sid_t *sid = Bios_GetSystemId();

	if (sid == NULL) {
		fprintf(file, "BIOS System ID: NULL\n");
		return;
	}
	fprintf(file, "BIOS Vendor  ID: [");
	fput_string(file, sid->vendor, sizeof(sid->vendor));
	fprintf(file, "]\n");

	fprintf(file, "BIOS Product ID: [");
	fput_bytes(file, sid->prodid, sizeof(sid->prodid));
	fprintf(file, "]\n");
}

int
bios_dump_resource_data(FILE *file, arc_cm_partial_resource_t *rsrc,
			int indent)
{
	static const char *resource_type_name[] = {
		"Null",
		"Port",
		"Interrupt",
		"Memory",
		"DMA",
		"DeviceSpecific",
		"Vector",
		"ProductName",
		"SerialNumber"
	};
	static const char *resource_share_disposition_name[] = {
		"Undetermined",
		"DeviceExclusive",
		"DriverExclusive",
		"Shared"
	};
	static const char *resource_interrupt_flag_name[] = {
		"LevelSensitive",
		"Latched"
	};
	static const char *resource_memory_flag_name[] = {
		"ReadWrite",
		"ReadOnly",
		"WriteOnly"
	};
	static const char *resource_port_flag_name[] = {
		"Memory",
		"IO"
	};

	fput_indent(file, indent);
	fprintf(file, "type: ");
	fput_name(file, resource_type_name, ARRAY_LENGTH(resource_type_name),
		  rsrc->type);
	fprintf(file, "\n");

	fput_indent(file, indent + 1);
	fprintf(file, "share_disposition: ");
	fput_name(file, resource_share_disposition_name,
		  ARRAY_LENGTH(resource_share_disposition_name),
		  rsrc->share_disposition);
	fprintf(file, ", flags: ");

	switch (rsrc->type) {
	case arc_CmResourceTypePort:
		fput_name(file, resource_port_flag_name,
			  ARRAY_LENGTH(resource_port_flag_name),
			  rsrc->flags);
		fprintf(file, "\n");
		fput_indent(file, indent + 1);
		fprintf(file, "start: 0x %x %x, length: 0x%x\n",
			rsrc->u.port.start.hiword,
			rsrc->u.port.start.loword,
			rsrc->u.port.length);
		break;
	case arc_CmResourceTypeInterrupt:
		fput_name(file, resource_interrupt_flag_name,
			  ARRAY_LENGTH(resource_interrupt_flag_name),
			  rsrc->flags);
		fprintf(file, "\n");
		fput_indent(file, indent + 1);
		fprintf(file, "level: %d, vector: %d, reserved1: %d\n",
			rsrc->u.interrupt.level,
			rsrc->u.interrupt.vector,
			rsrc->u.interrupt.reserved1);
		break;
	case arc_CmResourceTypeMemory:
		fput_name(file, resource_memory_flag_name,
			  ARRAY_LENGTH(resource_memory_flag_name),
			  rsrc->flags);
		fprintf(file, "\n");
		fput_indent(file, indent + 1);
		fprintf(file, "start: 0x %x %x, length: 0x%x\n",
			rsrc->u.memory.start.hiword,
			rsrc->u.memory.start.loword,
			rsrc->u.memory.length);
		break;
	case arc_CmResourceTypeDMA:
		fprintf(file, "0x%x\n", rsrc->flags);
		fput_indent(file, indent + 1);
		fprintf(file, "channel: %d, port: %d, reserved1: %d\n",
			rsrc->u.dma.channel,
			rsrc->u.dma.port,
			rsrc->u.dma.reserved1);
		break;
	case arc_CmResourceTypeDeviceSpecific:
		fprintf(file, "0x%x\n", rsrc->flags);
		fput_indent(file, indent + 1);
		fprintf(file,
			"datasize: %d, reserved1: %d, reserved2: %d\n",
			rsrc->u.device_specific_data.data_size,
			rsrc->u.device_specific_data.reserved1,
			rsrc->u.device_specific_data.reserved2);
		fput_indent(file, indent + 1);
		fprintf(file, "data: [");
		fput_bytes(file, (u_int8_t *)(rsrc + 1),
			   rsrc->u.device_specific_data.data_size);
		fprintf(file, "]\n");
		return (sizeof(*rsrc) +
			rsrc->u.device_specific_data.data_size);
	case arc_CmResourceTypeVendor:
		fprintf(file, "0x%x\n", rsrc->flags);
		fput_indent(file, indent + 1);
		fprintf(file, "vendor: [");
		fput_string(file, rsrc->u.vendor.vendor,
		       sizeof(rsrc->u.vendor.vendor));
		fprintf(file, "]\n");
		break;
	case arc_CmResourceTypeProductName:
		fprintf(file, "0x%x\n", rsrc->flags);
		fput_indent(file, indent + 1);
		fprintf(file, "product: [");
		fput_bytes(file, rsrc->u.product_name.product_name,
		       sizeof(rsrc->u.product_name.product_name));
		fprintf(file, "]\n");
		break;
	case arc_CmResourceTypeSerialNumber:
		fprintf(file, "0x%x\n", rsrc->flags);
		fput_indent(file, indent + 1);
		fprintf(file, "product: [");
		fput_bytes(file, rsrc->u.serial_number.serial_number,
		       sizeof(rsrc->u.serial_number.serial_number));
		fprintf(file, "]\n");
		break;
	default:
		fprintf(file, "0x%x\n", rsrc->flags);
		fput_indent(file, indent + 1);
		fprintf(file, "data: [");
		fput_bytes(file,
			   rsrc->u.vendor.vendor, sizeof(rsrc->u.vendor));
		fprintf(file, "]\n");
		break;
	}
	return (sizeof(*rsrc));
}

void
bios_dump_config_data(FILE *file, arc_config_t *cf, int data_len, int indent)
{
	int i, n, rsrc_size, parsed_size;
	arc_status_t rv;
	arc_cm_partial_resource_list_t *data;
	char data_buffer[256];

	if (data_len == 0)
		return; /* no configuration data */
	if (data_len > sizeof(data_buffer)) {
		fput_indent(file, indent);
		fprintf(file, "too large configuration data: %d bytes\n",
			data_len);
		return;
	}
	rv = Bios_GetConfigurationData(data_buffer, cf);
	if (rv != arc_ESUCCESS) {
		fput_indent(file, indent);
		fprintf(file, "error: %d\n", rv);
		return;
	}
	data = (arc_cm_partial_resource_list_t *)data_buffer;
	fput_indent(file, indent);
	fprintf(file, "config data version: %d, revision: %d, count: %d\n",
		data->version, data->revision, data->count);
	rsrc_size = data_len -
		(sizeof(*data) - sizeof(data->partial_descriptors[0]));
	n =  rsrc_size / sizeof(data->partial_descriptors[0]);
	if (data->count > n) {
		fput_indent(file, indent);
		fprintf(file, "resource count %d too big: use %d instead\n",
			data->count, n);
	} else {
		n = data->count;
	}
	parsed_size = 0;
	for (i = 0; i < n; i++) {
		parsed_size +=
			bios_dump_resource_data(file,
						&data->partial_descriptors[i],
						indent + 1);
	}
	if (data->count > n) {
		fput_indent(file, indent);
		fprintf(file, "raw count: [");
		fput_bytes(file, (u_int8_t *)&data->count,
			   sizeof(data->count));
		fprintf(file, "]\n");
		fput_indent(file, indent);
		fprintf(file, "raw data: %d:[", rsrc_size);
		fput_bytes(file, (u_int8_t *)&data->partial_descriptors[0],
			   rsrc_size);
		fprintf(file, "]\n");
	} else if (rsrc_size > parsed_size) {
		fput_indent(file, indent);
		fprintf(file, "unparsed data: %d:[", rsrc_size - parsed_size);
		fput_bytes(file,
			   (u_int8_t *)&data->partial_descriptors[0] +
			   parsed_size,
			   rsrc_size - parsed_size);
		fprintf(file, "]\n");
	}
}

void
bios_dump_config_component(FILE *file, arc_config_t *cf, int indent)
{
	static const char *config_class_name[] = {
		"System",
		"Processor",
		"Cache",
		"Adapter",
		"Controller",
		"Peripheral",
		"Memory"
	};
	static const char *config_type_name[] = {
		"ARC",

		"CPU",
		"FPU",
		"PrimaryICache",
		"PrimaryDCache",
		"SecondaryICache",
		"SecondaryDCache",
		"SecondaryCache",

		"EISAAdapter",
		"TCAdapter",
		"SCSIAdapter",
		"DTIAdapter",
		"MultiFunctionAdapter",

		"DiskController",
		"TapeController",
		"CDROMController",
		"WORMController",
		"SerialController",
		"NetworkController",
		"DisplayController",
		"ParallelController",
		"PointerController",
		"KeyboardController",
		"AudioController",
		"OtherController",

		"DiskPeripheral",
		"FloppyDiskPeripheral",
		"TapePeripheral",
		"ModemPeripheral",
		"MonitorPeripheral",
		"PrinterPeripheral",
		"PointerPeripheral",
		"KeyboardPeripheral",
		"TerminalPeripheral",
		"OtherPeripheral",
		"LinePeripheral",
		"NetworkPeripheral",

		"MemoryUnit"
	};
	static const char *config_flag_name[] = {
		"Failed",
		"ReadOnly",
		"Removable",
		"ConsoleIn",
		"ConsoleOut",
		"Input",
		"Output"
	};

	fput_indent(file, indent);
	fprintf(file, "class: ");
	fput_name(file, config_class_name, ARRAY_LENGTH(config_class_name),
		  cf->class);
	fprintf(file, ", type: ");
	fput_name(file, config_type_name, ARRAY_LENGTH(config_type_name),
		  cf->type);
	fprintf(file, "\n");

	fput_indent(file, indent + 1);
	fprintf(file, "flags: ");
	fput_flags(file, config_flag_name, ARRAY_LENGTH(config_flag_name),
		   cf->flags, sizeof(cf->flags) * 8);
	fprintf(file, "\n");

	fput_indent(file, indent + 1);
	fprintf(file, "version: %d, revision: %d\n",
		cf->version, cf->revision);

	fput_indent(file, indent + 1);
	switch (cf->class) {
	case arc_CacheClass:
		fprintf(file, "refill-size: %d, line-size: %d, size: %d\n",
			(cf->key >> 24) & 0xff,
			1 << ((cf->key >> 16) & 0xff),
			4096 << (cf->key & 0xffff));
		break;
	default:
		fprintf(file, "key: %d\n", cf->key);
		break;
	}

	fput_indent(file, indent + 1);
	fprintf(file, "affinity_mask: 0x%x\n", cf->affinity_mask);

	fput_indent(file, indent + 1);
	fprintf(file, "config_data_len: %d\n", cf->config_data_len);

	if (cf->id_len > 0) {
		fput_indent(file, indent + 1);
		fprintf(file, "id: %d:[", cf->id_len);
		fput_string(file, cf->id, cf->id_len);
		fprintf(file, "]\n");
	}

	bios_dump_config_data(file, cf, cf->config_data_len, indent + 1);
}

void
bios_dump_config_subtree(FILE *file, arc_config_t *cf, int indent)
{
	for (cf = Bios_GetChild(cf); cf != NULL; cf = Bios_GetPeer(cf)) {
		bios_dump_config_component(file, cf, indent);
		fputc('\n', file);
		bios_dump_config_subtree(file, cf, indent + 8);
	}
}

void
bios_dump_configuration(FILE *file)
{
	bios_dump_config_subtree(file, NULL, 0);
}

void
bios_dump_memory(FILE *file)
{
	u_int32_t total = 0;		/* Total physical memory size */
	arc_mem_t *descr = NULL;
	static const char *memory_type_name[] = {
		"ExeceptionBlock",
		"SystemParameterBlock",
		"FreeMemory",
		"BadMemory",
		"LoadedProgram",
		"FirmwareTemporary",
		"FirmwarePermanent",
		"FreeContigous"
	};

	while ((descr = Bios_GetMemoryDescriptor(descr)) != NULL) {
		u_int32_t
			seg_start = descr->BasePage * 4096,
			seg_size = descr->PageCount * 4096,
			seg_end = seg_start + seg_size - 1;

		total += seg_size;
		fprintf(file, "memory 0x%x..%x, size: %d bytes, type: ",
			seg_start, seg_end, seg_size);
		fput_name(file, memory_type_name,
			  ARRAY_LENGTH(memory_type_name),
			  descr->Type);
		fprintf(file, "\n");
	}
	fprintf(file, "Total memory = %d bytes\n", total);
}

void
bios_dump_display_status(FILE *file, int fd)
{
	arc_dsp_stat_t *dsp = Bios_GetDisplayStatus(fd), displaystatus;

	if (dsp == NULL) {
		fprintf(file, "display information of %d: NULL", fd);
		return;
	}
	displaystatus = *dsp;
	fprintf(file, "cursor: (%d, %d), cursor_max: (%d, %d)\n",
		displaystatus.CursorXPosition,
		displaystatus.CursorYPosition,
		displaystatus.CursorMaxXPosition,
		displaystatus.CursorMaxYPosition);
	fprintf(file,
		"color - "
		"fg: %d, bg: %d, intensity: %d, underscore: %d, reverse: %d\n",
		displaystatus.ForegroundColor,
		displaystatus.BackgroundColor,
		displaystatus.HighIntensity,
		displaystatus.Underscored,
		displaystatus.ReverseVideo);
}

