/*
 * $Id: //devel/tools/main/rfc868time/rfc868time.cpp#5 $ -- 2000-01-12
 *
 * written by :	Stephen J. Friedl
 *              Software Consultant
 *              steve@unixwiz.net
 *              http://unixwiz.net/tools/rfc868time.html
 *
 *	This is a Windows service that provides very simple RFC868 time
 *	services to the network. This is readonly with respect to the
 *	system clock, and shouldn't be used for actual time synchronization
 *	when there are better methods available.
 *
 *	We wrote this for system clock *auditing*. Throughout the data center,
 *	we have many machines that are supposed to agree on the time through
 *	various means - NTP, mainly - but for one reason or another, drift
 *	(misconfigured firewall, service didn't start, bad ntpd.conf, etc.).
 *
 *	So a central program polls every device in the place, asking for the
 *	time, and makes notes of which machines are not synchronized with
 *	what we think the master is.
 *
 *	This won't tell us *why* something is off, but it will at least alert
 *	us to the presence of a problem.
 *
 * RFC868 TIME
 * -----------
 *
 *	When a connection is made on port 37/tcp, the service write a four-byte
 *	time in network word order to the caller, and closes the socket.
 *
 *	This time is the number of seconds since 1/1/1900, and we simply
 *	add a fixed constant to the UNIX time to reliably return this value.
 *	This is the value of the local system clock, independent of any
 *	NTP-type adjustments which might be going on.
 *
 *	At no time does it *read* from the connection.
 *
 *	RFC868 provides for UDP transport as well, but we don't support it
 *	yet (and may never do so).
 *
 * INSTALLING / UNINSTALLING
 * --------------------------
 *
 *	Parameters:
 *
 *	-debug		enable more debugging to the DBMON log
 *	-install	install the service using *this* .EXE (but don't start)
 *	-remove		remove the service after stopping it
 *	-version	Show the version information, then exit with success
 *	-help		show a brief help listing, then exit with success
 *
 * THINGS TO DO
 * ------------
 *
 *	- add UDP support
 *	- add event log support
 *	- write a man page for it
 *	- find a way to bind just to one interface
 *	- provide popup support for fatal error messages
 *	- add function-call parameter security decoration
 *
 * HISTORY
 * -------
 *
 * v1.5 2011/03/05 - Now compile with ASLR and DEP; now properly show help
 *		and exit if the user types no commands; added support for a
 *		service description; now use all the _s safe functions; now
 *		drop all token privileges after we have the network open.
 * v1.4 2008/12/05 - cleaned up cmdline processing
 * v1.3 2007/07/24 - public release
 */
#include <windows.h>
#include <winsock2.h>
#include <stdlib.h>
#include <stdio.h>
#include <tchar.h>
#include <time.h>
#include "defs.h"

#define VERSION  "1.5"
#define VERSDATE "2011-03-05"

const char Version[]     = VERSION;

const char fullVersion[] = "Unixwiz.net " SERVICENAME " " VERSION " - " VERSDATE
	" - http://unixwiz.net/tools/";



int	Backlog = 5;

int	Debug = FALSE,
	runningAsService = FALSE;
TCHAR	*ProgName = 0;

HANDLE	hShutdownEvent = 0;

/*
 * clsocket()
 *
 *	clsocket a socket, reset its value *in the caller*
 */
inline void clsocket(SOCKET &sock)
{
	if ( sock != INVALID_SOCKET )
	{
		closesocket(sock);
		sock = INVALID_SOCKET;
	}
}

/*
 * setupsocket()
 *
 *	Given a protocol (TCP or UDP), a port number, and a description of
 *	the protocol, return a socket that's created, bound to the port, and
 *	in a listening state.
 *
 *	Note that even though we support UDP, we're probably not going to
 *	ever use it.
 */
static SOCKET setupsocket(bool isTCP, short portno, int backlog, WSAEVENT *hEvent)
{
	const TCHAR *name = isTCP ? _T("tcp") : _T("udp");

	SOCKET fd = socket(
		PF_INET,
		isTCP ? SOCK_STREAM : SOCK_DGRAM,
		isTCP ? IPPROTO_TCP : IPPROTO_UDP
	);

	if ( fd == INVALID_SOCKET )
	{
		odprintf(_T("Cannot create %s socket [err#%ld]"),
			name,
			WSAGetLastError()
		);

		return INVALID_SOCKET;
	}

	sockaddr_in rdateport;

	ZeroMemory(&rdateport, sizeof rdateport);

	rdateport.sin_family      = AF_INET;
	rdateport.sin_addr.s_addr = INADDR_ANY;
	rdateport.sin_port        = htons(portno);

	if ( bind(fd, (struct sockaddr *)&rdateport, sizeof rdateport) != 0)
	{
		odprintf(_T("Cannot bind to %d/%s [err#%ld]"),
			portno, name,
			WSAGetLastError() );

		clsocket(fd);
		return INVALID_SOCKET;
	}

	if ( listen(fd, backlog) != 0 )
	{
		odprintf(_T("Cannot listen on %s socket [err#%ld]"),
			name,
			WSAGetLastError() );
		clsocket(fd);
		return INVALID_SOCKET;
	}

	// Works! Yay!

	// create the handle event

	*hEvent = WSACreateEvent();

	// Associate event types FD_ACCEPT with this socket

	WSAEventSelect(fd, *hEvent, FD_ACCEPT);

	return fd;
}

/*
 * rfc868time()
 *
 *	Return the time as the number of seconds since 1/1/1900: this
 *	is just a constant offset to the UNIX time.
 */
static time_t rfc868time(void)
{
	time_t now;

	time(&now);

	return now + 2208988800U;
}


/*
 * tcpevent()
 *
 *	This is called when the TCP socket (which is listening for
 *	connections), gets an event.
 */
static void tcpevent(SOCKET s)
{
	if ( Debug ) odprintf(_T("Got TCP event on %d\n"), s);

	struct sockaddr peer;

	int socklen = sizeof peer;

	SOCKET sfd = accept(s, &peer, &socklen);

	if ( sfd == SOCKET_ERROR )
	{
		odprintf(_T("Could not accept connection [err#%ld]"),
			WSAGetLastError());
		return;
	}

	struct sockaddr_in *saddr = (struct sockaddr_in *)&peer;

	if ( Debug )
	  odprintf(" Got valid connection from %s", inet_ntoa(saddr->sin_addr));

	const unsigned int rtime = htonl( (u_long) rfc868time() );

	if ( send(sfd, (const char *)&rtime, 4, 0) != 4 )
		odprintf(_T("Note: could not send 4 bytes"));

	clsocket(sfd);
}

/*
 * remove_rights()
 *
 *	Remove every one of the rights we retain, making them unavailable
 *	for exploit. This is called only AFTER we gotten ourselves started
 *	up, after we're listening on the network, but before we respond to
 *	any external inputs.
 */
static bool remove_rights(void)
{
	HANDLE hToken;

	if ( ! OpenProcessToken( GetCurrentProcess(),
		TOKEN_QUERY | TOKEN_ADJUST_PRIVILEGES,
		&hToken) )
	{
		_tprintf(_T("Cannot get process token [%ld]\n"), GetLastError());
		return false;
	}

        bool rc = !!AdjustTokenPrivileges( hToken,
                TRUE,                   // bDisableAll Privs
                0, 0,                   // newstate + len
                0, 0);                  // prevstate + len

	CloseHandle(hToken);

	return rc;
}

/*
 * show_help()
 *
 *	Show a brief help listing to the standard output, but allow the
 *	caller to exit the program.
 */

static void show_help(const TCHAR *argv0)
{
static const char *const helptext[] = {

"  -help         Show this brief help listing",
"  -version      Show version information, then exit",
"  -install      Install the service using this .EXE (but does not start)",
"                Run \"NET START " SERVICENAME "\" after -install",
"  -installstart Install and then start the service",
"  -remove       Stops the service (if necessary), then uninstalls it",
"  -debug        Enables a bit more debugging to the dbmon log",

0 // ENDMARKER
};
	puts(fullVersion);

	putchar('\n');

	_tprintf(_T("Usage: %s [options]\n"), argv0);

	putchar('\n');

	for (const char * const *txtp = helptext; *txtp; txtp++ )
	{
		puts(*txtp);
	}
}


int __cdecl internal_main(int argc, TCHAR **argv)
{
	const TCHAR *argv0 = argv[0];

//	odprintf(_T("enter internal_main(argv[0] = %s, argc=%d)"),
//		argv[0],
//		argc);

	/*----------------------------------------------------------------
	 * Process command line
	 *
	 * --debug		enable full debug
	 */
	BOOL requestDebug = FALSE;

	for (int i = 1; i < argc; i++ )
	{
	TCHAR	*arg = argv[i];

		// coalesce multiple dashes into just one
		while (arg[0] == '-' && arg[1] == '-' )
			arg++;

		if ( _tcscmp(arg, _T("-help")) == 0 )
		{
			show_help(argv0);

			return EXIT_SUCCESS;
		}

		if ( _tcscmp(arg, _T("-debug")) == 0 )
		{
			requestDebug = TRUE;
		}

		else if ( _tcscmp(arg, _T("-install")) == 0 )
		{
			install_service(SERVICE_AUTO_START, false);
			return EXIT_SUCCESS;
		}
		else if ( _tcscmp(arg, _T("-installstart")) == 0 )
		{
			install_service(SERVICE_AUTO_START, true);
			return EXIT_SUCCESS;
		}
		else if ( _tcscmp(arg, _T("-remove")) == 0 )
		{
			remove_service();
			return EXIT_SUCCESS;
		}
		else if ( _tcscmp(arg, _T("-version")) == 0 )
		{
			puts(fullVersion);
			return EXIT_SUCCESS;
		}
		else
		{
			_tprintf(_T("ERROR: \"%s\" is invalid cmdline param (try -help)\n"),
				arg);
			return EXIT_FAILURE;
		}
	}

	// if we are not running as a service, the user should not be
	// able to just run the command line.

	if ( ! runningAsService )
	{
		show_help(argv0);

		return EXIT_SUCCESS;
	}

	Debug = requestDebug;

//	odprintf(_T("OPTIONS SET:"));
//	odprintf(_T("-debug          = %s"), printable(Debug));

	/*----------------------------------------------------------------
	 * CREATE SHUTDOWN EVENT
	 *
	 * When run in the foreground, the user can either hit control-C
	 * or press ESCAPE to exit, but when running as a service we must
	 * respond to a ServiceControl message from the SCM. Since we are
	 * generally waiting on the handle for the child process, we need
	 * to use some other waitable object to indicate shutdown. That
	 * is this.
	 *
	 * When the SCM calls ServiceControl (in a different thread from
	 * the main one) we simply "set" this event and our wait wakes up.
	 * Only in the service mode do we actually use this.
	 *
	 * Once created, the handle is marked as not available to the
	 * child process via inheritance. No point in them seeing more
	 * handles than they deserve...
	 */
	if ( runningAsService )
	{
		hShutdownEvent = CreateEvent(0,		// security attrs,
					FALSE,		// manual reset
					FALSE,		// initial state
					0 );		// name

		if ( hShutdownEvent == 0
		  || hShutdownEvent == INVALID_HANDLE_VALUE )
		{
			odprintf(_T("ERROR: cannot create shutdown event [%ld]"),
				GetLastError());
			hShutdownEvent = 0;
		}

		SetHandleInformation(hShutdownEvent, HANDLE_FLAG_INHERIT, 0);
	}

	mark_running();

	/*----------------------------------------------------------------
	 * DO REAL SERVICE WORK
	 *
	 * This creates a UDP and a TCP socket (one of each) for binding
	 * to port 37.
	 */

	// init winsock

	WORD    wVersion = MAKEWORD(2,2);
	WSADATA wsaData;
	DWORD   err;

	if ( (err = (DWORD)WSAStartup(wVersion, &wsaData)) != 0 )
	{
		odprintf(_T("ERROR: initializing Winsock [err#ld]"), err);
		// failure
	}


	SOCKET   tcpsocket = INVALID_SOCKET;
	WSAEVENT hTCPevent  = 0;
	tcpsocket = setupsocket(TRUE,  37, Backlog, &hTCPevent);

#ifdef SUPPORT_UDP
	SOCKET   udpsocket = INVALID_SOCKET;
	WSAEVENT hUDPevent  = 0;
	udpsocket = setupsocket(FALSE, 37, Backlog, &hUDPevent);
#endif

	remove_rights();


#define EVENT_SHUTDOWN   1
#define EVENT_TCP_LISTEN 2
#define EVENT_UDP_LISTEN 3

	BOOL bKeepGoing = TRUE;

	while ( bKeepGoing )
	{
	HANDLE	hArray[3];	// array of handles
	DWORD	dwXlate[3];	// translate return codes
	DWORD	nHandles = 0;

		if ( hShutdownEvent )
		{
			hArray [nHandles  ] = hShutdownEvent;
			dwXlate[nHandles++] = EVENT_SHUTDOWN;
		}

		if ( hTCPevent )
		{
			hArray [nHandles  ] = (HANDLE)hTCPevent;
			dwXlate[nHandles++] = EVENT_TCP_LISTEN;
		}

#if SUPPORT_UDP
		if ( hUDPevent )
		{
			hArray [nHandles  ] = (HANDLE)hUDPevent;
			dwXlate[nHandles++] = EVENT_UDP_LISTEN;
		}
#endif

		// add in any active sockets?

		if ( nHandles == 0 )
		{
			// huh? error
		}

		const DWORD dwrc = WaitForMultipleObjects(
			nHandles,		// count
			hArray,			// list of handles
			FALSE,			// don't wait for *all*
			INFINITE);		// wait forever

		odprintf(_T("Wait Multiple returned %d"), dwrc);

		if ( dwrc >= WAIT_OBJECT_0  &&  dwrc < (WAIT_OBJECT_0 + nHandles) )
		{
			// switch on the event

			switch ( dwXlate[dwrc] )
			{
			  case EVENT_SHUTDOWN:
				odprintf("Got shutdown event");
				bKeepGoing = FALSE;
				break;

			  case EVENT_TCP_LISTEN:
				tcpevent(tcpsocket);
				WSAResetEvent(hTCPevent);
				break;

#if SUPPORT_UDP
			  case EVENT_UDP_LISTEN:
				udpevent(udpsocket);
				WSAResetEvent(hUDPevent);
				break;
#endif
			}
		}
	}

	return 0;

}

/*
 * main()
 *
 *	This is the main entry point for service and standalone versions.
 *	We immediately determine the program name (for debugging) and
 *	figure out how we started.
 */
int __cdecl _tmain(int argc, TCHAR **argv)
{
	int rc = 0;

//	odprintf(_T("Entering main(%s)"), argv[0]);

	if ( (ProgName = _tcsrchr(argv[0], '\\')) != 0
	  || (ProgName = _tcsrchr(argv[0], '/' )) != 0 )
	{
		ProgName++;
	}
	else
	{
		ProgName = argv[0];
	}

	/*----------------------------------------------------------------
	 * See if we're running as a service or not.
	 */

	runningAsService =	GetStdHandle(STD_INPUT_HANDLE) == 0
			    ||	GetStdHandle(STD_OUTPUT_HANDLE) == 0
			    ||	GetStdHandle(STD_ERROR_HANDLE)  == 0;

	if ( runningAsService )
	{
		odprintf(_T("About to lauch service"));
		rc = launch_service();
	}
	else
	{
		rc = internal_main(argc, argv);
	}

	if ( Debug ) odprintf(_T("main exits %d"), rc);

	return rc;
}

