/*
 * $Id: //devel/tools/main/rfc868time/service.cpp#6 $
 *
 * written by :	Stephen J. Friedl
 *		Software Consultant
 *		steve@unixwiz.net
 *		1999-12-19
 *
 *	==================================================================
 *	NOTE: we've been using this module for NT service code for a long
 *	time, modifying it to fit whatever the current service is. Not all
 *	of this is required by RFC868time.
 *	==================================================================
 *
 *	This module manages all the NT Service-related code. Because this
 *	program can also run as a standalone program, we must take care
 *	to not build the service-ness into every part of the program.
 *
 *	Running as a Win32 service is straightforward, but there is a
 *	protocol required. A "service" can be one single executable, or
 *	it can be multiple services inside an EXE (each of which can be
 *	started and stopped independently). We take the former approach
 *	as this is all we need for now.
 *
 *	Services are programmed into the registry, and each EXE can have
 *	one or more logical services inside. When a service is started,
 *	the Service Control Manager (the SCM, in SERVICES.EXE) checks to
 *	see if the required EXE is already running. If not, it launches it.
 *
 *	The first thing that main() should do is start the Service Control
 *	Dispatcher, a worker thread that communicates via RPC to the main
 *	service controller. The init routine is given our local table of
 *	all services contained inside the EXE, and each entry has a "main"
 *	entry point.
 *
 *	When the dispatcher calls the appropriate ServiceMain, we immediately
 *	register a second function, ServiceControl, which the dispatcher uses
 *	to manipulate this service. We receive a HANDLE back from this
 *	routine, and we use it to inform the scheduler of our progress. In
 *	particular there is a STATUS object that lets it know what we are
 *	doing.
 *
 *	After we've registered our service, we inform the SCM that that we
 *	are starting by filling in a SERVICE_STATUS object and providing
 *	this info via the SetServiceStatus() function. We transit through
 *	several states during our rise and fall.
 *
 *	Once we're running, we go about our business, and we wait until
 *	our ServiceControl function is called by the SCM, and the only one
 *	we really care about is
 */
#include <windows.h>
#include <tchar.h>
#include <stdlib.h>
#include "defs.h"

static TCHAR ServiceName[] = _T(SERVICENAME);
static TCHAR ServiceDesc[] = _T(SERVICEDESC);


static SERVICE_STATUS		sStatus;
static SERVICE_STATUS_HANDLE	hServiceStatus = 0;

/*------------------------------------------------------------------------
 * This ShuttingDown flag is set by the ServiceControl function when we
 * receive the stop message from the SCM. In addition to setting the
 * proper stop event, we set this flag to let the main loop know what
 * we're supposed to be doing: are we looping after child process failure,
 * or do we exit?
 */
BOOL	bShuttingDown  = FALSE;

/*
 * set_status()
 *
 *	This is just a front-end function that lets us announce our status to
 *	the debugging log. We simply note the STOPPED/RUNNING/etc. status
 *	and then call the real SetService with the info. This relies on the
 *	*global* sStatus object, which is probably not that great an idea.
 */
static void set_status()
{
	const TCHAR *msg = 0;

	switch ( sStatus.dwCurrentState )
	{
	  case SERVICE_STOPPED:		msg = _T("STOPPED");		break;
	  case SERVICE_START_PENDING:	msg = _T("START_PENDING");	break;
	  case SERVICE_STOP_PENDING:	msg = _T("STOP_PENDING");	break;
	  case SERVICE_RUNNING:		msg = _T("RUNNING");		break;
	  case SERVICE_PAUSED:		msg = _T("PAUSED");		break;
	  case SERVICE_PAUSE_PENDING:	msg = _T("PAUSE_PENDING");	break;
	  case SERVICE_CONTINUE_PENDING:msg = _T("CONTINUE_PENDING");	break;
	  default:			msg = _T("-?-");		break;
	}

	if ( Debug ) odprintf(_T("set_status(%s)"), msg);

	SetServiceStatus(hServiceStatus, &sStatus);
}

/*
 * mark_stopped()
 * mark_running()
 *
 *	These functions inform the SCM of the simple status values that
 *	are suggested by the name, and (in the case of stopped), we pass
 *	the exit code too. These are simply wrapper functions for a bit
 *	of convenience.
 */
static void __stdcall mark_stopped(DWORD excode)
{
	sStatus.dwCurrentState  = SERVICE_STOPPED;
	sStatus.dwWin32ExitCode = excode;

	set_status();
}

void __stdcall mark_running()
{
	sStatus.dwCurrentState = SERVICE_RUNNING;

	set_status();
}

/*
 * announce_ServiceControl_action()
 *
 *	Given the dwControl action from the SCM passed to our ServiceControl
 *	function, announce the actual message to the debugging log. This
 *	has no purpose other than debugging, so it can be bypassed if the
 *	user doesn't care for the logging.
 */
static void announce_ServiceControl_action(DWORD dwControl)
{
	const TCHAR *msg = 0;

	switch (dwControl)
	{
	  case SERVICE_CONTROL_PAUSE:		msg = _T("PAUSE");	break;
	  case SERVICE_CONTROL_CONTINUE:	msg = _T("CONTINUE");	break;
	  case SERVICE_CONTROL_INTERROGATE:	msg = _T("INTERROGATE");break;
	  case SERVICE_CONTROL_STOP:		msg = _T("STOP");	break;
	  case SERVICE_CONTROL_SHUTDOWN:	msg = _T("SHUTDOWN");	break;
	  default:				msg = _T("-?-"); 	break;
	}

	if ( Debug ) odprintf(_T("enter ServiceControl(%s)"), msg);
}

/*
 * ServiceControl()
 *
 *	This is called by the Service Control Manager outside of the main
 *	thread of execution to make things happen with us. We only really
 *	care about stopping the service -- there is not really any such thing
 *	as "pause" and "continue" -- and when we get this we set the Event
 *	object that signals the main thread to stop.
 */
static void __stdcall ServiceControl(DWORD dwControl)
{
	announce_ServiceControl_action(dwControl);

	switch (dwControl)
	{
	  case SERVICE_CONTROL_SHUTDOWN:
	  case SERVICE_CONTROL_STOP:

		// signal the shutdown right now
		sStatus.dwCurrentState  = SERVICE_STOP_PENDING;
		sStatus.dwCheckPoint    = 0;
		sStatus.dwWaitHint      = 1000;		// 1 second !
		sStatus.dwWin32ExitCode = 0;
		set_status();

		bShuttingDown = TRUE;
		SetEvent(hShutdownEvent);
		break;

	  case SERVICE_CONTROL_INTERROGATE:
	  default:
		sStatus.dwCheckPoint = 0;
		set_status();

		break;
	}

}

/*
 * ServiceMain()
 *
 *	This is called by the Service Control Dispatcher to launch our
 *	service, and ultimately we want to start the same "internal" main
 *	that we do for the standalone version.
 *
 *	This function is never called directly. It is always provided in
 *	the Services[] table above as a structure member and called by
 *	the dispatcher.
 */
static void __stdcall ServiceMain(DWORD argc, TCHAR **argv)
{
	if ( Debug ) odprintf(_T("enter ServiceMain(%s)"), argv[0]);

	/*----------------------------------------------------------------
	 * The very first thing we must do is register a control routine
	 * and fetch a handle to the dispatcher. This handle uniquely
	 * identifies this service and allows us to give status feedback
	 * to the dispatcher.
	 *
	 * We don't know what to do if we get a failure from this: it
	 * should never happen, and it is probably quite fatal.
	 */
	hServiceStatus = RegisterServiceCtrlHandler(argv[0], ServiceControl);

	if (hServiceStatus == 0)
	{
		odprintf(_T("Cannot RegisterServiceCtrlHandler [%ld]"),
			GetLastError());
		return;
	}

	/*----------------------------------------------------------------
	 * Now we have a handle to the status object, so we "fill in the
	 * blanks" to let the dispatcher know about us.
	 *
	 * The service type is "WIN32 OWN PROCESS", which says that this
	 * .EXE has only one service. It's possible to provide multiple
	 * logical services under a single process, but this one does not
	 * do that.
	 *
	 * CurrentState says we're launching our service, and we say it's
	 * running after the main process has launched and is running for
	 * (say) one second.
	 *
	 * We only accept STOP and SHUTDOWN controls, and after our
	 * telling so prevents the dispatcher from allowing any other
	 * kinds of controls. Sorry, no PAUSE Or CONTINUE.
	 */
	sStatus.dwServiceType			= SERVICE_WIN32_OWN_PROCESS;
	sStatus.dwCurrentState			= SERVICE_START_PENDING;
	sStatus.dwControlsAccepted		= SERVICE_ACCEPT_STOP
						| SERVICE_ACCEPT_SHUTDOWN;
	sStatus.dwWin32ExitCode			= 0;
	sStatus.dwServiceSpecificExitCode	= 0;
	sStatus.dwCheckPoint			= 0;
	sStatus.dwWaitHint			= 1 * 1000;

	set_status();

	DWORD dwrc;

	__try
	{
		if ( Debug ) odprintf(_T("Launching internal_main()..."));
		dwrc = internal_main(argc, argv);
	}
	__except ( 1 )
	{
		odprintf(_T("ERROR: Got exception in internal_main()!"));
		dwrc = EXIT_FAILURE;
	}
	if ( Debug ) odprintf(_T("internal_main returns %d"), dwrc);

	if ( sStatus.dwCurrentState != SERVICE_STOPPED )
		mark_stopped(dwrc);

	if ( Debug ) odprintf(_T("ServiceMain returns"));
}

/*
 * launch_service()
 *
 *	This performs the startup of the service from the main routine after
 *	we have decided that we really are a service. We then rely on the
 *	call to ServiceMain() to actually lanch the real program.
 *
 *	NOTE: this really does rely on the caller having a good idea that
 *	we are launching as a service, because in "standalone" mode we
 *	typically get a long delay plus timeout. We do not know of another
 *	way to get around this issue, so be careful.
 */
int __stdcall launch_service()
{

	static SERVICE_TABLE_ENTRY	Services[] = {
		{ ServiceName,	ServiceMain },
		{ 0 }
	};

	int	rc;

	if ( Debug ) odprintf(_T("Starting Service Control Dispatcher..."));

	if ( ! StartServiceCtrlDispatcher(Services) )
	{
		odprintf(_T("ERROR: cannot start service [%ld]"),
			GetLastError() );
		rc = EXIT_FAILURE;
	}
	else
		rc = EXIT_SUCCESS;

	if ( Debug ) odprintf(_T("launch_service returns %d"), rc);

	return rc;
}

/*
 * printable_starttype()
 *
 *	Return a printable version of the service starting type.
 */

static const TCHAR *printable_starttype(DWORD dwStartType)
{
        switch (dwStartType)
        {
          STRCASE(SERVICE_AUTO_START);
          STRCASE(SERVICE_BOOT_START);
          STRCASE(SERVICE_DEMAND_START);
          STRCASE(SERVICE_DISABLED);
          STRCASE(SERVICE_SYSTEM_START);
          default: return _T("SERVICE_???");
        }
}

#if 0
static const TCHAR *printable_servicecontrol(DWORD dwControl)
{
        switch (dwControl)
        {
          STRCASE(SERVICE_CONTROL_STOP);
          STRCASE(SERVICE_CONTROL_PAUSE);
          STRCASE(SERVICE_CONTROL_CONTINUE);
          STRCASE(SERVICE_CONTROL_INTERROGATE);
          STRCASE(SERVICE_CONTROL_SHUTDOWN);
          STRCASE(SERVICE_CONTROL_PARAMCHANGE);
          STRCASE(SERVICE_CONTROL_NETBINDADD);
          STRCASE(SERVICE_CONTROL_NETBINDREMOVE);
          STRCASE(SERVICE_CONTROL_NETBINDENABLE);
          STRCASE(SERVICE_CONTROL_NETBINDDISABLE);

          default: return _T("SERVICE_CONTROL_???");
        }
}
#endif

/*
 * printable_state()
 *
 *      Given a state that we're transitioning to, return a printable
 *      version of it. This is only for reporting purposes.
 */
static const TCHAR *printable_state(DWORD dwState)
{
        switch (dwState)
        {
          STRCASE(SERVICE_STOPPED);
          STRCASE(SERVICE_START_PENDING);
          STRCASE(SERVICE_STOP_PENDING);
          STRCASE(SERVICE_RUNNING);
          STRCASE(SERVICE_CONTINUE_PENDING);
          STRCASE(SERVICE_PAUSE_PENDING);
          STRCASE(SERVICE_PAUSED);

          default: return _T("SERVICE_???");
        }
}



/*
 * open_scm()
 *
 *      This attempts to connect to the Service Controller Manager (with
 *      given access mask), and if we're not able to do it, it exits
 *      with failure. This is really only a helper...
 */
static SC_HANDLE open_scm(DWORD dwAccess)
{
        SC_HANDLE hSCManager = OpenSCManager(NULL, NULL, dwAccess);

        if ( hSCManager == 0 )
        {
		const DWORD err = GetLastError();

		if ( err == ERROR_ACCESS_DENIED )
			_tprintf( _T("ERROR: cannot OpenScManager (must be admin!)") );
		else
			_tprintf( _T("ERROR: cannot OpenScManager [err=%ld]\n"), err );
                exit(EXIT_FAILURE);
        }

        return hSCManager;
}

/*
 * supportsLocalService()
 *
 *	Figure out if this OS supports the LocalService service account type
 *	or not. Support started in XP & 2003, but NOT in 2000. This means we
 *	have to key on the major/minor numbers to find out.
 *
 *		6.0	Windows Server "Longhorn" 	
 *		6.0	Windows Vista 	
 *		5.2	Windows Server 2003 R2 	
 *		5.2	Windows Server 2003 	
 *		5.1	Windows XP 	
 *		5.0	Windows 2000 
 */
static BOOL supportsLocalService(void)
{
	OSVERSIONINFO osver;
	ZeroMemory(&osver, sizeof osver);
	osver.dwOSVersionInfoSize = sizeof osver;

	if ( ! GetVersionEx(&osver) ) return FALSE;

	// NT4? no way
	if ( osver.dwMajorVersion < 5 ) 	return FALSE;

	// Version 6 = Vista
	if ( osver.dwMajorVersion > 5 )		return TRUE;

	// 5.0 = win2000; 5.1 = XP
	if ( osver.dwMinorVersion > 0 )	return TRUE;

	return FALSE;
}


/*
 * install_service()
 *
 *	Install the service so it launches with Windows. The EXE is
 *	run from the *current location*, so be sure to park the
 *	binary where it will ultimately live.
 *
 *	REMINDER: this runs in user space, not service space.
 */
void install_service(DWORD dwStartType, bool dostart)
{
	/*----------------------------------------------------------------
	 * FIND OUR OWN PATH
	 *
	 * Figure out where this executable is located: this is where
	 * the service manager will later look for it to run it.
	 */
	TCHAR	szPath[512];

	if ( GetModuleFileName( NULL, szPath, COUNTOF(szPath) ) == 0 )
	{
		_tprintf(_T("Unable to install %s - %s\n"),
			ServiceName,
			prterr( GetLastError() ) );
		exit(EXIT_FAILURE);
	}

	/*----------------------------------------------------------------
	 * Figure out which kind of logon account we use for this service.
	 * We'd really rather use something with as few rights as possible,
	 * and LocalService is the lowest of the bunch.
	 */
	const TCHAR *acctname = 0;
	const TCHAR *acctpass = 0;

	if ( supportsLocalService() )
	{
		acctname = _T("NT AUTHORITY\\LocalService");
		acctpass = _T("-unused-");
	}

	SC_HANDLE hSCManager = open_scm(SC_MANAGER_CREATE_SERVICE);

	SC_HANDLE hService = CreateService(
		hSCManager,
		ServiceName,			// name of service
		ServiceDesc,			// name to display
		SERVICE_ALL_ACCESS,		// desired access
		SERVICE_WIN32_OWN_PROCESS,	// service type
		dwStartType,			// start type
		SERVICE_ERROR_NORMAL,		// error control type
		szPath,				// service's binary
		NULL,				// no load order grp
		NULL,				// no tag identifier
		_T(""),				// dependencies
		acctname,			// LocalSystem account
		acctpass);			// no password

	if ( hService == 0 )
	{
		_tprintf(_T("ERROR: cannot CreateService(%s) [%s]\n"),
			ServiceName,
			prterr() );
	}
	else
	{
		// full description.

		SERVICE_DESCRIPTION sdesc;
		sdesc.lpDescription = DETAILEDDESC;

		ChangeServiceConfig2(
			hService,
			SERVICE_CONFIG_DESCRIPTION,
			(void *) & sdesc );

		_tprintf(_T("Service %s installed (%s)\n"),
			ServiceName,
			printable_starttype(dwStartType) );
	}

	// start the service if requested
	if ( hService  &&  dostart )
	{
		if ( StartService(hService, NULL, 0 ) )
			_tprintf(_T("Service %s started\n"), ServiceName);
	}

	if ( hService )
		CloseServiceHandle(hService);

	CloseServiceHandle(hSCManager);
}

/*
 * remove_service()
 *
 *	Stop the service (if running), then remove it from the
 *	services database.
 *
 *	REMINDER: this runs in user space, not service space.
 */
void remove_service(void)
{
	SC_HANDLE hSCManager = open_scm( 0 );	/* odd access mask: hmmm */
	SC_HANDLE hService   = 0;

	__try
	{
		const DWORD dwAccess = DELETE
		                     | SERVICE_QUERY_STATUS
		                     | SERVICE_STOP;

		if ( (hService = OpenService(hSCManager,
					ServiceName, dwAccess)) == 0)
		{
			_tprintf(_T("OpenService(%s) failed - %s\n"),
				ServiceName,
				prterr() );
			__leave;
		}

		/* don't need this any more */
		CloseServiceHandle(hSCManager); hSCManager = 0;

		/*--------------------------------------------------------
		 * TRY TO STOP THE SERVICE
		 *
	 	 * This kicks the service to get it shutting down, but
		 * also waits until it *has* shut down (or at least when
		 * the SCM has given up).
		 *
		 * We suspect there are some timing issues going on here
		 * that could be heavily tuned, but we don't know what
		 * they are. Our DBMutex service stops so quickly that we
		 * really don't need to wait all that long anyway.
		 */
		SERVICE_STATUS sStat;

		if ( ControlService( hService, SERVICE_CONTROL_STOP, &sStat) )
		{
			_tprintf(_T("Stopping %s."), ServiceName);

			do
			{
				Sleep(100);
				_tprintf( _T(".") );

			} while ( QueryServiceStatus( hService, &sStat )
			     && sStat.dwCurrentState == SERVICE_STOP_PENDING );

			_tprintf( _T("\nService %s "), ServiceName);

			if ( sStat.dwCurrentState == SERVICE_STOPPED )
				_tprintf( _T("stopped as requested\n") );
			else
				_tprintf( _T("failed to stop (state=%s)\n"),
					printable_state(sStat.dwCurrentState));
		}

		/*--------------------------------------------------------
		 * REMOVE THE SERVICE
		 *
		 * At this point we're pretty sure it's stopped (at least
		 * we tried), so delete it.
		 */
		if ( DeleteService(hService) )
			_tprintf( _T("%s removed.\n"), ServiceName);
		else
			_tprintf( _T("DeleteService(%s) failed - %s\n"),
				ServiceName,
				prterr() );
	}
	__finally
	{
		if (hService)   CloseServiceHandle(hService);

		if (hSCManager) CloseServiceHandle(hSCManager);
	}
}

/*
 * prterr()
 *
 *	Given an error code, return a printable name of the ERROR_x
 *	macro for it. Yes, the long strings are available via the
 *	FormatMessage() function, but we've always preferred to do
 *	it this way.
 *
 *	Note that one can type "NET HELPMSG 1234" at a CMD prompt
 *	to do this lookup on the command line.
 */
const TCHAR * __stdcall prterr(DWORD err)
{
static TCHAR errbuf[256];

	switch (err)
	{
	  STRCASE(ERROR_SUCCESS);
	  STRCASE(ERROR_ACCESS_DENIED);
	  STRCASE(ERROR_SERVICE_EXISTS);
	  STRCASE(ERROR_SERVICE_DOES_NOT_EXIST);
	  STRCASE(ERROR_SERVICE_CANNOT_ACCEPT_CTRL);
	  STRCASE(ERROR_DEPENDENT_SERVICES_RUNNING);
	  STRCASE(ERROR_INVALID_SERVICE_CONTROL);
	  STRCASE(ERROR_SERVICE_REQUEST_TIMEOUT);
	  STRCASE(ERROR_SERVICE_NO_THREAD);
	  STRCASE(ERROR_SERVICE_DATABASE_LOCKED);
	  STRCASE(ERROR_SERVICE_ALREADY_RUNNING);
	  STRCASE(ERROR_INVALID_SERVICE_ACCOUNT);
	  STRCASE(ERROR_SERVICE_DISABLED);
	  STRCASE(ERROR_FAILED_SERVICE_CONTROLLER_CONNECT);
	  STRCASE(ERROR_EXCEPTION_IN_SERVICE);
	  STRCASE(ERROR_SERVICE_SPECIFIC_ERROR);
	  STRCASE(ERROR_INVALID_PARAMETER);
	  STRCASE(ERROR_INVALID_NAME);
	  STRCASE(ERROR_INVALID_HANDLE);
	  STRCASE(ERROR_DUPLICATE_SERVICE_NAME);
	  STRCASE(ERROR_CIRCULAR_DEPENDENCY);
	  STRCASE(ERROR_SERVICE_MARKED_FOR_DELETE);
	  STRCASE(ERROR_ALREADY_EXISTS);
	  STRCASE(ERROR_SHUTDOWN_IN_PROGRESS);
	  STRCASE(ERROR_DATABASE_DOES_NOT_EXIST);

	  default:
		_stprintf_s(errbuf, COUNTOF(errbuf), _T("err#%ld"), err);
		return errbuf;
	}
}

const TCHAR * __stdcall prterr(void)
{
	return prterr( GetLastError() );
}
