/*
 * $Id: //devel/tools/main/winforms/winforms.cpp#1 $
 *
 * written by :	Stephen J Friedl
 *		Tustin, California USA
 *		http://www.unixwiz.net/tools/
 *
 *	This program helps manage forms on a Windows print server, and
 *	its development was prompted by our painful efforts to resolve
 *	media issues with a PPD. Using the Adobe/Microsoft PostScript
 *	driver, we found that media just "disappeared", and this was
 *	getting very tiring: we had to *learn* how the whole system
 *	works.
 *
 *	The Windows print system maintains centralized "forms" management,
 *	and a "form" is a named, dimensioned piece of media. The
 *	EnumForms/AddForm/GetForm/DeleteForm API is used to manipulate
 *	these, but in practice they're all stored in the registry
 *
 *	HKEY_LOCAL_MACHINE\SYSTEM\CurrentControlSet\Control\Print\Forms
 *
 *	One can use 
 *		Control Panel
 *		-> Printers
 *		-> File:Server Properties
 *		-> Forms tab
 *
 *	but we wanted to be able to do queries and scriptable management.
 *	This was prompted by doing development of a PPD for use with
 *	the Microsoft/Adobe PostScript driver, and we had all kinds of
 *	problems with the forms.
 *
 *	Unlike winprinfo, this program doesn't actually need to know a
 *	specific printer: by opening NULL, it talks to the print *server*,
 *	which is where the forms are stored.
 *
 * COMMAND LINE
 * ------------
 *
 * --help	    Show a brief help listing
 *
 * --version	    Show the version number of this program.
 *
 * --printer=P	    Query using printer P: this is the name of the
 *                  printer *instance* (not the printer *type*) as
 *                  known by Windows.  Upstairs Laser", etc. We can
 *		    also use \\SERVER\PRINTER notation.
 *
 * --add=formspec   Add the given form name to the database
 *
 * --del=formname   Delete the given form name from the database
 *
 * --points	    Show paper dimensions in points
 *
 * --inches	    Show paper dimensions in inches
 *
 * A "formspec" is a quadruple value that gives the name, type, and
 * size of a media, and we can illustrate this by looking at some
 * of the types that we could define:
 *
 *	--add "Letter:B:215.90:279.40"		<-- builtin
 *	--add "A4:B:210.00:297.00"		<-- builtin
 *	--add "8x10:P:206.73:257.53"		<-- printer-defined
 */
#define UNICODE
#define _UNICODE
#define STRICT
#include <windows.h>
#include <tchar.h>
#include <assert.h>
#include <stdlib.h>
#include <stdio.h>
#include <stdarg.h>

#define SUPPORT_PER_PRINTER_FORMS	0

/*------------------------------------------------------------------------
 * VERSION HISTORY
 *
 * 2003-10-19 1.0.1 - Initial release
 */
static const TCHAR *VersionID =
_T("winforms 1.0.1 - 2003-10-19 - http://www.unixwiz.net/tools/");

/*------------------------------------------------------------------------
 * PORTABILITY HELPERS
 */
#define	UNUSED_PARAMETER(x)	((void)(x))

/*lint -sem(die, r_no) */


/*------------------------------------------------------------------------
 * UNIT OF MEASURE CONVERSIONS
 *
 * Virtually everything returned by the driver is in integral units (tenths
 * of a millimeter, 1000ths of a millimeter, etc.) but we use everything in
 * floating values. We also do lots of reporting in points, which is useful
 * when dealing with the PostScript PPD.
 */
#define MMTOPTS(x)	((x) * 72. / 25.4)
#define MMTOINCH(x)	((x) / 25.4)

#define TPR	_tprintf

static TCHAR	*PrinterName = 0,
                *PrintablePrinterName = 0;
static BOOL	Show_points     = FALSE,
		Show_inches     = FALSE,
		Do_enum_forms   = FALSE;

/* forms to add or delete */


static struct formaction {
	const TCHAR		*name;
	enum { ADD, SET, DEL } action;
	FORM_INFO_1		form1;
} Actions[256];
static int	nActions = 0;

static void  parse_forminfo(FORM_INFO_1 *pForm, TCHAR *str);
static int   format_formbits(TCHAR *outbuf, DWORD flags);
static void  die(const TCHAR *format, ...);
static const TCHAR *printable_error(DWORD err);

/*
 * getarg()
 *
 *	This is just a helper function that looks for a parameter to a
 *	command-line argument. If the user gave an arg in the form
 *
 *		--foo=bar
 *
 *	then argv[0] contains "--foo" and arg points to "bar". In this
 *	case we simply return arg.
 *
 *	But if the user gave 
 *
 *		--foo bar
 *
 *	then arg is NULL and we have to look ahead in argv to see if there
 *	is another word in the list. If so, we return it and increment the
 *	pointer for the next loop.
 */
static TCHAR *getarg(TCHAR *arg, TCHAR **&argv)
{
	if ( arg != 0 ) return arg;

	if ( argv[1] == 0 )
	{
		die( _T("ERROR: the \"%s\" option requires a parameter"),
			argv[0]);
	}

	return *++argv;
}

static void show_help(void)
{
static const TCHAR *const text[] = {
	_T(""),
	_T("Usage: winforms [options]"),
	_T(""),
	_T("  --help            Show this brief help listing"),
	_T("  --del=F           Delete form F"),
	_T("  --add=F:...       Add form F (with parameters)"),
	_T("  --printer=P       Query Windows printer P"),
	_T("  --points          Show paper dimensions in points"),
	_T("  --inches          Show paper dimensions in inches"),
	_T("  --noforms         Skip the EnumForms() step"),
	_T("  --showforms       Perform the EnumForms() step"),
	_T(""),
	_T("  Adding a form is done with a 'form spec' that's built"),
	_T("  from a single string: --add=\"NAME:flags:width:height\""),
	_T("  Dimensions are in millimeters (decimal OK)"),
	_T(""),
	_T("  If the $TESTPRINTER variable is found in the environment,"),
	_T("  that name is used as the default printer unless --printer"),
	_T("  is given."),
	_T(""),

	0 /* ENDMARKER */
};

	_putts(VersionID);

	for ( const TCHAR *const *pp = text; *pp; pp++ )
		TPR(_T("%s\n"), *pp);

	exit(EXIT_FAILURE);
}

static void enumerate_forms(HANDLE hPrinter);


int __cdecl _tmain(int argc, TCHAR **argv)
{
HANDLE	hPrinter = 0;

	UNUSED_PARAMETER(argc);

	/*----------------------------------------------------------------
	 * PARSE COMMAND LINE
	 *
	 * Look at the environment first to get the name of the printer,
	 * but we don't really need it anymore now that we query the local
	 * printserver via NULL.
	 */
	PrinterName = _tgetenv(_T("TESTPRINTER"));

	for ( argv++; *argv; argv++ )
	{
		/*--------------------------------------------------------
		 * If the user provided the parameter in "--foo=bar" form,
		 * split out the last part from the first. This gets rid
		 * of the '=' sign and makes the string compare on the first
		 * part easy (no need to count).
		 */
		TCHAR	*arg = 0;

#define ARGMATCH(a,b)	(_tcscmp((a), _T(b)) == 0)

		if ( (arg = _tcschr(*argv, '=')) != 0 )
		{
			*arg++ = '\0';
		}

		if ( ARGMATCH(*argv, "--help") )
		{
			show_help();
			return EXIT_SUCCESS;
		}
		else if ( ARGMATCH(*argv, "--version") )
		{
			_putts(VersionID);
			return EXIT_SUCCESS;
		}
		else if ( ARGMATCH(*argv, "--printer") )
		{
			PrinterName = getarg(arg, argv);
		}
		else if ( ARGMATCH(*argv, "--del") )
		{
			struct formaction *fa = &Actions[nActions++];

			fa->action = formaction::DEL;
			fa->name   = getarg(arg, argv);
		}
		else if ( ARGMATCH(*argv, "--add") )
		{
			struct formaction *fa = &Actions[nActions++];

			fa->action = formaction::ADD;
			parse_forminfo(&fa->form1, getarg(arg, argv));
			fa->name = fa->form1.pName;
		}
		else if ( ARGMATCH(*argv, "--set") )
		{
			struct formaction *fa = &Actions[nActions++];

			/* might be NAME/NAME:... or NAME:... */
			TCHAR *str = getarg(arg, argv);

			TCHAR *spec = _tcschr(str, '/');

			if ( spec )
			{
				*spec++ = '\0';
				parse_forminfo(&fa->form1, spec);
				fa->name = str;
			}
			else
			{
				parse_forminfo(&fa->form1, str);
				fa->name = fa->form1.pName;
			}

			fa->action = formaction::SET;
		}

		else if ( ARGMATCH(*argv, "--showforms") )
		{
			Do_enum_forms = TRUE;
		}

		else if ( ARGMATCH(*argv, "--noforms") )
		{
			Do_enum_forms = FALSE;
		}
		else if ( ARGMATCH(*argv, "--points") )
		{
			Show_points = TRUE;
		}
		else if ( ARGMATCH(*argv, "--nopoints") )
		{
			Show_points = FALSE;
		}
		else if ( ARGMATCH(*argv, "--inches") )
		{
			Show_inches = TRUE;
		}
		else if ( ARGMATCH(*argv, "--noinches") )
		{
			Show_inches = FALSE;
		}
		else
		{
			die( _T("ERROR: \"%s\" is an invalid cmdline option"),
				argv[0]);
		}
	}

	/*----------------------------------------------------------------
	 * SANITY CHECKING ON PARAMETERS
	 *
	 * We're allowed to provide a printer name, but there is not 
	 * really much point in that: by using NULL as the printer name,
	 * we are querying the local printer server.
	 */
	if ( PrinterName == 0 )
		PrintablePrinterName = _T("(local printserver)");
	else
		PrintablePrinterName = PrinterName;

	_putts(VersionID);

	TPR( _T("\nQuerying printer \"%s\"\n"), PrintablePrinterName);

	/*----------------------------------------------------------------
	 * OPEN THE PRINTER
	 *
	 * This call gives us a HANDLE to the printer which we use to get
	 * the DEVMODE and other things. It also serves to validate that
	 * we even got the printer name right: it's a fatal error if we
	 * cannot do it.
	 */
/*
	PRINTER_DEFAULTS pdfl;
	ZeroMemory(&pdfl, sizeof pdfl);
	pdfl.pDatatype     = 0;
	pdfl.pDevMode      = 0;
	pdfl.DesiredAccess = SERVER_ALL_ACCESS;
 */

	if ( ! OpenPrinter(PrinterName, &hPrinter, NULL) )
	{
		die(_T("ERROR: cannot OpenPrinter(%s) [%s]"),
			PrintablePrinterName,
			printable_error( GetLastError()) );
	}
	TPR(_T("Printer %s is open\n"), PrintablePrinterName);

	/* if not adding or deleting, then we're clearing enuming */

	if ( nActions == 0 )
		Do_enum_forms = TRUE;

	/*----------------------------------------------------------------
	 * Run through the list of "actions" and add/delete/set as needed.
	 */
	for (int i = 0; i < nActions; i++)
	{
		const struct formaction *fa = &Actions[i];

		const TCHAR *actionname = _T("???");
		BOOL rc = FALSE;

		switch (fa->action)
		{
		  case formaction::DEL:
			actionname = _T("DEL");
			rc = DeleteForm(hPrinter, const_cast<TCHAR *>(fa->name));
			break;

		  case formaction::ADD:
			actionname = _T("ADD");
			rc = AddForm(hPrinter, 1, (BYTE *)&fa->form1);
			break;

		  case formaction::SET:
			actionname = _T("SET");
			rc = SetForm(hPrinter, const_cast<TCHAR *>(fa->name),
				1, (BYTE *)&fa->form1);
			break;
		}

		if ( rc )
		{
			_tprintf( _T("%s of form %s ok\n"),
				actionname, fa->name);
		}
		else
		{
			_tprintf( _T("%s of form %s failed [err=%s]\n"),
				actionname, fa->name,
				printable_error( GetLastError() ) );
		}
	}

	/*----------------------------------------------------------------
	 * ENUMERATE FORMS
	 *
	 * We have a printer open, so use the EnumForms() API to get the
	 * full list of all the forms. As with most of the EnumXXX() calls,
	 * it's done in two steps: first we fetch the number of bytes
	 * required, allocate the space, then call it again to fill in
	 * that allocated space.
	 */
	if ( Do_enum_forms )
		enumerate_forms(hPrinter);

	return EXIT_SUCCESS;
}

/*
 * die()
 *
 *	Given a printf-style argument list, format the message to the
 *	standard error stream, append a newline, and exit with error
 *	status.
 */
static void  die(const TCHAR *format, ...)
{
va_list	args;

	va_start(args, format);
	_vftprintf(stderr, format,args);
	va_end(args);
	_fputtc('\n', stderr);

	exit(EXIT_FAILURE);
}

/*
 * printable_error()
 *
 *	Given an error number, return a printable version of it. We should
 *	(but don't) go through the trouble of looking up the error text
 *	via the FormatMessage() API: instead we just pick off a few of
 *	the most common errors and return the error # for the rest.
 *
 *	NOTE: not threadsafe!
 */
static const TCHAR *printable_error(DWORD err)
{
static TCHAR errbuf[256];

	switch (err)
	{
	  case ERROR_INVALID_PARAMETER:
		return _T("INVALID_PARAMETER");

	  case ERROR_FILE_EXISTS:
		return _T("FILE_EXISTS");

	  case ERROR_INVALID_FORM_NAME:
		return _T("INVALID_FORM_NAME");

	  case ERROR_INVALID_FORM_SIZE:
		return _T("INVALID_FORM_SIZE");

	  case ERROR_INVALID_PRINTER_NAME:
		return _T("INVALID_PRINTER_NAME");

	  case ERROR_ACCESS_DENIED:
		return _T("ACCESS_DENIED");

	  case ERROR_INVALID_HANDLE:
		return _T("INVALID_HANDLE");

	  default:	_stprintf(errbuf, _T("Err#%ld"), err);
			return errbuf;
	}
}

/*
 * format_formbits()
 *
 *	Given an output buffer and the "flags" field of an enumerated
 *	form, decode them as USER/BUILTIN/PRINTER. Return the number 
 *	of characters formatted to the output place.
 */
static int format_formbits(TCHAR *outbuf, DWORD flags)
{
	TCHAR 	*p = outbuf;

	switch ( flags )
	{
	  case FORM_USER:	*p++ = 'U';	break;
	  case FORM_BUILTIN:	*p++ = 'B';	break;
	  case FORM_PRINTER:	*p++ = 'P';	break;
	  default:		*p++ = '?';	break;
	}

	*p = '\0';

	return (int)(p - outbuf);
}

/*
 * enumerate_forms()
 *
 *	This enumerates all the forms in the system: we show the name
 *	and flags and dimensions.
 *
 *	NOTE: the FORM_INFO_1 structure contains the size in mm, plus
 *	the imageable area: we've found that the imageable area is 
 *	always of the full page, so it looks to us like it's simply
 *	aliasing the two size dimensions. In particular, it does NOT
 *	account for hardware printable margins: so we don't report
 *	them.
 */
static void enumerate_forms(HANDLE hPrinter)
{
	FORM_INFO_1 *pForm = 0;
	DWORD dwNeeded = 0, nForms = 0;

	if ( ! EnumForms(hPrinter, 1, 0, 0, &dwNeeded, &nForms)
	  && GetLastError() != ERROR_INSUFFICIENT_BUFFER )
	{
		die( _T("ERROR: cannot EnumForms [err=%s]"),
			printable_error( GetLastError() ) );
	}

	if ( (pForm = (FORM_INFO_1 *)malloc(dwNeeded)) == 0 )
	{
		die( _T("ERROR: cannot get %ld bytes for EnumForms"), dwNeeded);
	}

	if ( ! EnumForms(hPrinter, 1, (unsigned char *)pForm,
		dwNeeded, &dwNeeded, &nForms) )
	{
		die( _T("ERROR: cannot Enumforms [%ld]"), GetLastError() );
	}

	TPR( _T("\nPRINTSERVER FORMS (%d forms, %d bytes)\n"),
		nForms,
		dwNeeded);

	for (int i = 0; i < (int)nForms; i++ )
	{
	TCHAR	linebuf[256], *pline = linebuf;
	const FORM_INFO_1 *pf = pForm + i;
	double	sizemm[2];

		/* sizes are in thousandths of a mm */
		sizemm[0] = pf->Size.cx / 1000.;
		sizemm[1] = pf->Size.cy / 1000.;

		pline = linebuf;
		pline += _stprintf(pline, _T(" [%3d] "), i );
		pline += format_formbits(pline, pf->Flags);

		const int prefix = (int)(pline - linebuf);

		pline += _stprintf(pline, _T(" %8.3f %8.3f mm    %s"),
			sizemm[0],
			sizemm[1],
			pf->pName);

		_putts( linebuf );

		if ( Show_points )
		{
			TPR(_T("%*s %8.3f %8.3f pts\n"),
				prefix, _T(""),
				MMTOPTS(sizemm[0]),
				MMTOPTS(sizemm[1]));
		}

		if ( Show_inches )
		{
			TPR(_T("%*s %8.3f %8.3f in\n"),
				prefix, _T(""),
				MMTOINCH(sizemm[0]),
				MMTOINCH(sizemm[1]));
		}
	}

	free( pForm );
}

/*
 * strip()
 *
 *	Remove trailing whitespace from the given string, returning the
 *	input string.
 */
static TCHAR *strip(TCHAR *s)
{
	TCHAR *sv   = s;
	TCHAR *lnsp = 0;

	assert(s != 0);

        for ( TCHAR *q = s; *q; q++)
                if (!_istspace(*q))
                        lnsp = q;
        if ( lnsp )
                lnsp[1] = '\0';
        else
                *s = '\0';

	return sv;
}

/*
 * trims()
 *
 *	Given a string, remove trailing spaces and skip leading
 *	ones. If the string ends up being empty, return NULL.
 */
static TCHAR *trims(TCHAR *s)
{
	assert(s != 0);

	while ( _istspace(*s) )		/* leading spaces */
		s++;

	if (strip(s)[0] == '\0')	/* empty string? */
		s = 0;

	return s;
}

/*
 * parse_forminfo()
 *
 *	Given a form spec from the user
 *
 *		"name : type : mmwidth : mmheight"
 *
 *	parse the string into the FORM_INFO_1 structure. We *should*
 *	really make this be a simple parser returning true or false,
 *	but for this simple application we simply die() if we cannot
 *	parse a user's spec.
 */
static void parse_forminfo(FORM_INFO_1 *pForm, TCHAR *str)
{
	assert(pForm != 0);
	assert(str   != 0);

	const TCHAR *delim = _T(":");

	ZeroMemory(pForm, sizeof *pForm);

	if ( (pForm->pName = trims(_tcstok(str, delim))) == 0 )
	{
		die( _T("ERROR: form requires a name") );
	}

	/*----------------------------------------------------------------
	 * The second token is the type: user, builtin, or printer.
	 */
	TCHAR *tok = trims(_tcstok(0, delim));

	if ( _tcslen(tok) != 1 )
		die( _T("ERROR: invalid form type") );

	switch (tok[0])
	{
	  case 'u':
	  case 'U':	pForm->Flags = FORM_USER;    break;

	  case 'b':
	  case 'B':	pForm->Flags = FORM_BUILTIN; break;

	  case 'p':
	  case 'P':	pForm->Flags = FORM_PRINTER; break;

	  default:	die( _T("ERROR: %s is an invalid form type"), tok);
	}

	/*----------------------------------------------------------------
	 * The next two tokens are the width and height in millimeters.
	 * We don't do that much checking other than to insure that they
	 * 
	 */
	const TCHAR *sizex_str = trims( _tcstok(0, delim) );
	const TCHAR *sizey_str = trims( _tcstok(0, delim) );

	if ( sizex_str == 0 || sizey_str == 0 )
	{
		die( _T("ERROR: missing dimensions for form %s"),
			pForm->pName);
	}

	double sx = 0, sy = 0;

	if (  _stscanf(sizex_str, _T("%lf"), &sx) != 1 
	  ||  _stscanf(sizey_str, _T("%lf"), &sy) != 1  )
	{
		die( _T("Bad size string(s): \"%s\" \"%s\""),
			sizex_str, sizey_str);
	}

	pForm->ImageableArea.left   = 0;
	pForm->ImageableArea.top    = 0;
	pForm->ImageableArea.right  = pForm->Size.cx = (LONG)(sx * 1000);
	pForm->ImageableArea.bottom = pForm->Size.cy = (LONG)(sy * 1000);
}
