/*
 * $Id: //devel/tools/main/winprinfo/winprinfo.cpp#8 $
 *
 * written by :	Stephen J Friedl
 *		Tustin, California USA
 *		steve@unixwiz.net
 *
 *	This program is used to query everything we can find about a
 *	Windows printer. There are several kinds of APIs that we use
 *	to get this information, and we query them all. We created it
 *	to allow us to debug driver issues (specifically: getting a
 *	PPD right for a PostScript printer), but it may also be of
 *	use to an application developer.
 *
 *	Though some of these queries can be run on a Win 95/98/Me system,
 *	we have given zero thought to this. We are ONLY concerned about
 *	NT/2000/XP here, and as such have taken a strictly Unicode route.
 *	We think it's going to be a lot of work to backport this to the
 *	9x systems: good luck.
 *
 *	These are the things we query:
 *
 *	DEVMODE - these are the default parameters used for printing a job,
 *		and they're manipulated from the usual printer setup GUI.
 *		The DEVMODE itself is a single large object, and we use
 *		the DeviceCapabilities API call to extract the interesting
 *		parts.
 *
 *	GetDeviceCaps - These are queries of any kind of graphics device:
 *		printer and video. They are always small integral values
 *
 *	EnumForms - The printer server maintains a small database of "forms",
 *		which are more or less the names of different kinds of media.
 *
 *	PrinterDriver - we fetch all the info about the driver itself,
 *		including the path to the binaries and the support files
 *		used.
 *
 * NOTE:
 *
 *	This code was written with MSVC 6, and it requires the
 *	Feb 2003 SDK; newer compilers probably don't need the SDK.
 *
 *	It works fine with MSVC7 as well.
 *
 * COMMAND LINE
 * ------------
 *
 * --help	Show a brief help listing
 *
 * --enum	Enumerate all the local printers
 *
 * --full       Include more data on each enumeration
 *
 * --default    Show the default printer
 *
 * --enumserver=S  Enumerate server \\S
 *
 * --enumproviders  Enumerate local print providers
 *
 * --enumprocessors Enumerates all the print processors
 *
 * --version    Show the version number of this program.
 *
 * --printer=P	Query using printer P: this is the name of the printer
 *              *instance* (not the printer *type*) as known by Windows.
 *              "Upstairs Laser", etc. This can include a server name
 *		as in \\SERVER\PRINTER.
 *
 * --port=P	This lets us attach a port to the query, but we're not
 *		sure if this is even necessary. The port is the method
 *		by which the driver talks to the printer, and it can be
 *		something like "COM1:" or "EPR1:". The default is the
 *		empty string, which seems to work.
 *
 * --points	Show paper dimensions in points
 *
 * --inches	Show paper dimensions in inches
 *
 * --noforms	Skip the EnumForms part
 * --forms	Do the EnumForms part
 */
#define UNICODE
#define _UNICODE
#define STRICT
#include <windows.h>
#include <tchar.h>
#include <assert.h>
#include <stdlib.h>
#include <stdio.h>
#include <stdarg.h>

/*------------------------------------------------------------------------
 * VERSION HISTORY
 *
 * 2006-07-21 1.1.0 - Added enumeration of driver configuration
 *
 * 2005-05-24 1.0.9 - Added --default
 *
 * 2005-05-04 1.0.8 - Added --enumproviders & --enumprocessors
 *
 * 2005-02-15 1.0.7 - Added PRINTER_INFO_2 data
 *
 * 2004-04-15 1.0.6 - Added "top" and "left" to the margin reports
 *
 * 2003-10-13 1.0.5 - Lots of PC-Lint-suggested cleanups
 *
 * 2003-10-10 1.0.4 - Added remote-server support
 *
 * 2003-10-10 1.0.3 - Fixed error message when --printer option omitted
 *
 * 2003-10-09 1.0.2 - Added "--enum" parameter to enumerate printers
 *
 * 2003-10-02 1.0.1 - Initial release
 */
static const TCHAR *VersionID =
_T("winprinfo 1.1.0 - 2006-07-22 - http://www.unixwiz.net/tools/");

/*------------------------------------------------------------------------
 * PC-LINT stuff
 */
/*lint -sem(die, r_no) */

/*------------------------------------------------------------------------
 * PORTABILITY HELPERS
 */
#define	UNUSED_PARAMETER(x)	((void)(x))

#define	isizeof(x)	(int)(sizeof(x))

#define ASIZE(arr)	(sizeof(arr)/sizeof((arr)[0]))


static const TCHAR	*PrinterName = 0,
                        *PortName    = _T("");

/*------------------------------------------------------------------------
 * UNIT OF MEASURE CONVERSIONS
 *
 * Virtually everything returned by the driver is in integral units (tenths
 * of a millimeter, 1000ths of a millimeter, etc.) but we use everything in
 * floating values. We also do lots of reporting in points, which is useful
 * when dealing with the PostScript PPD.
 */
#define MMTOPTS(x)	((x) * 72. / 25.4)
#define MMTOINCH(x)	((x) / 25.4)

/*------------------------------------------------------------------------
 * DEVMODE STUFF
 *
 * Once we've open the printer and fetched the DEVMODE, we store it in
 * this global variable for reference throughout the program. We really
 */
static DEVMODE  *pDevmode    = 0;

static DEVMODE *fetch_DEVMODE(HANDLE hPrinter, const TCHAR *pname);


static const TCHAR *printable_DC_TRUETYPE(DWORD n);
static const TCHAR *printable_PRINTRATEUNIT(DWORD n);
static void show_string_array(int width, WORD cap, const TCHAR *capname);
static void show_point_array(WORD cap, const TCHAR *capname, const TCHAR *s);
static DWORD setupcap(WORD cap, int itemsize, void **pPtr);
static void enumerate_printers(const TCHAR *svr, BOOL bProviders);
static void enumerate_processors(const TCHAR *svr, const TCHAR *env);
static BYTE *allocGetPrinter(HANDLE hPrinter, DWORD Level);
static const TCHAR *printable_PINFO2_Attributes(DWORD n);
static BYTE *allocGetPrinterDriver(HANDLE hPrinter, TCHAR *env, DWORD Level);


/*------------------------------------------------------------------------
 * The DEVMODE contains information about the loaded papers, but the
 * information is split between two queries (one to get the paper names,
 * one to get the sizes). This lets us rendesvous them to a single
 * place.
 */
static struct paperinfo {
	TCHAR	name[64+1];
	double	sizemm[2];
} Papers[256];

static int	nPapers = 0;

static void get_paper_info(void);

#define TPR	_tprintf

static void  die(const TCHAR *format, ...);
static const TCHAR *printable_error(DWORD err);

static BOOL	Show_points     = FALSE,
		Show_inches     = FALSE,
		Do_EnumForms    = TRUE,
		Full            = FALSE;

/*
 * getarg()
 *
 *	This is just a helper function that looks for a parameter to a
 *	command-line argument. If the user gave an arg in the form
 *
 *		--foo=bar
 *
 *	then argv[0] contains "--foo" and arg points to "bar". In this
 *	case we simply return arg.
 *
 *	But if the user gave 
 *
 *		--foo bar
 *
 *	then arg is NULL and we have to look ahead in argv to see if there
 *	is another word in the list. If so, we return it and increment the
 *	pointer for the next loop.
 */
static TCHAR *getarg(TCHAR *arg, TCHAR **&argv)
{
	if ( arg != 0 ) return arg;

	if ( argv[1] == 0 )
	{
		die( _T("ERROR: the \"%s\" option requires a parameter"),
			argv[0]);
	}

	return *++argv;
}

static void show_help(void)
{
static const TCHAR *const text[] = {
	_T(""),
	_T("Usage: winprinfo [options]"),
	_T(""),
	_T("  --help            Show this brief help listing"),
	_T("  --default         Show the default printer"),
	_T("  --printer=P       Query local Windows printer P"),
	_T("  --printer=\\\\S\\P   Query printer P on server S"),
	_T("  --port=P          Query with port P"),
	_T("  --points          Show paper dimensions in points"),
	_T("  --inches          Show paper dimensions in inches"),
	_T("  --noforms         Skip the EnumForms() step"),
	_T("  --enum            Enumerate all the local printers"),
	_T("  --enumserver=S    Enumerate all the printers on server \\\\S"),
	_T("  --enumprocessors  Enumerate all the local print processors"),
	_T("  --enumproviders   Enumerate all the print providors"),
 	_T("  --full            Add more information to the enumeration"),
	_T(""),

	0 /* ENDMARKER */
};

	_putts(VersionID);

	for ( const TCHAR *const *pp = text; *pp; pp++ )
		_tprintf(_T("%s\n"), *pp);

	exit(EXIT_FAILURE);
}

static void show_extent(const TCHAR *fmt, double mmx, double mmy);
static void enumerate_forms(HANDLE hPrinter);
static void show_default(void);


int __cdecl _tmain(int argc, TCHAR **argv)
{
HANDLE	hPrinter  = 0;
int	Use_paper = 0;

	UNUSED_PARAMETER(argc);

	/*----------------------------------------------------------------
	 * PARSE COMMAND LINE
	 *
	 * We use a gnu-style parsing, and 
	 *
	 * --printer foo
	 */
	for ( argv++; *argv; argv++ )
	{
		/*--------------------------------------------------------
		 * If the user provided the parameter in "--foo=bar" form,
		 * split out the last part from the first. This gets rid
		 * of the '=' sign and makes the string compare on the first
		 * part easy (no need to count).
		 */
		TCHAR	*arg = 0;

#define ARGMATCH(a,b)	(_tcscmp((a), _T(b)) == 0)

		if ( (arg = _tcschr(*argv, '=')) != 0 )
		{
			*arg++ = '\0';
		}

		if ( ARGMATCH(*argv, "--help") )
		{
			show_help();
			return EXIT_SUCCESS;
		}
		else if ( ARGMATCH(*argv, "--enumserver") )
		{
			enumerate_printers( getarg(arg, argv), FALSE );

			return EXIT_SUCCESS;
		}
		else if ( ARGMATCH(*argv, "--full") )
		{
			Full = TRUE;
		}

		else if ( ARGMATCH(*argv, "--default") )
		{
			show_default();
			return EXIT_SUCCESS;
		}

		else if ( ARGMATCH(*argv, "--enum") )
		{
			enumerate_printers(0, FALSE);

			return EXIT_SUCCESS;
		}
		else if ( ARGMATCH(*argv, "--enumproviders") )
		{
			enumerate_printers(0, TRUE);

			return EXIT_SUCCESS;
		}
		else if ( ARGMATCH(*argv, "--version") )
		{
			_putts(VersionID);
			return EXIT_SUCCESS;
		}
		else if ( ARGMATCH(*argv, "--printer") )
		{
			PrinterName = getarg(arg, argv);
		}
		else if ( ARGMATCH(*argv, "--port") )
		{
			PortName = getarg(arg, argv);
		}
		else if ( ARGMATCH(*argv, "--points") )
		{
			Show_points = TRUE;
		}
		else if ( ARGMATCH(*argv, "--nopoints") )
		{
			Show_points = FALSE;
		}
		else if ( ARGMATCH(*argv, "--inches") )
		{
			Show_inches = TRUE;
		}
		else if ( ARGMATCH(*argv, "--noinches") )
		{
			Show_inches = FALSE;
		}
		else if ( ARGMATCH(*argv, "--forms") )
		{
			Do_EnumForms = TRUE;
		}
		else if ( ARGMATCH(*argv, "--noforms") )
		{
			Do_EnumForms = FALSE;
		}
		else if ( ARGMATCH(*argv, "--paper") )
		{
			Use_paper = _ttoi( getarg(arg, argv) );
		}
		else if ( ARGMATCH(*argv, "--enumprocessors") )
		{
			enumerate_processors(0, 0);

			return EXIT_SUCCESS;
		}

		else
		{
			die( _T("ERROR: \"%s\" is an invalid cmdline option"),
				argv[0]);
		}
	}

	/*----------------------------------------------------------------
	 * SANITY CHECKING ON PARAMETERS
	 *
	 * We positively MUST have a printer name, and it's a serious
	 * error to have this be missing. We've seen that the port name
 	 * doesn't seem to be needed, so we set it as a default above
	 * to the empty string.
	 */
	if ( PrinterName == 0 )
	{
		TPR(_T("ERROR: missing \"--printer printername\" param\n") );
		TPR(_T("\n"));
		show_help();
	}

	_putts(VersionID);

	_tprintf( _T("\nQuerying printer \"%s\" via port \"%s\"\n"),
		PrinterName,
		PortName);


	/*----------------------------------------------------------------
	 * OPEN THE PRINTER
	 *
	 * This call gives us a HANDLE to the printer which we use to get
	 * the DEVMODE and other things. It also serves to validate that
	 * we even got the printer name right: it's a fatal error if we
	 * cannot do it.
	 */
	if ( ! OpenPrinter(const_cast<TCHAR *>(PrinterName), &hPrinter, 0) )
	{
		die(_T("ERROR: cannot OpenPrinter(%s) [%s]"),
			PrinterName,
			printable_error( GetLastError()) );
	}
	TPR(_T("Printer %s is open\n"), PrinterName);

	/*----------------------------------------------------------------
	 * GET DRIVER INFO
	 *
	 * Ask the system all about he driver information, and show the
	 * user. The dependent files is a little tricky because we
	 * have to wander through the string looking for the double NUL.
	 */
	BYTE *pDriver4 = allocGetPrinterDriver(hPrinter, 0, 4);

	if ( pDriver4 )
	{
		DRIVER_INFO_4 *pdi = (DRIVER_INFO_4 *)pDriver4;

		_tprintf(_T("Printer Driver Information (DRIVER_INFO_4):\n"));
		_tprintf(_T(" cVersion         = %d\n"),   pdi->cVersion);
		_tprintf(_T(" pName            = %s\n"), pdi->pName);
		_tprintf(_T(" pEnvironment     = %s\n"), pdi->pEnvironment);
		_tprintf(_T(" pDriverPath      = %s\n"), pdi->pDriverPath);
  		_tprintf(_T(" pDataFile        = %s\n"), pdi->pDataFile);
  		_tprintf(_T(" pConfigFile      = %s\n"), pdi->pConfigFile);
  		_tprintf(_T(" pHelpFile        = %s\n"), pdi->pHelpFile);
  		_tprintf(_T(" pMonitorName     = %s\n"), pdi->pMonitorName);
  		_tprintf(_T(" pDefaultDataType = %s\n"), pdi->pDefaultDataType);

  		const TCHAR *pfx  = _T(" pDependentFiles  = ");
  		const TCHAR *pfx2 = _T("                    ");

		const TCHAR *p = pdi->pDependentFiles;

		while ( *p )
		{
			_tprintf(_T("%s%s\n"), pfx, p);

			pfx = pfx2;

			p += _tcslen(p) + 1;
		}
	}
	else
	{
		_tprintf(_T("Cannot GetPrinterDriver [err#%d]\n"), GetLastError() );
	}

	/*----------------------------------------------------------------
	 * FETCH DEVMODE
	 *
	 * The DEVMODE is a consolidation of a whole raft of printer
	 * parameters, but we're not strictly sure we need to get it.
	 */
#if 1
	pDevmode = fetch_DEVMODE(hPrinter, PrinterName);
#else
	pDevmode = 0;
#endif

	// update the devmode to pick a media?

	HDC hPrinterDC = CreateDC(0, PrinterName, 0, pDevmode);

	if ( hPrinterDC == 0 )
	{
		die( _T("ERROR: cannot CreateDC(%s,%s) [err=%ld]"),
			_T("NULL"), PrinterName, GetLastError() );
	}

	/*----------------------------------------------------------------
	 * Start fetching data from the devmode with DeviceCapabilities().
	 * "cap" is a token indicating a specific capability, such as
	 * DC_DRIVER or DC_COLLATE, that indicates something we want to
	 * know about.
	 *
	 * There are two kinds of queries: those that have a strictly
	 * numeric return, plus those that have a numeric return AND
	 * fill an array with information. We use both of them.
	 *
	 * The NUMCAP() macro is used for those calls that have no output
	 * component - they simply return an interesting value, and they
	 * are reported directly.
	 *
	 * The DCAP() macro is used when we *do* expect to have an
	 * output buffer, and we provide that buffer as a second param.
	 * BUT: that buffer must be large enough, so this requires two
	 * calls -- first we find out how many items will be returned,
	 * we allocate that space, then call it again with that buffer.
	 */
#define DCAP(cap, out)	DeviceCapabilities(\
                                PrinterName, \
                                PortName, \
                                (cap), \
                                (TCHAR *)(out), \
                                pDevmode )

#define NUMCAP(cap)	DCAP((cap), 0)
	LONG n;

	TPR(_T("DEVMODE PARAMETERS\n") );

	TPR(_T("  Numeric parameters:\n") );
//	TPR(_T("    DC_BINADJUST       %ld\n"), NUMCAP(DC_BINADJUST    ));
	TPR(_T("    DC_COLLATE         %ld\n"), NUMCAP(DC_COLLATE      ));
	TPR(_T("    DC_COLORDEVICE     %ld\n"), NUMCAP(DC_COLORDEVICE  ));
	TPR(_T("    DC_COPIES          %ld\n"), NUMCAP(DC_COPIES       ));
	TPR(_T("    DC_DRIVER          %ld\n"), NUMCAP(DC_DRIVER       ));
	TPR(_T("    DC_DUPLEX          %ld\n"), NUMCAP(DC_DUPLEX       ));
	TPR(_T("    DC_EMF_COMPLIANT   %ld\n"), NUMCAP(DC_EMF_COMPLIANT));
	TPR(_T("    DC_EXTRA           %ld\n"), NUMCAP(DC_EXTRA        ));
	TPR(_T("    DC_FIELDS          0x%lx\n"), NUMCAP(DC_FIELDS     ));
 
	n = NUMCAP(DC_MINEXTENT);
	show_extent( _T("    DC_MINEXTENT       "), LOWORD(n)/10., HIWORD(n)/10.);

	n = NUMCAP(DC_MAXEXTENT);
	show_extent( _T("    DC_MAXEXTENT       "), LOWORD(n)/10., HIWORD(n)/10.);

	TPR(_T("    DC_ORIENTATION     %ld\n"), NUMCAP(DC_ORIENTATION  ));
	TPR(_T("    DC_PRINTERMEM      %ld kbytes\n"), NUMCAP(DC_PRINTERMEM));
	TPR(_T("    DC_PRINTRATEPPM    %ld\n"), NUMCAP(DC_PRINTRATEPPM ));
	TPR(_T("    DC_PRINTRATE       %ld\n"), NUMCAP(DC_PRINTRATE    ));
	TPR(_T("    DC_PRINTRATEUNIT   %ld (%s)\n"),
		NUMCAP(DC_PRINTRATEUNIT),
		printable_PRINTRATEUNIT(NUMCAP(DC_PRINTRATEUNIT)));
	TPR(_T("    DC_SIZE            %ld\n"), NUMCAP(DC_SIZE         ));
	TPR(_T("    DC_STAPLE          %ld\n"), NUMCAP(DC_STAPLE       ));
	TPR(_T("    DC_VERSION         %ld\n"), NUMCAP(DC_VERSION      ));
	TPR(_T("    DC_TRUETYPE        %ld (%s)\n"),
		NUMCAP(DC_TRUETYPE),
		printable_DC_TRUETYPE(NUMCAP(DC_TRUETYPE)));

	/*----------------------------------------------------------------
	 * ARRAYS: These are string arrays whose sizes are defined in the
	 * API documentation - we simply reproduce them here.
	 */

	show_string_array(24, DC_BINNAMES,         _T("DC_BINNAMES"));
	show_string_array(64, DC_FILEDEPENDENCIES, _T("DC_FILEDEPENDENCIES"));
	show_string_array(64, DC_MEDIAREADY,       _T("DC_MEDIAREADY"));
	show_string_array(64, DC_MEDIATYPENAMES,   _T("DC_MEDIATYPENAMES"));
	show_string_array(32, DC_PERSONALITY,      _T("DC_PERSONALITY"));

	show_point_array( DC_ENUMRESOLUTIONS,
	              _T("DC_ENUMRESOLUTIONS"),
	              _T("dpi") );

	/*----------------------------------------------------------------
	 * PAPER INFORMATION
	 *
	 * The paper information is gathered in two steps because it hasj
	 * two arrays. "get_paper_info()" fetches 
	 */
	get_paper_info();

	TPR( _T("\n") );
	TPR( _T("PAPER SIZES FROM THE DEVMODE\n") );

	for (int i = 0; i < nPapers; i++)
	{
	const struct paperinfo	*p = &Papers[i];
	TCHAR		linebuf[256], *linep;

		linep = linebuf;

		linep += _stprintf(linep, _T(" [%2d] "), i);

		const int prefix = (int)(linep - linebuf);

		linep += _stprintf(linep, _T(" %7.2f %7.2f mm  %s"),
			p->sizemm[0],
			p->sizemm[1],
			p->name);

		_putts(linebuf);

		if ( Show_points )
		{
			_tprintf(_T("%*s %7.2f %7.2f pts\n"),
				prefix, _T(""),
				MMTOPTS(p->sizemm[0]),
				MMTOPTS(p->sizemm[1]) );
		}

		if ( Show_inches )
		{
			_tprintf(_T("%*s %7.2f %7.2f in\n"),
				prefix, _T(""),
				MMTOINCH(p->sizemm[0]),
				MMTOINCH(p->sizemm[1]) );
		}
	}

	if ( pDevmode )
	{
		free(pDevmode);
		pDevmode = 0;
	}


	/*----------------------------------------------------------------
	 * GetDeviceCaps
	 *
	 * These are generic graphics queries using the Device Context
	 * object. Each of these capabilities returns a single integer,
	 * and we simply report them all in turn.
	 *
	 * NOTE: we don't fetch *all* the capabilities because we don't 
	 * have a need for them all (some are for video devices only).
	 * We also don't have any kind of post-lookup function that can
	 * convert a bit-mapped result into a printable name: for instance,
	 * the TECHNOLOGY token can be DT_PLOTTER, DT_RASPRINTER, and
	 * several others. We really should provide a lookup function
	 * for these.
	 */
#define GCAP(x)	GetDeviceCaps(hPrinterDC, (x))

	TPR(_T("\n") );
	TPR(_T("DEVCAPS\n") );

	static const struct lookup { 
		const TCHAR	*name;
		int		token;
		const TCHAR	*sfx;
	} captable[] = {
#define CAPNAME(n)	_T(#n), n
{ CAPNAME(DRIVERVERSION),  	_T("     driver Version")       },
{ CAPNAME(TECHNOLOGY),          _T("     ") },
{ CAPNAME(PLANES),              _T("     number of color planes") },
{ CAPNAME(NUMBRUSHES),          _T("     number of device-specfic brushes") },
{ CAPNAME(NUMFONTS),            _T("     number of device-specfic fonts") },
{ CAPNAME(NUMPENS),             _T("     number of device-specfic pens") },
{ CAPNAME(NUMCOLORS),           _T("     number of device-specfic colors") },
{ CAPNAME(HORZSIZE),		_T("mm   width of image area")	},
{ CAPNAME(ASPECTX),             _T("     relative horz width of dev pixel") },
{ CAPNAME(ASPECTY),             _T("     relative vert width of dev pixel") },
{ CAPNAME(ASPECTXY),            _T("     relative diag width of dev pixel") },
{ CAPNAME(CLIPCAPS),		_T("     1=can clip to rectangle") },
{ CAPNAME(VERTSIZE),		_T("mm   height of image area")	},
{ CAPNAME(HORZRES),		_T("pix  width of image area")	},
{ CAPNAME(VERTRES),		_T("pix  height of image area")	},
{ CAPNAME(LOGPIXELSX),		_T("dpi  X resolution")		},
{ CAPNAME(LOGPIXELSY),		_T("dpi  Y resolution")		},
{ CAPNAME(PHYSICALWIDTH),	_T("pix  width of full paper")	},
{ CAPNAME(PHYSICALHEIGHT),	_T("pix  height of full paper")	},
{ CAPNAME(PHYSICALOFFSETX),	_T("pix  X hardware margin (left)") },
{ CAPNAME(PHYSICALOFFSETY),	_T("pix  Y hardware margin (top)")  },
{ CAPNAME(SCALINGFACTORX),	_T("     X scaling factor")     },
{ CAPNAME(SCALINGFACTORY),	_T("     Y scaling factor")     },

{ 0, 0, 0 } /* ENDMARKER */
};

	for ( const struct lookup *lp = captable; lp->name; lp++ )
	{
		_tprintf(_T("  %-20s = %6d %s\n"),
			lp->name,
			GCAP(lp->token),
			lp->sfx);
	}

	DeleteDC( hPrinterDC );
	hPrinterDC = 0;

	/*----------------------------------------------------------------
	 * ENUMERATE FORMS
	 *
	 * We have a printer open, so use the EnumForms() API to get the
	 * full list of all the forms. As with most of the EnumXXX() calls,
	 * it's done in two steps: first we fetch the number of bytes
	 * required, allocate the space, then call it again to fill in
	 * that allocated space.
	 */

	if ( Do_EnumForms )
	{
		enumerate_forms(hPrinter);
	}

	BYTE *pInfo2 = allocGetPrinter(hPrinter, 2);

	if (pInfo2)
	{
		PRINTER_INFO_2 *p = (PRINTER_INFO_2 *)pInfo2;

		_tprintf(_T("\nPRINTER_INFO_2 from spooler\n"));

#define N(x)	((x) ? (x) : _T("-null-"))

		_tprintf(_T("  ServerName      %s\n"), N(p->pServerName));
		_tprintf(_T("  PrinterName     %s\n"), N(p->pPrinterName));
		_tprintf(_T("  ShareName       %s\n"), N(p->pShareName));
		_tprintf(_T("  PortName        %s\n"), N(p->pPortName));
		_tprintf(_T("  DriverName      %s\n"), N(p->pDriverName));
		_tprintf(_T("  Comment         %s\n"), N(p->pComment));
		_tprintf(_T("  Location        %s\n"), N(p->pLocation));
		_tprintf(_T("  SepFile         %s\n"), N(p->pSepFile));
		_tprintf(_T("  PrintProcessor  %s\n"), N(p->pPrintProcessor));
		_tprintf(_T("  Datatype        %s\n"), N(p->pDatatype));
		_tprintf(_T("  Parameters      %s\n"), N(p->pParameters));
		_tprintf(_T("  Attributes      0x%08lx %s\n"),
			p->Attributes,
			printable_PINFO2_Attributes(p->Attributes));
		_tprintf(_T("  Priority        %ld\n"), p->Priority);
		_tprintf(_T("  DefaultPriority %ld\n"), p->DefaultPriority);
		_tprintf(_T("  StartTime       %02ld:%02ld\n"),
			p->StartTime / 60,
			p->StartTime % 60);
		_tprintf(_T("  UntilTime       %02ld:%02ld\n"),
			p->UntilTime / 60,
			p->UntilTime % 60);
		_tprintf(_T("  Status          0x%08lx\n"), p->Status);
		_tprintf(_T("  cJobs           0x%08lx\n"), p->cJobs);
		_tprintf(_T("  AveragePPM      %ld\n"),     p->AveragePPM);

		delete [] pInfo2;
	}



	ClosePrinter(hPrinter);


	return EXIT_SUCCESS;
}

/*
 * setupcap()
 *
 *	Given a capability token and the size of each item, fetch the
 *	item count and allocate the space for all the items. This does
 *	no allocation if count is zero or error, but otherwise we
 *	fetch the space too in the pointed-to parameter. This is strictly
 *	a helper to make the caller code easier.
 *
 *	Return is the # of items that the caller returned.
 */

static DWORD setupcap(WORD cap, int itemsize, void **pPtr)
{
	*pPtr = 0;

	DWORD nItems = NUMCAP(cap);

	if ( nItems == (DWORD)-1)
		nItems = 0;

	else if ( nItems > 0 )
	{
		/* add a small fudge factor for safety */
		const size_t nbytes = (size_t)( (nItems + 10) * itemsize );

		*pPtr = malloc(nbytes);

		if ( DCAP(cap, *pPtr) == (DWORD)-1 )
			return 0;
	}

	return nItems;
}

/*
 * show_string_array()
 *
 *	Given a capability macro (and associated string name), fetch
 *	the list of strings from the driver. We first query without
 *	any string pointer to get the count, then allocate the bytes
 *	required for the list. The strings are always allocated in
 *	units of {WIDTH} characters, and they're all NUL terminated
 *	unless they strings are fully {WIDTH} wide.
 */
static void show_string_array(int width, WORD cap, const TCHAR *capname)
{
	/*----------------------------------------------------------------
	 * Get the number of items found for this capability. If the 
	 * number of items is <= 0, then the count is all we need: we
	 * don't have to fetch the strings.
	 */
	TCHAR      *pItems = 0;
	const int   nItems = setupcap( cap,
	                               width*isizeof(TCHAR),
	                               (void **)&pItems );

	_tprintf( _T("\n") );
	_tprintf( _T("  %-20s %ld\n"), capname, nItems);

	if ( nItems <= 0 ) return;

	for (int i = 0; i < nItems; i++)
	{
		const TCHAR *s = &pItems[i*width];

		_tprintf(_T("    [%d] = %.*s\n"), i, width, s);
	}

	free(pItems);
}

static void show_point_array(WORD cap,
	const TCHAR *capname,
	const TCHAR *sfx )
{
	assert(capname != 0);

	POINT  *pItems = 0;
	DWORD   nItems = setupcap( cap,
	                         sizeof(POINT),
	                         (void **)&pItems );

	_tprintf( _T("\n") );
	_tprintf( _T("  %-20s %ld\n"), capname, nItems);

	for (DWORD i = 0; i < nItems; i++)
	{
		_tprintf(_T("    [%d]   x=%ld %s / y=%ld %s\n"),
			i,
			pItems[i].x, sfx,
			pItems[i].y, sfx);
	}

	if ( pItems ) free(pItems);
}

/*
 * die()
 *
 *	Given a printf-style argument list, format the message to the
 *	standard error stream, append a newline, and exit with error
 *	status.
 */
static void  die(const TCHAR *format, ...)
{
va_list	args;

	va_start(args, format);
	_vftprintf(stderr, format,args);
	va_end(args);
	_fputtc('\n', stderr);

	exit(EXIT_FAILURE);
}

/*
 * nstrcpy()
 *
 *	Perform a "strcpy" operation from source to destination, but return
 *	the # of characters copied (excluding the NUL byte).
 */
static size_t nstrcpy(TCHAR *dst, const TCHAR *src)
{
	assert(dst != 0);
	assert(src != 0);

	const TCHAR *dst_save = dst;

	while ( (*dst = *src++) != 0 )
		dst++;

        return (size_t)(dst - dst_save);
}

/*
 * printable_PRINTRATEUNIT()
 *
 */

static const TCHAR *printable_PRINTRATEUNIT(DWORD n)
{
	switch (n)
	{
	  case PRINTRATEUNIT_CPS:	return _T("chars/sec");
	  case PRINTRATEUNIT_IPM:	return _T("inches/min");
	  case PRINTRATEUNIT_LPM:	return _T("lines/min");
	  case PRINTRATEUNIT_PPM:	return _T("pages/min");
	  default:			return _T("?");
	}
}

/*
 * printable_DC_TRUETYPE()
 *
 *	Given a DWORD of a DC_TRUETYPE bitmap from DeviceCapabilities,
 *	return a string that represents the printable versions of the
 *	capabilities.
 *
 *	NOTE: this is not threadsafe!
 */
static const TCHAR *printable_DC_TRUETYPE(DWORD n)
{
static TCHAR buf[256], *p = buf;

	if ( n ==         0 )	return _T("-none-");
	if ( n == (DWORD)-1 )	return _T("-error-");

	buf[0] = '\0';

#define CHKVAL(token)	if ( n & (token) )  \
			{ n &= ~token;  p += nstrcpy(p, _T(" ") _T(#token)); }

	CHKVAL(DCTT_BITMAP);
	CHKVAL(DCTT_DOWNLOAD);
	CHKVAL(DCTT_DOWNLOAD_OUTLINE);
	CHKVAL(DCTT_SUBDEV);

#undef CHKVAL

	/* catch any leftovers */
	if ( n ) p += _stprintf(p, _T(" ??%d"), n);

	return buf + 1;
}

/*
 * get_paper_info()
 *
 *	Run through all the stuff we know about each of the available papers,
 *	which includes the name and the size in 0.1 millimeter units. These are
 *	put into the generic "paper info" structure for later reporting.
 */
static void get_paper_info(void)
{
	const int namewidth = 64;
	int i;

	/* get sizes */

	POINT *pPoints = 0;
	const int nPoints  = (int) setupcap( DC_PAPERSIZE,
	                               sizeof(POINT),
	                               (void **)&pPoints);

	if ( nPoints > nPapers ) nPapers = nPoints;

	for (i = 0; i < nPoints; i++)
	{
	struct paperinfo	*p = &Papers[i];

		p->sizemm[0] = pPoints[i].x / 10.;
		p->sizemm[1] = pPoints[i].y / 10.;
	}

	/* get names */

	TCHAR *pNames  = 0;
	const int nNames   = (int) setupcap( DC_PAPERNAMES,
	                               namewidth*sizeof(TCHAR),
	                               (void **)&pNames);

	if ( nNames > nPapers ) nPapers = nNames;

	for (i = 0; i < nNames; i++)
	{
	struct paperinfo	*p = &Papers[i];
	const TCHAR *s = &pNames[i*namewidth];

		memcpy(p->name, s, namewidth);
		p->name[namewidth] = '\0';
	}
}

/*
 * fetch_DEVMODE()
 *
 *	Given a HANDLE and name of a printer, fetch the DEVMODE for it.
 *	This is done in two steps: first we ask for the number of bytes
 *	required, allocate that memory, then actually fetch the memory.
 *
 *	Return is the pointer to the DEVMODE, which is allocated from
 *	the heap. There is no error return - if we can't get the data,
 *	we exit with error.
 *
 *	NOTE: the "pname" param really should be /const/, but the Win32
 *	API doesn't allow it. *bah*
 */
static DEVMODE *fetch_DEVMODE(HANDLE hPrinter, const TCHAR *pname)
{
DEVMODE	*pdm = 0;

	LONG n = DocumentProperties( 0,	// hWnd
		hPrinter,
		const_cast<TCHAR *>(pname),
		0,			// DEVMODE output
		0,			// DEVMODE input
		0 );			// mode

	if ( n <= 0 )
	{
		die( _T("ERROR: cannot get DEVMODE size [err=%ld]"),
			GetLastError() );
	}

	if ( (pdm = (DEVMODE *)malloc(n)) == 0 )
		die( _T("ERROR: cannot get %ld bytes for DEVMODE"), n);


	if ( DocumentProperties( 0,		// hWnd
		hPrinter,
		const_cast<TCHAR *>(pname),
		pdm,				// DEVMODE output
		0,				// DEVMODE Input
		DM_OUT_BUFFER ) != IDOK)	// mode
	{
		die( _T("ERROR: can't get DocumentProperties [%ld]"),
			GetLastError());
	}

	TPR(_T("fetched DEVMODE, %d bytes\n"), n);

	return pdm;
}

/*
 * printable_error()
 *
 *	Given an error number, return a printable version of it. We should
 *	(but don't) go through the trouble of looking up the error text
 *	via the FormatMessage() API: instead we just pick off a few of
 *	the most common errors and return the error # for the rest.
 *
 *	NOTE: not threadsafe!
 */
static const TCHAR *printable_error(DWORD err)
{
static TCHAR errbuf[256];

	switch (err)
	{
	  case ERROR_INVALID_PRINTER_NAME:
		return _T("INVALID_PRINTER_NAME");

	  case ERROR_ACCESS_DENIED:
		return _T("ACCESS_DENIED");

	  case ERROR_INVALID_HANDLE:
		return _T("INVALID_HANDLE");

	  default:	_stprintf(errbuf, _T("Err#%ld"), err);
			return errbuf;
	}
}

/*
 * show_extent()
 *
 *	This is used to show the min/max extent as reported from the
 *	early devmode capabilities. The native reports are always
 *	in tenths of millimeters, but the user can ask that they also
 *	be provided in points and/or inches.
 *
 *	The "prefix" is the leading part of the line, and the actual
 *	dimensions follow right after. For subsequent lines (if any),
 *	the text of the tag (DC_EXTENT) is replaced with dots to make
 *	it more clear that it's additional information on the same
 *	tag.
 *
 *	_____________________ prefix
 *	  DC_MINEXTENT       w= 147.9 h= 199.9 mm
 *	  ............       w= 419.2 h= 566.6 pts
 *	  ............       w=   5.8 h=   7.9 in
 */
static void show_extent(const TCHAR *pfx, double mmx, double mmy)
{
	/* MM */
	TPR( _T("%sw=%6.1f h=%6.1f mm\n"), pfx, mmx, mmy);

	TCHAR	pfx2[256];

	_tcscpy(pfx2, pfx);

	for (TCHAR *p = pfx2; *p; p++)
	{
		if ( *p == '_' || _istupper(*p) || _istdigit(*p))
		{
			*p = '.';
		}
	}

	if ( Show_points )
	{
		TPR( _T("%sw=%6.1f h=%6.1f pts\n"),
			pfx2,
			MMTOPTS(mmx),
			MMTOPTS(mmy) );
	}

	if ( Show_inches )
	{
		TPR( _T("%sw=%6.1f h=%6.1f in\n"),
			pfx2,
			MMTOINCH(mmx),
			MMTOINCH(mmy) );
	}
}

/*
 * format_formbits()
 *
 *	Given an output buffer and the "flags" field of an enumerated
 *	form, decode them as USER/BUILTIN/PRINTER. Return the number 
 *	of characters formatted to the output place.
 */
static int format_formbits(TCHAR *outbuf, DWORD flags)
{
	TCHAR 	*p = outbuf;

	switch ( flags )
	{
	  case FORM_USER:	*p++ = 'U'; break;
	  case FORM_BUILTIN:	*p++ = 'B'; break;
	  case FORM_PRINTER:	*p++ = 'P'; break;
	  default:		*p++ = '?'; break;
	}

	return (int)(p - outbuf);
}

/*
 * enumerate_forms()
 *
 *	This enumerates all the forms in the system: we show the name
 *	and flags and dimensions.
 *
 *	NOTE: the FORM_INFO_1 structure contains the size in mm, plus
 *	the imageable area: we've found that the imageable area is 
 *	always of the full page, so it looks to us like it's simply
 *	aliasing the two size dimensions. In particular, it does NOT
 *	account for hardware printable margins: so we don't report
 *	them.
 */
static void enumerate_forms(HANDLE hPrinter)
{
	FORM_INFO_1 *pForm = 0;
	DWORD dwNeeded = 0, nForms = 0;

	if ( ! EnumForms(hPrinter, 1, 0, 0, &dwNeeded, &nForms)
	  && GetLastError() != ERROR_INSUFFICIENT_BUFFER )
	{
		die( _T("ERROR: cannot EnumForms [err=%s] [ln=%d]"),
			printable_error(GetLastError()),
			__LINE__);
	}

	if ( dwNeeded == 0 )
	{
		die( _T("ERROR: no forms defined in the system!") );
	}

	if ( (pForm = (FORM_INFO_1 *)malloc(dwNeeded)) == 0 )
	{
		die( _T("ERROR: cannot get %d bytes for EnumForms"), dwNeeded);
	}

	if ( ! EnumForms(hPrinter, 1, (unsigned char *)pForm,
		dwNeeded, &dwNeeded, &nForms) )
	{
		die( _T("ERROR: cannot Enumforms [%ld]"), GetLastError() );
	}

	_tprintf( _T("\nPRINTSERVER FORMS (%d forms, %d bytes)\n"),
		nForms,
		dwNeeded);

	for (int i = 0; i < (int)nForms; i++ )
	{
	TCHAR	linebuf[256], *pline = linebuf;
	const FORM_INFO_1 *pf = pForm + i;
	double	sizemm[2];

		/* sizes are in thousandths of a mm */
		sizemm[0] = pf->Size.cx / 1000.;
		sizemm[1] = pf->Size.cy / 1000.;

		pline = linebuf;
		pline += _stprintf(pline, _T(" [%3d] "), i );
		pline += format_formbits(pline, pf->Flags);

		const int prefix = (int)(pline - linebuf);

		pline += _stprintf(pline, _T(" %7.2f %7.2f mm    %s"),
			sizemm[0],
			sizemm[1],
			pf->pName);

		_putts( linebuf );

		if ( Show_points )
		{
			_tprintf(_T("%*s %7.2f %7.2f pts\n"),
				prefix, _T(""),
				MMTOPTS(sizemm[0]),
				MMTOPTS(sizemm[1]));
		}

		if ( Show_inches )
		{
			_tprintf(_T("%*s %7.2f %7.2f in\n"),
				prefix, _T(""),
				MMTOINCH(sizemm[0]),
				MMTOINCH(sizemm[1]));
		}
	}

	free( pForm );
}

/*
 * printable_PINFO1_Flags()
 *
 *	The PRINTER_INFO_1 structure contains flags, and this returns
 *	a static buffer that holds a description of them.
 */
static const TCHAR *printable_PINFO1_Flags(DWORD flags)
{
static TCHAR buf[256];
TCHAR       *p = buf, *p0;

	if ( flags == 0 ) return _T("-none-");

	p += nstrcpy(p, _T("PRINTER_ENUM_"));
	p0 = p;

#define CHKFLAG(flg,s)	if ( (flags & (flg)) == 0 ) ; else { \
				if ( p > p0 ) *p++ = '|'; \
				p += nstrcpy(p, _T(s)); }

	CHKFLAG(PRINTER_ENUM_EXPAND, "EXPAND");
	CHKFLAG(PRINTER_ENUM_CONTAINER, "CONTAINER");
	CHKFLAG(PRINTER_ENUM_ICON1,     "ICON1");
	CHKFLAG(PRINTER_ENUM_ICON2,     "ICON2");
	CHKFLAG(PRINTER_ENUM_ICON3,     "ICON3");
	CHKFLAG(PRINTER_ENUM_ICON4,     "ICON4");
	CHKFLAG(PRINTER_ENUM_ICON5,     "ICON5");
	CHKFLAG(PRINTER_ENUM_ICON6,     "ICON6");
	CHKFLAG(PRINTER_ENUM_ICON7,     "ICON7");
	CHKFLAG(PRINTER_ENUM_ICON8,     "ICON8");

	*p = '\0';

	return buf;
}

/*
 * enumerate_printers()
 *
 *	Given a server name (which might be NULL), run through the list
 *	of all printers on that server. This just lists the names: we can
 *	later query the info for a printer at a time.
 *
 *	If the second argument is TRUE, we're enumerating local providers.
 */
static void enumerate_printers(const TCHAR *svr, BOOL bProvider)
{
	DWORD dwNeeded   = 0;
	DWORD dwReturned = 0;

	TCHAR serverbuf[256];

	if ( svr  &&  svr[0] != '\\' )
	{
		_stprintf(serverbuf, _T("\\\\%s"), svr);

		svr = serverbuf;
	}


	const DWORD enum_flags = ( (svr != 0) || bProvider )
		? PRINTER_ENUM_NAME
		: PRINTER_ENUM_LOCAL;

	if ( ! EnumPrinters(enum_flags,
		const_cast<TCHAR *>(svr),	/* name */
		1,				/* Level */
		0,				/* buffer */
		0,				/* size of buffer */
		&dwNeeded,
		&dwReturned )
	  && GetLastError() != ERROR_INSUFFICIENT_BUFFER )
	{
		die( _T("ERROR: cannot EnumPrinters [%ld]"), GetLastError() );
	}

	PRINTER_INFO_1 *pInfo = (PRINTER_INFO_1 *)malloc(dwNeeded);

	if ( pInfo == 0 )
	{
		die( _T("ERROR: can't get %ld bytes needed for PRINTER_INFO_1"),
			dwNeeded);
	}

	if ( ! EnumPrinters(enum_flags,
		const_cast<TCHAR *>(svr),	/* name */
		1,				/* level */
		(BYTE *)pInfo,			/* info buffer */
		dwNeeded,			/* size of buffer */
		&dwNeeded,
		&dwReturned) )
	{
		die( _T("ERROR: cannot EnumPrinters [%ld]"), GetLastError() );
	}

	for (int i = 0; i < (int)dwReturned; i++ )
	{
	PRINTER_INFO_1 *p = &pInfo[i];

		_tprintf( _T("[%d] name    = \"%s\"\n"),i, p->pName);
		_tprintf( _T("    comment = \"%s\"\n"),    p->pComment);
		_tprintf( _T("    descr   = \"%s\"\n"),    p->pDescription);
		_tprintf( _T("    flags   = %s\n\n"), printable_PINFO1_Flags(
		                                        p->Flags));
	}

	free(pInfo);
}

/*
 * allocGetPrinter()
 *
 *	This is an allocating front-end to the GetPrinter() function:
 *	we provide a level, and it returns a BYTE pointer to the fully
 *	allocated space. The caller must delete[] that pointer.
 *
 *	Return is NULL if we can't get it for whatever reason.
 */
static BYTE *allocGetPrinter(HANDLE hPrinter, DWORD Level)
{
	BYTE *pData   = 0;
	DWORD nBytes  = 0;
	DWORD nNeeded = 0;
	
	if ( ! GetPrinter(hPrinter, Level, 0, 0, &nNeeded)
	  && GetLastError() == ERROR_INSUFFICIENT_BUFFER )
	{
		pData = new BYTE [ nBytes = nNeeded + 128 ];

		if ( GetPrinter(hPrinter, Level, pData, nBytes, &nNeeded) )
			return pData;
	}

	return 0;
}

/*
 * allocGetPrinterDriver()
 *
 *	This is an allocating front-end to the GetPrinter() function:
 *	we provide a level, and it returns a BYTE pointer to the fully
 *	allocated space. The caller must delete[] that pointer.
 *
 *	Return is NULL if we can't get it for whatever reason.
 */
static BYTE *allocGetPrinterDriver(HANDLE hPrinter, TCHAR *env, DWORD Level)
{
	BYTE *pData   = 0;
	DWORD nBytes  = 0;
	DWORD nNeeded = 0;
	
	if ( ! GetPrinterDriver(hPrinter, env, Level, 0, 0, &nNeeded)
	  && GetLastError() == ERROR_INSUFFICIENT_BUFFER )
	{
		pData = new BYTE [ nBytes = nNeeded + 128 ];

		if ( GetPrinterDriver(hPrinter, env, Level, pData, nBytes, &nNeeded) )
			return pData;
	}

	return 0;
}

/*
 * printable_PINFO2_Attributes()
 *
 *	The PRINTER_INFO_2 object has an Attributes field that
 *	we can translate to some bits which are formatted to a
 *	static buffer that is returned to the caller. We don't
 *	do any meaninful checkingto be sure we don't overflow
 *	anything...
 */
static const TCHAR *printable_PINFO2_Attributes(DWORD n)
{
static TCHAR	tbuf[512],
		*t = tbuf;

#define PATTR(a) if ( ! (n & PRINTER_ATTRIBUTE_##a) ) ; else \
		t += nstrcpy(t, _T(#a) _T(" "))

	PATTR(DEFAULT);
	PATTR(DIRECT);
	PATTR(DO_COMPLETE_FIRST);
	PATTR(ENABLE_BIDI);
	PATTR(ENABLE_DEVQ);
	PATTR(FAX);
	PATTR(KEEPPRINTEDJOBS);
	PATTR(QUEUED);
	PATTR(SHARED);
	PATTR(WORK_OFFLINE);
	PATTR(PUBLISHED);
	PATTR(NETWORK);
	PATTR(HIDDEN);
	PATTR(LOCAL);
	PATTR(RAW_ONLY);

	if ( t == tbuf ) t += nstrcpy(t, _T("-none-") );

	return tbuf;
}

/*
 * enumerate_processors()
 *
 *	Run through all the print processors on the current system.
 */
static void enumerate_processors(const TCHAR *svr, const TCHAR *env)
{

	DWORD dwNeeded;
	DWORD dwReturned;

	/*----------------------------------------------------------------
	 * First figure out the directory where the print proccessor DLLs
	 * are located; this depends on the environment.
	 */

	TCHAR dirbuf[1024];

	if ( ! GetPrintProcessorDirectory(
		const_cast<TCHAR *>(svr),		// server name,
		const_cast<TCHAR *>(env),		// environment
		1,					// level
		(BYTE *)dirbuf,
		sizeof(dirbuf),
		&dwNeeded) )
	{
		die(_T("ERROR: cannot GetPrintProcessorDirectory"));
	}

	_tprintf(_T("Print processor dir:\n    %s\n\n"), dirbuf );

	/*----------------------------------------------------------------
	 * Now enumerate the processors themselves
	 */
	
	if ( ! EnumPrintProcessors(
		const_cast<TCHAR *>(svr),	// server name
		NULL,				// environment
		1,				// level
		0,				// data buffer
		0,				// size of buffer 
		&dwNeeded,
		&dwReturned )
	  &&  GetLastError() != ERROR_INSUFFICIENT_BUFFER )
	{
		die( _T("ERROR: cannot EnumPrintProcessors [%ld]"),
			GetLastError() );
	}

	PRINTPROCESSOR_INFO_1 *pInfo = (PRINTPROCESSOR_INFO_1 *)malloc(dwNeeded);

	if ( pInfo == 0 )
	{
		die( _T("ERROR: can't get %ld bytes for PRINTPROCESSOR_INFO_1"),
			dwNeeded);
	}

	if ( ! EnumPrintProcessors(
		const_cast<TCHAR *>(svr),	// server name
		NULL,				// environment
		1,				// level
		(BYTE *)pInfo,			// data buffer
		dwNeeded,			// size of buffer 
		&dwNeeded,
		&dwReturned ) )
	{
		die( _T("ERROR: cannot EnumPrintProcessors [%ld]"),
			GetLastError() );
	}

	for (int i = 0; i < (int)dwReturned; i++ )
	{
	TCHAR *prName = pInfo[i].pName;

		_tprintf( _T("[%d] Processor = \"%s\"\n"), i, prName);

		DATATYPES_INFO_1 tinfo[1024];
		DWORD dwNTypes;

		if ( ! EnumPrintProcessorDatatypes(
			const_cast<TCHAR *>(svr),	// server 
			prName,				// processor name
			1,				// level
			(BYTE *)tinfo,			// buffer
			sizeof tinfo,			// size of buffer
			&dwNeeded,
			&dwNTypes) )
		{
			_tprintf(_T("    ERROR: cannot enum data types [%ld]\n"),
				GetLastError() );
			continue;
		}


		for (int j = 0; j < (int)dwNTypes; j++)
		{
			_tprintf(_T("    Datatype  = \"%s\"\n"),
				tinfo[j].pName);
		}
		_tprintf(_T("\n"));
	}

	free(pInfo);
}

/*
 * show_default()
 *
 *	Show the user the default printer. A TRUE response means we
 *	have a printer, ERROR_FILE_NOT_FOUND means there is no
 *	default, and any other error is an error.
 */
static void show_default(void)
{
	TCHAR dbuf[256];
	DWORD dw = ASIZE(dbuf);

	if ( GetDefaultPrinter(dbuf, &dw) )
		_tprintf( _T("Default = \"%s\"\n"), dbuf);
	else
	{
		const DWORD err = GetLastError();

		if ( err == ERROR_FILE_NOT_FOUND )
			_tcscpy(dbuf, _T("-none-") );
		else
			_stprintf(dbuf, _T("[Err#%ld]"), err);

		_tprintf( _T("Default = %s\n"), dbuf);
	}
}
