/*
 * $Id: //devel/tools/main/ygpm/imuser.cpp#1 $
 *
 * Written by : Stephen J. Friedl
 *              Software Consultant
 *              Tustin, California USA
 *              http://www.unixwiz.net
 *              2002/05/21
 *
 *      NOTE: This code is in the public domain
 *
 *	This module holds all the methods and static variables for the
 *	IMUser class: we have one of these for every user we're watching
 *	for DSLR instant messages. T
 *
 *	Each object is initialized with the DSLR user number (always)
 *	and optionally a name (really just any text string), but we keep
 *	track of other parameters used in the process, such as the I/O
 *	buffers used to talk with the web server. But the I/O itself is
 *	NOT done here.
 *
 *	We don't think this is structured all that well, but it seems to
 *	be doing the trick for the short term.
 */
#include "ygpmcommon.h"
#include "ygpm.h"

/*static*/ IMUser *   IMUser::m_users[MAXUSERS];
/*static*/ int        IMUser::m_nusers = 0;

/*
 * IMUser constructors
 *
 *	The constructor positively must take a DSL Reports user ID, and the
 *	"name" part is optional (we'll fill it in with something generic if
 *	not provided).
 */

IMUser :: IMUser(long uid, const char *uname)
	: m_userid(uid)
	, m_name( 0 )
	, m_url( 0 )
	, m_urlsize( 0 )
	, m_have_PMs(FALSE)
{
	char idbuf[32];

	if ( uname == 0 )
	{
		sprintf(idbuf, "User#%ld", uid);
		uname = idbuf;
	}

	m_name = strdup(uname);

	m_readbuf[0] = '\0';
}


IMUser :: ~IMUser()
{
	if ( m_name ) free( m_name );
	if ( m_url  ) free( m_url  );
}

/*
 * IMUser :: build_request()
 *
 *	Construct the full request that we will pass off to the other
 *	end as a single HTTP request. This is the only place that we
 *	really understand anything about the format of a URL: after
 *	this step it's just a chunk of bytes.
 *
 *	This request includes the GET, the URL of the web site if we're
 *	running a proxy, the HTTP/1.0 tag, and the terminating CR/LF.
 *
 *	We can also include the user agent to let Justin know just which
 *	software is doing the checking: free advertising for me :-)
 */
void IMUser :: build_request(
	const char *proxyname,
	const char *useragent,
	const char *hostname )
{
	char	urlbuf[256],
	        *p = urlbuf;

	// release if previously allocated
	if ( m_url ) free(m_url);

	p += sprintf(p, "GET ");

	if ( proxyname )
	 p += sprintf(p, "%s", proxyname);
	p += sprintf(p, "/instantcheck/%ld", id() );
	p += sprintf(p, " HTTP/1.%d\r\n", hostname ? 1 : 0);

	if ( useragent )
	 p += sprintf(p, "User-Agent: %s\r\n", useragent );

	if ( hostname )
		p += sprintf(p, "Host: %s\r\n", hostname);

	p += sprintf(p, "\r\n");

	m_url     = strdup(urlbuf);
	m_urlsize = (int)(p - urlbuf);
}

/*
 * adduser()
 *
 *	Given two-part user information, perform the physical storage
 *	of the data into the work-to-do table. We do check ahead of
 *	time to make sure this DSLR user ID is not already in the table
 *	once.
 */

/*static*/
void IMUser :: add(long uid, const char *uname)
{
	assert(uid > 0);

	/*----------------------------------------------------------------
	 * Make sure that the current user ID is not already in the table,
	 * then be sure that there is room to add it.
	 */
	for (int i = 0; i < m_nusers; i++ )
	{
		if ( m_users[i]->id() == uid )
		{
			printf("WARNING: ID #%ld already in the table\n", uid);
			return;
		}
	}

	if ( m_nusers > MAXUSERS )
	{
		die("ERROR: too many users to watch (max=%d)", MAXUSERS);
	}

	IMUser *u = new IMUser(uid, uname);

	m_users[m_nusers++] = u;
	
}

/*
 * adduser()
 *
 *	Given a string that represents the numeric user ID at DSLR, plus an
 *	optional name associated with that user, store it into the table of
 *	work we have to do.
 *
 *	The user number is always first - must be numeric! - and it might
 *	be followed by a space or colon to separate from the user name that
 *	follows.
 */

void IMUser :: add(const char *userstring)
{
	assert(userstring != 0);

	/*----------------------------------------------------------------
	 * COLLECT USER INFORATION
	 */
	char       *p  = 0;
	const long uid = strtol(userstring, &p, 10);

	if ( p == userstring )
	{
		die("ERROR: \"%s\" doesn't start with a numeric ID!",
			userstring);
	}

	if ( uid <= 0 )
	{
		die("ERROR: \"%s\" userstring has bad numeric ID", userstring);
	}

	if ( *p == ':' || isspace(*p))
		p++;

	if ( *p == '\0' )
		p = 0;

	add(uid, p);
}


int IMUser::usercount()
{
	return m_nusers;
}

const char *IMUser :: name() const
{
	return m_name;
}

long IMUser::id() const
{
	return m_userid;
}

/*
 * IMUser :: have_PMs()
 *
 *	Given a flag that indicates whether the current IM user has PMs or not,
 *	see if they must be notified about this development. We only do a notify
 *	on a no-msgs to have-msgs transition.
 */

void IMUser :: have_PMs(int have)
{
	if ( have   &&  ! m_have_PMs )
	{
		notify_PM_present(this);
	}
	else if ( ! have &&  m_have_PMs )
	{
		log("resetting PM flag for %s", name() );
	}

	m_have_PMs = have;
}
