/*
 * $Id: //devel/tools/main/ygpm/ygpm.cpp#4 $
 *
 * written by :	Stephen J. Friedl
 *		Software Consultant
 *		steve@unixwiz.net
 *		2002/05/21
 *
 *	This program is a DSL Reports/Broadband Reports IM checker, the
 *	C++ version. We used the perl version (out since Dec 2001) to
 *	be a more or less proof of concept, and though it works fine,
 *	we've found that having to install perl on NT/2000 systems is
 *	a lot of effort. And it doesn't work at all on Win 95/98/ME
 *	systems: this is a bummer.
 *
 *	So this version works largely the same: we fetch the magic
 *	web page from DSLR for the given user ID and parse the response.
 *	If it contains the you-have-PM URL, then we notify the user.
 *	Otherwise, we don't.
 *
 * URL MANAGEMENT
 * --------------
 *
 *	The URL we're fetching is http://www.dslreports.com/instantcheck/###
 *	where ### is the DSL Reports user number. This URL doesn't require
 *	a login, provides no cookie, and is very lightweight in the server.
 *
 *	The return value is a small web page, and the key values are the
 *	IM indicator images:
 *
 *		http://i.dslr.net/bb/imsg.gif	- you have PMs
 *		http://i.dslr.net/bb/imsg1.gif  - no PMs available
 *
 *	We simply look in the returned web page for these strings and
 *	respond accordingly.
 *
 * HAVE/DON'T HAVE LOGIC
 * ---------------------
 *
 *	At regular intervals (60 seconds?) we query the server to see if
 *	we have any IMs, and when we detect that an IM has arrived, we
 *	notify the user with a synchronous dialog box. This blocks waiting
 *	for the user to respond.
 *
 *	This done, we mark the user as now having a PM, and we now disable
 *	notifications until we see there are *no* PMs. To catch this no-PM
 *	state, we increase our scanning to every 15 seconds as long as
 *	there are PMs, throttling back to the "normal" rate once we see
 *	that the user has read the IMs.
 *
 *	This approach means that users with longer interval times are 
 *	less likely to miss a notification if they read a PM, respond,
 *	and receive another PM just before the next check interval. We've
 *	tuned this substantially and belive this to be the right way to
 *	do it.
 *	
 * COMMAND LINE 
 * ------------
 *
 * -u ###:Name	specify the DSL Reports user number (usually six digits,
 *		unless you're Justin, in which case you're "1"). The ID
 *		can be followed by a colon, which provides a display.
 *
 * -i ##        specify query interval in seconds. Default = 60. The number
 *              may be followed with s=seconds m=minutes h=hours to get the
 *              appropriate multiplier. No fractions allowed.
 *
 *		NOTE: time<60 sec is not reasonable (per Justin)
 *
 * -p PXY	specify proxy stuff (unimplemented)
 *
 * -D		enable a bit of debugging
 *
 * -A		disable the User-Agent part of the request
 *
 * -N pgm       run "pgm" when an IM has arrived
 *
 * -H host	Connect to host instead of www.dslreports.com
 *
 * -U useragnt  Use string <useragnt> as the User Agent
 */
#include "ygpmcommon.h"
#include <stdarg.h>
#include "ygpm.h"

#ifdef _WIN32
#  include "getopt.i"
#endif

/*------------------------------------------------------------------------
 * HISTORY
 *
 * 1.10 2007/07/23 - Added -Uuseragent param; http/1.1
 *
 * 1.09 2005/03/07 - Added -Hhostname parameter
 *
 * 1.08 2004/03/24 - removed quite a bit of "error" reporting unless
 *                   debugging is turned on; won't allow a query
 *                   interval <60 seconds.
 *
 * 1.07 2003/10/02 - cleaned up help reporting
 *
 * 1.06 2002/09/19 - now don't hardcode the full URL for the image
 *
 * 1.05 2002/05/22 - playing with the sleep time a bit
 *
 * 1.04 2002/05/21 - minor adjustments to popup box
 *
 * 1.03 2002/05/21 - added timestamped logging, lots of PC-Lint cleanups
 *
 * 1.02 2002/05/21 - added version tags to useragent & console window
 *
 * 1.01 2002/05/21 - public release to Advanced Security
 *
 * 1.00 2002/05/21 - testing with Kasia
 *
 * 0.99 2002/05/20 - still in development
 */

#define VERSION  "1.10"
#define VERSDATE "2007-07-23"

const char Version[]     = VERSION;

const char fullVersion[] = "Unixwiz.net YGPM " VERSION " - " VERSDATE
	" - http://www.unixwiz.net/tools/";

#define MIN_QUERY_INTERVAL	60

/*------------------------------------------------------------------------
 * TUNABLE PARAMETERS
 *
 * DSLREPORTS is the web site we're trying to visit, and it *MUST* have
 * the leading http:// part: we require this whend building a proxy
 * request.
 *
 * "scan interval" is how often we normally check 
 */

#define		DSLREPORTS	"http://www.dslreports.com"

static int		scan_interval = 60,
                        PM_interval   = 15;

static const char	*Hostname = DSLREPORTS;

static char		*proxy   = 0;	// proxy server?
int		        Debug    = 0;

const char		*Useragent = "unixwiz.net YGPM " VERSION;
const char              *Notify_program = 0;

static int  process_readbuf( IMUser *im );
static long parse_interval(const char *s);
static void usage(const char *argv0);

extern "C"
int __cdecl main(int argc, char **argv)
{
	/*----------------------------------------------------------------
	 * PROCESS COMMAND LINE
	 *
	 * Using our normal getopt() processing, crack the command line
	 * parameters.
	 */
	int	c;

	while ( (c = getopt(argc, argv, "H:u:i:p:DAVN:TU:")) != EOF )
	{
		switch (c)
		{
		  case 'U':
			Useragent = optarg;
			break;

		  case 'H':
			Hostname = optarg;
			break;

		  case 'T':
			timestamp_seconds = TRUE;
			break;

		  case 'N':			// -N pgm - notify progrtam
			Notify_program = optarg;
			break;

		  case 'V':			// -V - report version
			puts(fullVersion);
			exit(EXIT_SUCCESS);

		  case 'A':			// -u - disable user agent
			Useragent = 0;
			break;

		  case 'D':
			Debug++;
			break;

		  case 'u':
			IMUser::add(optarg);
			break;

		  case 'i':
			scan_interval = parse_interval(optarg);
			break;

		  case 'p':
			proxy = optarg;
			break;

		  default:
			if ( optopt == '?' )
				usage(argv[0]);
			else
				die("(try \"%s -?\" for help)", argv[0]);
		}
	}

	/*----------------------------------------------------------------
	 * SANITY CHECKING OF PARAMETERS
	 *
	 * make sure we have stuff that makes sense: error if not
	 */
	if ( optind < argc )
	{
		printf("ERROR: too many parameters\n");
		usage(argv[0]);
	}

	if ( IMUser::usercount() == 0 )
	{
		printf("ERROR: missing users to check IMs for\n");
		usage(argv[0]);
	}


#ifdef _WIN32
	SetConsoleTitle("unixwiz.net YGPM " VERSION);
#endif

	puts(fullVersion);
	printf("Query interval = %d secs to %s\n", scan_interval, Hostname);

	/*----------------------------------------------------------------
	 * Scanning faster than once a minute is just unreasonble (per
	 * Justin, and per common sense). Sorry.
	 */
	if ( scan_interval < MIN_QUERY_INTERVAL )
	{
		printf("Sorry, refresh interval %d unreasonble; using %d\n",
			scan_interval,
			MIN_QUERY_INTERVAL);

		scan_interval = MIN_QUERY_INTERVAL;
	}

	/*----------------------------------------------------------------
	 * DETERMINE TCP CONNECT() TARGET
	 *
	 * We're going to be sending our TCP connect() requests to either
	 * http://www.dslreports.com:80, or to the proxy server specified
	 * by the user, and this parses that information. Ultimately we
	 * are going to populate "target" with the information we require
	 * for this.
	 *	
	 * If the proxy server address contains a colon, it's followed by
	 * the port information.
	 */
	const char *hostname = strrchr(Hostname, '/');

	if ( hostname == 0 )
		hostname = Hostname;
	else
		hostname++;		// skip the final slash

#ifdef _WIN32
	init_winsock(Debug, 2);
#endif
	struct sockaddr_in target;

	memset(&target, 0, sizeof target);

	target.sin_family = AF_INET;
	target.sin_port   = htons(80);

	if ( proxy )
	{
		hostname = proxy;

		char *p = strrchr(proxy, ':');

		if ( p )
		{
			*p++ = '\0';	// smoosh the colon

			target.sin_port = htons( (unsigned short) atoi(p) );
		}
		else
		{
			target.sin_port = htons( 8080 );
		}
	}

	/* look up the user name */

	struct hostent *hp = gethostbyname(hostname);

	if ( hp == 0 )
		die("ERROR: cannot find IP address for {%s}", hostname);

	memcpy(&target.sin_addr, hp->h_addr, 4);

	if ( Debug )
	{
		printf("connecting to %s:%d\n",
			inet_ntoa(target.sin_addr),
			ntohs(target.sin_port));
	}

	/*----------------------------------------------------------------
	 * CREATE FULL HTTP REQUESTS
	 *
	 * For each IM user object, build the full HTTP request so later
	 * on we can just send a chunk of bytes. This request buffer is
	 * never changed from loop to loop, so building it statically is
	 * much more efficient.
	 */
	const char *website = proxy ? DSLREPORTS : 0;

	for (int i = 0; i < IMUser::m_nusers; i++ )
	{
		IMUser *u = IMUser::m_users[i];

		u->build_request(website, Useragent, hostname);

		printf("Checking user %ld (%s)\n",
			u->id(),
			u->name());
	}


	/*----------------------------------------------------------------
	 * THE BIG LOOP!
	 *
	 * Run through the list of all users and query the PMs for each.
	 * If any IMuser has PMs we do a notification, then speed up the
	 * cycling to quickly detect when the has *read* the IMs.
	 *
	 * If we are not able to create a socket 10 times in a row, we
	 * have something really wrong and just ought to exit.
	 */

	int max_fetchtime = 15;

	for ( int errcount = 0; errcount < 10; )
	{
		int any = FALSE;

		for (int j = 0; j < IMUser::m_nusers; j++ )
		{
			IMUser *u = IMUser::m_users[j];

			SOCKET sfd = create_nonblocking_socket();

			if ( SOCKET_IS_VALID(sfd) )
			{
				int rc = dorequest( sfd,
				                    &target,
				                    u,
				                    max_fetchtime );

				closesocket(sfd);

				if ( rc ) any += process_readbuf(u);

				errcount = 0;
			}
			else
			{
				errcount++;
			}
		}

		/*--------------------------------------------------------
		 * COMPUTE SLEEP TIME
		 *
		 * Normally we have two amounts of time we sleep for:
		 *
		 * scan_interval - the user's check-for-PM frequency
	 	 *
		 * PM_interval   - internal check rate to see when the user
		 *                 actually reads the PM. It's normally
		 *                 much faster.
		 *
		 * For no really good reason, we like to align our sleep
		 * times with the second hand on the clock, so we wake up
		 * again at the top of the minute (or other appropriate
		 * modulo of the sleep time we're using).
		 *
		 * This ends up rounding the sleep time *down*, which is
		 * OK with us.
		 */
		int sleeptime = any ? PM_interval : scan_interval;

		time_t now;
		time(&now);

		sleep_seconds(sleeptime - (now % sleeptime));
	}

	printf("ERROR: too many errors\n");

	return EXIT_FAILURE;
}

/*
 * process_readbuf()
 *
 *	Given a user object with a presumably-full read buffer, run
 *	through it looking for the two image URLs of interest:
 *
 *	http://i.dslr.net/bb/imsg.gif  - you have PMs!
 *	http://i.dslr.net/bb/imsg1.gif - no PMs available
 *
 *	The returned string is not all that large, but we prefer to keep
 *	the string searches a bit more reasonable
 */

static int process_readbuf( IMUser *im )
{
	static const char SEARCHER[] = "/bb/";
	static const char PM_YES  [] = "imsg.gif";
	static const char PM_NO   [] = "imsg1.gif";

	assert(im != 0);

	/*----------------------------------------------------------------
	 * First find the location of the leading part of the URL: if
	 * we don't have that leading part, there is no point in looking
	 * for the tail part.
	 */
	const char *p = strstr( im->m_readbuf, SEARCHER );

	if ( p == 0 ) return 0;

	p += sizeof(SEARCHER)-1;

	if ( Debug ) printf("considering {%s}\n", p);

	if ( strncmp(p, PM_YES, sizeof(PM_YES)-1) == 0 )
	{
		im->have_PMs(TRUE);
		return TRUE;
	}

	if ( strncmp(p, PM_NO,  sizeof(PM_NO)-1) == 0 )
	{
		im->have_PMs(FALSE);
		return FALSE;
	}

	return FALSE;
}

/*
 * parse_interval()
 *
 *	Given a string with digits and an optional measurement token
 *	after, return the number of seconds represented by this string.
 *	The string has to have *some* digits, and the base integer
 *	must be postiive.
 *
 *	Trailing tokens
 *
 *		s	seconds
 *		m	minutes
 *		h	hours
 *
 *	We do NOT recognize fractional parts, and there can't be any
 *	spaces between the digits and the trailing token.
 */
static long parse_interval(const char *s)
{
	assert(s != 0);

	char *p;

	long interval = strtol(s, &p, 10);

	if ( s == p )
	{
		die("ERROR: {%s} is bad interval - must start with digit", s);
	}

	if ( interval == 0 )
	{
		die("ERROR: {%s} is bad interval - must be nonzero", s);
	}

	if ( *p == '\0'  ||  strcmp(p, "s") == 0 )
	{
		/* unadorned - seconds */
	}

	else if ( strcmp(p, "m") == 0 )
	{
		interval *= 60;			// minutes
	}

	else if ( strcmp(p, "h") == 0 )
	{
		interval *= 60*60;		// hours
	}

	else
	{
		die("ERROR: unknown unit of measure in {%s}", s);
	}

	return interval;
}

/*
 * usage()
 *
 *	Show the user how to run the program
 */

#define pr	(void) printf

static void usage(const char *argv0)
{
	assert(argv0 != 0 );

	puts(fullVersion);

	pr("\n");

	pr("usage: %s [options] -u id:name [-u....]\n", argv0);

	pr("\n");
	pr(" -?         show this help listing\n");
	pr(" -u #:n     add DSLR user # (with display name N) to work list\n");
	pr(" -D         add a bit of debugging\n");
	pr(" -i ##      scan interval is ## seconds (##m = mins)\n");
	pr(" -p pxy     use pxy as a proxy server (:port allowed)\n");
	pr(" -A         disable the User-Agent token\n");
	pr(" -H host    Connect to <host> instead of %s\n", Hostname);
	pr(" -U str     Use <str> as the user agent\n");
	pr("\n");
	pr("\n");
	pr("The \"id\" must be a numeric DSLR User id, and can be followed\n");
	pr("by \":Name\" which is used to display who has PMs available (it\n");
	pr("need not match the *actual* DSLR user name\n");

	exit(EXIT_SUCCESS);
}
