/*
 * $Id: //devel/tools/main/ygpm/selectloop.cpp#2 $
 *
 *	This is the "select engine" that talks to the network in order to
 *	fetch the response from the target web server. This could be much
 *	simpler if we decided to forgo the timeouts, but for a robust
 *	application we simply have to manage all the timeouts correctly.
 *	So this means a nonblocking socket and a LOT of select management.
 *
 *	There are three steps:
 *
 *	- CONNECT
 *	- WRITE
 *	- READ
 *
 *	The connect is always done by its own routine (doconnect), and it
 *	returns only when we have a valid connection to the other end.
 *
 *	Reading and writing are done in the same loop: we write until
 *	we have no more, then start to read. When the read returns 0,
 *	it means we got a valid closed socket.
 */
#include "ygpmcommon.h"
#include "ygpm.h"

/*
 * create_nonblocking_socket()
 *
 *	Create a streaming TCP/IP socket that is properly nonblocking.
 *	This means that any I/O -- connect, read, write -- on it will
 *	return *immediately* and we have to take steps to manage our
 *	own waiting and timeouts>
 */
SOCKET __stdcall create_nonblocking_socket(void)
{
	/*----------------------------------------------------------------
	 * First create the socket itself: the error reporting depends
	 * on wthe  
	 */
	SOCKET sfd = socket(PF_INET, SOCK_STREAM, IPPROTO_TCP);

	if ( ! SOCKET_IS_VALID(sfd) )
	{
#ifdef _WIN32
		die("ERROR: cannot create socket [err=#%ld]",
			WSAGetLastError());
#else
		die("ERROR: cannot create socket [%s]", strerror(errno));
#endif
	}

	/*----------------------------------------------------------------
	 * MAKE IT NONBLOCKING
	 *
	 * This is sadly depending on the operating system :-(
	 */
#ifdef O_NONBLOCK

	fcntl(sfd, F_SETFL, fcntl(sfd, F_GETFL, 0) | O_NONBLOCK);

#elif defined(_WIN32)

	u_long block = 0;

	/*lint --e(569) // loss of sign in FIOBIO */
	(void) ioctlsocket(sfd, FIONBIO, &block);
#endif

	return sfd;
}


/*
 * doconnect()
 *
 *	Given a socket descriptor, a target to connect to, and the maximum
 *	time we should wait for it, connect() to the other end. Since this
 *	is a nonblocking socket, we have to do a bunch more work to manage
 *	both the immediate and the delayed socket errors.
 *
 *	Return is TRUE if all is well, or FALSE if we couldn't connect
 *	for whatever reason.
 */
static int doconnect( SOCKET sfd,
                      const struct sockaddr_in *target,
                      int maxtime_secs )
{
	assert( target != 0 );

	/*----------------------------------------------------------------
	 * TRY DIRECT CONNECT() 
	 *
	 * The "regular" connect() is nonblocking, but it still might manage
	 * to succeed or fail immediately (especially if there is a proxy
	 * server on the local network for it).
	 */
	int rc = connect(sfd, (struct sockaddr *)target, sizeof *target);

	if ( errno == 0 )
	{
		return TRUE;            /* ALL OK */
	}
	else if ( errno != EINPROGRESS )
	{
		return FALSE;           /* GENERIC ERROR */
	}

	/*----------------------------------------------------------------
	 * WAIT FOR CONNECT TO FINISH
	 *
	 * The connect() couldn't complete right away, so we have to wait
	 * for up to the limit of time for it to do so. Once we have the
	 * result, fetch the error code (success or failure) and likewise
	 */

	fd_set wfds;
	FD_ZERO(&wfds);
	FD_SET(sfd, &wfds);

	struct timeval tv;
	init_timeval(&tv, maxtime_secs);

	/*lint --e(713) // loss of precision in sfd+1 */
	rc = select(sfd + 1, 0, &wfds, 0, &tv);

	if ( rc == 0 )
	{
		if ( Debug ) printf("timeout during connect/select\n");
		return FALSE;
	}
	if ( rc < 0 )
	{
		if ( Debug ) printf("error during connect/select\n");
		return FALSE;
	}

	/* select worked, so get socket error/completion code */
	int       err    = 0;
	socklen_t errlen = sizeof err;

	if ( getsockopt(sfd, SOL_SOCKET, SO_ERROR, (char *)&err, &errlen) < 0 )
	{
		printf("ERROR: cannot getsockopt in connect\n");

		return FALSE;
	}

	if ( err == 0 )
	{
		return TRUE;
	}
	else
	{
		printf("ERROR: connect/select delayed return %d\n", err);
		errno = err;
		return FALSE;
	}
}

/*
 * dorequest()
 *
 *	This talks on the given socket and makes a connection to the
 *	other end, sends our request buffer, and waits for a response
 *	to come back.
 *
 *	Since the socket is nonblocking, we havbe to 
 *
 */

int __stdcall dorequest( SOCKET sfd,
                         const struct sockaddr_in *target,
                         IMUser *u,
                         int maxtime_secs )
{
	assert(target != 0);
	assert(u      != 0);

	u->m_readbuf[0] = '\0';

	/*----------------------------------------------------------------
	 * DEAL WITH TIMEOUTS
	 *
	 * Given the max time we will spend trying to do a connect, compute
	 * the max *ending* time of this interaction. This allows us to have
	 * a hard upper limit on the transaction: our wait times are always
	 * computed relative to this ending time.
	 */
	time_t	clkmax;

	time(&clkmax);

	clkmax += maxtime_secs;

	/*----------------------------------------------------------------
	 * CONNECT TO THE OTHER END
	 *
	 * Try to connect to the other end, and if we can't do it, we
	 * return FALSE to bail on the whole thing.
	 */

	if ( ! doconnect(sfd, target, maxtime_secs) )
	{
		printf("ERROR: cannot connect to remote\n");
		return FALSE;
	}

	if ( Debug ) printf("connected\n");

	/*----------------------------------------------------------------
	 * SET UP BUFFERING
	 *
	 * We are first sending to then receving from the socket, so we
	 * set up our buffer pointers to do this easily.
	 *
	 * "sendnext" and "readnext" both point to the next location in
	 * the key buffers that we should send and receive at, and once
	 * each of these pointers reaches its max value, we're done with
	 * that part of the transaction.
	 *
	 * Our loop is such that sending works first, then receiving,
	 * and only one at a time.
	 */
	const char *sendnext = u->m_url,
	           *sendmax  = u->m_url + u->m_urlsize;

	char	   *readnext = u->m_readbuf,
	           *readmax  = u->m_readbuf + sizeof(u->m_readbuf) - 1;

	*readnext = '\0';

	while ( (sendnext < sendmax)  ||  (readnext < readmax) )
	{
		/*--------------------------------------------------------
		 * COMPUTE TIMEOUT
		 *
		 * We only are allowed to do our selecting for so long:
		 * after that max time we simply have to give up on the
		 * whole thing.
		 */
		struct timeval tv;

		time_t now; 
		time(&now);

		if ( now >= clkmax )
		{
			if ( Debug )
				printf("TIMEOUT in timeval calc\n");
			return FALSE;
		}

		init_timeval(&tv, clkmax - now);

		/*--------------------------------------------------------
		 * DETERMINE READ/WRITE DIRECTION
		 *
		 * We are either reading or writing, but not both.  If we
		 * have more data to write: do so. Otherwise we're reading.
	
		 */
		fd_set fdset,         // base "real" set of file descriptors
		       *rfds = 0,     // ptr to wait-for-readable descriptors
		       *wfds = 0;     // ptr to wait-for-writable descriptors

		FD_ZERO(&fdset);
		FD_SET(sfd, &fdset);

		if ( sendnext < sendmax )	wfds = &fdset;
		else                            rfds = &fdset;

		if ( Debug > 1 )
		{
			printf("selecting(%ld secs) %s\n",
				tv.tv_sec,
				wfds ? "WRITING" : "READING" );
		}

		/*--------------------------------------------------------
		 * WAIT FOR I/O
		 *
	 	 * Now we're waiting for our socket to be ready to read
	 	 * or to write, and we wait up to our max time for this
		 * to happen.
		 *
		 * Errors are always bad news, and we bail immediately
		 * when it happens.
		 */
		/*lint --e(713) // loss of precision in sfd+1 */
		int rc = select(sfd+1, rfds, wfds, 0, &tv);

		if ( rc == 0 )
		{
			if ( Debug )
			    printf("*timeout in select (waiting for %s)\n",
				rfds ? "read" : "write" );
			return FALSE;
		}
		else if ( rc < 0 )
		{
			if ( Debug )
			    printf("*error in select (waiting for %s)\n",
				rfds ? "read" : "write" );
			return FALSE;
		}

		/*--------------------------------------------------------
		 * TRY TO WRITE?
		 *
		 * If our want-to-write pointer is set, then we're in the
		 * write part of our loop (and this means we *must*) still
		 * have more to do. Try to write the remainder of the data,
		 * updating the buffer on success. This is nonblocking.
		 */
		if ( wfds )
		{
			assert(sendnext < sendmax);

			const int nleft = (int)(sendmax - sendnext);

			const int nw = send(sfd, sendnext, nleft, 0);

			if ( nw <= 0 )
			{
				printf("ERROR: cannot write [%d]\n", errno);
				return FALSE;
			}

			sendnext += nw;
		}

		/*--------------------------------------------------------
		 * TRY TO READ?
		 *
		 * Likewise write, try to fill the read buffer with data
		 * from the socket. A read of zero means that we've gotten
		 * an normal EOF, so we can exit the conversation in a
		 * graceful manner.
		 */
		else if ( rfds )
		{
			assert(readnext < readmax);

			const int nleft = (int)(readmax - readnext);

			const int nr = recv(sfd, readnext, nleft, 0);

			if ( nr > 0 )
			{
				/* got valid data */
				*( readnext += nr ) = '\0';
			}
			else if ( nr == 0 )
			{
				/* GOT EOF */
				break;
			}
			else /* nr < 0 */
			{
				printf("ERROR: cannot read [%d]\n", errno);
				return FALSE;
			}
		}
	}

	/*----------------------------------------------------------------
	 * The return value is TRUE onlyif we actually have any data in
	 * our read buffer. Though *technically* an empty read could be
	 * part of a valid TCP transaction, in practice we really do need
	 * to see some actual data in order to do anything.
	 */
	return (readnext >= u->m_readbuf);
}
